# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# -------------------------------------------------------------------------------
# This module implements the forwarding mode related commands for Etba for
# special multicast packets like igmp, PIM and link local multicast.
# None of these  are config commands that will be reflected in running-config.
# -------------------------------------------------------------------------------

import BasicCli
import CliCommand
import CliMatcher
import CliToken.Platform
from CliPlugin import BridgingEtbaCli
import LazyMount
import ShowCommand

igmpSnoopingHwConfig = None

matcherMulticast = CliMatcher.KeywordMatcher( 'multicast',
      helpdesc=( 'Control trapping of IGMP, PIM and link-local multicast packets '
                  'when IGMP snooping is enabled/disabled' ) )
matcherSwForwarding = CliMatcher.KeywordMatcher( 'sw-forwarding',
      helpdesc='Configure forwarding of special multicast packets' )
nodeEtba = CliCommand.guardedKeyword( 'etba',
      helpdesc='Ebra Test Bridge configuration commands',
      guard=BridgingEtbaCli.etbaGuard )

# --------------------------------------------------------------------------------
# [ no | default ] platform etba sw-forwarding multicast
#                            flooding-without-igmpsnooping ( igmp | linkLocal | pim )
# --------------------------------------------------------------------------------
class PlatformSwForwardWithoutIgmpSnoopingCmd( CliCommand.CliCommandClass ):
   syntax = ( 'platform etba sw-forwarding multicast flooding-without-igmpsnooping '
                                                      '( igmp | linkLocal | pim )' )
   noOrDefaultSyntax = syntax
   data = {
      'platform': CliToken.Platform.platformMatcherForConfig,
      'etba': nodeEtba,
      'sw-forwarding': matcherSwForwarding,
      'multicast': matcherMulticast,
      'flooding-without-igmpsnooping': ( 'Configure software flooding of IGMP, PIM '
         'and link-local multicast packets when IGMP snooping is disabled' ),
      'igmp': 'IgmpSwFlooding when IGMP snooping is disabled',
      'linkLocal': 'LinkLocalMulicastSwFlooding when IGMP snooping is disabled',
      'pim': 'PimSwFlooding when IGMP snooping is disabled',
   }

   @staticmethod
   def handler( mode, args ):
      if 'igmp' in args:
         igmpSnoopingHwConfig.igmpSwFloodingWhenSnoopingDisabled = True
      elif 'linkLocal' in args:
         igmpSnoopingHwConfig.linkLocalMulticastSwFloodingWhenSnoopingDisabled = True
      elif 'pim' in args:
         igmpSnoopingHwConfig.pimSwFloodingWhenSnoopingDisabled = True
      else:
         assert False, 'Unknown type'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'igmp' in args:
         igmpSnoopingHwConfig.igmpSwFloodingWhenSnoopingDisabled = False
      elif 'linkLocal' in args:
         igmpSnoopingHwConfig.linkLocalMulticastSwFloodingWhenSnoopingDisabled = \
                                                                                False
      elif 'pim' in args:
         igmpSnoopingHwConfig.pimSwFloodingWhenSnoopingDisabled = False
      else:
         assert False, 'Unknown type'

BasicCli.GlobalConfigMode.addCommandClass( PlatformSwForwardWithoutIgmpSnoopingCmd )

# --------------------------------------------------------------------------------
# [ no | default ] platform etba sw-forwarding multicast ( linkLocal | pim )
# --------------------------------------------------------------------------------
class PlatformSwForwardCmd( CliCommand.CliCommandClass ):
   syntax = 'platform etba sw-forwarding multicast ( linkLocal | pim )'
   noOrDefaultSyntax = syntax
   data = {
      'platform': CliToken.Platform.platformMatcherForConfig,
      'etba': nodeEtba,
      'sw-forwarding': matcherSwForwarding,
      'multicast': matcherMulticast,
      'linkLocal': 'LinkLocalMulticastSwForwarding when IGMP snooping is enabled',
      'pim': 'PimSwForwarding when IGMP snooping is enabled',
   }

   @staticmethod
   def handler( mode, args ):
      if 'linkLocal' in args:
         igmpSnoopingHwConfig.linkLocalMulticastSwForwarding = True
      elif 'pim' in args:
         igmpSnoopingHwConfig.pimSwForwarding = True
      else:
         assert False, 'Unknown type'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'linkLocal' in args:
         igmpSnoopingHwConfig.linkLocalMulticastSwForwarding = False
      elif 'pim' in args:
         igmpSnoopingHwConfig.pimSwForwarding = False
      else:
         assert False, 'Unknown type'

BasicCli.GlobalConfigMode.addCommandClass( PlatformSwForwardCmd )

# pylint: disable=consider-using-f-string
# --------------------------------------------------------------------------------
# show platform etba sw-forwarding multicast
# --------------------------------------------------------------------------------
class PlatformEtbaSwForwardingMulticastCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform etba sw-forwarding multicast'
   data = {
      'platform': CliToken.Platform.platformMatcherForShow,
      'etba': nodeEtba,
      'sw-forwarding': matcherSwForwarding,
      'multicast': matcherMulticast,
   }

   @staticmethod
   def handler( mode, args ):
      print( 'Etba sw-forwarding-mode for special multicast packets:' )
      print( '--------------------------------------------------------' )
      print( "%-48s : %s" % ( 'pimSwForwarding',
            igmpSnoopingHwConfig.pimSwForwarding ) )
      print( "%-48s : %s" % ( 'linkLocalMulticastSwForwarding',
       igmpSnoopingHwConfig.linkLocalMulticastSwForwarding ) )
      print( "%-48s : %s" % ( 'igmpSwFloodingWhenSnoopingDisabled',
       igmpSnoopingHwConfig.igmpSwFloodingWhenSnoopingDisabled ) )
      print( "%-48s : %s" % ( 'pimSwFloodingWhenSnoopingDisabled',
       igmpSnoopingHwConfig.pimSwFloodingWhenSnoopingDisabled ) )
      print( "%-48s : %s" % ( 'linkLocalMulticastSwFloodingWhenSnoopingDisabled',
       igmpSnoopingHwConfig.linkLocalMulticastSwFloodingWhenSnoopingDisabled ) )
      print()

BasicCli.addShowCommandClass( PlatformEtbaSwForwardingMulticastCmd )

def Plugin( entityManager ):
   global igmpSnoopingHwConfig
   # Note this is intentionally not a config mount. These CLI commands ares supported
   # only on etba and not on the product. On products, this entity is written by the
   # hardware forwarding agent.
   igmpSnoopingHwConfig = LazyMount.mount( entityManager,
                                           'bridging/igmpsnooping/hardware/config',
                                           'Bridging::IgmpSnooping::HwConfig',
                                           'w' )
