#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Arnet import IpGenAddr
import Cell
import CliParser
import ConfigMount
import LazyMount
import SmashLazyMount
import Plugins
import ShowCommand
import CliCommand
import CliMatcher
import BasicCli
import CliPlugin.TechSupportCli
from CliPlugin import SftModel
from CliPlugin import InbandTelemetryCliLib
from CliPlugin import IntfCli
from CliPlugin.ClassificationCliLib import CommitAbortModelet
from CliPlugin.PolicyMapCliLib import PolicyOpChkr
from CliPlugin.SftModel import FtrCounters
from CliPlugin.TrafficPolicyCliModel import Rule, Matches, NumericalRange, Actions
from CliPlugin.InbandTelemetryCliModel import (
   Profile,
   IntfList,
   ModelIntProfileSummary,
   InbandTelemetry,
   InbandTelemetryProfiles,
   IntTrackingModel,
   IntFlowDetailModel,
   IntFlowDetailNodeInfo,
   IntFlowDetailIntervalStats,
   SamplePolicyModel,
)
from CliPlugin.SftCliShow import (
   ShowCounters,
   generateShowTrackingFilter,
   flowKeyMatch,
   getVrfNameByVrfIdFromArpVrfIdMap,
   ftConfig,
   ftCounters,
   interfaceCounters,
   ipfixStats,
   shmemEntityManager,
   ethIntfStatusDir,
)
from CliPlugin.SftCliLib import (
   lfibSourceToTopLabelType,
   AllowedPacketDirection,
)
from CliPlugin.FlowTrackingCliLib import (
   getFlowGroups,
   IP6_GROUP,
   getIfIndexMap,
   EGRESS_IP6_GROUP,
   IP_GROUP,
   EGRESS_IP_GROUP,
   exporterKw,
   exporterNameMatcher,
   flowTableKw,
   ftrTypeKwStr,
   getFtrTypeFromArgs,
   inbandShowKw,
   isInbandTelemetryAgentRunning,
   telemetryShowKw,
   trackerKw,
   trackerNameMatcher,
   trackingShowKw,
)
from CliPlugin.TrafficPolicyCli import ( ShowPendingCmd, matchRuleName,
                               DscpConfigCmd, ActionsConfigCmdBase,
                               MatchRuleConfigCmdBase )
from CliPlugin.InbandTelemetryCliLib import ( IP_PROTOCOL_RESERVED,
                                              IntSamplePolicyContext,
                                              IntSamplePolicyMatchRuleContext )
from CliPlugin.SamplePolicyCliLib import ( SamplePolicyConfigCmd,
                                 SampleActionCmd )
from SftLib import (
   ifindexToIntf,
   vniOrUnknown,
)
import Tac
import Tracing
from CliMode.InbandTelemetry import ( FEATURE,
                                      InbandTelemetryConfigMode,
                                      ProfileConfigMode, IntSamplePolicyConfigMode,
                                      IntSamplePolicyMatchRuleConfigMode,
                                      IntSamplePolicyActionRuleConfigMode )
from InbandTelemetryTypes import ( tacCollectorLogBufferSize,
                                   tacPortProfileType,
                                   tacCollectorType,
                                   tacOperState, )
import CliToken.Monitor
import CliToken.Cli
import CliToken.Flow
import SharedMem

from Toggles.InbandTelemetryCommonToggleLib import (
   toggleFeatureIfa1BasedInbandTelemetryEnabled,
   toggleFeatureIfa2BasedInbandTelemetryEnabled,
   toggleFeatureInbandTelemetrySamplePolicyEnabled,
   toggleFeatureInbandTelemetryVxlanEnabled,
   toggleFeatureInbandTelemetryTh3TransitSupportEnabled,
      )
from FlowTrackerCliUtil import (
   getInbandTelemetryFlowTrackingCliConfigPath,
   ftrTypeInbandTelemetry,
   ipStr,
   tcpFlagStr,
)
import Smash
from TypeFuture import TacLazyType

AddressFamily = TacLazyType( "Arnet::AddressFamily" )
MplsLabel = TacLazyType( 'Arnet::MplsLabel' )
IpGenPrefix = TacLazyType( 'Arnet::IpGenPrefix' )

__defaultTraceHandle__ = Tracing.Handle( 'InbandTelemetryCli' )
t0 = Tracing.trace0
t1 = Tracing.trace1
t8 = Tracing.trace8

#------------------------------------------------------------------------------------
# Global variables
#------------------------------------------------------------------------------------
inbandTelemetryConfig = None
inbandTelemetryStatus = None
intFlowTrackingCollectors = None
policiesCliConfig = None
policiesStatusRequestDir = None
policiesStatus = None
hwCapability = None
em = None
activeAgentDir = None
intFtCounters = None

inbandTelemetryHwConfig = None
inbTmHeaderStatus = None
inbTmMetadataStatus = None
inbTmRulesStatus = None

minSampleRate = 1
defSampleRate = 1 << 20
maxSampleRate = 1 << 24
minLogNum = 64
maxLogNum = 4 * 1024
deflogNum = 1024
maxConfigVersion = Tac.Value(
      "InbandTelemetry::CommandMaskType" ).cliCommandMask
zeroIpv4GenAddr = IpGenAddr( "0.0.0.0" )
zeroIpv6GenAddr = IpGenAddr( "::" )
intVersions = Tac.Type( "InbandTelemetry::IntVersion" )

#------------------------------------------------------------------------------------
# Utility functions
#------------------------------------------------------------------------------------
# pylint: disable-msg=R1703

def copyIntParam( srcParam, dstParam ):
   dstParam.deviceId = srcParam.deviceId
   dstParam.probeMarker = srcParam.probeMarker
   dstParam.probeMarkerDisabled = srcParam.probeMarkerDisabled
   dstParam.probeProtocol = srcParam.probeProtocol
   dstParam.intDetectionMethod = srcParam.intDetectionMethod
   dstParam.udpDstPort1 = srcParam.udpDstPort1
   dstParam.udpDstPort1Enabled = srcParam.udpDstPort1Enabled
   dstParam.udpDstPort2 = srcParam.udpDstPort2
   dstParam.udpDstPort2Enabled = srcParam.udpDstPort2Enabled
   dstParam.intVersion = srcParam.intVersion

def isDefaultIntParam( param ):
   defaultParam = Tac.newInstance( 'InbandTelemetry::InbandTelemetryParam' )
   return identicalIntParam( param, defaultParam )


def identicalIntParam( oldParam, newParam ):
   if oldParam.deviceId == newParam.deviceId and \
      oldParam.probeMarker == newParam.probeMarker and \
      oldParam.probeMarkerDisabled == newParam.probeMarkerDisabled and \
      oldParam.probeProtocol == newParam.probeProtocol and \
      oldParam.intDetectionMethod == newParam.intDetectionMethod and \
      oldParam.udpDstPort1 == newParam.udpDstPort1 and  \
      oldParam.udpDstPort1Enabled == newParam.udpDstPort1Enabled and  \
      oldParam.udpDstPort2 == newParam.udpDstPort2 and  \
      oldParam.udpDstPort2Enabled == newParam.udpDstPort2Enabled and \
      oldParam.intVersion == newParam.intVersion:
      return True
   return False

def identicalProfile( profile, currentEntry ):
   if not profile or not currentEntry:
      return False
   if profile.type != currentEntry.type or \
           profile.collectorName != currentEntry.collectorName or \
           profile.collectorType != currentEntry.collectorType or \
           profile.sampleRate != currentEntry.sampleRate or \
           profile.samplePolicy != currentEntry.samplePolicy or \
           profile.truncate != currentEntry.truncate or \
           profile.truncationSize != currentEntry.truncationSize or \
           profile.egressDrop != currentEntry.egressDrop:
      return False
   return True

def identicalProfileList( oldPortProfiles, newPortProfiles, profileType="" ):
   if set( oldPortProfiles ) != set( newPortProfiles ):
      return False
   for profile in oldPortProfiles:
      # default edge/core profiles need not be compared as they are immutable
      if profileType != "vxlan" and profile == 'default':
         continue
      oldProfile = oldPortProfiles[ profile ]
      newProfile = newPortProfiles[ profile ]
      if not identicalProfile( oldProfile, newProfile ):
         return False
   return True

def identicalTelemetryConfig( oldConfig, newConfig ):
   if not identicalIntParam( oldConfig.intParam, newConfig.intParam ):
      return False
   if oldConfig.enable != newConfig.enable:
      return False

   if not identicalProfileList( oldConfig.corePortProfiles,
         newConfig.corePortProfiles ):
      return False
   if not identicalProfileList( oldConfig.edgePortProfiles,
         newConfig.edgePortProfiles ):
      return False
   if not identicalProfileList( oldConfig.vxlanPortProfiles,
         newConfig.vxlanPortProfiles, profileType="vxlan" ):
      return False

   oldIntfToEdgePortProfiles = set( oldConfig.intfToEdgePortProfiles.values() )
   newIntfToEdgePortProfiles = set( newConfig.intfToEdgePortProfiles.values() )
   oldIntfToCorePortProfiles = set( oldConfig.intfToCorePortProfiles.values() )
   newIntfToCorePortProfiles = set( newConfig.intfToCorePortProfiles.values() )
   oldIntfToVxlanPortProfiles = set( oldConfig.intfToVxlanPortProfiles.values() )
   newIntfToVxlanPortProfiles = set( newConfig.intfToVxlanPortProfiles.values() )

   if oldIntfToEdgePortProfiles != newIntfToEdgePortProfiles or \
         oldIntfToCorePortProfiles != newIntfToCorePortProfiles or \
         oldIntfToVxlanPortProfiles != newIntfToVxlanPortProfiles:
      return False
   if set( oldConfig.collectors ) != set( newConfig.collectors ):
      return False
   # TODO: when collector log is supported uncomment the below
   # if oldConfig.collectors[ 'DefaultLog' ].maxPktsToBuffer != \
   #   newConfig.collectors[ 'DefaultLog' ].maxPktsToBuffer:
   #   return False
   return True

def isDefaultTelemetryConfig( config ):
   if not isDefaultIntParam( config.intParam ):
      return False
   if list( config.corePortProfiles ) != [ 'default' ]:
      return False
   if list( config.edgePortProfiles ) != [ 'default' ]:
      return False
   if config.vxlanPortProfiles:
      return False
   if policiesCliConfig.pmapType.pmapInput:
      return False
   # TODO: when collector log is supported uncomment the below
   # if list( config.collectors ) != [ 'DefaultLog' ]:
   #   return False
   # if 'DefaultLog' in config.collectors and \
   #   config.collectors[ 'DefaultLog' ].maxPktsToBuffer != 64:
   #   return False
   if config.intfToCorePortProfiles or config.intfToEdgePortProfiles or\
         config.intfToVxlanPortProfiles:
      return False
   return True

def copyProfile( newProfile, oldProfile ):
   newProfile.collectorName = oldProfile.collectorName
   newProfile.collectorType = oldProfile.collectorType
   newProfile.sampleRate = oldProfile.sampleRate
   newProfile.samplePolicy = oldProfile.samplePolicy
   newProfile.truncate = oldProfile.truncate
   newProfile.truncationSize = oldProfile.truncationSize
   newProfile.egressDrop = oldProfile.egressDrop

def getPortProfileNames( profileType ):
   if profileType == 'core':
      return inbandTelemetryConfig.corePortProfiles
   elif profileType == 'edge':
      return inbandTelemetryConfig.edgePortProfiles
   else:
      return inbandTelemetryConfig.vxlanPortProfiles

def getFlowTrackerNames():
   intConfigCollectorNames = { dest.name for dest in
                               inbandTelemetryConfig.collectors.values()
                               if dest.type ==
                               tacCollectorType.CollectorFlowTracker }
   intFlowTrackingCollectorNames = \
                        { dest.name for dest in
                        intFlowTrackingCollectors.flowTrackerConfig.values() }
   flowTrackerNames = intConfigCollectorNames | intFlowTrackingCollectorNames

   return list( flowTrackerNames )

def defaultProfileGuard( mode, token ):
   if token == 'default':
      return CliParser.guardNotThisPlatform
   return None

def edgeProfileGuard( mode, token ):
   if mode.profileContext.profileType_ == tacPortProfileType.IntEdge:
      return None
   return CliParser.guardNotThisPlatform

def nonCoreProfileGuard( mode, token ):
   if mode.profileContext.profileType_ != tacPortProfileType.IntCore:
      return None
   return CliParser.guardNotThisPlatform

def telemetryGuard( mode, token ):
   if hwCapability.inbandTelemetrySupported:
      return None
   return CliParser.guardNotThisPlatform

# Edge ports are supported only on Non TH3/TH4 devices
def edgePortGuard( mode, token ):
   if not hwCapability.transitOnlyMode:
      return None
   return CliParser.guardNotThisPlatform

def profileFlowTrackerGuard( mode, token ):
   # Flow tracker not supported on TH3/TH4 platforms
   if hwCapability.transitOnlyMode:
      return CliParser.guardNotThisPlatform
   if mode.profileContext.profileType_ == tacPortProfileType.IntVxlan and\
         token == 'ingress':
      return None
   if mode.profileContext.profileType_ != tacPortProfileType.IntVxlan and\
         token == 'egress':
      return None
   return CliParser.guardNotThisPlatform

def ifa2Guard( mode, token ):
   if hwCapability.defaultIntVersion == intVersions.Version2:
      return None
   return CliParser.guardNotThisPlatform

def ifaNativeV1Guard( mode, token ):
   if hwCapability.defaultIntVersion == intVersions.VersionNativeV1:
      return None
   return CliParser.guardNotThisPlatform

def vxlanModeGuard( mode, token ):
   if hwCapability.vxlanModeSupported:
      return None
   return CliParser.guardNotThisPlatform

def getProfileTypeFromArgs( args ):
   if 'core' in args:
      profileType = args[ 'core' ]
   elif 'edge' in args:
      profileType = args[ 'edge' ]
   else:
      profileType = args[ 'vxlan' ]
   return profileType

def getProfileNamesFromContext( mode, context ):
   profileType = getProfileTypeFromArgs( context.sharedResult )
   return getPortProfileNames( profileType )

def handleIntMirrorSession( mode ):
   InbandTelemetryCliLib.maybeUpdateIntMirrorSession( mode )

def incrementConfigVersion( config ):
   if config.version == maxConfigVersion:
      config.version = 0
   else:
      config.version += 1

def collectionCliData():
   data = {
         'collection': 'Set collector configuration',
         'flow': 'Flow configuration commands',
         'tracker': 'Set collector as flow tracker',
         '<flow-tracker-name>': flowTrackerNameMatcher
         }
   return data

def collectionCliHandler( mode, args ):
   collectionName = args[ '<flow-tracker-name>' ]
   collectionType = tacCollectorType.CollectorFlowTracker
   profile = mode.profileContext.currentEntry_
   if collectionName not in inbandTelemetryConfig.collectors:
      inbandTelemetryConfig.collectors.newMember( collectionName, collectionType )
   profile.collectorName = collectionName
   profile.collectorType = collectionType

def collectionCliNoOrDefaultHandler( mode, args ):
   profile = mode.profileContext.currentEntry_
   if profile.collectorName == 'DefaultLog':
      return
   profile.collectorName = ""
   profile.collectorType = tacCollectorType.CollectorNone

coreProfileNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: getPortProfileNames( 'core' ),
   "Inband Telemetry Core Profile Name",
   pattern=r'(?!summary$)[A-Za-z0-9_:{}\[\]-]+' )

edgeProfileNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: getPortProfileNames( 'edge' ),
   "Inband Telemetry Edge Profile Name",
   pattern=r'(?!summary$)[A-Za-z0-9_:{}\[\]-]+' )

nonDefaultCoreProfileNameMatcher = CliCommand.Node( CliMatcher.DynamicNameMatcher(
   lambda mode: getPortProfileNames( 'core' ),
   "Inband Telemetry Core Profile Name",
   pattern=r'(?!summary$)[A-Za-z0-9_:{}\[\]-]+' ), guard=defaultProfileGuard )

nonDefaultEdgeProfileNameMatcher = CliCommand.Node( CliMatcher.DynamicNameMatcher(
   lambda mode: getPortProfileNames( 'edge' ),
   "Inband Telemetry Edge Profile Name",
   pattern=r'(?!summary$)[A-Za-z0-9_:{}\[\]-]+' ), guard=defaultProfileGuard )

vxlanProfileNameMatcher = CliCommand.Node( CliMatcher.DynamicNameMatcher(
   lambda mode: getPortProfileNames( 'vxlan' ),
   "Inband Telemetry Vxlan Profile Name",
   pattern=r'(?!summary$)[A-Za-z0-9_:{}\[\]-]+' ) )

flowTrackerNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: getFlowTrackerNames(),
   "Flow Tracker Name" )

profileNameMatcher = CliCommand.Node( matcher=CliMatcher.DynamicNameMatcher(
      getProfileNamesFromContext, "Profile name",
      passContext=True ), storeSharedResult=True )

coreKeyword = CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
   'core', helpdesc='core profile' ), storeSharedResult=True )

edgeKeyword = CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
   'edge', helpdesc='edge profile' ), storeSharedResult=True )

vxlanKeyword = CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
   'vxlan', helpdesc='vxlan profile' ), storeSharedResult=True,
   guard=vxlanModeGuard )

profileKeyword = CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
   'profile', helpdesc='Show inband telemetry profiles' ) )

probeMarkerMatcherUpper = CliMatcher.PatternMatcher(
   pattern='0x?[a-fA-F0-9]{1,8}', helpname='0x0-0xffffffff',
   helpdesc="Upper 32 bits of the probe marker" )

probeMarkerMatcherLower = CliMatcher.PatternMatcher(
   pattern='0x?[a-fA-F0-9]{1,8}', helpname='0x0-0xffffffff',
   helpdesc="Lower 32 bits of the probe marker" )

class IpProtocolMatcher( CliMatcher.IntegerMatcher ):
   '''Extends IntegerMatcher, but filters out reserved values'''
   def match( self, mode, context, token ):
      if token in IP_PROTOCOL_RESERVED:
         return CliMatcher.noMatch
      return super().match(
            mode, context, token )

#------------------------------------------------------------------------------------
# Modelet under Intf mode for inband telemetry command
#------------------------------------------------------------------------------------

class IntfTelemetryModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.name.startswith( ( 'Ethernet', 'Port-Channel' ) )\
            and not mode.intf.isSubIntf()

IntfCli.IntfConfigMode.addModelet( IntfTelemetryModelet )

#------------------------------------------------------------------------------------
# Modelet under vxlan Intf mode for inband telemetry command
#------------------------------------------------------------------------------------

class VxlanIntfTelemetryModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.name.startswith( 'Vxlan' )

if toggleFeatureInbandTelemetryVxlanEnabled():
   IntfCli.IntfConfigMode.addModelet( VxlanIntfTelemetryModelet )

#------------------------------------------------------------------------------------
# The "monitor telemetry inband" mode context
#------------------------------------------------------------------------------------

class InbandTelemetryContext:
   def __init__( self, mode ):
      self.mode = mode
      self.config = inbandTelemetryConfig
      self.currentEntry_ = None
      self.previousEntry_ = Tac.newInstance( 'InbandTelemetry::Config', "intConfig" )
      if not isDefaultIntParam( inbandTelemetryConfig.intParam ):
         intParam = Tac.newInstance( 'InbandTelemetry::InbandTelemetryParam' )
         copyIntParam( inbandTelemetryConfig.intParam, intParam )
         self.previousEntry_.intParam = intParam
      # TODO: when collector log is supported uncomment the below
      # dftLog = self.previousEntry_.collectors.newMember(
      #   'DefaultLog', tacCollectorType.CollectorLog )
      # dftLog.maxPktsToBuffer = inbandTelemetryConfig.collectors[
      #   'DefaultLog' ].maxPktsToBuffer
      self.previousEntry_.enable = inbandTelemetryConfig.enable

   def copyProfileContents( self, dstProfile, srcProfile ):
      dstProfile.egressDrop = srcProfile.egressDrop
      dstProfile.sampleRate = srcProfile.sampleRate
      dstProfile.samplePolicy = srcProfile.samplePolicy
      dstProfile.collectorName = srcProfile.collectorName
      dstProfile.collectorType = srcProfile.collectorType

   def copyEditEntry( self ):
      newEntry = Tac.newInstance( 'InbandTelemetry::Config', "intConfig" )
      intParam = Tac.newInstance( 'InbandTelemetry::InbandTelemetryParam' )
      if not isDefaultIntParam( self.config.intParam ):
         copyIntParam( self.config.intParam, intParam )
      newEntry.intParam = intParam
      for collector in self.config.collectors:
         collectorType = self.config.collectors[ collector ].type
         newEntry.collectors.newMember( collector, collectorType )
         newEntry.collectors[ collector ].maxPktsToBuffer = self.config.collectors[
            collector ].maxPktsToBuffer
      for coreProfile in self.config.corePortProfiles:
         newEntry.corePortProfiles.newMember( coreProfile,
                                              tacPortProfileType.IntCore )
         newProfile = newEntry.corePortProfiles[ coreProfile ]
         self.copyProfileContents( newProfile,
               self.config.corePortProfiles[ coreProfile ] )
      for edgeProfile in self.config.edgePortProfiles:
         newEntry.edgePortProfiles.newMember( edgeProfile,
                                              tacPortProfileType.IntEdge )
         newProfile = newEntry.edgePortProfiles[ edgeProfile ]
         self.copyProfileContents( newProfile,
               self.config.edgePortProfiles[ edgeProfile ] )
      for vxlanProfile in self.config.vxlanPortProfiles:
         newEntry.vxlanPortProfiles.newMember( vxlanProfile,
               tacPortProfileType.IntVxlan )
         newProfile = newEntry.vxlanPortProfiles[ vxlanProfile ]
         self.copyProfileContents( newProfile,
               self.config.vxlanPortProfiles[ vxlanProfile ] )

      for intf, profile in self.config.intfToCorePortProfiles.items():
         newEntry.intfToCorePortProfiles[ intf ] = profile
      for intf, profile in self.config.intfToEdgePortProfiles.items():
         newEntry.intfToEdgePortProfiles[ intf ] = profile
      for intf, profile in self.config.intfToVxlanPortProfiles.items():
         newEntry.intfToVxlanPortProfiles[ intf ] = profile
      newEntry.enable = self.config.enable
      self.currentEntry_ = newEntry
      return newEntry

   def currentEntry( self ):
      return self.currentEntry_

   def commit( self ):
      if self.currentEntry_:
         if identicalTelemetryConfig( self.config, self.currentEntry_ ):
            # no need to commit anything from currentEntry_ as it has not changed
            return
         if not identicalIntParam( self.currentEntry_.intParam,
                                   self.config.intParam ):
            if not self.currentEntry_.intParam.probeProtocol == \
                  self.config.intParam.probeProtocol:
               # increment version of all the portProfiles to signify a change in
               # probeProtocol and a need to refresh TCAM entries
               for profile in self.config.corePortProfiles:
                  self.config.corePortProfiles[ profile ].version += 1
               for profile in self.config.edgePortProfiles:
                  self.config.edgePortProfiles[ profile ].version += 1
            intParam = Tac.newInstance( 'InbandTelemetry::InbandTelemetryParam' )
            copyIntParam( self.currentEntry_.intParam, intParam )
            self.config.intParam = intParam
         # TODO: when collector log is supported uncomment the below
         # self.config.collectors[ 'DefaultLog' ].\
         #   maxPktsToBuffer = self.currentEntry_.collectors[
         #      'DefaultLog' ].maxPktsToBuffer

         # Remove collectors that are no longer used
         collectorsNotInUse = []
         for collector in self.config.collectors:
            collectorInUse = False
            # TODO: when collector log is supported uncomment the below
            # if collector == 'DefaultLog':
            #   continue
            for coreProfile in self.config.corePortProfiles:
               if collector == self.config.corePortProfiles[
                     coreProfile ].collectorName:
                  collectorInUse = True
                  break
            for edgeProfile in self.config.edgePortProfiles:
               if collector == self.config.edgePortProfiles[
                     edgeProfile ].collectorName:
                  collectorInUse = True
                  break
            for vxlanProfile in self.config.vxlanPortProfiles:
               if collector == self.config.vxlanPortProfiles[
                     vxlanProfile ].collectorName:
                  collectorInUse = True
                  break
            if not collectorInUse:
               collectorsNotInUse.append( collector )

         for collector in collectorsNotInUse:
            del self.config.collectors[ collector ]

         # Set enable upon successfully committing from inband telemetry mode's
         # context. If config.enable was already false and currentEntry_.enable
         # is also false, no need to increment version and handle int mirror session
         versionIncNeeded = self.config.enable or self.currentEntry_.enable
         self.config.enable = self.currentEntry_.enable
         if versionIncNeeded:
            incrementConfigVersion( self.config )
            # Attempt to create an internal inband telemetry mirror
            # session when the feature is enabled.
            # This must be gated through HW capabilities for future
            # implementations that don't require such an operation.
            handleIntMirrorSession( self.mode )

#------------------------------------------------------------------------------------
# The "monitor telemetry MODE" mode command
#------------------------------------------------------------------------------------

class MonitorTelemetryInband( CliCommand.CliCommandClass ):
   syntax = '''monitor telemetry inband'''
   noOrDefaultSyntax = syntax
   data = {
      'monitor': CliToken.Monitor.monitorMatcher,
      'telemetry': 'Telemetry configuration',
      'inband': CliCommand.guardedKeyword( 'inband',
                                           helpdesc='Inband telemetry',
                                           guard=telemetryGuard ),
   }

   @staticmethod
   def handler( mode, args ):
      context = InbandTelemetryContext( mode )
      context.copyEditEntry()
      childMode = mode.childMode( InbandTelemetryConfigMode,
                                  context=context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      for name in policiesCliConfig.pmapType.pmap:
         # pylint: disable=protected-access
         IntSamplePolicyConfigCmd._removePolicy( mode, name )
      coreProfiles = inbandTelemetryConfig.corePortProfiles
      for profile in coreProfiles:
         if profile != 'default':
            del inbandTelemetryConfig.corePortProfiles[ profile ]
      edgeProfiles = inbandTelemetryConfig.edgePortProfiles
      for profile in edgeProfiles:
         if profile != 'default':
            del inbandTelemetryConfig.edgePortProfiles[ profile ]
      inbandTelemetryConfig.vxlanPortProfiles.clear()
      # TODO: when collector log is supported uncomment the below
      #inbandTelemetryConfig.collectors[ 'DefaultLog' ].maxPktsToBuffer = 64
      # TODO: when collector log is supported uncomment the below
      # if collector != 'DefaultLog':
      #   del inbandTelemetryConfig.collectors[ collector ]
      inbandTelemetryConfig.collectors.clear()
      # Set version to 0, enable to False and reset int param when setting
      # inbandTelemetryConfig to default
      intParam = Tac.newInstance( 'InbandTelemetry::InbandTelemetryParam' )
      inbandTelemetryConfig.intParam = intParam
      if inbandTelemetryConfig.enable:
         inbandTelemetryConfig.enable = False
         incrementConfigVersion( inbandTelemetryConfig )
         # Delete the internal inband telemetry mirror
         # session when the feature is disabled.
         # This must be gated through HW capabilities for future
         # implementations that don't require such an operation.
         handleIntMirrorSession( mode )

#--------------------------------------------------------------------------------
# [ no | default ] disabled
#--------------------------------------------------------------------------------
class DisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'disabled'
   noOrDefaultSyntax = syntax
   data = {
      'disabled': 'Disable inband telemetry feature'
   }

   @staticmethod
   def handler( mode, args ):
      mode.telemetryContext.currentEntry_.enable = False

   defaultHandler = handler

   @staticmethod
   def noHandler( mode, args ):
      currentEntry = mode.telemetryContext.currentEntry_
      currentEntry.enable = True

def getSamplePolicyNames( mode ):
   if not policiesCliConfig:
      return []
   return policiesCliConfig.pmapType.pmapInput

samplePolicyNameMatcher = CliMatcher.DynamicNameMatcher(
   getSamplePolicyNames, "Sample Policy Name",
   pattern=r'[A-Za-z0-9_:{}\[\]-]+' )


#[ no ] sample policy POLICY_NAME
class IntSamplePolicyConfigCmd( SamplePolicyConfigCmd ):
   data = {
      'POLICY_NAME': samplePolicyNameMatcher
   }
   data.update( SamplePolicyConfigCmd._baseData )

   @staticmethod
   def _feature():
      return FEATURE

   @classmethod
   def _context( cls, name, mode ):
      return IntSamplePolicyContext( policiesCliConfig, policiesStatusRequestDir,
                                     policiesStatus, name )

#[ no ]  match  <match-rule-name> ipv4|ipv6
class IntSamplePolicyMatchRuleConfigCmd( MatchRuleConfigCmdBase ):
   syntax = 'match RULE_NAME ( ipv4 | ipv6 )'
   noOrDefaultSyntax = 'match RULE_NAME [ ipv4 | ipv6 ]'
   data = {
      'match': 'Configure match rule',
      'RULE_NAME': matchRuleName,
      'ipv4': 'IPv4 match rule',
      'ipv6': 'IPv6 match rule',
   }

   @staticmethod
   def _feature():
      return FEATURE

   @staticmethod
   def _context( trafficPolicyContext, ruleName, matchOption ):
      return IntSamplePolicyMatchRuleContext( trafficPolicyContext, ruleName,
                                              matchOption )

class IntSamplePolicyActionsConfigCmd( ActionsConfigCmdBase ):
   data = ActionsConfigCmdBase.data.copy()

   @staticmethod
   def _feature():
      return FEATURE


def deviceIdRange( mode, context ):
   IntV2 = Tac.Type( "InbandTelemetry::IntVersion" ).Version2
   if hwCapability and hwCapability.defaultIntVersion == IntV2:
      return ( 0x0, 0xfffff )
   else:
      return ( 0x0, 0xffffffff )

deviceIdMatcher = CliMatcher.DynamicIntegerMatcher(
                           helpdesc='Device ID value',
                           rangeFn=deviceIdRange )


class DeviceIdCmd( CliCommand.CliCommandClass ):
   syntax = '''device-id DEVICE_ID_VAL'''
   noOrDefaultSyntax = '''device-id ...'''
   data = {
      'device-id' : 'Set the device id for inband telemetry',
      'DEVICE_ID_VAL': deviceIdMatcher
   }

   @staticmethod
   def handler( mode, args ):
      currentEntry = mode.telemetryContext.currentEntry_
      deviceId = args[ 'DEVICE_ID_VAL' ]
      intParam = Tac.newInstance( 'InbandTelemetry::InbandTelemetryParam' )
      copyIntParam( currentEntry.intParam, intParam )
      # Currently intParam.deviceId is represented as string to accomodate
      # other type of deviceIds like interface name based deviceId.
      # Currently through CLI, we are only supporting integer based device-ids
      deviceIdStr = f'0x{deviceId:x}'
      intParam.deviceId = deviceIdStr
      if not identicalIntParam( currentEntry.intParam, intParam ):
         currentEntry.intParam = intParam

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      currentEntry = mode.telemetryContext.currentEntry_
      if not currentEntry.intParam:
         return
      intParam = Tac.newInstance( 'InbandTelemetry::InbandTelemetryParam' )
      copyIntParam( currentEntry.intParam, intParam )
      intParam.deviceId = '0x0'
      if not identicalIntParam( currentEntry.intParam, intParam ):
         currentEntry.intParam = intParam

class ProbeMarkerCmd( CliCommand.CliCommandClass ):
   syntax = '''probe marker <markerA> <markerB>'''
   noOrDefaultSyntax = '''probe marker ...'''
   data = {
      'probe' : 'Configure probe marker for inband telemetry',
      'marker': 'Set the probe marker value',
      '<markerA>' : probeMarkerMatcherUpper,
      '<markerB>' : probeMarkerMatcherLower
      }

   @staticmethod
   def handler( mode, args ):
      if hwCapability.defaultIntVersion == intVersions.Version2:
         mode.addError( "'probe marker' command is not supported on this platform" )
         return
      markerA = args[ '<markerA>' ]
      markerB = args[ '<markerB>' ]
      currentEntry = mode.telemetryContext.currentEntry_
      probeMarker = ( int( markerA, 16 ) << 32 ) + \
                    int( markerB, 16 )
      intParam = Tac.newInstance( 'InbandTelemetry::InbandTelemetryParam' )
      copyIntParam( currentEntry.intParam, intParam )
      intParam.probeMarker = probeMarker
      intParam.probeMarkerDisabled = False
      if not identicalIntParam( currentEntry.intParam, intParam ):
         currentEntry.intParam = intParam

   @staticmethod
   def noHandler( mode, args ):
      currentEntry = mode.telemetryContext.currentEntry_
      intParam = Tac.newInstance( 'InbandTelemetry::InbandTelemetryParam' )
      copyIntParam( currentEntry.intParam, intParam )
      intParam.probeMarker = 0
      intParam.probeMarkerDisabled = True
      if not identicalIntParam( currentEntry.intParam, intParam ):
         currentEntry.intParam = intParam

   @staticmethod
   def defaultHandler( mode, args ):
      currentEntry = mode.telemetryContext.currentEntry_
      if not currentEntry.intParam:
         return
      intParam = Tac.newInstance( 'InbandTelemetry::InbandTelemetryParam' )
      copyIntParam( currentEntry.intParam, intParam )
      intParam.probeMarker = intParam.defaultProbeMarker
      intParam.probeMarkerDisabled = False
      if not identicalIntParam( currentEntry.intParam, intParam ):
         currentEntry.intParam = intParam

#--------------------------------------------------------------------------------
# [ no | default ] probe marker ip protocol PROTOCOL_VAL
#--------------------------------------------------------------------------------
class ProbeProtocolCmd( CliCommand.CliCommandClass ):
   syntax = '''probe marker ip protocol PROTOCOL_VAL'''
   noOrDefaultSyntax = '''probe marker ip protocol ...'''
   data = {
      'probe': 'Configure probe marker for inband telemetry',
      'marker': 'Set the probe marker value',
      'ip': CliCommand.guardedKeyword( 'ip',
         helpdesc='Configure IP header to probe InbandTelemetry packets',
         guard=ifa2Guard ),
      'protocol': 'Set IP protocol/next-header field to probe '
            'InbandTelemetry packets',
      'PROTOCOL_VAL': IpProtocolMatcher( 0, 254,
         helpdesc='0-254 except 6 ( reserved for TCP ) and 17 ( reserved for UDP )' )
   }

   @staticmethod
   def handler( mode, args ):
      currentEntry = mode.telemetryContext.currentEntry_
      probeProtocol = args[ 'PROTOCOL_VAL' ]
      intParam = Tac.newInstance( 'InbandTelemetry::InbandTelemetryParam' )
      copyIntParam( currentEntry.intParam, intParam )
      intParam.probeProtocol = probeProtocol
      if not identicalIntParam( currentEntry.intParam, intParam ):
         currentEntry.intParam = intParam

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      currentEntry = mode.telemetryContext.currentEntry_
      if not currentEntry.intParam:
         return
      intParam = Tac.newInstance( 'InbandTelemetry::InbandTelemetryParam' )
      copyIntParam( currentEntry.intParam, intParam )
      intParam.probeProtocol = intParam.defaultProbeProtocol
      if not identicalIntParam( currentEntry.intParam, intParam ):
         currentEntry.intParam = intParam

class ProbeMarkerUdpCmd( CliCommand.CliCommandClass ):
   syntax = 'probe marker udp destination port { PORTS }'
   noOrDefaultSyntax = 'probe marker udp destination port ...'
   data = {
      'probe': 'Configure probe marker for inband telemetry',
      'marker': 'Set the probe marker value',
      'udp': CliCommand.guardedKeyword( 'udp',
         helpdesc="Set the probe packet's UDP parameters",
         guard=ifaNativeV1Guard ),
      'destination': "Set the probe packet's UDP destination port",
      'port': "Set the probe packet's UDP destination port",
      'PORTS': CliCommand.Node( CliMatcher.IntegerMatcher( 1, 65535,
                                helpdesc='Destination port value' ),
                                maxMatches=2 )
   }

   @staticmethod
   def handler( mode, args ):
      currentEntry = mode.telemetryContext.currentEntry_
      intParam = Tac.newInstance( 'InbandTelemetry::InbandTelemetryParam' )
      copyIntParam( currentEntry.intParam, intParam )
      ports = args[ 'PORTS' ]
      intParam.udpDstPort1 = ports[ 0 ]
      intParam.udpDstPort1Enabled = True
      if len( ports ) > 1:
         intParam.udpDstPort2 = ports[ 1 ]
         intParam.udpDstPort2Enabled = True

      else:
         intParam.udpDstPort2 = intParam.defaultUdpDstPort
         intParam.udpDstPort2Enabled = False
      if not identicalIntParam( currentEntry.intParam, intParam ):
         currentEntry.intParam = intParam

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      currentEntry = mode.telemetryContext.currentEntry_
      if not currentEntry.intParam:
         return
      intParam = Tac.newInstance( 'InbandTelemetry::InbandTelemetryParam' )
      copyIntParam( currentEntry.intParam, intParam )
      intParam.udpDstPort1 = intParam.defaultUdpDstPort
      intParam.udpDstPort1Enabled = False
      intParam.udpDstPort2 = intParam.defaultUdpDstPort
      intParam.udpDstPort2Enabled = False
      if not identicalIntParam( currentEntry.intParam, intParam ):
         currentEntry.intParam = intParam

class EgressCollectionLogNumCmd( CliCommand.CliCommandClass ):
   syntax = '''egress collection log <num>'''
   noOrDefaultSyntax = '''egress collection log ...'''
   data = {
      'egress': CliCommand.guardedKeyword(
         'egress', helpdesc='Configure egress parameters',
         guard=profileFlowTrackerGuard ),
      'collection' : 'Set collector configuration',
      'log' : 'Set collector as log',
      '<num>' : CliMatcher.IntegerMatcher( minLogNum, maxLogNum,
                                           helpdesc='Max packets to buffer' )
      }

   @staticmethod
   def handler( mode, args ):
      currentEntry = mode.telemetryContext.currentEntry_
      collectorCfg = currentEntry.collectors[ 'DefaultLog' ]
      maxPktsToBuffer = args[ '<num>' ]
      collectorCfg.maxPktsToBuffer = maxPktsToBuffer

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      currentEntry = mode.telemetryContext.currentEntry_
      collectorCfg = currentEntry.collectors[ 'DefaultLog' ]
      collectorCfg.maxPktsToBuffer = tacCollectorLogBufferSize.minSize

#------------------------------------------------------------------------------------
# The profile  mode context
#------------------------------------------------------------------------------------

class ProfileModeContext:
   def __init__( self, mode, profileName, profileType ):
      self.mode = mode
      self.profile_ = None
      self.profileName_ = profileName
      self.profileType_ = profileType
      self.currentEntry_ = None
      self.previousEntry_ = None
      if profileType == tacPortProfileType.IntEdge:
         portProfileConfig = inbandTelemetryConfig.edgePortProfiles
      elif profileType == tacPortProfileType.IntCore:
         portProfileConfig = inbandTelemetryConfig.corePortProfiles
      else:
         portProfileConfig = inbandTelemetryConfig.vxlanPortProfiles
      if profileName in portProfileConfig:
         self.profile_ = portProfileConfig[ profileName ]
         prevProfile = Tac.newInstance( 'InbandTelemetry::PortProfileConfig',
                                        profileName, profileType )
         copyProfile( prevProfile, self.profile_ )
         self.previousEntry_ = prevProfile

   def copyEditEntry( self ):
      newProfile = Tac.newInstance( 'InbandTelemetry::PortProfileConfig',
                                    self.profileName_, self.profileType_ )
      copyProfile( newProfile, self.profile_ )
      self.currentEntry_ = newProfile
      return newProfile

   def newEditEntry( self ):
      newProfile = Tac.newInstance( 'InbandTelemetry::PortProfileConfig',
                                    self.profileName_, self.profileType_ )
      self.currentEntry_ = newProfile

   def profileName( self ):
      return self.profileName_

   def profileType( self ):
      return self.profileType_

   def currentEntry( self ):
      return self.currentEntry_

   def commit( self ):
      chkr = PolicyOpChkr( policiesStatusRequestDir, policiesStatus )
      if identicalProfile( self.profile_, self.currentEntry_ ):
         return
      if self.profile_ is None:
         if self.profileType_ == tacPortProfileType.IntEdge:
            self.profile_ = inbandTelemetryConfig.edgePortProfiles.newMember(
               self.profileName_, self.profileType_ )
         elif self.profileType_ == tacPortProfileType.IntCore:
            self.profile_ = inbandTelemetryConfig.corePortProfiles.newMember(
               self.profileName_, self.profileType_ )
         else:
            self.profile_ = inbandTelemetryConfig.vxlanPortProfiles.newMember(
               self.profileName_, self.profileType_ )
      if self.profileType_ == tacPortProfileType.IntEdge:
         for intf in inbandTelemetryConfig.intfToEdgePortProfiles:
            if inbandTelemetryConfig.intfToEdgePortProfiles[ intf ] != \
               self.profileName_:
               continue
            result = chkr.verify( 'intf', intf )
            commitStatus = result[ 0 ]
            if not commitStatus:
               reason = result[ 1 ].error if result[ 1 ] else 'unknown'
               self.mode.addError( 'Failed to commit edge profile : %s' %
                                   reason )
      self.profile_.collectorName = self.currentEntry_.collectorName
      self.profile_.collectorType = self.currentEntry_.collectorType
      self.profile_.sampleRate = self.currentEntry_.sampleRate
      self.profile_.samplePolicy = self.currentEntry_.samplePolicy
      self.profile_.truncate = self.currentEntry_.truncate
      self.profile_.truncationSize = self.currentEntry_.truncationSize
      self.profile_.egressDrop = self.currentEntry_.egressDrop
      self.profile_.version += 1


#------------------------------------------------------------------------------------
# The "profile <profile-type> <profile-name>" mode command
#------------------------------------------------------------------------------------
class ProfileConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'profile ( ( core CORE_PROFILE ) | ( edge EDGE_PROFILE ) '
   if toggleFeatureInbandTelemetryVxlanEnabled():
      syntax += '| ( vxlan VXLAN_PROFILE ) )'
   else:
      syntax += ')'
   noOrDefaultSyntax = syntax
   data = {
      'profile' : 'Configure inband telemetry profile',
      'core' : 'Configure core profile',
      'CORE_PROFILE': nonDefaultCoreProfileNameMatcher,
      'edge': CliCommand.guardedKeyword( 'edge',
                                          helpdesc='Configure edge profile',
                                          guard=edgePortGuard ),
      'EDGE_PROFILE': nonDefaultEdgeProfileNameMatcher,
   }
   if toggleFeatureInbandTelemetryVxlanEnabled():
      data[ 'vxlan' ] = CliCommand.guardedKeyword( 'vxlan',
            helpdesc='Configure vxlan profile',
            guard=vxlanModeGuard )
      data[ 'VXLAN_PROFILE' ] = vxlanProfileNameMatcher

   @staticmethod
   def handler( mode, args ):
      if "edge" in args:
         profileType = tacPortProfileType.IntEdge
         profileName = args[ 'EDGE_PROFILE' ]
      elif "core" in args:
         profileType = tacPortProfileType.IntCore
         profileName = args[ 'CORE_PROFILE' ]
      else:
         profileType = tacPortProfileType.IntVxlan
         profileName = args[ 'VXLAN_PROFILE' ]
      context = ProfileModeContext( mode, profileName, profileType )

      if ( profileType == tacPortProfileType.IntEdge and
            profileName in inbandTelemetryConfig.edgePortProfiles ) or \
         ( profileType == tacPortProfileType.IntCore and
            profileName in inbandTelemetryConfig.corePortProfiles ) or \
         ( profileType == tacPortProfileType.IntVxlan and
            profileName in inbandTelemetryConfig.vxlanPortProfiles ):
         context.copyEditEntry()
      else:
         context.newEditEntry()
      childMode = mode.childMode( ProfileConfigMode,
                                  context=context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if "edge" in args:
         profileName = args[ 'EDGE_PROFILE' ]
         if profileName in inbandTelemetryConfig.edgePortProfiles:
            del inbandTelemetryConfig.edgePortProfiles[ profileName ]
      elif "core" in args:
         profileName = args[ 'CORE_PROFILE' ]
         if profileName in inbandTelemetryConfig.corePortProfiles:
            del inbandTelemetryConfig.corePortProfiles[ profileName ]
      else:
         profileName = args[ 'VXLAN_PROFILE' ]
         if profileName in inbandTelemetryConfig.vxlanPortProfiles:
            del inbandTelemetryConfig.vxlanPortProfiles[ profileName ]

class SampleRateCmd( CliCommand.CliCommandClass ):
   syntax = '''ingress sample rate <rate-value>'''
   noOrDefaultSyntax = '''ingress sample rate ...'''
   data = {
      'ingress': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher(
            'ingress', helpdesc='Configure ingress parameters' ),
         guard=nonCoreProfileGuard ),
      'sample'  : 'Configure sampling parameters',
      'rate': 'Configure sampling rate',
      '<rate-value>' :
                 CliMatcher.IntegerMatcher( minSampleRate, maxSampleRate,
                    helpdesc='Sampling Rate Value' )
   }

   @staticmethod
   def handler( mode, args ):
      sampleRate = args[ '<rate-value>' ]
      profile = mode.profileContext.currentEntry_
      profile.sampleRate = sampleRate

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profile = mode.profileContext.currentEntry_
      profile.sampleRate = defSampleRate

class EgressCollectionLogCmd( CliCommand.CliCommandClass ):
   syntax = '''egress collection log'''
   noOrDefaultSyntax = syntax
   data = {
      'egress': CliCommand.guardedKeyword(
         'egress', helpdesc='Configure egress parameters',
         guard=profileFlowTrackerGuard ),
      'collection' : 'Set collector configuration',
      'log' : 'Set collector as log',
      }

   @staticmethod
   def handler( mode, args ):
      collectionName = 'DefaultLog'
      profile = mode.profileContext.currentEntry_
      profile.collectorName = collectionName
      profile.collectorType = tacCollectorType.CollectorLog

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profile = mode.profileContext.currentEntry_
      if profile.collectorName != 'DefaultLog':
         return
      profile.collectorName = ""
      profile.collectorType = tacCollectorType.CollectorNone

class IngressSamplePolicyCmd( CliCommand.CliCommandClass ):
   syntax = '''ingress sample policy POLICY_NAME'''
   noOrDefaultSyntax = '''ingress sample policy [ POLICY_NAME ]'''
   data = {
      'ingress': CliCommand.guardedKeyword(
         'ingress', helpdesc='Configure ingress parameters',
         guard=edgeProfileGuard ),
      'sample': 'Configure sampling parameters',
      'policy': 'Configure sample policy',
      'POLICY_NAME': samplePolicyNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      policyName = args[ 'POLICY_NAME' ]
      profile = mode.profileContext.currentEntry_
      profile.samplePolicy = policyName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profile = mode.profileContext.currentEntry_
      policyName = args.get( 'POLICY_NAME' )
      if not policyName or policyName == profile.samplePolicy:
         profile.samplePolicy = ""

class EgressCollectionFlowTrackerCmd( CliCommand.CliCommandClass ):
   syntax = '''egress collection flow tracker <flow-tracker-name>'''
   noOrDefaultSyntax = '''egress collection flow tracker ...'''
   data = collectionCliData()
   data[ 'egress' ] = CliCommand.guardedKeyword(
         'egress', helpdesc='Configure egress parameters',
         guard=profileFlowTrackerGuard )
   handler = collectionCliHandler
   noOrDefaultHandler = collectionCliNoOrDefaultHandler

class EgressDropDisabledCmd( CliCommand.CliCommandClass ):
   syntax = '''egress drop disabled'''
   noOrDefaultSyntax = syntax
   data = {
      'egress': CliCommand.guardedKeyword(
         'egress', helpdesc='Configure egress parameters',
         guard=profileFlowTrackerGuard ),
      'drop': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher(
            'drop', helpdesc='Set egress drop configuration' ),
         guard=edgeProfileGuard ),
      'disabled': 'Disable egress drop',
      }

   @staticmethod
   def handler( mode, args ):
      profile = mode.profileContext.currentEntry_
      profile.egressDrop = False

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profile = mode.profileContext.currentEntry_
      profile.egressDrop = True

class IngressCollectionFlowTrackerCmd( CliCommand.CliCommandClass ):
   syntax = '''ingress collection flow tracker <flow-tracker-name>'''
   noOrDefaultSyntax = '''ingress collection flow tracker ...'''
   data = collectionCliData()
   data[ 'ingress' ] = CliCommand.guardedKeyword(
         'ingress', helpdesc='Configure ingress parameters',
         guard=profileFlowTrackerGuard )
   handler = collectionCliHandler
   noOrDefaultHandler = collectionCliNoOrDefaultHandler

class InbandTelemetryInterface( CliCommand.CliCommandClass ):
   syntax = '''telemetry inband profile ( ( core [ CORE_PROFILE ] ) |'''  \
            ''' ( edge [ EDGE_PROFILE ] ) )'''
   noOrDefaultSyntax = syntax
   data = {
      'inband': CliCommand.guardedKeyword( 'inband',
                                           helpdesc='Apply inband variant',
                                           guard=telemetryGuard ),
      'telemetry' : 'Apply telemetry feature',
      'profile' : 'Apply inband telemetry profile',
      'core': 'Apply core profile',
      'CORE_PROFILE' : coreProfileNameMatcher,
      'edge': CliCommand.guardedKeyword( 'edge',
                                          helpdesc='Configure edge profile',
                                          guard=edgePortGuard ),
      'EDGE_PROFILE' : edgeProfileNameMatcher
      }

   @staticmethod
   def handler( mode, args ):
      intfName = mode.intf.name
      newProfileVersion = 0
      oldProfileVersion = 0
      if "edge" in args:
         profileType = 'edge'
         profileName = args[ 'EDGE_PROFILE' ] if 'EDGE_PROFILE' in args \
                       else 'default'
         if profileName in inbandTelemetryConfig.edgePortProfiles:
            newProfileVersion = \
                  inbandTelemetryConfig.edgePortProfiles[ profileName ].version
      else:
         profileType = 'core'
         profileName = args[ 'CORE_PROFILE' ] if 'CORE_PROFILE' in args \
                       else 'default'
         if profileName in inbandTelemetryConfig.corePortProfiles:
            newProfileVersion = \
                  inbandTelemetryConfig.corePortProfiles[ profileName ].version

      if profileType == 'edge':
         intfToPortProfiles = inbandTelemetryConfig.intfToEdgePortProfiles
         if intfName in inbandTelemetryConfig.intfToCorePortProfiles:
            oldProfile = inbandTelemetryConfig.intfToCorePortProfiles[ intfName ]
            del inbandTelemetryConfig.intfToCorePortProfiles[ intfName ]
            if oldProfile in inbandTelemetryConfig.corePortProfiles:
               oldProfileVersion =\
                     inbandTelemetryConfig.corePortProfiles[ oldProfile ].version
               # Increment the leaving profile version(oldProfileVersion) by 1
               inbandTelemetryConfig.corePortProfiles[ oldProfile ].version += 1
      else:
         intfToPortProfiles = inbandTelemetryConfig.intfToCorePortProfiles
         if intfName in inbandTelemetryConfig.intfToEdgePortProfiles:
            oldProfile = inbandTelemetryConfig.intfToEdgePortProfiles[ intfName ]
            del inbandTelemetryConfig.intfToEdgePortProfiles[ intfName ]
            if oldProfile in inbandTelemetryConfig.edgePortProfiles:
               oldProfileVersion = \
                     inbandTelemetryConfig.edgePortProfiles[ oldProfile ].version
               # Increment the leaving profile version(oldProfileVersion) by 1
               inbandTelemetryConfig.edgePortProfiles[ oldProfile ].version += 1
      if intfName in intfToPortProfiles:
         if intfToPortProfiles[ intfName ] == profileName:
            return
         else:
            # Remove the profile before adding a new profile to an interface
            oldProfile = intfToPortProfiles[ intfName ]
            del intfToPortProfiles[ intfName ]
            if profileType == 'edge':
               if oldProfile in inbandTelemetryConfig.edgePortProfiles:
                  oldProfileVersion =\
                        inbandTelemetryConfig.edgePortProfiles[ oldProfile ].version
                  inbandTelemetryConfig.edgePortProfiles[ oldProfile ].version += 1
            else:
               if oldProfile in inbandTelemetryConfig.corePortProfiles:
                  oldProfileVersion =\
                        inbandTelemetryConfig.corePortProfiles[ oldProfile ].version
                  inbandTelemetryConfig.corePortProfiles[ oldProfile ].version += 1

      # Add interface to newProfile and increment version of newProfile.
      # max() is used to ensure a change in version
      intfToPortProfiles[ intfName ] = profileName
      newProfileVersion = max( newProfileVersion, oldProfileVersion ) + 1
      if profileType == 'edge' and profileName in\
            inbandTelemetryConfig.edgePortProfiles:
         inbandTelemetryConfig.edgePortProfiles[ profileName ].version =\
               newProfileVersion
      elif profileType == 'core' and profileName\
            in inbandTelemetryConfig.corePortProfiles:
         inbandTelemetryConfig.corePortProfiles[ profileName ].version\
               = newProfileVersion

      if inbandTelemetryConfig.enable:
         incrementConfigVersion( inbandTelemetryConfig )
      # Attempt to create an internal inband telemetry mirror
      # session when an interface is associated with an edge
      # profile when such an association does not already exist.
      # This must be gated through HW capabilities for future
      # implementations that don't require such an operation.
      handleIntMirrorSession( mode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'CORE_PROFILE' in args:
         profileName = args[ 'CORE_PROFILE' ]
      elif 'EDGE_PROFILE' in args:
         profileName = args[ 'EDGE_PROFILE' ]
      else:
         profileName = None
      if "edge" in args:
         profileType = 'edge'
      else:
         profileType = 'core'
      intfName = mode.intf.name
      if profileType == 'edge':
         intfToPortProfiles = inbandTelemetryConfig.intfToEdgePortProfiles
      else:
         intfToPortProfiles = inbandTelemetryConfig.intfToCorePortProfiles

      if intfName in intfToPortProfiles:
         if not profileName or ( intfToPortProfiles[ intfName ] == profileName ):
            if not profileName:
               profileName = intfToPortProfiles[ intfName ]

            del intfToPortProfiles[ intfName ]

            if profileType == 'edge':
               portProfiles = inbandTelemetryConfig.edgePortProfiles
            else:
               portProfiles = inbandTelemetryConfig.corePortProfiles

            if profileName in portProfiles:
               portProfiles[ profileName ].version += 1

            if inbandTelemetryConfig.enable:
               incrementConfigVersion( inbandTelemetryConfig )
            # Delete the internal inband telemetry mirror session
            # when no interface is associated with an edge profile.
            # This must be gated through HW capabilities for future
            # implementations that don't require such an operation.
            handleIntMirrorSession( mode )

class InbandTelemetryVxlanInterface( CliCommand.CliCommandClass ):
   syntax = '''telemetry inband profile vxlan VXLAN_PROFILE'''
   noOrDefaultSyntax = '''telemetry inband profile vxlan [ VXLAN_PROFILE ]'''
   data = {
      'inband': CliCommand.guardedKeyword( 'inband',
                                           helpdesc='Apply inband variant',
                                           guard=telemetryGuard ),
      'telemetry': 'Apply telemetry feature',
      'profile': 'Apply inband telemetry profile',
      'vxlan': CliCommand.guardedKeyword( 'vxlan',
         helpdesc='Apply vxlan profile', guard=vxlanModeGuard ),
      'VXLAN_PROFILE': vxlanProfileNameMatcher,
      }

   @staticmethod
   def handler( mode, args ):
      intfName = mode.intf.name
      profileName = args[ 'VXLAN_PROFILE' ]
      intfToPortProfiles = inbandTelemetryConfig.intfToVxlanPortProfiles
      if intfName in intfToPortProfiles:
         if intfToPortProfiles[ intfName ] == profileName:
            return
         else:
            # Remove intf from old profile
            del intfToPortProfiles[ intfName ]
      intfToPortProfiles[ intfName ] = profileName
      if inbandTelemetryConfig.enable:
         incrementConfigVersion( inbandTelemetryConfig )

      handleIntMirrorSession( mode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'VXLAN_PROFILE' in args:
         profileName = args[ 'VXLAN_PROFILE' ]
      else:
         profileName = None
      intfName = mode.intf.name
      intfToPortProfiles = inbandTelemetryConfig.intfToVxlanPortProfiles
      if intfName in intfToPortProfiles:
         if not profileName or ( intfToPortProfiles[ intfName ] == profileName ):
            if not profileName:
               profileName = intfToPortProfiles[ intfName ]
            del intfToPortProfiles[ intfName ]
            if inbandTelemetryConfig.enable:
               incrementConfigVersion( inbandTelemetryConfig )
            handleIntMirrorSession( mode )

#------------------------------------------------------------------------------------
# "show monitor telemetry inband profile [ ( edge | core | vxlan) [ NAME ] ]" summary
#------------------------------------------------------------------------------------
def getModelProfileNameToIntf( intfToPortProfiles, profileName ):
   profileNameToIntf = {}
   modelProfileNameToIntf = {}
   for intf, profName in sorted( intfToPortProfiles.items() ):
      if profName not in profileNameToIntf:
         profileNameToIntf.setdefault( profName, [] ).append( intf )
      else:
         profileNameToIntf[ profName ].append( intf )
   if profileName:
      reqList = []
      reqList = profileNameToIntf.get( profileName, [] )
      profileNameToIntf.clear()
      profileNameToIntf[ profileName ] = reqList
   if profileNameToIntf:
      for profName in sorted( profileNameToIntf ):
         modelProfileNameToIntf[ profName ] = IntfList(
               Intfs=profileNameToIntf[ profName ] )
   else:
      modelProfileNameToIntf = None
   return modelProfileNameToIntf

def nameInterfaceLinker( profileType, profileName ):
   modelProfileNameToIntfCore = None
   modelProfileNameToIntfEdge = None
   modelProfileNameToIntfVxlan = None
   if profileType is None or profileType == 'Core':
      modelProfileNameToIntfCore = getModelProfileNameToIntf( inbandTelemetryConfig.
            intfToCorePortProfiles, profileName )
   if profileType is None or profileType == 'Edge':
      modelProfileNameToIntfEdge = getModelProfileNameToIntf( inbandTelemetryConfig.
            intfToEdgePortProfiles, profileName )
   if profileType is None or profileType == 'Vxlan':
      modelProfileNameToIntfVxlan = getModelProfileNameToIntf( inbandTelemetryConfig.
            intfToVxlanPortProfiles, profileName )
   return modelProfileNameToIntfCore, modelProfileNameToIntfEdge,\
         modelProfileNameToIntfVxlan

def showMonitorTelemetryInbandSummary( mode, args ):
   profileNameToIntfCore = {}
   profileNameToIntfEdge = {}
   profileNameToIntfVxlan = {}
   profileType = None
   profileName = args.get( 'NAME' )
   if "core" in args:
      profileType = 'Core'
   elif "edge" in args:
      profileType = 'Edge'
   elif "vxlan" in args:
      profileType = 'Vxlan'

   profileNameToIntfCore, profileNameToIntfEdge, profileNameToIntfVxlan =\
         nameInterfaceLinker( profileType, profileName )
   return ModelIntProfileSummary(
            coreIntfList=profileNameToIntfCore, edgeIntfList=profileNameToIntfEdge,
            vxlanIntfList=profileNameToIntfVxlan )

class ShowMonitorTelemetryInbandSummary( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor telemetry inband profile [ ( core | edge '
   if toggleFeatureInbandTelemetryVxlanEnabled():
      syntax += '| vxlan ) [ NAME ] ] summary'
   else:
      syntax += ') [ NAME ] ] summary'
   data = {
         'monitor': CliToken.Monitor.monitorMatcherForShow,
         'telemetry': CliCommand.Node(
            matcher=CliMatcher.KeywordMatcher(
            'telemetry', helpdesc='Show monitor telemetry information' ),
            guard=telemetryGuard ),
         'inband': 'Show monitor inband telemetry',
         'profile': profileKeyword,
         'core': coreKeyword,
         'edge': edgeKeyword,
         'NAME': profileNameMatcher,
         'summary': 'Show inband telemetry profile summary',
         }
   if toggleFeatureInbandTelemetryVxlanEnabled():
      data[ 'vxlan' ] = vxlanKeyword
   cliModel = ModelIntProfileSummary
   handler = showMonitorTelemetryInbandSummary
   privileged = True

#------------------------------------------------------------------------------------
# The "show monitor telemetry inband profile [ ( edge | core | vxlan )
# [ NAME ] ]" command
#------------------------------------------------------------------------------------

def getEgressDrop( profileName ):
   egressDrop = 'enabled'
   portProfileConfig = inbandTelemetryConfig.edgePortProfiles
   if not portProfileConfig[ profileName ].egressDrop:
      egressDrop = 'disabled'
   return egressDrop

def getProfileStateAndReason( profileType, profileName ):
   if profileType == 'edge':
      profileStatus = inbandTelemetryStatus.edgePortProfileStatus
   elif profileType == 'core':
      profileStatus = inbandTelemetryStatus.corePortProfileStatus
   else:
      profileStatus = inbandTelemetryStatus.vxlanPortProfileStatus
   if profileName not in profileStatus:
      return 'unknown', ''
   portProfileState = profileStatus[ profileName ].portProfileOperStatus.state
   profileErrorReason = profileStatus[ profileName ].portProfileOperStatus.reason
   if portProfileState == 'operStateUnknown':
      profileState = 'unknown'
   elif portProfileState == 'operStateActive':
      profileState = 'active'
      profileErrorReason = ""
   else:
      profileState = 'inactive'
   return profileState, profileErrorReason

def getPortProfileConfig( profileType ):
   if profileType == 'edge':
      portProfileConfig = inbandTelemetryConfig.edgePortProfiles
   elif profileType == 'core':
      portProfileConfig = inbandTelemetryConfig.corePortProfiles
   else:
      portProfileConfig = inbandTelemetryConfig.vxlanPortProfiles
   return portProfileConfig

def getProfileModel( profileType, profileName ):
   portProfileConfig = getPortProfileConfig( profileType )
   collection = portProfileConfig[ profileName ].collectorName
   profileState, profileErrorReason = getProfileStateAndReason( profileType,
                                                                profileName )
   extraArgs = {}

   if profileType == 'edge':
      extraArgs[ "sampleRate" ] = inbandTelemetryConfig.\
            edgePortProfiles[ profileName ].sampleRate

      if toggleFeatureInbandTelemetrySamplePolicyEnabled():
         extraArgs[ "samplePolicy" ] = inbandTelemetryConfig.\
               edgePortProfiles[ profileName ].samplePolicy

      extraArgs[ "egressDrop" ] = getEgressDrop( profileName )

   if profileType == 'vxlan':
      extraArgs[ "sampleRate" ] = inbandTelemetryConfig.\
            vxlanPortProfiles[ profileName ].sampleRate

   return Profile( profileType=profileType,
                   egressCollection=collection,
                   profileStatus=profileState,
                   profileErrorReason=profileErrorReason,
                   **extraArgs )

def getProfiles( profileType, profileName ):
   config = {}
   portProfileConfig = getPortProfileConfig( profileType )
   if profileName == '':
      for portProfileName in portProfileConfig:
         config[ portProfileName ] = getProfileModel( profileType, portProfileName )
   else:
      if profileName in portProfileConfig:
         config[ profileName ] = getProfileModel( profileType, profileName )
   return config

def getPolicyModel( policyName ):
   rules = []
   pmapConfig = policiesCliConfig.pmapType.pmapInput[ policyName ]
   ruleNames = list( pmapConfig.currCfg.classPrio.values() )
   for ruleName in ruleNames:
      dscps = []
      rule = pmapConfig.currCfg.rawClassMap[ ruleName ]
      match = list( rule.match.items() )[ 0 ][ 1 ]
      matchOpt = "ipv4" if match.option == 'matchIpAccessGroup' else "ipv6"
      for dscpRange in match.structuredFilter.dscp:
         dscps.append( NumericalRange( low=dscpRange.rangeStart,
                                       high=dscpRange.rangeEnd ) )
      dscpList = sorted( dscps, key=lambda x: x.low )
      policyActions = pmapConfig.currCfg.classAction[ ruleName ].policyAction
      actions = Actions( sample='sample' in policyActions,
                         sampleAll='sampleAll' in policyActions )
      rules.append( Rule( ruleString=ruleName, matchOption=matchOpt,
                          matches=Matches( dscps=dscpList ), actions=actions ) )
   return SamplePolicyModel( rules=rules )

def getPolicies():
   config = {}
   for policyName in policiesCliConfig.pmapType.pmapInput:
      config[ policyName ] = getPolicyModel( policyName )
      return config

def showInbandTelemetryProfileHandler( mode, args ):
   if 'NAME' in args:
      profileName = args[ 'NAME' ]
   else:
      profileName = ''
   if 'edge' in args:
      profileType = 'edge'
   elif 'core' in args:
      profileType = 'core'
   elif 'vxlan' in args:
      profileType = 'vxlan'
   else:
      profileType = ''
   coreProfiles = None
   edgeProfiles = None
   vxlanProfiles = None
   if profileType == 'core':
      coreProfiles = getProfiles( profileType, profileName )
   elif profileType == 'edge':
      edgeProfiles = getProfiles( profileType, profileName )
   elif profileType == 'vxlan':
      vxlanProfiles = getProfiles( profileType, profileName )
   else:
      coreProfiles = getProfiles( 'core', profileName )
      edgeProfiles = getProfiles( 'edge', profileName )
      vxlanProfiles = getProfiles( 'vxlan', profileName )
   transitOnly = hwCapability.transitOnlyMode
   return InbandTelemetryProfiles( coreProfiles=coreProfiles,
                                   edgeProfiles=edgeProfiles,
                                   vxlanProfiles=vxlanProfiles,
                                   intVersion=hwCapability.defaultIntVersion,
                                   transitOnly=transitOnly )

class ShowInbandTelemetryProfile( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor telemetry inband profile [ ( edge | core '
   if toggleFeatureInbandTelemetryVxlanEnabled():
      syntax += '| vxlan ) [ NAME ] ]'
   else:
      syntax += ') [ NAME ] ]'

   data = { 'monitor': CliToken.Monitor.monitorMatcherForShow,
            'telemetry': CliCommand.Node(
               matcher=CliMatcher.KeywordMatcher(
                  'telemetry', helpdesc='Show monitor telemetry information' ),
               guard=telemetryGuard ),
            'inband': 'Show monitor inband telemetry',
            'profile': profileKeyword,
            'edge': edgeKeyword,
            'core': coreKeyword,
            'NAME': profileNameMatcher
   }
   if toggleFeatureInbandTelemetryVxlanEnabled():
      data[ 'vxlan' ] = vxlanKeyword

   cliModel = InbandTelemetryProfiles
   handler = showInbandTelemetryProfileHandler
   privileged = True

class AbortCmd( CliCommand.CliCommandClass ):
   syntax = '''abort'''
   data = {
      'abort': CliToken.Cli.abortMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      mode.abort()

#------------------------------------------------------------------------------------
# The "show monitor telemetry inband" command
#------------------------------------------------------------------------------------

def showInbandTelemetryHandler( mode, args ):
   probeMarker = inbandTelemetryConfig.intParam.probeMarker
   coreProfiles = getProfiles( 'core', '' )
   edgeProfiles = getProfiles( 'edge', '' )
   vxlanProfiles = getProfiles( 'vxlan', '' )
   transitOnly = hwCapability.transitOnlyMode
   profiles = InbandTelemetryProfiles( coreProfiles=coreProfiles,
                                       edgeProfiles=edgeProfiles,
                                       vxlanProfiles=vxlanProfiles,
                                       intVersion=hwCapability.defaultIntVersion,
                                       transitOnly=transitOnly )
   inProgressReason = ""
   if not inbandTelemetryStatus.aggHwStatus or inbandTelemetryStatus.version != \
      ( inbandTelemetryStatus.aggHwStatus.version & Tac.Value(
         "InbandTelemetry::CommandMaskType" ).cliCommandMask ):
      operStatus = "In Progress"

      if inbandTelemetryHwConfig.version != inbTmHeaderStatus.version:
         inProgressReason = inbTmHeaderStatus.reason

      if inbandTelemetryHwConfig.version != inbTmMetadataStatus.version:
         if inProgressReason:
            inProgressReason += ", "
            inProgressReason += inbTmMetadataStatus.reason
         else:
            inProgressReason = inbTmMetadataStatus.reason
      if inbandTelemetryHwConfig.version != inbTmRulesStatus.version:
         if inProgressReason:
            inProgressReason += ", "
            inProgressReason += inbTmRulesStatus.reason
         else:
            inProgressReason = inbTmRulesStatus.reason
   else:
      if inbandTelemetryStatus.aggHwStatus.intOperStatus.state == \
            tacOperState.operStateActive:
         operStatus = "Success"
      elif inbandTelemetryStatus.aggHwStatus.intOperStatus.state == \
           tacOperState.operStateInactive:
         operStatus = \
               "Error, %s" % inbandTelemetryStatus.aggHwStatus.intOperStatus.reason
      else:
         operStatus = ""

   udpDstPorts = []

   if inbandTelemetryConfig.intParam.udpDstPort1Enabled:
      udpDstPorts.append( inbandTelemetryConfig.intParam.udpDstPort1 )

   if inbandTelemetryConfig.intParam.udpDstPort2Enabled:
      udpDstPorts.append( inbandTelemetryConfig.intParam.udpDstPort2 )

   policies = getPolicies()
   return InbandTelemetry( enabled=inbandTelemetryConfig.enable,
         deviceId=inbandTelemetryConfig.intParam.deviceId,
         udpDstPorts=udpDstPorts, probeMarker=probeMarker,
         probeProtocol=inbandTelemetryConfig.intParam.probeProtocol,
         operStatus=operStatus,
         inProgressReason=inProgressReason,
         profiles=profiles, policies=policies,
         intDetectionMethod=hwCapability.defaultIntDetectionMethod,
         intVersion=hwCapability.defaultIntVersion )

class ShowInbandTelemetry( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor telemetry inband'
   data = { 'monitor': CliToken.Monitor.monitorMatcherForShow,
            'telemetry': CliCommand.Node(
               matcher=CliMatcher.KeywordMatcher(
                  'telemetry', helpdesc='Show monitor telemetry information' ),
               guard=telemetryGuard ),
            'inband': 'Show monitor inband telemetry',
   }
   cliModel = InbandTelemetry
   handler = showInbandTelemetryHandler
   privileged = True


# SHOW COMMANDS
#------------------------------------------------------------
# show flow tracking inband telemetry flow-table [detail]
# [ tracker <tracker-name> ] [ group <group-name> ]
# [ src-ip <ip> ] [ dst-ip <ip> ] [ src-port <port> ] [ dst-port <port> ]
# [ protocol <protocol> ] [ vrf <vrf> ] [ vlan <vlan> ]
# [ interface <interface-range> ]
#------------------------------------------------------------
class ShowIntFlowTable:
   def showFlowDetail( self, flowDetailModel, groupAf, entry, bgp, mpls,
                       ifIndexMap ):
      flowDetailModel.srcEthAddr = entry.srcEthAddr
      flowDetailModel.dstEthAddr = entry.dstEthAddr
      flowDetailModel.tcpFlags = tcpFlagStr( entry.tcpFlags )
      flowDetailModel.lastPktTime = entry.lastPktTime
      flowDetailModel.tos = entry.tos
      flowDetailModel.ingressVlanId = entry.ingressVlanId()
      flowDetailModel.ingressIntf = ifindexToIntf( ifIndexMap, entry.inIfIndex ) if \
         entry.inIfIndex else ''
      flowDetailModel.egressVlanId = entry.egressVlanId()
      flowDetailModel.egressIntf = ifIndexMap.get( entry.outIfIndex, 'unknown' )
      flowDetailModel.vni = vniOrUnknown( entry.vni )
      if entry.dot1qVlanTag.hasValidVid():
         flowDetailModel.dot1qVlanId = entry.dot1qVlanTag.vid
         if entry.dot1qCustomerVlanTag.hasValidVid():
            flowDetailModel.dot1qCustomerVlanId = entry.dot1qCustomerVlanTag.vid
      if bgp:
         flowDetailModel.srcAs = bgp.srcAs
         flowDetailModel.dstAs = bgp.dstAs
         flowDetailModel.nextHopIp = IpGenAddr( ipStr( bgp.nextHopIp ) )
         flowDetailModel.bgpNextHopIp = IpGenAddr( ipStr( bgp.bgpNextHopIp ) )
         flowDetailModel.srcPrefixLen = bgp.srcPrefixLen
         flowDetailModel.dstPrefixLen = bgp.dstPrefixLen
      else:
         flowDetailModel.srcAs = 0
         flowDetailModel.dstAs = 0
         zeroAddr = ( zeroIpv4GenAddr if groupAf == AddressFamily.ipv4
               else zeroIpv6GenAddr )
         flowDetailModel.nextHopIp = zeroAddr
         flowDetailModel.bgpNextHopIp = zeroAddr
         flowDetailModel.srcPrefixLen = 0
         flowDetailModel.dstPrefixLen = 0

      if mpls:
         mplsDetail = SftModel.MplsFlowDetail()
         if mpls.topLabel != MplsLabel.null:
            mplsDetail.topLabel = mpls.topLabel
            mplsDetail.topLabelType = lfibSourceToTopLabelType.get( mpls.source )
         mplsDetail.labelStack = [
            mpls.label.get( i ) for i in range( mpls.labelStackSize )
         ]
         mplsDetail.labelStackTruncated = mpls.labelStackTruncated
         if mpls.getRawAttribute( 'fec' ) != IpGenPrefix():
            mplsDetail.fec = mpls.fec
         t1( "MPLS flow entry", mpls.key.str(), "topLabelType:",
             repr( mplsDetail.topLabelType ), "source:", repr( mpls.source ) )
         flowDetailModel.mpls = mplsDetail

      flowDetailModel.sampledPktsReceived = entry.pkts
      flowDetailModel.sampledBytesReceived = entry.bytes
      flowDetailModel.hwPktsReceived = entry.hwPkts
      flowDetailModel.hwBytesReceived = entry.hwBytes

      if groupAf == AddressFamily.ipv6:
         flowDetailModel.flowLabel = entry.flowLabel

   def isEntryMatchingFilter( self, key, entry, ifIndexMap=None, srcIpAddr=None,
         dstIpAddr=None, srcPort=None, dstPort=None, vrfName=None, vlanId=None,
         protocol=None, intfs=None, direction=None ):

      if not flowKeyMatch( key, vrfName, vlanId,
            srcIpAddr, dstIpAddr, srcPort, dstPort, protocol, direction ):
         return False

      # If there are no interfaces on which to filter, consider the flow a match.
      if not intfs:
         return True

      # It is possible a flow entry's ingress interface name cannot be located via
      # the index. In this case, filter the entry.
      if entry.inIfIndex not in ifIndexMap:
         return False

      # This mapping contains special interfaces e.g. 0 maps to "unknown" while
      # 0x3fffffff maps to "CPU". These are to cover possible logical interface
      # terminations for a flow for which there is no corresponding Arnet::IntfId. In
      # practice, these will never match a provided interface filter.
      entryInIntfName = ifIndexMap[ entry.inIfIndex ]
      return entryInIntfName in intfs

   def populateFlowModel( self, key, entry ):
      flowModel = SftModel.FlowModel()
      flowModel.bytesReceived = entry.bytes
      flowModel.pktsReceived = entry.pkts
      flowModel.pktsReceived += entry.hwPkts
      flowModel.bytesReceived += entry.hwBytes
      flowModel.startTime = entry.startTime

      flowKeyModel = SftModel.FlowKeyModel()
      flowKeyModel.vrfName = getVrfNameByVrfIdFromArpVrfIdMap( key.vrfId )
      flowKeyModel.vlanId = key.vlanId
      flowKeyModel.srcAddr = IpGenAddr( ipStr( key.srcAddr ) )
      flowKeyModel.dstAddr = IpGenAddr( ipStr( key.dstAddr ) )
      flowKeyModel.ipProtocolNumber = key.ipProtocolNumber
      try:
         flowKeyModel.ipProtocol = key.ipProtocol
      except ValueError:
         # unassigned protocol number will not have ipProtocol Enum
         pass
      flowKeyModel.srcPort = key.srcPort
      flowKeyModel.dstPort = key.dstPort
      flowKeyModel.direction = key.direction

      flowModel.key = flowKeyModel

      return flowModel

   def showFlowTableEntries( self, grName, groupModel, flowTable, ifIndexMap,
         displayType, srcIpAddr, dstIpAddr, srcPort, dstPort, vrfName, vlanId,
         protocol, intfs, direction ):
      flowEntry = {}
      bgpEntry = {}
      grDirection = AllowedPacketDirection.ingress

      if grName in ( EGRESS_IP_GROUP, EGRESS_IP6_GROUP ):
         grDirection = AllowedPacketDirection.egress

      if direction is None:
         direction = grDirection

      if grName in ( IP_GROUP, EGRESS_IP_GROUP ):
         groupAf = AddressFamily.ipv4
         flowEntry = flowTable.ipFlowEntry
         bgpEntry = flowTable.bgpFlowEntry
         mplsEntry = flowTable.mplsFlowEntry
      elif grName in ( IP6_GROUP, EGRESS_IP6_GROUP ):
         groupAf = AddressFamily.ipv6
         flowEntry = flowTable.ip6FlowEntry
         bgpEntry = flowTable.bgp6FlowEntry
         mplsEntry = flowTable.mpls6FlowEntry
      else:
         groupAf = AddressFamily.ipunknown

      bgp = None
      mpls = None
      for key, entry in flowEntry.items():
         if key.direction != grDirection:
            continue

         if not self.isEntryMatchingFilter( key, entry, ifIndexMap, srcIpAddr,
               dstIpAddr, srcPort, dstPort, vrfName, vlanId, protocol, intfs,
               direction ):
            continue
         if displayType == "detail":
            bgp = bgpEntry.get( key )
            mpls = mplsEntry.get( key )
         flowModel = self.showFlowEntry( flowTable,
                                         groupAf,
                                         key,
                                         entry,
                                         bgp=bgp,
                                         mpls=mpls,
                                         ifIndexMap=ifIndexMap,
                                         displayType=displayType )
         groupModel.flows.append( flowModel )

   def showFlowTable( self, mode, args ):
      ftrType = getFtrTypeFromArgs( args )
      groupName = args.get( 'GROUP_NAME' )
      trackerName = args.get( 'TRACKER_NAME' )
      displayType = args.get( 'detail' )
      srcIpAddr = args.get( 'SRC_IPV4' ) or args.get( 'SRC_IPV6' )
      dstIpAddr = args.get( 'DST_IPV4' ) or args.get( 'DST_IPV6' )
      srcPort = args.get( 'SRC_PORT' )
      dstPort = args.get( 'DST_PORT' )
      protocol = args.get( 'PROTOCOL' )
      vrfName = args.get( 'VRF' )
      vlanId = args.get( 'VLAN' )
      intfs = set( args.get( 'INTFS', () ) )
      direction = args.get( 'DIRECTION' )

      ifIndexMap = getIfIndexMap( ethIntfStatusDir )

      trackingModel = self.fetchTrackingModel()
      if not trackingModel.running:
         return trackingModel

      for trName in ftConfig[ ftrType ].hwFtConfig:
         if trackerName and trName != trackerName:
            continue

         flowTable = self.getFlowTable( shmemEntityManager, trName )
         if not flowTable:
            continue

         trackerModel = SftModel.TrackerModel()
         trackingModel.trackers[ trName ] = trackerModel
         trackerModel.numFlows = 0

         for grName in getFlowGroups( ftrType, trName ):
            if groupName and grName != groupName:
               continue
            groupModel = SftModel.GroupModel()
            trackerModel.groups[ grName ] = groupModel
            self.showFlowTableEntries( grName, groupModel, flowTable, ifIndexMap,
                  displayType, srcIpAddr, dstIpAddr, srcPort, dstPort, vrfName,
                  vlanId, protocol, intfs, direction )
            trackerModel.numFlows += len( groupModel.flows )
            if groupName:
               break
         if trackerName:
            break
      return trackingModel

   def getFlowTable( self, sMount, trackerName ):
      mountPath = 'flowtracking/{}/flowTable/{}'.format( ftrTypeInbandTelemetry,
                                                         trackerName )
      entityType = 'InbandTelemetry::IntFlowTable'
      smashFlowTable = sMount.doMount( mountPath, entityType,
                                       Smash.mountInfo( 'reader' ) )
      return smashFlowTable

   def fetchTrackingModel( self ):
      trackingModel = IntTrackingModel()
      trackingModel.running = isInbandTelemetryAgentRunning( em, activeAgentDir )
      trackingModel.ftrType = ftrTypeInbandTelemetry
      return trackingModel

   def showIntFlowDetail( self, flowDetailModel, entry ):
      if entry is None:
         flowDetailModel.pathTransistions = 0
         flowDetailModel.pathPackets = 0
         flowDetailModel.hopCountExceeded = False
         flowDetailModel.devicesInPath = 0
         flowDetailModel.flowIntervals = 0
         return

      flowDetailModel.pathTransistions = entry.intPathId
      flowDetailModel.pathPackets = entry.intPathNumPackets
      flowDetailModel.hopCountExceeded = bool( entry.intPathOverflowed )
      flowDetailModel.devicesInPath = entry.intPathNumNodes
      flowDetailModel.flowIntervals = entry.intPathNumIntervals
      for nodeIdx in range( entry.intPathNumNodes ):
         nodeInfo = IntFlowDetailNodeInfo()
         nodeInfo.deviceId = entry.intNodeInfo[ nodeIdx ].deviceId
         nodeInfo.ingressPortId = entry.intNodeInfo[ nodeIdx ].ingressPort
         nodeInfo.egressPortId = entry.intNodeInfo[ nodeIdx ].egressPort
         nodeInfo.egressQueueId = entry.intNodeInfo[ nodeIdx ].egressQueueId
         nodeInfo.ttl = entry.intNodeInfo[ nodeIdx ].intTTL
         nodeInfo.lastPacketCongestion = entry.intNodeInfo[ nodeIdx ].congestion
         nodeInfo.lastPacketLatency = entry.intNodeInfo[ nodeIdx ].latency
         flowDetailModel.devicesInformation.append( nodeInfo )
      for statIdx in range( entry.intPathNumIntervals ):
         intrvlStats = IntFlowDetailIntervalStats()
         for nodeIdx in range( entry.intPathNumNodes ):
            globalStatEntry = entry.intGlobalIntervalStats[ statIdx ]
            statEntry = entry.intIntervalStats[
                  ( nodeIdx * hwCapability.intFlowTrackingIntervalsPerNode ) +
                        statIdx ]
            intrvlStats.timestamp = globalStatEntry.intervalStartTime
            intrvlStats.pkts = globalStatEntry.numPackets
            intrvlStats.congestions.append( statEntry.congestion )
            intrvlStats.avgLatencies.append( int( statEntry.avgLatency ) )
            intrvlStats.maxLatencies.append( int( statEntry.maxLatency ) )
            intrvlStats.minLatencies.append( int( statEntry.minLatency ) )
         flowDetailModel.flowIntervalStats.append( intrvlStats )

   def showFlowEntry( self, flowTable, groupAf, key, entry, bgp=None,
                      mpls=None, ifIndexMap=None, displayType=None ):

      flowModel = self.populateFlowModel( key, entry )

      if displayType == "detail":
         flowDetailModel = IntFlowDetailModel()

         intFlowEntry = ( flowTable.intIpFlowEntry if groupAf == AddressFamily.ipv4
                                                   else flowTable.intIp6FlowEntry )
         intFlowEntryKey = intFlowEntry.get( key )
         self.showFlowDetail( flowDetailModel, groupAf, entry, bgp, mpls,
                              ifIndexMap )
         self.showIntFlowDetail( flowDetailModel, intFlowEntryKey )
         flowModel.flowDetail = flowDetailModel

      return flowModel

def showIntFlowTable( mode, args ):
   t = ShowIntFlowTable()
   return t.showFlowTable( mode, args )

ftrTypeInbandTelemetryKwStrs = ftrTypeKwStr[ ftrTypeInbandTelemetry ].split()

class ShowIntFlowTrackingFlowTable( ShowCommand.ShowCliCommandClass ):
   syntax = 'show flow tracking ' + ftrTypeKwStr[ ftrTypeInbandTelemetry ] + \
         ' flow-table [ SHOW_TRACKING_FILTER ] '
   data = {
      'flow': CliToken.Flow.flowMatcherForShow,
      'tracking': trackingShowKw,
      ftrTypeInbandTelemetryKwStrs[ 0 ]: telemetryShowKw,
      ftrTypeInbandTelemetryKwStrs[ 1 ]: inbandShowKw,
      'flow-table': flowTableKw,
      'SHOW_TRACKING_FILTER': generateShowTrackingFilter()
   }

   handler = showIntFlowTable
   cliModel = IntTrackingModel

class ShowIntFlowCounters( ShowCounters ):
   def getFlowTable( self, sMount, trackerName ):
      mountPath = 'flowtracking/{}/flowTable/{}'.format( ftrTypeInbandTelemetry,
                                                         trackerName )
      entityType = 'InbandTelemetry::IntFlowTable'
      smashFlowTable = sMount.doMount( mountPath, entityType,
                                       Smash.mountInfo( 'reader' ) )
      return smashFlowTable

def showIntFlowCounters( mode, args ):
   t = ShowIntFlowCounters()
   model = FtrCounters()
   model.ftrType = ftrTypeInbandTelemetry
   model.running = isInbandTelemetryAgentRunning( em, activeAgentDir )
   t.showCounters( mode, args, model )
   return model

class ShowIntCounters( ShowCommand.ShowCliCommandClass ):
   syntax = 'show flow tracking ' + ftrTypeKwStr[ ftrTypeInbandTelemetry ] + \
            ' counters [ tracker TRACKER_NAME [ exporter EXPORTER_NAME ] ]'

   data = {
      'flow': CliToken.Flow.flowMatcherForShow,
      'tracking': trackingShowKw,
      ftrTypeInbandTelemetryKwStrs[ 0 ]: telemetryShowKw,
      ftrTypeInbandTelemetryKwStrs[ 1 ]: inbandShowKw,
      'counters': 'Show flow tracking counters',
      'tracker': trackerKw,
      'TRACKER_NAME': trackerNameMatcher,
      'exporter': exporterKw,
      'EXPORTER_NAME': exporterNameMatcher,
   }
   handler = showIntFlowCounters
   cliModel = FtrCounters

BasicCli.GlobalConfigMode.addCommandClass( MonitorTelemetryInband )
# TODO: when collector log is supported uncomment the below
#InbandTelemetryConfigMode.addCommandClass( EgressCollectionLogNumCmd )
InbandTelemetryConfigMode.addCommandClass( DeviceIdCmd )
if toggleFeatureIfa1BasedInbandTelemetryEnabled() or \
   toggleFeatureInbandTelemetryTh3TransitSupportEnabled():
   InbandTelemetryConfigMode.addCommandClass( ProbeMarkerCmd )
if toggleFeatureInbandTelemetryTh3TransitSupportEnabled():
   InbandTelemetryConfigMode.addCommandClass( ProbeMarkerUdpCmd )
InbandTelemetryConfigMode.addCommandClass( ProfileConfigCmd )
InbandTelemetryConfigMode.addCommandClass( DisabledCmd )
InbandTelemetryConfigMode.addModelet( CommitAbortModelet )
IntfTelemetryModelet.addCommandClass( InbandTelemetryInterface )
if toggleFeatureInbandTelemetryVxlanEnabled():
   VxlanIntfTelemetryModelet.addCommandClass( InbandTelemetryVxlanInterface )
ProfileConfigMode.addCommandClass( SampleRateCmd )
# TODO: when collector log is supported uncomment the below
#ProfileConfigMode.addCommandClass( EgressCollectionLogCmd )
ProfileConfigMode.addCommandClass( EgressCollectionFlowTrackerCmd )
ProfileConfigMode.addCommandClass( EgressDropDisabledCmd )
if toggleFeatureInbandTelemetryVxlanEnabled():
   ProfileConfigMode.addCommandClass( IngressCollectionFlowTrackerCmd )
ProfileConfigMode.addModelet( CommitAbortModelet )
BasicCli.addShowCommandClass( ShowInbandTelemetry )
BasicCli.addShowCommandClass( ShowInbandTelemetryProfile )
BasicCli.addShowCommandClass( ShowMonitorTelemetryInbandSummary )
BasicCli.addShowCommandClass( ShowIntFlowTrackingFlowTable )
BasicCli.addShowCommandClass( ShowIntCounters )
if ( not toggleFeatureIfa1BasedInbandTelemetryEnabled() ) and \
       toggleFeatureIfa2BasedInbandTelemetryEnabled():
   InbandTelemetryConfigMode.addCommandClass( ProbeProtocolCmd )
if toggleFeatureInbandTelemetrySamplePolicyEnabled():
   InbandTelemetryConfigMode.addCommandClass( IntSamplePolicyConfigCmd )
   IntSamplePolicyConfigMode.addCommandClass( IntSamplePolicyMatchRuleConfigCmd )
   IntSamplePolicyConfigMode.addModelet( CommitAbortModelet )
   IntSamplePolicyMatchRuleConfigMode.addCommandClass( DscpConfigCmd )
   IntSamplePolicyMatchRuleConfigMode.addCommandClass(
      IntSamplePolicyActionsConfigCmd )
   IntSamplePolicyActionRuleConfigMode.addCommandClass( SampleActionCmd )
   IntSamplePolicyMatchRuleConfigMode.addModelet( CommitAbortModelet )
   IntSamplePolicyActionRuleConfigMode.addModelet( CommitAbortModelet )
   IntSamplePolicyConfigMode.addShowCommandClass( ShowPendingCmd )
   IntSamplePolicyMatchRuleConfigMode.addShowCommandClass( ShowPendingCmd )
   IntSamplePolicyActionRuleConfigMode.addShowCommandClass( ShowPendingCmd )
   ProfileConfigMode.addCommandClass( IngressSamplePolicyCmd )

#-------------------------------------------------------------------------------
# Support for default interface command.
# Clean up all the configuration set by cmds in interface config mode.
#-------------------------------------------------------------------------------
class InbandTelemetryIntfJanitor( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      intfName = self.intf_.name
      if intfName in inbandTelemetryConfig.intfToEdgePortProfiles:
         del inbandTelemetryConfig.intfToEdgePortProfiles[ intfName ]
      elif intfName in inbandTelemetryConfig.intfToCorePortProfiles:
         del inbandTelemetryConfig.intfToCorePortProfiles[ intfName ]
      elif intfName in inbandTelemetryConfig.intfToVxlanPortProfiles:
         del inbandTelemetryConfig.intfToVxlanPortProfiles[ intfName ]
      if inbandTelemetryConfig.enable:
         incrementConfigVersion( inbandTelemetryConfig )

#--------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#--------------------------------------------------------------------------------
@Plugins.plugin( provides=( "InbandTelemetryCli", ) )
def Plugin( entityManager ):
   global inbandTelemetryConfig, inbandTelemetryStatus, hwCapability, em
   global policiesCliConfig, policiesStatus, policiesStatusRequestDir
   global activeAgentDir, intFtCounters
   global inbandTelemetryHwConfig, inbTmHeaderStatus
   global inbTmMetadataStatus, inbTmRulesStatus
   global intFlowTrackingCollectors
   global ethIntfStatusDir
   global shmemEntityManager

   em = entityManager
   shmemEntityManager = SharedMem.entityManager( sysdbEm=em )

   samplePoliciesConfigPath = 'inbandtelemetry/samplePolicies/input/cli'
   samplePoliciesCliConfigType = 'TrafficPolicy::TrafficPolicyConfig'
   policiesStatusPath = 'cell/%d/inbandtelemetry/samplePolicies/status' % \
                        Cell.cellId()
   policiesStatusType = 'Tac::Dir'
   policiesStatusRequestDirPath = 'inbandtelemetry/samplePolicies/statusRequest/cli'
   policiesStatusRequestDirType = 'PolicyMap::PolicyMapStatusRequestDir'

   hardwareHeaderStatusPath = 'hardware/inbandtelemetry/inbtmheaderstatus'
   hardwareHeaderStatusType = 'AleInbandTelemetry::InbTmHeaderStatus'
   hardwareMetadataStatusPath = 'hardware/inbandtelemetry/inbtmmetadatastatus'
   hardwareMetadataStatusType = 'AleInbandTelemetry::InbTmMetadataStatus'
   hardwareRulesStatusPath = 'hardware/inbandtelemetry/inbtmrulesstatus'
   hardwareRulesStatusType = 'AleInbandTelemetry::InbTmRulesStatus'

   mountGroup = entityManager.mountGroup()
   policiesStatus = mountGroup.mount( policiesStatusPath, policiesStatusType, 'ri' )
   mountGroup.close( callback=None, blocking=False )

   inbandTelemetryConfig = ConfigMount.mount( entityManager,
                                              "inbandtelemetry/config",
                                              "InbandTelemetry::Config", "w" )
   inbandTelemetryStatus = LazyMount.mount( entityManager, "inbandtelemetry/status",
                                            "InbandTelemetry::Status", "r" )
   intFlowTrackingCollectors = LazyMount.mount(
      entityManager,
      getInbandTelemetryFlowTrackingCliConfigPath(),
      "FlowTracking::Config", "r" )
   hwCapability = LazyMount.mount( entityManager, "inbandtelemetry/hwCapability",
                                   "InbandTelemetry::HwCapability", "r" )
   policiesCliConfig = ConfigMount.mount( entityManager, samplePoliciesConfigPath,
                                          samplePoliciesCliConfigType, 'wi' )
   policiesStatusRequestDir = LazyMount.mount( entityManager,
                                               policiesStatusRequestDirPath,
                                               policiesStatusRequestDirType, 'wc' )
   intFtCounters = SmashLazyMount.mount( entityManager,
                                         "flowtracking/inbandTelemetry/counters",
                                         "Smash::FlowTracker::FtCounters",
                                         SmashLazyMount.mountInfo( 'reader' ) )
   IntfCli.Intf.registerDependentClass( InbandTelemetryIntfJanitor )
   activeAgentDir = LazyMount.mount( em, 'flowtracking/activeAgent',
                                     'Tac::Dir', 'ri' )

   inbTmHeaderStatus = LazyMount.mount( entityManager,
                                        hardwareHeaderStatusPath,
                                        hardwareHeaderStatusType, "r" )
   inbTmMetadataStatus = LazyMount.mount( entityManager,
                                          hardwareMetadataStatusPath,
                                          hardwareMetadataStatusType, "r" )
   inbTmRulesStatus = LazyMount.mount( entityManager,
                                       hardwareRulesStatusPath,
                                       hardwareRulesStatusType, "r" )
   inbandTelemetryHwConfig = LazyMount.mount( entityManager,
                                              "inbandtelemetry/hwConfig",
                                              "InbandTelemetry::HwConfig", "r" )
   ethIntfStatusDir = LazyMount.mount( em, "interface/status/eth/intf",
                                       "Interface::EthIntfStatusDir", "r" )
   ftrType = ftrTypeInbandTelemetry
   interfaceCounters[ ftrType ] = SmashLazyMount.mount( em,
                              'flowtracking/%s/interfaceCounters' % ftrType,
                              'Smash::SampledFlowTracker::InterfaceCounters',
                              SmashLazyMount.mountInfo( 'reader' ) )
   ftConfig[ ftrType ] = LazyMount.mount( em,
                                'hardware/flowtracking/config/%s' % ftrType,
                                'HwFlowTracking::Config', 'r' )
   ftCounters[ ftrType ] = SmashLazyMount.mount( em,
                                       'flowtracking/%s/counters' % ftrType,
                                       'Smash::FlowTracker::FtCounters',
                                       SmashLazyMount.mountInfo( 'reader' ) )
   ipfixStats[ ftrType ] = SmashLazyMount.mount( em,
                                   'flowtracking/%s/ipfix/statistics' % ftrType,
                                   'Smash::Ipfix::CollectorStatistics',
                                   SmashLazyMount.mountInfo( 'reader' ) )

CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2020-10-29 22:19:56',
   cmds=[ 'show monitor telemetry inband',
          'show monitor telemetry inband profile summary'
   ],
   cmdsGuard=lambda: hwCapability.inbandTelemetrySupported )

CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2020-10-29 22:19:56',
   cmds=[ 'show flow tracking telemetry inband',
          'show flow tracking telemetry inband counters'
   ],
   cmdsGuard=lambda: not hwCapability.transitOnlyMode and
   hwCapability.inbandTelemetrySupported )
