#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliGlobal
import CliMatcher
import CliCommand
import ConfigMount
import ReversibleSecretCli
from CliToken.Monitor import monitorMatcher
from CliMode.InfluxTelemetry import InfluxTelemetryBaseMode
from CliMode.InfluxTelemetry import DestinationInfluxdbMode
from CliMode.InfluxTelemetry import SourceSocketMode
from CliPlugin.AaaCli import usernameMatcher
from CliPlugin.VrfCli import VrfNameExprFactory
from IpLibConsts import DEFAULT_VRF

gv = CliGlobal.CliGlobal( {
   'config': None,
} )

#----------------------------------------------------------
# monitor telemetry influx
#----------------------------------------------------------
class InfluxTelemetryConfigMode( InfluxTelemetryBaseMode,
                                 BasicCli.ConfigModeBase ):
   name = 'mon-telemetry-influx'

   def __init__( self, parent, session ):
      InfluxTelemetryBaseMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class MonitorInfluxTelemetryCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor telemetry influx'
   noOrDefaultSyntax = syntax
   data = {
      'monitor': monitorMatcher,
      'telemetry': 'Telemetry configuration',
      'influx': 'Influx telemetry',
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( InfluxTelemetryConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      gv.config.destinationInfluxdbConfig.clear()
      gv.config.vrfName = DEFAULT_VRF
      gv.config.sourceSocketConfig.clear()
      gv.config.sourceGroups.clear()
      gv.config.globalTags.clear()

#----------------------------------------------------------
# [ no | default ] destination influxdb WORD
#----------------------------------------------------------
class DestinationInfluxdbConfigMode( DestinationInfluxdbMode,
                                     BasicCli.ConfigModeBase ):
   name = 'influxdb destination configuration'

   def __init__( self, parent, session, dest, destName ):
      self.dest = dest
      self.destName = destName
      DestinationInfluxdbMode.__init__( self, destName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class DestinationInfluxdbCmd( CliCommand.CliCommandClass ):
   syntax = 'destination influxdb NAME'
   noOrDefaultSyntax = syntax
   data = {
      'destination': 'Configure telemetry output destinations',
      'influxdb': 'Configure InfluxDB destinations',
      'NAME': CliMatcher.PatternMatcher(
         pattern='.+',
         helpdesc='InfluxDB connection name', helpname='WORD' ),
   }

   @staticmethod
   def handler( mode, args ):
      destName = args[ 'NAME' ]
      gv.config.destinationInfluxdbConfig.newMember( destName )
      dest = gv.config.destinationInfluxdbConfig[ destName ]
      dest.vrfName = gv.config.vrfName
      childMode = mode.childMode( DestinationInfluxdbConfigMode,
                                 dest=dest, destName=destName )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      destName = args[ 'NAME' ]
      del gv.config.destinationInfluxdbConfig[ destName ]
      if not gv.config.destinationInfluxdbConfig:
         gv.config.vrfName = DEFAULT_VRF

#----------------------------------------------------------
# [ no | default ] url URL
#----------------------------------------------------------
class UrlCmd( CliCommand.CliCommandClass ):
   syntax = 'url URL'
   noOrDefaultSyntax = 'url ...'
   data = {
      'url': 'Configure a URL for the destination',
      'URL': CliMatcher.PatternMatcher(
         pattern='(http(s)?|udp|unix)://.+',
         helpdesc='URL of the destination', helpname='URL' ),
   }

   @staticmethod
   def handler( mode, args ):
      mode.dest.url = args.get( 'URL', '' )

   noOrDefaultHandler = handler

#----------------------------------------------------------
# [ no | default ] database name WORD
#----------------------------------------------------------
class DatabaseNameCmd( CliCommand.CliCommandClass ):
   syntax = 'database name NAME'
   noOrDefaultSyntax = 'database name ...'
   data = {
      'database': 'Database configuration',
      'name': 'Set the name of the database',
      'NAME': CliMatcher.PatternMatcher(
         pattern='.+',
         helpdesc='Database name', helpname='WORD' ),
   }

   @staticmethod
   def handler( mode, args ):
      mode.dest.databaseName = args.get( 'NAME', '' )

   noOrDefaultHandler = handler

#----------------------------------------------------------
# [ no | default ] retention policy POLICY
#----------------------------------------------------------
class RetentionPolicyCmd( CliCommand.CliCommandClass ):
   syntax = 'retention policy POLICY'
   noOrDefaultSyntax = 'retention policy ...'
   data = {
      'retention': 'Configure data retention',
      'policy': 'Set the name of the retention policy',
      'POLICY': CliMatcher.PatternMatcher(
         pattern='.+',
         helpdesc='Retention policy name', helpname='WORD' ),
   }

   @staticmethod
   def handler( mode, args ):
      mode.dest.retentionPolicyName = args.get( 'POLICY', 'default' )

   noOrDefaultHandler = handler

#----------------------------------------------------------
# [ no | default ] vrf VRF
#----------------------------------------------------------
class VrfCmd( CliCommand.CliCommandClass ):
   syntax = 'vrf VRF'
   noOrDefaultSyntax = 'vrf ...'
   data = {
      'vrf': 'Configure vrf',
      'VRF': VrfNameExprFactory( inclDefaultVrf=True ),
   }

   @staticmethod
   def handler( mode, args ):
      gv.config.vrfName = args.get( 'VRF', DEFAULT_VRF )
      for dest in gv.config.destinationInfluxdbConfig.values():
         dest.vrfName = gv.config.vrfName

   noOrDefaultHandler = handler

#------------------------------------------------------------------
# [ no | default ] username USERNAME [ password <0|7> PASSWORD ]
#------------------------------------------------------------------
class UsernameCmd( CliCommand.CliCommandClass ):
   syntax = 'username USERNAME password PASSWORD'
   noOrDefaultSyntax = 'username ...'
   data = {
      'username': 'Configure login details with destination',
      'USERNAME': usernameMatcher,
      'password': 'Assign login password',
      'PASSWORD': ReversibleSecretCli.defaultReversiblePwdCliExpr,
   }

   @staticmethod
   def handler( mode, args ):
      username = args.get( 'USERNAME', '' )
      password = args.get( 'PASSWORD', ReversibleSecretCli.getDefaultSecret() )
      mode.dest.username = username
      mode.dest.password = password

   noOrDefaultHandler = handler

#----------------------------------------------------------
# [ no | default ] source socket WORD
#----------------------------------------------------------
class SourceSocketConfigMode( SourceSocketMode,
                              BasicCli.ConfigModeBase ):
   name = 'socket source configuration'

   def __init__( self, parent, session, socket, socketName ):
      self.socket = socket
      self.socketName = socketName
      SourceSocketMode.__init__( self, socketName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class SourceSocketCmd( CliCommand.CliCommandClass ):
   syntax = 'source socket NAME'
   noOrDefaultSyntax = syntax
   data = {
      'source': 'Configure telemetry input paths',
      'socket': 'socket connection',
      'NAME': CliMatcher.PatternMatcher(
         pattern='.+',
         helpdesc='Label of the socket connection',
         helpname='WORD' ),
   }

   @staticmethod
   def handler( mode, args ):
      socketName = args[ 'NAME' ]
      gv.config.sourceSocketConfig.newMember( socketName )
      socket = gv.config.sourceSocketConfig[ socketName ]
      childMode = mode.childMode( SourceSocketConfigMode,
                                  socket=socket, socketName=socketName )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      socketName = args[ 'NAME' ]
      del gv.config.sourceSocketConfig[ socketName ]

#----------------------------------------------------------
# [ no | default ] url URL
#----------------------------------------------------------
class SourceSocketUrlCmd( CliCommand.CliCommandClass ):
   syntax = 'url URL'
   noOrDefaultSyntax = 'url ...'
   data = {
      'url': 'Configure socket URL',
      'URL': CliMatcher.PatternMatcher(
         pattern='(tcp[46]?|udp[46]?|unix(gram)?)://.+',
         helpdesc='URL of the socket', helpname='URL' ),
   }

   @staticmethod
   def handler( mode, args ):
      mode.socket.url = args.get( 'URL', '' )

   noOrDefaultHandler = handler

#----------------------------------------------------------
# [ no | default ] connection limit <n>
#----------------------------------------------------------
class ConnectionLimitCmd( CliCommand.CliCommandClass ):
   syntax = 'connection limit LIMIT'
   noOrDefaultSyntax = 'connection limit ...'
   data = {
      'connection': 'Configure connection settings',
      'limit': 'Set the maximum number of connections allowed',
      'LIMIT': CliMatcher.IntegerMatcher(
         0, 0xFFFFFFFF, helpdesc='Number of connections allowed' ),
   }

   @staticmethod
   def handler( mode, args ):
      mode.socket.connectionLimit = args.get( 'LIMIT', 0 )

   noOrDefaultHandler = handler

# ----------------------------------------------------------------
# [ no | default ] source group ( ( standard disabled ) | GROUP )
# ----------------------------------------------------------------
def sourceGroupOpts( mode, context ):
   return {
      sourceGroup: '%s source group' % sourceGroup
      for sourceGroup in gv.config.sourceGroups if sourceGroup != 'standard'
   }

class SourceGroupStandardCmd( CliCommand.CliCommandClass ):
   syntax = 'source group ( ( standard disabled ) | GROUP )'
   noOrDefaultSyntax = syntax
   data = {
      'source': 'Configure telemetry input paths',
      'group': 'Subscribe to a predefined group of sources',
      'standard': 'Configure standard set of telemetry',
      'disabled': 'Disable standard set of telemetry',
      # the dynamic keyword matcher shows all registered groups that aren't standard.
      # The "standard" group is a bit special cased here. To register new
      # source groups they should be added to the package DefaultConfigPlugin
      'GROUP': CliMatcher.DynamicKeywordMatcher(
         keywordsFn=sourceGroupOpts, passContext=True )
   }

   @staticmethod
   def handler( mode, args ):
      if 'standard' in args:
         gv.config.sourceGroups[ 'standard' ] = 'disabled'
      else:
         gv.config.sourceGroups[ args[ 'GROUP' ] ] = 'enabled'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'standard' in args:
         gv.config.sourceGroups[ 'standard' ] = 'enabled'
      else:
         gv.config.sourceGroups[ args[ 'GROUP' ] ] = 'disabled'

#----------------------------------------------------------
# [ no | default ] tag global KEY VALUE
#----------------------------------------------------------
class TagGlobalCmd( CliCommand.CliCommandClass ):
   syntax = 'tag global KEY VALUE'
   noOrDefaultSyntax = 'tag global KEY ...'
   data = {
      'tag': 'Add extra tags to the telemetry output',
      'global': 'Add tags to all telemetry output globally',
      'KEY': CliMatcher.PatternMatcher(
         pattern='.+',
         helpdesc='Specify the key of the global tag pair', helpname='WORD' ),
      'VALUE': CliMatcher.PatternMatcher(
         pattern='.+',
         helpdesc='Specify the value of the global tag pair', helpname='WORD' ),
   }

   @staticmethod
   def handler( mode, args ):
      key = args[ 'KEY' ]
      value = args[ 'VALUE' ]
      gv.config.globalTags[ key ] = value

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      key = args[ 'KEY' ]
      del gv.config.globalTags[ key ]

BasicCli.GlobalConfigMode.addCommandClass( MonitorInfluxTelemetryCmd )
InfluxTelemetryConfigMode.addCommandClass( DestinationInfluxdbCmd )
DestinationInfluxdbConfigMode.addCommandClass( UrlCmd )
DestinationInfluxdbConfigMode.addCommandClass( DatabaseNameCmd )
DestinationInfluxdbConfigMode.addCommandClass( RetentionPolicyCmd )
DestinationInfluxdbConfigMode.addCommandClass( VrfCmd )
DestinationInfluxdbConfigMode.addCommandClass( UsernameCmd )
InfluxTelemetryConfigMode.addCommandClass( SourceSocketCmd )
SourceSocketConfigMode.addCommandClass( SourceSocketUrlCmd )
SourceSocketConfigMode.addCommandClass( ConnectionLimitCmd )
InfluxTelemetryConfigMode.addCommandClass( SourceGroupStandardCmd )
InfluxTelemetryConfigMode.addCommandClass( TagGlobalCmd )

def Plugin( entityManager ):
   gv.config = ConfigMount.mount(
      entityManager, 'telegrafMgr/config',
      'TelegrafMgr::TelegrafMgrConfig', 'w' )
