# Copyright (c) 2005-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

#-------------------------------------------------------------------------------
# This module implements the framework for interface configuration.  In
# particular, it provides:
# -  the Intf class
# -  the "config-if" mode
# -  the "[no] interface <name>" command
# -  the "show interfaces [<name>]" command
# -  the "show interfaces [<name>] description" command
# -  the "show interfaces [<name>] counters" command
# -  the "show interfaces [<name>] counters discards" command
# -  the "show interfaces [<name>] counters delta" command
# -  the "clear counters [<name>] [session]" command
# -  the "[no] mac-address router MAC_ADDR" command
# -  the "[no] mtu [value]" command
# -  the "[no] shutdown" command
# -  the "[no] description <desc>" command.
# -  the "[no|default] load-interval <interval>" command (interface level)
# -  the "[no|default] load-interval defaul <interval>" command (global config level)
# -  the "[no] logging event link-status (interface level)
# -  the "[no] logging event link-status global (global config level)
#-------------------------------------------------------------------------------

import copy
import functools
import itertools
import re
import threading
import weakref
from collections import defaultdict

import Ark
import Tac
import Arnet
import BasicCli
import BasicCliModes
import CliCommand
import CliExtensions
import CliMatcher
import CliMode.InterfaceDefaults
from CliMode.Intf import IntfMode
from CliModel import UnknownEntityError
import CliParser
import CliToken.Clear
import CliToken.Logging
import CliToken.Service
import ConfigMount
import Intf.IntfRange as IntfRange # pylint: disable=consider-using-from-import
import Intf.Log as Log # pylint: disable=consider-using-from-import
import LazyMount
import MultiRangeRule
import QuickTrace
import ShowCommand
import SmashLazyMount
import TacSigint
import Tracing
from CliPlugin import IntfModel, WaitForWarmupCli
# pylint: disable-next=consider-using-from-import
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.MacAddr as MacAddr # pylint: disable=consider-using-from-import
from CliPlugin.VirtualIntfRule import IntfMatcher
import Ethernet

qv = QuickTrace.Var
qt0 = QuickTrace.trace0
qt8 = QuickTrace.trace8

aclDropConfigDir = None
allIntfConfigDir = None
allIntfStatusDir = None
globalIntfConfig = None
allIntfAclDropCounters = {}
routerMacHwCap = None

# This extension is called when the Cli
# tries to display an attribute of an intf that
# is only meaningful if bridging is enabled, yet
# the forwarding model of an intf is dataLink.
# extension is a function  that takes an eth
# intf name and the Cli mode, and then
# returns the reason (a string) that
# the intf is dataLink.  (E.g. "is in po3" ).
# DataLink Explanation hook has been moved from Ebra
# to here for 2 reasons.
# a.) It does not make sense to add Ebra dependency in packages
#     that want to use the hook.
# b.) Hooks are added by different packages like
# Lag/Mirror etc.. and it can be used by dependent packages.
dataLinkExplanationHook = CliExtensions.CliHook()
# This is the hook for intfForwardingModelQuietDataLink
quietDataLinkExplanationHook = CliExtensions.CliHook()
# This is the hook for intfForwardingModelUnauthorized
unauthorizedExplanationHook = CliExtensions.CliHook()
# This is the hook for intfForwardingModelRecirculation
recirculationExplanationHook = CliExtensions.CliHook()
# This is the hook for intfForwardingModelLayer1Patch
layer1PatchExplanationHook = CliExtensions.CliHook()
# This is the new hook for intfForwardingModelLinkQualification
linkQualificationExplanationHook = CliExtensions.CliHook()

_forwardingModelMap = { "intfForwardingModelLinkQualification": "linkQualification",
                        "intfForwardingModelLayer1Patch": "layer1Patch",
                        "intfForwardingModelLayer1PatchAndLldp":
                           "layer1PatchAndLldp",
                        "intfForwardingModelRecirculation": "recirculation",
                        "intfForwardingModelQuietDataLink" : "quietDataLink",
                        "intfForwardingModelUnauthorized" : "unauthorized",
                        "intfForwardingModelDataLink" : "dataLink",
                        "intfForwardingModelBridged" : "bridged",
                        "intfForwardingModelRouted" : "routed"
                      }

intfMtuGuardHook = []
subintfMtuGuardHook = []

intfIdDisplayContextDir = None
intfIdDisplayContextHelper = None

def tryWaitForWarmup( mode ):
   WaitForWarmupCli.tryWaitForWarmupAfterVlanPortChange( mode )

# Helper methods for create and deleting IntfConfig requests from the Cli
def cliCreateIntfConfigRequest( intfConfigDir, intfName ):
   # Enqueue a request for the interface
   intfConfigDir.intfConfigRequest[ Tac.Value(
      "Interface::IntfConfigRequest", intf=intfName, requestor='Cli' ) ] = True

def cliDeleteIntfConfigRequest( intfConfigDir, intfName ):
   # Del the request.
   del intfConfigDir.intfConfigRequest[ Tac.Value(
      "Interface::IntfConfigRequest", intf=intfName, requestor='Cli' ) ]

# Return the requestors for a given interface
def intfConfigRequestors( intfConfigDir, intfName ):
   return list( intfConfigDir.intfConfig[ intfName ].requestor )

def intfsSupported( mode, supportedFunc, warning, intf=None, mod=None,
                    intfType=None ):
   # returns interfaces that support supportedFunc
   intfs = Intf.getAll( mode, intf, mod, intfType=intfType )
   if not intfs:
      return None
   intfs = [ i for i in intfs if supportedFunc( i ) ]
   if not intfs:
      mode.addWarning( "%s not supported on %s" %
                       ( warning, intf or "any interface", ) )
   return intfs

def counterSupportedIntfs( mode, intf=None, mod=None, intfType=None ):
   # FIXME: `methodcaller` instead of lambdas.
   return intfsSupported( mode, lambda i: i.countersSupported(), "Counters",
                          intf=intf, mod=mod, intfType=intfType )

def dpCounterSupportedIntfs( mode, intf=None, mod=None, intfType=None ):
   return intfsSupported( mode, lambda i: i.dpCountersSupported(), "Counters",
                          intf=intf, mod=mod, intfType=intfType )

def countersErrorsSupportedIntfs( mode, intf=None, mod=None ):
   return intfsSupported( mode, lambda i: i.countersErrorsSupported(), "Counters",
                          intf=intf, mod=mod )

def counterDiscardSupportedIntfs( mode, intf=None, mod=None ):
   return intfsSupported( mode, lambda i: i.countersDiscardSupported(),
                          "Counter discards are", intf=intf, mod=mod )

def countersRateSupportedIntfs( mode, intf=None, mod=None ):
   return intfsSupported( mode, lambda i: i.countersRateSupported(),
                          "Counter rates are", intf=intf, mod=mod )

def perVniCounterSupportedIntfs( mode, intf=None, mod=None ):
   return intfsSupported( mode, lambda i: i.perVniCountersSupported(),
                          "Per-VNI counters", intf=intf, mod=mod )

def perVniCountersRateSupportedIntfs( mode, intf=None, mod=None ):
   return intfsSupported( mode, lambda i: i.perVniCountersRateSupported(),
                          "Per-VNI counter rates are", intf=intf, mod=mod )

def _getAclDropConfigKey( intf, aclDropCounters ):
   for i in aclDropCounters:
      if aclDropConfigDir.get( i ):
         keyFound = aclDropConfigDir[ i ].intf.get( intf.name )
         # retVal can be either key or None
         # We return key when the counters should be accessed. Otherwise
         # return None to indicate the key cannot be read at this time
         if keyFound is not None:
            return i
      else:
         return None
   return None

def _maybeMountAclDropCounters():
   for name, entry in aclDropConfigDir.items():
      if name not in allIntfAclDropCounters:
         allIntfAclDropCounters[ name ] = SmashLazyMount.mount(
               myEntManager,
               "interface/status/counter/acldrop/" + entry.writerMountName,
               "Smash::Interface::AllDropCounterDir",
               SmashLazyMount.mountInfo( 'reader' ) )

def moduleAndPort( intfName ):
   """
    Returns the module and port number of the interface, or None if this is not
    a physical interface.  If this is a port in a fixed config system, returns
    None for the module but the right number for the port number.
   """
   # match against interface names with 2 or more levels
   m = re.match( r'(\D+)(\d+)/(\d+)(?:/\d+)*$', intfName )
   if m is not None:
      return ( int( m.group( 2 ) ), int( m.group( 3 ) ) )
   # match against interface names with 1 or more levels
   m = re.match( r'(\D+)(\d+)(?:/\d+)*$', intfName )
   if m is not None:
      return ( None, int( m.group( 2 ) ) )
   return ( None, None )

#-------------------------------------------------------------------------------
# Generic interface helper class. This is used as a base class for all
# dependent classes of Interface that wants to help Intf class to provide
# certain functionality like lookup for show commands, clean-up interface
# configuration on deletion or setting it to default
#-------------------------------------------------------------------------------
class IntfDependentBase:
   """This class is responsible for cleaning up per-interface configurations
   when the interface is deleted (via "no interface" command) or when the
   interface is set to its default state (via "default interface" command)
   """
   def __init__( self, intf, mode ):
      self.intf_ = intf
      self.mode_ = mode

   def destroy( self ):
      # Some tests use only the config state without having the
      # corresponding status and we expect to deal with defaulting
      # those interfaces as well
      if self.intf_.config():
         self.setDefault()

   def setDefault( self ):
      pass

# global storage for session counters
threadLocalData_ = threading.local()

def _sessionCounterDir():
   # TODO: eventually this should just be stored in the mode.session.sessionData
   # like deltaCounterDir is. However it's ok for it to be a per thread attribute
   if getattr( threadLocalData_, 'sessionCounterDir', None ) is None:
      threadLocalData_.sessionCounterDir = {}

   return threadLocalData_.sessionCounterDir

def _deltaCounterDir( mode ):
   deltaCounterDir = mode.session.sessionData( 'intfDeltaCounterDir', None )
   if deltaCounterDir is None:
      deltaCounterDir = {}
      mode.session.sessionDataIs( 'intfDeltaCounterDir', deltaCounterDir )

   return deltaCounterDir

class DependentClassRegistry( defaultdict ):
   def __init__( self ):
      super().__init__( list )

   def registerDependentClass( self, derived, priority=20 ):
      self[ priority ].append( derived )

   def __iter__( self ):
      for _priority, classes in sorted( self.items() ):
         yield from classes

#-------------------------------------------------------------------------------
# Generic interface class.  There is a subclass of this class for every physical
# interface type that is supported by the CLI.
#
# This class should never be instantiated directly; only its subclasses should
# be instantiated.
#
# Note that existence of an instance of this class is independent of existence
# of the corresponding interface and C++ objects.  Whether or not the interface
# exists can be tested by calling the lookup() method.  The interface can be
# created and destroyed by calling the create() and destroy() methods.
#-------------------------------------------------------------------------------
@functools.total_ordering
class Intf:
   #----------------------------------------------------------------------------
   # A class for a protocol module that wishes to hook into the base
   # CLI commands for an Intf should use this API to
   # register itself.  The dependent class must have a constructor
   # that takes two parameters: an Intf instance and sysdbRoot.  As of
   # 2007-10-10, IraIpIntfCli.IpIntf is registered to ensure
   # that the IpIntf gets destroyed in 'no interface' and that the network
   # address gets displayed in 'show interface'
   #----------------------------------------------------------------------------
   depClsRegistry_ = DependentClassRegistry()
   registerDependentClass = depClsRegistry_.registerDependentClass

   #----------------------------------------------------------------------------
   # Subclasses of this class must register themselves by calling
   # Intf.addPhysicalIntfType.  This enables this class to query them all to
   # find all existing interfaces, which is needed in order to perform the
   # "show interfaces" command.  This function also modifies the Intf.matcher to
   # support names of particular types of interface.
   #----------------------------------------------------------------------------
   physicalIntfClasses_ = []

   @staticmethod
   def addPhysicalIntfType( clazz,
                            intfRangeType,
                            withIpSupport=False,
                            withArpSupport=False,
                            withRoutingProtoSupport=True,
                            matcherWithRoutingProtoSupport=None ):
      # The withRoutingProtoSupport is checked only when withIpSupport is True.
      # When matcherWithRoutingProtoSupport is None, the clazz.matcher is used.
      # In EthPhyIntf case, the clazz.matcher includes both Ethernet and Management
      # interfaces. The former supports routing protocol while the later does
      # not. In this case, the matcherWithRoutingProtoSupport should be given
      # explicitly.

      Intf.physicalIntfClasses_.append( clazz )
      Intf.matcher |= clazz.matcher
      Intf.matcherWOSubIntf |= clazz.matcher

      if intfRangeType:
         if not hasattr( intfRangeType, "__iter__" ):
            intfRangeType = ( intfRangeType, )
         for irt in intfRangeType:
            irt.cliIntfClazzIs( clazz )
      if withIpSupport:
         Intf.matcherWithIpSupport |= clazz.matcher
         if withRoutingProtoSupport:
            if matcherWithRoutingProtoSupport:
               Intf.matcherWithRoutingProtoSupport |= \
                  matcherWithRoutingProtoSupport
            else:
               Intf.matcherWithRoutingProtoSupport |= clazz.matcher
      if withIpSupport or withArpSupport:
         Intf.matcherWithArpSupport |= clazz.matcher

   @staticmethod
   def addSubIntf( subMatcher,
                   clazz,
                   withIpSupport=False,
                   withArpSupport=False,
                   withRoutingProtoSupport=True ):
      Intf.matcher |= subMatcher
      if clazz not in Intf.physicalIntfClasses_:
         Intf.physicalIntfClasses_.append( clazz )
      if withIpSupport:
         Intf.matcherWithIpSupport |= subMatcher
         if withRoutingProtoSupport:
            Intf.matcherWithRoutingProtoSupport |= subMatcher
      if withIpSupport or withArpSupport:
         Intf.matcherWithArpSupport |= subMatcher

   @staticmethod
   def getShortname( longname ):
      return IntfMode.getShortname( longname )

   #----------------------------------------------------------------------------
   # Defines whether or not the interface is internal to the switch, These
   # interfaces will not be be enumerated by Intf.getAll unless the interface
   # name / type is specifically requested or if the exposeInternal flag is
   # specified.
   #----------------------------------------------------------------------------
   def internal( self ):
      return False

   #----------------------------------------------------------------------------
   # A matcher to match interface names and return a corresponding Intf object.
   # To begin with this matcher is empty, but each call to
   # Intf.addPhysicalIntfType() modifies this matcher to add a pattern matching
   # names of interfaces of that type.
   #----------------------------------------------------------------------------
   matcher = IntfMatcher()
   # interface matchers that exclude the support for sub interface.
   matcherWOSubIntf = IntfMatcher()
   # interface matchers that support IP address configuration
   matcherWithIpSupport = IntfMatcher()
   # interface matcher that support IP and IP routing protocol.
   # This is a subset of matcherWithIpSupport
   matcherWithRoutingProtoSupport = IntfMatcher()
   # interface matcher that supports ARP/ND entries
   # this is a superset of matcherWithIpSupport and Recirc-Channel
   matcherWithArpSupport = IntfMatcher()

   # If your command should support interface range, use the following instead of
   # single-interface matchers.
   rangeMatcher = IntfRange.intfRangeMatcher

   # Range matchers for physical interfaces don't support non-existent
   # physical interfaces. So noSingletons is set to make the range
   # matchers match only interface ranges and the single interface matcher,
   # matcherWithIpSupport is used to match non-existent interfaces also.
   # The value of this matcher may be either IntfList or a single Intf
   # object and the CLI command handler needs to handle accordingly.
   rangeMatcherWithIpSupport = IntfRange.IntfRangeMatcher(
      explicitIntfTypes=IntfRange.intfTypesWithIpSupport )

   #----------------------------------------------------------------------------
   # Create a new Intf instance.
   #----------------------------------------------------------------------------
   def __init__( self, name, mode ):
      self.name_ = name
      self.mode_ = mode
      self.sysdbRoot_ = self.mode_.sysdbRoot
      self.intfCounter = None
      self.shortname_ = IntfMode.getShortname( name )

      self.counterFreshnessTime = 0

   name = property( lambda self: self.name_ )
   shortname = property( lambda self: self.shortname_ )

   def __str__(self):
      return self.name_

   #----------------------------------------------------------------------------
   # This method indicates whether or not interface creation is currently
   # supported.
   #----------------------------------------------------------------------------
   def intfCreationCurrentlySupported( self, mode ):
      return True

   def userVisible( self ):
      status = self.status()
      if not status:
         return True
      return status.userVisible

   #----------------------------------------------------------------------------
   # Creates the interface corresponding to this object.  This function is
   # called whenever "config-if" mode is entered (i.e. by the "interface"
   # command).
   #----------------------------------------------------------------------------
   def create( self, startupConfig=False ):
      self.createPhysical( startupConfig )

   #----------------------------------------------------------------------------
   # Destroys the interface corresponding to this object.  This function is
   # called by the "no interface" command, for virtual interfaces only.
   #----------------------------------------------------------------------------
   def destroy( self ):
      for cls in self.depClsRegistry_:
         cls( self, self.mode_ ).destroy()
      self.destroyPhysical()

   #----------------------------------------------------------------------------
   # If any comment exists for that interface, it will remove it.
   # This command is applicable to physical as well as virtual interfaces.
   #----------------------------------------------------------------------------
   def removeComments( self ):
      commentKey = "if-%s" % self.shortname_
      BasicCliModes.removeCommentWithKey( commentKey )

   #----------------------------------------------------------------------------
   # Sets the interface corresponding to this object to its default state.
   # This function is called by the "default interface" command. This command
   # is applicable to physical as well as virtual interfaces.
   #----------------------------------------------------------------------------
   def setDefault( self ):
      for cls in self.depClsRegistry_:
         cls( self, self.sysdbRoot_ ).setDefault()

   #----------------------------------------------------------------------------
   # Searches for the interface corresponding to this object, returning True if
   # the interface exists. Also check if the status has a config attached to
   # it because if it doesn't it means it is in a transitive state about to
   # be created or deleted so not visible.
   #----------------------------------------------------------------------------
   def lookup( self ):
      config = self.config() or self.dynConfig()
      return self.lookupPhysical() and config

   #---------------------------------------------------------------------------
   # return True if the default is "shutdown", False if "no shutdown"
   #---------------------------------------------------------------------------
   def defaultShutdownConfig( self ):
      return False

   #----------------------------------------------------------------------------
   # Returns the module number of the interface, or None if this is not a
   # physical interface.
   #----------------------------------------------------------------------------
   def module( self ):
      # match against interface names with 2 or more levels
      m = re.match( r'(\D+)(\d+)/(\d+)(?:/\d+)*$', self.name )
      if m is not None:
         return int( m.group( 2 ) )
      else:
         return None

   #----------------------------------------------------------------------------
   # Returns the slice ID associated with the interface, or None if this is not a
   # physical interface.
   #----------------------------------------------------------------------------
   def slotId( self ):
      return None

   #----------------------------------------------------------------------------
   # Returns a sorted list of Intf objects for all existing interfaces of any
   # type.  If intf is specified and it exists, returns a list containing intf.
   # If intf is specified and it does not exist, prints an error message and
   # returns None.  If mod is specified, restricts the output to physical
   # interfaces on that module. If intfType is specified, restricts the output
   # to interfaces of that type or subclasses of that type. If intfType is specified
   # as a list of types, restricts the output to interfaces of at least one type
   # in the list or subclasses of at least one type in the list. If config is
   # specified, only check that the interface config is present, and skip the
   # check entirely if it's a single interface. This allows non-existent single
   # interface to be accepted which is important for startup-config and CliSave
   # tests. It is expected the caller will call intf.create() afterwards. We
   # only do this for single interface to avoid accidentally creating a range
   # of interface by mistake.
   #----------------------------------------------------------------------------
   @staticmethod
   def getAll( mode, intf=None, mod=None, intfType=None, silent=False,
               config=False, exposeInactive=False, exposeUnconnected=False,
               exposeInternal=False, exposeOnlyUserVisible=True, sort=True ):
      def getCliIntfs( intfs ):
         '''Create CliIntf objects for each* interface.
         If we are configuring, assume all interfaces are valid.
         Otherwise, cross-reference with current (sub)intf config.
         Yield, instead of building large collections.'''
         CliIntf = functools.partial( intfs.type().getCliIntf, mode )
         if config:
            for i in intfs:
               yield CliIntf( i )
         else:
            existing = set( allIntfConfigDir.intfConfig )
            existing.update( allIntfStatusDir.intfStatus )
            for i in intfs:
               if i in existing:
                  yield CliIntf( i )
               else:
                  hasNonexistentIntfs[ 0 ] = True

      if intfType:
         try: # Try as iterable. A non-iterable `intfType` will raise `TypeError`.
            assert all( issubclass( t, Intf ) for t in intfType )
         except TypeError: # Try as a type. A non-type `Intf` will raise `TypeError`.
            assert issubclass( intfType, Intf )
      hasNonexistentIntfs = [ False ] # This needs to be a list for pass-by-ref.
      intfs = []
      exposeInactive = exposeInactive or globalIntfConfig.exposeInactiveLanes
      exposeUnconnected = ( exposeUnconnected or
                            globalIntfConfig.exposeUnconnectedLanes )
      exposeInternal = ( exposeInternal or
                         globalIntfConfig.exposeInternalInterfaces )
      if intf is not None:
         if isinstance( intf, MultiRangeRule.IntfList ):
            possibleIntfs = getCliIntfs( intf )

            # We need to find out if this is a single interface.
            # Generators have no `len`, but we can use `tee` to peek.
            possibleIntfs, peek = itertools.tee( possibleIntfs, 2 )
            if next( peek, None ) is None: # No interfaces at all.
               if silent:
                  return []
               raise UnknownEntityError( "Interface does not exist" )

            isSingleIntf = next( peek, None ) is None
         else:
            possibleIntfs = [ intf ]
            isSingleIntf = True

         for i in possibleIntfs:
            intfConfig = i.config() or i.dynConfig()
            if intfType and not isinstance( i, intfType ):
               err = 'Interface is not of appropriate type for this operation'
               mode.addError( err )
               return []

            if config:
               if not isSingleIntf:
                  # For config command, we only make sure the interface
                  # config exists because we don't care about status.
                  if not intfConfig:
                     hasNonexistentIntfs[ 0 ] = True
                     continue
            elif not i.lookup():
               # check both config and status
               hasNonexistentIntfs[ 0 ] = True
               continue

            # Interfaces that do not have configs are exempt from filtering
            if i.config() is None:
               intfs.append( i )
               continue

            if not i.userVisible() and exposeOnlyUserVisible:
               continue

            if not i.active() and not exposeInactive:
               continue

            if i.unconnected() and not exposeUnconnected:
               continue

            intfs.append( i )
      else:
         for clazz in Intf.physicalIntfClasses_:
            # If a type list has been provided by the caller then filter out all
            # interface types that do not inherit from the types in the provided type
            # list.
            if intfType and not issubclass( clazz, intfType ):
               continue

            for theIntf in clazz.getAllPhysical( mode ):
               if theIntf.internal() and not exposeInternal:
                  continue

               # Configurable interfaces that do not have configs are exempt from
               # the rest of the filtering.
               if theIntf.config() is None:
                  intfs.append( theIntf )
                  continue

               if not theIntf.userVisible() and exposeOnlyUserVisible:
                  continue

               if not theIntf.active() and not exposeInactive:
                  continue

               if theIntf.unconnected() and not exposeUnconnected:
                  continue

               intfs.append( theIntf )


      if mod is not None:
         intfs = [ i for i in intfs if i.module() == mod ]

      # we pay a penalty for sorting large nubmer of interfaces, we should only do
      # if our caller cares about this
      if sort:
         intfs.sort()

      if hasNonexistentIntfs[ 0 ] and not silent:
         if not intfs: # pylint: disable=no-else-raise
            raise UnknownEntityError( "Interface does not exist" )
         else:
            mode.addWarning( "Some interfaces do not exist" )
      # Set the counterFreshnessTime on all the selected interfaces
      # to the same value. For agents that do batch prefetching of counters
      # this would avoid duplicate prefetches when Cli takes some time to step
      # through the interfaces and finds the already prefetched counter to be
      # stale
      counterFreshnessTime = Tac.now() - 1
      for intf in intfs: # pylint: disable=redefined-argument-from-local
         intf.counterFreshnessTime = counterFreshnessTime
      return intfs

   #----------------------------------------------------------------------------
   # Prints information about this interface, or throws an exception if it does
   # not exist.  This function is called by the "show interfaces" command.
   #
   # This function prints a common header but then delegates the rest of the
   # output to the subclass via the pure virtual 'showPhysical' function.
   #----------------------------------------------------------------------------
   def show( self, mode ):
      intfStatusModel = self.getIntfStatusModel()
      self.populateBaseModel( intfStatusModel )
      self.showPhysical( mode, intfStatusModel )
      return intfStatusModel

   #----------------------------------------------------------------------------
   # Prints information about this interface's configuration and any config
   # errors. This is implemented per-interface by overriding the
   # getIntfConfigSanityModel function to return a custom model extending
   # InterfaceConfigSaniy. The default model prints the intf is unsupported.
   #----------------------------------------------------------------------------
   def showConfigSanity( self, mode ):
      intfConfigSanityModel = self.getIntfConfigSanityModel()
      self.populateBaseModel( intfConfigSanityModel )
      return intfConfigSanityModel

   def populateBaseModel( self, intfStatusModel ):
      intfStatusModel.forwardingModel = self.forwardingModel()
      ( intfStatusModel.lineProtocolStatus,
        intfStatusModel.interfaceStatus ) = self.getStatus()
      intfStatusModel.hardware = self.hardware()
      ( intfStatusModel.physicalAddress,
        intfStatusModel.burnedInAddress ) = self.physicalAddress()
      intfStatusModel.description = self.description().strip()
      intfStatusModel.bandwidth = self.bandwidth() * 1000
      intfStatusModel.l3MtuConfigured = self.l3MtuConfigured()
      intfStatusModel.mtu = self.mtu()
      intfStatusModel.lastStatusChangeTimestamp = self.uptime()
      addrList = self.showNetworkAddr()
      while addrList:
         addr = addrList.pop()
         if isinstance( addr, IntfModel.InterfaceAddressIp6Base ):
            if not intfStatusModel.interfaceAddressIp6:
               intfStatusModel.interfaceAddressIp6 = addr
            else:
               raise RuntimeError( "multiple IPv6 address CLI providers" )
         elif isinstance( addr, IntfModel.InterfaceAddressIp4Base):
            if not intfStatusModel.interfaceAddress:
               intfStatusModel.interfaceAddress = [ addr ]
            else:
               raise RuntimeError( "multiple IPv4 address CLI providers" )

      for hook in IntfModel.interfaceInfoHook.extensions():
         hook( self.name, intfStatusModel )

   def forwardingModel( self ):
      return _forwardingModelMap[ self.status().forwardingModel ]

   def getStatus( self ):
      conf = self.config() or self.dynConfig()
      stat = self.status()
      lineProtocolStatus = self.lineProtocolState()
      if not conf:
         interfaceStatus = "uninitialized"
      elif lineProtocolStatus == "notApplicable":
         interfaceStatus = "notApplicable"
      elif not stat.active:
         interfaceStatus = "inactive"
      elif not conf.adminEnabled:
         interfaceStatus = "disabled"
      elif not conf.enabled or lineProtocolStatus != "up":
         interfaceStatus = conf.enabledStateReason or "notconnect"
      else:
         interfaceStatus = "connected"

      return ( lineProtocolStatus, interfaceStatus.lower() )

   def physicalAddress( self ):
      address = self.addrStr()
      if address is None:
         return ( None, None )

      quad = '[0-9a-f]{4}'
      pattern = fr'({quad}\.{quad}\.{quad})'
      m = re.search( fr'address is {pattern}(?: \(bia {pattern}\))?',
                     address )
      physAddr, biAddr = m.groups()
      return ( physAddr, biAddr )

   #----------------------------------------------------------------------------
   # Sets the MAC address of the interface to a specified value or the default
   # value.
   #----------------------------------------------------------------------------
   def setMacAddr( self, addr, routerMacConfigured=False, physical=False ):
      if addr and Ethernet.convertMacAddrToDisplay( addr ) == '0000.0000.0000':
         # setting to 0.0.0 is equivalent to clearing the config
         addr = None
      if physical:
         self.config().addr = '00:00:00:00:00:00'
         self.config().routerMacConfigured = True
      elif addr is None:
         self.config().addr = '00:00:00:00:00:00'
         self.config().routerMacConfigured = False
      elif Ethernet.isUnicast( addr ):
         self.config().addr = addr
         self.config().routerMacConfigured = routerMacConfigured
      else:
         self.mode_.addError( 'Cannot set the interface MAC address to a '
                              'multicast address' )

   def l3MtuConfigured( self ):
      if not self.config():
         return False
      return self.config().l3MtuConfigured

   def mtu( self ):
      return self.status().mtu

   def uptime( self ):
      if self.status().operStatusChange.time == 0:
         return None

      return self.status().operStatusChange.time + Tac.utcNow() - Tac.now()

   def showNetworkAddr( self ):
      """If available, display the L3 network address in show interface
      command."""

      addresses = []
      for cls in self.depClsRegistry_:
         if hasattr( cls, 'showNetworkAddr' ):
            intfDependent = cls( self, self.sysdbRoot_ )
            if addr := intfDependent.showNetworkAddr():
               addresses.append( addr )
      return addresses

   #----------------------------------------------------------------------------
   # Returns the configured interface description.
   #----------------------------------------------------------------------------
   def description( self ):
      config = self.config() or self.dynConfig()
      return config.description if config else ""

   def prettyIntfStateStrings( self ):
      """ Return tuple of four values suitable for pretty-printing
      line status in various places in the CLI output. The first
      two items in the tuple fill in the blanks:
         Ethernet1 is ___, line protocol is ___
      The third and fourth items are used in 'show int description'.
      """
      lps = self.lineProtocolState().lower()
      conf = self.config() or self.dynConfig()
      if not conf:
         # There was no Intfconfig object
         return ('down', '%s (uninitialized)' % lps, 'uninit down', lps)
      if not conf.adminEnabled:
         return ('administratively down', '%s (disabled)' % lps, 'admin down', lps)
      elif not conf.enabled or lps != 'up':
         downReasonString = conf.enabledStateReason or 'notconnect'
         return ( 'down', f'{lps} ({downReasonString})', 'down', lps )
      else:
         return ('up', '%s (connected)' % lps, 'up', lps)

   def getIntfState( self ):
      lps = self.lineProtocolState()
      conf = self.config() or self.dynConfig()
      if not conf:
         # There was no Intfconfig object
         return 'uninitDown'
      elif lps == 'notApplicable':
         return 'notApplicable'
      elif not conf.adminEnabled:
         return 'adminDown'
      elif not conf.enabled or lps != 'up':
         return 'down'
      else:
         return 'up'

   def lineProtocolApplicable( self ):
      return True

   def lineProtocolState( self ):
      if not self.lineProtocolApplicable():
         return 'notApplicable'
      operStatus = self.status().operStatus
      # Strip off 'intfOper' and return the rest of the enum name
      return operStatus[8].lower() + operStatus[9:]

   #----------------------------------------------------------------------------
   # Overloads the rich comparison operators so that Intf objects have value
   # semantics and are sorted by name.
   #----------------------------------------------------------------------------
   def __lt__( self, other ):
      if isinstance( other, Intf ):
         return Arnet.intfNameKey( self.name_ ) < Arnet.intfNameKey( other.name_ )
      else:
         return NotImplemented

   def __eq__( self, other ):
      if isinstance( other, Intf ):
         return Arnet.intfNameKey( self.name_ ) == Arnet.intfNameKey( other.name_ )
      else:
         return NotImplemented

   def __hash__( self ):
      return hash( self.name_ )

   #----------------------------------------------------------------------------
   # Hooks/support for "clear counters".
   # See TestIntfCli for examples of what the derived class is supposed to do.
   #----------------------------------------------------------------------------

   #----------------------------------------------------------------------------
   def counter( self ):
      '''Returns the current IntfCounter object for this interface.'''
      if not self.intfCounter:
         if not self.countersSupported():
            return None
         status = self.status()
         if not status:
            return None
         self.intfCounter = status.counter
         if not self.intfCounter:
            qt8( "Intf.counter-", qv( self.name ), "counter is None" )
            return None
         currentTime = Tac.now()
         # For agents that support counter update i.e counterRefreshTime is non-
         # zero, if the counter is older than the counterFreshnessTime then request
         # an update by setting the counterUpdateReqTime and wait till the agent
         # updates the counter and sets the counterRefreshTime to the time when it
         # is done updating counter
         if self.intfCounter.counterRefreshTime and \
            self.intfCounter.counterRefreshTime < self.counterFreshnessTime:
            self.intfCounter.counterUpdateReqTime = self.counterFreshnessTime
            try:
               Tac.waitFor(
                      lambda: self.intfCounter.counterRefreshTime >=
                              self.counterFreshnessTime,
                      timeout=1, warnAfter=0, maxDelay=0.1, description="",
                      sleep=True )
            except Tac.Timeout :
               self.mode_.addWarning(
                  "Timeout getting fresh counters (start: %f, timeout: %f)" %
                  ( currentTime, Tac.now() ) )

      return self.intfCounter

   def dpCounters( self ):
      return None

   def getLatestDpCounterCheckpoint( self ):
      return None

   #----------------------------------------------------------------------------
   # return <intfType>IntfCounterDir must be implemented by subclass
   #----------------------------------------------------------------------------
   def getIntfCounterDir( self ):
      '''returns <intfType>IntfCounterDir
      Must be implemented by subclass'''
      raise NotImplementedError()

   def getCounterCheckpoint( self, className=None, sessionOnly=False ):
      '''Depending on sessionOnly, return the global or session counter checkpoint.
      If it doesn't exist only create it and the underlying directory structure
      only if a className is passed'''
      if sessionOnly:
         y = _sessionCounterDir().get( self.name )
         if not y:
            if not className:
               return None
            y = Tac.newInstance( "%sDir" % (className), self.name )
            _sessionCounterDir()[ self.name ] = y
      else:
         x = self.getIntfCounterDir()
         if x is None:
            return None
         x = getattr( x, 'checkpointCounterDir', x )
         y = x.intfCounterDir.get( self.name )
         if not y:
            if not className:
               return None
            qt8( "Intf.getCounterCheckpoint-", qv( self.name ), "y is None" )
            y = x.newIntfCounterDir( self.name )

      cc = y.intfCounter.get( 'lastClear' )
      # Get the status of the interface to access it's genId
      status = self.status()

      # Lookup of an existing checkpoint, only return if the genId is the same
      if not className:
         if cc and cc.genId == status.genId:
            return cc
         return None

      # Create new one if it doesn't exist or has wrong genId
      if not cc or cc.genId != status.genId:
         if cc:
            del y.intfCounter[ 'lastClear' ]
         cc = y.newIntfCounter( 'lastClear' )
         cc.genId = status.genId
      return cc

   def getLatestCounterCheckpoint( self ):
      '''Get the latest counter checkpoint wheter global or for this session only'''

      ccGlobal = self.getCounterCheckpoint( sessionOnly=False )
      ccSession = self.getCounterCheckpoint( sessionOnly=True )
      if ccGlobal and ccSession:
         # both there,  compare timestamps
         if ccGlobal.rates.statsUpdateTime > ccSession.rates.statsUpdateTime:
            return ccGlobal
         else:
            return ccSession
      else:
         # at least one not there, return one if any that is there
         if ccGlobal:
            return ccGlobal
         else:
            return ccSession

   def getLatestVniCounterCheckpoint( self ):
      # This function must be implemented by a VxlanIntf derived class
      pass

   def clearCounters( self, sessionOnly=False ):
      if self.countersSupported():
         ckpt = self.getCounterCheckpoint( 'Interface::IntfCounter',
                                           sessionOnly=sessionOnly)
         if ckpt is None or self.counter() is None:
            return
         ckpt.statistics = self.counter().statistics
         # Because (current) counters are updated periodically (and
         # statsUpdateTime records latest sample time) and show
         # counter shows counters gathered in latest periodic sample,
         # statsUpdateTime can be saved in ckpt as is (from latest
         # sample). It can still be used to compare global or session
         # chkpt, because: if both session and global clear happen
         # within same sample window, then the chkpt value would be
         # same for both and either can be used later, otherwise (they
         # happens in different sample window), then
         # chkpt.statsUpdateTime can be used correctly. This also make
         # rates calculation to be accurate after clear counters.
         ckpt.rates = self.counter().rates
         # Update the value of linkStatusChanges.
         ckpt.linkStatusChanges = self.status().operStatusChange.count

   def storeLastCounters( self, mode, className=None ):
      z = _deltaCounterDir( mode ).get( self.name )
      if z is None:
         if className is None:
            z = Tac.newInstance( "Interface::IntfCounterDir", self.name )
         else:
            z = Tac.newInstance( "%sDir" % (className), self.name )

      y = z.intfCounter.get( self.name )
      if y is None:
         y = z.newIntfCounter( self.name )

      _deltaCounterDir( mode )[ self.name ] = z

      return y

   def updateLastCounters( self, mode ):
      lastCounter = self.storeLastCounters( mode )
      counter = self.counter()
      if counter:
         lastCounter.statistics = counter.statistics

   def updateLastVniCounters( self, mode ):
      # This function must be implemented by a VxlanIntf derived class
      pass

   def countersSupported( self ):
      return True

   def dpCountersSupported( self ):
      return False

   def countersErrorsSupported( self ):
      return False

   def countersRateSupported( self ):
      return self.countersSupported()

   def countersDiscardSupported( self ):
      return self.countersSupported()

   def perVniCountersSupported( self ):
      return False

   def perVniCountersRateSupported( self ):
      return False

   def updateInterfaceCountersRateModel( self ):
      return None

   def getIntfStatusModel( self ):
      return IntfModel.InterfaceStatus( name=self.status().intfId )

   def getIntfConfigSanityModel( self ):
      return IntfModel.InterfaceConfigSanity( name=self.status().intfId )

   def getUpdateTime( self, checkpoint ):
      currentTime = self.counter().statistics.lastUpdate

      # when counters have been cleared, return zero until they have been updated
      if checkpoint and checkpoint.statistics.lastUpdate == currentTime:
         return 0.0

      # convert the 'uptime' to UTC
      return Ark.switchTimeToUtc( currentTime )

   def updateInterfaceCountersModel( self, checkpoint=None, notFoundString=None,
                                     zeroOut=False ):
      intfCountersModel = IntfModel.InterfaceCounters( _name=self.status().intfId )
      if not zeroOut:
         stat = captureCounterState( counter=self.counter(),
                                     checkpoint=checkpoint,
                                     notFoundString=notFoundString )
         intfCountersModel.inOctets = stat( 'inOctets' )
         intfCountersModel.inUcastPkts = stat( 'inUcastPkts' )
         intfCountersModel.inMulticastPkts = stat( 'inMulticastPkts' )
         intfCountersModel.inBroadcastPkts = stat( 'inBroadcastPkts' )
         intfCountersModel.inDiscards = stat( 'inDiscards' )
         intfCountersModel.outOctets = stat( 'outOctets' )
         intfCountersModel.outUcastPkts = stat( 'outUcastPkts' )
         intfCountersModel.outMulticastPkts = stat( 'outMulticastPkts' )
         intfCountersModel.outBroadcastPkts = stat( 'outBroadcastPkts' )
         intfCountersModel.outDiscards = stat( 'outDiscards' )
         # NOTE: do NOT use 'stat()' since last update time should never be a delta
         intfCountersModel.lastUpdateTimestamp = self.getUpdateTime( checkpoint )
      else:
         intfCountersModel.inOctets = 0
         intfCountersModel.inUcastPkts = 0
         intfCountersModel.inMulticastPkts = 0
         intfCountersModel.inBroadcastPkts = 0
         intfCountersModel.inDiscards = 0
         intfCountersModel.outOctets = 0
         intfCountersModel.outUcastPkts = 0
         intfCountersModel.outMulticastPkts = 0
         intfCountersModel.outBroadcastPkts = 0
         intfCountersModel.outDiscards = 0
         intfCountersModel.lastUpdateTimestamp = 0.0

      return intfCountersModel

   def updateVxlanInterfaceCountersModel( self, checkpoint=None, notFoundString=None,
                                          zeroOut=False, vni=None ):
      # This function must be implemented by a VxlanIntf derived class
      return None

   def routingSupported( self ):
      # This method indicates whether or not routing may be supported
      # on the interface i.e whether to allow routing related commands
      # on the interface or not.
      return True

   def routingCurrentlySupported( self ):
      # This method indicates whether or not routing is currently supported
      # on the interface i.e whether or not the interface appears in the output
      # of "show" commands.
      intfStatus = allIntfStatusDir.intfStatus.get( self.name )
      if intfStatus and intfStatus.forwardingModel == "intfForwardingModelRouted":
         return True
      return False

   def vrfSupported( self ):
      # Whether we allow 'vrf forwarding' command on the interface.
      return self.routingSupported()

   def isUnicastInetAddress( self, a1 ):
      # Check for non-unicast.

      if IpAddrMatcher.validateMulticastIpAddr( a1.address ) is None:
         return (False, "IP address must be unicast" )

      elif  Arnet.Subnet( a1 ).isBroadcastAddress():
         return (False, "IP address must not be broadcast" )

      elif Arnet.Subnet( a1 ).isAllZerosAddress():

         return (False, "IP address must not have a zero host number" )

      elif IpAddrMatcher.isMartianIpAddr( a1.address ):

         return ( False, "Not a valid host address" )

      elif IpAddrMatcher.isInvalidThisNetworkAddress( a1 ):

         return ( False, "Not a valid local router address or subnet mask" )

      else:
         return ( True, "" )

   def isValidHostInetAddress( self, a1 ):
      if a1.len >= 32:
         return ( False, "Prefix length must be less than 32" )
      elif IpAddrMatcher.isLoopbackIpAddr( a1.address ):
         return ( False, "IP address must not be loopback" )
      else:
         return self.isUnicastInetAddress( a1 )

   def isUnicastInet6Address( self, a1 ):
      # Check for non-unicast.
      if a1.address.isMulticast:
         return ( False, "Address must be unicast" )
      if a1.address.isUnspecified:
         return ( False, "Invalid unicast address" )
      if a1.address.isLoopback:
         # Loopback address can be configured ONLY on lo. But lo is
         # not configurable thru EOS CLI. Reject it.
         return ( False, "Invalid address for this interface" )
      return ( True, "" )

   def isValidHostInet6Address( self, a1 ):
      if a1.len >= 128:
         return ( False, "Prefix length must be less than 128" )

      return self.isUnicastInet6Address( a1 )

   #----------------------------------------------------------------------------
   # Only a subinterface will return True
   #----------------------------------------------------------------------------
   def isSubIntf( self ):
      return False

   #----------------------------------------------------------------------------
   # Only an interface that can be created dynamically will return a dyn config
   #----------------------------------------------------------------------------
   def dynConfig( self ):
      return None

   def active( self ):
      intfConfig = self.config() or self.dynConfig()
      if intfConfig is None:
         return False

      return intfConfig.enabledStateReason != 'inactive'

   def unconnected( self ):
      intfConfig = self.config() or self.dynConfig()
      if intfConfig is None:
         return False

      return intfConfig.prettyName.startswith( ( 'Un', 'Ue' ) )

   #----------------------------------------------------------------------------
   # Method stubs to be defined in child classes
   #----------------------------------------------------------------------------

   def createPhysical( self, startupConfig=False ):
      raise NotImplementedError()

   def destroyPhysical( self ):
      raise NotImplementedError()

   def lookupPhysical( self ):
      raise NotImplementedError()

   def showPhysical( self, mode, intfStatusModel ):
      raise NotImplementedError()

   def config( self ):
      raise NotImplementedError()

   def status( self ):
      raise NotImplementedError()

   def bandwidth( self ):
      raise NotImplementedError()

   def hardware( self ):
      raise NotImplementedError()

   def addrStr( self ):
      raise NotImplementedError()

class VirtualIntf( Intf ):
   # This is an abstract class, so don't warn about abstract inherited
   # methods
   # pylint: disable-msg=W0223
   def noInterface( self ):
      self.destroy()

class VxlanVirtualIntf( VirtualIntf ):
   # This is an abstract class for vxlan interface cli
   # pylint: disable-msg=W0223
   def noInterface( self ):
      self.destroy()

   def getLatestVniCounterCheckpoint( self ):
      # This function must be implemented by a VxlanIntf derived class
      raise NotImplementedError()

   def updateLastVniCounters( self, mode ):
      # This function must be implemented by a VxlanIntf derived class
      raise NotImplementedError()

   def updateVxlanInterfaceCountersModel( self, checkpoint=None, notFoundString=None,
                                          zeroOut=False, vni=None ):
      # This function must be implemented by a VxlanIntf derived class
      raise NotImplementedError()

# canShutdownIfHook extensions accept one argument: a list of
# IntfConfigMode objects on which 'shutdown' is being called, and
# returns True if the interfaces can be shutdown and False otherwise.
canShutdownIfHook = CliExtensions.CliHook()

# canNoShutdownIfHook extensions accept one argument: a list of
# IntfConfigMode objects on which 'no shutdown' is being called, and
# returns True if the interfaces can be no shutdown and False otherwise.
canNoShutdownIfHook = CliExtensions.CliHook()

# shutdownIfHook extensions accept one argument: a list of
# IntfConfigMode objects on which 'shutdown' is being called. This
# extension provides a callback to VirtualIntfs so that they can do
# housekeeping when interface is being shutdown or brought back up
shutdownIfHook = CliExtensions.CliHook()

class HiddenIntfFilterMatcher( CliMatcher.Matcher ):
   """This is a rule that filters Intf.matcher with specified tags.
   It's not for general use. Instead, it's used as a Hidden matcher
   by intfRangeWithSingleExpression() so it can parse an interface with
   specified tags at startup-config time when the interface has
   not been created."""
   __slots__ = ( 'matcher_', 'tags_' )
   def __init__( self, matcher, tags, priority=None ):
      # by default use the underlying matcher's priority and value function
      self.matcher_ = matcher
      self.tags_ = tags
      CliMatcher.Matcher.__init__( self, priority=CliParser.PRIO_LOW )

   def match( self, mode, context, token ):
      if self.tags_ and context.state is None:
         # first token
         m = IntfRange.intfTagRe_.match( token )
         if m and m.group( 1 ) not in self.tags_:
            # We have a tag that is not allowed
            return CliParser.noMatch
      return self.matcher_.match( mode, context, token )

   def completions( self, mode, context, token ):
      # We are hidden, so no completion
      return []

   def valueFunction( self, context ):
      return self.matcher_.valueFunction( context )

def intfRangeWithSingleExpression( name, explicitIntfTypes=None ):
   """
   WARNING: it's probably NOT what you want to use!

   One weakness of IntfRangeMatcher is that cannot match physical interfaces that
   do not have an interface config, such as Ethernet. If you have a config command
   with interface range for physical interfaces, it needs to come after interface
   modes.

   There are a couple of commands that do not do exactly this.
   1. The interface command itself. We use this expression so we could parse single
      physical interface and create the interface config.
   2. Some IGMP snooping commands take an interface range but outputs individual
      interfaces in running-config, and it comes before interface mode (not sure if
      it's a good idea to change it now).
   3. Danz commands that are applied to an interface but take another interface as a
      parameter. For example, `switchport tap default interface et2` will cause
      issues on reload because the interface configs have not been created yet.

   In these cases, we add a hidden single-interface matcher (IntfCli.Intf.matcher)
   so we could parse it.

   The value of this rule is either a MultiRangeRule.IntfList or a single
   Intf object. The Cli command handler should pass the value to
   Intf.getAll() to return a list of interfaces. If getAll() returns None,
   no interfaces are available.
   """
   tags = ( { x.tagLong for x in explicitIntfTypes } if explicitIntfTypes
            else None )
   nameSingle = name + '_SINGLE'
   nameRange = name + '_RANGE'
   class IntfRangeConfigExpression( CliCommand.CliExpression ):
      expression = nameSingle + " | " + nameRange
      data = {
         nameSingle : HiddenIntfFilterMatcher( Intf.matcher, tags ),
         nameRange : IntfRange.IntfRangeMatcher(
            explicitIntfTypes=explicitIntfTypes,
            earlySingletonReject=True )
      }
      @staticmethod
      def adapter( mode, args, argsList ):
         intf = args.get( nameSingle ) or args.get( nameRange )
         if intf:
            args[ name ] = intf
   return IntfRangeConfigExpression

#-------------------------------------------------------------------------------
# The "config-if" mode.
#-------------------------------------------------------------------------------
class IntfConfigModeDependent( IntfDependentBase ):
   def setDefault( self ):
      IntfConfigMode.setDefault( self.intf_ )

class IntfConfigMode( IntfMode, BasicCli.ConfigModeBase ):

   #----------------------------------------------------------------------------
   # Attributes required of every Mode class.
   #----------------------------------------------------------------------------
   name = 'Interface configuration'
   nameRe = re.compile( '([A-Za-z]+)' )

   #----------------------------------------------------------------------------
   # Constructs a new IntfConfigMode instance for a particular Intf object.
   #
   # Note that the intf attribute must be set up before calling the superclass
   # constructor, since the Modelet instances created by the superclass
   # constructor need to use it.
   #----------------------------------------------------------------------------
   def __init__( self, parent, session, intf ):
      self.intf = intf
      self.intfRangeRef = None
      IntfMode.__init__( self, self.intf.name, '%s' % self.intf.shortname )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def instanceRuleKey( self ):
      # Share instance rules for interfaces of the same type
      #
      # We use interface tag + class name, because
      # 1. Same class name can refer to different types (EthPhyIntf)
      # 2. Same tag can refer to different types (ma0 and ma1)
      m = self.nameRe.match( self.intf.name )
      tag = m.group( 1 )
      return self.intf.__class__.__name__ + ':' + tag

   @staticmethod
   def setDefault( intf ):
      intf.config().adminEnabled = True
      if ( intf.name.startswith( "Ethernet" ) and
           globalIntfConfig.defaultEthernetShutdown ):
         intf.config().adminEnabled = False
      intf.config().description = ''

      intf.config().linkStatusLogging = IntfConfigMode.defaultLoggingLinkStatus(
         intf )

      intf.config().loadInterval = defaultLoadInterval()
      intf.removeComments()

   # Where available the default interface configuration should be used
   # to set the default link status logging mode. Where this is not available
   # we have to fallback on "useGlobal" as this is the legacy approach.
   #
   # TODO BUG892322: This technically introduces a race on card initialization
   #                 if the user defaults a fabric interface before FRU has
   #                 processed the card and populated the default configuration.
   #                 This should be acceptable because fabric interfaces are the only
   #                 interfaces which diverge from "useGlobal", they are also only
   #                 ever configurable on fixed systems meaning that the possibility
   #                 of a race are miniscule. This bug will have to be fixed before
   #                 configurable fabric interfaces get enabled on modular systems.
   @staticmethod
   def defaultLoggingLinkStatus( intf ):
      if defaultIntfConfig := intf.config().defaultConfig:
         return defaultIntfConfig.linkStatusLogging
      else:
         return 'useGlobal'

   def maybeIntfRangeCliHook( self, hook, hookName, hookArgsCallback=None ):
      ''' If this intfMode is part of a range, call
      IntfRangeConfigMode.intfRangeHook and return the
      result. Otherwise, just call the hook and return the result. See
      the IntfRangeConfigMode.intfRangeHook docstring for a
      description of the arguments.'''
      intfRange = self.intfRange()
      if intfRange is not None:
         return intfRange.intfRangeCliHook( self, hook, hookName,
                                            hookArgsCallback )
      else:
         hookArgs = [ [self] ]
         if hookArgsCallback:
            hookArgsCallback( hookArgs )
         return hook( *hookArgs )

   def setShutdown( self, no=False ):
      if no:
         hooks = canNoShutdownIfHook.extensions()
         hookName = 'canNoShutdownHook'
      else:
         hooks = canShutdownIfHook.extensions()
         hookName = 'canShutdownHook'

      if self.session.commandConfirmation():
         for hook in hooks:
            if not self.maybeIntfRangeCliHook( hook, hookName ):
               return

      self.intf.config().adminEnabled = no

      for shutdownHook in shutdownIfHook.extensions():
         self.maybeIntfRangeCliHook( shutdownHook, 'shutdownHook' )

   def setDescription( self, desc ):
      self.intf.config().description = desc

   def noDescription( self ):
      self.intf.config().description = ''

   def setLoadInterval( self, interval ):
      self.intf.config().loadInterval = newLoadInterval( interval )

   def noLoadInterval( self ):
      self.intf.config().loadInterval = defaultLoadInterval()

   def setLoggingLinkStatus( self ):
      self.intf.config().linkStatusLogging = 'on'

   def setLoggingLinkStatusUseGlobal( self ):
      self.intf.config().linkStatusLogging = 'useGlobal'

   def noLoggingLinkStatus( self ):
      self.intf.config().linkStatusLogging = 'off'

   def intfRangeIs( self, obj ):
      self.intfRangeRef = weakref.ref(obj)

   def intfRange( self ):
      return self.intfRangeRef() if self.intfRangeRef else None


interfaceKwMatcher = CliMatcher.KeywordMatcher( 'interface',
                                   helpdesc='Interface Configuration' )
intfAfterServiceKwMatcher = CliMatcher.KeywordMatcher( 'interface',
                                   helpdesc='Change interface related parameters' )

class InterfaceDefaultsConfigMode( CliMode.InterfaceDefaults.InterfaceDefaultsMode,
                                   BasicCli.ConfigModeBase ):
   name = "Interface Defaults Configuration"

   def __init__( self, parent, session ):
      CliMode.InterfaceDefaults.InterfaceDefaultsMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class IntfDefaults( CliCommand.CliCommandClass ):
   syntax = "interface defaults"
   noOrDefaultSyntax = syntax
   data = {
      "interface" : interfaceKwMatcher,
      "defaults" : ( "Set attribute values to use in absence of specific "
                     "configuration (user configurable defaults)" )
      }

   @staticmethod
   def handler( mode, args ):
      newMode = mode.childMode( InterfaceDefaultsConfigMode )
      mode.session_.gotoChildMode( newMode )

   noOrDefaultHandler = InterfaceDefaultsConfigMode.clear

BasicCli.GlobalConfigMode.addCommandClass( IntfDefaults )

#-------------------------------------------------------------------------------
# The interface-defaults mode "[no] mtu" command.
#-------------------------------------------------------------------------------
class GlobalL3MtuCmd( CliCommand.CliCommandClass ):
   syntax = "mtu MTU"
   noOrDefaultSyntax = "mtu ..."
   data = {
           'mtu' : 'Set default MTU for all Layer 3 interfaces',
           'MTU' : CliMatcher.IntegerMatcher( 68, 65535,
              helpdesc='Maximum transmission unit in bytes' ),
   }

   @staticmethod
   def handler( mode, args ):
      globalL3Mtu = args[ 'MTU' ]
      globalIntfConfig.l3Mtu = globalL3Mtu

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      globalIntfConfig.l3Mtu = globalIntfConfig.l3MtuDefault

InterfaceDefaultsConfigMode.addCommandClass( GlobalL3MtuCmd )

#-------------------------------------------------------------------------------
# The "show interfaces [<name>]" command, in enable mode.
#  show interfaces [<name>]
#  show interfaces module <modnum>
#-------------------------------------------------------------------------------
moduleExpressionFactory_ = None

def registerModularExpression( moduleExpr ):
   global moduleExpressionFactory_
   moduleExpressionFactory_ = moduleExpr

# This is a wrapper expression factory that can be used in ShowIntfCommand
# as moduleExpressionFactory_ is None initially and cannot be used directly.
class ModuleWrapperFactory( CliCommand.CliExpressionFactory ):
   def generate( self, name ):
      if moduleExpressionFactory_:
         return moduleExpressionFactory_.generate( name )
      else:
         # Nobody set moduleExpressionFactory. Just return an invalid expression.
         class NullExpr( CliCommand.CliExpression ):
            expression = "INVALID"
            data = dict( INVALID=CliCommand.Node( CliMatcher.KeywordMatcher(
               "<INVALID>", helpdesc="INVALID" ), hidden=True ) )
         return NullExpr

moduleWrapperFactory = ModuleWrapperFactory()

interfacesShowKw = CliMatcher.KeywordMatcher(
   'interfaces', helpdesc='Details related to interfaces' )

interfacesShowAllKw = CliMatcher.KeywordMatcher(
   'all', helpdesc='Include internal interfaces' )

class ShowIntfCommand( ShowCommand.ShowCliCommandClass ):
   '''Defines a base class for all "show interfaces" commands.

   Basic Syntax:
      show interface [ INTF | module MOD ] EXTRA_SYNTAX

   The subclass may customize this syntax using the following controls:
      allowAllSyntax: When set to True, this adds a token which can be used to
                      describe interfaces: 'all'. This modifies the command syntax to
                      look like:
                      'show interface [ INTF | all | module MOD ] EXTRA_SYNTAX'
      moduleAtEnd: When set True, this allows for the ' module MOD ' to be specified
                   at the end of the command:
                   'show interface EXTRA_SYNTAX [ module MOD ]'
   '''
   allowAllSyntax = False
   moduleAtEnd = False

   baseSyntax_ = 'show interfaces [ INTF {allSyntax}| MOD ] {extraSyntax}'
   baseSyntaxWithModuleAtEnd_ = 'show interfaces ( [ INTF {allSyntax}| MOD ] '\
                                '{extraSyntax} ) | ( {extraSyntax} MOD )'

   baseData_ = {
      'interfaces' : interfacesShowKw,
      'INTF' : Intf.rangeMatcher,
      'MOD' : moduleWrapperFactory,
   }
   allSyntaxData_ = {
      'all' : interfacesShowAllKw,
   }

   ALLOWED_FIELDS = tuple( list( ShowCommand.ShowCliCommandClass.ALLOWED_FIELDS ) +
                           [ 'moduleAtEnd', 'allowAllSyntax' ] )
   dynamicSyntax_ = True

   @classmethod
   def _generateSyntaxAndData( cls ):
      data = {}
      baseSyntax = cls.baseSyntax_
      data.update( cls.baseData_ )

      if cls.moduleAtEnd:
         baseSyntax = cls.baseSyntaxWithModuleAtEnd_

      allSyntax = ''
      if cls.allowAllSyntax:
         cls._assert( 'all' not in cls.syntax,
                      '\'all\' syntax is automatically generated' )
         allSyntax = '| all '
         data.update( cls.allSyntaxData_ )

      cls._assert( cls.syntax.startswith( 'show interfaces ' ),
                   'must define syntax with \'show interfaces ...\'' )
      cls._assert( 'INTF' not in cls.syntax and 'MOD' not in cls.syntax,
                   'INTF/MOD syntax is automatically generated' )
      extraSyntax = cls.syntax.replace( 'show interfaces ', '', 1 )

      cls.syntax = baseSyntax.format( allSyntax=allSyntax, extraSyntax=extraSyntax )

      for k in data:
         cls._assert( k not in cls.data,
                      f'\'{k}\' should not be defined in data' )

      cls.userData_ = copy.copy( cls.data )
      cls.data.update( data )

   @classmethod
   def _expandData( cls ):
      data = super()._expandData()
      for k, v in data.items():
         if k not in cls.userData_:
            if isinstance( v, CliMatcher.Matcher ):
               data[ k ] = CliCommand.Node( matcher=v, forceMerge=True )
            else:
               v.forceMerge_ = True
      return data

def showInterfaces( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   intfStatuses = IntfModel.InterfaceStatuses()
   intfs = Intf.getAll( mode, intf, mod, sort=False,
                        exposeInternal=( 'all' in args ) )
   if not intfs:
      return intfStatuses
   intfs = [ i for i in intfs if i.lookup() ] # See bug 9124
   if not intfs:
      return intfStatuses

   for x in intfs:
      intfStatuses.interfaces[ x.name ] = x.show( mode )
      TacSigint.check()

   return intfStatuses

class ShowInterfaces( ShowIntfCommand ):
   syntax = "show interfaces "
   data = {}
   cliModel = IntfModel.InterfaceStatuses
   handler = showInterfaces
   allowAllSyntax = True

BasicCli.addShowCommandClass( ShowInterfaces )

#-------------------------------------------------------------------------------
# The "show interfaces [<name>] description" command, in enable mode.
#  show interfaces [<name>] description
#  show interfaces module <modnum> description
#-------------------------------------------------------------------------------
def showInterfacesDescription( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   intfDescriptions = IntfModel.InterfacesDescriptions()
   intfs = Intf.getAll( mode, intf, mod, exposeInternal=( 'all' in args ) )
   if not intfs:
      return intfDescriptions

   intfs = [ i for i in intfs if i.lookup() ] # See bug 9124
   if not intfs:
      return intfDescriptions

   for x in intfs:
      intfDescription = IntfModel.InterfacesDescriptions.InterfaceDescription()
      intfDescription.description = x.description()
      intfDescription.lineProtocolStatus = x.lineProtocolState()
      intfDescription.interfaceStatus = x.getIntfState()
      intfDescriptions.interfaceDescriptions[ x.name ] = intfDescription

   return intfDescriptions

class ShowIntfDescription( ShowIntfCommand ):
   syntax = 'show interfaces description'
   data = dict( description='Details on description strings of interfaces' )
   cliModel = IntfModel.InterfacesDescriptions
   handler = showInterfacesDescription
   allowAllSyntax = True

BasicCli.addShowCommandClass( ShowIntfDescription )

#-------------------------------------------------------------------------------
# The "show interfaces [<name>] config-sanity" command, in enable mode.
#  show interfaces [<name>] config-sanity
#  show interfaces module <modnum> config-sanity
#-------------------------------------------------------------------------------

def showInterfacesConfigSanity( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   intfConfigSanities = IntfModel.InterfaceStatuses()
   intfs = Intf.getAll( mode, intf, mod )

   for i in intfs:
      if i.lookup():
         intfConfigSanities.interfaces[ i.name ] = i.showConfigSanity( mode )

   return intfConfigSanities

class ShowIntfConfigSanity( ShowIntfCommand ):
   syntax = 'show interfaces config-sanity'
   data = {
      'config-sanity' : 'Check for interface configuration errors',
   }
   cliModel = IntfModel.InterfaceStatuses
   handler = showInterfacesConfigSanity

BasicCli.addShowCommandClass( ShowIntfConfigSanity )

def captureCounterState( counter, checkpoint, notFoundString='n/a' ):
   """
   Returns a function which takes a specific counter name as an
   argument (such as "inOctets") and returns the current value of that
   specific counter (or the 'notFoundString' text if the counter is
   not defined) given an 'IntfCounter' object and a counter
   checkpoint.

   Note that for counters defined as 'Ark::MaybeU63', a value of
   'None' in Python corresponds to Ark::MaybeU63::null being
   true. 'notFoundString' will be returned in these cases.
   """
   def stat( attr ):
      if counter is None:
         # Unders some circumstances (e.g. a linecard being hotswapped), the counter
         # argument may be None.  In theory, the right thing to do in this case
         # would be to return notFoundString.  This would however cause some non-
         # optional counter attributes in the interface counters model (e.g.,
         # inOctets, inUcastPkts, etc.) to be missing.  Because of that we for
         # now simply return 0.  We have BUG90662 to track a more complete solution.
         return 0

      sysdbValue = getattr( counter.statistics, attr, None )

      if sysdbValue is None:
         sysdbValue = getattr( counter.rates, attr, None )

      if sysdbValue is None:
         return notFoundString

      checkpointValue = 0
      if checkpoint:
         checkpointValue = getattr( checkpoint.statistics, attr, None )

         if checkpointValue is None:
            checkpointValue = getattr( checkpoint.rates, attr, 0 )

      return sysdbValue - checkpointValue

   return stat

#-------------------------------------------------------------------------------
# The "show interfaces [<name>] counters delta" command, in enable mode.
#  show interfaces [<name>] counters delta
#  show interfaces module <modnum> counters delta
#  show interfaces counters delta module <modnum>
#-------------------------------------------------------------------------------
countersDeltaKw = CliMatcher.KeywordMatcher( 'delta',
                                       helpdesc='Interface counters delta' )

#-------------------------------------------------------------------------------
# The "show interfaces [<name>] counters" command, in enable mode.
#  show interfaces [<name>] counters
#  show interfaces module <modnum> counters
#  show interfaces counters module <modnum>
#-------------------------------------------------------------------------------
countersKw = CliMatcher.KeywordMatcher( 'counters',
                                       helpdesc='Interface counters' )

def clearLastCounters( mode ):
   _deltaCounterDir( mode ).clear()

def showInterfacesCounters( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   direction = args.get( 'incoming' ) or args.get( 'outgoing' )

   intfsCounters = IntfModel.InterfacesCounters()
   if direction is not None:
      intfsCounters._outputType = direction # pylint: disable=protected-access

   # Intf.getAll  may return None
   intfs = counterSupportedIntfs( mode, intf, mod )
   if not intfs:
      return intfsCounters

   for x in intfs:
      if 'delta' in args:
         intf = _deltaCounterDir( mode ).get( x.name )
         zeroOut = intf is None or intf.intfCounter.get( x.name ) is None
         checkpoint = None if zeroOut else intf.intfCounter.get( x.name )

         intfCountersModel = x.updateInterfaceCountersModel( checkpoint=checkpoint,
                                                             zeroOut=zeroOut )
         intfsCounters.interfaces[ x.status().intfId ] = intfCountersModel
      else:
         checkpoint = x.getLatestCounterCheckpoint()
         intfCountersModel = x.updateInterfaceCountersModel( checkpoint=checkpoint )
         intfsCounters.interfaces[ x.status().intfId ] = intfCountersModel
         x.updateLastCounters( mode )

   return intfsCounters

#-------------------------------------------------------------------------------
# The "show interfaces counters incoming", "show interfaces counters outgoing",
# "show interfaces counters delta incoming", and
# "show interfaces counters delta outgoing" commands, in enable mode.
# show interfaces [module <modnum>|<name>] counters [delta] [incoming|outgoing]
# show interfaces counters [delta] [incoming|outgoing] module <modnum>
#-------------------------------------------------------------------------------
class ShowIntfCounters( ShowIntfCommand ):
   syntax = "show interfaces counters [ delta ] [ incoming | outgoing ]"
   data = dict( counters=countersKw,
                delta='Interface counters delta',
                incoming='Incoming counters',
                outgoing='Outgoing counters' )
   cliModel = IntfModel.InterfacesCounters
   handler = showInterfacesCounters
   moduleAtEnd = True

BasicCli.addShowCommandClass( ShowIntfCounters )

#-------------------------------------------------------------------------------
# The "show interfaces counters discards" command, in enable mode.
#  show interfaces [<name>] counters discards
#  show interfaces module <modnum> counters discards
#  show interfaces counters discards module <modnum>
#-------------------------------------------------------------------------------
def showInterfacesCountersDiscards( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   intfsCountersDiscards = IntfModel.InterfacesCountersDiscards()
   intfs = counterDiscardSupportedIntfs( mode, intf, mod )
   if not intfs:
      return intfsCountersDiscards

   for x in intfs:
      intfId = x.status().intfId
      intfCountersDiscardsModel = IntfModel.InterfaceCounterDiscards( _name=intfId )
      stat = captureCounterState( counter=x.counter(),
                                  checkpoint=x.getLatestCounterCheckpoint(),
                                  notFoundString='N/A' )

      inDiscards = stat( 'inDiscards' )
      intfCountersDiscardsModel.inDiscards = inDiscards

      # 'inDiscards' should allways be updated, but leave this check
      # in just in case to avoid a run-time error.
      intfsCountersDiscards.inDiscardsTotal += ( inDiscards if inDiscards
                                                 != 'N/A' else 0 )

      outDiscards = stat( 'outDiscards' )

      # If 'outDiscards' is not being updated for this interface, then
      # don't update any totals.
      if outDiscards != 'N/A':
         intfCountersDiscardsModel.outDiscards = outDiscards
         if intfsCountersDiscards.outDiscardsTotal is not None:
            intfsCountersDiscards.outDiscardsTotal += outDiscards
         else:
            intfsCountersDiscards.outDiscardsTotal = outDiscards

      intfsCountersDiscards.interfaces[ intfId ] = intfCountersDiscardsModel

   return intfsCountersDiscards

class ShowIntfCountersDiscards( ShowIntfCommand ):
   syntax = "show interfaces counters discards"
   data = dict( counters=countersKw,
                discards='Packet discard counters' )
   cliModel = IntfModel.InterfacesCountersDiscards
   handler = showInterfacesCountersDiscards

BasicCli.addShowCommandClass( ShowIntfCountersDiscards )

#-------------------------------------------------------------------------------
# The "show interfaces counters ingress acl drop" command, in enable mode.
#  show interfaces [<name>] counters ingress acl drop
#-------------------------------------------------------------------------------
def showInterfacesCountersAclDrops( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   _maybeMountAclDropCounters()

   intfsCountersAclDrop = IntfModel.InterfacesCountersAclDrop()
   intfs = counterSupportedIntfs( mode, intf, mod )

   if not intfs:
      return intfsCountersAclDrop

   for x in intfs:
      intfId = x.status().intfId
      intfCountersAclDropModel = IntfModel.InterfacesCounterAclDrop( _name=intfId )
      aclDropKey =  _getAclDropConfigKey( x, allIntfAclDropCounters )
      if aclDropKey is None:
         continue
      aclDropCounterDir = allIntfAclDropCounters[ aclDropKey ].counter.get( intfId )
      if aclDropCounterDir:
         intfCountersAclDropModel.aclDrops = aclDropCounterDir.inAclDrops
         intfsCountersAclDrop.interfaces[ intfId ] = intfCountersAclDropModel

   return intfsCountersAclDrop

ingressDirectionKw = CliMatcher.KeywordMatcher( 'ingress',
                                          helpdesc='Ingress counter information' )

class ShowIntfCountersIngressAclDrop( ShowIntfCommand ):
   syntax = 'show interfaces counters ingress acl drop'
   data = dict( counters=countersKw,
                ingress=ingressDirectionKw,
                acl='Ingress ACL information',
                drop='Ingress ACL drop information' )
   cliModel = IntfModel.InterfacesCountersAclDrop
   handler = showInterfacesCountersAclDrops
   moduleAtEnd = True

BasicCli.addShowCommandClass( ShowIntfCountersIngressAclDrop )

#-------------------------------------------------------------------------------
# The "show interfaces [<name>] counters rates" command.
#   show interfaces [<name>] counters rates
#   show interfaces module <modnum> counters rates
#-------------------------------------------------------------------------------
countersRatesKw = CliMatcher.KeywordMatcher( 'rates',
                                             helpdesc='Input/Output rate counters' )

class InterfaceCountersRatesConfig:
   def __init__( self ):
      self._scaleKbpsAndPps = False

   @property
   def scaleKbpsAndPps( self ):
      return self._scaleKbpsAndPps

   @scaleKbpsAndPps.setter
   def scaleKbpsAndPps( self, value ):
      self._scaleKbpsAndPps = value

interfaceCountersConfig = InterfaceCountersRatesConfig()

def showInterfacesCountersRates( mode, args ):
   intfArg = args.get( 'INTF' )
   modArg = args.get( 'MOD' )
   intfsCountersRates = IntfModel.InterfaceCountersRates()
   # pylint: disable=protected-access
   intfsCountersRates._scaleKbpsAndPps = interfaceCountersConfig.scaleKbpsAndPps
   # Intf.getAll may return None
   intfs = countersRateSupportedIntfs( mode, intfArg, modArg )
   if not intfs:
      return intfsCountersRates

   for oneIntf in intfs:
      if oneIntf.addrStr() or oneIntf.hardware() == 'vxlan':
         intfCountersRates = oneIntf.updateInterfaceCountersRateModel()
         if intfCountersRates is None:
            mode.addWarning( f"Counters unavailable for {oneIntf}" )
            continue
         intfsCountersRates.interfaces[ oneIntf.name_ ] = intfCountersRates

   return intfsCountersRates

class ShowIntfCountersRates( ShowIntfCommand ):
   syntax = "show interfaces counters rates"
   data = dict( counters=countersKw,
                rates=countersRatesKw )
   cliModel = IntfModel.InterfaceCountersRates
   handler = showInterfacesCountersRates

BasicCli.addShowCommandClass( ShowIntfCountersRates )

#-------------------------------------------------------------------------------
# The "show interfaces [<name>] counters errors" command.
#-------------------------------------------------------------------------------
countersErrorsKw = CliMatcher.KeywordMatcher( 'errors',
                                              helpdesc='Error counters' )

def showInterfacesCountersErrors( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   allInterfacesCountersErrors = IntfModel.InterfacesErrorCounters()
   allIntf = countersErrorsSupportedIntfs( mode, intf=intf, mod=mod)
   if not allIntf:
      return allInterfacesCountersErrors
   for oneIntf in allIntf:
      intfModel = oneIntf.getCountersErrorsModel()
      if intfModel is None:
         mode.addWarning( f"Counters unavailable for {oneIntf}" )
         continue
      allInterfacesCountersErrors.interfaceErrorCounters[ oneIntf.name ] = intfModel
   return allInterfacesCountersErrors

class ShowIntfCountersErrors( ShowIntfCommand ):
   syntax = "show interfaces counters errors"
   data = dict( counters=countersKw,
                errors=countersErrorsKw )
   cliModel = IntfModel.InterfacesErrorCounters
   handler = showInterfacesCountersErrors
   moduleAtEnd = True

BasicCli.addShowCommandClass( ShowIntfCountersErrors )

#-------------------------------------------------------------------------------
# The "show interfaces [<name>] counters half-duplex" command.
#-------------------------------------------------------------------------------

countersHalfDuplexKw = CliMatcher.KeywordMatcher( 'half-duplex',
                                                  helpdesc='Half-duplex error '
                                                           'counters' )

def showInterfacesCountersHalfDuplexErrors( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   allIntfsCountersHalfDuplexErrors = IntfModel.InterfacesHalfDuplexErrorCounters()
   allIntf = countersErrorsSupportedIntfs( mode, intf=intf, mod=mod )
   # allIntf may be None.
   if not allIntf:
      return allIntfsCountersHalfDuplexErrors
   halfDuplexCounters = allIntfsCountersHalfDuplexErrors.intfHalfDuplexErrorCounters
   for oneIntf in allIntf:
      intfModel = oneIntf.getCountersHalfDuplexErrorsModel()
      if intfModel is None:
         mode.addWarning( f"Counters unavailable for {oneIntf}" )
         continue
      halfDuplexCounters[ oneIntf.name ] = intfModel
   return allIntfsCountersHalfDuplexErrors

class ShowIntfCountersHalfDuplexErrors( ShowIntfCommand ):
   syntax = "show interfaces counters half-duplex"
   data = {
      "counters": countersKw,
      "half-duplex": countersHalfDuplexKw,
   }
   cliModel = IntfModel.InterfacesHalfDuplexErrorCounters
   handler = showInterfacesCountersHalfDuplexErrors
   moduleAtEnd = True

BasicCli.addShowCommandClass( ShowIntfCountersHalfDuplexErrors )

#-------------------------------------------------------------------------------
# The "clear counters [interface] [session]" command, in "privileged exec" mode.
#-------------------------------------------------------------------------------

# Allow clear counters callbacks
clearCountersHook = []

def registerClearCountersHook( hook ):
   clearCountersHook.append( hook )

def clearCounters( mode, intf=None, mod=None, sessionOnly=False ):
   intfs = Intf.getAll( mode, intf, mod )
   if not intfs:
      return

   if not sessionOnly:
      if intf is None:
         Log.logClearCounters( "", "all interfaces" )
      else:
         if len( intfs ) > 1:
            Log.logClearCounters( "", "interfaces %s to %s" %
                                        ( intfs[0].shortname,
                                          intfs[-1].shortname ) )
         else:
            Log.logClearCounters( "", "interface " + intfs[ 0 ].shortname )

   for i in intfs:
      i.clearCounters( sessionOnly=sessionOnly )
   clearLastCounters( mode )

   for hook in clearCountersHook:
      hook( mode, intfs, sessionOnly, intf is None and mod is None )

counterSessionKw = CliMatcher.KeywordMatcher(
   'session',
   helpdesc='Clear for this CLI session only' )

class ClearCountersCmd( CliCommand.CliCommandClass ):
   syntax = "clear counters [ INTF ] [ session ]"
   data = dict( clear=CliToken.Clear.clearKwNode,
                counters=countersKw,
                INTF=Intf.rangeMatcher,
                session=counterSessionKw )
   @staticmethod
   def handler( mode, args ):
      intf = args.get( 'INTF' )
      clearCounters( mode, intf, mod=None, sessionOnly='session' in args )

BasicCli.EnableMode.addCommandClass( ClearCountersCmd )

#-------------------------------------------------------------------------------
# The "[no|default] shutdown" command, in "config-if" mode.
#-------------------------------------------------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = "shutdown"
   noSyntax = syntax
   defaultSyntax = syntax
   data = dict( shutdown="Administratively shut off the interface" )

   @staticmethod
   def handler( mode, args ):
      mode.setShutdown( no=False )

   @staticmethod
   def noHandler( mode, args ):
      mode.setShutdown( no=True )

   @staticmethod
   def defaultHandler( mode, args ):
      # if default state is shutdown, pass no=False (shutdown)
      no = not mode.intf.defaultShutdownConfig()
      mode.setShutdown( no=no )

IntfConfigMode.addCommandClass( ShutdownCmd )

#-------------------------------------------------------------------------------
# The "[no] description <desc>" command, in "config-if" mode.
#-------------------------------------------------------------------------------
class DescriptionCmd( CliCommand.CliCommandClass ):
   syntax = "description DESC"
   noOrDefaultSyntax = "description ..."
   data = dict( description='Description string to associate with the interface',
                DESC=CliMatcher.StringMatcher(
                   helpname='LINE',
                   helpdesc='Description for this interface' ) )
   @staticmethod
   def handler( mode, args ):
      mode.setDescription( args[ 'DESC' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noDescription()

IntfConfigMode.addCommandClass( DescriptionCmd )

#-------------------------------------------------------------------------------
# The "[no|default] load-interval <interval>" command, in "config-if" mode.
#-------------------------------------------------------------------------------

# instantiate temp LoadInterval object so that we can get the min/max values
LoadInterval = Tac.Type( 'Interface::LoadInterval' )

intervalMatcher = CliMatcher.IntegerMatcher(
   LoadInterval.minVal,
   LoadInterval.maxVal,
   helpdesc='Number of seconds' )

class LoadIntervalCmd( CliCommand.CliCommandClass ):
   syntax = "load-interval INTERVAL"
   noOrDefaultSyntax = "load-interval ..."
   data = {
      "load-interval" : 'Time used in calculating interface utilization',
      "INTERVAL" : intervalMatcher
      }
   @staticmethod
   def handler( mode, args ):
      mode.setLoadInterval( args[ 'INTERVAL' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noLoadInterval()

IntfConfigMode.addCommandClass( LoadIntervalCmd )

#-------------------------------------------------------------------------------
# The "[no|default] load-interval default <interval>" command, in "config" mode.
#-------------------------------------------------------------------------------
class LoadIntervalDefaultCmd( CliCommand.CliCommandClass ):
   syntax = "load-interval _default_ INTERVAL"
   noOrDefaultSyntax = "load-interval _default_ ..."
   data = {
      "load-interval" : 'Specify global interval for load calculation on interfaces',
      "_default_" : CliMatcher.KeywordMatcher(
         'default',
         helpdesc='Default load-interval configuration' ),
      "INTERVAL" : intervalMatcher
      }
   @staticmethod
   def handler( mode, args ):
      globalIntfConfig.loadInterval = newLoadInterval( args[ 'INTERVAL' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      globalIntfConfig.loadInterval = defaultLoadInterval()

BasicCli.GlobalConfigMode.addCommandClass( LoadIntervalDefaultCmd )

# helper function to get actual load-interval value:
#   if default at interface level
#      use global one (default or user-defined)
#   else
#      use user-defined at interface level
def getActualLoadIntervalValue( intfLoadInterv ):
   if intfLoadInterv.useDefault:
      globLoadInterv = globalIntfConfig.loadInterval
      if globLoadInterv.useDefault:
         return globLoadInterv.defaultVal
      else:
         return globLoadInterv.val
   else:
      return intfLoadInterv.val

# helper function to get printable string for load interval value
# (convert to minutes / seconds as appropriate)
def getLoadIntervalPrintableString( seconds ):
   seconds = int( seconds + 0.5 )
   minutes, seconds = divmod( seconds, 60 )
   intervalStr = ''
   if minutes > 0:
      intervalStr = '%d minute' % minutes
      if minutes > 1:
         intervalStr += 's'
      if seconds != 0:
         intervalStr += ', '
   if seconds:
      intervalStr += '%d second' % seconds
      if seconds > 1:
         intervalStr += 's'
   return intervalStr

# return LoadInterval object w/ default value
def defaultLoadInterval():
   li = Tac.newInstance( "Interface::LoadInterval" )
   li.useDefault = True
   return li

# return LoadInterval object w/ non-default value
def newLoadInterval( value ):
   li = Tac.newInstance( "Interface::LoadInterval" )
   li.useDefault = False
   li.val = value
   return li

#-------------------------------------------------------------------------------
# Provide global 'logging event FEATURE' and interface mode
# 'logging event FEATURE [use-global]' base classes.
#-------------------------------------------------------------------------------
class LoggingEventGlobalCmd( CliCommand.CliCommandClass ):
   # Subclass should define:
   # syntax = "logging event FEATURE"
   # noOrDefaultSyntax
   # data = { "FEATURE" : ... }
   #
   # and implement handler/noOrDefaultHandler
   baseData_ = {
      'logging' : CliToken.Logging.loggingForConfig,
      'event' : CliToken.Logging.eventForConfig,
   }
   data = {}

   dynamicSyntax_ = True

   @classmethod
   def _generateSyntaxAndData( cls ):
      cls._assert( cls.syntax.startswith( "logging event " ),
                   "must define syntax with 'logging event ...'" )
      for k in cls.baseData_:
         cls._assert( k not in cls.data, "'%s' should not be defined in data" % k )
      cls.data.update( cls.baseData_ )

class LoggingEventIntfCmd( CliCommand.CliCommandClass ):
   # Subclass should define:
   # syntax = "logging event FEATURE"
   # data = { "FEATURE" : ... }
   #
   # and implement handler/noOrDefaultHandler
   #
   # and the following handlers:
   #
   # 1. _enableHandler( cls, mode, args ) ( handler without using 'use-global' )
   # 2. _disableHandler( cls, mode, args ) ( no-handler )
   # 3. _useGlobalHandler( cls, mode, args ) (default-handler)
   baseData_ = {
      'logging' : CliToken.Logging.loggingForConfigIf,
      'event' : CliToken.Logging.eventForConfigIf,
      'use-global' : CliToken.Logging.useGlobalForConfigIf
   }
   data = {}

   @classmethod
   def _mustImplementHandler( cls, mode, args ):
      assert False, "Subclass must implement _enable/_disable/_useGlobalHandler"

   _enableHandler = _disableHandler = _useGlobalHandler = _mustImplementHandler

   dynamicSyntax_ = True

   @classmethod
   def _generateSyntaxAndData( cls ):
      cls._assert( cls.syntax.startswith( "logging event " ),
                   "must define syntax with 'logging event ...'" )
      for k in cls.baseData_:
         cls._assert( k not in cls.data, "'%s' should not be defined in data" % k )
      cls.syntax += ' [ use-global ]'
      cls.noOrDefaultSyntax = cls.syntax
      cls.data.update( cls.baseData_ )

   @classmethod
   def handler( cls, mode, args ):
      if 'use-global' in args:
         cls._useGlobalHandler( mode, args )
      else:
         cls._enableHandler( mode, args )

   @classmethod
   def noHandler( cls, mode, args ):
      cls._disableHandler( mode, args )

   @classmethod
   def defaultHandler( cls, mode, args ):
      cls._useGlobalHandler( mode, args )

#-------------------------------------------------------------------------------
# The "[no|default] logging event link-status global" command, in "config" mode.
#-------------------------------------------------------------------------------
class LoggingEventLinkStatusGlobal( LoggingEventGlobalCmd ):
   syntax = "logging event link-status global"
   noOrDefaultSyntax = syntax
   data = {
      'link-status' : 'UPDOWN messages',
      'global' : 'Global link status configuration'
      }

   @staticmethod
   def handler( mode, args ):
      globalIntfConfig.linkStatusLogging = 'on'

   @staticmethod
   def noHandler( mode, args ):
      globalIntfConfig.linkStatusLogging = 'off'

   # default is 'on'
   defaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( LoggingEventLinkStatusGlobal )

#-------------------------------------------------------------------------------
# The "[no] logging event link-status" command.
# The "logging event link-status use-global" command.
#-------------------------------------------------------------------------------
class LoggingEventLinkStatusIntf( LoggingEventIntfCmd ):
   syntax = "logging event link-status"
   data = {
      'link-status' : 'UPDOWN messages',
   }
   @classmethod
   def _enableHandler( cls, mode, args ):
      mode.setLoggingLinkStatus()

   @classmethod
   def _disableHandler( cls, mode, args ):
      mode.noLoggingLinkStatus()

   @classmethod
   def _useGlobalHandler( cls, mode, args ):
      mode.setLoggingLinkStatusUseGlobal()

   @classmethod
   def defaultHandler( cls, mode, args ):
      mode.intf.config().linkStatusLogging = IntfConfigMode.defaultLoggingLinkStatus(
         mode.intf )

IntfConfigMode.addCommandClass( LoggingEventLinkStatusIntf )

inactiveKwMatcher = CliMatcher.KeywordMatcher( 'inactive',
   helpdesc='Inactive subordinate lanes in multi-lane mode' )
exposeKwMatcher = CliMatcher.KeywordMatcher( 'expose',
                                             helpdesc='Expose information' )
unconnectedKwMatcher = CliMatcher.KeywordMatcher( 'unconnected',
   helpdesc='Unconnected/Internal ethernet ports' )

class ExposeLanesCommand( CliCommand.CliCommandClass ):
   syntax = "service interface inactive | unconnected expose"
   noOrDefaultSyntax = syntax
   data = {
      "service": CliToken.Service.serviceKw,
      "interface": intfAfterServiceKwMatcher,
      "inactive": inactiveKwMatcher,
      "unconnected": unconnectedKwMatcher,
      "expose": exposeKwMatcher
      }

   @staticmethod
   def handler( mode, args ):
      if 'inactive' in args: # pylint: disable=simplifiable-if-statement
         globalIntfConfig.exposeInactiveLanes = True
      else:
         globalIntfConfig.exposeUnconnectedLanes = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'inactive' in args: # pylint: disable=simplifiable-if-statement
         globalIntfConfig.exposeInactiveLanes = False
      else:
         globalIntfConfig.exposeUnconnectedLanes = False

BasicCli.GlobalConfigMode.addCommandClass( ExposeLanesCommand )

# -------------------------------------------------------------------------------
# The "service interface inactive port-id allocation disabled"
# -------------------------------------------------------------------------------
class DisableInactiveIntfPortAllocationCommand( CliCommand.CliCommandClass ):
   syntax = "service interface inactive port-id allocation disabled"
   noOrDefaultSyntax = syntax
   data = {
      "service": CliToken.Service.serviceKw,
      "interface": intfAfterServiceKwMatcher,
      "inactive": inactiveKwMatcher,
      "port-id": "Change interface Port ID",
      "allocation": "Change interface Port ID allocation",
      "disabled": "Disable Port ID allocation"
   }

   @staticmethod
   def handler( mode, args ):
      globalIntfConfig.inactiveIntfPortIdAllocationEnabled = False

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      globalIntfConfig.inactiveIntfPortIdAllocationEnabled = True

BasicCli.GlobalConfigMode.addCommandClass( DisableInactiveIntfPortAllocationCommand )

################################################################################
#-------------------------------------------------------------------------------
# Adds routed interface specific CLI commands to the "config-if" mode.
#-------------------------------------------------------------------------------
class RoutedIntfDependent( IntfDependentBase ):
   def setDefault( self ):
      RoutedIntfModelet.setDefaultMtu( self.intf_ )

class RoutedIntfModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.routingSupported()

   @staticmethod
   def setDefaultMtu( intf ):
      intf.config().l3MtuConfigured = False
      intf.config().mtu = globalIntfConfig.l3MtuDefault
      Tracing.trace0( "Modelet noMtu set", intf.name, "MTU to", \
              globalIntfConfig.l3MtuDefault )

   def setMtu( self, args ):
      mtu = args[ 'MTU' ]
      intf = self.mode.intf
      if intf.lookup():
         intfStatus = intf.status()
         # Only apply the limit if this is NONE of the below
         # 1. a startup-config
         # 2. a config-replace
         # 3. a config-session
         if ( not self.mode.session.skipConfigCheck() and
              not self.mode.session.inConfigSession() and
              intfStatus.maxMtu != 0 and
              intfStatus.maxMtu < mtu ):
            Tracing.trace0(   "Invalid MTU", mtu,
                              "for", intf.name,
                              "max is", intfStatus.maxMtu )
            self.mode.addErrorAndStop( "Max MTU for %s is %d" %
                                       ( intf.name, intfStatus.maxMtu ) )

      intf.config().l3MtuConfigured = True
      intf.config().mtu = mtu
      Tracing.trace0( "Modelet Set", intf.name, "MTU to", mtu )

   def noMtu( self, args ):
      # Clear the configured mtu (use default)
      intf = self.mode.intf
      RoutedIntfModelet.setDefaultMtu( intf )

   @staticmethod
   def mtuGuard( mode, token ):
      hookIter = subintfMtuGuardHook if mode.intf.isSubIntf() else intfMtuGuardHook
      for hook in hookIter:
         rc = hook( mode, token )
         if rc is not None:
            return rc
         return None

#-------------------------------------------------------------------------------
# Associate the RoutedIntfModelet with the "config-if" mode.
#-------------------------------------------------------------------------------
IntfConfigMode.addModelet( RoutedIntfModelet )

#-------------------------------------------------------------------------------
# The "[no] mtu" command.
# Also aliased to from the hidden "[no] ip mtu" command.
#-------------------------------------------------------------------------------
mtuKw = CliCommand.Node(
      CliMatcher.KeywordMatcher( 'mtu',
         helpdesc='Set IP Maximum Transmission Unit size in bytes' ),
      guard=RoutedIntfModelet.mtuGuard )
mtuValue = CliMatcher.IntegerMatcher( 68, 65535,
              helpdesc='Maximum transmission unit in bytes' )

class MtuCmd( CliCommand.CliCommandClass ):
   syntax = "mtu MTU"
   noOrDefaultSyntax = "mtu ..."
   data = dict( mtu=mtuKw,
                MTU=mtuValue )
   handler = RoutedIntfModelet.setMtu
   noOrDefaultHandler = RoutedIntfModelet.noMtu

RoutedIntfModelet.addCommandClass( MtuCmd )

myEntManager = None

#-------------------------------------------------------------------------------
#  show interfaces counters drop-precedence
#-------------------------------------------------------------------------------
def showInterfacesDpCounters( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   allDpCounters = IntfModel.DropPrecedenceCounters()
   intfs = dpCounterSupportedIntfs( mode, intf, mod )
   if not intfs:
      return allDpCounters

   fieldMap = {
         'inPkts': 'inPkts',
         'inBytes': 'inOctets',
         'outPkts': 'outPkts',
         'outBytes': 'outOctets'
      }
   for intf in intfs:
      intfId = intf.status().intfId
      intfDpCounters = IntfModel.IntfDropPrecedenceCounters()
      dpCounters = intf.dpCounters()
      if dpCounters is None:
         continue
      checkpointCounters = intf.getLatestDpCounterCheckpoint()

      def getDpStat( stat, clearStat, field ):
         currentCount = getattr( stat, field )
         clearCount = getattr( clearStat, field, 0 )
         return currentCount - clearCount

      for dp, stats in dpCounters.dropPrecedenceStat.items():
         perDpStats = IntfModel.DpCounters()
         for attr, cliHeader in fieldMap.items():
            deltaStat = \
               getDpStat( stats, checkpointCounters.dropPrecedenceStat[ dp ], attr )
            setattr( perDpStats, cliHeader, deltaStat )
         intfDpCounters.dpCounters[ dp ] = perDpStats

      allDpCounters.interfaces[ intfId ] = intfDpCounters

   return allDpCounters

class ShowIntfCountersDropPrecedence( ShowIntfCommand ):
   syntax = 'show interfaces counters drop-precedence'
   data = {
         'counters': countersKw,
         'drop-precedence': 'DP-tagged counters'
         }
   handler = showInterfacesDpCounters
   cliModel = IntfModel.DropPrecedenceCounters

BasicCli.addShowCommandClass( ShowIntfCountersDropPrecedence )

#--------------------------------------------------------------------------------
# [ no | default ] mac-address router MAC_ADDR
#-------------------------------------------------------------------------------
def macAddressRouterSupportedGuard( mode, token ):
   if routerMacHwCap.intfRouterMacSupported:
      return None
   return CliParser.guardNotThisPlatform

def macAddressRouterPhysicalSupportedGuard( mode, token ):
   if routerMacHwCap.intfRouterMacPhysicalSupported:
      intfName = mode.intf.name
      if '.' in intfName:
         # no subintf support
         return CliParser.guardNotThisPlatform
      if intfName.startswith( 'Ethernet' ):
         return None
      if intfName.startswith( 'Port-Channel' ):
         return None
   return CliParser.guardNotThisPlatform

nodeRouter = CliCommand.guardedKeyword( 'router',
      helpdesc='Set interface MAC address to use for Routing',
      guard=macAddressRouterSupportedGuard )

matcherMacAddress = CliMatcher.KeywordMatcher( 'mac-address',
      helpdesc='Set interface MAC address' )

nodePhysical = CliCommand.guardedKeyword( 'physical',
      helpdesc='physical address',
      guard=macAddressRouterPhysicalSupportedGuard )

class MacAddressRouterAddrCmd( CliCommand.CliCommandClass ):
   syntax = 'mac-address router ( MAC_ADDR | physical )'
   noOrDefaultSyntax = 'mac-address [ router ] ...'
   data = {
      'mac-address' : matcherMacAddress,
      'router': nodeRouter,
      'MAC_ADDR' : MacAddr.macAddrMatcher,
      'physical': nodePhysical,
   }

   @staticmethod
   def handler( mode, args ):
      isPhysical = 'physical' in args
      macAddr = None if isPhysical else args[ 'MAC_ADDR' ]
      mode.intf.setMacAddr( macAddr, routerMacConfigured=True, physical=isPhysical )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.intf.setMacAddr( None )

#-------------------------------------------------------------------------------
# Mount the GlobalIntfConfig and other necessary Sysdb objects.
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global aclDropConfigDir
   global allIntfConfigDir
   global allIntfStatusDir
   global globalIntfConfig
   global myEntManager
   global routerMacHwCap
   global intfIdDisplayContextDir

   myEntManager = entityManager

   globalIntfConfig = ConfigMount.mount( entityManager, "interface/config/global",
                                         "Interface::GlobalIntfConfig", "w" )
   allIntfConfigDir = LazyMount.mount( entityManager, 'interface/config/all',
                                       'Interface::AllIntfConfigDir', 'r' )
   allIntfStatusDir = LazyMount.mount( entityManager, "interface/status/all",
                                       "Interface::AllIntfStatusDir", "r" )
   aclDropConfigDir = LazyMount.mount( entityManager,
                                       "interface/aclDropCounter/aclDropConfigDir",
                                       "Tac::Dir", "ri" )
   routerMacHwCap = LazyMount.mount( entityManager, "routerMac/hwCapability",
                                     "RouterMac::HwCapability", "r" )

   mg = entityManager.mountGroup()
   intfIdDisplayContextDir = mg.mount( "interface/cli/displayContext",
                                       "Tac::Dir", "ri" )

   def doMountsComplete():
      global intfIdDisplayContextHelper
      intfIdDisplayContextHelper = Tac.newInstance(
         "Arnet::IntfIdDisplayContextHelper", intfIdDisplayContextDir )
   mg.close( doMountsComplete )

   Intf.registerDependentClass( RoutedIntfDependent )
   Intf.registerDependentClass( IntfConfigModeDependent )
