# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from itertools import chain
import re

import IntfInteractionsReferenceLib as IxnRef
import Tracing
from TypeFuture import TacLazyType

# The generated default trace-handle is something like
# CliPlugin::IntfInteractionReferenceMatcher, so I'm changing that.
traceHandle = Tracing.Handle( "IntfInteractionReferenceMatcher" )
t0 = traceHandle.trace0

XcvrType = TacLazyType( "Xcvr::XcvrType" )

class InteractionReferenceId:
   """
   Bundles a set of criteria that map to a particular
   IntfInteractionReference-based class.

   Members (optional unless otherwise specified)
   ---------------------------------------------
   phys: set( string ) (mandatory*)
      Set of strings describing the phys between the front-panel and the
      switch-chip, inclusive. Strings come from descriptions in L1Topo.
   xcvrSlot: string (mandatory*)
      Basically an alias for xcvrStatus.xcvrType
   productRe: string
      Match against entMib model name
   portRange: iterable of ints
      Ints correspond to front-panel port numbers/transceiver slot IDs. This
      criteria combined with the previous leads to the most explicit matching.
      Should be reserved for the special-est unique-est of cases.
   
   * 'phys' and 'xcvrSlot' will be "mandatory" for products that support L1Topo
     ("mandatory" in the sense that we won't assert out, but you won't get the
     match you want). For products that don't, matching will be done differently.
   """
   def __init__( self, phys=None, xcvrSlot="", productRe="", portRange=None ):
      self.phys = phys or set()
      self.xcvrSlot = xcvrSlot
      self.productRe = productRe
      self.portRange = portRange if portRange else list( range( 0 ) )

   def __repr__( self ):
      argString = ""
      if self.phys:
         # pylint: disable-next=consider-using-f-string
         argString += "phys=%s, " % self.phys
      if self.xcvrSlot:
         # pylint: disable-next=consider-using-f-string
         argString += "xcvrSlot=%s, " % self.xcvrSlot
      if self.productRe:
         # pylint: disable-next=consider-using-f-string
         argString += "productRe=%s, " % self.productRe
      if self.portRange:
         # pylint: disable-next=consider-using-f-string
         argString += "portRange=%s, " % self.portRange
      # pylint: disable-next=consider-using-f-string
      return "<InteractionReferenceId: %s>" % argString

################################################################################
# Reference Registry
#
# This is a central place to register InteractionReferenceIds to
# IntfInteractionReference-derived classes. The matching logic will favor the
# most specific RefId in the case that there are multiple matches.
Id = InteractionReferenceId  # will be used as key very often...
_intfInteractionReferenceRegistry = {
   # Jericho2 Template Types

   # Babbage Gearbox
   Id( { "Babbage-Gearbox", "Blackhawk" },
       XcvrType.qsfpPlus ): IxnRef.GearboxQsfp28Pair,
   Id( { "Babbage-Gearbox", "cmx42550", "Blackhawk" },
       XcvrType.qsfpPlus ): IxnRef.BabbageEnigmaQsfp,
   Id( { "Babbage-Gearbox", "cmx42550", "Blackhawk" },
       XcvrType.qsfpCmis ): IxnRef.BabbageEnigmaQsfp,
   Id( { "Babbage-Gearbox", "Blackhawk" },
       XcvrType.qsfpCmis ): IxnRef.GearboxQsfp28Pair,
   Id( { "B52-Gearbox", "Blackhawk" },
       XcvrType.qsfpPlus ): IxnRef.GearboxQsfp28Pair,
   Id( { "B52-Gearbox", "Blackhawk" },
       XcvrType.qsfpCmis ): IxnRef.GearboxQsfp28Pair,
   Id( { "B52-Gearbox", "Tofino2Core" },
       XcvrType.qsfpPlus ): IxnRef.GearboxQsfp28Pair,
   Id( { "B52LP-Gearbox", "Blackhawk" },
       XcvrType.qsfpPlus ): IxnRef.GearboxQsfp28Pair,
   Id( { "B52LP-Gearbox", "Blackhawk" },
       XcvrType.qsfpCmis ): IxnRef.GearboxQsfp28Pair,
   Id( { "B52LP-Gearbox", "Tofino2Core" },
       XcvrType.qsfpPlus ): IxnRef.GearboxQsfp28Pair,
   Id( { "BabbageLP-Gearbox", "Blackhawk" },
       XcvrType.qsfpCmis ): IxnRef.GearboxQsfp28Pair,
   Id( { "BabbageLP-Gearbox", "Blackhawk" },
       XcvrType.qsfpPlus ): IxnRef.GearboxQsfp28Pair,
   Id( { "BabbageLP-CrossGearbox", "Blackhawk" },
       XcvrType.qsfpPlus ): IxnRef.GearboxQsfp28Pair,

   # Babbage CrossGearbox
   Id( { "Babbage-CrossGearbox", "Blackhawk" },
       XcvrType.qsfpPlus ): IxnRef.GearboxQsfp28Pair,
   Id( { "Babbage-CrossGearbox", "cmx42550", "Blackhawk" },
       XcvrType.qsfpPlus ): IxnRef.BabbageEnigmaQsfp,
   Id( { "Babbage-CrossGearbox", "cmx42550", "Blackhawk" },
       XcvrType.qsfpCmis ): IxnRef.BabbageEnigmaQsfp,
   Id( { "Babbage-CrossGearbox", "Blackhawk" },
       XcvrType.qsfpCmis ): IxnRef.GearboxQsfp28Pair,

   # Barchetta2
   Id( { "bcm87728-gb", "Blackhawk" },
       XcvrType.qsfpCmis ): IxnRef.GearboxQsfp28Pair,
   Id( { "bcm87728-cgb", "Blackhawk" },
       XcvrType.qsfpCmis ): IxnRef.GearboxQsfp28Pair,
   Id( { "bcm87728-mux", "Blackhawk" },
       XcvrType.qsfpDd ): IxnRef.MuxQsfp56Pair,

   # Evora
   Id( { "bcm82391", "Falcon16" }, XcvrType.qsfpPlus ): IxnRef.QsfpMixedRate,
   Id( { "bcm82391", "Falcon16Gen3" }, XcvrType.sfpPlus ): IxnRef.Sfp,
   Id( { "bcm82391", "Blackhawk" }, XcvrType.qsfpPlus ): IxnRef.QsfpMixedRate,

   # Millenio
   Id( { 'bcm81356-gb', 'Blackhawk' },
       XcvrType.qsfpCmis ): IxnRef.GearboxQsfp28Pair,
   Id( { 'bcm81356-mux', 'Blackhawk' },
       XcvrType.qsfpDd ): IxnRef.MuxQsfp56Pair,

   # Sparrow
   Id( { 'Sparrow-FourWayGearbox', 'Peregrine' },
       XcvrType.qsfpPlus ): IxnRef.SparrowFourWayGearboxQsfp,

   # No External Phy
   Id( { "Blackhawk" }, XcvrType.qsfpPlus ): IxnRef.QsfpMixedRate,
   Id( { "Blackhawk" }, XcvrType.osfp ): IxnRef.OctalBlackhawk,
   Id( { "Blackhawk" }, XcvrType.qsfpDd ): IxnRef.OctalBlackhawk,
   Id( { "Blackhawk" }, XcvrType.sfpPlus ): IxnRef.AdvancedSfp,
   Id( { "Blackhawk" }, XcvrType.qsfpCmis ): IxnRef.Qsfp56MixedRate,
   Id( { "Blackhawk" }, XcvrType.dsfp ): IxnRef.Dsfp,
   Id( { "Blackhawk" }, XcvrType.sfpDd ): IxnRef.SfpDd,
   Id( { "BlackhawkGen3" }, XcvrType.qsfpDd ): IxnRef.Tomahawk450GQsfpDd,
   Id( { "Falcon16Gen3" }, XcvrType.sfpPlus ): IxnRef.AdvancedSfp,
   Id( { "Falcon16Gen3" }, XcvrType.qsfpPlus ): IxnRef.QsfpMixedRate,
   Id( { "Osprey" }, XcvrType.osfp ): IxnRef.Octal2x400G,
   Id( { "Osprey" }, XcvrType.qsfpDd ): IxnRef.Octal2x400G,
   Id( { "Peregrine" }, XcvrType.osfp ): IxnRef.OctalPeregrine,
   Id( { "Peregrine" }, XcvrType.qsfpDd ): IxnRef.OctalPeregrine,
   Id( { "Tofino2Core4Lane" }, XcvrType.sfpPlus ): IxnRef.AdvancedSfp,
   Id( { "D5" }, XcvrType.sfpPlus ): IxnRef.Sfp,
   Id( { "D5" }, XcvrType.qsfpPlus ): IxnRef.QsfpMixedRate,

   # Strata Template Types
   Id( { "Babbage-Retimer", "Blackhawk" },
       XcvrType.osfp ): IxnRef.OctalBlackhawk,
   Id( { "Babbage-Retimer", "Blackhawk" },
       XcvrType.qsfpDd ): IxnRef.OctalBlackhawk,
   Id( { "Babbage-Retimer", "Blackhawk" },
       XcvrType.sfpPlus ): IxnRef.AdvancedSfp,
   Id( { "BabbageLP-Retimer", "BlackhawkGen3" },
       XcvrType.qsfpDd ): IxnRef.Tomahawk450GQsfpDd,
   Id( { "B52-Retimer", "Blackhawk" },
       XcvrType.osfp ): IxnRef.OctalBlackhawk,
   Id( { "B52-Retimer", "Blackhawk" },
       XcvrType.qsfpDd ): IxnRef.OctalBlackhawk,
   Id( { "B52-Retimer", "BlackhawkGen3" },
       XcvrType.qsfpDd ): IxnRef.Tomahawk450GQsfpDd,
   Id( { "B52LP-Retimer", "Blackhawk" },
       XcvrType.osfp ): IxnRef.OctalBlackhawk,
   Id( { "B52LP-Retimer", "Blackhawk" },
       XcvrType.qsfpDd ): IxnRef.OctalBlackhawk,
   Id( { "B52LP-Retimer", "BlackhawkGen3" },
       XcvrType.qsfpDd ): IxnRef.Tomahawk450GQsfpDd,
   Id( { "Merlin" }, XcvrType.sfpPlus ): IxnRef.Sfp,
   Id( { "MerlinQ", "BCM54182" }, XcvrType.rj45 ): IxnRef.Rj45,
   Id( { "BCM84898", "Falcon16LowSpeed" }, XcvrType.rj45 ): IxnRef.Rj45,
   Id( { "BCM84898", "Falcon16Gen3" }, XcvrType.rj45 ): IxnRef.Rj45,
   Id( { "BCM84898", "Falcon16" }, XcvrType.rj45 ): IxnRef.Rj45,
   Id( { "BCM84898", "Blackhawk" }, XcvrType.rj45 ): IxnRef.SpeedGroupRj45,
   Id( { "BCM54994", "Falcon16" }, XcvrType.rj45 ): IxnRef.Rj45,
   Id( { "Falcon16" }, XcvrType.qsfpPlus ): IxnRef.BasicQsfp,
   Id( { "Falcon16" }, XcvrType.sfpPlus ): IxnRef.AdvancedSfp,
   Id( { "Falcon16LowSpeed" }, XcvrType.sfpPlus ): IxnRef.AdvancedSfp,
   Id( { "MerlinQ", "BCM54998E" }, XcvrType.rj45 ): IxnRef.Rj45,
   Id( { "Merlin", "BCM54998ES" }, XcvrType.rj45 ): IxnRef.Rj45,
   Id( { "Merlin", "BCM54998S" }, XcvrType.rj45 ): IxnRef.Rj45,
   Id( { "GPhy" }, XcvrType.rj45 ): IxnRef.Rj45,
   Id( { "MerlinQ", "BCM54185" }, XcvrType.sfpPlus ): IxnRef.Sfp,
   Id( { "Merlin", "BCM84898" }, XcvrType.rj45 ): IxnRef.Rj45,
   Id( { "MerlinQ", "BCM84898" }, XcvrType.rj45 ): IxnRef.Rj45,

   # Non-L1-Topo Template Types (i.e. super-hacky-fun-stuff)

   # Capitola ports:
   # * Et1-6,31-42,67-72 are standard QSFP+
   # * Et7-10,27-30,43-46,63-66 are 40G-only QSFP+
   # * Et11-26,47-62 are Sesto2 port pairs; primary-secondary are odd-even pairs
   Id( xcvrSlot=XcvrType.qsfpPlus, productRe="DCS-7280QRA?-C72",
       portRange=( list( range( 1, 7 ) ) + list( range( 31, 43 ) ) +
                   list( range( 67, 73 ) ) ) ): IxnRef.BasicQsfp,
   Id( xcvrSlot=XcvrType.qsfpPlus, productRe="DCS-7280QRA?-C72",
       portRange=( list( range( 7, 11 ) ) + list( range( 27,
             31 ) ) + list( range( 43, 47 ) ) +
                   list( range( 63, 67 ) ) ) ): IxnRef.Qsfp40gOnly,
   Id( xcvrSlot=XcvrType.qsfpPlus, productRe="DCS-7280QRA?-C72",
       portRange=( list( range( 11, 27 ) ) + list( range( 47,
             63 ) ) ) ): IxnRef.Sesto2Pair,

   # Nice ports:
   # * Et1-6,8,10,12,26,28,30-36 are standard QSFP+
   # * Et7,9,11,25,27,29 are standard QSFP28 (but use the same reference as QSFP+)
   # * Et13-24 are Sesto2 port pairs
   Id( xcvrSlot=XcvrType.qsfpPlus, productRe="DCS-7280QRA-C36S",
       portRange=( list( range( 1, 13 ) ) + list( range( 25,
             37 ) ) ) ): IxnRef.BasicQsfp,
   Id( xcvrSlot=XcvrType.qsfpPlus, productRe="DCS-7280QRA-C36S",
       portRange=( list( range( 13, 25 ) ) ) ): IxnRef.Sesto2Pair,

   # Woodacre2 ports:
   # * Et7/1-30/1,53/1-56/1 are Babbage-pairs where both ports only have /1
   #   interfaces. The lack of subordinate interfaces actually means that by
   #   default, the secondary ports can remain active for many more configurations
   #   on the primary compared to standard Babbage-pairs.
   Id( phys={ "BabbageLP-Gearbox", "Blackhawk" },
       xcvrSlot=XcvrType.qsfpPlus, productRe="DCS-7280CR3A[KM]?-72",
       portRange=list( chain( range( 7, 31 ), range( 53, 57 ) ) ) ):
   IxnRef.QsfpGearboxSingleIntfSingleSlot,

   # Brownsville2
   # Due to the J2M logical port limitation, port 13-28 on Brownsville2
   # are restricted to non-breakout mode and only have /1 interfaces.
   Id( phys={ "bcm87728-gb", "Blackhawk" },
       xcvrSlot=XcvrType.qsfpCmis, productRe="DCS-7280CR3A[KM]?-32S",
       portRange=list( range( 13, 29 ) ) ):
   IxnRef.DepopulatedQsfpGearboxPair,

   # Pebble and Pebble-Prime linecards
   Id( phys={ "BlackhawkGen3" }, xcvrSlot=XcvrType.qsfpDd,
       productRe="7388-16CD2?$", 
       portRange=( list( range( 1, 17, 2 ) ) ) ): IxnRef.Tomahawk450GQsfpDd,
   Id( phys={ "BlackhawkGen3" }, xcvrSlot=XcvrType.qsfpDd,
       productRe="7388-16CD2?$",
       portRange=( list( range( 2, 17, 2 ) ) ) ): IxnRef.BlancoSecondary,
   # If blanco is installed, secondary intfs on pebble will lose their
   # serdes, and thus have no information on Phys.
   Id( set(), XcvrType.qsfpDd, 
       productRe="7388-16CD2?$" ): IxnRef.NoPhyTemplate,

   # Gardena4Th4
   # Have to handle BlackhawGen3 direct connections to two QSFP ports, and handling
   # BlackhawkGen3 partial remap + Barchetta2 External Phy.
   # TH4-50G Direct connection to two QSFPs:
   Id( phys={ "BlackhawkGen3" }, xcvrSlot=XcvrType.qsfpPlus,
       productRe="DCS-7060CX5-56D8",
       portRange=( list( range( 5, 24, 2 ) ) + list( range( 33, 52, 2 ) ) ) ):
   IxnRef.Tomahawk450GQsfpPlusPrimary,
   Id( phys={ "BlackhawkGen3" }, xcvrSlot=XcvrType.qsfpPlus,
       productRe="DCS-7060CX5-56D8",
       portRange=( list( range( 6, 25, 2 ) ) + list( range( 34, 53, 2 ) ) ) ):
   IxnRef.Tomahawk450GQsfpPlusSecondary,

   # TH4-50G + Barchetta2 connections:
   Id( phys={ "bcm87728-gb", "BlackhawkGen3" }, xcvrSlot=XcvrType.qsfpPlus,
       productRe="DCS-7060CX5-56D8",
       portRange=( [ 1, 2, 25, 26, 29, 30, 53, 54 ] ) ):
   IxnRef.Tomahawk450GBarchetta2FirstPorts,
   Id( phys={ "bcm87728-gb", "BlackhawkGen3" }, xcvrSlot=XcvrType.qsfpPlus,
       productRe="DCS-7060CX5-56D8", portRange=( [ 3, 27, 31, 55 ] ) ):
   IxnRef.Tomahawk450GBarchetta2SecondPrimary,
   Id( phys={ "bcm87728-gb", "BlackhawkGen3" }, xcvrSlot=XcvrType.qsfpPlus,
       productRe="DCS-7060CX5-56D8", portRange=( [ 4, 28, 32, 56 ] ) ):
   IxnRef.Tomahawk450GBarchetta2SecondSecondary,
}

def getMatcherRegistry():
   """
   Getter for protected collection _intfInteractionReferenceRegistry.

   Not to be widely used, but some test environments might be easier to construct
   if we have access to the registry.
   """
   return _intfInteractionReferenceRegistry

################################################################################
# API to match criteria with reference
def findMatch( refId ):
   match = None
   # Find a baseline first by matching against phys + xcvr-slot. Products that
   # don't support L1Topo are expected to have refId.phys==set().
   matchingKeys = [ k for k in _intfInteractionReferenceRegistry
                    if k.phys == refId.phys and k.xcvrSlot == refId.xcvrSlot ]
   if len( matchingKeys ) > 1:
      # Need to refine matches with additional criteria
      t0( refId, "initially matched to", matchingKeys )
      defaultProductReMatchKeys = [ key for key in matchingKeys
                                        if not key.productRe ]
      # Filter first based on product
      productReMatchKeys = [ key for key in matchingKeys
                             if re.match( key.productRe, refId.productRe ) ]
      if productReMatchKeys:
         t0( refId, "product matched to", productReMatchKeys[ 0 ].productRe )
      else:
         t0( refId, "product not matched to any, default: ",
                    defaultProductReMatchKeys )
         productReMatchKeys = defaultProductReMatchKeys
      # Filter further on port-range
      defaultPortMatchedKeys = [ key for key in productReMatchKeys
                                    if not key.portRange ]
      portMatchedKeys = [ key for key in productReMatchKeys if
                          set( refId.portRange ).issubset( set( key.portRange ) ) ]
      if len( portMatchedKeys ) == 1:
         match = _intfInteractionReferenceRegistry[ portMatchedKeys[ 0 ] ]
      else:
         match = _intfInteractionReferenceRegistry[ defaultPortMatchedKeys[ 0 ] ]
   elif len( matchingKeys ) == 1:
      match = _intfInteractionReferenceRegistry[ matchingKeys[ 0 ] ]
   # The "else" case means that there is no match. In the grand scheme of things,
   # it means that serdes domains with this reference ID won't appear in the
   # command output.
   t0( refId, "matched to", match )
   return match

