# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from collections import namedtuple

import Ark
import Arnet
import BasicCli
from CliPlugin.EthIntfCli import L1Intf
# pylint: disable-next=consider-using-from-import
import CliPlugin.IntfMacModel as IntfMacModel
import CliPlugin.IntfCli as IntfCli # pylint: disable=consider-using-from-import
import CliPlugin.TechSupportCli
import LazyMount
import PhyStatusLib
from TypeFuture import TacLazyType


systemName = None
ethPhyIntfConfigSliceDir = None
ethPhyIntfStatusDir = None
entityManager = None


#--------------------------------------------------------------------
#
#   show interface [interface name] mac [detail]
#
# Here's the idea: MAC status information is stored in EthPhyIntfStatus.
# We will only look at those interfaces that have physical ports associated
# with them. We will also make an explicit assumption that for every physical
# port, the MAC display information in the ethPhyIntfStatus object is
# valid.
#
# If a platform wishes to provide hook to display additional information,
# they can do so. That will be displayed as part of the "detail" option.
#
# There is one monkey wrench in this scheme: Unlike PHYs, where PHY type
# can differentiate who is the "owner" to display additional info, there
# is nothing of that sort at the interface level. When the CLI Plugins are
# loaded, multiple registrations can happen for specific display.
# The scheme to loop through the registered functions is not great,
# but will do for now.
#--------------------------------------------------------------------

def _maybeCallShowIntMacRegisteredFunctions( table, *args ):

   for entry in table.values():
      checker = entry[ 'checkFunc' ]
      if not checker or checker( systemName ):
         entry[ 'func' ]( *args )


def _getPrintableInterfaceList( intfs ):

   ShowIntMacData = namedtuple( 'ShowIntMacData',
                                'phyIntfStatus ethConfig ethStatus' )
   # mount phy topology
   phyTopology = PhyStatusLib.mountPhyTopology( entityManager )

   # Look in the phy topology for phy config/status objects first.
   phyStatusList = {}

   intfs = { intf.name : intf for intf in intfs }
   for intfName in Arnet.sortIntf( intfs ):
      phyData = PhyStatusLib.phyConfigStatusAtPos(
         entityManager.root(), phyTopology, intfName,
         TacLazyType( "PhyEee::PhyPosition" ).asicPhy, firstLaneOnly=True )
      if phyData:
         assert len( phyData ) == 1
         phyStatusList[ intfName ] = phyData[ 0 ][ 1 ]

   # We can still attempt to print mac info if there is no phyStatus
   for intf in intfs:
      # Putting an interface into phyStatusList gets the loop
      # below to add the interface to the list that we will
      # display, but setting it to None indicates that we
      # have no phyStatus for this interface.
      if ( isinstance( intfs[ intf ], L1Intf ) and
           intf not in phyStatusList ):
         phyStatusList[ intf ] = None

   # Grab other information necessary for show int mac to work. If all information
   # is not present, don't return data on that interface.
   interfaceList = {}
   for intfName, phyIntfStatus in phyStatusList.items():
      intf = intfs[ intfName ]
      ethIntfStatus = intf.status()
      ethIntfConfig = intf.config()

      if ethIntfConfig and ethIntfStatus:
         interfaceList[ intfName ] = ShowIntMacData(
               phyIntfStatus, ethIntfConfig, ethIntfStatus )

   # Let's sort it and send a list of tuples back
   printableInterfaceList = sorted( interfaceList.items(),
                                    key=lambda k: Arnet.intfNameKey( k[ 0 ] ) )
   return printableInterfaceList


def showInterfacesMacCommandHandler( mode, intfsMac, intfs ):

   printableInterfaceList = _getPrintableInterfaceList( intfs )

   if not printableInterfaceList:
      return

   for t in printableInterfaceList:
      intf = t[0]
      intfData = t[1]

      hwPhyStatus = intfData.phyIntfStatus
      ethStatus = intfData.ethStatus

      macData = IntfMacModel.InterfaceMacData()

      macData.adminEnabled = intfData.ethConfig.adminEnabled

      # pylint: disable-next=isinstance-second-argument-not-valid-type
      if isinstance( hwPhyStatus,
            TacLazyType( 'Hardware::PhyStatus::PhyStatusGen3' ) ):
         macData.phyState = hwPhyStatus.phyState.current
         macData.phyStateChangeCount = hwPhyStatus.phyState.changes
         macData.phyStateLastChangeTime = Ark.switchTimeToUtc(
                                                hwPhyStatus.phyState.lastChange )
      elif hasattr( hwPhyStatus, 'phyState' ):
         macData.phyState = hwPhyStatus.phyState
         macData.phyStateChangeCount = hwPhyStatus.phyStateChanges
         macData.phyStateLastChangeTime = Ark.switchTimeToUtc(
                                                hwPhyStatus.lastPhyStateChange )
      # Don't bother populating the phy data if phyState is not present

      macData.intfState = ethStatus.linkStatus if ethStatus.active else "inactive"
      macData.intfStateChangeCount = ethStatus.operStatusChange.count
      macData.intfStateLastChangeTime = Ark.switchTimeToUtc(
                                             ethStatus.operStatusChange.time )

      macData.macRxLocalFault = ethStatus.macRxLocalFault
      macData.macRxLocalFaultChangeCount = ethStatus.macRxLocalFaultChanges
      macData.macRxLastLocalFaultChangeTime = Ark.switchTimeToUtc(
                                                ethStatus.lastMacRxLocalFaultChange )

      macData.macRxRemoteFault = ethStatus.macRxRemoteFault
      macData.macRxRemoteFaultChangeCount = ethStatus.macRxRemoteFaultChanges
      macData.macRxLastRemoteFaultChangeTime = \
            Ark.switchTimeToUtc( ethStatus.lastMacRxRemoteFaultChange )

      # If there is additional status, it is stored as a set of strings. Let's
      # just dump them for the user. This is only used in the "detail" output.
      if intfsMac.detail():
         macData.additionalStatus.update(
                     list( ethStatus.macLayerAdditionalStatus.items() ) )

      intfsMac.interfaces[intf] = macData


def showInterfacesMac( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   exposeInternal = 'all' in args

   intfsMac = IntfMacModel.InterfacesMac()
   intfsMac.detailIs( 'detail' in args )

   intfs = IntfCli.Intf.getAll( mode, intf, mod, exposeInternal=exposeInternal )
   if not intfs:
      return intfsMac

   _maybeCallShowIntMacRegisteredFunctions( showIntMacDisplayHandlers_,
                                            mode,
                                            intfsMac,
                                            intfs )
   return intfsMac


class ShowIntfMac( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces mac [ detail ]'
   data = {
         'mac' : 'Display MAC status',
         'detail' : 'Display information in detail'
         }
   handler = showInterfacesMac
   cliModel = IntfMacModel.InterfacesMac
   allowAllSyntax = True

BasicCli.addShowCommandClass( ShowIntfMac )

showIntMacDisplayHandlers_ = {}


def registerShowIntMacDisplayFunction( name,
                                       displayFunction,
                                       checkFunction=None ):

   '''
   A check function, if provided, will be called with the sysname as parameter.
   If that function returns true then the actual display function is called.
   This allows callers to pass in a check function that can determine at run time
   (based on the actual HW etc.) whether or not to run their display function.
   '''

   assert name not in showIntMacDisplayHandlers_
   entry = { 'func' : displayFunction, 'checkFunc' : checkFunction }
   showIntMacDisplayHandlers_[ name ] = entry


def Plugin( em ):
   global entityManager
   global ethPhyIntfConfigSliceDir
   global ethPhyIntfStatusDir

   entityManager = em

   ethPhyIntfConfigSliceDir = LazyMount.mount( em, "interface/config/eth/phy/slice",
                                        "Tac::Dir", "ri" )

   ethPhyIntfStatusDir = LazyMount.mount( em, "interface/status/eth/phy/all",
                                   "Interface::AllEthPhyIntfStatusDir", "r" )

   global systemName
   systemName = entityManager.sysname()

#-------------------------------------------------------------------------------
# Register show tech command
#-------------------------------------------------------------------------------
# Timestamps are made up to maintain logical order within show tech-support
CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2014-06-26 18:06:45',
   cmds=[ 'show interfaces all mac detail' ] )

