#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliMatcher
from CliPlugin.IntfMaintenanceModels import IntfMaintenanceDefaultProfile
from CliPlugin.IntfGroupCli import intfGroupNameMatcher
from CliPlugin.MaintenanceCliLib import intfProfileType, bgpProfileType
from CliPlugin.MaintenanceCliLib import intfGroupType
from CliPlugin.MaintenanceCliLib import showMaintenanceMatcher, defaultMatcher
from CliPlugin.MaintenanceCliLib import profileNameMatcher
from CliPlugin.MaintenanceCliLib import defaultProfile, createGroupKey
from CliPlugin.MaintenanceCliLib import maintRunningShowCliCheck
from CliPlugin.MaintenanceModels import InterfaceGroup, InterfaceGroups
from CliPlugin.MaintenanceModels import InterfaceProfile, InterfaceProfiles
from CliPlugin.MaintenanceModels import intfGroupsHook, intfProfilesHook
import LazyMount
import ShowCommand
import Tac

intfProfileStatusDir = None
defaultIntfProfile = None
defaultBgpProfile = None
intfGroupConfigDir = None
intfGroupStatusDir = None
maintenanceGroupDir = None
groupToUnitDir = None

GroupOrigin = Tac.Type( "Group::GroupOrigin" )

matcherInterface = CliMatcher.KeywordMatcher( 'interface',
      helpdesc='Interface-specific details' )
matcherProfile = CliMatcher.KeywordMatcher( 'profiles',
      helpdesc='Profiles information' )

#-------------------------------------------------------------------------------
# show maintenance groups interface [ GROUP_NAME ]
#-------------------------------------------------------------------------------
def showInterfaceGroup( intfGroup ):
   ''' Returns the populated intfGroup
       Note: This function expects the intfGroup to be valid
   '''
   units = []
   origin = intfGroup.origin
   interfaces = list( intfGroup.intf )
   bgpProfile_ = intfProfile_ = 'Default'
   if defaultBgpProfile.profileName != defaultProfile:
      bgpProfile_ = defaultBgpProfile.profileName
   if defaultIntfProfile.profileName != defaultProfile:
      intfProfile_ = defaultIntfProfile.profileName
   maintGroup = maintenanceGroupDir.maintenanceGroup.get( intfGroup.key )
   if maintGroup:
      for profile in maintGroup.profile:
         if profile.type == bgpProfileType:
            bgpProfile_ = profile.name
         elif profile.type == intfProfileType:
            intfProfile_ = profile.name
   groupName = intfGroup.groupName
   intfGroupKey = createGroupKey( intfGroupType, groupName )
   intfGroupToUnit = groupToUnitDir.groupToUnit.get( intfGroupKey ) 
   if intfGroupToUnit:
      for unit in intfGroupToUnit.unitName:
         units.append( unit )
   return InterfaceGroup( origin=origin, interfaces=interfaces,
                          bgpProfile=bgpProfile_,
                          intfProfile=intfProfile_,
                          units=units )

def showInterfaceGroups_( mode, intfGroupName=None ):
   allIntfGroups = {}
   if not intfGroupName:
      for groupStatus in intfGroupStatusDir.status.values():
         if groupStatus.origin == GroupOrigin.dynamic:
            continue
         intfGroup = showInterfaceGroup( groupStatus )
         allIntfGroups[ groupStatus.groupName ] = intfGroup
   else:
      intfGroupStatus = intfGroupStatusDir.status.get( intfGroupName )
      if intfGroupStatus:
         intfGroup = showInterfaceGroup( intfGroupStatus )
         allIntfGroups[ intfGroupName ] = intfGroup
   return allIntfGroups

def showInterfaceGroupsHook( mode, groupsModel, intfGroupName=None ):
   groupsModel.intfGroups = showInterfaceGroups_( mode, intfGroupName )

class ShowInterfaceGroups( ShowCommand.ShowCliCommandClass ):
   syntax = 'show maintenance groups interface [ GROUP_NAME ]'
   data = {
      'maintenance' : showMaintenanceMatcher,
      'groups' : 'Groups information',
      'interface' : matcherInterface,
      'GROUP_NAME' : intfGroupNameMatcher
   }
   cliModel = InterfaceGroups

   @staticmethod
   @maintRunningShowCliCheck
   def handler( mode, args ):
      allIntfGroups = showInterfaceGroups_( mode, args.get( 'GROUP_NAME' ) )
      return InterfaceGroups( intfGroups=allIntfGroups )

BasicCli.addShowCommandClass( ShowInterfaceGroups )

#-------------------------------------------------------------------------------
# show maintenance profiles interface [ profile_name ]
#-------------------------------------------------------------------------------
def showInterfaceProfile( profileStatus ):
   ''' Return the details for a single interface profile.
       Note: this function assumes profileStatus to be valid
   '''
   interfaceProfile = InterfaceProfile()
   interfaceProfile.rateMonLoadInterval = profileStatus.rateMonLoadInterval
   interfaceProfile.rateMonThreshold = profileStatus.rateMonThresholdInKbps
   interfaceProfile.shutdown = profileStatus.shutdown
   interfaceProfile.shutdownMaxDelay = profileStatus.shutdownMaxDelay
   return interfaceProfile

def showInterfaceProfiles_( mode, intfProfileName=None ):
   allInterfaceProfiles = {}
   if not intfProfileName:
      for profile in intfProfileStatusDir.status:
         if profile == '#DEFAULTPROFILE':
            continue
         profileStatus = intfProfileStatusDir.status.get( profile )
         if profileStatus:
            intfProfile = showInterfaceProfile( profileStatus )
            allInterfaceProfiles[ profile ] = intfProfile
   else:
      intfProfileStatus = intfProfileStatusDir.status.get( intfProfileName )
      if intfProfileStatus:
         intfProfile = showInterfaceProfile( intfProfileStatus )
         allInterfaceProfiles[ intfProfileName ] = intfProfile
   return allInterfaceProfiles


def showInterfaceProfilesHook( mode, profilesModel, intfProfileName=None ):
   profilesModel.intfProfiles = showInterfaceProfiles_( mode, intfProfileName )

class ShowInterfaceProfiles( ShowCommand.ShowCliCommandClass ):
   syntax = 'show maintenance profiles interface [ PROFILE_NAME ]'
   data = {
      'maintenance' : showMaintenanceMatcher,
      'profiles' : matcherProfile,
      'interface' : matcherInterface,
      'PROFILE_NAME' : profileNameMatcher( lambda mode: intfProfileStatusDir.status )
   }
   cliModel = InterfaceProfiles

   @staticmethod
   @maintRunningShowCliCheck
   def handler( mode, args ):
      allInterfaceProfiles = showInterfaceProfiles_( mode,
            args.get( 'PROFILE_NAME' ) )
      return InterfaceProfiles( intfProfiles=allInterfaceProfiles )

BasicCli.addShowCommandClass( ShowInterfaceProfiles )

#---------------------------------------------------------------------------------
# show maintenance profiles interface default
#---------------------------------------------------------------------------------
class ShowIntfMaintenanceProfileDefaultCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show maintenance profiles interface default'
   data = {
      'maintenance' : showMaintenanceMatcher,
      'profiles' : matcherProfile,
      'interface' : matcherInterface,
      'default' : defaultMatcher
   }
   cliModel = IntfMaintenanceDefaultProfile

   @staticmethod
   @maintRunningShowCliCheck
   def handler( mode, args ):
      intfDefaultProfile = IntfMaintenanceDefaultProfile()
      defaultProfileName = defaultIntfProfile.profileName
      systemDefaultProfileName = \
         Tac.Type( 'Maintenance::Profile::DefaultProfile' ).systemDefaultProfileName
      if defaultProfileName == systemDefaultProfileName:
         intfDefaultProfile.profileName = 'Default'
      else:
         intfDefaultProfile.profileName = defaultProfileName
      defaultProfileStatus = intfProfileStatusDir.status.get( defaultProfileName )
      if not defaultProfileStatus:
         return intfDefaultProfile
      intfDefaultProfile.profileAttributes = showInterfaceProfile( 
         defaultProfileStatus )
      return intfDefaultProfile

BasicCli.addShowCommandClass( ShowIntfMaintenanceProfileDefaultCmd )

def Plugin( entityManager ):
   global defaultIntfProfile
   global intfGroupConfigDir, intfGroupStatusDir, intfProfileStatusDir
   global maintenanceGroupDir, defaultBgpProfile, groupToUnitDir

   intfGroupsHook.addExtension( showInterfaceGroupsHook )
   intfProfilesHook.addExtension( showInterfaceProfilesHook )

   intfGroupConfigDir = LazyMount.mount( entityManager,
                                         'group/config/interface',
                                         'IntfGroup::ConfigDir', 'r' )
   intfGroupStatusDir = LazyMount.mount( entityManager,
                                         'group/status/interface',
                                         'IntfGroup::StatusDir', 'r' )
   maintenanceGroupDir = LazyMount.mount( 
      entityManager,
      'maintenance/group/config',
      'Maintenance::MaintenanceGroup::ConfigDir', 'r' )
   intfProfileStatusDir = LazyMount.mount( entityManager,
                                           'maintenance/profile/status/interface',
                                           'IntfMaintenanceProfile::StatusDir', 'r' )
   defaultIntfProfile = LazyMount.mount(
      entityManager,
      'maintenance/profile/config/default/interface',
      'Maintenance::Profile::DefaultProfile',
      'r' )
   defaultBgpProfile = LazyMount.mount(
      entityManager,
      'maintenance/profile/config/default/bgp',
      'Maintenance::Profile::DefaultProfile',
      'r' )
   groupToUnitDir = LazyMount.mount( entityManager,
                                     'maintenance/mapping/group',
                                     'Maintenance::GroupToUnitDir', 'r' )
