# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import io
import re
import sys

import BasicCli
import BasicCliModes
from CliMode.IntfProfile import IntfProfileMode
import CliCommand
import CliMatcher
import CliParser
import CliSave
import CliSession
import CliToken.Cli
import ConfigMount
from CliPlugin.EthIntfCli import EthIntfModelet
# pylint: disable-next=consider-using-from-import
import CliPlugin.SessionCli as SessionCli
import CliPlugin.IntfCli as IntfCli # pylint: disable=consider-using-from-import
import IntfProfileCliLib
import LazyMount
import MainCli
import ShowCommand
import Tac
import Tracing
import Url

t0 = Tracing.trace0
t1 = Tracing.trace1
t2 = Tracing.trace2

#--------------------------------------------------------------
# CLI commands/modes:
# The "interface profile <profileName>" group-change mode:
# (config)# interface profile FOO
# (config-intf-profile-FOO)# command <COMMAND-1>
# (config-intf-profile-FOO)# command <COMMAND-2>
# (config-intf-profile-FOO)# exit
#
# "[ no | default] profile PROFILE" command in config-if mode
#--------------------------------------------------------------

profileConfig = None
ethPhyIntfConfigDir = None

tokenProfileName = CliMatcher.DynamicNameMatcher( lambda mode: profileConfig.profile,
                                                  'Interface profile name' )

def getProfileConfig( profileName ):
   return profileConfig.profile.get( profileName )

def getProfileCmds( profileName ):
   profConfig = getProfileConfig( profileName )
   if profConfig:
      return list( profConfig.command.values() )
   return []

def getProfileListCmds( profileList ):
   """Return a list of all the commands defined in all the profiles
   in the profileList."""
   pCmds = []
   if profileList:
      for profile in profileList:
         pCmds.extend( getProfileCmds( profile ) )
   return pCmds

def getModifiedProfileCmds( intf, profileNames, newProfCmds ):
   """Return the profile commands for intf, replacing the commands associated
   with @profileNames with the commands in @newProfCmds."""
   profAppCfg = profileConfig.intfToProfile.get( intf )
   pCmds = []
   if profAppCfg:
      for profile in profAppCfg.profile.values():
         if profile not in profileNames:
            pCmds.extend( getProfileCmds( profile ) )
         else:
            pCmds.extend( newProfCmds )
   return pCmds

def getAppliedProfileCmds( intf ):
   """Return the profile commands that were applied to the interface."""
   profAppCfg = profileConfig.intfToProfile.get( intf )
   if profAppCfg:
      return profAppCfg.profileCmdsApplied.splitlines()
   return []

def filterDefaultCmds( canonicalCmds, intfToProfileCmds, intfDefaultCmds ):
   # For each cmd in canonicalCmds, remove the one in intfDefaultCmds unless
   # it's explicitly in intfToProfileCmds. This is related to the following
   # case:
   #
   # canonicalCmds[ intf ] : [ "description foo", "shutdown" ]
   # intfToProfileCmds[ intf ]: "description foo"
   # intfDefaultCmds[ intf ] : "shutdown"
   #
   # ==> [ "description foo" ] since "shutdown" is inherited from the default
   # command, not the profile.
   for intf, cmds in canonicalCmds.items():
      defaultCmds = intfDefaultCmds.get( intf, '' ).splitlines()
      if defaultCmds:
         newCmds = []
         profileCmds = intfToProfileCmds.get( intf, [] )
         for cmd in cmds.splitlines():
            if cmd in defaultCmds and cmd not in profileCmds:
               t0( "remove default cmd '%s' from intf %s" % ( cmd, intf ) )
            else:
               newCmds.append( cmd )
         canonicalCmds[ intf ] = '\n'.join( newCmds )

class ConfigManager:
   """
   We need to translate a set of commands (applied from profiles for each interface)
   to canonical form. We cache this information. Note if we ever support loading
   CliSave plugins without restarting ConfigAgent, we need a way to invalidate the
   cache.
   """

   # The cache will have unique mapping from "profile commands applied" to
   # "canonical commands", so we don't imagine we'll churn this very often, but
   # theoretically we could still grow large if frequent profile config changes
   # are made. So just set a limit here and once reached we reset the cache.
   MAX_CACHE_SIZE = 256

   CONFIG_CACHE = {} # ( intfType, config, defaultConfig ) -> canonical config

   # match interface type and subintf (dot)
   INTF_RE = re.compile( r'^([a-zA-Z\-]+)[^\.]*(\.*)' )

   @classmethod
   def fetchConfig( cls, em, sessionName, intfNames=None ):
      if not intfNames:
         return {}
      configBuffer = io.StringIO()
      intfFilter = set( intfNames ) if intfNames is not None else None
      sessionName = sessionName or CliSession.currentSession( em )
      if sessionName:
         CliSave.saveSessionConfig( em, configBuffer, sessionName,
                                    showProfileExpanded=True,
                                    intfFilter=intfFilter )
      else:
         CliSave.saveRunningConfig( em, configBuffer,
                                    showProfileExpanded=True,
                                    intfFilter=intfFilter )
      config = configBuffer.getvalue().rstrip()
      filteredConfig = IntfProfileCliLib.filterIntfConfig( config, intfFilter )
      t1( "fetchConfig:", config, "filtered", filteredConfig, "session", sessionName,
          "intfNames", intfNames )
      return filteredConfig

   @classmethod
   def intfType( cls, intf ):
      # each unique interface type might parse different commands, so we maintain
      # cache differently (e.g., a profile applied to port-channel might not 100%
      # match if applied to ethernet interface). See BUG995591.
      m = cls.INTF_RE.match( intf )
      return m.groups()

   @classmethod
   def saveIntfProfileCmdsToCache( cls, intfToProfileCmds,
                                   canonicalIntfToProfileCmds,
                                   intfDefaultCmds ):
      # intfToProfileCmds: intf -> list of commands we just applied
      # canonicalIntfToProfileCmds: intf -> canonical interface config
      t0( "saveIntfProfileCmdsToCache", intfToProfileCmds,
          canonicalIntfToProfileCmds,
          intfDefaultCmds )

      # To avoid unbounded caching, clear the cache whenever we hit the limit
      if len( cls.CONFIG_CACHE ) > cls.MAX_CACHE_SIZE:
         cls.CONFIG_CACHE.clear()
      for intf, cmdList in intfToProfileCmds.items():
         intfType = cls.intfType( intf )
         config = '\n'.join( cmdList )
         intfDefaultConfig = intfDefaultCmds.get( intf, '' )
         cacheKey = ( intfType, config, intfDefaultConfig )
         canonicalConfig = canonicalIntfToProfileCmds.get( intf, '' )
         t1( "saving config cache", cacheKey, "->", canonicalConfig )
         cls.CONFIG_CACHE[ cacheKey ] = canonicalConfig

   @classmethod
   def getIntfProfileCmds( cls, em, sessionName, intfToProfileCmds,
                           intfDefaultCmds ):
      # We just applied intfToProfileCmds and want to get the canonical commands
      # back. Get it from the cache if possible, or we'll just have to fetch from
      # session config.
      t0( "getIntfProfileCmds:", intfToProfileCmds, "default", intfDefaultCmds,
          "session", sessionName )
      canonicalIntfToProfileCmds = {}
      for intf, cmdList in intfToProfileCmds.items():
         intfType = cls.intfType( intf )
         config = '\n'.join( cmdList )
         defaultConfig = intfDefaultCmds.get( intf, '' )
         cacheKey = ( intfType, config, defaultConfig )
         if config:
            canonicalCmds = cls.CONFIG_CACHE.get( cacheKey )
            if canonicalCmds is not None:
               # we have it in cache
               canonicalIntfToProfileCmds[ intf ] = canonicalCmds
            else:
               t0( "getIntfProfileCmds: missed cache for", intfToProfileCmds,
                   "default", defaultConfig, "session", sessionName )
               canonicalIntfToProfileCmds = cls.fetchConfig(
                  em, sessionName,
                  intfNames=intfToProfileCmds )
               if intfDefaultCmds:
                  filterDefaultCmds( canonicalIntfToProfileCmds,
                                     intfToProfileCmds,
                                     intfDefaultCmds )
               # save to cache
               cls.saveIntfProfileCmdsToCache( intfToProfileCmds,
                                               canonicalIntfToProfileCmds,
                                               intfDefaultCmds )
               break
         else:
            canonicalIntfToProfileCmds[ intf ] = ''
      t0( "getIntfProfileCmds: get", canonicalIntfToProfileCmds )
      return canonicalIntfToProfileCmds

def makeUrlFromFile( mode, fileObj, content ):
   fileObj.write( content )
   fileObj.flush()
   return Url.parseUrl( 'file:%s' % fileObj.name, 
                        Url.Context( *Url.urlArgsFromMode( mode ) ) )

def loadConfigInSession( mode, config ):
   if not config:
      return True
   with ConfigMount.ConfigMountDisabler( disable=False ):
      t2( "load config in session:", config, mode.session.disableGuards_ )
      errors = MainCli.loadConfig( config + [ 'end' ], mode.session,
                                   initialModeClass=SessionCli.ConfigSessionMode,
                                   autoComplete=True )
   return not errors

def getIntfDefaultCmds( mode, intfNames ):
   # Run default command on one interface and get the default commands.
   # These commands should not be run after profile is applied.
   em = mode.entityManager
   sessionName = CliSession.uniqueSessionName( em, prefix="ifdefault" )
   with CliSession.TemporaryConfigSession( em, sessionName ):
      loadConfigInSession(
         mode, [ "default interface %s" % intf for intf in intfNames ] )
      intfDefaultCfg = ConfigManager.fetchConfig( em, sessionName,
                                                  intfNames=intfNames )
   return intfDefaultCfg

def splitProfileConfig( currIntfCfg, intfToProfileCmds, intfDefaultCmds ):
   # split current expanded interface config into profile and non-profile commands

   profileCfg = []
   nonProfileCfg = []
   for intfName, newProfileCmds in intfToProfileCmds.items():
      oldProfileCmds = getAppliedProfileCmds( intfName )
      if oldProfileCmds == newProfileCmds:
         continue
      overrideCmds = profileConfig.intfOverrideCmd.get( intfName )
      if overrideCmds:
         overrideCmds = overrideCmds.cmd
      else:
         overrideCmds = None
      appProfCfg = IntfProfileCliLib.AppliedProfileConfig(
         intfName,
         currIntfCfg.get( intfName, '' ),
         oldProfileCmds,
         newProfileCmds,
         overrideCmds=overrideCmds,
         defaultCmds=intfDefaultCmds.get( intfName, '' ).splitlines() )
      profileCfg.extend( appProfCfg.newProfileConfig() )
      nonProfileCfg.extend( appProfCfg.nonProfileConfig() )

   t0( "profile config:", profileCfg )
   t0( "non-profile config:", nonProfileCfg )
   return profileCfg, nonProfileCfg

def applyProfileConfig( mode, intfToProfileCmds, keepProfileOnIntf ):
   """ Given a mapping of interfaces to profile commands to apply to the interface,
   apply the profile commands to the interfaces.
   Returns a boolean indicating success."""
   t0( "applyProfileConfig:", intfToProfileCmds )
   em = mode.entityManager
   sessionName = CliSession.currentSession( em )
   inConfigSession = bool( sessionName )
   abortSession = not inConfigSession

   if not intfToProfileCmds:
      return True

   # First get the default config under an interface, so we don't run them
   # at the end of the process.
   intfDefaultCmds = getIntfDefaultCmds( mode, intfNames=intfToProfileCmds )
   t0( "intfDefaultCmds", intfDefaultCmds )

   currIntfCfg = ConfigManager.fetchConfig( mode.entityManager, sessionName,
                                            intfNames=intfToProfileCmds )

   profileCfg, nonProfileCfg = splitProfileConfig( currIntfCfg,
                                                   intfToProfileCmds,
                                                   intfDefaultCmds )
   if not profileCfg and not nonProfileCfg:
      return True

   oldIntfToProfile = {}
   if keepProfileOnIntf:
      for intfName in intfToProfileCmds:
         profile = profileConfig.intfToProfile.get( intfName )
         if profile:
            oldIntfToProfile[ intfName ] = list( profile.profile.values() )

   if not inConfigSession:
      t0( "create session" )
      sessionName = CliSession.uniqueSessionName( em, prefix="ifprof" )
      CliSession.enterSession( sessionName, entityManager=em )

   try:
      # Apply interface profile to interfaces
      #
      # intfToProfileCmds: intfId to a list of commands defined in the profile
      # sessionName: current session name or None
      # profileCfg: profile commands applied to all affected interfaces
      # nonProfileCfg: non-profile commands applied to all affected interfaces

      # now we are in the a config session
      # copy to the current session
      t0( "load profileCfg" )
      success = loadConfigInSession( mode, profileCfg )

      t0( "get applied profile commands for the interfaces" )
      profCmdsApplied = ConfigManager.getIntfProfileCmds( mode.entityManager,
                                                          sessionName,
                                                          intfToProfileCmds,
                                                          intfDefaultCmds )

      t0( "load nonProfileCfg" )
      if not loadConfigInSession( mode, nonProfileCfg ):
         success = False

      if keepProfileOnIntf:
         with ConfigMount.ConfigMountDisabler( disable=False ):
            for intfName, profileNames in oldIntfToProfile.items():
               updateProfileConfig( intfName, profileNames )

      if not inConfigSession:
         t0( "commit session" )
         CliSession.commitSession( em, sessionName=sessionName )
         CliSession.exitSession( em )
         abortSession = False

      t0( "save profCmdsApplied", profCmdsApplied )
      for intfName in intfToProfileCmds:
         profileAppCfg = profileConfig.newIntfToProfile( intfName )
         profileAppCfg.profileCmdsApplied = profCmdsApplied.get( intfName, '' )
         t0( intfName, "clear intfOverrideCmd", profileAppCfg )
         del profileConfig.intfOverrideCmd[ intfName ]

      return success

   finally:
      if abortSession:
         CliSession.abortSession( em )
         CliSession.exitSession( em )
      if not success:
         mode.addWarning( "Not all profile commands could be applied." )

def updateProfileConfig( intfName, profileNames ):
   if not profileNames:
      t0( "clear profiles applied to", intfName )
      del profileConfig.intfToProfile[ intfName ]
      return
   if profileConfig.profile.values() == profileNames:
      return
   profileAppCfg = profileConfig.newIntfToProfile( intfName )
   profileAppCfg.profile.clear()
   for profile in profileNames:
      t0( intfName, "add profile", profile )
      profileAppCfg.profile.enq( profile )

#---------------------------------------------------------------
# The "interface profile <profileName>" mode
#
# IntfProfileConfigMode is a Group-Change Mode that supports:
# (config)# interface profile FOO
# (config-intf-profile-FOO)# command <COMMAND-1>
# (config-intf-profile-FOO)# command <COMMAND-2>
# (config-intf-profile-FOO)# exit
#
# Group-change mode commands:
# (config-intf-profile-FOO)# abort
# (config-intf-profile-FOO)# show active
# (config-intf-profile-FOO)# show pending
# (config-intf-profile-FOO)# show diff
#---------------------------------------------------------------
class IntfProfileConfigMode( IntfProfileMode, BasicCli.ConfigModeBase ):
   name = 'interface profile configuration'
   showActiveCmdRegistered_ = True

   def __init__( self, parent, session, profileName ):
      self.profileName = profileName
      self.profile = Tac.newInstance( 'IntfProfile::Profile', profileName )
      self.currProfile = getProfileConfig( profileName )
      copyProfileConfig( self.currProfile, self.profile )

      IntfProfileMode.__init__( self, profileName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def commitProfile( self ):
      profile = profileConfig.profile.newMember( self.profileName )
      copyProfileConfig( self.profile, profile )

   def onExit( self ):
      """
      Update the affected profiles and interface configs, if necessary.
      Validate the commands in the profile against each intf used.
      """
      if self.needsUpdate():
         appliedIntfs = {} # Mapping of intf name to list of profile names
         intfProfCmds = {} # Mapping of intf name to profile cmds to apply
         newProfCmds = list( self.profile.command.values() )
         for intfName, config in profileConfig.intfToProfile.items():
            profileList = list( config.profile.values() )
            if self.profileName in profileList:
               appliedIntfs[ intfName ] = profileList
               intfProfCmds[ intfName ] = getModifiedProfileCmds(
                                             intfName,
                                             [ self.profileName ],
                                             newProfCmds )
         if newProfCmds:
            self.saveCommands( newProfCmds )
         applyProfileConfig( self, intfProfCmds, True )
         self.commitProfile()
      BasicCli.ConfigModeBase.onExit( self )

   def saveCommands( self, cmds ):
      if self.profile:
         self.profile.command.clear()
         for cmd in cmds:
            self.profile.command.enq( cmd )

   def abort( self ):
      self.profile = None
      self.session_.gotoParentMode()

   def needsUpdate( self ):
      if self.profile is None:
         return False # We aborted
      if self.currProfile is None:
         return True
      return list( self.currProfile.command.values() ) !=\
            list( self.profile.command.values() )

   def addProfileCommand( self, command ):
      if self.profile is not None:
         if command in self.profile.command.values():
            return
         if command.startswith( "profile " ):
            self.addError( "Nested profiles are not supported." )
            return
         self.profile.command.enq( command )

   def removeProfileCommand( self, command ):
      if self.profile:
         for idx, cmd in self.profile.command.items():
            if cmd == command:
               del self.profile.command[ idx ]

def copyProfileConfig( srcProfile, dstProfile ):
   if srcProfile:
      dstProfile.command.clear()
      for profileCmd in srcProfile.command.values():
         dstProfile.command.enq( profileCmd )

class EnterIntfProfileCommandClass( CliCommand.CliCommandClass ):
   syntax = """interface profile <profileName>"""
   noOrDefaultSyntax = """interface profile <profileName>"""

   data = { "interface": IntfCli.interfaceKwMatcher,
            "profile": "Configure interface profiles",
            "<profileName>": tokenProfileName }

   @staticmethod
   def handler( mode, args ):
      profileName = args[ '<profileName>' ]
      childMode = mode.childMode( IntfProfileConfigMode, profileName=profileName )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profileName = args[ '<profileName>' ]
      lastMode = mode.session_.modeOfLastPrompt()
      if ( isinstance( lastMode, IntfProfileConfigMode ) and lastMode.profile
           and lastMode.profileName == profileName ):
         # If we are deleting ourselves, make sure we don't write it back when
         # we exit.
         lastMode.profile = None
      if profileName not in profileConfig.profile:
         return
      appliedIntfs = {} # Mapping of intf name to list of profile names
      intfProfCmds = {} # Mapping of intf name to profile cmds to apply
      for intfName, config in profileConfig.intfToProfile.items():
         profileList = list( config.profile.values() )
         if profileName in profileList:
            appliedIntfs[ intfName ] = profileList
            intfProfCmds[ intfName ] = getModifiedProfileCmds( intfName,
                                                               [ profileName ],
                                                               [] )
      applyProfileConfig( mode, intfProfCmds, True )
      del profileConfig.profile[ profileName ]

BasicCli.GlobalConfigMode.addCommandClass( EnterIntfProfileCommandClass )

class ConfigureIntfProfile( CliCommand.CliCommandClass ):
   syntax = """command COMMAND"""
   noOrDefaultSyntax = """command COMMAND"""

   data = { "command": "Add commands to the profile",
            "COMMAND": CliMatcher.StringMatcher( helpname='LINE',
                                                 helpdesc='Command string' )
   }

   @staticmethod
   def handler( mode, args ):
      command = args[ 'COMMAND' ]
      firstToken = command.split( maxsplit=1 )[ 0 ]
      if firstToken in ( 'comment', '!!' ):
         # do not support comments
         #
         # this is not 100% robust as user can use a prefix of "comment", but then
         # they'll just see some weird behavior.
         mode.addErrorAndStop( "comments are not supported in interface profiles" )
      mode.addProfileCommand( command )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      command = args[ 'COMMAND' ]
      mode.removeProfileCommand( command )

IntfProfileConfigMode.addCommandClass( ConfigureIntfProfile )

class AbortIntfProfile( CliCommand.CliCommandClass ):
   syntax = """abort"""
   data = { "abort": CliToken.Cli.abortMatcher }

   @staticmethod
   def handler( mode, args ):
      mode.abort()

IntfProfileConfigMode.addCommandClass( AbortIntfProfile )

#-----------------------------------------------------------------
# "[ no | default] profile PROFILE" command in config-if mode
# Attach an interface profile to an actual interface
#-----------------------------------------------------------------
profileApplyKw = CliMatcher.KeywordMatcher( "profile",
                                            "Apply a profile to this interface" )
class ApplyIntfProfile( CliCommand.CliCommandClass ):
   syntax = "profile { PROFILE }"
   noOrDefaultSyntax = "profile [ { PROFILE } ]"
   data = { "profile": profileApplyKw,
            "PROFILE": tokenProfileName }

   @staticmethod
   def handler( mode, args ):
      profileList = args[ 'PROFILE' ]
      newProfileCmds = getProfileListCmds( profileList )
      t0( "new profile commands", newProfileCmds )
      intfName = mode.intf.name
      oldProfileCmds = getAppliedProfileCmds( intfName )
      t0( "old profile commands", oldProfileCmds )
      if mode.session.startupConfig():
         if profileConfig.profile:
            # If profiles are already configured before interface commands,
            # it means we are using the legacy order.
            t0( "useOverrideCmd = False" )
            profileConfig.useOverrideCmd = False
      elif not profileConfig.useOverrideCmd:
         t0( "useOverrideCmd = True" )
         profileConfig.useOverrideCmd = True

      if newProfileCmds != oldProfileCmds:
         applyProfileConfig( mode, { intfName: newProfileCmds }, False )
      updateProfileConfig( intfName, profileList )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      removeList = args.get( 'PROFILE' )
      intfName = mode.intf.name
      profAppCfg = profileConfig.intfToProfile.get( intfName )
      if not profAppCfg:
         return

      # Figure out which commands to apply based on the profiles we
      # want to remove
      currProfList = list( profAppCfg.profile.values() )
      newProfCmds = []
      newProfList = []
      if removeList:
         newProfList = [ profile for profile in currProfList
                                 if profile not in removeList ]
         if newProfList == currProfList:
            # The profiles to remove are not configured, no changes necessary.
            return
         newProfCmds = getModifiedProfileCmds( intfName, removeList, [] )

      # Apply the commands
      oldProfileCmds = getAppliedProfileCmds( mode.intf.name )
      if oldProfileCmds != newProfCmds:
         applyProfileConfig( mode, { intfName: newProfCmds }, False )
      updateProfileConfig( intfName, newProfList )

EthIntfModelet.addCommandClass( ApplyIntfProfile )

class OverrideIntfProfile( CliCommand.CliCommandClass ):
   """This is a special config command in that it is not generated by a normal
   CliSave plugin, but by CliSave infrastructure (see CliSave.processProfileCmds()).
   It actually does not touch Sysdb but keeps information in the current CliSession
   that certain commands in the profile are overriden.
   """
   syntax = "profile override COMMAND"
   noOrDefaultSyntax = syntax
   data = { "profile": profileApplyKw,
            "override": "Override a profile command not yet in effect",
            "COMMAND": CliMatcher.StringMatcher( helpname="COMMAND",
                                                 helpdesc="Command to override" )
            }

   @staticmethod
   def _getDefaultCommand( command ):
      if command.startswith( "default " ):
         return ''
      if command.startswith( 'no ' ):
         command = command[ 3: ]
      return "default " + command

   @staticmethod
   def _runCmd( mode, command ):
      if not command:
         return False
      # run the command
      try:
         mode.session.runCmd( command )
         return True
      except CliParser.GuardError as e:
         mode.addError( "(unavailable: %s)" % e.guardCode )
      except CliParser.AlreadyHandledError as e:
         mode.session_.handleAlreadyHandledError( e )
      except Exception as e: # pylint: disable=broad-except
         mode.addError( "Error running comand '%s': %s" % ( command, str( e ) ) )
      return False

   @staticmethod
   def handler( mode, args ):
      intfName = mode.intf.name
      command = args[ 'COMMAND' ]
      command = OverrideIntfProfile._getDefaultCommand( command )
      if OverrideIntfProfile._runCmd( mode, command ):
         overrideCfg = profileConfig.newIntfOverrideCmd( intfName )
         overrideCfg.cmd[ command ] = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # this command isn't really useful, but let's do something reasonable
      intfName = mode.intf.name
      command = args[ 'COMMAND' ]
      OverrideIntfProfile._runCmd( mode, command )
      overrideCfg = profileConfig.intfOverrideCmd.get( intfName )
      if overrideCfg:
         del overrideCfg.cmd[ command ]

EthIntfModelet.addCommandClass( OverrideIntfProfile )

#-----------------------------------------------------------------
# Making sure "default interface" removes the profile
#-----------------------------------------------------------------
class IntfProfileCleaner( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      del profileConfig.intfToProfile[ self.intf_.name ]
      del profileConfig.intfOverrideCmd[ self.intf_.name ]

#-----------------------------------------------------------------
# show active|pending|diff for IntfProfileConfigMode
#-----------------------------------------------------------------
def _showList( profile, profileName, output=None ):
   if output is None:
      output = sys.stdout
   if profile is None or not profile.command:
      return
   output.write( 'interface profile %s\n' % profileName )
   for oCmd in profile.command.values():
      output.write( '   command %s\n' % oCmd )

def _showDiff( mode ):
   # generate diff between active and pending
   import difflib # pylint: disable=import-outside-toplevel
   activeOutput = io.StringIO()
   _showList( mode.currProfile, mode.profileName, output=activeOutput )
   pendingOutput = io.StringIO()
   _showList( mode.profile, mode.profileName, output=pendingOutput )
   diff = difflib.unified_diff( activeOutput.getvalue().splitlines(),
                                pendingOutput.getvalue().splitlines(),
                                lineterm='' )
   print( '\n'.join( list( diff ) ) )

class ActivePendingDiffCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show [ active | diff | pending ]'
   data = {
      'active': BasicCliModes.showActiveNode,
      'pending' : 'Show pending list in this session',
      'diff' : 'Show the difference between active and pending list',
   }
   privileged = True

   @staticmethod
   def handler( mode, args ):
      if 'active' in args:
         return _showList( mode.currProfile, mode.profileName )
      elif 'diff' in args:
         return _showDiff( mode )
      else:
         return _showList( mode.profile, mode.profileName )

IntfProfileConfigMode.addShowCommandClass( ActivePendingDiffCmd )

def Plugin( entityManager ):
   global profileConfig
   global ethPhyIntfConfigDir

   profileConfig = ConfigMount.mount( entityManager, "interface/profile/config", 
                                      "IntfProfile::Config", "w" )
   ethPhyIntfConfigDir = LazyMount.mount( entityManager,
                                          "interface/config/eth/phy/all",
                                          "Interface::AllEthPhyIntfConfigDir", "r" )
   IntfCli.Intf.registerDependentClass( IntfProfileCleaner )
