#!/usr/bin/env python3
# Copyright (c) 2009-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#----------------------------------------------------------------------
# This module implements:
#   - the "config-if-range" mode
#----------------------------------------------------------------------

import collections

from CliPlugin import IntfCli
import BasicCli
import CliParser
import MultiRangeRule
import BasicCliUtil
import CliCommand
import CliExtensions
import CliMatcher
import FileUrl
import Intf.IntfRange as IntfRange # pylint: disable=consider-using-from-import
import CliRangeExpansion
import Tac
import TacSigint
import Url

class IntfRangeConfigMode( BasicCli.ConfigModeBase ):
   name = 'Interface range configuration'

   class RangeParseContext:
      ''' Place for callbacks to store state when they are being
      called in the context of an intf range command. '''
      def __init__( self, intfModes ):
         self.intfModes_ = intfModes # interface modes that run this command
         self.data_ = {}

      def hasHookData( self, hookName, hook ):
         return hook in self.data_.get( hookName, {} )

      def getHookData( self, hookName, hook ):
         return self.data_[ hookName ][ hook ]

      def setHookData( self, hookName, hook, data ):
         self.data_.setdefault( hookName, {} )[ hook ] = data

   def __init__( self, parent, session, intfList ):
      '''Enter config-if-range mode.  "intfList" should be an IntfRange.IntfList.'''
      self.intfList = intfList
      # Create a fake config-if mode for my representative member.
      # Its modelets are the ones that I need to emulate.
      intfType = intfList.type()
      self.individualIntfModes_ = []
      for iName in intfList:
         intf = intfType.getCliIntf( parent, iName )
         # Since we never call gotoIntfMode for individual interfaces while running
         # the commands, we explicitly create the intf for once and for all, at the
         # beginning. This is redundant for some intf types, for example, lag intfs
         # which are anyway accepted only if created by cli a priori; however is
         # "necessary" for physical intfs, so that Sysdb can record Cli as one of
         # requestor for the intf, and doesn't remove the intf( and its configs),
         # when, for exmaple, Fru drop its reference of the intf.
         intf.create()
         mode = IntfCli.IntfConfigMode( parent, session, intf )
         self.individualIntfModes_.append( mode )
         mode.intfRangeIs( self )
      self.modeKey = "if-range"
      listStr = str(self.intfList)
      if len(listStr) > 30:
         # Prompts that are too long look silly, and they also cause readline
         # to do really weird things, like not display the left portion of the
         # prompt at all, which confuses CliTest immensely.  So, shorten the
         # prompt.  Try to split it at commas if possible.
         le = len(listStr)
         # This code works even for strings with no "," because
         # str.find will return -1.
         c1 = 10 + listStr[10:16].find(",")
         c2 = le - 12 + listStr[-12:-6].find(",")
         listStr = listStr[:c1+1] + "..."+ listStr[c2:]      
      self.longModeKey = "if-%s" % listStr # pylint: disable=consider-using-f-string
      BasicCli.ConfigModeBase.__init__( self, parent, session, 
                                        multiInstance = True,
                                        multiModes = self.individualIntfModes_ )
      self.rangeParseContext_ = None

   def historyKey( self ):
      # Keep it the same as config-if mode
      return "(config-if)#"

   def getCompletions( self, tokens, partialToken, startWithPartialToken=False ):
      # Any of my member modes is as good as any other for completions.
      return self.individualIntfModes_[0].getCompletions( tokens, 
                                                          partialToken,
                                                          startWithPartialToken )

   # pylint: disable-next=inconsistent-return-statements
   def parse( self, tokens, autoComplete=True, authz=True, acct=True ):
      # This is very exciting.  Not many modes override mode.parse().
      # What I do is ask all of my babies to parse for me.  I never use
      # my modeRule at all.  This way, the configuration applies to all
      # interfaces.

      # Try parsing a range expansion
      if CliRangeExpansion.tryRangeParse( self, tokens, autoComplete=autoComplete,
                                          authz=authz, acct=acct ):
         return 

      # Try the parse myself.  This handles things like "exit"
      # without doing something crazy (like invoking "exit" once for
      # each interface in the range.)
      try:
         return BasicCli.ConfigModeBase.parse( self, tokens,
                                               autoComplete=autoComplete,
                                               authz=authz, acct=acct )
      except CliParser.ParseError:
         pass

      # Ask each child mode to parse; however, since the command is the same,
      # I supress authorization and accounting except the first command.
      results = {}
      error = None
      for m in self.individualIntfModes_:
         try:
            results[ m ] = m.parse( tokens, autoComplete=autoComplete, authz=authz,
                                    acct=acct )
         except CliParser.GuardError as e:
            if e.guardCode != CliParser.guardNotThisInterface:
               raise
            # Allow command to proceed if the guard code indicates that
            # this particular interface does not support the command.
            error = e

      if error and not results:
         # all interfaces raised guard error, just raise it too
         raise error # pylint: disable=raising-bad-type

      CmdHandler = collections.namedtuple( 'CmdHandler', [ 'handler', 'dummy' ] )
      cmdHandler = CmdHandler( self._invokeValueFunc, True )
      return { "cmdHandler": cmdHandler,
               "allowCache": False,
               "kargs": { "results": results, "authz": authz, "acct": acct },
               "aaa": None,
               'cmdDeprecatedBy': None }

   def _invokeValueFunc( self, mode, results, authz, acct ):
      intfModes = [ m for m in self.individualIntfModes_ if m in results ]
      self.rangeParseContext_ = IntfRangeConfigMode.RangeParseContext( intfModes )
      try:
         for m in intfModes:
            # pylint: disable-msg=W0212
            m.session._invokeValueFunc( m, results[ m ], authz, acct )

            # We do authorization for each child mode (since RBAC might reject it),
            # but not accounting
            acct = False

            # Allow keyboard interrupt
            TacSigint.check()
      finally:
         self.rangeParseContext_ = None

   def intfRangeCliHook( self, intfMode, hook, hookName, hookArgsCallback ):
      ''' Call the hook for the first interface mode in the range,
      and re-use the result in subsequent calls within the same
      intf range context. hookName is a string that is unique to the
      type of hook being called. hookArgsCallback, if not None, is
      called with the list of arguments that will be passed to the
      hook, and can add to that list.'''
      if self.rangeParseContext_:
         if not self.rangeParseContext_.hasHookData( hookName, hook ):
            # First intf mode in the range list.
            # Call the hook and cache the result.
            hookArgs = [ self.rangeParseContext_.intfModes_ ]
            if hookArgsCallback:
               hookArgsCallback( hookArgs )
            res = hook( *hookArgs )
            self.rangeParseContext_.setHookData( hookName, hook, res )
            return res
         else:
            # Re-use the value returned when the hook was called for
            # the first intf mode.
            return self.rangeParseContext_.getHookData( hookName, hook )
      else:
         # some plugins call this function directly without range
         hookArgs = [ [ intfMode ] ]
         if hookArgsCallback:
            hookArgsCallback( hookArgs )
         return hook( *hookArgs )

   # For performance reasons we do this so we don't end up fetching running
   # config for each mode
   def _showActiveRunningConfig( self, url ):
      try:
         with url.open() as f:
            if len( self.individualIntfModes_ ) > 1:
               content = f.readlines()
            else:
               # If only one mode, just pass the iterator
               content = f
            for m in self.individualIntfModes_:
               BasicCliUtil.showRunningConfigWithFilter( content, m.filterExp(), [] )

      except OSError as e:
         # pylint: disable-next=consider-using-f-string
         self.addError( "Cannot display running-config: %s" % e )

   def showActive( self ):
      """ Base implementation of show active. Sub mode can overide to
      provide alternative implementation. """
      url = FileUrl.localRunningConfig( *Url.urlArgsFromMode( self ))
      self._showActiveRunningConfig( url )

   def showActiveAll( self, showDetail ):
      url = FileUrl.localRunningConfigAll( *Url.urlArgsFromMode( self ),
                                            showDetail=showDetail )
      self._showActiveRunningConfig( url )

# extraGotoIntfRangeModeHook extensions accept two arguments: the mode
# and a list of interfaces and returns nothing
extraGotoIntfRangeModeHook = CliExtensions.CliHook()

canCreateIntfsHook = CliExtensions.CliHook()

#-------------------------------------------------------------------------------
# The "interface <name>" command, in "config" mode.
#-------------------------------------------------------------------------------
def gotoIntfRangeMode( mode, intf, create=None, exposeInactive=False, 
                       exposeUnconnected=False ):
   # This accepts single or interface range. If it's a single interface, and
   # it does not exist, the command creates it. If it's an interface range,
   # all interfaces must already exist unless 'create' is specified.
   #
   # Set startupConfig to True to skip the single-interface lookup.
   if isinstance( intf, MultiRangeRule.IntfList ):
      for hook in canCreateIntfsHook.extensions():
         error = hook( intf )
         if error:
            mode.addError( error )
            return

   if isinstance( intf, MultiRangeRule.IntfList ) and create:
      # IntfRangeConfigMode will create the interfaces for us
      childMode = mode.childMode( IntfRangeConfigMode, intfList=intf )
      mode.session_.gotoChildMode( childMode )
      intfs = IntfCli.Intf.getAll( mode, intf, silent=True, config=True,
                                   exposeInactive=exposeInactive,
                                   exposeUnconnected=exposeUnconnected )
   else:
      intfs = IntfCli.Intf.getAll( mode, intf, silent=True, config=True,
                                   exposeInactive=exposeInactive,
                                   exposeUnconnected=exposeUnconnected )
      if not intfs:
         mode.addError( "Interface does not exist" )
         return
      if len( intfs ) > 1:
         intfList = intf.newIntfList( intfs )
         childMode = mode.childMode( IntfRangeConfigMode, intfList=intfList )
      else:
         # It's a single interface. 
         # Note that we create the interface before entering the child mode, 
         # so that if for some reason the interface can't be created, we remain 
         # in the parent mode.
         intf = intfs[ 0 ]
         if not intf.intfCreationCurrentlySupported( mode ):
            return
         intf.create( mode.session_.startupConfig() )
         childMode = mode.childMode( IntfCli.IntfConfigMode, intf=intf )
      mode.session_.gotoChildMode( childMode )

   for hook in extraGotoIntfRangeModeHook.extensions():
      hook( mode, intfs )

def forceExpose( mode, intfs ):
   if mode.session_.isInteractive():
      # For interactive, always need to specify "expose" manually
      return False
   # During non-interactive period, such as copying configurations, we
   # should always expose the inactive and unconnected interfaces if a single
   # interface is specified (see BUG108174, BUG109034, BUG119701, and BUG384414 ).
   return not ( isinstance( intfs, MultiRangeRule.IntfList ) and len( intfs ) > 1 )

class IntfRangeCommand( CliCommand.CliCommandClass ):
   allowCache = False
   syntax = """interface
               ( ( range [ create ] INTFS )
               | ( [ create ] INTF_AND_RANGE ) [ inactive | unconnected expose ] )"""
   noOrDefaultSyntax = syntax
   data = {
      'interface': IntfCli.interfaceKwMatcher,
      'range': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'range',
                                            helpdesc='none' ),
         hidden=True ),
      'create': 'Create interfaces',
      # to parse Ethernet interfaces in startup config, we need to accept single
      # interface matcher as range matcher doesn't work.
      'INTFS': IntfRange.intfRangeMatcher,
      'INTF_AND_RANGE': IntfCli.intfRangeWithSingleExpression( 'INTFS' ),
      'inactive': IntfCli.inactiveKwMatcher,
      'unconnected': IntfCli.unconnectedKwMatcher,
      'expose': IntfCli.exposeKwMatcher
      }

   @staticmethod
   def commonArgs_( mode, args ):
      intfs = args[ 'INTFS' ]
      force = forceExpose( mode, intfs )
      return ( intfs, 'inactive' in args or force, 'unconnected' in args or force )

   @classmethod
   def handler( cls, mode, args ):
      intfs, exposeInactive, exposeUnconnected = cls.commonArgs_( mode, args )
      create = 'create' in args
      gotoIntfRangeMode( mode, intfs, create=create,
                         exposeInactive=exposeInactive,
                         exposeUnconnected=exposeUnconnected )

   @classmethod
   def noHandler( cls, mode, args ):
      intfs, exposeInactive, exposeUnconnected = cls.commonArgs_( mode, args )
      intfs = IntfCli.Intf.getAll( mode, intfs, silent=True, config=True,
                                   exposeInactive=exposeInactive,
                                   exposeUnconnected=exposeUnconnected )
      for intf in intfs:
         if intf.config():
            intf.noInterface()

   @classmethod
   def defaultHandler( cls, mode, args ):
      intfs, exposeInactive, exposeUnconnected = cls.commonArgs_( mode, args )
      intfs = IntfCli.Intf.getAll( mode, intfs, silent=True, config=True,
                                   exposeInactive=exposeInactive,
                                   exposeUnconnected=exposeUnconnected )
      for intf in intfs:
         if intf.config():
            intf.setDefault()

BasicCli.GlobalConfigMode.addCommandClass( IntfRangeCommand )
