#!/usr/bin/env python3
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet.IntCmdLib
import BasicCli
import CliMatcher
import FileUrl
import ShowCommand
import Url
from CliPlugin import IntfCli
from CliPlugin.FileCli import ( runningConfigAfterShowKw,
                                startupConfigAfterShowKw,
                                allKwMatcher, detailKwMatcher )

def _intfNames( ranges ):
   '''The syntax has '{ INTFS }', so we have a list of intf ranges.'''
   return [ intf for intfs in ranges for intf in intfs ]

def intfFilter( mode, url, intfNames ):
   '''This is a helper function for the show running-config interface and
   show startup-config interface commands.  The commands pass in the url
   to open to get to the config along with a list of Intf or IntfRange
   objects.  We use the Arnet IntCmdLib to do the work of matching the right
   lines from the config file.'''
   try:
      f = url.open()
      try:
         (matchedLines, error) = \
            Arnet.IntCmdLib.matchIntCmds( intfNames, f, subIntf=True )
         if error:
            mode.addError( error )
         else:
            for line in matchedLines:
               print( line, end='' )
      finally:
         f.close()
   except OSError as e:
      mode.addError( f"Error displaying {url.url} ({e.strerror})" )

#####################################################################################
## Implementation of the show running-config interface command
##     show running-config [all] [detail] interface <intf-range> [<intf-range> ... ]
#####################################################################################

showConfigIntfKwMatcher = CliMatcher.KeywordMatcher(
   'interfaces',
   helpdesc='Filter config to include only the given interfaces' )

class ShowRunningConfigIntf( ShowCommand.ShowCliCommandClass ):
   '''Filter the running config to include only the given interfaces.'''

   syntax = ( "show running-config [ all ] [ detail ] interface { INTFS }"
              " [ profile expanded ]" )
   data = {
      "running-config" : runningConfigAfterShowKw,
      "all" : allKwMatcher,
      "detail" : detailKwMatcher,
      "interface" : showConfigIntfKwMatcher,
      "INTFS" : IntfCli.Intf.rangeMatcher,
      "profile": "Option about interface profiles",
      "expanded": "Show interface config with profile content expanded",
   }
   privileged = True

   @staticmethod
   def handler( mode, args ):
      intfs = _intfNames( args[ 'INTFS' ] )
      showExpanded = 'expanded' in args
      if 'all' in args:
         url = FileUrl.localRunningConfigAll( *Url.urlArgsFromMode( mode ),
                                              showDetail=( 'detail' in args ),
                                              showProfileExpanded=showExpanded,
                                              intfFilter=intfs )
      else:
         url = FileUrl.localRunningConfig( *Url.urlArgsFromMode( mode ),
                                           showProfileExpanded=showExpanded,
                                           intfFilter=intfs )
      intfFilter( mode, url, intfs )

BasicCli.addShowCommandClass( ShowRunningConfigIntf )

#####################################################################################
## Implementation of the show startup-config interface command
##     show startup-config interface <intf-range> [<intf-range> ... ]
#####################################################################################
class ShowStartupConfigIntf( ShowCommand.ShowCliCommandClass ):
   '''Filter the startup config to include only the given interfaces.'''

   syntax = "show startup-config interface { INTFS }"
   data = {
      "startup-config" : startupConfigAfterShowKw,
      "interface" : showConfigIntfKwMatcher,
      "INTFS" : IntfCli.Intf.rangeMatcher
   }
   privileged = True

   @staticmethod
   def handler( mode, args ):
      startupConfigUrl = FileUrl.localStartupConfig( *Url.urlArgsFromMode( mode ) )
      intfFilter( mode, startupConfigUrl, _intfNames( args[ 'INTFS' ] ) )

BasicCli.addShowCommandClass( ShowStartupConfigIntf )

