# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module contains the definition of CliParser rules for matching an IPv6
# link-local address.
#
# It is defined in Intf/CliPlugin as it explicitly depends on both Arnet and
# Intf. Defining it in Arnet introduces a dependency cycle on Intf.
#-------------------------------------------------------------------------------

import Arnet
from Intf.IntfRange import intfListFromCanonical
import CliMatcher
import CliParser
import CliParserCommon
import re

class Ip6LlAddrMatcher( CliMatcher.Matcher ):
   '''Type of matcher that matches a link-local IPv6 address'''
   def __init__( self, helpdesc, **kargs ):
      super().__init__( helpdesc=helpdesc, **kargs )

      # Strings found in the exception of an incomplete ipv6 address
      self.ip6AddrIncompleteRe_ = \
         re.compile( '(incomplete ipv6 address|expected)' )

   def match( self, mode, context, token ):
      try:
         addrStr, intfName = token.split( '%' )
      except ValueError:
         return CliParserCommon.noMatch

      # The empty string is a valid IntfId, but it is invalid for our purposes
      if intfName == '':
         return CliParserCommon.noMatch

      m = Arnet.Ip6AddrCompiledRe.match( addrStr )
      if m is None or m.group( 0 ) != addrStr:
         return CliParserCommon.noMatch

      try:
         addr = Arnet.Ip6Addr( addrStr )
      except IndexError:
         return CliParserCommon.noMatch

      if not addr.isLinkLocal:
         return CliParserCommon.noMatch

      try:
         # First try if it's an EOS name
         intfId = Arnet.IntfId( intfName )
      except IndexError:
         try:
            # If it fails, perhaps its a short name -- convert to long EOS name
            intfId = Arnet.IntfId( intfListFromCanonical( [ intfName ] )[ 0 ] )
         except IndexError:
            return CliParserCommon.noMatch

      result = addr.stringValue + "%" + intfId.stringValue
      return CliParserCommon.MatchResult( result, result )

   def completions( self, mode, context, token ):
      try:
         addrStr, intfName = token.split( '%' )
      except ValueError:
         addrStr = token
         intfName = None

      if intfName == '':
         intfName = None

      m = Arnet.Ip6AddrCompiledRe.match( addrStr )
      if m is None or m.group( 0 ) != addrStr:
         return []

      partial = m.group( 0 )
      addr = None
      try:
         addr = Arnet.Ip6Addr( partial )
      except IndexError as e:
         m = self.ip6AddrIncompleteRe_.search( str( e ) )
         if m is None:
            return []
         # Ensure the partial address starts with 'fe80'
         for a, b in zip( partial, 'fe80' ):
            if a != b:
               return []
      except: # pylint: disable=bare-except
         return []

      if addr and not addr.isLinkLocal:
         return []

      completionText = 'fe80::A:B:C:D%<interface>'
      return [ CliParser.Completion( completionText, self.helpdesc_, False ) ]

   def __str__( self ):
      return '<IPv6 link-local address>'
