# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import TableOutput
from Arnet import sortIntf
from ArnetModel import Ip6Address
from CliModel import (
   Model,
   Dict,
   Float,
   List,
)
import datetime

class RouterNeighbor( Model ):
   address = Ip6Address( help="IPv6 link local address of neighbor" )
   lastRa = Float( help="UTC time at which the last RA was received from"
                        "the neighbor" )

class RouterNeighbors( Model ):
   routerNeighbors = List( valueType=RouterNeighbor,
           help="IPv6 link local address of neighbors discovered by"
           "Router Advertisement" )

class InterfaceNeighbors( Model ):
   interfaces = Dict( keyType=str, valueType=RouterNeighbors,
                      help="A mapping of interface to its neighbors" )
  
   def createTable( self ):
      headers = [ "Interface", "IPv6 Address", "Last RA Received" ]
      table = TableOutput.createTable( headers, indent=0 )
      columnFormat = TableOutput.Format( justify='center', wrap=True )
      table.formatColumns( *( [ columnFormat ] * len( headers ) ) )
      return table

   def render( self ):
      if not self.interfaces:
         return
      table = self.createTable()   
      for intf in sortIntf( self.interfaces ):
         for neighbor in self.interfaces[ intf ].routerNeighbors:
            relativeTime = int( Tac.utcNow() - neighbor.lastRa )
            # convert to HH:MM:SS format
            lastRaTime = str( datetime.timedelta( seconds=relativeTime ) )
            table.newRow( intf, neighbor.address, lastRaTime + " ago" )
      print( table.output() )

class VrfNeighbors( Model ):
   vrfs = Dict( keyType=str, valueType=InterfaceNeighbors,
           help="A mapping of VRF name to its neighbors" )

   def render( self ):
      for vrf in sorted( self.vrfs ):
         print( "VRF: " + vrf )
         self.vrfs[ vrf ].render()
