#!/usr/bin/env python3
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=no-value-for-parameter
# pylint: disable=consider-using-f-string

#---------------------------------------------------------------------
# This module implements the following show commands:
#
# show rib next-hop ( ip | ipv6 ) dependency
#
# show rib route ( ip | ipv6 ) [ VRF ] [ PREFIX ] [ PROTOCOL ]
#    [ fib policy excluded ] [ orr-position POSITION ] [ debug ]
# show rib multicast route ( ip | ipv6 ) [ VRF ] [ PREFIX ] [ PROTOCOL ]
# show rib route ( ip | ipv6 ) internal [ VRF ]
#
# show rib route summary [ ip | ipv6 ] [ VRF ]
# show rib route summary [ ip | ipv6 ] brief
#
# show rib [ multicast ] loop ( ip | ipv6 ) [ VRF ] route [ PREFIX ]
#
# show rib next-hop resolution route ( ipv4 | ipv6 ) [ VRF ]
#
# show rib [ multicast ] next-hop ( ip | ipv6 ) [ VRF ] [ PROTOCOL ]
#    [ orr-position POSITION ]
#
# show rib next-hop evpn [ VRF ] [ PROTOCOL ]
#
# show rib ready [ VRF ]
#
# show vrf leak connected ( ipv4 | ipv6 )
#
# show tech-support iprib graceful-restart
#---------------------------------------------------------------------

'''IpRib Show Commands'''

import AgentCommandRequest
import Arnet
import ConnectedRouteLeakModel as Crlm
import IpRib
from CliPlugin import IpRibCliLib
from CliPlugin.IpRibLib import getProtocolShowString
from CliPlugin.IraServiceCli import getEffectiveProtocolModel
from CliPlugin.ShowRibReadyModels import ( ProtoReadyModel,
                                           RibReadyModel,
                                           RibReadyStateModel )
from CliPlugin.VrfCli import ( ALL_VRF_NAME,
                               DEFAULT_VRF,
                               VrfExecCmdDec,
                               generateVrfCliModel,
                               getVrfNames,
                               vrfExists )
from CliPlugin.IraCommonCli import ( ribReadyHook,
                           ribResolutionRouteHook,
                           ribRouteHook,
                           ribRouteModel )
import IpRibModel
import QuickTrace
import sys
import Tac
import TacSigint
from IpLibTypes import ProtocolAgentModelType as ProtoAgentModel
from CliPrint import CliPrint
from contextlib import contextmanager
from Toggles.IpRibToggleLib import toggleShowTechIpRibUpdateEnabled
from Toggles.IpRibToggleLib import toggleNoRouteConfigTrieUsageEnabled

printer = CliPrint().lib

vrfMounter = IpRibCliLib.IpRibVrfMounter()
ribMounter = IpRibCliLib.IpRibCliMounter( vrfMounter, "rib" )
mribMounter = IpRibCliLib.IpRibCliMounter( vrfMounter, "mrib" )
orrPositionMounter = IpRibCliLib.IpRibOrrPositionMounter()
orrPositionRibMounter = \
      IpRibCliLib.IpRibOrrPositionCliMounter( orrPositionMounter, "rib" )
tunnelRibMounter = IpRibCliLib.TunnelRibMounter()
fibRtStatusMounter = IpRibCliLib.FibRouteStatusMounter( vrfMounter )
nhDependencyMounter = IpRibCliLib.NhDependencyMounter()
allVrfIgpReadyMounter = IpRibCliLib.AllVrfIgpReadyMounter()
allVrfRibReadyMounter = IpRibCliLib.AllVrfRibReadyMounter()
fibReadyDirMounter = IpRibCliLib.FibReadyDirMounter()
fecMounter = IpRibCliLib.ForwardingStatusMounter()

entMan = None
qv = QuickTrace.Var
qt0 = QuickTrace.trace0
qt8 = QuickTrace.trace8

AddressFamily = Tac.Type( 'Arnet::AddressFamily' )

ribReadyVrfModel = generateVrfCliModel( RibReadyModel,
                                        'Per VRF Rib ready summary',
                                        uncheckedModel=True )

def getCliMounter( rib="rib", position="" ):
   # If a position is specified, fetch orrPositionRibMounter
   if position != "":
      return orrPositionRibMounter

   if rib == "rib":
      return ribMounter
   elif rib == "mrib":
      return mribMounter
   assert False
   return None

def getCliShmemMounter( rib="rib" ):
   return IpRibCliLib.IpRibShmemCliMounter( entMan, rib )

#--------------------------------------------------------------------------------
# Implements show rib next-hop ( ip | ipv6 ) dependency
#--------------------------------------------------------------------------------
def showNhDependency( mode, args ):
   nhDepGraph = nhDependencyMounter.getNhDependencyGraph()
   viaStatus = ribMounter.getViaStatus()
   vrfStatus = vrfMounter.getVrfIdStatus()
   inputAf = "ipv4" if "ip" in args else "ipv6"
   nhgEntryStatus = ribMounter.getNhgEntryStatus()

   nhp = Tac.newInstance( "Routing::Rib::NhDependencyPrinter", nhDepGraph,
                          viaStatus, vrfStatus, inputAf, nhgEntryStatus )

   fd = sys.stdout.fileno()
   outputFormat = mode.session_.outputFormat()
   nhp.renderNhDependency( fd, outputFormat )
   return IpRibModel.RecursiveNhDependencyByVrf

#--------------------------------------------------------------------------------
# Implements show rib route ( ip | ipv6 ) [ VRF ] [ PREFIX ] [ PROTOCOL ]
#               [ fib policy excluded ] [ orr-position POSITION ] [ debug ]
#            show rib multicast route ( ip | ipv6 ) [ VRF ] [ PREFIX ]
#               [ PROTOCOL ]
#            show rib route ( ip | ipv6 ) internal [ VRF ]
#--------------------------------------------------------------------------------
def correctProtocolAndAddrFamily( protocol, af ):
   return ( ( protocol != 'ospf' or af == 'ipv4' ) and
            ( protocol != 'ospf3' or af == 'ipv6' ) )

@contextmanager
def initPrinterCtx( fd, outputFormat ):
   prt = printer.initPrinter( fd, outputFormat, True )
   try:
      yield prt
   finally:
      printer.deinitPrinter( prt )

@contextmanager
def objCtx( prt ):
   if prt is None:
      yield
   else:
      printer.startRender( prt )
      yield
      printer.endRender( prt )

@contextmanager
def dictCtx( prt, dictName ):
   if prt is None:
      yield
   else:
      printer.startDict( prt, dictName )
      yield
      printer.endDict( prt, dictName )

@contextmanager
def simpleDictEntryCtx( prt, dictName ):
   if prt is None:
      yield
   else:
      printer.startSimpleDictEntry( prt, dictName )
      yield
      printer.endSimpleDictEntry( prt )

def showIpRouteRev2EmptyOutput( prt ):
   with objCtx( prt ), dictCtx( prt, 'ribRoutesByProtocol' ):
      pass

def showIpRouteOrrPosition( prt, vrfName, vrfId, mode, fd, outputFormat,
                            rib, af, prefix, protocol, fibExcluded, tag,
                            protocolOrHost, position, displayHostRoutes,
                            revision ):
   orrPosition = orrPositionMounter.getPosition( position )
   if orrPosition is None:
      if revision < 3:
         showIpRouteRev2EmptyOutput( prt )
      qt0( 'ORR position %s does not exist' % position )
      return IpRibModel.RibRoutesByVrf

   # Only Default Vrf is supported
   if vrfName != DEFAULT_VRF:
      if revision < 3:
         showIpRouteRev2EmptyOutput( prt )
      qt0( 'Only default VRF is supported for BGP ORR position' )
      return IpRibModel.RibRoutesByVrf

   # If protocol is specified, it should be Isis
   if protocolOrHost is not None and protocol != 'isis':
      if revision < 3:
         showIpRouteRev2EmptyOutput( prt )
      qt0( 'Only isis protocol is supported for BGP ORR position' )
      return IpRibModel.RibRoutesByVrf

   if revision < 3:
      prt = None

   with objCtx( prt ), dictCtx( prt, 'vrfs' ):
      cliMounter = getCliMounter( rib, orrPosition )
      ( routeConfig, routeConfigTrie ) = \
            cliMounter.getRouteConfig( af, vrfName, orrPosition )

      if routeConfig is None or \
         ( not toggleNoRouteConfigTrieUsageEnabled() and routeConfigTrie is None ):
         qt0( 'VRF %s : Mounts Incomplete' % vrfName )
         return IpRibModel.RibRoutesByVrf

      igpResultStatusReaderHelper = cliMounter.getIgpResultStatusReaderHelper()

      rr = Tac.newInstance( "Routing::Rib::RouteRenderer",
                            vrfName, vrfId, routeConfig,
                            routeConfigTrie, None, None,
                            None, None, None,
                            None, None, None, None,
                            None, None, displayHostRoutes,
                            None, None, None, None,
                            None, None,
                            vrfMounter.getVrfIdStatus(), None,
                            None, None, igpResultStatusReaderHelper,
                            False, orrPosition, revision, False )

      if not mode.session_.shouldPrint():
         rr.jsonModelRevision = mode.session_.requestedModelRevision()

      with simpleDictEntryCtx( prt, vrfName ), TacSigint.immediateMode():
         # A tag value of 0 is interpreted as any other tag value.
         # In "0 if tag is None else tag", 0 is arbitrary.
         # It prevents a TypeError from occurring (expected U32, got NoneType).
         rr.renderRoute( fd, outputFormat, prefix, protocol, fibExcluded,
                         tag is not None, 0 if tag is None else tag )

   return IpRibModel.RibRoutesByVrf

def showIpRouteByVrf( prt, vrfName, vrfId, mode, fd, outputFormat,
                      rib, af, prefix, protocol, fibExcluded, tag,
                      internal, displayHostRoutes, debug ):
   cliMounter = getCliMounter( rib )
   ( routeConfig, routeConfigTrie ) = cliMounter.getRouteConfig( af, vrfName )
   viaSetConfig = cliMounter.getAllViaSetConfig()
   ipv4ViaConfig = cliMounter.getIpViaConfig( vrfName, 'ipv4' )
   ipv6ViaConfig = cliMounter.getIpViaConfig( vrfName, 'ipv6' )
   # Evpn vias are in default vrf
   evpnViaConfig = cliMounter.getEvpnViaConfig( DEFAULT_VRF )
   mplsViaConfig = cliMounter.getMplsViaConfig( vrfName )
   ipv4ViaCfgByProtoByVrf = cliMounter.getAllViaConfigPerTransport( 'ipv4' )
   ipv6ViaCfgByProtoByVrf = cliMounter.getAllViaConfigPerTransport( 'ipv6' )

   viaStatus = cliMounter.getViaStatus()
   viaSetStatus = cliMounter.getAllViaSetStatus()

   winningRouteStatus = cliMounter.getWinningRouteStatus( af, vrfName )
   loopingRouteStatus = cliMounter.getLoopingRouteStatus( af, vrfName )
   nhgEntryStatus = cliMounter.getNhgEntryStatus()
   tunnelFib = cliMounter.getTunnelFib()
   ribConfigBgp = cliMounter.getRibConfigBgp( vrfId )
   fec4Status = fecMounter.getFec4Status()

   routeIpv4Status = fibRtStatusMounter.getFib4RouteStatus( vrfName )
   routeIpv6Status = fibRtStatusMounter.getFib6RouteStatus( vrfName )

   cliShmemMounter = getCliShmemMounter( rib )
   viaMetricStatus = cliShmemMounter.getViaMetricStatus( force=True )
   revision = mode.session_.requestedModelRevision()

   if ( routeConfig is None or
        ( not toggleNoRouteConfigTrieUsageEnabled() and routeConfigTrie is None ) or
        viaSetConfig is None or ipv4ViaConfig is None or
        ipv6ViaConfig is None or evpnViaConfig is None or
        viaStatus is None or viaMetricStatus is None or
        winningRouteStatus is None or loopingRouteStatus is None or
        nhgEntryStatus is None or tunnelFib is None or
        routeIpv4Status is None or routeIpv6Status is None ):
      qt0( 'VRF %s : Mounts Incomplete' % vrfName )
      return

   igpResultStatusReaderHelper = cliMounter.getIgpResultStatusReaderHelper()

   rr = Tac.newInstance( "Routing::Rib::RouteRenderer", vrfName, vrfId,
                         routeConfig, routeConfigTrie, viaSetConfig,
                         ipv4ViaConfig, ipv6ViaConfig, evpnViaConfig,
                         mplsViaConfig, ipv4ViaCfgByProtoByVrf,
                         ipv6ViaCfgByProtoByVrf, viaStatus, viaMetricStatus,
                         winningRouteStatus, loopingRouteStatus,
                         displayHostRoutes, nhgEntryStatus, tunnelFib, None,
                         ribConfigBgp, routeIpv4Status, routeIpv6Status,
                         vrfMounter.getVrfIdStatus(), viaSetStatus,
                         tunnelRibMounter.tunnelRibNameIdMap, fec4Status,
                         igpResultStatusReaderHelper, False, "", revision, debug )

   if not mode.session_.shouldPrint():
      rr.jsonModelRevision = mode.session_.requestedModelRevision()

   with simpleDictEntryCtx( prt, vrfName ), TacSigint.immediateMode():
      # A tag value of 0 is interpreted as any other tag value.
      # In "0 if tag is None else tag", 0 is arbitrary.
      # It prevents a TypeError from occurring (expected U32, got NoneType).
      rr.renderRoute( fd, outputFormat, prefix, protocol, fibExcluded,
                      tag is not None, 0 if tag is None else tag )

   if internal:
      print( "Internal State for VRF", vrfName )
      sys.stdout.flush()
      # Dumps output to the cli instance that's invoking the command. If set to None
      # only cohab tests will be able to see the output (which gets dumped to stdout)
      dumpCtx = Tac.Type( 'Ark::DumpContext' ).dumpContext( fd, 'dumpNormal', False )
      indentCount = 2
      if af == 'ipv4':
         ipv4ViaConfig.dumpState( dumpCtx, indentCount )
      if af == 'ipv6':
         ipv6ViaConfig.dumpState( dumpCtx, indentCount )
      if vrfName == DEFAULT_VRF:
         evpnViaConfig.dumpState( dumpCtx, indentCount )
      viaStatus.dumpState( dumpCtx, indentCount )
      viaMetricStatus.dumpState( dumpCtx, indentCount )
      viaSetConfig.dumpState( dumpCtx, indentCount )
      if routeConfig is not None:
         routeConfig.dumpState( dumpCtx, indentCount )
      winningRouteStatus.dumpState( dumpCtx, indentCount )
      print( "Tunnel RIB" )
      sys.stdout.flush()
      tunnelRibMounter.tunnelRib.dumpState( dumpCtx, indentCount )

def showIpRoute( mode, args ):
   rib = 'mrib' if 'multicast' in args else 'rib'
   af = 'ipv4' if 'ip' in args else 'ipv6'
   vrfArg = args.get( 'VRF', DEFAULT_VRF )
   prefix = args.get( 'PREFIX' )
   protocolOrHost = args.get( 'PROTOCOL' )
   internal = 'internal' in args
   fibExcluded = 'excluded' in args
   tag = args.get( 'TAG' )
   position = args.get( 'POSITION', None )
   displayHostRoutes = False
   shamlink = 'shamlink' in args
   debug = 'debug' in args

   if protocolOrHost is None:
      protocol = Tac.Type( "Routing::Rib::RoutingProtocol" ).routingProtocols
   elif protocolOrHost == 'host':
      protocol = 'connected'
      displayHostRoutes = True
   else:
      protocol = IpRibCliLib.ipRibProtocolString.externalToInternal( protocolOrHost )

   if prefix is None:
      prefix = Tac.Value( "Arnet::IpGenPrefix" )
   else:
      if not isinstance( prefix, str ):
         prefix = prefix.stringValue

      prefix = Arnet.IpGenPrefix( prefix )

   fd = sys.stdout.fileno()
   outputFormat = mode.session_.outputFormat()
   revision = IpRibModel.RibRoutesByVrf.__revision__
   if not mode.session_.shouldPrint():
      revision = mode.session_.requestedModelRevision()

   with initPrinterCtx( fd, outputFormat ) as prt:
      # Support ribd mode
      # Sham link runs in multi-Agent and doesn't export OSPF routes to IpRib.
      # These routes are hidden routes and present only in gated.
      # Hence, we are re-using the existing
      # single agent code to send a dget command to gated
      if getEffectiveProtocolModel( mode ) == ProtoAgentModel.ribd or \
            shamlink:
         # Host routes, internal state, and multicast are not (yet) supported. If
         # any of these were specified by the user, an empty result is returned.
         # Call single agent code only if none of these were specified.
         if ( not displayHostRoutes and not internal and rib == 'rib' and
              not ( revision >= 3 and vrfArg == ALL_VRF_NAME ) or shamlink ):
            # Certain permutations of filters may not be supported.
            # Ex ipv4 + ospf3 and ipv6 + ospf
            if correctProtocolAndAddrFamily( protocol, af ):
               # A temporary solution to wrap the old output in the new model
               prt = None if revision < 3 else prt
               if shamlink and vrfArg == ALL_VRF_NAME:
                  # vrf all is supported for shamlink
                  vrfsToConsider = vrfMounter.getAllVrfNames( mode )
               else:
                  # Only user provided vrf is considered
                  vrfsToConsider = [ vrfArg ]
               vrfsToConsider.sort( key=lambda vrf:( vrf != DEFAULT_VRF, vrf ) )
               with objCtx( prt ), dictCtx( prt, 'vrfs' ):
                  for vrfArg in vrfsToConsider:
                     if vrfArg == DEFAULT_VRF and shamlink:
                        # shamlink not supported in default vrf
                        continue
                     with simpleDictEntryCtx( prt, vrfArg ):
                        # invokes showRibdRoute
                        ribRouteHook.notifyExtensions( af, mode, vrfArg, protocol,
                                                       prefix,
                                                       fibExcluded=fibExcluded,
                                                       shamlink=shamlink )
                        sys.stdout.flush()
         else:
            mode.addError( 'Not supported' )
            return None

         # Deferred Model - return the model class
         return IpRibModel.RibRoutesByVrf

      # Before printing the beginning of the json dict
      # Check if the vrfName exists for single vrf input
      vrfId = vrfMounter.getVrfId( vrfArg )
      if vrfId is None:
         if revision >= 3 and vrfArg != ALL_VRF_NAME:
            mode.addError( 'VRF %s does not exist' % vrfArg )
            return None
         elif revision < 3:
            showIpRouteRev2EmptyOutput( prt )
            return IpRibModel.RibRoutesByVrf

      if fibExcluded:
         if ( not correctProtocolAndAddrFamily( protocol, af ) or displayHostRoutes
              or internal or rib != 'rib' ):
            mode.addError( 'Not supported' )
            return None

      # BGP Optimal-Route-Reflection Position Rib( AID/9125 )
      if position is not None:
         return showIpRouteOrrPosition( prt, vrfArg, vrfId, mode, fd, outputFormat,
                                        rib, af, prefix, protocol, fibExcluded, tag,
                                        protocolOrHost, position, displayHostRoutes,
                                        revision )

      if revision < 3:
         showIpRouteByVrf( None, vrfArg, vrfId, mode, fd, outputFormat,
                           rib, af, prefix, protocol, fibExcluded, tag,
                           internal, displayHostRoutes, debug )
         return IpRibModel.RibRoutesByVrf

      with objCtx( prt ), dictCtx( prt, 'vrfs' ):
         firstVrf = True
         for vrfName in vrfMounter.iterVrfNames( mode, vrfArg ):
            vrfId = vrfMounter.getVrfId( vrfName )
            if vrfId is None:
               qt0( 'VRF %s missing' % vrfName )
               continue

            if firstVrf:
               firstVrf = False
            else:
               printer.addFrills( prt, '\n', 0 )

            showIpRouteByVrf( prt, vrfName, vrfId, mode, fd, outputFormat,
                              rib, af, prefix, protocol, fibExcluded, tag,
                              internal, displayHostRoutes, debug )

   return IpRibModel.RibRoutesByVrf

#--------------------------------------------------------------------------------
# Implements show rib route summary [ ip | ipv6 ] [ VRF ]
#            show rib route summary [ ip | ipv6 ] brief
#--------------------------------------------------------------------------------
omittedProtocols = ( 'reserved', 'cbf' )

def getVrfRouteCountsToRibRouteSummaryHierarchical( cliMounter, vrfs,
                                                    addressFamilies, isBrief=False,
                                                    model=None ):
   if model is None:
      model = IpRibModel.RibRouteSummary()
   firstVrf = vrfs[ 0 ]
   for vrf in vrfs:
      for af in addressFamilies:
         ( routeConfig, _ ) = cliMounter.getRouteConfig( af, vrf )
         for protocol, protocolRouteConfig in routeConfig.protocolConfig.items():
            isFlattened = IpRibCliLib.isRouteConfigFlattened( protocol, af )
            if protocol in omittedProtocols or isFlattened:
               continue
            showString = getProtocolShowString( protocol )
            routeCnt = len( protocolRouteConfig.route )
            # If summary is brief total count of routes
            # should be put in a single summary
            if isBrief:
               if firstVrf not in model.ribRouteSummary:
                  model.ribRouteSummary[ firstVrf ] = \
                     IpRibModel.RibRouteSummaryForVrf()
               vrfModel = model.ribRouteSummary[ firstVrf ]
            else:
               if vrf not in model.ribRouteSummary:
                  model.ribRouteSummary[ vrf ] = IpRibModel.RibRouteSummaryForVrf()
               vrfModel = model.ribRouteSummary[ vrf ]
            vrfModel.routeCountPerProtocol[ showString ] = \
               vrfModel.routeCountPerProtocol.get( showString, 0 ) + routeCnt
   return model

def getVrfRouteCountsToRibRouteSummaryFlattened( cliMounter, vrfs,
                                                 addressFamilies, isBrief=False,
                                                 model=None ):
   if model is None:
      model = IpRibModel.RibRouteSummary()
   vrfCounts = {}
   # Since for flattened protocols routes for all vrfs
   # can be found in the same table
   # their number per vrf can be counted in a single iteration
   # over this table, or from the table's length (for brief summary)
   firstVrf = vrfs[ 0 ]
   for af in addressFamilies:
      ( routeConfig, _ ) = cliMounter.getRouteConfig( af, firstVrf )
      for protocol, protocolRouteConfig in routeConfig.protocolConfig.items():
         isFlattened = IpRibCliLib.isRouteConfigFlattened( protocol, af )
         if protocol in omittedProtocols or not isFlattened:
            continue
         showString = getProtocolShowString( protocol )
         routes = protocolRouteConfig.route
         # If summary is brief total count of routes
         # should be put in a single summary
         if isBrief:
            vrfModel = model.ribRouteSummary[ firstVrf ]
            vrfModel.routeCountPerProtocol[ showString ] = \
               vrfModel.routeCountPerProtocol.get( showString, 0 ) + len( routes )
         else:
            for route in routes:
               vrfCounts[ route.vrfId ] = vrfCounts.get( route.vrfId, 0 ) + 1
            for vrf in vrfs:
               vrfId = vrfMounter.getVrfId( vrf )
               vrfSummary = model.ribRouteSummary[ vrf ]
               vrfSummary.routeCountPerProtocol[ showString ] = \
                  vrfSummary.routeCountPerProtocol.get( showString, 0 ) + \
                  vrfCounts.pop( vrfId, 0 )
            vrfCounts.clear()
   return model

def getRibRouteSummaryForVrfFromShark( mode, ribRouteSummary, vrfName,
                                       addressFamilies, modelForVrf=None ):
   if modelForVrf is None:
      modelForVrf = IpRibModel.RibRouteSummaryForVrf()

   vrfId = vrfMounter.getVrfId( vrfName )
   summaryForVrf = ribRouteSummary.vrfRibRouteSummary.get( vrfId )
   if summaryForVrf is None:
      qt0( 'Summary for VRF %s missing' % vrfName )
      return modelForVrf

   summaries = []
   if 'ipv4' in addressFamilies:
      summaries.append( summaryForVrf.v4RibRouteSummary.ribRouteCount )
   if 'ipv6' in addressFamilies:
      summaries.append( summaryForVrf.v6RibRouteSummary.ribRouteCount )

   for ribRouteCount in summaries:
      for proto, routeCount in ribRouteCount.items():
         # pylint: disable-next=consider-using-in
         if proto == 'reserved' or proto == 'cbf':
            continue
         protoString = getProtocolShowString( proto )
         if protoString not in modelForVrf.routeCountPerProtocol:
            modelForVrf.routeCountPerProtocol[ protoString ] = routeCount.count
         else:
            modelForVrf.routeCountPerProtocol[ protoString ] += routeCount.count

   return modelForVrf

def getAddressFamilies( addressFamily ):
   addressFamilies = []
   if addressFamily is None:
      addressFamilies = [ 'ipv4', 'ipv6' ]
   elif addressFamily == 'ip':
      addressFamilies = [ 'ipv4' ]
   else:
      addressFamilies = [ addressFamily ]
   return addressFamilies

def showRibRouteSummary( mode, args ):
   vrf = args.get( 'VRF', DEFAULT_VRF )
   addressFamilies = getAddressFamilies( args.get( 'ip' ) or args.get( 'ipv6' ) )
   vrfs = vrfMounter.getAllVrfNames( mode ) if vrf == ALL_VRF_NAME else [ vrf ]
   cliMounter = getCliMounter( 'rib' )
   model = IpRibModel.RibRouteSummary()

   for vrf in vrfs:
      if vrfMounter.getVrfId( vrf ) is None:
         return model
   getVrfRouteCountsToRibRouteSummaryHierarchical( cliMounter, vrfs, addressFamilies,
                                                   False, model )
   getVrfRouteCountsToRibRouteSummaryFlattened( cliMounter, vrfs, addressFamilies,
                                                False, model )
   return model

def showRibRouteSummaryBrief( mode, args ):
   addressFamilies = getAddressFamilies( args.get( 'ip' ) or args.get( 'ipv6' ) )
   cliMounter = getCliMounter( 'rib' )
   vrfs = vrfMounter.getAllVrfNames( mode )
   firstVrf = vrfs[ 0 ]
   model = IpRibModel.RibRouteSummary()

   getVrfRouteCountsToRibRouteSummaryHierarchical( cliMounter, vrfs, addressFamilies,
                                                   True, model )
   getVrfRouteCountsToRibRouteSummaryFlattened( cliMounter, vrfs, addressFamilies,
                                                True, model )
   return model.ribRouteSummary[ firstVrf ]

#--------------------------------------------------------------------------------
# Implements show rib [ multicast ] loop ( ip | ipv6 ) [ VRF ] route [ PREFIX ]
#--------------------------------------------------------------------------------
def showIpRouteLooped( mode, args ):
   rib = 'mrib' if 'multicast' in args else 'rib'
   af = 'ipv4' if 'ip' in args else 'ipv6'
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   prefix = args.get( 'PREFIX' )

   revision = mode.session_.requestedModelRevision()

   if prefix is None:
      prefix = Tac.Value( "Arnet::IpGenPrefix" )
   else:
      if not isinstance( prefix, str ):
         prefix = prefix.stringValue

      prefix = Arnet.IpGenPrefix( prefix )

   vrfId = vrfMounter.getVrfId( vrfName )
   if vrfId is None:
      return IpRibModel.LoopedRoutes()

   cliMounter = getCliMounter( rib )

   loopingRouteStatus = cliMounter.getLoopingRouteStatus( af, vrfName )

   if loopingRouteStatus is None:
      return IpRibModel.LoopedRoutes()

   rr = Tac.newInstance( "Routing::Rib::RouteRenderer", vrfName, vrfId, None,
                         None, None, None, None, None, None, None, None, None, None,
                         None, loopingRouteStatus, True, None, None, None, None,
                         None, None, vrfMounter.getVrfIdStatus(), None,
                         tunnelRibMounter.tunnelRibNameIdMap, None, None, False, "",
                         revision, False )

   fd = sys.stdout.fileno()
   outputFormat = mode.session_.outputFormat()

   with TacSigint.immediateMode():
      rr.renderLoopRoute( fd, outputFormat, prefix )

   sys.stdout.flush()

   # Deferred Model - return an empty model
   return IpRibModel.LoopedRoutes

#--------------------------------------------------------------------------------
# Implements show rib next-hop resolution route ( ipv4 | ipv6 ) [ VRF ]
#--------------------------------------------------------------------------------
def showResolutionRoutes( mode, args ):
   af = 'ipv4' if 'ipv4' in args else 'ipv6'
   vrfName = args.get( 'VRF', DEFAULT_VRF )

   if getEffectiveProtocolModel( mode ) == ProtoAgentModel.ribd:
      # Support ribd mode. Call Rib agent function showResolutionRibdRoute
      ribResolutionRouteHook.notifyExtensions( af, mode, vrfName )
      return IpRibModel.ResolutionRoutes

   vrfId = vrfMounter.getVrfId( vrfName )
   if vrfId is None:
      return IpRibModel.ResolutionRoutes

   cliMounter = getCliMounter( 'rib' )
   # We don't need the routeConfigTrie
   ( routeConfig, _ ) = cliMounter.getRouteConfig( af, vrfName )
   ipv4ViaConfig = cliMounter.getIpViaConfig( vrfName, 'ipv4' )
   ipv6ViaConfig = cliMounter.getIpViaConfig( vrfName, 'ipv6' )
   # Evpn vias are in default vrf
   evpnViaConfig = cliMounter.getEvpnViaConfig( DEFAULT_VRF )
   mplsViaConfig = cliMounter.getMplsViaConfig( vrfName )
   viaSetConfig = cliMounter.getViaSetConfig( vrfName )
   viaStatus = cliMounter.getViaStatus()
   nhResPolicyStatus = cliMounter.getNhResStatus( vrfName )
   nhgEntryStatus = cliMounter.getNhgEntryStatus()
   tunnelFib = cliMounter.getTunnelFib()

   ipv4ViaCfgByProtoByVrf = cliMounter.getAllViaConfigPerTransport( 'ipv4' )
   ipv6ViaCfgByProtoByVrf = cliMounter.getAllViaConfigPerTransport( 'ipv6' )

   cliShmemMounter = getCliShmemMounter( 'rib' )
   viaMetricStatus = cliShmemMounter.getViaMetricStatus( force=True )

   revision = mode.session_.requestedModelRevision()

   if ( routeConfig is None or
        ipv4ViaConfig is None or
        ipv6ViaConfig is None or
        evpnViaConfig is None or
        viaSetConfig is None or
        viaStatus is None or
        viaMetricStatus is None or
        nhResPolicyStatus is None or
        nhgEntryStatus is None or
        tunnelFib is None ):
      qt0( "Mounts incomplete" )
      return IpRibModel.ResolutionRoutes

   igpResultStatusReaderHelper = cliMounter.getIgpResultStatusReaderHelper()

   rr = Tac.newInstance( "Routing::Rib::RouteRenderer", vrfName, vrfId, routeConfig,
                         None, viaSetConfig, ipv4ViaConfig, ipv6ViaConfig,
                         evpnViaConfig, mplsViaConfig, ipv4ViaCfgByProtoByVrf,
                         ipv6ViaCfgByProtoByVrf, viaStatus, viaMetricStatus, None,
                         None, True, nhgEntryStatus, tunnelFib, nhResPolicyStatus,
                         None, None, None, vrfMounter.getVrfIdStatus(), None,
                         tunnelRibMounter.tunnelRibNameIdMap, None,
                         igpResultStatusReaderHelper, False, "", revision, False )

   fd = sys.stdout.fileno()
   outputFormat = mode.session_.outputFormat()
   # BUG282681 - Once we modify NextHopResPolicyConfig/Status to be per-AF,
   #             the appropriate nhResPolicyStatus should be fetched, then we do
   #             not need to pass in af into renderResolutionRoute
   with TacSigint.immediateMode():
      rr.renderResolutionRoute( fd, outputFormat, af )

   sys.stdout.flush()

   # Deferred Model - return an empty Model
   return IpRibModel.ResolutionRoutes

#--------------------------------------------------------------------------------
# Implements show rib [ multicast ] next-hop ( ip | ipv6 ) [ VRF ] [ PROTOCOL ]
#               [ orr-position POSITION ]
#--------------------------------------------------------------------------------
def showIpRibNextHop( mode, args ):
   rib = 'mrib' if 'multicast' in args else 'rib'
   af = 'ipv4' if 'ip' in args else 'ipv6'

   transport = getattr( Tac.Value( "Routing::Rib::Transport" ), af )
   if 'evpn' in args:
      transport = Tac.Value( "Routing::Rib::Transport" ).evpn

   vrfName = args.get( 'VRF', DEFAULT_VRF )
   protocolOrHost = args.get( 'PROTOCOL' )
   detail = 'detail' in args
   position = args.get( 'POSITION', None )

   displayHostRoutes = False

   # only specific protocols are supported for evpn transport, renderer code
   # handles that based on passed transport
   if protocolOrHost is None:
      protocol = Tac.Type( "Routing::Rib::RoutingProtocol" ).routingProtocols
   elif protocolOrHost == 'host':
      protocol = 'connected'
      displayHostRoutes = True
   else:
      protocol = IpRibCliLib.ipRibProtocolString.externalToInternal( protocolOrHost )

   vrfId = vrfMounter.getVrfId( vrfName )
   if vrfId is None:
      return IpRibModel.RibNextHopsByProtocol()

   ipv4ViaConfig = None
   ipv6ViaConfig = None
   evpnViaConfig = None
   mplsViaConfig = None

   revision = mode.session_.requestedModelRevision()

   # BGP Optimal-Route-Reflection Position Rib( AID/9125 )
   # ORR is not supported for evpn transport today
   if position is not None:
      # If protocol is specified, it should be Bgp
      if protocolOrHost is not None and protocol != 'bgp':
         return IpRibModel.RibNextHopsByProtocol()

      # Only Default Vrf is supported
      if vrfName != DEFAULT_VRF:
         return IpRibModel.RibNextHopsByProtocol()
      orrPosition = orrPositionMounter.getPosition( position )
      if orrPosition is None:
         return IpRibModel.RibNextHopsByProtocol()

      cliMounter = getCliMounter( rib, orrPosition )
      if af == 'ipv4':
         ipv4ViaConfig = cliMounter.getIpViaConfig( vrfName, 'ipv4', orrPosition )
      elif af == 'ipv6':
         ipv6ViaConfig = cliMounter.getIpViaConfig( vrfName, 'ipv6', orrPosition )
      ipv4ViaCfgByProtoByVrf = \
            cliMounter.getAllViaConfigPerTransport( 'ipv4', orrPosition )
      ipv6ViaCfgByProtoByVrf = \
            cliMounter.getAllViaConfigPerTransport( 'ipv6', orrPosition )

      viaMetricStatus = cliMounter.getViaMetricStatus( orrPosition )

      if ( ( ipv4ViaConfig is None and ipv6ViaConfig is None ) or
            viaMetricStatus is None ):
         return IpRibModel.RibNextHopsByProtocol()

      rr = Tac.newInstance( "Routing::Rib::RouteRenderer",
                            vrfName, vrfId, None, None, None, ipv4ViaConfig,
                            ipv6ViaConfig, None, None, ipv4ViaCfgByProtoByVrf,
                            ipv6ViaCfgByProtoByVrf, None, viaMetricStatus, None,
                            None, displayHostRoutes, None, None, None, None, None,
                            None, vrfMounter.getVrfIdStatus(), None, None, None,
                            None, False, orrPosition, revision, False )
   else:
      cliMounter = getCliMounter( rib )

      if transport == Tac.Value( "Routing::Rib::Transport" ).evpn:
         evpnViaConfig = cliMounter.getEvpnViaConfig( vrfName )
      else:
         if af == 'ipv4':
            ipv4ViaConfig = cliMounter.getIpViaConfig( vrfName, 'ipv4' )
         elif af == 'ipv6':
            ipv6ViaConfig = cliMounter.getIpViaConfig( vrfName, 'ipv6' )
         else:
            mplsViaConfig = cliMounter.getMplsViaConfig( vrfName )

      viaSetConfig = cliMounter.getViaSetConfig( vrfName )
      ipv4ViaCfgByProtoByVrf = cliMounter.getAllViaConfigPerTransport( 'ipv4' )
      ipv6ViaCfgByProtoByVrf = cliMounter.getAllViaConfigPerTransport( 'ipv6' )

      viaStatus = cliMounter.getViaStatus()

      nhgEntryStatus = cliMounter.getNhgEntryStatus()
      tunnelFib = cliMounter.getTunnelFib()
      fec4Status = fecMounter.getFec4Status()

      cliShmemMounter = getCliShmemMounter( rib )
      viaMetricStatus = cliShmemMounter.getViaMetricStatus( force=True )

      if ( ( ipv4ViaConfig is None and ipv6ViaConfig is None and
            evpnViaConfig is None )
           or ( viaStatus is None or viaMetricStatus is None or
                 nhgEntryStatus is None or tunnelFib is None ) ):
         return IpRibModel.RibNextHopsByProtocol()

      rr = Tac.newInstance( "Routing::Rib::RouteRenderer",
                            vrfName, vrfId, None, None, viaSetConfig,
                            ipv4ViaConfig, ipv6ViaConfig, evpnViaConfig,
                            mplsViaConfig, ipv4ViaCfgByProtoByVrf,
                            ipv6ViaCfgByProtoByVrf, viaStatus, viaMetricStatus,
                            None, None,
                            displayHostRoutes, nhgEntryStatus, tunnelFib,
                            None, None, None, None,
                            vrfMounter.getVrfIdStatus(), None,
                            tunnelRibMounter.tunnelRibNameIdMap, fec4Status,
                            None, detail, "", revision, False )

   fd = sys.stdout.fileno()
   outputFormat = mode.session_.outputFormat()

   with TacSigint.immediateMode():
      rr.renderNextHop( fd, outputFormat, protocol,
         Tac.Value( "Routing::Rib::Transport", transport ) )

   sys.stdout.flush()

   # Deferred Model - return an empty model
   return IpRibModel.RibNextHopsByProtocol

#--------------------------------------------------------------------------------
# Implements show rib ready [ VRF ]
#
# This command is implemented in both 'ribd' and 'multi-agent' protocol models.
# Although, in the multi-agent model, this command has been capified as well.
# The output for this command differs a bit too, in both these models.
# The registration for the command was earlier done in gated but has been moved to
# IpRib to prevent dependency issues. The command callback
# method takes care of whether to call the 'ribd' implementation or 'multi-agent'
# implementation based on the protocol model in use.
#
# Sample output of command
# =======================
# Vrf default
# =======================
# Start time : 0:00:43 ago
#
#    State                             StateReached      TimeTaken( in sec )
# --------------------------------- ------------------ ---------------------
#    Fib ready                         True                            2.490
#    ...
#
#    Protocol     Configured     ConfigPending     Ready    TimeTaken( in sec )
# -------------- ------------   ----------------  -------  --------------------
#    connected    True           False             True                   1.700
#    ...
#--------------------------------------------------------------------------------
@VrfExecCmdDec( getVrfsFunc=getVrfNames, cliModel=ribReadyVrfModel )
def showRibReady( mode, args ): # pylint: disable-msg=inconsistent-return-statements
   vrfName = args.get( 'VRF', DEFAULT_VRF )

   if getEffectiveProtocolModel( mode ) == ProtoAgentModel.ribd:
      # We are running with protocol model ribd. So the output should be fetched via
      # the method gated has registered as one of the extensions in its CliPlugin
      for hook in ribReadyHook.extensions():
         return hook( mode, vrfName )

   ribReadyModel = RibReadyModel()
   # pylint: disable-msg=protected-access
   ribReadyModel._protocolModel = "multi-agent"

   if not vrfExists( vrfName ):
      mode.addError( "Vrf %s doesn't exist" % ( vrfName ) )
      return

   vrfIgpReady = allVrfIgpReadyMounter.allVrfIgpReady.vrfIgpReady.get( vrfName )
   vrfRibReady = allVrfRibReadyMounter.allVrfRibReady.vrfRibReady.get( vrfName )

   if not vrfIgpReady or not vrfRibReady:
      return

   ribReadyModel._vrf = vrfName # pylint: disable-msg=protected-access
   ribReadyModel.startTime = IpRibCliLib.toUtc( vrfRibReady.startTime )
   ribReadyModel.gracefullyRestarting = vrfRibReady.gracefullyRestarting
   # The routes dict is use by ribd command, but this needs to be set to
   # None so that an empty dict does not show up in the json output
   ribReadyModel.routes = None

   def roundedTimeInterval( readyTime ):
      timeTaken = readyTime - vrfRibReady.startTime
      if timeTaken > 0:
         return round( timeTaken, 3 )
      else:
         # ConnectedRoute and StaticRoute processing is independent of when IpRib
         # starts and startTime is recorded. If ConnectedRoute and StaticRoute have
         # nothing to do and declare themselves ready even before IpRib starts,
         # this situation will occur.
         return 0.0

   stateModel = RibReadyStateModel()
   stateModel.igpReady = vrfIgpReady.ready
   if vrfIgpReady.ready:
      stateModel.igpReadyTime = IpRibCliLib.toUtc( vrfIgpReady.readyTime )
      stateModel.timeTakenForIgpReady = roundedTimeInterval( vrfIgpReady.readyTime )
   stateModel.ribReady = vrfRibReady.vrfReady
   if vrfRibReady.vrfReady:
      stateModel.ribReadyTime = IpRibCliLib.toUtc( vrfRibReady.vrfReadyTime )
      stateModel.timeTakenForRibReady = \
                                    roundedTimeInterval( vrfRibReady.vrfReadyTime )
   stateModel.recursiveResolution = vrfRibReady.vrfRecursiveResolutionReady
   if vrfRibReady.vrfRecursiveResolutionReady:
      stateModel.recursiveResolutionTime = \
                     IpRibCliLib.toUtc( vrfRibReady.vrfRecursiveResolutionReadyTime )
      stateModel.timeTakenForRecursiveResolution = \
                  roundedTimeInterval( vrfRibReady.vrfRecursiveResolutionReadyTime )
   if ( fibReady := fibReadyDirMounter.fibReadyDir.get( vrfName ) ) is not None:
      stateModel.fibReady = fibReady.vrfReady
   # To make sure that we do not incorrectly display fibReadyTime as negative value
   # when we switch from ribd to multi-agent model, we check that fibReady is set
   # and so is fibReadyTime
   if stateModel.fibReady and vrfRibReady.fibReadyTime:
      stateModel.fibReadyTime = IpRibCliLib.toUtc( vrfRibReady.fibReadyTime )
      stateModel.timeTakenForFibReady = \
                                    roundedTimeInterval( vrfRibReady.fibReadyTime )

   ribReadyModel.states = stateModel

   protocolModels = {}
   for protoEntry in vrfRibReady.vrfRibProtoReady:
      proto = vrfRibReady.vrfRibProtoReady[ protoEntry ]
      protocol = proto.routeProtoName
      if protocol == "staticConfig":
         protocol = "static"
      elif protocol == "ospf3":
         protocol = "ospfv3"
      model = ProtoReadyModel()
      model.configPending = proto.configPending
      model.configured = proto.configured
      model.ready = proto.ready
      if proto.readyTime:
         model.readyTime = IpRibCliLib.toUtc( proto.readyTime )
         model.timeTaken = roundedTimeInterval( proto.readyTime )
      protocolModels[ protocol ] = model

   ribReadyModel.protocols = protocolModels

   return ribReadyModel

#--------------------------------------------------------------------------------
# Implements show vrf leak connected ( ipv4 | ipv6 )
#--------------------------------------------------------------------------------
def showVrfLeakConnected( mode, args ):
   af = 'ipv4' if 'ipv4' in args else 'ipv6'

   crlStatus = ribMounter.getCrLeakStatus( af )
   model = Crlm.LeakedConnectedRoutesByVrf()
   vrfIdStatus = vrfMounter.getVrfIdStatus()
   model.initFromTacModel( crlStatus, vrfIdStatus )

   return model

#--------------------------------------------------------------------------------
# Implements show tech-support iprib graceful-restart
#            show tech-support iprib optimal-route-reflection
#            show tech-support iprib via-set-config
#            show tech-support iprib via-set-status
#            show tech-support iprib via-status
#            show tech-support iprib resolved-via
#--------------------------------------------------------------------------------
def showTechIpRib( mode, args ):
   acrCmd = []
   vrf = args.get( 'VRF' )
   protoOrHost = args.get( 'PROTOCOL' )
   activeFilters = set()

   if vrf:
      if vrfExists( vrf ):
         acrCmd.append( f"VRF {vrf}" )
         activeFilters.add( 'VRF' )
      else:
         if vrf == "all":
            error = "all vrf is the default value"
         else:
            error = f"Vrf {vrf} doesn't exist"
         mode.addErrorAndStop( error )

   # do processing to convert external to internal protocol names
   if protoOrHost:
      if protoOrHost == 'host':
         proto = 'connected'
      else:
         proto = IpRibCliLib.ipRibProtocolString.externalToInternal( protoOrHost )
      # Add the protocol to ACR command if it exists
      acrCmd.append( f"PROTOCOL {proto}" )
      activeFilters.add( 'PROTOCOL' )


   cliToAcr = {
      'graceful-restart' : 'DUMP_GR_STATE',
      'optimal-route-reflection' : 'DUMP_ORR',
      'via-config' : 'DUMP_VIA_CONFIG',
      'via-set-config' : 'DUMP_VIA_SET_CONFIG',
      'via-set-status' : 'DUMP_VIA_SET_STATUS',
      'via-status' : 'DUMP_VIA_STATUS',
      'resolved-via' : 'DUMP_RESOLVED_VIA',
   }

   # only implemented filters are set as allowed
   cliToAllowedFilters = {
      'graceful-restart' : { 'VRF' },
      'optimal-route-reflection' : set(),
      'via-config' : { 'VRF' },
      'via-set-config' : { 'PROTOCOL' },
      'via-set-status' : { 'PROTOCOL' },
      'via-status' : { 'VRF', 'PROTOCOL' },
      'resolved-via' : set(),
   }

   # initially set to all filters available
   invalidFilters = { 'VRF', 'PROTOCOL' }

   if toggleShowTechIpRibUpdateEnabled():
      for cmd in args[ 'CATEGORIES' ]:
         validFilters = cliToAllowedFilters[ cmd ]
         invalidFilters.difference_update( validFilters )
         # Check if invalid filters are applied
         if invalidFilters.intersection( activeFilters ):
            error = "{} does not support the filter(s): {}".format(
                  cmd, invalidFilters.intersection( activeFilters ) )
            mode.addErrorAndStop( error )

         acrCmd.append( cliToAcr[ cmd ] )
   else:
      for cmd in args:
         # skip show, tech-support, iprib, filters, verbose and NUM
         if cmd not in cliToAcr:
            continue

         acrCmd.append( cliToAcr[ cmd ] )

   if "verbose" in args:
      acrCmd.append( "VERBOSE {}".format( args[ "NUM" ] ) )

   acrCmdStr = " ".join( acrCmd )
   AgentCommandRequest.runSocketCommand( entMan, IpRib.agentName(), 'debug',
                                         acrCmdStr )

# ----------------------------------------------------
# Implements show agent IpRib debug dynamic-endpoints
# ----------------------------------------------------
def showAgentDebugIpRib( mode, args ):
   acrCmd = []

   cliToAcr = {
      'dynamic-endpoints' : 'DUMP_DYNAMIC_ENDPOINTS',
      'unicast-rib' : 'DUMP_CUSTOM_UNICAST_RIB',
   }

   for cmd in args:
      # skip show, tech-support, iprib, filters, verbose and NUM
      if cmd not in cliToAcr:
         continue

      acrCmd.append( cliToAcr[ cmd ] )

   if "verbose" in args:
      acrCmd.append( "VERBOSE {}".format( args[ "NUM" ] ) )

   acrCmdStr = " ".join( acrCmd )
   AgentCommandRequest.runSocketCommand( entMan, IpRib.agentName(), 'debug',
                                         acrCmdStr )

def Plugin( entityManager ):
   global entMan

   entMan = entityManager

   vrfMounter.setEntityManager( entityManager )
   ribMounter.setEntityManager( entityManager )
   orrPositionMounter.setEntityManager( entityManager )
   orrPositionRibMounter.setEntityManager( entityManager )
   mribMounter.setEntityManager( entityManager )
   tunnelRibMounter.setEntityManager( entityManager )
   fibRtStatusMounter.setEntityManager( entityManager )
   nhDependencyMounter.setEntityManager( entityManager )
   allVrfIgpReadyMounter.setEntityManager( entityManager )
   allVrfRibReadyMounter.setEntityManager( entityManager )
   fibReadyDirMounter.setEntityManager( entityManager )
   fecMounter.setEntityManager( entityManager )
   ribRouteModel[ 'IpRib' ] = IpRibModel
