# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

# CLI expressions and helpers for IpRibLib types.

from CliPlugin.TunnelRibCli import (
   getCustomTunnelRibNames,
   systemTunnelRibName,
   systemColoredTunnelRibName,
)
import Tac
import CliCommand
from CliCommand import singleNode
import CliMatcher
from CliToken.IpRibLibCliTokens import ( matcherTunnelRib,
                                         matcherSystemTunnelRib,
                                         matcherColoredTunnelRib,
                                         matcherSystemColoredTunnelRib )

excludeTunnelNames = '^(?!{}$|{}$|{}$)'.format( systemTunnelRibName,
                                                systemColoredTunnelRibName,
                                                'colored' )

tunnelMatcher = CliMatcher.DynamicNameMatcher(
   getCustomTunnelRibNames, 'Name of custom tunnel RIB',
   excludeTunnelNames + r'([A-Za-z0-9_-]+)$', helpname='NAME' )

# These matchers are added separately so as to be able to set maxMatches
# on them individually
ipRibsSysConMatcher = CliMatcher.KeywordMatcher(
   'system-connected',
   helpdesc='Resolve using system connected RIB' )

ipRibsSysUniMatcher = CliMatcher.KeywordMatcher(
   'system-unicast-rib',
   helpdesc='Resolve using system unicast RIB' )

# support hidden system-unicast-rib token. See BUG479809
systemConnected = CliMatcher.KeywordMatcher(
      'system-connected',
      helpdesc='Resolve using system connected RIB' )

resolutionRibsTunnelExprString = '''
( tunnel-rib ( {} | ( colored {} ) | TUNNEL_RIB ) )  '''.format( systemTunnelRibName,
                                             systemColoredTunnelRibName )

nonColorRibsTunnelExprString = '''
( tunnel-rib ( %s | TUNNEL_RIB ) )  ''' % ( systemTunnelRibName )

# one shared matcher object to ensure that system-connected and
# system-unicast-rib are mutually exclusive and another to ensure
# that system-tunnel-rib and named tunnel rib are mutually exclusive
systemRibSharedMatcherObj = object()
tunnelRibSharedMatcherObj = object()

class ResolutionRibsExprFactory( CliCommand.CliExpressionFactory ):
   validExprNames = {
      'SYS_CON_ONLY_RIBS': 'system-connected',
      'IP_ONLY_RIBS': 'system-connected | system-unicast-rib',
      'TUNNEL_ONLY_RIBS': '{ %s }' % resolutionRibsTunnelExprString,
      'RIBS': '''
            { system-connected
            | system-unicast-rib
            | %s } ''' % ( resolutionRibsTunnelExprString ),
      'RIBS_HIDDEN': '''
            { system-connected
            | system-unicast-rib
            | %s } ''' % ( resolutionRibsTunnelExprString ),
      'NON_COLOR_TUNNEL_RIBS': ' %s ' % nonColorRibsTunnelExprString,
      'IP_NON_COLOR_TUNNEL_ONLY_RIBS': '''
            { system-connected
            | system-unicast-rib
            | %s } ''' % ( nonColorRibsTunnelExprString ),
   }
   def generate( self, name ):
      assert name in self.validExprNames
      class Expr( CliCommand.CliExpression ):
         expression = self.validExprNames[ name ]
         data = {
            'system-connected' : 
            CliCommand.Node( matcher=ipRibsSysConMatcher, 
                             maxMatches=1,
                             sharedMatchObj=systemRibSharedMatcherObj ),
            'system-unicast-rib' : 
            CliCommand.Node( matcher=ipRibsSysUniMatcher, 
                             maxMatches=1, 
                             hidden=name=='RIBS_HIDDEN',
                             sharedMatchObj=systemRibSharedMatcherObj ),
            'tunnel-rib': matcherTunnelRib,
            systemTunnelRibName:
            CliCommand.Node( matcher=matcherSystemTunnelRib,
                             maxMatches=1,
                             sharedMatchObj=tunnelRibSharedMatcherObj),
            'colored': singleNode( matcher=matcherColoredTunnelRib ),
            systemColoredTunnelRibName: 
            singleNode( matcher=matcherSystemColoredTunnelRib ),
            'TUNNEL_RIB':
            CliCommand.Node( matcher=tunnelMatcher,
                             maxMatches=1,
                             sharedMatchObj=tunnelRibSharedMatcherObj )
         }

         @staticmethod
         def adapter( mode, args, argsList ):
            '''
            This method creates the necessary ResolutionRibConfig for each
            resolution rib token and sets the corresponding resolution methods.

            argsList:  is a list where each element of the list is a list of the 
            cliCommand nodes followed by the arguments supplied to that node.

            for example: the resolution rib profile:
               'system-connected tunnel-rib foo'
            supplies  the following argList
            eg. [['system-connected','']['tunnel-rib','']['TUNNEL_RIB', 'foo']]
            '''
            ribs = []
            resRibSrcType = Tac.Type( 'Routing::Rib::ResolutionRibSource' )
            for index, ( matcher, _ ) in enumerate( argsList ):
               if matcher == 'system-connected':
                  resRib = Tac.Value( 'Routing::Rib::ResolutionRibConfig',
                                      resRibSrcType.systemConnected )
                  ribs.append( resRib )
               elif matcher == 'system-unicast-rib':
                  resRib = Tac.Value( 'Routing::Rib::ResolutionRibConfig',
                                      resRibSrcType.systemUnicastRib )
                  if name == 'RIBS_HIDDEN':
                     resRib = Tac.Value( 'Routing::Rib::ResolutionRibConfig',
                                         resRibSrcType.systemConnected )
                  ribs.append( resRib )
               elif matcher == 'tunnel-rib':
                  resRib = Tac.Value( 'Routing::Rib::ResolutionRibConfig',
                                      resRibSrcType.tunnelRib )
                  if argsList[ index + 1 ][ 1 ] != 'colored':
                     # Next token will specify the tunnel rib name
                     resRib.tunnelRibName = argsList[ index + 1 ][ 1 ]
                  else:
                     # We include additional colored token before tunnel rib name
                     assert argsList[ index + 1 ][ 0 ] == 'colored'
                     resRib.tunnelRibName = argsList[ index + 2 ][ 1 ]
                  ribs.append( resRib )

            resRibProf = Tac.Value( 'Routing::Rib::ResolutionRibProfileConfig' )
            for idx, resRibConf in enumerate( ribs ):
               resRibProf.resolutionMethod[ idx ] = resRibConf

            args[ name ] = resRibProf

      return Expr

def getProtocolShowString( protocol ):
   protocolShowStrings = {
      'bgp' : 'BGP',
      'cbf' : 'CBF',
      'connected' : 'Connected',
      'dynamicPolicy' : 'Dynamic policy',
      'isis' : 'IS-IS',
      'ospf' : 'OSPF',
      'ospf3' : 'OSPFv3',
      'routeInput' : 'Route input',
      'routeInputRouteCacheConfig' : 'Route Input Route Cache',
      'staticConfig' : 'Static',
      'staticRouteCacheConfig' : 'Static Route Cache',
      'vrfLeak' : 'VRF leak',
      'bgpLu' : 'BGP labeled unicast',
      'rip' : 'RIP',
      'gribi' : 'GRIBI'
   }
   return protocolShowStrings[ protocol ]
