# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliPlugin import IpGenAddrMatcher
from CliPlugin import TunnelIntfCli
from CliPlugin.Ssl import profileNameMatcher
from CliPlugin.VrfCli import VrfExprFactory
from CliPlugin import AclCli
import AclCliLib
import BasicCli
import BasicCliModes
import CliCommand
import CliMatcher
from CliMode.Ipsec import IkePolicyMode
from CliMode.Ipsec import IpsecEntropyMode
from CliMode.Ipsec import IpsecProfileMode
from CliMode.Ipsec import IpSecurityMode
from CliMode.Ipsec import SecurityAssociationMode
from CliMode.Ipsec import SecurityAssociationVxlanMode
from CliMode.Ipsec import KeyControllerMode
from CliMode.Ipsec import IpsecIkeCryptoSuiteCliMode
from CliMode.Ipsec import IpsecSaCryptoSuiteCliMode
import CliParser
import CliToken.Ip
import ConfigMount
import CliGlobal
import LazyMount
import Tac
import Toggles.IpsecToggleLib
from Toggles.AclToggleLib import toggleIpsecServiceAclEnabled
from Toggles.AclToggleLib import toggleIpsecServiceAclVrfEnabled
from Toggles.IpsecToggleLib import toggleIpsecClearCommandEnabled
from Toggles.IpsecToggleLib import toggleMultiCryptoEnabled
from Toggles.IpsecToggleLib import togglePfConnectionLossEnabled
from Toggles.IpsecToggleLib import toggleIpsecRemoteIpAnyEnabled
from Toggles.PolicyMapToggleLib import togglePolicyBasedVpnEnabled
from TypeFuture import TacLazyType
import ReversibleSecretCli
from IpLibConsts import DEFAULT_VRF
from socket import IPPROTO_UDP, IPPROTO_ESP
import CliSession

#Assigning globals.
gv = CliGlobal.CliGlobal( dict( ipsecConfigMode=None,
                                ikeConfigDir=None,
                                ikeStatusDir=None,
                                allVrfStatusLocal=None,
                                tunIntfConfigDir=None,
                                tunIntfStatusDir=None,
                                hwCapabilities=None,
                                ipsecCapabilities=None,
                                aclConfig=None,
                                aclCpConfig=None,
                                ipsecPathStatus=None ) )

dpdActionEnum = Tac.Type( 'Ipsec::Ike::IpsecDpdAction' )
IpsecHmacAlgorithmEnum = Tac.Type( 'Ipsec::IpsecHmacAlgorithm' )
IpsecAuthTypeEnum = Tac.Type( 'Ipsec::Ike::IpsecAuthType' )
IpsecEspAlgorithmEnum = Tac.Type( 'Ipsec::IpsecEspAlgorithm' )
IpsecDhGroupEnum = Tac.Type( 'Ipsec::IpsecDhGroup' )
IpsecEncapProtocolEnum = Tac.Type( 'Ipsec::Ike::IpsecEncapProtocol' )
IpsecFlowParallelization = TacLazyType( 'Ipsec::Ike::FlowParallelization' )
IpsecEncapSrcPortRange = TacLazyType( 'Ipsec::Ike::EncapSrcPortRange' )
IpsecSaTypeEnum = Tac.Type( 'Ipsec::IpsecSaType' )
IpsecCryptoAlgoDefault = Tac.Type( 'Ipsec::IpsecCryptoAlgoDefault' )

DUP_ENTRY_ERROR_STR = "This entry already exists with {} suite name: {}"

ikeEncryptionTypeData = {
   'aes128': 'AES - 128-bit Advanced Encryption Standard',
   'aes192': 'AES - 192-bit Advanced Encryption Standard',
   'aes256': 'AES - 256-bit Advanced Encryption Standard',
   '3des': 'Three key triple DES',
}

saEncryptionTypeData = {
   '3des': 'Three key triple DES',
   'aes128': 'AES - 128-bit Advanced Encryption Standard',
   'aes256': 'AES - 256-bit Advanced Encryption Standard',
   'null': 'Null Encryption',
}

saGcmEncryptionTypeData = {
   'aes128gcm64': '128 bit AES-GCM with 64 bit ICV',
   'aes128gcm128': '128 bit AES-GCM with 128 bit ICV',
   'aes256gcm128': '256 bit AES-GCM with 128 bit ICV',
}

ikeIntegrityTypeData = {
   'md5': 'Message Digest Algorithm 5',
   'sha1': 'Secure Hash Standard 1 (160 bit)',
   'sha256': 'Secure Hash Standard 2 (256 bit)',
   'sha384': 'Secure Hash Standard 2 (384 bit)',
   'sha512': 'Secure Hash Standard 2 (512 bit)',
}

saIntegrityTypeData = {
   'md5': 'Message Digest Algorithm 5',
   'sha1': 'sha1 security profile',
   'sha256': 'sha256 security profile',
   'sha384': 'sha384 security profile',
   'sha512': 'sha512 security profile',
   'null': 'null security profile',
}

dhGroupTypeData = {
   '1': 'Diffie-Hellman group 1 (768 bit)',
   '2': 'Diffie-Hellman group 2 (1024 bit)',
   '5': 'Diffie-Hellman group 5 (1536 bit)',
   '14': 'Diffie-Hellman group 14 (2048 bit)',
   '15': 'Diffie-Hellman group 15 (3072 bit)',
   '16': 'Diffie-Hellman group 16 (4096 bit)',
   '17': 'Diffie-Hellman group 17 (6144 bit)',
   '19': 'Diffie-Hellman group 19 (256 bit ecp)',
   '20': 'Diffie-Hellman group 20 (384 bit ecp)',
   '21': 'Diffie-Hellman group 21 (521 bit ecp)',
   '24': 'Diffie-Hellman group 24 (2048 bit, 256 bit subgroup)'
}

encryptionMap = {
   '3des': IpsecEspAlgorithmEnum.des,
   'aes128': IpsecEspAlgorithmEnum.aes128,
   'aes192': IpsecEspAlgorithmEnum.aes192,
   'aes256': IpsecEspAlgorithmEnum.aes256,
   'aes128gcm64': IpsecEspAlgorithmEnum.aes128gcm64,
   'aes128gcm128': IpsecEspAlgorithmEnum.aes128gcm128,
   'aes256gcm128': IpsecEspAlgorithmEnum.aes256gcm128,
   'null': IpsecEspAlgorithmEnum.nullesp,
}

integrityMap = {
   'md5': IpsecHmacAlgorithmEnum.md5,
   'sha1': IpsecHmacAlgorithmEnum.sha1,
   'sha256': IpsecHmacAlgorithmEnum.sha256,
   'sha384': IpsecHmacAlgorithmEnum.sha384,
   'sha512': IpsecHmacAlgorithmEnum.sha512,
   'null': IpsecHmacAlgorithmEnum.nullhash,
}

defaultIkeSuite = Tac.Value( "Ipsec::CryptoSuite",
                              IpsecCryptoAlgoDefault.ikeEncryptionDefault,
                              IpsecCryptoAlgoDefault.ikeIntegrityDefault,
                              IpsecCryptoAlgoDefault.ikeDhGroupDefault )

defaultSaSuite = Tac.Value( "Ipsec::CryptoSuite",
                             IpsecCryptoAlgoDefault.saEspAesDefault,
                             IpsecCryptoAlgoDefault.saEspShaDefault,
                             IpsecCryptoAlgoDefault.saPfsGroupDefault )

#------------------------------------------------------------------------
# Hook for TunnelIntf to confirm if an Ipsec profile is present in config.
#-------------------------------------------------------------------------
def isIpsecProfilePresent( profileName ):
   errMessage = ''
   if profileName not in ikeConfig().ipsecProfile:
      errMessage = f'IPsec profile {profileName} has not been created'
      return ( False, errMessage )
   return ( True, errMessage )

def ipsecConfigMode():
   return gv.ipsecConfigMode

def ikeConfig():
   return gv.ikeConfigDir

def ikeStatus():
   return gv.ikeStatusDir

def capabilities():
   return gv.ipsecCapabilities

def createOrGetIpsecProfileConfig( profile ):
   profile = ikeConfig().ipsecProfile.get( profile )
   if profile is None:
      profile = ikeConfig().ipsecProfile.newMember ( profile )
   return profile

def createOrUpdateTransformSetConfig( tsKey, espAes, espSha ):
   ts = ikeConfig().securityAssoc.get( tsKey )
   if ts is None:
      ts = ikeConfig().securityAssoc.newMember( tsKey, espAes, espSha )
   else:
      ts.espAes = espAes
      ts.espSha = espSha

def getOrCreateAppliedPolicy( policy ):
   appliedPolicy = ikeStatus().appliedPolicy.get( policy )
   if appliedPolicy is None:
      appliedPolicy  = ikeStatus().appliedPolicy.newMember( policy )
   return appliedPolicy

def getOrCreateAppliedSa( sa ):
   appliedSa = ikeStatus().appliedSa.get( sa )
   if appliedSa is None:
      appliedSa  = ikeStatus().appliedSa.newMember( sa )
   return appliedSa

def deletePolicyMapping( policy, profile ):
   policyMap = getOrCreateAppliedPolicy( policy )
   del policyMap.appliedPolicy[ profile.profile ]
   if not policyMap.appliedPolicy:
      del ikeStatus().appliedPolicy[ policy ]

def deleteSaMapping( sa, profile ):
   applied = getOrCreateAppliedSa( sa )
   del applied.appliedSec[ profile.profile ]
   if not applied.appliedSec:
      del ikeStatus().appliedSa[sa]

def vxlanSecurityGuard( mode, token ):
   if gv.hwCapabilities.vxlanSecuritySupported:
      return None
   return CliParser.guardNotThisPlatform

#-------------------------------------------------------------------------------
# Auto-completers
#-------------------------------------------------------------------------------
ipsecProfileNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: ikeConfig().ipsecProfile,
      pattern='[a-zA-Z0-9_-]+',
      helpdesc='Name of the IPsec profile',
      helpname='WORD',
      priority=CliParser.PRIO_LOW )

ikePolicyNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: ikeConfig().ikePolicy,
      pattern='[a-zA-Z0-9_-]+',
      helpdesc='Name of the IKE policy',
      helpname='WORD',
      priority=CliParser.PRIO_LOW )

saPolicyNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: ikeConfig().securityAssoc,
      pattern='[a-zA-Z0-9_-]+',
      helpdesc='Name of the SA policy',
      helpname='WORD',
      priority=CliParser.PRIO_LOW )

ikeCryptoSuiteNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: ikeConfig().ikeCryptoSuiteConfig,
      pattern='[a-zA-Z0-9_-]+',
      helpdesc='Name of the IKE crypto suite',
      helpname='WORD',
      priority=CliParser.PRIO_LOW )

saCryptoSuiteNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: ikeConfig().saCryptoSuiteConfig,
      pattern='[a-zA-Z0-9_-]+',
      helpdesc='Name of the SA crypto suite',
      helpname='WORD',
      priority=CliParser.PRIO_LOW )

policyCryptoSuiteOrderMatcher = CliMatcher.EnumMatcher( {
      'after': 'Insert the crypto suite '
              'after the following crypto suite',
      'before': 'Insert the crypto suite '
               'before the following crypto suite',
      } )

#-------------------------------------------------------------------------------
# Aliases for some useful token rules.
#-------------------------------------------------------------------------------
class IpsecIkeConfigMode( IkePolicyMode, BasicCli.ConfigModeBase ):
   name = 'ISAKMP configuration'

   def __init__( self, parent, session, name ):
      self.ikeName = name
      self.ike = self.getOrCreateIkeConfig( ikeConfig() )
      self.ikeParams = Tac.nonConst( self.ike.ikeParams )
      if toggleMultiCryptoEnabled():
         # IKE crypto suite
         self.suiteType = 'ike'
      IkePolicyMode.__init__( self, name )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def getOrCreateIkeConfig( self, config ):
      ikePolicyConfig = config.ikePolicy.get( self.ikeName )
      if ikePolicyConfig is None:
         ikePolicyConfig = config.ikePolicy.newMember( self.ikeName )
      return ikePolicyConfig

   def onExit( self ):
      if self.ike.ikeParams != self.ikeParams:
         self.ike.ikeParams = self.ikeParams

class IpsecVxlanSecurityAssocConfigMode( SecurityAssociationVxlanMode,
                                         BasicCli.ConfigModeBase ):
   name = 'IPsec Vxlan SA configuration'

   def __init__( self, parent, session, name ):
      self.saName = name
      self.secAssoc = self.getOrCreateSAConfig( ikeConfig() )
      self.saParams = Tac.nonConst( self.secAssoc.saParams )
      if self.saParams.saType != IpsecSaTypeEnum.IpsecVxlanSa:
         self.saParams.saType = IpsecSaTypeEnum.IpsecVxlanSa
      # Only 256 bit hash integrity currently supported
      if self.saParams.espSha != IpsecHmacAlgorithmEnum.sha256:
         self.saParams.espSha = IpsecHmacAlgorithmEnum.sha256
      # Only aes256-gcm encryption currently supported
      if self.saParams.espAes != IpsecEspAlgorithmEnum.aes256gcm128:
         self.saParams.espAes = IpsecEspAlgorithmEnum.aes256gcm128
      # No pfs is configured
      if self.saParams.pfsGroup != IpsecDhGroupEnum.IpsecDhGroupNone:
         self.saParams.pfsGroup = IpsecDhGroupEnum.IpsecDhGroupNone
      SecurityAssociationVxlanMode.__init__( self, self.saName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def getOrCreateSAConfig( self, config ):
      secAssocConfig = config.securityAssoc.get( self.saName )
      if secAssocConfig is None:
         secAssocConfig = config.securityAssoc.newMember( self.saName )
      return secAssocConfig

   def onExit( self ):
      if self.secAssoc.saParams != self.saParams:
         self.secAssoc.saParams = self.saParams

class IpsecSecurityAssocConfigMode( SecurityAssociationMode,
                                    BasicCli.ConfigModeBase ):
   name = 'IPsec SA configuration'

   def __init__( self, parent, session, name ):
      self.saName = name
      self.secAssoc = self.getOrCreateSAConfig( ikeConfig() )
      self.saParams = Tac.nonConst( self.secAssoc.saParams )
      if self.saParams.saType != IpsecSaTypeEnum.IpsecKeyMgmtSa:
         self.saParams.saType = IpsecSaTypeEnum.IpsecKeyMgmtSa
      if toggleMultiCryptoEnabled():
         # SA crypto suite
         self.suiteType = 'sa'
      SecurityAssociationMode.__init__( self, self.saName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def getOrCreateSAConfig( self, config ):
      secAssocConfig = config.securityAssoc.get( self.saName )
      if secAssocConfig is None:
         secAssocConfig = config.securityAssoc.newMember( self.saName )
      return secAssocConfig

   def onExit( self ):
      if self.secAssoc.saParams != self.saParams:
         self.secAssoc.saParams = self.saParams

# Helper to set encapSrcPortRange attr
def setEncapSrcPortRange( profileParams, encapSrcPortRange ):
   fp = Tac.nonConst( profileParams.flowParallelization )
   fp.encapSrcPortRange = encapSrcPortRange
   profileParams.flowParallelization = fp

class IpsecProfileConfigMode( IpsecProfileMode, BasicCli.ConfigModeBase ):
   name = 'IPsec Profile configuration'

   def __init__( self, parent, session, name ):
      self.profileName = name
      self.profile = self.getOrCreateProfileConfig( ikeConfig() )
      self.profileParams = Tac.nonConst( self.profile.profileParams )
      IpsecProfileMode.__init__( self, self.profileName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def getOrCreateProfileConfig( self, config ):
      ipsecProfileConfig = config.ipsecProfile.get( self.profileName )
      if ipsecProfileConfig is None:
         ipsecProfileConfig = config.ipsecProfile.newMember( self.profileName )
         # Set default flow entropy (use dynamic port range)
         profileParams = Tac.nonConst( ipsecProfileConfig.profileParams )
         setEncapSrcPortRange(
               profileParams, IpsecFlowParallelization().dynamicPortRange )
         ipsecProfileConfig.profileParams = profileParams
      return ipsecProfileConfig

   def onExit( self ):
      if self.profile.profileParams != self.profileParams:
         self.profile.profileParams = self.profileParams

class IpsecEntropyUdpConfigMode( IpsecEntropyMode, BasicCli.ConfigModeBase ):
   name = 'IPsec UDP flow entropy configuration'

   def __init__( self, parent, session, name ):
      self.parent = parent
      self.protoName = name
      # Copy all configurable UDP entropy params (there's currently only one)
      self.encapSrcPortRange = (
            parent.profileParams.flowParallelization.encapSrcPortRange )
      IpsecEntropyMode.__init__( self, ( parent.profileName, self.protoName ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      if ( self.parent.profileParams.flowParallelization.encapSrcPortRange !=
            self.encapSrcPortRange ):
         setEncapSrcPortRange( self.parent.profileParams, self.encapSrcPortRange )

class IpsecConfigMode( IpSecurityMode, BasicCli.ConfigModeBase ):
   name = 'IP Security'

   def __init__( self, parent, session ):
      IpSecurityMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def _ipsecModeExitSessionCommitHandler( self ):
      ipsecConfigMode().exit = ipsecConfigMode().exit + 1

   def onExit( self ):
      if self.session.inConfigSession():
         CliSession.registerSessionOnCommitHandler(
               self.session_.entityManager, "IpsecModeExit",
               lambda m, onSessionCommit: self._ipsecModeExitSessionCommitHandler() )
      else:
         self._ipsecModeExitSessionCommitHandler()

class KeyControllerConfigMode( KeyControllerMode, BasicCli.ConfigModeBase ):
   name = 'IPsec Key Controller Config Mode'

   def __init__( self, parent, session ):
      self.controllerCfg = Tac.nonConst( ikeConfig().keyController )
      KeyControllerMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      if ikeConfig().keyController != self.controllerCfg:
         ikeConfig().keyController = self.controllerCfg

# --------------------------------------------------------------------------------
# "[ no | default ] ip access-group <name> [ vrf VRF ]", in "ip security" mode
# --------------------------------------------------------------------------------
def setServiceAcl( mode, args ):
   aclName = args[ 'ACL' ]
   if toggleIpsecServiceAclVrfEnabled():
      vrfName = args.get( 'VRF' )
   else:
      vrfName = DEFAULT_VRF

   AclCliLib.setServiceAcl( mode, 'ipsec', IPPROTO_UDP, gv.aclConfig, gv.aclCpConfig,
                            aclName, 'ip', vrfName, port=[ 500, 4500 ] )
   AclCliLib.setServiceAcl( mode, 'esp', IPPROTO_ESP, gv.aclConfig, gv.aclCpConfig,
                            aclName, 'ip', vrfName )

def noServiceAcl( mode, args=None ):
   args = args or {}
   aclName = args.get( 'ACL' )
   vrfName = args.get( 'VRF' )
   if toggleIpsecServiceAclVrfEnabled():
      vrfName = vrfName or DEFAULT_VRF
   AclCliLib.noServiceAcl( mode, 'ipsec', gv.aclConfig, gv.aclCpConfig,
                           aclName, 'ip', vrfName )
   AclCliLib.noServiceAcl( mode, 'esp', gv.aclConfig, gv.aclCpConfig, aclName, 'ip',
                           vrfName )

vrfWithNameExprFactory = VrfExprFactory(
      helpdesc='Configure the VRF in which to apply the access control list',
      inclDefaultVrf=True )
class IpAccessGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'ip access-group ACL'
   noOrDefaultSyntax = 'ip access-group [ ACL ]'
   data = {
      'ip': AclCli.ipKwForServiceAclMatcher,
      'access-group': AclCli.accessGroupKwMatcher,
      'ACL': AclCli.ipAclNameMatcher,
      'VRF': vrfWithNameExprFactory,
   }
   if toggleIpsecServiceAclVrfEnabled():
      syntax += ' [ VRF ]'
      noOrDefaultSyntax += ' [ VRF ]'
   handler = setServiceAcl
   noOrDefaultHandler = noServiceAcl

if toggleIpsecServiceAclEnabled():
   IpsecConfigMode.addCommandClass( IpAccessGroupCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] fips restrictions", in "ip security" mode
#--------------------------------------------------------------------------------
def handleFipsRestrictions( mode, args ):
   ikeConfig().fipsRestrictions = True

def handleNoFipsRestrictions( mode, args=None ):
   ikeConfig().fipsRestrictions = ikeConfig().fipsRestrictionsDefault

class FipsRestrictionsCmd( CliCommand.CliCommandClass ):
   syntax = 'fips restrictions'
   noOrDefaultSyntax = syntax
   data = {
      'fips': 'FIPS',
      'restrictions': 'Use FIPS compliant IPsec protocol suite',
   }
   handler = handleFipsRestrictions
   noOrDefaultHandler = handleNoFipsRestrictions

IpsecConfigMode.addCommandClass( FipsRestrictionsCmd )

#----------------------------------------------------------------------------------
# "[ no | default ] ethernet untagged allowed", in "ip security" mode
#----------------------------------------------------------------------------------
def untaggedAllowedGuard( mode, token ):
   if gv.ipsecCapabilities.untaggedAllowed:
      return CliParser.guardNotThisPlatform
   return None

ethernet = CliCommand.guardedKeyword( 'ethernet', 'Ethernet header',
                                      untaggedAllowedGuard )

def handleUntaggedAllowed( mode, args ):
   ikeConfig().untaggedAllowed = True

def handleNoUntaggedAllowed( mode, args=None ):
   ikeConfig().untaggedAllowed = ikeConfig().untaggedAllowedDefault

class UntaggedAllowedCmd( CliCommand.CliCommandClass ):
   syntax = 'ethernet untagged allowed'
   noOrDefaultSyntax = syntax
   data = {
      'ethernet': ethernet,
      'untagged': 'Untagged packets',
      'allowed': 'Allowed'
   }
   handler = handleUntaggedAllowed
   noOrDefaultHandler = handleNoUntaggedAllowed

IpsecConfigMode.addCommandClass( UntaggedAllowedCmd )

# --------------------------------------------------------------------------------
# "[ no ] hardware encryption disabled", in "ip security" mode
# --------------------------------------------------------------------------------
def hwCryptoSuppportedGuard( mode, token ):
   if gv.ipsecCapabilities and not gv.ipsecCapabilities.hwCryptoSupported:
      return CliParser.guardNotThisPlatform
   return None

hwGuardedKw = CliCommand.guardedKeyword( 'hardware',
         helpdesc='Hardware support',
         guard=hwCryptoSuppportedGuard )

def handleHwEncryption( mode, args ):
   ikeConfig().hwCryptoEnabled = False
   mode.addWarning( "Hardware encryption disabled. "
                    "A SFE restart is needed for this change to take effect." )

def handleNoHwEncryption( mode, args=None ):
   ikeConfig().hwCryptoEnabled = capabilities().hwCryptoByDefault
   mode.addWarning( "Hardware encryption enabled. "
                    "A SFE restart is needed for this change to take effect." )

class HardwareEncryptionCmd( CliCommand.CliCommandClass ):
   syntax = 'hardware encryption disabled'
   noOrDefaultSyntax = syntax
   data = {
      'hardware': hwGuardedKw,
      'encryption': 'Encryption Support',
      'disabled': 'Disable hardware encryption'
   }
   handler = handleHwEncryption
   noOrDefaultHandler = handleNoHwEncryption

if Toggles.IpsecToggleLib.toggleHwEncryptionCommandEnabled():
   IpsecConfigMode.addCommandClass( HardwareEncryptionCmd )

#-------------------------------------------------------------------------------
# 'ip security'
#-------------------------------------------------------------------------------
class IpSecurity( CliCommand.CliCommandClass ):
   syntax = 'ip security'
   noOrDefaultSyntax = syntax
   data = { 'ip': CliToken.Ip.ipMatcherForConfig,
            'security': 'IP security' }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( IpsecConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = ikeConfig()
      config.ikePolicy.clear()
      config.saCryptoSuiteConfig.clear()
      config.ikeCryptoSuiteConfig.clear()
      config.securityAssoc.clear()
      if toggleIpsecServiceAclEnabled():
         noServiceAcl( mode, args )
      config.fipsRestrictions = config.fipsRestrictionsDefault
      config.untaggedAllowed = config.untaggedAllowedDefault
      if Toggles.IpsecToggleLib.toggleHwEncryptionCommandEnabled():
         config.hwCryptoEnabled = capabilities().hwCryptoByDefault
      noKeyController( mode, args )
      config.ipsecProfile.clear()

BasicCli.GlobalConfigMode.addCommandClass( IpSecurity )

#-------------------------------------------------------------------------------
# '[no] profile <name>'
#-------------------------------------------------------------------------------
class IpsecProfile( CliCommand.CliCommandClass ):
   syntax = 'profile <profileName>'
   noOrDefaultSyntax = 'profile <profileName>'
   data = { 'profile': 'Profile name',
            '<profileName>': ipsecProfileNameMatcher
          }

   @staticmethod
   def handler(  mode, args ):
      name = args.get( '<profileName>' )
      childMode = mode.childMode( IpsecProfileConfigMode, name=name )
      mode.session_.gotoChildMode( childMode )
      childMode.getOrCreateProfileConfig( ikeConfig() )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      name = args[ '<profileName>' ]
      config = ikeConfig()
      if name in config.ipsecProfile:
         keyControllerCfg = config.keyController
         if keyControllerCfg.configured and keyControllerCfg.profileName == name:
            mode.addError( f'Cannot remove IPsec profile {name}. '
                           'It is applied to the key controller' )
            return
         # Check to see if this profile has been applied to a Tunnel/Path interface
         appliedProfile = ikeStatus().appliedProfiles.get( name )
         if appliedProfile:
            # It is possible that the profile has not been removed from the
            # applied profiles when we have removed it from the key controller
            # but has not exited from the ipsec mode. We want to allow
            # deleting the profile in this case
            if appliedProfile.appliedIntfId or appliedProfile.appliedPath:
               mode.addError( f'Cannot remove IPsec profile {name}. '
                              'It is applied to an interface' )
               return

         del config.ipsecProfile[ name ]

IpsecConfigMode.addCommandClass( IpsecProfile )

#------------------------------------------------------------------------------------
#sa policy <name>
#------------------------------------------------------------------------------------
class IpsecSecurityAssociation( CliCommand.CliCommandClass ):
   if Toggles.IpsecToggleLib.toggleTunnelSecAssocEnabled():
      syntax = 'sa policy [ vxlan security ] <saName>'
   else:
      syntax = 'sa policy <saName>'
   noOrDefaultSyntax = syntax
   data = { 'sa': 'Security Association',
            'policy': 'Policy to be applied on the data path',
            'vxlan': CliCommand.Node(
               matcher=CliMatcher.KeywordMatcher( 'vxlan',
                  helpdesc='VXLANSec Security Association' ),
               guard=vxlanSecurityGuard ),
            'security': 'Security policy',
            '<saName>': saPolicyNameMatcher
          }

   @staticmethod
   def handler( mode, args ):
      name = args.get( '<saName>' )
      secAssocConfig = ikeConfig().securityAssoc.get( name )
      # Verify the SA name configured doesn't conflict with another SA type
      if secAssocConfig is not None:
         saType = secAssocConfig.saParams.saType
         if ( 'vxlan' in args and saType != IpsecSaTypeEnum.IpsecVxlanSa ) or \
            ( 'vxlan' not in args and saType != IpsecSaTypeEnum.IpsecKeyMgmtSa ):
            mode.addError( "Conflicting SA name" )
            return
      if 'vxlan' in args:
         childMode = mode.childMode( IpsecVxlanSecurityAssocConfigMode, name=name )
      else:
         childMode = mode.childMode( IpsecSecurityAssocConfigMode, name=name )
      mode.session_.gotoChildMode( childMode )
      childMode.getOrCreateSAConfig( ikeConfig() )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      sa = args.get( '<saName>' )
      config = ikeConfig()
      if sa in config.securityAssoc:
         if sa in ikeStatus().appliedSa:
            mode.addError( f'Cannot remove Security associsation {sa}. '
                           'It is applied to a profile' )
            return
         else:
            del config.securityAssoc[ sa ]

IpsecConfigMode.addCommandClass( IpsecSecurityAssociation )

#--------------------------------------------------------------------------
# esp encryption [ 3des | aes128 | aes256 | aes128gcm64 | aes128gcm128 | null ]
#---------------------------------------------------------------------------
class IpsecSaEncryption( CliCommand.CliCommandClass ):
   syntax = 'esp encryption [ ESP_CIPHER ]'
   noOrDefaultSyntax = 'esp encryption ...'
   data = { 'esp': 'Encapsulation Security Payload',
            'encryption': 'Encryption type',
            'ESP_CIPHER': CliMatcher.EnumMatcher( {
               '3des': 'Three key triple DES',
               'aes128': 'AES - 128-bit Advanced Encryption Standard',
               'aes256': 'AES - 256-bit Advanced Encryption Standard',
               'aes128gcm64': '128 bit AES-GCM with 64 bit ICV',
               'aes128gcm128': '128 bit AES-GCM with 128 bit ICV',
               'aes256gcm128': '256 bit AES-GCM with 128 bit ICV',
               'null': 'Null Encryption',
            } )
          }

   @staticmethod
   def handler( mode, args ):
      #aes128 is the default encryption algo
      encr = args.get( 'ESP_CIPHER', 'aes128' )
      mode.saParams.espAes = encryptionMap[ encr ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.saParams.espAes = mode.saParams.espAesDefault

IpsecSecurityAssocConfigMode.addCommandClass( IpsecSaEncryption )

#--------------------------------------------------------------------------------
# esp integrity [ md5 | sha1 | sha256 | sha384 | sha512 | null ]
#---------------------------------------------------------------------------
class IpsecSaIntegrity( CliCommand.CliCommandClass ):
   syntax = 'esp integrity [ HASHALGO ]'
   noOrDefaultSyntax = 'esp integrity ...'
   data = { 'esp': 'Encapsulation Security Payload',
            'integrity': 'Integrity type',
            'HASHALGO': CliMatcher.EnumMatcher( {
               'md5': 'md5 security profile',
               'sha1': 'sha1 security profile',
               'sha256': 'sha256 security profile',
               'sha384': 'sha384 security profile',
               'sha512': 'sha512 security profile',
               'null': 'null security profile',
            } )
          }

   @staticmethod
   def handler( mode, args ):
      espHashAlgo = args.get( 'HASHALGO', 'sha1' )
      mode.saParams.espSha = integrityMap[ espHashAlgo ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.saParams.espSha = mode.saParams.espShaDefault

IpsecSecurityAssocConfigMode.addCommandClass( IpsecSaIntegrity )

#-------------------------------------------------------------------------------
# ' ike policy <name>'
#-------------------------------------------------------------------------------
class IpsecIkePolicy( CliCommand.CliCommandClass ):
   '''Enter the Ike mode to enter the IKE profile, encryption method
      integrity, lifetime and Diffie-Hellman group number'''
   syntax = 'ike policy <policy>'
   noOrDefaultSyntax = 'ike policy <policy>'
   data = { 'ike': 'Internet Security Association and Key Mgmt Protocol',
            'policy': 'Assign a name',
            '<policy>': ikePolicyNameMatcher
          }

   @staticmethod
   def handler( mode, args ):
      ikeName = args.get( '<policy>' )
      childMode = mode.childMode( IpsecIkeConfigMode, name=ikeName )
      mode.session_.gotoChildMode( childMode )
      childMode.getOrCreateIkeConfig( ikeConfig() )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      policy = args.get( '<policy>' )
      config = ikeConfig()
      if policy in config.ikePolicy:
         if policy in ikeStatus().appliedPolicy:
            mode.addError( f'Cannot remove Policy {policy}. It is currently applied '
                           'to a profile.' )
            return
         else:
            del config.ikePolicy[ policy ]

IpsecConfigMode.addCommandClass( IpsecIkePolicy )

#-------------------------------------------------------------------------------
# 'version ( 1 | 2 )
#-------------------------------------------------------------------------------
class IpsecIkeVersion( CliCommand.CliCommandClass ):
   syntax = 'version ( 1 | 2 )'
   noOrDefaultSyntax = 'version ...'

   data = { 'version': 'IKE Version',
            '1': 'IKEv1',
            '2': 'IKEv2'
          }

   @staticmethod
   def handler ( mode, args ):
      if args.get( '1' ):
         version = '1'
         versionE = 'ikeVersion1'
      elif args.get( '2' ):
         version = '2'
         versionE = 'ikeVersion2'
      mode.ikeParams.version = version
      # incrementally get rid of the string-based version and replace
      # it with the enum
      mode.ikeParams.versionEnum = versionE

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.ikeParams.version = mode.ikeParams.versionDefault

IpsecIkeConfigMode.addCommandClass( IpsecIkeVersion )

#-------------------------------------------------------------------------------
# 'authentication ( pre-share | rsa-sig | pki )
#-------------------------------------------------------------------------------
class IpsecIkeAuthentication( CliCommand.CliCommandClass ):
   syntax = 'authentication ( pre-share | rsa-sig | pki )'

   noOrDefaultSyntax = 'authentication ...'

   data = { 'authentication': 'Authentication type',
            'pre-share': 'Pre-Shared Key',
            'rsa-sig': 'Rivest-Shamir-Adleman Signature',
            'pki': 'Public Key Infrastructure',
          }

   @staticmethod
   def handler( mode, args ):
      if args.get( 'pre-share' ):
         auth = IpsecAuthTypeEnum.pre_share
      elif args.get( 'rsa-sig' ):
         auth = IpsecAuthTypeEnum.rsa_sig
      elif args.get( 'pki' ):
         auth = IpsecAuthTypeEnum.pki
      mode.ikeParams.auth = auth

   @staticmethod
   def noOrDefaultHandler( mode, args):
      mode.ikeParams.auth = mode.ikeParams.authDefault

IpsecIkeConfigMode.addCommandClass( IpsecIkeAuthentication )

#-------------------------------------------------------------------------------
# 'encryption [ aes128 | aes192 | aes256 | 3des ]
#-------------------------------------------------------------------------------
class IpsecIkeEncryption( CliCommand.CliCommandClass ):
   syntax = 'encryption [ CIPHER ]'
   noOrDefaultSyntax = 'encryption ...'
   data = { 'encryption': 'Encryption type',
            'CIPHER': CliMatcher.EnumMatcher( {
               'aes128': 'AES - 128-bit Advanced Encryption Standard',
               'aes192': 'AES - 192-bit Advanced Encryption Standard',
               'aes256': 'AES - 256-bit Advanced Encryption Standard',
               '3des': 'Three key triple DES',
            } )
          }

   @staticmethod
   def handler( mode, args ):
      encr = args.get( 'CIPHER' )
      mode.ikeParams.encryption = ( encryptionMap[ encr ] if encr
            else mode.ikeParams.encryptionDefault )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.ikeParams.encryption = mode.ikeParams.encryptionDefault

IpsecIkeConfigMode.addCommandClass( IpsecIkeEncryption )

#-------------------------------------------------------------------------------
# integrity ( md5 | sha1 | sha256 | sha384 | sha512 )
#-------------------------------------------------------------------------------
class IpsecIkeHash( CliCommand.CliCommandClass ):
   syntax = 'integrity ( HASHALGO )'
   noOrDefaultSyntax = 'integrity ...'
   data = { 'integrity': 'Integrity algorithm',
            'HASHALGO': CliMatcher.EnumMatcher( {
               'md5': 'Message Digest Algorithm 5',
               'sha1': 'Secure Hash Standard 1 (160 bit)',
               'sha256': 'Secure Hash Standard 2 (256 bit)',
               'sha384': 'Secure Hash Standard 2 (384 bit)',
               'sha512': 'Secure Hash Standard 2 (512 bit)',
            } )
          }

   @staticmethod
   def handler( mode, args ):
      hashAlgo = args.get( 'HASHALGO' )
      mode.ikeParams.integrity = integrityMap[ hashAlgo ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.ikeParams.integrity = mode.ikeParams.integrityDefault

IpsecIkeConfigMode.addCommandClass( IpsecIkeHash )

matcherDHGroupNum = CliMatcher.EnumMatcher( {
                    '1': 'Diffie-Hellman group 1 (768 bit)',
                    '2': 'Diffie-Hellman group 2 (1024 bit)',
                    '5': 'Diffie-Hellman group 5 (1536 bit)',
                    '14': 'Diffie-Hellman group 14 (2048 bit)',
                    '15': 'Diffie-Hellman group 15 (3072 bit)',
                    '16': 'Diffie-Hellman group 16 (4096 bit)',
                    '17': 'Diffie-Hellman group 17 (6144 bit)',
                    '19': 'Diffie-Hellman group 19 (256 bit ecp)',
                    '20': 'Diffie-Hellman group 20 (384 bit ecp)',
                    '21': 'Diffie-Hellman group 21 (521 bit ecp)',
                    '24': 'Diffie-Hellman group 24 (2048 bit, 256 bit subgroup)' } )

groupNumToIpsecDhGroupMap = {
      '1': IpsecDhGroupEnum.modp768,
      '2': IpsecDhGroupEnum.modp1024, 
      '5': IpsecDhGroupEnum.modp1536, 
      '14': IpsecDhGroupEnum.modp2048,
      '15': IpsecDhGroupEnum.modp3072,
      '16': IpsecDhGroupEnum.modp4096,
      '17': IpsecDhGroupEnum.modp6144,
      '19': IpsecDhGroupEnum.ecp256,
      '20': IpsecDhGroupEnum.ecp384,
      '21': IpsecDhGroupEnum.ecp521,
      '24': IpsecDhGroupEnum.modp2048s256,
      }

def getDHGroupFromNum( dhGroupNum ):
   dhGroup = groupNumToIpsecDhGroupMap.get( dhGroupNum )
   return dhGroup

#-----------------------------------------------------------------------------------
# dh-group
#-----------------------------------------------------------------------------------
class IpsecIkeDHGroup( CliCommand.CliCommandClass ):
   syntax = 'dh-group GROUP_NUM'
   noOrDefaultSyntax = 'dh-group ...'
   data = { 'dh-group': 'Diffie-Hellman Group',
            'GROUP_NUM': matcherDHGroupNum 
          }

   @staticmethod
   def handler( mode, args ):
      dhGroupNum = args[ 'GROUP_NUM' ]
      mode.ikeParams.dhGroup = getDHGroupFromNum( dhGroupNum )

   @staticmethod
   def noOrDefaultHandler( mode, args):
      mode.ikeParams.dhGroup = mode.ikeParams.dhGroupDefault

IpsecIkeConfigMode.addCommandClass( IpsecIkeDHGroup )

#-----------------------------------------------------------------------------------
# ike-lifetime
#-----------------------------------------------------------------------------------
LifetimeUnit = TacLazyType( 'Ipsec::LifetimeUnit' )
hoursMatcher = CliMatcher.IntegerMatcher( 1, 24, helpdesc='Lifetime (hours)' )
minutesMatcher = CliMatcher.IntegerMatcher( 10, 1440, helpdesc='Lifetime (minutes)' )

class IpsecIkeLifetime( CliCommand.CliCommandClass ):
   syntax = 'ike-lifetime ( MINUTES minutes ) | ( HOURS [ hours ] )'
   noOrDefaultSyntax = 'ike-lifetime [ ( MINUTES minutes ) | ( HOURS [ hours ] ) ]'
   data = { 'ike-lifetime': 'Set ikeLifetime for ISAKMP security association',
            'hours': "Set lifetime in hours",
            'HOURS': hoursMatcher,
            'minutes': "Set lifetime in minutes",
            'MINUTES': minutesMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      lifeTime = args.get( 'MINUTES' )
      if lifeTime:
         mode.ikeParams.ikeLifetimeUnit = LifetimeUnit.minutes
         lifeTime = lifeTime * 60
      else:
         mode.ikeParams.ikeLifetimeUnit = LifetimeUnit.hours
         lifeTime = args.get( 'HOURS' ) * 3600
      mode.ikeParams.ikeLifetime = Tac.newInstance( 'Ipsec::Ike::IpsecLifetime',
                                                    lifeTime )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.ikeParams.ikeLifetime = mode.ikeParams.ikeLifetimeDefault

IpsecIkeConfigMode.addCommandClass( IpsecIkeLifetime )

#-----------------------------------------------------------------------------------
# local-id <ip>
#-----------------------------------------------------------------------------------
genIpAddrMatcher = IpGenAddrMatcher.ipGenAddrMatcher

class IpsecIkeLocalIdentity( CliCommand.CliCommandClass ):
   syntax = 'local-id ( <ip> | ( fqdn <fqdn> ) )'
   noOrDefaultSyntax = 'local-id [ <ip>  | ( fqdn [ <fqdn> ] ) ]'

   data = { 'local-id': 'Local IKE Identification',
            '<ip>': genIpAddrMatcher,
            'fqdn': 'Fully Qualified Domain Name',
            '<fqdn>': CliMatcher.StringMatcher( helpdesc="FQDN/UFQDN as local ID" ),
          }

   @staticmethod
   def handler( mode, args ):
      ip = args.get( "<ip>" )
      fqdn = args.get( "<fqdn>" )
      if ip:
         mode.ikeParams.localId = Tac.Value( "Ipsec::HostIdentity",
                                             ip=ip )
      elif fqdn:
         mode.ikeParams.localId = Tac.Value( "Ipsec::HostIdentity",
                                             fqdn=fqdn )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.ikeParams.localId = Tac.Value( "Ipsec::HostIdentity" )

IpsecIkeConfigMode.addCommandClass( IpsecIkeLocalIdentity )

#-----------------------------------------------------------------------------------
# remote-id <ip> | fqdn <fqdn>
#-----------------------------------------------------------------------------------
class IpsecIkeRemoteIdentity( CliCommand.CliCommandClass ):
   syntax = 'remote-id ( <ip> | ( fqdn <fqdn> ) )'
   noOrDefaultSyntax = 'remote-id [ <ip> | ( fqdn [ <fqdn> ] ) ]'

   data = { 'remote-id': 'Remote peer IKE Identification',
            '<ip>': genIpAddrMatcher,
            'fqdn': "Fully Qualified Domain Name",
            '<fqdn>': CliMatcher.StringMatcher( helpdesc="FQDN/UFQDN as remote ID" ),
          }

   @staticmethod
   def handler( mode, args ):
      ip = args.get( '<ip>' )
      fqdn = args.get( '<fqdn>' )
      if ip:
         mode.ikeParams.remoteId = Tac.Value( 'Ipsec::HostIdentity',
                                              ip=ip )
      elif fqdn:
         mode.ikeParams.remoteId = Tac.Value( 'Ipsec::HostIdentity',
                                              fqdn=fqdn )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.ikeParams.remoteId = Tac.Value( "Ipsec::HostIdentity" )

IpsecIkeConfigMode.addCommandClass( IpsecIkeRemoteIdentity )

#-----------------------------------------------------------------------------------
# remote-ip any
#-----------------------------------------------------------------------------------
class IpsecIkeRemoteIp( CliCommand.CliCommandClass ):
   syntax = 'remote-ip any'
   noOrDefaultSyntax = syntax

   data = { 'remote-ip': 'Remote peer IP address',
            'any': 'Allow session from any IP address',
          }

   @staticmethod
   def handler( mode, args ):
      mode.ikeParams.remoteIpAny = True

   @staticmethod
   def noOrDefaultHandler( mode, args=None ):
      mode.ikeParams.remoteIpAny = False

if toggleIpsecRemoteIpAnyEnabled():
   IpsecIkeConfigMode.addCommandClass( IpsecIkeRemoteIp )

#-----------------------------------------------------------------------------------
# pfs group
#-----------------------------------------------------------------------------------
class IpsecPfsgroup( CliCommand.CliCommandClass ):
   syntax = 'pfs dh-group GROUP_NUM'
   noOrDefaultSyntax = 'pfs dh-group ...'
   data = { 'pfs': 'Perfect Forward Secrecy',
            'dh-group': 'Diffie-Hellman Group',
            'GROUP_NUM': matcherDHGroupNum 
          }

   @staticmethod
   def handler ( mode, args ):
      dhGroupNum = args[ 'GROUP_NUM' ]
      mode.saParams.pfsGroup = getDHGroupFromNum( dhGroupNum )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.saParams.pfsGroup = IpsecDhGroupEnum.IpsecDhGroupNone

IpsecSecurityAssocConfigMode.addCommandClass( IpsecPfsgroup )

#-----------------------------------------------------------------------------------
# sa lifetime ( MINUTES minutes ) | ( HOURS [ hours ] )
# no sa lifetime [ ( MINUTES minutes ) | ( HOURS [ hours ] ) ]
#-----------------------------------------------------------------------------------
class IpsecSALifetime( CliCommand.CliCommandClass ):
   syntax = 'sa lifetime ( MINUTES minutes ) | ( HOURS [ hours ] )'
   noOrDefaultSyntax = 'sa lifetime [ ( MINUTES minutes ) | ( HOURS [ hours ] ) ]'
   data = { 'sa' : 'Security Association',
            'lifetime' : 'Lifetime for IPsec Security Association '\
                         '( default unit is hours )',
            'hours' : 'Lifetime in hours ( default )',
            'HOURS': hoursMatcher,
            'minutes': 'Lifetime in minutes',
            'MINUTES': minutesMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      lifeTime = args.get( 'MINUTES' )
      if lifeTime:
         mode.saParams.saLifetimeUnit = LifetimeUnit.minutes
         lifeTime = lifeTime * 60
      else:
         mode.saParams.saLifetimeUnit = LifetimeUnit.hours
         lifeTime = args.get( 'HOURS' ) * 3600
      mode.saParams.saLifetime = Tac.newInstance( 'Ipsec::Ike::IpsecLifetime' ,
                                                  lifeTime )
   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.saParams.saLifetime = mode.saParams.saLifetimeDefault

IpsecSecurityAssocConfigMode.addCommandClass( IpsecSALifetime )

#-----------------------------------------------------------------------------------
# [no | default] anti-replay window <size>
#-----------------------------------------------------------------------------------
class IpsecAntiReplayWindow( CliCommand.CliCommandClass ):
   syntax = 'anti-replay window <size>'
   noOrDefaultSyntax = 'anti-replay window [ <size> ]'
   data = { 'anti-replay': 'IPsec duplicate IP datagram detection',
            'window': 'Anti-Replay Window',
            '<size>': CliMatcher.IntegerMatcher( 1024, 65535,
                                                 helpdesc='Set Window size' )
         }

   @staticmethod
   def handler( mode, args ):
      replayWindowSize = args.get( '<size>' )
      mode.saParams.replayWindowSize = Tac.newInstance(
            'Ipsec::Ike::ReplayWindowSize', replayWindowSize )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.saParams.replayWindowSize = mode.saParams.replayWindowSizeDefault

IpsecSecurityAssocConfigMode.addCommandClass( IpsecAntiReplayWindow )

#-----------------------------------------------------------------------------------
# [no | default] anti-replay detection
#
# Below command will set replay window to 0
# per cli-review feedback.
# no anti-replay
#
# Below commands will reset the replay window to default
# anti-replay detection
# default anti-replay [ detection ]
#-----------------------------------------------------------------------------------
class IpsecAntiReplayDetection( CliCommand.CliCommandClass ):
   syntax = 'anti-replay detection'
   noOrDefaultSyntax = 'anti-replay [ detection ]'
   data = { 'anti-replay': 'IPsec duplicate IP datagram detection',
            'detection': 'Disable Anti-Replay Window detection',
          }

   @staticmethod
   def handler( mode, args ):
      mode.saParams.replayWindowSize = Tac.newInstance(
            'Ipsec::Ike::ReplayWindowSize',
            mode.saParams.replayWindowSizeDefault )

   @staticmethod
   def noHandler( mode, args ):
      mode.saParams.replayWindowSize = Tac.newInstance(
            'Ipsec::Ike::ReplayWindowSize', 0 )

   @staticmethod
   def defaultHandler( mode, args ):
      mode.saParams.replayWindowSize = Tac.newInstance(
            'Ipsec::Ike::ReplayWindowSize', mode.saParams.replayWindowSizeDefault )

IpsecSecurityAssocConfigMode.addCommandClass( IpsecAntiReplayDetection )

#-----------------------------------------------------------------------------------
# sa lifetime <packetLimit> thousand-packets
#-----------------------------------------------------------------------------------
class IpsecPacketLimit( CliCommand.CliCommandClass ):
   syntax = 'sa lifetime <packetLimit> thousand-packets'
   noOrDefaultSyntax = 'sa lifetime <packetLimit> thousand-packets'

   data = {
         'sa': 'Security Association',
         'lifetime' : 'Lifetime for IPsec Security Association '\
                      '( default unit is hours )',
         '<packetLimit>' : CliMatcher.IntegerMatcher( 1, 4000000,
                           helpdesc="Packet limit in thousands" ),
         'thousand-packets' : 'Packet limit in thousands',
   }

   @staticmethod
   def handler( mode, args ):
      mode.saParams.packetLimit = args.get( '<packetLimit>' ) * 1000

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.saParams.packetLimit = mode.saParams.packetLimitDefault

IpsecSecurityAssocConfigMode.addCommandClass( IpsecPacketLimit )

#-----------------------------------------------------------------------------------
# sa lifetime <byteLimit> megabytes
#-----------------------------------------------------------------------------------
class IpsecMegaByteLimit( CliCommand.CliCommandClass ):
   syntax = 'sa lifetime <byteLimit> megabytes'
   noOrDefaultSyntax = 'sa lifetime <byteLimit> megabytes'

   data = {
         'sa' : 'Security Association',
         'lifetime' : 'Lifetime for IPsec Security Association '\
                      '( default unit is hours )',
         '<byteLimit>' : CliMatcher.IntegerMatcher( 1, 6144000,
                         helpdesc="Byte limit in MB ( 1024 KB )" ),
         'megabytes' : 'Byte limit in MB ( 1024 KB )',
   }

   @staticmethod
   def handler( mode, args ):
      mode.saParams.byteLimit = args.get( '<byteLimit>' ) * ( 1024 ** 2 )
      mode.saParams.byteLimitUnit = 'MegaBytes'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.saParams.byteLimit = mode.saParams.byteLimitDefault
      mode.saParams.byteLimitUnit = 'GigaBytes'

IpsecSecurityAssocConfigMode.addCommandClass( IpsecMegaByteLimit )

#-----------------------------------------------------------------------------------
# sa lifetime <byteLimit> gigabytes
#-----------------------------------------------------------------------------------
class IpsecGigaByteLimit( CliCommand.CliCommandClass ):
   syntax = 'sa lifetime <byteLimit> gigabytes'
   noOrDefaultSyntax = 'sa lifetime <byteLimit> gigabytes'

   data = {
         'sa' : 'Security Association',
         'lifetime' : 'Lifetime for IPsec Security Association '\
                      '( default unit is hours )',
         '<byteLimit>' : CliMatcher.IntegerMatcher( 1, 6000,
                         helpdesc="Byte limit in GB ( 1024 MB )" ),
         'gigabytes' : 'Byte limit in GB ( 1024 MB )',
   }

   @staticmethod
   def handler( mode, args ):
      mode.saParams.byteLimit = args.get( '<byteLimit>' ) * ( 1024 ** 3 )
      mode.saParams.byteLimitUnit = 'GigaBytes'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.saParams.byteLimit = mode.saParams.byteLimitDefault
      mode.saParams.byteLimitUnit = 'GigaBytes'

IpsecSecurityAssocConfigMode.addCommandClass( IpsecGigaByteLimit )

# -----------------------------------------------------------------------------------
# sa idle-timeout <time> [ seconds ]
# no sa idle-timeout
# -----------------------------------------------------------------------------------
class IpsecSAIdleTimeout( CliCommand.CliCommandClass ):
   syntax = 'sa idle-timeout <time> seconds'
   noOrDefaultSyntax = 'sa idle-timeout ...'
   data = { 'sa': 'Security Association',
            'idle-timeout': 'Set idle timeout for Securiry Association',
            '<time>': CliMatcher.IntegerMatcher( 60, 2 ** 32 - 1,
                      helpdesc='Idle timeout in seconds' ),
            'seconds': 'Idle timeout in seconds',
          }

   @staticmethod
   def handler( mode, args ):
      mode.saParams.saIdleTimeout = Tac.newInstance( 'Ipsec::Ike::IpsecIdleTimeout',
                                                      args[ '<time>' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.saParams.saIdleTimeout = mode.saParams.saIdleTimeoutDefault

if togglePolicyBasedVpnEnabled():
   IpsecSecurityAssocConfigMode.addCommandClass( IpsecSAIdleTimeout )

#-----------------------------------------------------------------------------------
# encapsulation vxlan security
#-----------------------------------------------------------------------------------
class IpsecEncapsulationVxlan( CliCommand.CliCommandClass ):
   syntax = 'encapsulation vxlan security [ tag esp ]'
   noOrDefaultSyntax = syntax

   data = {
         'encapsulation': 'Encapsulation format',
         'vxlan' : 'VXLAN',
         'security' : 'VXLAN payload encryption',
         'tag' : 'Tag format',
         'esp' : 'Format UDP-ESP',
   }

   @staticmethod
   def handler( mode, args ):
      tagEsp = 'tag' in args
      mode.saParams.vxlanEncapTagEsp = tagEsp
      mode.saParams.vxlanEncap = not tagEsp

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.saParams.vxlanEncap = False
      mode.saParams.vxlanEncapTagEsp = False

if Toggles.IpsecToggleLib.toggleTunnelSecAssocEnabled():
   IpsecVxlanSecurityAssocConfigMode.addCommandClass( IpsecEncapsulationVxlan )

#-----------------------------------------------------------------------------------
# traffic unprotected allow
#-----------------------------------------------------------------------------------
class IpsecVxlanUnprotectedTraffic( CliCommand.CliCommandClass ):
   syntax = 'traffic unprotected allow'
   noOrDefaultSyntax = syntax

   data = {
         'traffic': 'Traffic security policy',
         'unprotected' : 'Unprotected fallback mode without encryption',
         'allow' : 'Allow policy option',
   }

   @staticmethod
   def handler( mode, args ):
      mode.saParams.unprotectedTraffic = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.saParams.unprotectedTraffic = False

if Toggles.IpsecToggleLib.toggleTunnelSecAssocEnabled():
   IpsecVxlanSecurityAssocConfigMode.addCommandClass( IpsecVxlanUnprotectedTraffic )
   IpsecVxlanSecurityAssocConfigMode.addCommandClass( IpsecSALifetime )
   IpsecVxlanSecurityAssocConfigMode.addCommandClass( IpsecAntiReplayDetection )

#-----------------------------------------------------------------------------------
# connection [add | start | route ]
#-----------------------------------------------------------------------------------
class IpsecConnectionType( CliCommand.CliCommandClass ):
   syntax = 'connection [ add | start | route ]'
   noOrDefaultSyntax = 'connection ...'
   data = { 'connection': 'IPsec Connection (Initiator/Responder/Dynamic)',
            'add': 'Responder',
            'start': 'Initiator',
            'route': 'Dynamic'
          }

   @staticmethod
   def handler( mode, args ):
      if args.get( 'add' ):
         mode.profileParams.connectionType = 'add'
      if args.get('start'):
         mode.profileParams.connectionType = 'start'
      if args.get('route'):
         mode.profileParams.connectionType = 'route'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.profileParams.connectionType = mode.profileParams.connectionTypeDefault

IpsecProfileConfigMode.addCommandClass( IpsecConnectionType )

#-----------------------------------------------------------------------------------
# mode [ tunnel | transport ]
#-----------------------------------------------------------------------------------
class IpsecModeType( CliCommand.CliCommandClass ):
   syntax = 'mode [ tunnel | transport ]'
   noOrDefaultSyntax = 'mode ...'
   data = { 'mode': 'IPsec mode type',
            'tunnel': 'Tunnel mode',
            'transport': 'Transport mode'
          }

   @staticmethod
   def handler( mode, args ):
      if args.get( 'tunnel' ):
         mode.profileParams.mode = 'tunnel'
         mode.profileParams.modeEnum = 'IPSEC_TUNNEL_MODE'
      if args.get( 'transport' ):
         mode.profileParams.mode = 'transport'
         mode.profileParams.modeEnum = 'IPSEC_TRANSPORT_MODE'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.profileParams.mode = 'tunnel'
      mode.profileParams.modeEnum = 'IPSEC_TUNNEL_MODE'

IpsecProfileConfigMode.addCommandClass( IpsecModeType )

#----------------------------------------------------------------------------------
# sa-policy <name>
#----------------------------------------------------------------------------------
class SetIpsecTransformSet( CliCommand.CliCommandClass ):
   syntax = 'sa-policy SA_POLICY_NAME'
   noOrDefaultSyntax = syntax
   data = { 'sa-policy': 'Security Assoc Name',
            'SA_POLICY_NAME': saPolicyNameMatcher
          }

   @staticmethod
   def handler( mode, args ):
      saName = args[ 'SA_POLICY_NAME' ]

      if mode.profile:
         # prevent event handlers from being called unnecessarily
         if saName == mode.profile.securityAssocName:
            return
      else:
         mode.addError( ' Ipsec profile cannot be created' )
         return

      # Check if the sa-policy is present
      if saName not in ikeConfig().securityAssoc:
         mode.addError( f' IPsec SA policy {saName} not created' )
         return

      mode.profile.securityAssocName = saName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      saName = args[ 'SA_POLICY_NAME' ]

      if mode.profile:
         # Check if the sa-policy is present
         if saName != mode.profile.securityAssocName:
            mode.addError( f' IPsec SA {saName} not applied' )
            return
         mode.profile.securityAssocName = ''


IpsecProfileConfigMode.addCommandClass( SetIpsecTransformSet )

#----------------------------------------------------------------------------------
# ike-policy <policy>
#----------------------------------------------------------------------------------
class SetIsakmpPolicy( CliCommand.CliCommandClass ):
   syntax = 'ike-policy IKE_POLICY_NAME'
   noOrDefaultSyntax = syntax
   data = { 'ike-policy': 'ISAKMP policy',
            'IKE_POLICY_NAME': ikePolicyNameMatcher
          }

   @staticmethod
   def handler( mode, args ):
      policyName = args[ 'IKE_POLICY_NAME' ]

      profile = mode.getOrCreateProfileConfig( ikeConfig() )
      if profile:
         # prevent event handlers from being called unnecessarily
         if policyName == profile.ikePolicyName:
            return
      else:
         mode.addError( ' Ipsec profile cannot be created' )
         return

      # Check if the ike-policy is present
      if policyName not in ikeConfig().ikePolicy:
         mode.addError( f' Ike policy {policyName} not created' )
         return

      profile.ikePolicyName = policyName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      policyName = args[ 'IKE_POLICY_NAME' ]

      if mode.profile:
         # Check if the ike-policy is present
         if policyName != mode.profile.ikePolicyName:
            mode.addError( f' Ike policy {policyName} not applied' )
            return
         mode.profile.ikePolicyName = ''

IpsecProfileConfigMode.addCommandClass( SetIsakmpPolicy )

#-----------------------------------------------------------------------------------
# "shared-key ( CLEARTEXT | ( 0 TYPE0KEY ) | ( 7 TYPE7KEY ) | ( 8a TYPE8AKEY ) )"
#-----------------------------------------------------------------------------------
keyValueMatcher = CliMatcher.PatternMatcher( 
      pattern=r'^[-+`~&!@#$%^*(),.:;=/|{}<>_a-zA-Z0-9\\\[\]\'\"]+$',
      helpname='WORD', helpdesc='Clear text key value' )
sharedKeyCliExpr = ReversibleSecretCli.ReversiblePasswordCliExpression(
      cleartextMatcher=keyValueMatcher )

class IpsecPreSharedKey( CliCommand.CliCommandClass ):
   syntax = 'shared-key SHARED_KEY'
   noOrDefaultSyntax = 'shared-key ...'
   data = { 'shared-key': 'Shared key',
            'SHARED_KEY': sharedKeyCliExpr
          }

   @staticmethod
   def handler( mode, args ):
      mode.profile.ikeSharedKey = args[ 'SHARED_KEY' ]
   
   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.profile.ikeSharedKey = ReversibleSecretCli.getDefaultSecret()

IpsecProfileConfigMode.addCommandClass( IpsecPreSharedKey )

# -------------------------------------------------------------------------------
#  pki-profile PROFILE
# -------------------------------------------------------------------------------
class PkiProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'pki-profile PROFILE'
   noOrDefaultSyntax = 'pki-profile ...'
   data = { 'pki-profile': 'Configure PKI profile for IPsec',
            'PROFILE': profileNameMatcher
          }

   @staticmethod
   def handler( mode, args ):
      mode.profileParams.pkiProfile = args[ 'PROFILE' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      default = mode.profileParams.pkiProfileDefault
      mode.profileParams.pkiProfile = default

IpsecProfileConfigMode.addCommandClass( PkiProfileCmd )

#-------------------------------------------------------------------------------
#  flow parallelization encapsulation udp
#-------------------------------------------------------------------------------
class IpsecFlowParallelizationCmd( CliCommand.CliCommandClass ):
   syntax = 'flow parallelization encapsulation udp'
   noOrDefaultSyntax = 'flow parallelization encapsulation ...'
   data = { 'flow': 'Flow',
            'parallelization': 'Parallelize flows',
            'encapsulation': 'Encapsulation protocol',
            'udp': 'UDP'
          }

   @staticmethod
   def handler( mode, args ):
      fp = Tac.nonConst( mode.profileParams.flowParallelization )
      if 'udp' in args:
         fp.encapProtocol = IpsecEncapProtocolEnum.udp
      else:
         fp.encapProtocol = IpsecEncapProtocolEnum.no_encap
      mode.profileParams.flowParallelization = fp

   noOrDefaultHandler = handler

IpsecProfileConfigMode.addCommandClass( IpsecFlowParallelizationCmd )

#-------------------------------------------------------------------------------
#  flow entropy udp
#-------------------------------------------------------------------------------
def encapSrcPortRangeGuard( mode, token ):
   encapSrcPortRangeSupported = gv.ipsecCapabilities.encapSrcPortRangeSupported
   return None if encapSrcPortRangeSupported else CliParser.guardNotThisPlatform

entropyGuardedKw = CliCommand.guardedKeyword(
      'entropy', 'Distinguish flows', encapSrcPortRangeGuard )

class IpsecFlowEntropyCmd( CliCommand.CliCommandClass ):
   syntax = 'flow entropy udp'
   noOrDefaultSyntax = syntax
   data = { 'flow': 'Flow',
            'entropy': entropyGuardedKw,
            'udp': 'UDP'
          }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( IpsecEntropyUdpConfigMode, name='udp' )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noHandler( mode, args ):
      # Set no flow entropy (use IKE-negotiated source port)
      setEncapSrcPortRange( mode.profileParams, IpsecEncapSrcPortRange() )

   @staticmethod
   def defaultHandler( mode, args ):
      # Set default flow entropy (use dynamic port range)
      setEncapSrcPortRange(
            mode.profileParams, IpsecFlowParallelization().dynamicPortRange )

IpsecProfileConfigMode.addCommandClass( IpsecFlowEntropyCmd )

#-------------------------------------------------------------------------------
# port source offset OFFSET length PORT_LENGTH
#-------------------------------------------------------------------------------
class UdpSourcePortRangeCmd( CliCommand.CliCommandClass ):
   syntax = ( 'port source offset OFFSET length PORT_LENGTH' )
   noOrDefaultSyntax = 'port source offset ...'
   data = { 'port': 'Flow entropy UDP port range',
            'source': 'Source port range for UDP encapsulated packets',
            'offset': 'Starting port number',
            'OFFSET': CliMatcher.IntegerMatcher( 1, 65535,
                  helpdesc='UDP port number' ),
            'length': 'Number of ports',
            'PORT_LENGTH': CliMatcher.IntegerMatcher( 1, 65535,
               helpdesc='UDP port length' ),
   }

   @staticmethod
   def handler( mode, args ):
      offset = args[ 'OFFSET' ]
      portLength = args[ 'PORT_LENGTH' ]

      if portLength + offset > 0x10000:
         mode.addErrorAndStop(
               f'Offset + Length must not be greater than {0x10000}' )

      # Verify that portLength is a power of 2, 0 is allowed
      if bin( portLength ).count( '1' ) > 1:
         mode.addErrorAndStop( f'Length must be a power of 2, e.g. {0x1000}' )

      mode.encapSrcPortRange = IpsecEncapSrcPortRange( offset, portLength )

   @staticmethod
   def noHandler( mode, args ):
      # Set no flow entropy
      mode.encapSrcPortRange = IpsecEncapSrcPortRange()

   @staticmethod
   def defaultHandler( mode, args ):
      # Set flow entropy with default source port range
      mode.encapSrcPortRange = IpsecFlowParallelization().dynamicPortRange

IpsecEntropyUdpConfigMode.addCommandClass( UdpSourcePortRangeCmd )

#-------------------------------------------------------------------------------
# dpd <interval> <timeout> [ clear | hold | restart ]
#
# dpd hold and restart are not yet supported fully by Ipsec agent even though
# it is supported by strongswan. We would like to guard these cli options until
# support is added. None of our config guide or TOI talk about dpd hold and
# restart and therefore we should be fine
#-------------------------------------------------------------------------------
def guardDpdHoldAction( mode, token ):
   return CliParser.guardNotThisPlatform

def guardDpdRestartAction( mode, token ):
   return CliParser.guardNotThisPlatform

dpdHoldConfigKw = CliCommand.guardedKeyword( 'hold',
                        helpdesc='Re-negotiate connection on demand',
                        guard=guardDpdHoldAction )
dpdRestartConfigKw = CliCommand.guardedKeyword( 'restart',
                        helpdesc='Restart connection immediately',
                        guard=guardDpdRestartAction )

class IpsecDpd( CliCommand.CliCommandClass ):
   syntax = 'dpd <interval> <timeout> [ clear | hold | restart ]'
   noOrDefaultSyntax = 'dpd ...'
   data = { 'dpd': 'Dead Peer Detection',
            '<interval>': CliMatcher.IntegerMatcher( 2, 3600,
                       helpdesc='Interval(in seconds) between keep-alive messages' ),
            '<timeout>': CliMatcher.IntegerMatcher( 10, 3600,
                    helpdesc='Time(in seconds) after which the action is applied.' ),
            'clear': 'Delete all connections',
            'hold': dpdHoldConfigKw,
            'restart': dpdRestartConfigKw,
          }

   @staticmethod
   def handler( mode, args ):
      interval = args.get( '<interval>' )
      timeout = args.get( '<timeout>' )
      action = dpdActionEnum.none
      if args.get( 'clear' ):
         action = dpdActionEnum.clear
      elif args.get( 'hold' ):
         action = dpdActionEnum.hold
      elif args.get( 'restart' ):
         action = dpdActionEnum.restart

      dpd = Tac.Value( 'Ipsec::Ike::DeadPeerDetection' )
      dpd.interval = interval
      dpd.timeout = timeout
      dpd.action = action
      mode.profileParams.dpd = dpd

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.profileParams.dpd = Tac.Value( 'Ipsec::Ike::DeadPeerDetection' )

IpsecProfileConfigMode.addCommandClass( IpsecDpd )

#-------------------------------------------------------------------------------
# "key controller"
#-------------------------------------------------------------------------------
def setKeyController( mode, args ):
   childMode = mode.childMode( KeyControllerConfigMode )
   mode.session_.gotoChildMode( childMode )
   if not childMode.controllerCfg.configured:
      childMode.controllerCfg.configured = True
      ikeConfig().keyController = childMode.controllerCfg

def noKeyController( mode, args ):
   controllerCfg = Tac.nonConst( ikeConfig().keyController )
   controllerCfg.profileName = ""
   controllerCfg.dhLifetime = ikeConfig().keyController.dhLifetimeDefault
   controllerCfg.holdTime = ikeConfig().keyController.holdTimeDefault
   controllerCfg.dhGroup = ikeConfig().keyController.dhGroupDefault
   controllerCfg.configured = False
   ikeConfig().keyController = controllerCfg

class IpsecKeyController( CliCommand.CliCommandClass ):
   '''Enter the key controller mode to apply the profile we want to use'''
   syntax = 'key controller'
   noOrDefaultSyntax = 'key controller'
   data = { 'key': 'Config IPsec key controller',
            'controller': 'Enter key controller config mode' }

   handler = setKeyController
   noOrDefaultHandler = noKeyController

IpsecConfigMode.addCommandClass( IpsecKeyController )

#-------------------------------------------------------------------------------
# "profile <profile-name>" in "key controller" mode
#-------------------------------------------------------------------------------
def setKeyControllerProfile( mode, args ):
   profileName = args[ '<profile-name>' ]

   [ val, message ] = isIpsecProfilePresent( profileName )
   if not val:
      mode.addError( message )
      return

   if mode.controllerCfg.profileName != profileName:
      mode.controllerCfg.profileName = profileName

def noSetKeyControllerProfile( mode, args ):
   mode.controllerCfg.profileName = ""

class IpsecKeyControllerProfile( CliCommand.CliCommandClass ):
   '''Apply profile to the key controller'''
   syntax = 'profile <profile-name>'
   noOrDefaultSyntax = 'profile ...'
   data = { 'profile': 'Apply IPsec profile',
            '<profile-name>': ipsecProfileNameMatcher
          }

   handler = setKeyControllerProfile
   noOrDefaultHandler = noSetKeyControllerProfile

KeyControllerConfigMode.addCommandClass( IpsecKeyControllerProfile )

#-----------------------------------------------------------------------------------
# 'lifetime <time> hours' for key controller
#-----------------------------------------------------------------------------------
class IpsecControllerDHLifetime( CliCommand.CliCommandClass ):
   syntax = 'lifetime <time> hours'
   noOrDefaultSyntax = 'lifetime ...'
   data = { 'lifetime': 'Set DH key lifetime',
            '<time>': hoursMatcher,
            'hours' : 'Lifetime in hours',
          }

   @staticmethod
   def handler( mode, args ):
      mode.controllerCfg.dhLifetime = \
            Tac.newInstance( 'Ipsec::Ike::IpsecLifetime',
                             args.get( '<time>' ) * 3600 )

   @staticmethod
   def noOrDefaultHandler( mode, args):
      mode.controllerCfg.dhLifetime = mode.controllerCfg.dhLifetimeDefault

KeyControllerConfigMode.addCommandClass( IpsecControllerDHLifetime )

#-----------------------------------------------------------------------------------
# 'holdtime <time> hours' for key controller
#-----------------------------------------------------------------------------------
class IpsecControllerHoldTime( CliCommand.CliCommandClass ):
   syntax = 'holdtime <time> hours'
   noOrDefaultSyntax = 'holdtime ...'
   data = { 'holdtime': 'Set hold duration for old key material',
            '<time>': hoursMatcher,
            'hours': 'Holdtime in hours',
          }

   @staticmethod
   def handler( mode, args ):
      mode.controllerCfg.holdTime = \
            Tac.newInstance( 'Ipsec::Ike::IpsecLifetime',
                             args.get( '<time>' ) * 3600 )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.controllerCfg.holdTime = mode.controllerCfg.holdTimeDefault

if togglePfConnectionLossEnabled():
   KeyControllerConfigMode.addCommandClass( IpsecControllerHoldTime )

#-----------------------------------------------------------------------------------
# 'dh-group GROUP_NUM' for key controller
#-----------------------------------------------------------------------------------
class IpsecControllerDHGroup( CliCommand.CliCommandClass ):
   syntax = 'dh-group GROUP_NUM'
   noOrDefaultSyntax = 'dh-group ...'
   data = { 'dh-group': 'Set Diffie-Hellman group',
            'GROUP_NUM': matcherDHGroupNum 
          }

   @staticmethod
   def handler( mode, args ):
      dhGroupNum = args[ 'GROUP_NUM' ]
      mode.controllerCfg.dhGroup = getDHGroupFromNum( dhGroupNum )

   @staticmethod
   def noOrDefaultHandler( mode, args):
      mode.controllerCfg.dhGroup = mode.controllerCfg.dhGroupDefault

KeyControllerConfigMode.addCommandClass( IpsecControllerDHGroup )

# ------------------------------------------------------------------------------
# 'clear ip security connection path <name>'
# ------------------------------------------------------------------------------

clearMatcher = CliToken.Clear.clearKwNode
pathNameMatcher = CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9_-]+',
                        helpname='WORD', helpdesc='Name of the path' )

class ClearIpsecPathCmd( CliCommand.CliCommandClass ):
   syntax = "clear ip security connection path PATHNAME"
   data = { 'clear': clearMatcher,
            'ip': CliToken.Ip.ipMatcherForClear,
            'security': 'IP security information',
            'connection': 'Connection information',
            'path': 'Path connection',
            'PATHNAME': pathNameMatcher
          }

   @staticmethod
   def handler( mode, args ):
      path = args.get( 'PATHNAME' )
      for connKey in gv.ipsecPathStatus.ipsecPathStatus:
         pathStatus = gv.ipsecPathStatus.ipsecPathStatus.get( connKey )
         if pathStatus and path == pathStatus.pathName:
            pathKey = ikeConfig().clearConnection.newMember( connKey )
            pathKey.clearCnt += 1
            break

if toggleIpsecClearCommandEnabled():
   BasicCli.EnableMode.addCommandClass( ClearIpsecPathCmd )

# ----------------------------------------------------
# Base class for crypto suite mode
# ----------------------------------------------------
class IpsecCryptoSuiteModeBase:
   def __init__( self, suiteName, suiteType ):
      self.suiteType = suiteType
      self.suiteName = suiteName
      self.suite = self.getOrCreateSuiteConfig()
      self.suiteParams = None
      self.suiteNotConfigured = True

   def getOrCreateSuiteConfig( self ):
      if self.suiteType == 'ike':
         return ikeConfig().ikeCryptoSuiteConfig.newMember( self.suiteName )
      else:
         return ikeConfig().saCryptoSuiteConfig.newMember( self.suiteName )

   def commitSuiteCfg( self ):
      if self.suiteNotConfigured:
         if self.suiteType == 'ike':
            del ikeConfig().ikeCryptoSuiteConfig[ self.suiteName ]
         else:
            del ikeConfig().saCryptoSuiteConfig[ self.suiteName ]
         return
      if self.suite.cryptoSuite != self.suiteParams:
         self.suite.cryptoSuite = self.suiteParams

# ----------------------------------------------------------------------
# crypto ike-suite <name>
#       encryption <encr> integrity <integ> dh-group <dhGroup>
# ----------------------------------------------------------------------
class IpsecIkeCryptoSuiteMode( IpsecCryptoSuiteModeBase, IpsecIkeCryptoSuiteCliMode,
                               BasicCli.ConfigModeBase ):
   name = 'IKE crypto suite configuration'

   def __init__( self, parent, session, ikeSuiteName ):
      IpsecCryptoSuiteModeBase.__init__( self, ikeSuiteName, 'ike' )
      IpsecIkeCryptoSuiteCliMode.__init__( self, ikeSuiteName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      if self.suiteNotConfigured:
         self.commitSuiteCfg()
      BasicCli.ConfigModeBase.onExit( self )

# ----------------------------------------------------------------------
# crypto sa-suite <name>
#       encryption <encr> integrity <integ> dh-group <dhGroup>
# ----------------------------------------------------------------------
class IpsecSaCryptoSuiteMode( IpsecCryptoSuiteModeBase, IpsecSaCryptoSuiteCliMode,
                              BasicCli.ConfigModeBase ):
   name = 'SA crypto suite configuration'

   def __init__( self, parent, session, saSuiteName ):
      IpsecCryptoSuiteModeBase.__init__( self, saSuiteName, 'sa' )
      IpsecSaCryptoSuiteCliMode.__init__( self, saSuiteName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      if self.suiteNotConfigured:
         self.commitSuiteCfg()
      BasicCli.ConfigModeBase.onExit( self )

class IpsecIkeCryptoSuiteModeCmd( CliCommand.CliCommandClass ):
   syntax = 'encryption ENCR integrity INTEG dh-group DHGROUP'
   data = {
         'encryption': 'Encryption type',
         'ENCR': CliMatcher.EnumMatcher( ikeEncryptionTypeData ),
         'integrity': 'Integrity algorithm',
         'INTEG': CliMatcher.EnumMatcher( ikeIntegrityTypeData ),
         'dh-group': 'Diffie-Hellman Group',
         'DHGROUP': CliMatcher.EnumMatcher( dhGroupTypeData ),
         }

   @staticmethod
   def handler( mode, args ):
      encryption = encryptionMap[ args[ 'ENCR' ] ]
      integrity = integrityMap[ args[ 'INTEG' ] ]
      dhGroup = groupNumToIpsecDhGroupMap[ args[ 'DHGROUP' ] ]

      rcvdSuiteParams = Tac.Value( "Ipsec::CryptoSuite",
                                    encryption, integrity, dhGroup )

      # Duplicate suite entry check
      for entry in ikeConfig().ikeCryptoSuiteConfig.values():
         if ( entry.name != mode.suiteName and
               entry.cryptoSuite == rcvdSuiteParams ):
            mode.addError( DUP_ENTRY_ERROR_STR.format( mode.suiteType, entry.name ) )
            return

      mode.suiteParams = rcvdSuiteParams
      # suite and its entry is valid
      mode.suiteNotConfigured = False
      mode.commitSuiteCfg()

class IpsecSaCryptoSuiteModeCmd( CliCommand.CliCommandClass ):
   syntax = 'encryption ( ( ENCR integrity INTEG ) | ( GCM_ENCR ) ) ' \
            '[ dh-group DHGROUP ]'
   data = {
         'encryption': 'Encryption type',
         'ENCR': CliMatcher.EnumMatcher( saEncryptionTypeData ),
         'GCM_ENCR': CliMatcher.EnumMatcher( saGcmEncryptionTypeData ),
         'integrity': 'Integrity algorithm',
         'INTEG': CliMatcher.EnumMatcher( saIntegrityTypeData ),
         'dh-group': 'Diffie-Hellman Group',
         'DHGROUP': CliMatcher.EnumMatcher( dhGroupTypeData ),
         }

   @staticmethod
   def handler( mode, args ):
      encr = args[ 'ENCR' ] if args.get( 'ENCR' ) else args[ 'GCM_ENCR' ]
      encryption = encryptionMap[ encr ]
      # Integrity for gcm encrs are ignored
      integrity = integrityMap[ args[ 'INTEG' ] ] if args.get( 'INTEG' ) else \
               IpsecHmacAlgorithmEnum.nonehash
      dhGroupNum = args.get( 'DHGROUP' )
      dhGroup = IpsecDhGroupEnum.IpsecDhGroupNone if dhGroupNum is None \
                else groupNumToIpsecDhGroupMap[ dhGroupNum ]

      rcvdSuiteParams = Tac.Value( "Ipsec::CryptoSuite",
                                    encryption, integrity, dhGroup )

      # Duplicate suite entry check
      for entry in ikeConfig().saCryptoSuiteConfig.values():
         if ( entry.name != mode.suiteName and
               entry.cryptoSuite == rcvdSuiteParams ):
            mode.addError( DUP_ENTRY_ERROR_STR.format( mode.suiteType, entry.name ) )
            return

      mode.suiteParams = rcvdSuiteParams
      # suite and its entry is valid
      mode.suiteNotConfigured = False
      mode.commitSuiteCfg()

# ---------------------------------------------------------
# crypto ( ike-suite <name> ) | ( sa-suite <name> )
# ---------------------------------------------------------
class IpsecCryptoSuiteCmd( CliCommand.CliCommandClass ):
   syntax = 'crypto ( ike-suite IKE_SUITE_NAME ) | ( sa-suite SA_SUITE_NAME )'
   noOrDefaultSyntax = syntax
   data = {
         'crypto': 'Defines a crypto algorithm suite',
         'ike-suite': 'IKE crypto suite',
         'IKE_SUITE_NAME': ikeCryptoSuiteNameMatcher,
         'sa-suite': 'SA crypto suite',
         'SA_SUITE_NAME': saCryptoSuiteNameMatcher,
         }

   @staticmethod
   def handler( mode, args ):
      if args.get( 'IKE_SUITE_NAME' ):
         name = args.get( 'IKE_SUITE_NAME' )
         childMode = mode.childMode( IpsecIkeCryptoSuiteMode, ikeSuiteName=name )
         mode.session_.gotoChildMode( childMode )
      else:
         name = args.get( 'SA_SUITE_NAME' )
         childMode = mode.childMode( IpsecSaCryptoSuiteMode, saSuiteName=name )
         mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if args.get( 'IKE_SUITE_NAME' ):
         name = args.get( 'IKE_SUITE_NAME' )
         ikeCryptoSuiteMap = ikeConfig().ikeCryptoSuiteConfig
         if name not in ikeCryptoSuiteMap:
            return
         BasicCliModes.removeCommentWithKey( f'ipsec-crypto-ike-{name}' )
         del ikeCryptoSuiteMap[ name ]
      else:
         name = args.get( 'SA_SUITE_NAME' )
         saCryptoSuiteMap = ikeConfig().saCryptoSuiteConfig
         if name not in saCryptoSuiteMap:
            return
         BasicCliModes.removeCommentWithKey( f'ipsec-crypto-sa-{name}' )
         del saCryptoSuiteMap[ name ]

def advanceSeqNo( mode ):
   SEQNO_STEP = 10.0
   if mode.suiteType == 'ike':
      seqNo = mode.ike.lastSeqNo + SEQNO_STEP
      mode.ike.lastSeqNo = seqNo
   else:
      seqNo = mode.secAssoc.lastSeqNo + SEQNO_STEP
      mode.secAssoc.lastSeqNo = seqNo
   return seqNo

def findPolicyCryptoSuite( mode, suiteName ):
   ( before, current, after ) = ( 0.0, None, None )
   matched = False
   if mode.suiteType == 'ike':
      cs = mode.ike.ikeCryptoSuite
   else:
      cs = mode.secAssoc.saCryptoSuite
   for ( seqNo, name ) in cs.items():
      if matched:
         after = seqNo
         break
      if suiteName == name.cryptoSuiteName:
         current = seqNo
         matched = True
      else:
         before = seqNo
   return ( before, current, after )

def addPolicyCryptoSuiteCommon( mode, suiteName, order, orderName=None ):
   if orderName is not None:
      # return if both suites are same
      if orderName == suiteName:
         return
      ( before, orderSeqno, after ) = findPolicyCryptoSuite( mode, orderName )
      if orderSeqno is None:
         mode.addError( f'{orderName} does not exist' )
         return
      # if suite already exists delete it
      delPolicyCryptoSuiteCommon( mode, suiteName )
      if order == 'before':
         seqNo = ( before + orderSeqno ) / 2
      else:
         seqNo = advanceSeqNo( mode ) if after is None else \
                                 ( ( orderSeqno + after ) / 2 )
   else:
      seqNo = findPolicyCryptoSuite( mode, suiteName )[ 1 ]
      if seqNo is not None:
         return
      seqNo = advanceSeqNo( mode )
   if mode.suiteType == 'ike':
      pSuite = mode.ike.newIkeCryptoSuite( seqNo )
   else:
      pSuite = mode.secAssoc.newSaCryptoSuite( seqNo )
   if pSuite:
      pSuite.cryptoSuiteName = suiteName

def delPolicyCryptoSuiteCommon( mode, suiteName ):
   ( before, seqNo, _ ) = findPolicyCryptoSuite( mode, suiteName )
   if seqNo is not None:
      if mode.suiteType == 'ike':
         if seqNo == mode.ike.lastSeqNo:
            mode.ike.lastSeqNo = before
         del mode.ike.ikeCryptoSuite[ seqNo ]
      else:
         if seqNo == mode.secAssoc.lastSeqNo:
            mode.secAssoc.lastSeqNo = before
         del mode.secAssoc.saCryptoSuite[ seqNo ]

def setPolicyCryptoSuite( mode, args ):
   name = args.get( 'SUITE' )
   order = args.get( 'ORDER' )
   name2 = args.get( 'SUITE2' )
   addPolicyCryptoSuiteCommon( mode, name, order, orderName=name2 )

def unsetPolicyCryptoSuite( mode, args ):
   delPolicyCryptoSuiteCommon( mode, args.get( 'SUITE' ) )

# ---------------------------------------------------------------------
# crypto suite <ikeSuiteName> [ after | before <ikeSuiteName2> ] )
# ---------------------------------------------------------------------
class IpsecIkePolicyCryptoSuiteCmd( CliCommand.CliCommandClass ):
   syntax = 'crypto suite SUITE [ORDER SUITE2]'
   noOrDefaultSyntax = 'crypto suite SUITE ...'
   data = {
            'crypto': 'Specify a crypto algorithm suite',
            'suite': 'IKE crypto suite',
            'SUITE': ikeCryptoSuiteNameMatcher,
            'ORDER': policyCryptoSuiteOrderMatcher,
            'SUITE2': ikeCryptoSuiteNameMatcher,
         }

   handler = setPolicyCryptoSuite
   noOrDefaultHandler = unsetPolicyCryptoSuite

# ---------------------------------------------------------------------
# crypto suite <saSuiteName> [ after | before <saSuiteName2> ] )
# ---------------------------------------------------------------------
class IpsecSaPolicyCryptoSuiteCmd( CliCommand.CliCommandClass ):
   syntax = 'crypto suite SUITE [ORDER SUITE2]'
   noOrDefaultSyntax = 'crypto suite SUITE ...'
   data = {
            'crypto': 'Specify a crypto algorithm suite',
            'suite': 'SA crypto suite',
            'SUITE': saCryptoSuiteNameMatcher,
            'ORDER': policyCryptoSuiteOrderMatcher,
            'SUITE2': saCryptoSuiteNameMatcher,
         }

   handler = setPolicyCryptoSuite
   noOrDefaultHandler = unsetPolicyCryptoSuite

if toggleMultiCryptoEnabled():
   IpsecConfigMode.addCommandClass( IpsecCryptoSuiteCmd )
   IpsecIkeCryptoSuiteMode.addCommandClass( IpsecIkeCryptoSuiteModeCmd )
   IpsecSaCryptoSuiteMode.addCommandClass( IpsecSaCryptoSuiteModeCmd )
   IpsecIkeConfigMode.addCommandClass( IpsecIkePolicyCryptoSuiteCmd )
   IpsecSecurityAssocConfigMode.addCommandClass( IpsecSaPolicyCryptoSuiteCmd )

#-------------------------------------------------------------------------------
# Plugin Func
#-------------------------------------------------------------------------------
def Plugin( entityManager ):

   gv.ikeConfigDir = ConfigMount.mount( entityManager, 'ipsec/ike/config',
                                        'Ipsec::Ike::Config', 'w' )
   gv.ikeStatusDir = LazyMount.mount( entityManager, 'ipsec/ike/status',
                                      'Ipsec::Ike::Status', 'r' )
   gv.tunIntfConfigDir = ConfigMount.mount( entityManager,
                                            'interface/config/tunnel/intf',
                                            'Interface::TunnelIntfConfigDir', 'w' )
   gv.tunIntfStatusDir = LazyMount.mount( entityManager,
                                          'interface/status/tunnel/intf',
                                          'Interface::TunnelIntfStatusDir', 'r' )
   gv.hwCapabilities = LazyMount.mount( entityManager,
                                        'bridging/hwcapabilities',
                                        'Bridging::HwCapabilities',
                                        'r' )
   gv.ipsecCapabilities = LazyMount.mount( entityManager,
                                           'ipsec/capabilities/status',
                                           'Ipsec::Capabilities::Status', 'r' )
   if toggleIpsecServiceAclEnabled():
      gv.aclConfig = ConfigMount.mount( entityManager, "acl/config/cli",
                                        "Acl::Input::Config", "w" )
      gv.aclCpConfig = ConfigMount.mount( entityManager, "acl/cpconfig/cli",
                                          "Acl::Input::CpConfig", "w" )
   gv.ipsecConfigMode = LazyMount.mount( entityManager, 'ipsec/configmode',
                                         'Ipsec::ConfigMode', 'w' )

   if toggleIpsecClearCommandEnabled():
      gv.ipsecPathStatus = LazyMount.mount( entityManager, 'ipsec/path/status',
                                         'Ipsec::Path::PathStatusDir', 'r' )

   TunnelIntfCli.ipsecHook.addExtension( isIpsecProfilePresent )

