# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

""" Implements Cli show commands for displaying kernel information
All commands call Tac.run()
Errors from the bash command are returned as is
"""

import re
import BasicCli
import ShowCommand
import Tac
from CliPlugin.IpsecModel import (
      ShowKernelIpsecModel,
      ShowKernelIpsecConnections,
      ShowKernelIpsecSecurityAssociations,
      ShowKernelIpsecWorkerThreads,
      ShowKernelIpsecStrongSwan,
      ShowKernelIpsecIkeSa,
      ShowKernelIpsecChildSa
)
from CliPlugin import VrfCli
from datetime import datetime
import time
import LazyMount
import SmashLazyMount

ipsecStatus = None
vrfIdStatus = None

# Below functions added to reduce nesting
def addSecurityAssociations( output, matched, matcher, lastIkeSaId, i ):
   sec = matcher.sub( "", i )[1:].strip()
   keyTuple = re.sub( r"(\{\d+\}|\[\d+\])", "", matched )
   uniqId = re.search( r"(\{\d+\}|\[\d+\])",  matched ).group(0)
   # Check if the matched tuple exists in the dictionary
   if keyTuple in output.securityAssociations:
      if re.search( r"\[\d+\]", matched ):
         # Check if ikeSa uniqId already exists for this tuple
         if uniqId in output.securityAssociations[ keyTuple ].ikeSa:
            output.securityAssociations[ keyTuple ].ikeSa[ uniqId ].\
                                                   summary.append( sec )
         else:
            output.securityAssociations[ keyTuple ].ikeSa[ uniqId ] = \
                                                   ShowKernelIpsecIkeSa()
            output.securityAssociations[ keyTuple ].ikeSa[ uniqId ].\
                                                   summary.append( sec )
            lastIkeSaId = uniqId
      else:
         # Check if childSa uniqId already exists for this tuple
         if uniqId in output.securityAssociations[ keyTuple ].\
                                          ikeSa[ lastIkeSaId ].childSa:
            output.securityAssociations[ keyTuple ].\
                           ikeSa[ lastIkeSaId ].childSa[ uniqId ].\
                           summary.append( sec )
         else:
            output.securityAssociations[ keyTuple ].\
                           ikeSa[ lastIkeSaId ].childSa[ uniqId ] = \
                           ShowKernelIpsecChildSa()
            output.securityAssociations[ keyTuple ].\
                           ikeSa[ lastIkeSaId ].childSa[ uniqId ].\
                           summary.append( sec )
   else:
      output.securityAssociations[ keyTuple ] = \
                                    ShowKernelIpsecSecurityAssociations()
      output.securityAssociations[ keyTuple ].ikeSa[ uniqId ] = \
                                    ShowKernelIpsecIkeSa()
      output.securityAssociations[ keyTuple ].ikeSa[ uniqId ].\
                                    summary.append( sec )
      lastIkeSaId = uniqId
   return lastIkeSaId

def addWorkerThreads( output, threads ):
   for m in threads:
      t = m.split()
      if "idle" in m:
         output.strongSwan.workerThreads.idle = int( t[ 0 ] )
         output.strongSwan.workerThreads.total = int( t[ 2 ] )
      elif "working" in m:
         output.strongSwan.workerThreads.working = t[ 0 ]
      elif "queue" in m:
         output.strongSwan.workerThreads.queue = t[ 2 ]
      elif "scheduled" in m:
         output.strongSwan.workerThreads.scheduled = int( t[ 1 ] )

def addStatus( output, strongSwan ):
   for m in strongSwan:
      if len( m.split() ) == 1:
         output.strongSwan.architecture = m.split()[ 0 ]
      else:
         if m.split()[ 0 ] == "strongSwan":
            output.strongSwan.version = m.split()[ 1 ]
         elif m.split()[ 0 ] == "Linux":
            output.strongSwan.linux = m.split()[ 1 ]

def showKernelIpsec( mode, args ):
   output = ShowKernelIpsecModel()

   vrfName = args.get( 'VRF', VrfCli.DEFAULT_VRF )
   vrfList = []
   if vrfName == 'all':
      for vrfId in ipsecStatus.vrfsInUse:
         ve = vrfIdStatus.vrfIdToName.get( vrfId )
         if not ve:
            continue
         vrfList.append( ve.vrfName )
   else:
      vrfList = [ vrfName ]

   first = True
   for v in sorted( vrfList ):
      doShowKernelIpsec( output, mode, args, v, first )
      first = False

   return output

def doShowKernelIpsec( output, mode, args, vrfName, first ):

   if 'datapath' in args:
      cmd = ''
      # for non-default VRF lookups we need to go into the VRF network
      # namespace by prepending 'ip netns exec ns-<vrfName>'.  For
      # this command we are just looking at /proc, so we do not care
      # about the 'netnsd'-generated mount namespace that strongswan
      # lives in
      if vrfName != VrfCli.DEFAULT_VRF:
         cmd += f'ip netns exec ns-{vrfName} '
      cmd += "cat /proc/net/xfrm_stat"

      output.setDatapath( True )
      items = Tac.run( cmd.split(), stdout=Tac.CAPTURE,
               stderr=Tac.CAPTURE, ignoreReturnCode=True, asRoot=True )
      items = items.splitlines()
      for i in items:
         itemList = i.split( '\t' )
         if len( itemList ) > 1:
            output.dataPath[ itemList[ 0 ].strip() ] = int( itemList[ 1 ] )
            output.addToDataPath( itemList[ 0 ], itemList[ 1 ] )
         else:
            # Error occurred, directly append the output to _showIpsec list
            output.appendShowIpsec( i )

   else:
      if not first:
         output.appendShowIpsec( '\n' )
      output.appendShowIpsec( f'VRF: {vrfName}' )
      cmd = []
      # Unlike the 'datpath' command this one is a wrapper around
      # 'stroke' which depends on the strongswan config files.
      # This means that we need to get into the mount namespace
      # that the SuperServerPlugin created for this instance of
      # strongswan.  Note that the 'netns' prefix command needs to
      # precede any 'ip netns exec' command
      cmd += [ '/usr/bin/ipsecmount', vrfName ]
      cmd += [ 'strongswan statusall' ]

      output.setDatapath( False )
      items = Tac.run( cmd, stdout=Tac.CAPTURE,
               stderr=Tac.CAPTURE, ignoreReturnCode=True, asRoot=True )
      items = items.splitlines()
      # Key is which group each of the fields/attributes belong to
      # i.e. Status, Listening, Connections, Security
      key = ""
      matcher = re.compile( "^.[^:]*" )
      headerMatch = re.compile( "^.*:" )
      # Keep track of last ikeSa uniqId to ensure correct childSa's
      lastIkeSaId = ""
      for i in items:
         if i.strip() in ( "none", "" ):
            continue
         hMatch = headerMatch.match(i)
         matches = False
         if hMatch is not None:
            matches = hMatch.group(0) == i
         if matches:
            for word in ( "Status", "Listening", "Connections", "Security" ):
               if word in hMatch.group(0):
                  key = word
                  break
            if key == "Status":
               output.strongSwan = ShowKernelIpsecStrongSwan()
               statusMatch = re.compile( r"(strongSwan.*)\)" )
               matched = statusMatch.search( hMatch.group(0) ).group( 1 )
               strongSwan = matched.split( "," )
               addStatus( output, strongSwan )
         else:
            matched = matcher.match( i ).group( 0 )
            if "uptime" in matched:
               dateRe = re.compile( r"\w{3}\s\d{1,2}\s\d{2}:\d{2}:\d{2}\s\d{4}" )
               date = dateRe.search( i ).group( 0 )
               datetimeFormat = float( datetime.strptime(
                                       date, '%b %d %H:%M:%S %Y' ).strftime("%s") )
               now = int( time.time() )
               output.strongSwan.uptime = int( now - datetimeFormat )
               output.strongSwan.startTime = datetimeFormat
            elif "malloc" in matched:
               malloc = re.sub( hMatch.group(0), "", i ).split( ',' )
               for m in malloc:
                  a = m.strip().split()
                  output.strongSwan.malloc[ a[ 0 ] ] = int( a[ 1 ] )
            elif "worker threads" in matched:
               threads = matcher.sub("", i)[1:].split( ',')
               output.strongSwan.workerThreads = ShowKernelIpsecWorkerThreads()
               addWorkerThreads( output, threads )
            elif "loaded plugins" in matched:
               plugins = re.sub( hMatch.group(0), "", i ).split()
               for p in plugins:
                  output.strongSwan.plugins.append( p )
            elif key == "Listening":
               output.listeningIpAddresses.append( i.strip() )
            elif key == "Connections":
               if matched not in output.connections:
                  output.connections[ matched ] = ShowKernelIpsecConnections()
               connection = matcher.sub("", i)[1:]
               output.connections[ matched ].summary.append( connection.strip() )
            elif key == "Security":
               lastIkeSaId = addSecurityAssociations( output, matched,
                                                            matcher, lastIkeSaId, i )
         output.appendShowIpsec( i )

vrfExprFactory = VrfCli.VrfExprFactory(
      helpdesc='VRF instance' )

#--------------------------------------------------------------------------------
# show kernel ipsec [ datapath ]
#--------------------------------------------------------------------------------
class KernelIpsecCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show kernel ipsec [ VRF ] [ datapath ]'
   data = {
      'kernel': 'Show kernel information',
      'ipsec': 'Show Ipsec information',
      'VRF': vrfExprFactory,
      'datapath': 'Show Ipsec Datapath information',
   }
   handler = showKernelIpsec
   cliModel = ShowKernelIpsecModel

BasicCli.addShowCommandClass( KernelIpsecCmd )

def Plugin( entityManager ):
   global vrfIdStatus
   global ipsecStatus

   ipsecStatus = LazyMount.mount( entityManager, 'ipsec/ike/status',
                                  'Ipsec::Ike::Status', 'r' )
   vrfIdStatus = SmashLazyMount.mount( entityManager, 'vrf/vrfIdMapStatus',
                                       'Vrf::VrfIdMap::Status',
                                       SmashLazyMount.mountInfo( 'reader' ),
                                       autoUnmount=True )
