# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-in

from itertools import chain
import time
import collections
from collections import defaultdict
import socket
import Arnet, Tracing
from ArnetModel import (
   IpGenericAddress,
   IpGenericPrefix
)
from CliModel import (
      Model,
      DeferredModel,
      Submodel,
      Bool,
      Dict,
      Enum,
      Int,
      Str,
      List,
      Float
      )
from IntfModels import Interface
import Tac
from IpLibConsts import DEFAULT_VRF
import Toggles.IpsecToggleLib
from Toggles.PolicyMapToggleLib import togglePolicyBasedVpnEnabled

# pylint: disable-msg=C0209

DEFAULT_VRF_ID = Tac.Type( 'Vrf::VrfIdMap::VrfId' ).defaultVrf

__defaultTraceHandle__ = Tracing.Handle( "IpsecCliModel" )
t8 = Tracing.trace8
t9 = Tracing.trace9

IpsecHmacAlgorithmEnum = Tac.Type( "Ipsec::IpsecHmacAlgorithm" )
IpsecAuthTypeEnum = Tac.Type( "Ipsec::Ike::IpsecAuthType" )
IpsecEspAlgorithmEnum = Tac.Type( "Ipsec::IpsecEspAlgorithm" )
IpsecDhGroupEnum = Tac.Type( "Ipsec::IpsecDhGroup" )
IpsecPeerDownReasonEnum = Tac.Type( "Ipsec::PeerDownReason" )
IpsecPeerConnStatusEnum = Tac.Type( "Ipsec::PeerConnStatus" )
IpsecLifetimeUnitEnum = Tac.Type( "Ipsec::LifetimeUnit" )
SslProfileState = Tac.Type( "Mgmt::Security::Ssl::ProfileState" )

AF_INET = 2
AF_INET6 = 10
DIR_IN = 0
DIR_OUT = 1

indent1 = 2 * ' '
indent2 = 4 * ' '
indent3 = 6 * ' '

IPSEC_DIRECTION_LIST = [ 'anyDir',
                         'inboundDir',
                         'outboundDir',
                         'fwdDir',
                         'unknownDir',]
IPSEC_DIRECTION_MAP = { 0 : 'Ipsec Any Policy',
                        1 : 'Ipsec Inbound Policy',
                        2 : 'Ipsec Outbound Policy',
                        3 : 'Ipsec Forward Policy',
                        -1 : 'Unknown Policy' }

IPSEC_PROTO_LIST = [ 'esp', 'ah', 'unknownProto', ]
IPSEC_PROTO_MAP = { 50 : 'Encapsulation Security Payload protocol',
                    51 : 'Authentication Header protocol',
                    -1 : 'Unknown Proto' }

IPSEC_MODE_LIST = [ 'transport',
                    'tunnel',
                    'routeOptimization',
                    'inboundTrigger',
                    'beet',
                    'unknownMode', ]
IPSEC_MODE_MAP = { 0 : 'Transport',
                   1 : 'Tunnel',
                   2 : 'Route optimization',
                   3 : 'Inbound trigger',
                   4 : 'Bound end-to-end tunnel',
                   -1 : 'Unknown Mode' }

IPSEC_INTEGRITY_MAP = { IpsecHmacAlgorithmEnum.md5  : '128bit Hash',
                        IpsecHmacAlgorithmEnum.sha1 : '160bit Hash',
                        IpsecHmacAlgorithmEnum.sha256 : '256bit Hash',
                        IpsecHmacAlgorithmEnum.sha384 : '384bit Hash',
                        IpsecHmacAlgorithmEnum.sha512 : '512bit Hash', 
                        IpsecHmacAlgorithmEnum.nullhash : 'Null Hash',
                        'unknownIntegrity' : 'Unknown', }

IPSEC_ENCRYPTION_MAP = { IpsecEspAlgorithmEnum.des : 'Triple DES', 
                         IpsecEspAlgorithmEnum.aes128 : '128-bit AES', 
                         IpsecEspAlgorithmEnum.aes192 : '192-bit AES', 
                         IpsecEspAlgorithmEnum.aes256 : '256-bit AES', 
                         IpsecEspAlgorithmEnum.aes128gcm64 :
                             '128 bit AES-GCM',
                         IpsecEspAlgorithmEnum.aes128gcm128 :
                             '128 bit AES-GCM',
                         IpsecEspAlgorithmEnum.aes256gcm128 :
                             '256 bit AES-GCM',
                         IpsecEspAlgorithmEnum.nullesp : 'Null encryption', 
                         'unknownEncr' : 'Unknown', }

IPSEC_AUTH_MAP = { IpsecAuthTypeEnum.pre_share: 'Pre-shared',
                   IpsecAuthTypeEnum.rsa_sig: 'RSA Signature',
                   IpsecAuthTypeEnum.pki: 'PKI',
                   'unknownAuth' : 'Unknown', }

IPSEC_PRIORITY_LIST = [ 'passThrough',
                       'regular',
                       'trap',
                       'fallback',
                       'unknownPriority', ]
IPSEC_PRIORITY_MAP = { 0 : 'Passthrough policies',
                       1 : 'Regular IPsec policies',
                       2 : 'Trap policies',
                       3 : 'Fallback drop policies',
                       -1 : 'Unknown Priority', }

IPSEC_XFRMPOLICYFLAGS_LIST = [ 'unknownPolicy', 'allowOverride',
                               'includeIcmp',
                               'unknownPolicy', ]
IPSEC_XFRMPOLICYFLAGS_MAP = { 1 : 'Allow policy override',
                              2 : 'Include ICMP payloads',
                              -1 : 'Unknown', }

IPSEC_DHGROUP_MAP = {
      IpsecDhGroupEnum.modp768 : '768 bit',
      IpsecDhGroupEnum.modp1024 : '1024 bit',
      IpsecDhGroupEnum.modp1536 : '1536 bit',
      IpsecDhGroupEnum.modp2048 : '2k bit',
      IpsecDhGroupEnum.modp3072 : '3072 bit',
      IpsecDhGroupEnum.modp4096 : '4k bit',
      IpsecDhGroupEnum.modp6144 : '6144 bit',
      IpsecDhGroupEnum.ecp256 : '256 bit ecp',
      IpsecDhGroupEnum.ecp384 : '384 bit ecp',
      IpsecDhGroupEnum.ecp521 : '521 bit ecp',
      IpsecDhGroupEnum.modp2048s256 : '2k bit, 256 bit subgroup',
      }

IPSEC_KEY_CONTROLLER_PEER_STATUS = {
      IpsecPeerConnStatusEnum.peerUp: 'up',
      IpsecPeerDownReasonEnum.dhGroupMismatch : 'down (DH group mismtach)',
      IpsecPeerDownReasonEnum.dhLenMismatch : 'down (DH length mismtach)',
      IpsecPeerDownReasonEnum.nonceLenMismatch : 'down (nonce Length mismtach)',
      IpsecPeerDownReasonEnum.encryptAlgoMismatch : 
            'down (encrypt algorithm mismtach)',
      IpsecPeerDownReasonEnum.hashAlgoMismatch : 'down (hash algorithm mismtach)',
      IpsecPeerDownReasonEnum.cryptoKeyGenerationErr :
            'down (crypto key generation error)',
      IpsecPeerDownReasonEnum.licenseDisabled :
            'down (ipsec license disabled)',
      'unknownPeerState': 'unknown',
      }

IPSEC_DHGROUP_NUMBER_MAP = { 
      IpsecDhGroupEnum.modp768 : '1', 
      IpsecDhGroupEnum.modp1024 : '2', 
      IpsecDhGroupEnum.modp1536 : '5', 
      IpsecDhGroupEnum.modp2048 : '14', 
      IpsecDhGroupEnum.modp3072 : '15', 
      IpsecDhGroupEnum.modp4096 : '16', 
      IpsecDhGroupEnum.modp6144 : '17', 
      IpsecDhGroupEnum.ecp256 : '19',
      IpsecDhGroupEnum.ecp384 : '20', 
      IpsecDhGroupEnum.ecp521 : '21', 
      IpsecDhGroupEnum.modp2048s256 : '24', 
      'unknownDhGroup' : 'unknown',
      }

IPSEC_LIFETIME_UNIT_LIST = [
      IpsecLifetimeUnitEnum.hours,
      IpsecLifetimeUnitEnum.minutes
      ]
IPSEC_CONNIDSRC_LIST = [
   'unused', 'tunnel', 'dps', 'policy'
]

DhRekeyStateShowMap = {
      'notRekeying': 'not rekeying',
      'initiated': 'initiated',
      'inSaInstalled': 'ingress SA installed',
      'inProgress': 'in progress',
      'peerOutSaInstalled': 'peer out SA installed',
      'invalid': 'invalid'
      }

SmStateShowMap = {
      'notRekeying': 'not rekeying',
      'dhRekeyInProgress': 'DH rekey in progress',
      'initiator': 'initator',
      'responder': 'responder',
      'simultaneous': 'simultaneous',
      'invalid': 'invalid'
      }

SaRekeyStateShowMap = {
      'initial': 'initial',
      'created': 'created',
      'installed': 'installed',
      'peerOutSaInstalled': 'peer out SA installed',
      'invalid': 'invalid'
      }

DynamicPathStateShowMap = {
      'initial': 'initial',
      'connected': 'connected',
      'established': 'established',
      'active': 'active',
      # Initiator DH Rekey scenarios
      'dhInitiatorRekeyInitiated': 'DH initiator rekey initiated',
      'dhInitiatorInSaInstalled': 'DH initiator in SA installed',
      'dhInitiatorInSaPktsReceived': 'DH initiator in SA packets received',
      'dhInitiatorOutSaInstalled': 'DH initiator out SA installed',
      'dhInitiatorRekeyComplete': 'DH initiator rekey complete',
      # Responder DH Rekey scenarios
      'dhResponderRekeyInitiated': 'DH responder rekey initiated',
      'dhResponderInSaInstalled': 'DH responder in SA installed',
      'dhResponderOutSaInstalled': 'DH responder out SA installed',
      'dhResponderRekeyComplete': 'DH responder rekey complete',
      # Simulataneous DH Rekey scenarios
      'dhInitiatorSimultaneous': 'DH initiator simultaneous',
      'dhResponderSimultaneous': 'DH responder simultaneous',
      'dhTransientOutSaInstalled': 'DH transient out sa installed',
      'dhInitiatorPktsRxSimultaneous': 'DH simultaneous after packets received',
      # Initiator SA Rekey scenarios
      'saInitiatorNonceRequestSent': 'SA initiator nonce request sent',
      'saInitiatorNonceRequestReceived': 'SA initiator nonce request received',
      'saInitiatorNonceResponseReceived': 'SA initiator nonce response received',
      'saInitiatorInSaInstalled': 'SA initiator in SA installed',
      'saInitiatorOutSaInstalled': 'SA initiator out SA installed',
      'saInitiatorRekeyComplete': 'SA initiator rekey complete',
      # Responder SA Rekey scenarios
      'saResponderNonceRequestReceived': 'SA responder nonce request received',
      'saResponderNonceResponseSent': 'SA responder nonce response sent',
      'saResponderInSaInstalled': 'SA responder in SA installed',
      'saResponderInSaPktsReceived': 'SA responder in SA packets received',
      'saResponderOutSaInstalled': 'SA responder out SA installed',
      'saResponderRekeyComplete': 'SA responder rekey complete',
      # Simulataneous SA Rekey scenarios
      'saInitiatorSimultaneous': 'SA initiator simultaneous',
      'saResponderSimultaneous': 'SA responder simultaneous',
      'invalid': 'invalid'
      }

SaTypeShowMap = {
      'unknownSa': 'unknown',
      'inSa': 'ingress',
      'outSa': 'egress',
      'inOutSa': 'invalid',
      'newSaInSa': 'new SA ingress',
      'newSaOutSa': 'new SA egress',
      'newDhInSa': 'new DH ingress',
      'newDhOutSa': 'new DH egress',
      'oldInSa': 'old ingress',
      'oldOutSa': 'old egress'
      }

class IpsecShowFormatStrings:
   appliedProfileFormatStr = '%-28s %s'
   profileFormatStr = '%-28s %-28s %-28s %-40s'
   connStateFormatStr = '%-16s %-16s %-16s %-12s %-10s %-16s %-16s %-12s'
   connStateFormatVrfStr = '%-16s %-16s %-16s %-12s %-10s %-16s %-16s %-12s %20s'
   ikePolicyFormatStr = '%-28s %-16s %-16s %-16s %-13s %-6s %-28s'
   saFormatStr = '%-28s %-16s %-16s %-13s %-28s'
   ikePolicyFormatStrCryptoEn = '%-28s %-16s %-16s %-16s %-16s %-13s %-6s %-28s'
   saFormatStrCryptoEn = '%-28s %-16s %-16s %-16s %-13s %-28s'
   peerFormatStr = '%-19s %-11s %-32s %-9s'

# Get the Ipsec up/rekey time in secs/minutes/hours
def ipsecTimeString( elapsed, output, units ):
   if elapsed > 0 and units:
      localName, localUnits = units[ 0 ]
      if len( units ) == 1:
         # Last unit, skip the devision
         value = elapsed
      else:
         value = elapsed % localUnits
      if value > 0:
         if value == 1:
            localName = localName[ : -1 ]
         result = [ '%d %s' % ( value, localName ) ]
         if output and output != '0 seconds':
            result.append( output )
         output = ', '.join( result )
      return ipsecTimeString( elapsed // localUnits,
             output, units[ 1 : ] )
   else:
      return output

class IpsecProfile( Model ):
   ''' CLI Model for Ipsec's Profile information. '''
   policyName = Str( help="Unique identifier of IKE Policy" )
   securityAssociation = Str( help="Transformset associated with profile", 
                           optional=True )
   pkiProfile = Str( help="PKI Profile Information", optional=True )
   pkiProfileStatus = Enum( values=( 'valid', 'notPresent', 'invalid' ),
                            help="PKI Profile Status", optional=True )

   def fillIpsecProfileInfo( self, ipsecProfile, sslStatus ):
      self.policyName = ipsecProfile.ikePolicyName
      self.securityAssociation = ipsecProfile.securityAssocName

      pkiProfile = ipsecProfile.profileParams.pkiProfile
      if pkiProfile != ipsecProfile.profileParams.pkiProfileDefault:
         self.pkiProfile = pkiProfile
         if pkiProfile in sslStatus.profileStatus:
            if ( sslStatus.profileStatus[ pkiProfile ].state !=
                 SslProfileState.valid ):
               self.pkiProfileStatus = 'invalid'
            else:
               self.pkiProfileStatus = 'valid'
         else:
            self.pkiProfileStatus = 'notPresent'

   def renderProfile( self, profileName ):
      if profileName is None:
         return

      if self.policyName == '':
         self.policyName = 'None'
      if self.securityAssociation == '':
         self.securityAssociation = 'None'
      pkiProfileStr = self.pkiProfile or "-"
      if self.pkiProfileStatus and self.pkiProfileStatus != "valid":
         pkiProfileStatusMap = {
            'invalid': 'Invalid',
            'notPresent': 'Not Present'
         }
         pkiProfileStatusStr = pkiProfileStatusMap[ self.pkiProfileStatus ]
         pkiProfileStr = f'{self.pkiProfile} ({pkiProfileStatusStr})'
      print( IpsecShowFormatStrings.profileFormatStr % ( profileName,
         self.policyName, self.securityAssociation, pkiProfileStr ) )

class IpsecProfiles( Model ):
   ''' CLI Model for show ip security profile [ name ]. '''
   ipsecProfiles = Dict( help="A mapping between a IPSEC profile name "
                              "and an IPSEC profile.", keyType=str, 
                         valueType=IpsecProfile )

   def render( self ):
      if self.ipsecProfiles is None:
         return

      print( IpsecShowFormatStrings.profileFormatStr % (
         'Profile name', 'IKE Policy Name', 'SA', 'PKI Profile' ) )

      for ipsecProfileKey, ipsecProfile in sorted( self.ipsecProfiles.items() ):
         ipsecProfile.renderProfile( ipsecProfileKey )

class ShowKernelIpsecConnections( Model ):
   summary = List( valueType=str, help="Details about a connection" )

class ShowKernelIpsecChildSa( Model ):
   summary = List( valueType=str, help="Details about a childSa" )

class ShowKernelIpsecIkeSa( Model ):
   summary = List( valueType=str, help="Details about an ikeSa" )
   childSa = Dict( valueType=ShowKernelIpsecChildSa,
                   help="ChildSa's relevant to this ikeSa" )

class ShowKernelIpsecSecurityAssociations( Model ):
   ikeSa = Dict( keyType=str, valueType=ShowKernelIpsecIkeSa,
                 help="Details about each security associations ikeSa's and "
                 "related childSa's", optional=True )

class ShowKernelIpsecWorkerThreads( Model ):
   queue = Str( help="Number of threads in the queue" )
   scheduled= Int( help="Number of scheduled threads" )
   idle = Int( help="Number of idle threads" )
   working = Str( help="Number of working threads" )
   total = Int( help="Total number of threads" )

class ShowKernelIpsecStrongSwan( Model ):
   version = Str( help="StrongSwan version number" )
   architecture = Str( help="Architecture running on" )
   linux = Str( help="Linux kernel version" )
   plugins = List( valueType=str, help="Plugins enabled", optional=True )
   malloc = Dict( valueType=int, help="Details of memory" )
   workerThreads = Submodel( valueType=ShowKernelIpsecWorkerThreads,
                             help="Details of all worker threads" )
   startTime = Float( help="Start time in UTC" )
   uptime = Int( help="Uptime in seconds" )

class ShowKernelIpsecModel( Model ):
   strongSwan = Submodel( valueType=ShowKernelIpsecStrongSwan,
                          help="Details about IKE charon daemon", optional=True )
   listeningIpAddresses = List( valueType=IpGenericAddress, help="IPsec Listening "
                                "IP Addresses", optional=True )
   connections = Dict( valueType=ShowKernelIpsecConnections,
                       help="IPsec Connections", optional=True )
   securityAssociations= Dict( valueType=ShowKernelIpsecSecurityAssociations,
                               help="IPsec Security Associations", optional=True )
   dataPath = Dict( keyType=str, valueType=int,
                    help="Details of /proc/net/xfrm_stat", optional=True )
   _showIpsec = List( valueType=str, help="Show Kernel IPsec command output to keep "
                      "CLI output the same", optional=True )
   _dataPath = Dict( keyType=str, valueType=str, help="Show Kernel IPsec Datapath "
                     "command output to keep CLI output the same", optional=True )
   _isDatapath = Bool( help="CLI command contained datapath" )

   def setDatapath( self, isDatapath ):
      self._isDatapath = isDatapath

   def appendShowIpsec( self, info ):
      self._showIpsec.append( info )

   def addToDataPath( self, key, value ):
      self._dataPath[ key ] = value

   def render( self ):
      if self._isDatapath:
         if self._showIpsec:
            # For datapath output, if this _showIpsec is not empty,
            # some error occurred. Print it out
            for k in self._showIpsec:
               print( k )
         else:
            for k in sorted( self._dataPath ):
               print( f"{k}\t{self._dataPath[ k ]}" )
      else:
         for k in self._showIpsec:
            print( k )

class IpsecDaemonLog( Model ):
   daemonLog = List( valueType=str, help="Log in charon.log", optional=True )

   def render( self ):
      for k in self.daemonLog:
         print( k )

class IpsecSecurityAssociation( Model ):
   ''' CLI Model for Ipsec Security Association information '''
   espAes = Enum( values=IPSEC_ENCRYPTION_MAP,
                     help='ESP Advanced Encryption Standard Algorithm' )
   espSha = Enum( values=IPSEC_INTEGRITY_MAP,
                     help="ESP Secure Hash Algorithm" )
   pfsGroup = Enum( values=IPSEC_DHGROUP_MAP,
                     help="PFS Group to force a new DH key exchange" )
   saLifetime = Int( help="Lifetime for the SA" )
   saLifetimeUnit = Enum( values=IPSEC_LIFETIME_UNIT_LIST,
         help="Time unit for configuring lifetime for the SA" )
   replayWindowSize = Int( help="Anti-Replay Window size to detect replay attack" )
   packetLimit = Int( help="Packet count limit for the SA" )
   byteLimit = Int( help="Byte count limit for the SA" )
   if Toggles.IpsecToggleLib.toggleMultiCryptoEnabled():
      cryptoSuites = List( valueType=str, help="Crypto suites used in SA" )
  
   def fillIpsecSecurityAssociationInfo( self, ipsecSecurityAssociation, ikeConfig ):
      saParams = ipsecSecurityAssociation.saParams
      if saParams.espSha in IPSEC_INTEGRITY_MAP:
         self.espSha = saParams.espSha
      else:
         self.espSha = 'unknownIntegrity'
      if saParams.espAes in IPSEC_ENCRYPTION_MAP:
         self.espAes = saParams.espAes
      else:
         self.espAes = 'unknownEncr'
      if saParams.pfsGroup in IPSEC_DHGROUP_MAP:
         self.pfsGroup = saParams.pfsGroup
      else:
         self.pfsGroup = IpsecDhGroupEnum.modp2048
      self.saLifetime = saParams.saLifetime
      self.saLifetimeUnit = saParams.saLifetimeUnit
      self.replayWindowSize = saParams.replayWindowSize
      self.packetLimit = saParams.packetLimit
      self.byteLimit = saParams.byteLimit
      if Toggles.IpsecToggleLib.toggleMultiCryptoEnabled():
         for suite in ipsecSecurityAssociation.saCryptoSuite.values():
            self.cryptoSuites.append( suite.cryptoSuiteName )
         if self.cryptoSuites:
            for firstSuiteName in ipsecSecurityAssociation.saCryptoSuite.values():
               firstSuiteConfig = \
                  ikeConfig.saCryptoSuiteConfig[ firstSuiteName.cryptoSuiteName ]
               self.espAes = firstSuiteConfig.cryptoSuite.encryption
               self.espSha = firstSuiteConfig.cryptoSuite.integrity
               if firstSuiteConfig.cryptoSuite.dhGroup in IPSEC_DHGROUP_MAP:
                  self.pfsGroup = firstSuiteConfig.cryptoSuite.dhGroup
               break

   def renderSA( self, saKey ):
      factor = 60 if self.saLifetimeUnit == IpsecLifetimeUnitEnum.minutes else 3600
      saLifetime = f"{ (self.saLifetime / factor) } { self.saLifetimeUnit }"
      if Toggles.IpsecToggleLib.toggleMultiCryptoEnabled():
         if self.cryptoSuites:
            print( IpsecShowFormatStrings.saFormatStrCryptoEn % (
               saKey, self.cryptoSuites[ 0 ],
               'n/a', 'n/a', saLifetime, 'n/a' ) )
            for suiteName in self.cryptoSuites[ 1 : ]:
               print( IpsecShowFormatStrings.saFormatStrCryptoEn % (
                  '', suiteName, '', '', '', '' ) )
         else:
            print( IpsecShowFormatStrings.saFormatStrCryptoEn % ( saKey, '-',
               IPSEC_ENCRYPTION_MAP[ self.espAes ],
               IPSEC_INTEGRITY_MAP[ self.espSha ], saLifetime,
               IPSEC_DHGROUP_MAP[ self.pfsGroup ] ) )
      else:
         espSha = IPSEC_INTEGRITY_MAP[ self.espSha ]
         espAes = IPSEC_ENCRYPTION_MAP[ self.espAes ]
         pfsGroup = IPSEC_DHGROUP_MAP[ self.pfsGroup ]
         print( IpsecShowFormatStrings.saFormatStr % ( saKey, espAes,
               espSha, saLifetime, pfsGroup ) )

class IpsecSecurityAssociations( Model ):
   ''' CLI Model for show ip security security-association [ <saName> ] '''
   ipsecSecurityAssociations = Dict( help="A mapping between a security "
                                          "association and it's information",
                                     keyType=str,
                                     valueType=IpsecSecurityAssociation )
   def render( self ):
      if self.ipsecSecurityAssociations is None:
         return

      if Toggles.IpsecToggleLib.toggleMultiCryptoEnabled():
         print( 'INFO: n/a - Value is taken from crypto suite' )
         print( IpsecShowFormatStrings.saFormatStrCryptoEn % ( 'SA Name',
               'Crypto Suite', 'ESP Encryption', 'ESP Integrity',
               'Lifetime', 'PFS Group' ) )
      else:
         print( IpsecShowFormatStrings.saFormatStr % ( 'SA Name', 'ESP Encryption',
               'ESP Integrity', 'Lifetime', 'PFS Group' ) )
      for ipsecSaKey, ipsecSa in sorted( 
            self.ipsecSecurityAssociations.items() ):
         ipsecSa.renderSA( ipsecSaKey )

class IpsecPolicy( Model ):
   ''' CLI Model for Ipsec's ISAKMP Policy information. '''
   version = Str( help="Version of IKE Policy" )
   integrity = Enum( values=IPSEC_INTEGRITY_MAP,
                     help="Hash algorithm to verify integrity mechanism" )
   encryption = Enum( values=IPSEC_ENCRYPTION_MAP,
                     help='Encryption Algorithm used in IKE' )
   dhGroup = Enum( values=( IPSEC_DHGROUP_MAP ),
                     help="Identifier for Diffie-Hellman (DH) group" )
   ipsecAuth = Enum( values=IPSEC_AUTH_MAP,
                     help='Peer Authentication method' )
   ikeLifetime = Int( help="Lifetime for the keying channel" )
   ikeLifetimeUnit = Enum( values=IPSEC_LIFETIME_UNIT_LIST,
         help="Time unit for configuring lifetime for the keying channel" )
   rekey = Bool( help="Rekey Enabled" )
   if Toggles.IpsecToggleLib.toggleMultiCryptoEnabled():
      cryptoSuites = List( valueType=str, help="Crypto suites used in IKE" )

   def fillIpsecPolicyInfo( self, ikePolicy, ikeConfig ):
      ikeParams = ikePolicy.ikeParams
      self.version = ikeParams.version
      if ikeParams.integrity in IPSEC_INTEGRITY_MAP:
         self.integrity = ikeParams.integrity
      else:
         self.integrity = 'unknownIntegrity'
      if ikeParams.encryption in IPSEC_ENCRYPTION_MAP:
         self.encryption = ikeParams.encryption
      else:
         self.encryption = 'unknownEncr'
      if ikeParams.auth in IPSEC_AUTH_MAP:
         self.ipsecAuth = ikeParams.auth
      else:
         self.ipsecAuth = 'unknownAuth'
      self.ikeLifetime = ikeParams.ikeLifetime
      self.ikeLifetimeUnit = ikeParams.ikeLifetimeUnit
      if ikeParams.dhGroup in IPSEC_DHGROUP_MAP:
         self.dhGroup = ikeParams.dhGroup
      else:
         self.dhGroup = IpsecDhGroupEnum.modp2048
      self.rekey = ikeParams.rekey
      if Toggles.IpsecToggleLib.toggleMultiCryptoEnabled():
         for suite in ikePolicy.ikeCryptoSuite.values():
            self.cryptoSuites.append( suite.cryptoSuiteName )
         if self.cryptoSuites:
            for firstSuiteName in ikePolicy.ikeCryptoSuite.values():
               firstSuiteConfig = \
                  ikeConfig.ikeCryptoSuiteConfig[ firstSuiteName.cryptoSuiteName ]
               self.encryption = firstSuiteConfig.cryptoSuite.encryption
               self.integrity = firstSuiteConfig.cryptoSuite.integrity
               self.dhGroup = firstSuiteConfig.cryptoSuite.dhGroup
               break

   def renderPolicy( self, policy ):
      factor = 60 if self.ikeLifetimeUnit == IpsecLifetimeUnitEnum.minutes else 3600
      ikeLifetime = f"{ (self.ikeLifetime / factor) } { self.ikeLifetimeUnit }"
      if self.rekey:
         rekey = "True"
      else:
         rekey = "False"
      ipsecAuth = IPSEC_AUTH_MAP[self.ipsecAuth]
      if Toggles.IpsecToggleLib.toggleMultiCryptoEnabled():
         if self.cryptoSuites:
            print( IpsecShowFormatStrings.ikePolicyFormatStrCryptoEn % ( policy,
                  ipsecAuth, self.cryptoSuites[ 0 ], 'n/a', 'n/a',
                  ikeLifetime, rekey, 'n/a' ) )
            for suiteName in self.cryptoSuites[ 1 : ]:
               print( IpsecShowFormatStrings.ikePolicyFormatStrCryptoEn % (
                      '', '', suiteName, '', '', '', '', '' ) )
         else:
            print( IpsecShowFormatStrings.ikePolicyFormatStrCryptoEn % ( policy,
                  ipsecAuth, '-',
                  IPSEC_ENCRYPTION_MAP[ self.encryption ],
                  IPSEC_INTEGRITY_MAP[ self.integrity ],
                  ikeLifetime, rekey, IPSEC_DHGROUP_MAP[ self.dhGroup ] ) )
      else:
         encryption = IPSEC_ENCRYPTION_MAP[ self.encryption ]
         dhGroup = IPSEC_DHGROUP_MAP[ self.dhGroup ]
         integrity = IPSEC_INTEGRITY_MAP[ self.integrity ]
         print( IpsecShowFormatStrings.ikePolicyFormatStr % ( policy,
               ipsecAuth, encryption, integrity,
               ikeLifetime, rekey, dhGroup ) )

class IpsecPolicies( Model ):
   ''' CLI Model for show isakmp [ policies | policy ]. '''
   ipsecPolicies = Dict( help="A mapping between IPSEC policy name "
                              "and an IPSEC policy", 
                         keyType=str, 
                         valueType=IpsecPolicy )

   def render( self ):
      if self.ipsecPolicies is None:
         return

      if Toggles.IpsecToggleLib.toggleMultiCryptoEnabled():
         print( 'INFO: n/a - Value is taken from crypto suite' )
         print( IpsecShowFormatStrings.ikePolicyFormatStrCryptoEn % ( 'Policy Name',
               'Authentication', 'Crypto Suite', 'Encryption', 'Integrity',
               'Lifetime', 'Rekey', 'DH Group' ) )
      else:
         print( IpsecShowFormatStrings.ikePolicyFormatStr % ( 'Policy Name',
               'Authentication', 'Encryption', 'Integrity',
               'Lifetime', 'Rekey', 'DH Group' ) )
      for ipsecPolicyKey, ipsecPolicy in sorted( self.ipsecPolicies.items() ):
         ipsecPolicy.renderPolicy( ipsecPolicyKey )

class IpsecDaemonState( Model ):
   ''' CLI Model for show ip security state daemon '''
   strongSwanEnabled = Bool( help="strongSwan enabled" )
   strongSwanRunning = Bool( help="strongSwan running" )

   def fillIpsecDaemonStateInfo( self, daemonConfig, daemonStatusDir ):
      self.strongSwanEnabled = False
      self.strongSwanRunning = False
      if( daemonConfig is None or daemonStatusDir is None or
          DEFAULT_VRF not in daemonStatusDir.daemonStatus ):
         return

      self.strongSwanEnabled = daemonConfig.enableSswan
      ds = daemonStatusDir.daemonStatus[ DEFAULT_VRF ]
      self.strongSwanRunning = ds.sswanRunning

   def render( self ):
      print( "Ipsec daemon State :" )
      if self.strongSwanEnabled is True:
         print( "  Enabled : True" )
      else:
         print( "  Enabled : False" )
      if self.strongSwanRunning is True:
         print( "  state : Running" )
      else:
         print( "  state : Not Running" )

class IpsecFipsStatus( Model ):
   ''' CLI Model for show ip security fips status '''
   __revision__ = 2
   ikeFipsEnabled = Bool( help="Control plane FIPS enabled" )
   platformFipsEnabled = Bool( help="Data path FIPS enabled" )

   def fillIpsecFipsStatusInfo( self, daemonConfig, daemonStatusDir,
           ipsecPlatStatus, ikeStatus, ikeConfig ):
      self.ikeFipsEnabled = False
      self.platformFipsEnabled = False
      if ( not daemonConfig.enableSswan or ipsecPlatStatus is None or
           daemonStatusDir is None or
           DEFAULT_VRF not in daemonStatusDir.daemonStatus or
           ikeStatus is None ):
         return
      self.ikeFipsEnabled = daemonStatusDir.daemonStatus[ DEFAULT_VRF ].fipsMode
      self.platformFipsEnabled = ipsecPlatStatus.platformFipsEnabled

   def render( self ):
      op = "Supported IKE encryption algorithms: AES-128, AES-256\n"
      op += "Supported IKE authentication algorithms: SHA1, SHA256, SHA384, SHA512\n"
      op += "Supported DH groups: 14,15,16,17,20,21,24\n"
      op += "FIPS restrictions for IKE control plane: %s\n" % ( "enabled"
              if self.ikeFipsEnabled else "disabled" )
      op += "FIPS restrictions for data path: %s\n" % ( "enabled"
              if self.platformFipsEnabled else "disabled" )
      print( op )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         dictRepr[ 'fipsSelfTestStatus' ] = "unknown"
      return dictRepr

class IpsecConnectionDetail ( Model ):
   ''' CLI Model for ipsec connection info of a tunnel'''
   connectionName = Str( help="Name of the IPsec connection" )
   srcAddr = IpGenericAddress( help="Source IP Address" )
   dstAddr = IpGenericAddress( help="Destination IP Address" )
   tunnelDict = Dict ( help="Tunnel interface dict, with SA info",
                       keyType=Interface,
                       valueType=str )
   pathDict = Dict( help="Ipsec path dict", keyType=str, valueType=str )
   if togglePolicyBasedVpnEnabled():
      policyConnectionState = Dict(
              help="Dict of IPsec policy connection states", keyType=str,
              valueType=str, optional=True )
   tunnelNs = Str( help="Tunnel namespace for this profile" )
   policyTunnelName = Str( optional=True,
           help="Tunnel name for Policy Based VPN connection" )
   policyBasedVpnTunnel = Bool( optional=True,
           help="This is a tunnel used for Policy Based VPN" )
   trafficSelectorSrcPrefix = IpGenericPrefix( optional=True,
           help="Traffic selector source prefix" )
   trafficSelectorDstPrefix = IpGenericPrefix( optional=True,
           help="Traffic selector destination prefix" )

   def fillIpsecConnectionDetails( self, connName, srcAddr, dstAddr, vrfName, state,
                                   ipsecType, tunId=None, pathName=None,
                                   policyName=None ):
      self.connectionName = connName
      self.srcAddr = srcAddr
      self.dstAddr = dstAddr
      self.tunnelNs = vrfName
      self.fillIpsecConnState( state, ipsecType, tunId, pathName, policyName )

   def fillIpsecConnState( self, state, ipsecType, tunId=None, pathName=None,
                           policyName=None ):
      if state != "Shut":
         if ipsecType == "IpsecTunnel":
            self.tunnelDict[ tunId ] = state
         elif ipsecType == "IpsecPath":
            self.pathDict[ pathName ] = state
         elif ipsecType == "IpsecTunnelPolicy":
            self.policyConnectionState[ policyName ] = state

class IpsecAppliedProfile( Model ):
   ''' CLI Model for ipsec profiles that have been applied'''
   appliedIntf = List( help="Tunnel interface List for this profile",
                       valueType=Interface )
   appliedPath = List( help="Path list for this profile",
                       valueType=str )
   ipsecConnectionDetails = Dict( help="A mapping between Ipsec profile "
                                       "info and corresponding connection",
                                  keyType=str, 
                                  valueType=IpsecConnectionDetail )

   def fillIpsecPolicyConnectionInfo( self, ipsecTunnelInfo ):
      for connectionId, connectionInfo in \
              ipsecTunnelInfo.tunnelConnectionInfo.items():
         connName = connectionInfo.connectionName
         connectionModel = IpsecConnectionDetail()
         policyName = ipsecTunnelInfo.tunId + '-conn' + str( connectionId )
         connectionModel.fillIpsecConnectionDetails(
                         connName, ipsecTunnelInfo.srcAddr,
                         ipsecTunnelInfo.dstAddr, ipsecTunnelInfo.vrfName,
                         connectionInfo.connectionState, "IpsecTunnelPolicy",
                         policyName=policyName )
         connectionModel.policyTunnelName = ipsecTunnelInfo.tunId
         connectionModel.trafficSelectorSrcPrefix = \
                         connectionInfo.selector.srcPrefix
         connectionModel.trafficSelectorDstPrefix = \
                         connectionInfo.selector.dstPrefix
         self.ipsecConnectionDetails[ connName ] = connectionModel

   def fillIpsecAppliedProfileInfo( self, ikeStatus, appliedProfile,
                                    vrfName, ipsecPathStatus, tunnelIntfStatusDir ):
      if vrfName == 'all':
         vrfList = list( ikeStatus.vrfStatus.keys() )
      else:
         vrfList = [ vrfName ]

      for intfId in appliedProfile.appliedIntfId:
         tunnelIntfStatus = tunnelIntfStatusDir.intfStatus.get( intfId )
         ipsecTunnelInfo = appliedProfile.appliedIntfId.get( intfId )
         if not all( ( ipsecTunnelInfo, tunnelIntfStatus ) ):
            t9( "Key missing for intfId: ", intfId )
            continue
         if ipsecTunnelInfo.vrfName in vrfList:
            self.appliedIntf.append( tunnelIntfStatus.intfId )
            t9( "self.appliedIntf.append ", intfId )
            connName = ipsecTunnelInfo.connectionName
            ipsecConnectionDetail = self.ipsecConnectionDetails.get( connName )
            if not ipsecConnectionDetail:
               connectionModel = IpsecConnectionDetail()
               connectionModel.fillIpsecConnectionDetails( 
                                                   connName,
                                                   ipsecTunnelInfo.srcAddr,
                                                   ipsecTunnelInfo.dstAddr,
                                                   ipsecTunnelInfo.vrfName,
                                                   ipsecTunnelInfo.connectionState,
                                                   "IpsecTunnel", tunId=intfId )
               self.ipsecConnectionDetails[connName] = connectionModel
            elif intfId not in ipsecConnectionDetail.tunnelDict:
               ipsecConnectionDetail.fillIpsecConnState( 
                                                ipsecTunnelInfo.connectionState,
                                                "IpsecTunnel", tunId=intfId )
            if intfId in ikeStatus.policyTunnelsByIntfId and \
               togglePolicyBasedVpnEnabled():
               ipsecConnectionDetail = self.ipsecConnectionDetails[ connName ]
               ipsecConnectionDetail.policyBasedVpnTunnel = True
               self.fillIpsecPolicyConnectionInfo( ipsecTunnelInfo )
         else:
            t9( "not self.appliedIntf.append ", intfId )
            continue
      for connectionKey in appliedProfile.appliedPath:
         pathStatus = ipsecPathStatus.ipsecPathStatus.get( connectionKey )
         if not pathStatus:
            continue
         # BUG690031 clean this up once we add the VRF support
         assert connectionKey.vrfId == DEFAULT_VRF_ID
         if pathStatus.vrfName == vrfName:
            assert pathStatus.vrfName == DEFAULT_VRF
            self.appliedPath.append( pathStatus.pathName )
            connName = pathStatus.connectionName
            pathName = pathStatus.pathName
            ipsecConnectionDetail = self.ipsecConnectionDetails.get( connName )
            if not ipsecConnectionDetail:
               connectionModel = IpsecConnectionDetail()
               connectionModel.fillIpsecConnectionDetails( 
                                                   connName,
                                                   connectionKey.srcAddr,
                                                   connectionKey.dstAddr,
                                                   # BUG690031
                                                   pathStatus.vrfName,
                                                   pathStatus.connectionState,
                                                   "IpsecPath", pathName=pathName )
               self.ipsecConnectionDetails[connName] = connectionModel
            elif pathName not in ipsecConnectionDetail.pathDict:
               ipsecConnectionDetail.fillIpsecConnState( pathStatus.connectionState,
                                                         "IpsecPath", 
                                                         pathName=pathName )

   def renderProfile( self, profileName ):
      appliedConnList = ''
      for path in Arnet.sortIntf( self.appliedPath ):
         if appliedConnList:
            appliedConnList = appliedConnList + ",\n%-28s %s" % ( '', path )
         else:
            appliedConnList = "%s" % ( path )
      for intfId in Arnet.sortIntf( self.appliedIntf ):
         if appliedConnList:
            appliedConnList = appliedConnList + ",\n%-28s %s" % ( '', intfId )
         else:
            appliedConnList = "%s" % ( intfId )
      if appliedConnList != '':
         print( IpsecShowFormatStrings.appliedProfileFormatStr % (
            profileName, appliedConnList ) )

class IpsecAppliedProfiles( Model ):
   ''' CLI Model for show ip security profile [ profileName ]. '''
   ipsecAppliedProfiles = Dict( help="A mapping between Ipsec profile "
                                     "name and profile applied to an"
                                     "interface", 
                                keyType=str, 
                                valueType=IpsecAppliedProfile )

   def render( self ):
      if self.ipsecAppliedProfiles is None:
         return

      print( IpsecShowFormatStrings.appliedProfileFormatStr % (
         'Profile Name', 'Interface' ) )
      for appProfileKey, appProfile in sorted( 
            self.ipsecAppliedProfiles.items() ):
         appProfile.renderProfile( appProfileKey )

class LifetimeCfg( Model ):
   ''' CLI Model for Lifetime interval Config in bytes or time '''

   xfrmPolicyByteLimitWarning = Int( 
         help="Warning about impending expiry of byte transfer limit" )
   xfrmPolicyByteLimitExpiry = Int( help="Expiry of byte transfer limit" )
   xfrmPolicyPacketLimitWarning = Int(
         help="Warning about impending expiry of packet transfer limit" )
   xfrmPolicyPacketLimitExpiry = Int(
         help="Expiry of packet transfer limit" )
   xfrmPolicyTimeLimitWarning = Int(
         help="Warning about impending expiry of time limit (in secs)" )
   xfrmPolicyTimeLimitExpiry = Int(
         help="Expiry of time limit (in secs)" )
   xfrmPolicyUseTimeWarning = Int(
         help="Warning about impending expiry of use time" )
   xfrmPolicyUseTimeExpiry = Int( help="Expiry of use time" )

   def fillLifetimeCfgInfo( self, ikeSALifetimeCfgStatus=None ):
      if ikeSALifetimeCfgStatus is not None:
         self.xfrmPolicyByteLimitWarning = ikeSALifetimeCfgStatus.softByteLimit
         self.xfrmPolicyByteLimitExpiry = ikeSALifetimeCfgStatus.hardByteLimit
         self.xfrmPolicyPacketLimitWarning = ikeSALifetimeCfgStatus.softPacketLimit
         self.xfrmPolicyPacketLimitExpiry = ikeSALifetimeCfgStatus.hardPacketLimit
         self.xfrmPolicyTimeLimitWarning = ikeSALifetimeCfgStatus.softAddExpire
         self.xfrmPolicyTimeLimitExpiry = ikeSALifetimeCfgStatus.hardAddExpire
         self.xfrmPolicyUseTimeWarning = ikeSALifetimeCfgStatus.softUseExpire
         self.xfrmPolicyUseTimeExpiry = ikeSALifetimeCfgStatus.hardUseExpire
      else:
         return


   def render( self ):
      print( "Lifetime interval Config for XFRM Policy:" )
      print( "  Xfrm Policy Byte Limit Warning: %d" %
             self.xfrmPolicyByteLimitWarning )
      print( "  Xfrm Policy Byte Limit Expiry: %d" %
             self.xfrmPolicyByteLimitExpiry )
      print( "  Xfrm Policy Packet Limit Warning: %d" %
             self.xfrmPolicyPacketLimitWarning )
      print( "  Xfrm Policy Packet Limit Expiry: %d" %
             self.xfrmPolicyPacketLimitExpiry )
      print( "  Xfrm Policy Time Limit Warning: %d" %
             self.xfrmPolicyTimeLimitWarning )
      print( "  Xfrm Policy Time Limit Expiry: %d" %
             self.xfrmPolicyTimeLimitExpiry )
      print( "  Xfrm Policy Use Time Warning: %d" %
             self.xfrmPolicyUseTimeWarning )
      print( "  Xfrm Policy Use Time Expiry: %d" %
             self.xfrmPolicyUseTimeExpiry )

class LifetimeCur( Model ):
   ''' CLI Model for Current Status of Policy in context of Lifetime'''

   bytesProcessed = Int( help="Number of bytes processed in tx & rx" )
   packetsProcessed = Int( help="Number of pkts processed in tx & rx" )
   policyAddTime = Int( help="Timestamp when the policy was added" )
   policyLastAccessTime = Int( help="Timestamp when policy was last accessed" )

   def fillLifetimeCurInfo( self, entMan, ikeSALifetimeCurStatus=None ):
      if ikeSALifetimeCurStatus is None:
         return
      self.bytesProcessed = ikeSALifetimeCurStatus.bytes
      self.packetsProcessed = ikeSALifetimeCurStatus.pkts
      self.policyAddTime = ikeSALifetimeCurStatus.addTime
      self.policyLastAccessTime = ikeSALifetimeCurStatus.useTime
      if self.policyLastAccessTime:
         # If the value is not zero, convert the use time to an utc time by
         # adding the difference between Tac utcNow and now
         self.policyLastAccessTime = int( self.policyLastAccessTime +
                                          Tac.utcNow() - Tac.now() )

   def render( self ):
      print( "Policy current status in context of Lifetime :" )
      print( "  Bytes: %d" % self.bytesProcessed )
      print( "  Packets: %d" % self.packetsProcessed )
      print( "  Add time: %d" % self.policyAddTime )
      print( "  Last used time: %d" % self.policyLastAccessTime )

class SecurityAssociation ( Model ):
   ''' CLI Model for show ip security state ike sa '''
   saddr = IpGenericAddress( help="Source IP Address" )
   daddr = IpGenericAddress( help="Dest IP Address" )
   spi = Int( help="Security Policy Index for ESP/AH" )
   proto = Enum( values=( IPSEC_PROTO_LIST ), 
                 help="IP Transform Protocol (AH/ESP)" )
   mode = Enum( values=( IPSEC_MODE_LIST ),
                help="mode of SA (tunnel, transport, beet, pass etc)" )
   reqid = Int( help="Request Id for a given SA pair" )
   replayWindow = Int( help="Sequence number window for replay protection" )
   seq = Int( help="Sequence Number to help with replay protection" )
   family = Enum( values=( 'ipv4', 'ipv6', 'unknownFamily' ), 
                  help="v4 or v6 family" )
   packetsOutOfReplayWindow = Int( help="Number of packets outside Replay Window" )
   replayErrors = Int( help="Number of Replay Errors" )
   statsIntegrityFailed = Int( help="Number of Integrity check failures for "
                                    "incoming packets" )
   lifetimeCurrent = Submodel( valueType=LifetimeCur, 
                               help="Lifetime interval current status in "
                                    "bytes/time:" )
   lifetimeCfg  = Submodel( valueType=LifetimeCfg, 
                            help="Lifetime interval config in bytes/time:" )
   connId = Int( help="Connection Id of SA", default=0 )

   def fillSecurityAssociationInfo( self, entMan, ikeStatus, ipsecCounterTable,
                                    saKey ):
      saStatus = ikeStatus.sa.get( saKey )
      if not saStatus:
         saStatus = ikeStatus.controllerSa.get( saKey )
      if not saStatus:
         return
      counterKey = Tac.Value( "IpsecCounters::IpsecCounterKey",
                              saStatus.reqid, saKey.id.spi, saKey.id.daddr )
      saCounterEntry = ipsecCounterTable.ipsecSACounterEntry.get( counterKey )
      lifetimeCfg = ikeStatus.lifetimeCfg.get( saKey )
      if not all( ( saCounterEntry, lifetimeCfg ) ):
         return
      said = saKey.id
      self.spi = said.spi
      self.daddr = said.daddr
      if said.proto in IPSEC_PROTO_MAP:
         self.proto = IPSEC_PROTO_LIST[said.proto-50]
      else:
         self.proto = 'unknownProto'
      self.saddr = saStatus.saddr
      if saStatus.mode in IPSEC_MODE_MAP:
         self.mode = IPSEC_MODE_LIST[saStatus.mode]
      else:
         self.mode = 'unknownMode'
      self.reqid = saStatus.reqid
      # Get Connection ID of SA
      if saStatus.connId:
         self.connId = saStatus.connId.connId
      self.replayWindow = saStatus.replay_window
      self.seq = saStatus.seq
      if saKey.family == AF_INET:
         self.family = 'ipv4'
      elif saKey.family == AF_INET6:
         self.family = 'ipv6'
      else:
         self.family = 'unknownFamily'

      self.packetsOutOfReplayWindow = saCounterEntry.replayWindowErrors
      self.replayErrors = saCounterEntry.replayErrors
      self.statsIntegrityFailed = saCounterEntry.integrityErrors

      self.lifetimeCurrent = LifetimeCur()
      self.lifetimeCurrent.fillLifetimeCurInfo( entMan, saCounterEntry )

      self.lifetimeCfg = LifetimeCfg()
      self.lifetimeCfg.fillLifetimeCfgInfo( lifetimeCfg )

class PolicySelector( Model ):
   ''' Policy Selector for a Flow. '''

   daddr = IpGenericAddress( help="Dest IP Address for Policy" )
   saddr = IpGenericAddress( help="Source IP Address for Policy" )
   index = Int( help="Index of policy in the IPsec Engine" )
   direction = Enum( values=( IPSEC_DIRECTION_LIST ),
                     help="Direction- in/out/fwd" )
   family = Enum( values=( 'ipv4', 'ipv6', 'unknownFamily' ), 
                  help="v4 or v6 family" )
   proto = Enum( values=( IPSEC_PROTO_LIST ), 
                 help="IP Transform Protocol (AH/ESP)" )
 
   def fillPolicySelectorInfo( self, ikeSecurityPolicySelectorStatus=None ):
      if ikeSecurityPolicySelectorStatus is None:
         return
      self.daddr = ikeSecurityPolicySelectorStatus.daddr
      self.saddr = ikeSecurityPolicySelectorStatus.saddr
      if ikeSecurityPolicySelectorStatus.dir in IPSEC_DIRECTION_MAP:
         self.direction = IPSEC_DIRECTION_LIST[ikeSecurityPolicySelectorStatus.dir]
      else: 
         self.direction = 'unknownDir'
      if ikeSecurityPolicySelectorStatus.family == AF_INET:
         self.family = 'ipv4'
      elif ikeSecurityPolicySelectorStatus.family == AF_INET6:
         self.family = 'ipv6'
      else:
         self.family = 'unknownFamily'
      if ikeSecurityPolicySelectorStatus.proto in IPSEC_PROTO_MAP:
         self.proto = IPSEC_PROTO_LIST[ikeSecurityPolicySelectorStatus.proto-50]
      else:
         self.proto = 'unknownProto'
      self.index = ikeSecurityPolicySelectorStatus.index

   def render( self ):
      print( "  Policy Selector for the Flow." )
      print( "  Dest ip addr: %s" % self.daddr )
      print( "  Source ip addr: %s" % self.saddr )
      proto = IPSEC_DIRECTION_LIST.index(self.direction)
      print( "  Direction: %s" % IPSEC_DIRECTION_MAP[ proto ] )
      print( "  Family: %s" % self.family )
      proto = IPSEC_PROTO_LIST.index(self.proto)
      print( "  Proto: %s" % IPSEC_PROTO_MAP[ 50 + proto ] )
      print( "  Index: %d" % self.index )

class SecurityPolicy ( Model ):
   ''' CLI Model for show ip security state ike policy '''
   selector = Submodel( valueType=PolicySelector, 
                        help="Policy Selector for a Flow" )
   priority = Enum( values=( IPSEC_PRIORITY_LIST ),
                    help="SA Priority for this Policy" )
   flags = Enum( values= ( IPSEC_XFRMPOLICYFLAGS_LIST ),
                 help="Override global policy vs allow matching ICMP payloads" )

   def fillSecurityPolicyInfo( self, ikeSecurityPolicyStatus=None ):
      if ikeSecurityPolicyStatus is not None:
         self.selector = PolicySelector()
         self.selector.fillPolicySelectorInfo( ikeSecurityPolicyStatus.selector )
         if ikeSecurityPolicyStatus.priority in IPSEC_PRIORITY_MAP:
            self.priority = IPSEC_PRIORITY_LIST[ikeSecurityPolicyStatus.priority]
         else: 
            self.priority = 'unknownPriority'
         self.flags = ikeSecurityPolicyStatus.flags
         if ikeSecurityPolicyStatus.flags in IPSEC_XFRMPOLICYFLAGS_MAP:
            self.flags = IPSEC_XFRMPOLICYFLAGS_LIST[ikeSecurityPolicyStatus.flags]
         else: 
            self.flags = 'unknownPolicy'
      else:
         return


   def renderSP( self, name ):
      print( "Ipsec Security Policy State :" )
      print( "  Name: %s" % name )
      flags = IPSEC_XFRMPOLICYFLAGS_LIST.index(self.flags)
      print( "  Flags: %s" % IPSEC_XFRMPOLICYFLAGS_MAP[ flags ] )
      priority = IPSEC_PRIORITY_LIST.index(self.priority)
      print( "  Priority: %s" % IPSEC_PRIORITY_MAP[ priority ] )
      if self.selector is not None:
         self.selector.render( )

def getIpsecConnList( ikeStatus=None, tunnelIntfStatusDir=None,
                      ipsecPathStatus=None, vrfName=None, ):
   connList = []

   def fillConnectionList( profileName ):
      assert ikeStatus
      profile = profiles.get( profileName )
      if not profile:
         return
      profileModel = IpsecAppliedProfile()
      profileModel.fillIpsecAppliedProfileInfo( ikeStatus, profile,
                                                vrfName, ipsecPathStatus,
                                                tunnelIntfStatusDir )
      for connName in profileModel.ipsecConnectionDetails:
         t9( "fillConnectionList for connEntry: ", connName )
         connEntry = profileModel.ipsecConnectionDetails[connName]
         connList.append( connEntry )

   if ikeStatus is not None:
      profiles = ikeStatus.appliedProfiles
      for profileName in profiles:
         t9( "fillTunnelList for", profileName )
         fillConnectionList( profileName )

   return connList

class IpsecDynamicDhStateM( Model ):
   ''' CLI Model for Dh State in dynamic paths'''
   localRekey = Int( help="Local rekey count" )
   peerRekey = Int( help="Peer rekey count" )
   dhRekeyState = Enum( help="Current DH rekey state",
                values=( "notRekeying", "initiated", "inSaInstalled",
                         "inProgress", "peerOutSaInstalled",
                         "invalid" ) )
   dhSmState = Enum( help="Current DH state machine state",
                values=( "notRekeying", "dhRekeyInProgress",
                         "initiator", "responder",
                         "simultaneous", "invalid" ) )

class IpsecDynamicSpiInfoM( Model ):
   ''' CLI Model for SPI values for dynamic paths'''
   localRekey = Int( help="Local rekey count" )
   peerRekey = Int( help="Peer rekey count" )
   inSpi = Int( help="Inbound SPI value" )
   outSpi = Int( help="Outbound SPI value" )

class IpsecDynamicSaInfoM( Model ):
   ''' CLI Model for SA Info in dynamic paths'''
   saType = Enum( help="SA type",
              values=( "unknownSa", "inSa", "outSa", "inOutSa",
                       "newSaInSa", "newSaOutSa", "newDhInSa",
                       "newDhOutSa", "oldInSa", "oldOutSa" ) )
   spi = Int( help="SPI value" )
   initiator = Bool( help="This peer is an initiator" )
   saRekeyState = Enum( help="SA rekey state",
                     values=( "initial", "created", "installed",
                              "peerOutSaInstalled", "invalid" ) )
   peerNonceType = Enum( help="Peer nonce type",
                      values=( "init", "request", "response", "invalid" ) )
   addTime = Float( help="Time when the SA was programmed" )
   pkts = Int( help="Packets seen on this path" )
   pairSpi = Int( help="Pair SPI" )

class IpsecDynamicPathInfoM( Model ):
   ''' CLI Model for dynamic part of show ip security connection '''
   saSmState = Enum( help="Current SA state",
         values=( "notRekeying", "dhRekeyInProgress", "initiator",
                  "responder", "simultaneous", "invalid" ) )
   pathState = Enum( help="Dynamic path state",
                values=( "initial", "connected", "established", "active",
                   "dhInitiatorRekeyInitiated", "dhInitiatorInSaInstalled",
                   "dhInitiatorInSaPktsReceived", "dhInitiatorOutSaInstalled",
                   "dhInitiatorRekeyComplete",
                   "dhResponderRekeyInitiated", "dhResponderInSaInstalled",
                   "dhResponderOutSaInstalled", "dhResponderRekeyComplete",
                   "dhInitiatorSimultaneous", "dhResponderSimultaneous",
                   "dhTransientOutSaInstalled", "saInitiatorNonceRequestSent",
                   "dhInitiatorPktsRxSimultaneous",
                   "saInitiatorNonceRequestReceived",
                   "saInitiatorNonceResponseReceived",
                   "saInitiatorInSaInstalled", "saInitiatorOutSaInstalled",
                   "saInitiatorRekeyComplete", "saResponderNonceRequestReceived",
                   "saResponderNonceResponseSent", "saResponderInSaInstalled",
                   "saResponderInSaPktsReceived", "saResponderOutSaInstalled",
                   "saResponderRekeyComplete", "saInitiatorSimultaneous",
                   "saResponderSimultaneous", "invalid" ) )
   lastEstablishedTime = Float( help="Time when session was last established" )
   lastKnownLocalRekeyCount = Int( help="Last rekey count" )
   sessionPktsOut = Int( help="Session output packets", default=0 )
   sessionPktsIn = Int( help="Session input packets", default=0 )
   dhStateInfo = List( help="DH state entries", valueType=IpsecDynamicDhStateM )
   dhCryptoInfo = List( help="DH crypto entries", valueType=IpsecDynamicSpiInfoM )
   saCryptoInfo = List( help="SA crypto entries", valueType=IpsecDynamicSpiInfoM )
   saInfo = List( help="SA info entries", valueType=IpsecDynamicSaInfoM )

class IpsecConnection( Model ):
   ''' CLI Model for Tunnel info scraped from strongswan statusall'''
   __revision__ = 2
   connName = Str( help="Name of the IPsec connection" )
   ipsecInputDataBytes = Int( help="Ipsec Input Data in bytes" )
   ipsecInputDataPkts = Int( help="Ipsec Input Data in packets" )
   ipsecOutputDataBytes = Int( help="Ipsec Output Data in bytes" )
   ipsecOutputDataPkts = Int( help="Ipsec Output Data in packets" )
   tunnelDict = Dict( help="Tunnel interface List for this connection",
                      keyType=Interface,
                      valueType=str, optional=True )
   pathDict = Dict( help="Path name for this connection", keyType=str,
                    valueType=str, optional=True )
   if togglePolicyBasedVpnEnabled():
      policyConnectionState = Dict(
                         help="Dict of IPsec policy connection states", keyType=str,
                         valueType=str, optional=True )
   saddr = IpGenericAddress( help="Source IP Address" )
   daddr = IpGenericAddress( help="Destination IP Address" )
   tunnelNs = Str( help="Tunnel namespace for this profile" )
   inboundSA = Submodel( valueType=SecurityAssociation, 
                                   help="SA for an inbound Flow",
                                   optional=True )
   outboundSA = Submodel( valueType=SecurityAssociation, 
                                   help="SA for an outbound Flow",
                                   optional=True )
   upTimeStr = Str( help="Ipsec connection uptime" )
   rekeyTime = Int( help="Seconds remaining until next rekey" )
   dynamicPath = Bool( help="Identifies if peer is a dynamic path" )
   remoteIp = IpGenericAddress( help="Remote IP" )
   vrfName = Str( help="VRF name" )
   initiator = Bool( help="This peer is an initiator" )
   dynPathDict = Dict( keyType=str, valueType=IpsecDynamicPathInfoM,
                       help="A mapping of paths to dynamic IPsec info" )
   policyTunnelName = Str( optional=True,
           help="Tunnel name for Policy Based VPN connection" )
   policyBasedVpnTunnel = Bool( optional=True,
           help="This is a tunnel used for Policy Based VPN" )
   trafficSelectorSrcPrefix = IpGenericPrefix( optional=True,
           help="Traffic selector source prefix" )
   trafficSelectorDstPrefix = IpGenericPrefix( optional=True,
           help="Traffic selector destination prefix" )

   def initConnection( self, connName ):
      self.connName = connName
      self.ipsecInputDataBytes = 0
      self.ipsecInputDataPkts = 0
      self.ipsecOutputDataBytes = 0
      self.ipsecOutputDataPkts = 0
      self.inboundSA = None
      self.outboundSA = None
      self.upTimeStr = "N/A"
      self.rekeyTime = 0

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         dictRepr[ 'ipsecIkeState' ] = 'unknownState'
         dictRepr[ 'ikev1StateTime' ] = 0
         dictRepr[ 'ikev2StateTime' ] = 0
         dictRepr[ 'ikev1StateTimeUnits' ] = 'hours'
         dictRepr[ 'ikev2StateTimeUnits' ] = 'hours'
      return dictRepr

class IpsecConnectionState( Model ):
   ''' CLI Model for show ip security connection '''
   __revision__ = 2
   _detailed = Bool( help="Display detailed output" )
   connections = Dict( help="A mapping between tunnel name and"
                       "IPsec tunnel",
                       keyType=str,
                       valueType=IpsecConnection )

   def getIpsecSessionStats( self, ikeStatus, reqid, ipsecCounterTable, inSa=True ):
      numBytes = 0
      numPkts = 0
      ipsecUptime = 0

      sessionKey = Tac.Value( "IpsecCounters::IpsecSessionKey", reqid )
      ipsecInfo = ipsecCounterTable.ipsecSessionCounterEntry.get( sessionKey )
      if not ipsecInfo:
         return numBytes, numPkts, ipsecUptime

      if inSa:
         numBytes = ipsecInfo.bytesIn
         numPkts = ipsecInfo.pktsIn
      else:
         numBytes = ipsecInfo.bytesOut
         numPkts = ipsecInfo.pktsOut
      ipsecUptime = ipsecInfo.lastUpTime

      return numBytes, numPkts, ipsecUptime

   def getIpsecConnEntry( self, ipsecConnTable, vrfName, reqid ):
      connVrfStatus = ipsecConnTable.connVrfStatus.get( vrfName )
      if connVrfStatus is not None:
         return connVrfStatus.connEntries.get( reqid )
      return None

   def enumToStr( self, state, prefix ):
      fullstr = str( state )
      prefixLen = len( prefix )
      if prefixLen < len( fullstr ):
         fullstr = fullstr[ prefixLen : ]
      res = fullstr[ 0 ].lower() + fullstr[ 1 : ]
      return res

   def getDynamicAndCommonPathInfo( self, connection, ikeStatus, ipsecPathStatus,
                                ipsecCntrTbl ):
      if connection is None or ikeStatus is None or ipsecPathStatus is None \
            or ipsecCntrTbl is None:
         return

      if not connection.pathDict:
         return

      def fillDynamicDhState( dynPathInfo, connState ):
         dynPathInfo.dhStateInfo = []
         for key in connState.dhState:
            dS = connState.dhState.get( key )
            valueEntry = IpsecDynamicDhStateM()
            valueEntry.localRekey = key.localDhRekeyCount
            valueEntry.peerRekey = key.peerDhRekeyCount
            valueEntry.dhRekeyState = self.enumToStr( dS.dhRekeyState, "dhRekey" )
            valueEntry.dhSmState = self.enumToStr( dS.dhSmState, "smState" )
            dynPathInfo.dhStateInfo.append( valueEntry )

      def fillDynamicCryptoInfo( dynPathInfo, connState ):
         dynPathInfo.dhCryptoInfo = []
         for key in connState.dhCryptoInfo:
            dS = connState.dhCryptoInfo.get( key )
            cryptoInfo = dS.cryptoInfo
            if cryptoInfo:
               valueEntry = IpsecDynamicSpiInfoM()
               valueEntry.localRekey = key.localDhRekeyCount
               valueEntry.peerRekey = key.peerDhRekeyCount
               valueEntry.inSpi = socket.ntohl( cryptoInfo.spiInitiator )
               valueEntry.outSpi = socket.ntohl( cryptoInfo.spiResponder )
               dynPathInfo.dhCryptoInfo.append( valueEntry )

      def fillDynamicSaCryptoInfo( dynPathInfo, connState ):
         dynPathInfo.saCryptoInfo = []
         for key in connState.saCryptoInfo:
            dS = connState.saCryptoInfo.get( key )
            saCryptoInfo = dS.cryptoInfo
            if saCryptoInfo:
               valueEntry = IpsecDynamicSpiInfoM()
               valueEntry.localRekey = key.localDhRekeyCount
               valueEntry.peerRekey = key.peerDhRekeyCount
               valueEntry.inSpi = socket.ntohl( saCryptoInfo.spiInitiator )
               valueEntry.outSpi = socket.ntohl( saCryptoInfo.spiResponder )
               dynPathInfo.saCryptoInfo.append( valueEntry )

      def fillDynamicSaInfo( dynPathInfo, connState ):
         dynPathInfo.saInfo = []
         for key in connState.pathConnSaInfo:
            saInfo = connState.pathConnSaInfo.get( key )
            if saInfo and saInfo.sa:
               saInfoM = IpsecDynamicSaInfoM()
               saInfoM.saType = self.enumToStr( saInfo.sa.saType, "saType" )
               saInfoM.spi = key.id.spi
               saInfoM.initiator = saInfo.sa.isInitiator
               saInfoM.saRekeyState = \
                   self.enumToStr( saInfo.saRekeyState, "saRekey" )
               saInfoM.peerNonceType = \
                   self.enumToStr( saInfo.peerNonceType, "nonce" )
               saInfoM.addTime = saInfo.addTime
               saInfoM.pkts = saInfo.pkts
               saInfoM.pairSpi = socket.ntohl( saInfo.sa.pairSpi )
               dynPathInfo.saInfo.append( saInfoM )

      pathName = next( iter( connection.pathDict ) )
      for connectionKey in ipsecPathStatus.ipsecPathStatus:
         pathStatus = ipsecPathStatus.ipsecPathStatus.get( connectionKey )
         if pathStatus is None:
            continue

         if pathStatus.pathName != pathName:
            connection.dynamicPath = False
            continue

         connection.remoteIp = pathStatus.remoteIp
         peerConnState = ikeStatus.ipsecPeerConnectionState.get(
                            pathStatus.remoteIp )
         if peerConnState is None:
            continue

         connection.initiator = peerConnState.initiator
         connection.tunnelNs = peerConnState.vrfName

         pathConnState = peerConnState.pathConnState.get( connectionKey )
         if pathConnState is None:
            continue

         dynPathInfo = IpsecDynamicPathInfoM()
         dynPathInfo.saSmState = \
            self.enumToStr( pathConnState.saSmState, "smState" )
         dynPathInfo.pathState = \
            self.enumToStr( pathConnState.pathState, "ps" )
         if pathConnState.sanityCheck and dynPathInfo.pathState == 'established':
            dynPathInfo.lastEstablishedTime = \
                  pathConnState.sanityCheck.lastEstablishedTime
         else:
            dynPathInfo.lastEstablishedTime = 0.0
         dynPathInfo.lastKnownLocalRekeyCount = \
            pathConnState.lastKnownLocalRekeyCount

         reqid = pathConnState.reqid
         sessionKey = Tac.Value( "IpsecCounters::IpsecSessionKey", reqid )
         saSessionEntry = ipsecCntrTbl.ipsecSessionCounterEntry.get( sessionKey )
         if saSessionEntry:
            dynPathInfo.sessionPktsIn = saSessionEntry.pktsIn
            dynPathInfo.sessionPktsOut = saSessionEntry.pktsOut

         fillDynamicDhState( dynPathInfo, pathConnState )
         fillDynamicCryptoInfo( dynPathInfo, peerConnState )
         fillDynamicSaCryptoInfo( dynPathInfo, pathConnState )
         fillDynamicSaInfo( dynPathInfo, pathConnState )

         connection.dynamicPath = True
         connection.dynPathDict[ pathStatus.pathName ] = dynPathInfo
         break

   def fillIpsecConnectionStateInfo( self, entMan, vrfName, ikeStatus=None,
                                     ipsecConnTable=None,
                                     tunnelIntfStatusDir=None, 
                                     ipsecCounterTable=None,
                                     ipsecPathStatus=None,
                                     tunnelName=None, 
                                     pathName=None,
                                     pathPeerIp=None,
                                     pathPeerFqdn=None,
                                     policyConnectionName=None,
                                     tunnels=False,
                                     paths=False,
                                     policies=False,
                                     detailed=False ):
      # Nothing to do if ikeStatus is invalid
      if ikeStatus is None:
         return

      self._detailed = detailed
      connList = []
      vrfList = []
      connectionsPerVrf = defaultdict( list )

      if vrfName == 'all':
         for v in ikeStatus.vrfStatus:
            vrfList.append( v )
            vrfConnList = getIpsecConnList( ikeStatus, tunnelIntfStatusDir,
                                            ipsecPathStatus, v )
            connList.extend( vrfConnList )
      else:
         connList = getIpsecConnList( ikeStatus, tunnelIntfStatusDir,
                                      ipsecPathStatus, vrfName )

         vrfList.append( vrfName )

      def fillConnDetails( connEntry, tunnelName=None, pathName=None,
                           policyConnectionName=None, tunnels=False,
                           paths=False, policies=False ):
         connection = IpsecConnection()
         connection.initConnection( connName )
         connection.saddr = connEntry.srcAddr
         connection.daddr = connEntry.dstAddr
         # this is the VRF, not the namespace...
         connection.tunnelNs = connEntry.tunnelNs
         connection.dynamicPath = False
         connection.dynPathDict = {}
         connection.policyTunnelName = connEntry.policyTunnelName
         connection.policyBasedVpnTunnel = connEntry.policyBasedVpnTunnel
         connection.trafficSelectorSrcPrefix = connEntry.trafficSelectorSrcPrefix
         connection.trafficSelectorDstPrefix = connEntry.trafficSelectorDstPrefix

         # display info only for given tunnel, if specified.
         if tunnelName is not None:
            assert tunnelName in connEntry.tunnelDict
            connection.tunnelDict[ tunnelName ] = connEntry.tunnelDict[ tunnelName ]
         elif pathName is not None:
            assert pathName in connEntry.pathDict
            connection.pathDict[ pathName ] = connEntry.pathDict[ pathName ]
         elif policyConnectionName is not None:
            assert policyConnectionName in connEntry.policyConnectionState
            connection.policyConnectionState[ policyConnectionName ] = \
                    connEntry.policyConnectionState[ policyConnectionName ]
         elif tunnels:
            connection.tunnelDict = connEntry.tunnelDict
         elif paths:
            connection.pathDict = connEntry.pathDict
         elif policies:
            connection.policyConnectionState = connEntry.policyConnectionState
         else:
            connection.tunnelDict = connEntry.tunnelDict
            connection.pathDict = connEntry.pathDict
            if togglePolicyBasedVpnEnabled():
               connection.policyConnectionState = connEntry.policyConnectionState

         t9( "Created Tunnel Entry for %s, src=%s dst=%s" %
               ( connName, connection.saddr, connection.daddr ) )
         return connection

      firstInboundSaByConn = {}
      firstOutboundSaByConn = {}
      # Get IPsec connection details for all connections
      for connEntry in connList:
         connection = None
         connName = connEntry.connectionName
         firstInboundSaByConn[ connName ] = True
         firstOutboundSaByConn[ connName ] = True
         t9( "filling conn details for conn: ", connName )
         if not tunnelName and not pathName and not pathPeerIp \
                 and not policyConnectionName:
            connection = fillConnDetails( connEntry, tunnels=tunnels, paths=paths,
                                          policies=policies )
            self.getDynamicAndCommonPathInfo( connection, ikeStatus,
                                          ipsecPathStatus, ipsecCounterTable )
            self.connections[connName] = connection
            connectionsPerVrf[ connection.tunnelNs ].append( connName )
            continue
         # pylint: disable-msg=R1723
         if tunnelName in connEntry.tunnelDict:
            # user specified tunnel name. get info for only this entry
            connection = fillConnDetails( connEntry, tunnelName=tunnelName )
            self.connections[connName] = connection
            connectionsPerVrf[ connection.tunnelNs ].append( connName )
            break
         elif pathName in connEntry.pathDict:
            connection = fillConnDetails( connEntry, pathName=pathName )
            self.getDynamicAndCommonPathInfo( connection, ikeStatus,
                                          ipsecPathStatus, ipsecCounterTable )
            self.connections[ connName ] = connection
            connectionsPerVrf[ connection.tunnelNs ].append( connName )
            break
         elif togglePolicyBasedVpnEnabled() and \
              policyConnectionName in connEntry.policyConnectionState:
            connection = fillConnDetails( connEntry,
                                          policyConnectionName=policyConnectionName )
            self.connections[ connName ] = connection
            connectionsPerVrf[ connection.tunnelNs ].append( connName )
            break
         elif pathPeerIp:
            pathEntry = ipsecPathStatus.ipsecPathRemoteIp.get( pathPeerIp )
            if not pathEntry:
               return
            for path in pathEntry.paths:
               if path in connEntry.pathDict:
                  connection = fillConnDetails( connEntry, pathName=path )
                  self.getDynamicAndCommonPathInfo( connection, ikeStatus,
                                                ipsecPathStatus, ipsecCounterTable )
                  self.connections[connName] = connection
                  connectionsPerVrf[ connection.tunnelNs ].append( connName )
                  break
            continue

      t9( "vrfList is: ", vrfList )
      # pylint: disable-msg=R1702
      for vrf in vrfList:
         vrfStatus = ikeStatus.vrfStatus.get( vrf )
         if vrfStatus is None:
            t9( "vrfStatus has no vrf key named : ", vrf )
            continue
         for saKey in chain( vrfStatus.sa, vrfStatus.controllerSa ):
            sa = SecurityAssociation()
            sa.fillSecurityAssociationInfo( entMan, vrfStatus, ipsecCounterTable,
                                            saKey )
            if sa.reqid is None:
               # fillSecurityAssociationInfo did not fill the sa info ( encountered
               # an invalid case and returned before assigned the values )
               # skip finding the matching connection in this case
               continue
            srcAddr = "%s" % sa.saddr
            dstAddr = "%s" % sa.daddr
            lifetimeCur = sa.lifetimeCurrent
            ipsecConnEntry = self.getIpsecConnEntry( ipsecConnTable,
                                                     vrf, sa.reqid )
            t9( f"Find Tunnel for SA: src={srcAddr} dst={dstAddr}" )
            for connectionName in connectionsPerVrf[ vrf ]:
               connection = self.connections[connectionName]
               tunnelSaddr = "%s" % connection.saddr
               tunnelDaddr = "%s" % connection.daddr
               isConnOut = tunnelSaddr == srcAddr and tunnelDaddr == dstAddr
               isConnIn = tunnelSaddr == dstAddr and tunnelDaddr == srcAddr

               if Toggles.IpsecToggleLib.toggleIpsecPatEnabled() or \
                  togglePolicyBasedVpnEnabled():
                  # We need to take into account the connection Id also in
                  # addition to the addresses
                  parts = connectionName.split( '-' )
                  connId = 0
                  if len( parts ) == 5:
                     connId = int( parts[ -1 ] )
                  isConnOut = isConnOut and connId == sa.connId
                  isConnIn = isConnIn and connId == sa.connId

               t9( "tunnel is {}(s={}, d={}) sa={} sd={}".format( connectionName,
                     tunnelSaddr, tunnelDaddr, srcAddr, dstAddr ) )
               if isConnOut:
                  if firstOutboundSaByConn[ connectionName ]:
                     # For the first SA we also copy over the session stats
                     numBytes, numPkts, ipsecUptime = self.getIpsecSessionStats(
                                                      vrfStatus, sa.reqid, 
                                                      ipsecCounterTable, False )
                     connection.ipsecOutputDataBytes = numBytes
                     connection.ipsecOutputDataPkts = numPkts
                     connection.upTimeStr = ipsecTimeString( 
                                            int( Tac.utcNow() - ipsecUptime ),
                                            "0 seconds", [ ( "seconds", 60 ),
                                            ( "minutes", 60 ), ( "hours", 24 ),
                                            ( "days", 365 ) ] )
                     firstOutboundSaByConn[ connectionName ] = False
                  if lifetimeCur:
                     connection.ipsecOutputDataBytes += lifetimeCur.bytesProcessed
                     connection.ipsecOutputDataPkts += lifetimeCur.packetsProcessed
                  if ipsecConnEntry:
                     outSa = ipsecConnEntry.activeOutSa
                     if ( outSa and outSa.sakey == saKey ):
                        # activeOutSa in ipsecConnEntry should be the correct sa we
                        # should use for the connection.outboundSA.
                        connection.outboundSA = sa
                        if lifetimeCur and sa.lifetimeCfg:
                           rekeyTime = int( lifetimeCur.policyAddTime +
                                 sa.lifetimeCfg.xfrmPolicyTimeLimitWarning -
                                 Tac.utcNow() )
                           if ( not connection.rekeyTime or \
                                connection.rekeyTime > rekeyTime ):
                              # update the connection.rekeyTime if it has not been
                              # assigned, or the value assigned by the latest inSa
                              # is higher than the rekeyTime of this activeOutSa
                              connection.rekeyTime = rekeyTime
                  if not connection.outboundSA:
                     # guarantee that a connection has an outboundSA even if
                     # the ipsecConnEntry is not ready for the latest SA
                     connection.outboundSA = sa
                  break
               if isConnIn:
                  if firstInboundSaByConn[ connectionName ]:
                     # as with the outbound one, for the first SA we
                     # copy over the session stats
                     numBytes, numPkts, ipsecUptime = self.getIpsecSessionStats(
                                                      vrfStatus, sa.reqid, 
                                                      ipsecCounterTable, True )
                     connection.ipsecInputDataBytes = numBytes
                     connection.ipsecInputDataPkts = numPkts
                     firstInboundSaByConn[ connectionName ] = False
                  if lifetimeCur:
                     connection.ipsecInputDataBytes += lifetimeCur.bytesProcessed
                     connection.ipsecInputDataPkts += lifetimeCur.packetsProcessed
                  if ipsecConnEntry:
                     inSas = list( ipsecConnEntry.inSa.values() )
                     if ( inSas and inSas[ -1 ].sakey == saKey ):
                        # use the last ( newest ) sa in inSa queue as the
                        # connection.inboundSA
                        connection.inboundSA = sa
                        # Since rekey is only done for controller out SAs,
                        # skip computing rekeyTime for controller in SAs.
                        if ( inSas[ -1 ].sakey not in vrfStatus.controllerSa ) and \
                           lifetimeCur and sa.lifetimeCfg:
                           rekeyTime = int( lifetimeCur.policyAddTime +
                                 sa.lifetimeCfg.xfrmPolicyTimeLimitWarning -
                                 Tac.utcNow() )
                           if ( not connection.rekeyTime or \
                                connection.rekeyTime > rekeyTime ):
                              # update the connection.rekeyTime if this inSa 
                              # rekeyTime is lower than the rekeyTime assigned 
                              # by the active outSa. We want to use the lower 
                              # rekeyTime of the inSa and the outSa as the 
                              # connection.rekeyTime
                              connection.rekeyTime = rekeyTime
                  if not connection.inboundSA:
                     # guarantee that a connection has an inboundSA
                     connection.inboundSA = sa
                  break

      for connectionName, connection in list( self.connections.items() ):
         emptyConnection = False
         if togglePolicyBasedVpnEnabled():
            if not connection[ 'tunnelDict' ] and not connection[ 'pathDict' ] \
               and not connection[ 'policyConnectionState' ]:
               emptyConnection = True
         else:
            if not connection[ 'tunnelDict' ] and not connection[ 'pathDict' ]:
               emptyConnection = True

         if emptyConnection:
            # pylint: disable=unsupported-delete-operation
            del self.connections[ connectionName ]

   def getDataBytes( self, policyBasedTunnel, dataBytes ):
      if policyBasedTunnel:
         return 'N/A'

      return '%s bytes' % dataBytes

   def getPolicyBasedTunnelStatus( self, connection ):
      if connection.policyBasedVpnTunnel and togglePolicyBasedVpnEnabled():
         return '(P)'
      return ''

   def printDataPkts( self, policyBasedTunnel, lastTunnel,
                      tunnelName, connection ):
      if policyBasedTunnel:
         return

      seperator = ''
      if not lastTunnel:
         seperator = ','

      print( IpsecShowFormatStrings.connStateFormatStr %
             ( tunnelName + seperator,
               "", "", "", "",
               '%s pkts' % connection.ipsecInputDataPkts,
               '%s pkts' % connection.ipsecOutputDataPkts, '' ) )

   def printNonDetailedView( self, connection, displayVrfColumn ):

      def getBriefTime( timeStr ):
         if timeStr == "0 seconds":
            return 'N/A'
         return timeStr.split( ', ', 1 )[ 0 ]

      upTime = getBriefTime( connection.upTimeStr )
      rekeyTimeStr = ipsecTimeString( connection.rekeyTime,
                                      "0 seconds", [ ( "seconds", 60 ),
                                      ( "minutes", 60 ), ( "hours", 24 ) ] )
      rekeyTime = getBriefTime( rekeyTimeStr )

      if list( connection.tunnelDict ):
         connNameDict = collections.OrderedDict( 
                                          sorted( connection.tunnelDict.items() ) )
      elif list( connection.pathDict ):
         connNameDict = collections.OrderedDict(
                                          sorted( connection.pathDict.items() ) )
      elif togglePolicyBasedVpnEnabled() and \
           list( connection.policyConnectionState ):
         connNameDict = collections.OrderedDict(
                          sorted( connection.policyConnectionState.items() ) )
      if not connNameDict:
         return

      if len( connNameDict ) > 1:
         if not displayVrfColumn:
            print( IpsecShowFormatStrings.connStateFormatStr %
                   ( next( iter( connNameDict ) ) +
                     self.getPolicyBasedTunnelStatus( connection ) + ',',
                     connection.saddr, connection.daddr,
                     next( iter( connNameDict.values() ) ), upTime,
                     self.getDataBytes( connection.policyBasedVpnTunnel,
                         connection.ipsecInputDataBytes ),
                     self.getDataBytes( connection.policyBasedVpnTunnel,
                         connection.ipsecOutputDataBytes ),
                     rekeyTime ) )
         else:
            print( IpsecShowFormatStrings.connStateFormatVrfStr %
                   ( next( iter( connNameDict ) ) +
                     self.getPolicyBasedTunnelStatus( connection ) + ',',
                     connection.saddr, connection.daddr,
                     next( iter( connNameDict.values() ) ), upTime,
                     self.getDataBytes( connection.policyBasedVpnTunnel,
                         connection.ipsecInputDataBytes ),
                     self.getDataBytes( connection.policyBasedVpnTunnel,
                         connection.ipsecOutputDataBytes ),
                     rekeyTime, connection.tunnelNs ) )

         # no VRFs on the packet counter lines
         if len( connNameDict ) == 2:
            # only two entries to print
            self.printDataPkts( connection.policyBasedVpnTunnel,
                    lastTunnel=True,
                    tunnelName=list( connNameDict )[ 1 ],
                    connection=connection )
            return

         self.printDataPkts( connection.policyBasedVpnTunnel, lastTunnel=False,
                 tunnelName=list( connNameDict )[ 1 ],
                 connection=connection )
      else:
         if not displayVrfColumn:
            print( IpsecShowFormatStrings.connStateFormatStr %
                   ( next( iter( connNameDict ) ) +
                     self.getPolicyBasedTunnelStatus( connection ),
                     connection.saddr, connection.daddr,
                     next( iter( connNameDict.values() ) ), upTime,
                     self.getDataBytes( connection.policyBasedVpnTunnel,
                         connection.ipsecInputDataBytes ),
                     self.getDataBytes( connection.policyBasedVpnTunnel,
                         connection.ipsecOutputDataBytes ),
                     rekeyTime ) )
         else:
            print( IpsecShowFormatStrings.connStateFormatVrfStr %
                   ( next( iter( connNameDict ) ) +
                     self.getPolicyBasedTunnelStatus( connection ),
                     connection.saddr, connection.daddr,
                     next( iter( connNameDict.values() ) ), upTime,
                     self.getDataBytes( connection.policyBasedVpnTunnel,
                         connection.ipsecInputDataBytes ),
                     self.getDataBytes( connection.policyBasedVpnTunnel,
                         connection.ipsecOutputDataBytes ),
                     rekeyTime, connection.tunnelNs ) )

         self.printDataPkts( connection.policyBasedVpnTunnel, lastTunnel=True,
                             tunnelName='', connection=connection )
         return

      # Print the rest of the tunnel entries
      for i in range( 2, len( connNameDict ) ):
         if i == ( len( connNameDict ) - 1 ):
            print( list( connNameDict )[ i ] )
         else:
            print( list( connNameDict )[ i ] + ',', end=' ' )

   def printSaInfo( self, sa, whichSA='Inbound' ):
      if sa:
         lcfg = sa.lifetimeCfg
         lcur = sa.lifetimeCurrent
         print( f"{indent1}{whichSA} SPI: 0x{socket.ntohl( sa.spi ):x}:" )
         print( "    Request ID: %d, Mode: %s, Replay window size: %d, Seq: 0x%x" %
                ( sa.reqid, sa.mode, sa.replayWindow, sa.seq ) )
         print( "%sErrors:" % ( indent2 ) )
         print( "%sPackets outside replay window: %d, Replay: %d,"
                " Integrity failed: %d" % ( indent3, sa.packetsOutOfReplayWindow,
                   sa.replayErrors, sa.statsIntegrityFailed ) )
         print( "%sLifetime config:" % ( indent2 ) )
         print( "%sSoft byte limit: %d, Hard byte limit: %d" %
                ( indent3, lcfg.xfrmPolicyByteLimitWarning,
                      lcfg.xfrmPolicyByteLimitExpiry ) )
         print( "%sSoft packet limit: %d, Hard packet limit: %d" %
                ( indent3, lcfg.xfrmPolicyPacketLimitWarning,
                      lcfg.xfrmPolicyPacketLimitExpiry ) )
         print( "%sSoft time limit: %d secs, Hard time limit: %d secs" %
                ( indent3, lcfg.xfrmPolicyTimeLimitWarning,
                      lcfg.xfrmPolicyTimeLimitExpiry ) )
         print( "%sLifetime current:" % ( indent2 ) )
         print( "%sCurrent bytes: %d" % ( indent3, lcur.bytesProcessed ) )
         print( "%sCurrent packets: %d" % ( indent3, lcur.packetsProcessed ) )
         if lcur.policyAddTime:
            aTime = "%s" % time.ctime(lcur.policyAddTime)
         else:
            aTime = '-'
         if lcur.policyLastAccessTime:
            uTime = "%s" % time.ctime(lcur.policyLastAccessTime)
         else:
            uTime = '-'
         print( f"{indent3}SA add time: {aTime}" )
         print( f"{indent3}SA last use time: {uTime}" )

   # Prints all the dynPathInfo information collected via getIpsecDynamicPathInfo
   # The display structure is closely tied with how dynPathInfo is populated
   # Some of the data comes in as elements in a list
   def printDynpathInfo( self, dynPathInfo=None ):
      if dynPathInfo is None:
         return
      print( "%sPackets in: %d, Packets out: %d" %
           ( indent1, dynPathInfo.sessionPktsIn, dynPathInfo.sessionPktsOut ) )
      print( "%sLast known rekey count: %s" %
                        ( indent1, dynPathInfo.lastKnownLocalRekeyCount ) )
      print( "{}Dynamic state: {}".format(
                 indent1, DynamicPathStateShowMap[ dynPathInfo.pathState ] ) )
      if dynPathInfo.lastEstablishedTime == Tac.endOfTime or \
            not dynPathInfo.lastEstablishedTime:
         estTimeStr = 'not applicable'
      else:
         estTime = int( dynPathInfo.lastEstablishedTime + Tac.utcNow() - Tac.now() )
         estTimeStr = "%s" % time.ctime( estTime )
      print( f"{indent1}Last established time: {estTimeStr}" )

      if dynPathInfo.dhCryptoInfo:
         print( "%sDH crypto:" % ( indent1 ) )
         while dynPathInfo.dhCryptoInfo:
            info = dynPathInfo.dhCryptoInfo[ 0 ]
            print( "%sLocal rekey: %d, Peer rekey: %d,"
                   " In SPI: 0x%x, Out SPI: 0x%x" % ( indent2, info.localRekey,
                            info.peerRekey, info.inSpi, info.outSpi ) )
            dynPathInfo.dhCryptoInfo = dynPathInfo.dhCryptoInfo[ 1 : ]

      if dynPathInfo.dhStateInfo:
         print( "%sDH state:" % ( indent1 ) )
         while dynPathInfo.dhStateInfo:
            state = dynPathInfo.dhStateInfo[ 0 ]
            print( "%sLocal rekey: %d, Peer rekey: %d, DH rekey state: %s,"
                   " Role: %s" % ( indent2, state.localRekey, state.peerRekey,
                            DhRekeyStateShowMap[ state.dhRekeyState ],
                            SmStateShowMap[ state.dhSmState ] ) )
            dynPathInfo.dhStateInfo = dynPathInfo.dhStateInfo[ 1 : ]

      if dynPathInfo.saCryptoInfo:
         print( "%sSA crypto:" % ( indent1 ) )
         while dynPathInfo.saCryptoInfo:
            info = dynPathInfo.saCryptoInfo[ 0 ]
            print( "%sLocal rekey: %d, Peer rekey: %d, In SPI 0x%x,"
                   " Out SPI 0x%x" % ( indent2, info.localRekey, info.peerRekey,
                            info.inSpi, info.outSpi ) )
            dynPathInfo.saCryptoInfo = dynPathInfo.saCryptoInfo[ 1 : ]

      if dynPathInfo.saSmState:
         print( "{}SA rekey role: {}".format( indent1,
                 SmStateShowMap[ dynPathInfo.saSmState ] ) )

      if dynPathInfo.saInfo:
         print( "%sSA state:" % ( indent1 ) )
         while dynPathInfo.saInfo:
            dynInfoSaM = dynPathInfo.saInfo[ 0 ]
            print( "{}SA type: {} , SPI: 0x{:x}".format( indent2,
                      SaTypeShowMap[ dynInfoSaM.saType ],
                      socket.ntohl( dynInfoSaM.spi ) ) )
            role = "initiator" if dynInfoSaM.initiator else "responder"
            print( "%sRole: %s, SA rekey state: %s, Peer nonce type: %s" %
                      ( indent3, role,
                        SaRekeyStateShowMap[ dynInfoSaM.saRekeyState ],
                        dynInfoSaM.peerNonceType ) )
            if dynInfoSaM.addTime:
               addTime = "%s" % time.ctime( dynInfoSaM.addTime )
            else:
               addTime = 'Not Programmed'
            print( "{}Add time: {}, Packets: {}, Pair SPI: 0x{:x}".format(
                         indent3, addTime, dynInfoSaM.pkts, dynInfoSaM.pairSpi ) )
            dynPathInfo.saInfo = dynPathInfo.saInfo[ 1 : ]

   def printTrafficSelector( self, connection ):
      print( "{}Traffic Selectors:".format( indent1 ) )
      print( "{}Source prefix: {}, Destination prefix: {}".format(
              indent2, connection.trafficSelectorSrcPrefix,
              connection.trafficSelectorDstPrefix ) )

   def printDetailedView( self, connection, dynPathInfo ):
      if list( connection.tunnelDict ):
         connNameDict = collections.OrderedDict( 
                                          sorted( connection.tunnelDict.items() ) )
      elif list( connection.pathDict ):
         connNameDict = collections.OrderedDict(
                                          sorted( connection.pathDict.items() ) )
      elif togglePolicyBasedVpnEnabled() and \
              list( connection.policyConnectionState ):
         connNameDict = collections.OrderedDict(
                            sorted( connection.policyConnectionState.items() ) )
      numConnections = len( connNameDict )
      for i in range( 0, numConnections ):
         if i == numConnections - 1:
            print( "%s:" % ( list( connNameDict )[ i ] ) )
         else:
            print( "%s," % ( list( connection.tunnelDict )[ i ] ),
                   end=' ' )
      print( "{}Source address: {}, Destination address: {}".format(
              indent1, connection.saddr, connection.daddr ) )

      policyBasedConnectionChildSa = False
      if connection.trafficSelectorSrcPrefix or connection.trafficSelectorDstPrefix:
         self.printTrafficSelector( connection )
         policyBasedConnectionChildSa = True

      if list( connection.tunnelDict ):
         connState = next( iter( connection.tunnelDict.values() ) )
      elif togglePolicyBasedVpnEnabled() and \
           list( connection.policyConnectionState ):
         connState = next( iter( connection.policyConnectionState.values() ) )
      else:
         connState = next( iter( connection.pathDict.values() ) )
      connStateStr = connState[ 0 ].lower() + connState[ 1 : ]
      print( f"{indent1}State: {connStateStr}" )
      print( f"{indent1}Uptime: {connection.upTimeStr}" )
      print( f"{indent1}VRF: {connection.tunnelNs}" )

      if togglePolicyBasedVpnEnabled() and not policyBasedConnectionChildSa and \
         list( connection.tunnelDict ):
         if connection.policyBasedVpnTunnel:
            print( f"{indent1}VPN type: policy based tunnel" )
         else:
            print( f"{indent1}VPN type: route based tunnel" )

      if list( connection.pathDict ):
         if connection.dynamicPath:
            pathState = 'dynamic'
         else:
            pathState = 'static'
         print( f"{indent1}Type: {pathState}" )
         print( f"{indent1}Remote IP: {connection.remoteIp}" )
         role = "initiator" if connection.initiator else "responder"
         print( f"{indent1}Role: {role}" )
      self.printDynpathInfo( dynPathInfo )
      self.printSaInfo( connection.inboundSA, whichSA='Inbound' )
      self.printSaInfo( connection.outboundSA, whichSA='Outbound' )

   def printConnection( self, connection, connectionName,
                        printHeader, multiVrf ):
      dynPathInfo = None
      try:
         dynPathInfo = connection.dynPathDict[ connectionName ]
      except KeyError:
         pass
      if self._detailed:
         self.printDetailedView( connection, dynPathInfo )
      else:
         if printHeader:
            vrfHead = 'Vrf' if multiVrf else ''
            print( IpsecShowFormatStrings.connStateFormatVrfStr % (
                                             'Tunnel', 'Source', 'Dest',
                                             'Status', 'Uptime', 'Input',
                                             'Output', 'Rekey Time', vrfHead ) )
         self.printNonDetailedView( connection, multiVrf )


   def render( self ):
      # Get the tunnel interfaces that are in use from app profiles
      firstTime = True

      # Basically, try to print in sorted tunnel list order as much
      # as possible.
      connectionDict = {}
      for connectionName in self.connections:
         connection = self.connections[ connectionName ]
         # get the first tunnelName from sorted list.
         if list( connection.tunnelDict ) or \
            ( togglePolicyBasedVpnEnabled() and
            list( connection.policyConnectionState ) ):
            tunnelName = None
            connectionName = None
            if list( connection.tunnelDict ):
               tunnelName = min( i.title() for i in connection.tunnelDict )
               connectionName = tunnelName
            elif togglePolicyBasedVpnEnabled():
               tunnelName = connection.policyTunnelName.title()
               connectionName = \
                       min( i.title() for i in connection.policyConnectionState )
            if tunnelName not in connectionDict:
               connectionDict[ tunnelName ] = {}
            connectionDict[ tunnelName ][ connectionName ] = connection
         elif list( connection.pathDict ):
            pathName = next( iter( connection.pathDict ) )
            connectionName = pathName
            if pathName not in connectionDict:
               connectionDict[ pathName ] = {}
            connectionDict[ pathName ][ connectionName ] = connection

      multiVrf = False
      firstVrf = None
      for connectionName, connections in connectionDict.items():
         for _, conn in connections.items():
            if not firstVrf:
               firstVrf = conn.tunnelNs
            else:
               if firstVrf != conn.tunnelNs:
                  multiVrf = True
                  break

      if togglePolicyBasedVpnEnabled() and not self._detailed:
         print( 'Legend: (P) policy based VPN tunnel' )

      for parentConnectionName in Arnet.sortIntf( connectionDict ):
         connectionNames = list( connectionDict[ parentConnectionName ].keys() )
         connectionNames.sort()
         for connectionName in connectionNames:
            connection = connectionDict[ parentConnectionName ][ connectionName ]
            self.printConnection( connection, connectionName,
                                  printHeader=firstTime,
                                  multiVrf=multiVrf )
            firstTime = False

class IpsecPeer( Model ):
   ''' CLI Model for one Ipsec key controller peer'''
   peerIp = IpGenericAddress( help="Peer IP Address" )
   peerStatus = Enum( values=( IPSEC_KEY_CONTROLLER_PEER_STATUS ),
                      help="Status of the IPsec key controller peer" )
   dhGroup = Int( help="Diffie-Hellman (DH) group number" )
   encryptAlgo = Enum( values=IPSEC_ENCRYPTION_MAP,
                       help='Encryption Algorithm' )
   hashAlgo = Enum( values=IPSEC_INTEGRITY_MAP,
                    help="INTEGRITY Algorithm" )
   uptime = Float( help="Seconds since the peer came up" )
   lastKeyUptime = Int( help="Seconds since the last key material received" )
   initiator = Bool( help="The peer is the initiator" )

   def fillPeerInfo( self, peerConnState ):
      self.peerIp = peerConnState.remoteIp
      self.dhGroup = peerConnState.dhGroup

      if peerConnState.peerConnStatus != IpsecPeerConnStatusEnum.peerUp:
         self.uptime = 0.0
      else:
         self.uptime = Tac.utcNow() - peerConnState.upTime
      if not peerConnState.lastDhKeyAddTime:
         self.lastKeyUptime = 0
      else:
         self.lastKeyUptime = int( Tac.utcNow() - peerConnState.lastDhKeyAddTime )
      if peerConnState.encryptAlgo in IPSEC_ENCRYPTION_MAP:
         self.encryptAlgo = peerConnState.encryptAlgo
      else:
         self.encryptAlgo = 'unknownEncr'
      if peerConnState.hashAlgo in IPSEC_INTEGRITY_MAP:
         self.hashAlgo = peerConnState.hashAlgo
      else:
         self.hashAlgo = 'unknownIntegrity'
      if peerConnState.peerConnStatus == IpsecPeerConnStatusEnum.peerUp:
         self.peerStatus = IpsecPeerConnStatusEnum.peerUp
      elif peerConnState.peerDownReason in IPSEC_KEY_CONTROLLER_PEER_STATUS:
         self.peerStatus = peerConnState.peerDownReason
      else:
         self.peerStatus = 'unknownPeerState'
      self.initiator = peerConnState.initiator

def getCryptoSuiteStr( encryptAlgo, hashAlgo ):
   if encryptAlgo == IpsecEspAlgorithmEnum.aes128gcm64:
      hashAlgoStr = '64bit Hash'
   elif encryptAlgo == IpsecEspAlgorithmEnum.aes128gcm128 or \
        encryptAlgo == IpsecEspAlgorithmEnum.aes256gcm128:
      hashAlgoStr = '128bit Hash'
   else:
      hashAlgoStr = IPSEC_INTEGRITY_MAP[ hashAlgo ]
   return f"{IPSEC_ENCRYPTION_MAP[ encryptAlgo ]} , {hashAlgoStr}"

class IpsecPeerState( Model ):
   ''' CLI Model for show ip security key controller peer '''
   _detailed = Bool( help="Display detailed output" )
   peers = Dict( help="A mapping between peer address and "
                      "IPsec peer info of the key controller",
                 keyType=IpGenericAddress,
                 valueType=IpsecPeer )

   def fillPeersInfo( self, status, detailed, peerIpStr=None ):
      self._detailed = detailed
      if peerIpStr is None:
         for peerIp in status.ipsecPeerConnectionState:
            ipsecPeer = IpsecPeer()
            ipsecPeer.fillPeerInfo( status.ipsecPeerConnectionState[ peerIp ] )
            self.peers[ peerIp ] = ipsecPeer
      else:
         peerIp = Tac.Value( "Arnet::IpGenAddr", stringValue=peerIpStr )
         if peerIp in status.ipsecPeerConnectionState:
            ipsecPeer = IpsecPeer()
            ipsecPeer.fillPeerInfo( status.ipsecPeerConnectionState[ peerIp ] )
            self.peers[ peerIp ] = ipsecPeer

   def render( self ):
      if self._detailed:
         peerDetailFormatStr = "Peer: %s\nStatus: %s\nDH Group: %s\n"\
                               "Crypto suite: %s\nUp time: %s\n"\
                               "Last key material received: %s\nInitiator: %s\n"
         for peerIp, peer in sorted( self.peers.items() ):
            uptimeStr = "N/A"
            lastKeyUptimeStr = "N/A"
            if peer.uptime > 0:
               uptimeStr = ipsecTimeString( int( peer.uptime ),
                                            "0 seconds", [ ( "seconds", 60 ),
                                            ( "minutes", 60 ), ( "hours", 24 ),
                                            ( "days", 365 ) ] )
            if peer.lastKeyUptime > 0:
               lastKeyUptimeStr = \
                     ipsecTimeString( peer.lastKeyUptime,
                                      "0 seconds", [ ( "seconds", 60 ),
                                      ( "minutes", 60 ), ( "hours", 24 ),
                                      ( "days", 365 ) ] )
            peerCryptoSuiteStr = getCryptoSuiteStr( peer.encryptAlgo, peer.hashAlgo )
            print( peerDetailFormatStr % ( peerIp,
                        IPSEC_KEY_CONTROLLER_PEER_STATUS[ peer.peerStatus ],
                        peer.dhGroup, peerCryptoSuiteStr,
                        uptimeStr, lastKeyUptimeStr, peer.initiator ) )
         
      else:
         print( IpsecShowFormatStrings.peerFormatStr % (
            "Peer", "DH Group", "Crypto Suite", "Status" ) )
         print( IpsecShowFormatStrings.peerFormatStr % (
            "----", "--------", "------------", "------" ) )
         for peerIp, peer in sorted( self.peers.items() ):
            peerStatusStr = "down"
            if peer.peerStatus == IpsecPeerConnStatusEnum.peerUp:
               peerStatusStr = "up"
            peerCryptoSuiteStr = getCryptoSuiteStr( peer.encryptAlgo, peer.hashAlgo )
            print( IpsecShowFormatStrings.peerFormatStr % (
               peerIp, peer.dhGroup, peerCryptoSuiteStr, peerStatusStr ) )

class IpsecLocalKey( Model ):
   ''' CLI Model for Ipsec key controller local key info '''
   dhGroup = Enum( values=IPSEC_DHGROUP_NUMBER_MAP,
                   help="Diffie-Hellman (DH) group number" )
   encryptAlgo = Enum( values=IPSEC_ENCRYPTION_MAP,
                       help='Encryption Algorithm' )
   hashAlgo = Enum( values=IPSEC_INTEGRITY_MAP,
                    help="Integrity Algorithm" )
   firstKeyAddTime = Float( help="Seconds the initial DH key was created" )
   firstDhKeyAddTime = Int( help="Seconds the initial DH key was created" )
   lastKeyUptime = Int( help="Seconds since the last key was created" )
   rekeyTime = Int( help="Seconds remaining until next rekey" )

   def fillLocalKeyInfo( self, config, status ):
      keyControllerCfg = config.keyController
      localKeyState = status.localKeyState
      if keyControllerCfg.dhGroup in IPSEC_DHGROUP_NUMBER_MAP:
         self.dhGroup = keyControllerCfg.dhGroup
      else:
         self.dhGroup = 'unknownDhGroup'
      self.firstKeyAddTime = localKeyState.firstDhKeyAddTime
      self.firstDhKeyAddTime = int( Tac.utcNow() - localKeyState.firstDhKeyAddTime )
      if not localKeyState.lastDhKeyAddTime:
         self.lastKeyUptime = 0
      else:
         self.lastKeyUptime = int( Tac.utcNow() - localKeyState.lastDhKeyAddTime )
      if not self.lastKeyUptime:
         self.rekeyTime = 0
      else:
         nextRekeyTime = \
            int( keyControllerCfg.dhLifetime - self.lastKeyUptime )
         self.rekeyTime = nextRekeyTime if nextRekeyTime > 0 else 0

      self.encryptAlgo = 'unknownEncr'
      self.hashAlgo = 'unknownIntegrity'
      profileName = config.keyController.profileName
      profileCfg = config.ipsecProfile.get( profileName )
      if profileCfg:
         sa = config.securityAssoc[ profileCfg.securityAssocName ]
         if sa:
            saParams = sa.saParams
            if saParams.espAes in IPSEC_ENCRYPTION_MAP:
               self.encryptAlgo = saParams.espAes
            if saParams.espSha in IPSEC_INTEGRITY_MAP:
               self.hashAlgo = saParams.espSha

   def render( self ):
      firstKeyAddTimeStr = "N/A"
      lastKeyUptimeStr = "N/A"
      rekeyTimeStr = "N/A"
      if self.firstDhKeyAddTime > 0:
         firstKeyAddTimeStr = time.asctime( time.gmtime( self.firstDhKeyAddTime ) )
         firstKeyAddTimeStr = ipsecTimeString( self.firstDhKeyAddTime,
                                             "0 seconds", [ ( "seconds", 60 ),
                                             ( "minutes", 60 ), ( "hours", 24 ),
                                             ( "days", 365 ) ] )
      if self.lastKeyUptime > 0:
         lastKeyUptimeStr = ipsecTimeString( self.lastKeyUptime,
                                             "0 seconds", [ ( "seconds", 60 ),
                                             ( "minutes", 60 ), ( "hours", 24 ),
                                             ( "days", 365 ) ] )
      if self.rekeyTime > 0:
         rekeyTimeStr = ipsecTimeString( self.rekeyTime,
                                         "0 seconds", [ ( "seconds", 60 ),
                                         ( "minutes", 60 ), ( "hours", 24 ) ] )
      cryptoSuiteStr = getCryptoSuiteStr( self.encryptAlgo, self.hashAlgo )
      localKeyFormatStr = "DH group: %s\n" \
                          "Crypto suite: %s\n" \
                          "Initial DH key created: %s\n" \
                          "Last DH key created: %s\n" \
                          "DH rekey: %s"
      print( localKeyFormatStr % ( IPSEC_DHGROUP_NUMBER_MAP[ self.dhGroup ],
                                  cryptoSuiteStr, firstKeyAddTimeStr,
                                  lastKeyUptimeStr, rekeyTimeStr ) )

class IpsecLocalKeyState( Model ):
   ''' CLI Model for configured Ipsec key controller local key '''
   keyController = Submodel( valueType=IpsecLocalKey, 
                             help="Configured IPsec Key Controller",
                             optional=True )
   def fillLocalKeyState( self, config, status ):
      self.keyController = None
      keyControllerCfg = config.keyController
      if keyControllerCfg.configured:
         self.keyController = IpsecLocalKey()
         self.keyController.fillLocalKeyInfo( config, status )

   def render( self ):
      if self.keyController is not None:
         self.keyController.render()

class IpsecConnectionId( DeferredModel ):
   connectionIdSrc = Enum(
                values=( IPSEC_CONNIDSRC_LIST ), help="Connection ID source" )
   connectionId = Int( help="Connection ID value" )

class IpsecConnectionKey( DeferredModel ):
   vrfId = Int( help="VRF ID of the path" )
   srcAddr = IpGenericAddress( help="Source IP address of the path" )
   dstAddr = IpGenericAddress( help="Destination IP address of the path" )
   connectionId = Submodel( valueType=IpsecConnectionId,
                        help="Connection ID of the connection key" )

class IpsecDhKeyPair( DeferredModel ):
   localDhRekeyCount = Int( help="Local DH rekey count" )
   peerDhRekeyCount = Int( help="Peer DH rekey count" )

class IpsecSaEvent( DeferredModel ):
   eventTime = Float( help="UTC timestamp of the event" )
   eventName = Str( help="Name of the event" )
   spi = Int( help="Spi value of the SA key" )
   pathState = Enum( values=DynamicPathStateShowMap.keys(),
                     help="Dynamic path state" )
   dhKeyPair = Submodel( valueType=IpsecDhKeyPair, help="Current BGP DH key pair" )

class IpsecPathEvent( DeferredModel ):
   eventTime = Float( help="UTC timestamp of the event" )
   eventName = Str( help="Name of the event" )
   connKey = Submodel( valueType=IpsecConnectionKey,
                       help="Connection key of the path" )
   pathState = Enum( values=DynamicPathStateShowMap.keys(),
                     help="Dynamic path state" )
   dhKeyPair = Submodel( valueType=IpsecDhKeyPair,
                         help="Current BGP DH key pair" )

class IpsecPeerEvent( DeferredModel ):
   eventTime = Float( help="UTC timestamp of the event" )
   eventName = Str( help="Name of the event" )
   remoteIp = IpGenericAddress( help="Remote peer IP address" )
   dhKeyPair = Submodel( valueType=IpsecDhKeyPair, help="Current BGP DH key pair" )

class IpsecLocalEvent( DeferredModel ):
   eventTime = Float( help="UTC timestamp of the event" )
   eventName = Str( help="Name of the event" )
   rekeyCounter = Int( help="Local DH rekey counter" )

class IpsecSaEventEntry( DeferredModel ):
   events = List( valueType=IpsecSaEvent, help="Sa events list" )

class IpsecPathEventEntry( DeferredModel ):
   events = List( valueType=IpsecPathEvent, help="Path events list" )

class IpsecPeerEventEntry( DeferredModel ):
   events = List( valueType=IpsecPeerEvent, help="Peer events list" )

class IpsecLocalEventType( DeferredModel ):
   events = List( valueType=IpsecLocalEvent, help="Local events list" )

class IpsecRecordEvents( DeferredModel ):
   saEvents = Dict( keyType=str, valueType=IpsecSaEventEntry, optional=True,
                     help="Sa events, keyed by SA key" )
   pathEvents = Dict( keyType=str, valueType=IpsecPathEventEntry, optional=True,
                     help="Path events, keyed by connection key" )
   peerEvents = Dict( keyType=str, valueType=IpsecPeerEventEntry, optional=True,
                     help="Peer events, keyed by remote IP" )
   localEvents = Submodel( valueType=IpsecLocalEventType, optional=True,
                      help="Local events" )
