# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliMatcher
import CliParser
import LazyMount
import BasicCli
import Tac
from CliPlugin import IpAddrMatcher
from CliPlugin import IpsecModel
from CliPlugin import VrfCli
from CliPlugin import TunnelIntfCli
from CliPlugin import TechSupportCli
import CliToken.Ip
import CliToken.Ipsec
import ShowCommand
import SmashLazyMount
from Toggles.PolicyMapToggleLib import togglePolicyBasedVpnEnabled
from Toggles.IpsecToggleLib import toggleAutoVpnSnapshotEnabled
import IpsecAgent
import AgentCommandRequest

#Assigning globals.
ipsecDaemonConfigDir = None
ipsecDaemonStatusDir = None
ipsecIkeConfigDir = None
ipsecIkeStatusDir = None
ipsecConnTableDir = None
tunnelIntfStatusDir = None
ipsecPathStatus = None
entMan = None
ipsecCounterTable = None
ipsecPlatformStatus = None
sslStatus = None

def haveValidIpsecLicense( func ):
   ''' Decorator for config commands. If a valid IPsec license is not 
   found, throw a warning.'''
   def newFunc( mode, *args, **kwargs ):
      if not ipsecIkeStatusDir.ipsecLicenseEnabled:
         mode.addWarning( 'No valid IPsec license found. '
                          'IPsec is disabled.' )
      return func( mode, *args, **kwargs )
   return newFunc

matcherSecurity = CliMatcher.KeywordMatcher( 'security',
      helpdesc='IP security information' )
matcherDaemon = CliMatcher.KeywordMatcher( 'daemon',
      helpdesc='Show IPsec Daemon information' )

profileNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: ipsecIkeConfigDir.ipsecProfile,
      pattern='[a-zA-Z0-9_-]+',
      helpdesc='name of the profile',
      helpname='PROFILE_NAME',
      priority=CliParser.PRIO_LOW )

policyNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: ipsecIkeConfigDir.ikePolicy,
      pattern='[a-zA-Z0-9_-]+',
      helpdesc='name of the policy',
      helpname='POLICY_NAME',
      priority=CliParser.PRIO_LOW )

saNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: ipsecIkeConfigDir.securityAssoc,
      pattern='[a-zA-Z0-9_-]+',
      helpdesc='name of the TS',
      helpname='TS_NAME',
      priority=CliParser.PRIO_LOW )

vrfExprFactory = VrfCli.VrfExprFactory(
      helpdesc='VRF instance' )

#--------------------------------------------------------------------------------
# show ip security fips status
#--------------------------------------------------------------------------------
@haveValidIpsecLicense
def showIpsecFipsStatus( mode, args ):
   ''' function to create Ipsec Fips Status. '''

   statusModel = IpsecModel.IpsecFipsStatus()
   statusModel.fillIpsecFipsStatusInfo( ipsecDaemonConfigDir, ipsecDaemonStatusDir,
                                        ipsecPlatformStatus, ipsecIkeStatusDir,
                                        ipsecIkeConfigDir )
   return statusModel

class IpSecurityFipsStatusCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip security fips status'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'security': matcherSecurity,
      'fips': 'FIPS related information',
      'status': 'Display IPsec FIPS status information',
   }
   handler = showIpsecFipsStatus
   cliModel = IpsecModel.IpsecFipsStatus

BasicCli.addShowCommandClass( IpSecurityFipsStatusCmd )

#--------------------------------------------------------------------------------
# show ip security security-association [ TS_NAME ]
#--------------------------------------------------------------------------------
@haveValidIpsecLicense
def showIpsecSecurityAssociations( mode, args ):
   ''' function to create list of transform-sets. '''
   secAssocName = args.get( 'TS_NAME' )
   def fillSecurityAssociationInfo( SA, secAssocsModel, securityAssociations ):
      secAssocModel = IpsecModel.IpsecSecurityAssociation()
      secAssocModel.fillIpsecSecurityAssociationInfo( securityAssociations[ SA ],
                                                      ipsecIkeConfigDir )
      secAssocsModel.ipsecSecurityAssociations[ SA ] = secAssocModel

   secAssocsModel = IpsecModel.IpsecSecurityAssociations()
   ipsecSecurityAssociations = ipsecIkeConfigDir.securityAssoc
   if secAssocName is None:
      for secAssocName in ipsecSecurityAssociations:
         fillSecurityAssociationInfo( secAssocName, secAssocsModel, 
               ipsecSecurityAssociations )
   else:
      if secAssocName in ipsecSecurityAssociations:
         fillSecurityAssociationInfo( secAssocName, secAssocsModel, 
               ipsecSecurityAssociations )
      else:
         mode.addError( f'SA {secAssocName} has not been configured' )

   return secAssocsModel

class IpSecuritySecurityAssociationCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip security security-association [ TS_NAME ]'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'security': matcherSecurity,
      'security-association': 'Display Security Associations',
      'TS_NAME': saNameMatcher,
   }
   handler = showIpsecSecurityAssociations
   cliModel = IpsecModel.IpsecSecurityAssociations

BasicCli.addShowCommandClass( IpSecuritySecurityAssociationCmd )

#--------------------------------------------------------------------------------
# show ip security policy [ POLICY_NAME ]
#--------------------------------------------------------------------------------
@haveValidIpsecLicense
def showIpsecPolicies( mode, args ):
   ''' function to create list of policies. '''

   polName = args.get( 'POLICY_NAME' )
   def fillPolicyInfo( polName, policiesModel, policies ):
      policyModel = IpsecModel.IpsecPolicy()
      policyModel.fillIpsecPolicyInfo( policies[ polName ], ipsecIkeConfigDir )
      policiesModel.ipsecPolicies[polName] = policyModel

   policiesModel = IpsecModel.IpsecPolicies()
   policies = ipsecIkeConfigDir.ikePolicy
   if polName is None:
      for polName in policies:
         fillPolicyInfo( polName, policiesModel, policies )
   else:
      if polName in policies:
         fillPolicyInfo( polName, policiesModel, policies )
      else:
         mode.addError( f'Policy {polName} has not been configured' )

   return policiesModel

class IpSecurityPolicyCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip security policy [ POLICY_NAME ]'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'security': matcherSecurity,
      'policy': 'Display Policy information',
      'POLICY_NAME': policyNameMatcher,
   }
   handler = showIpsecPolicies
   cliModel = IpsecModel.IpsecPolicies

BasicCli.addShowCommandClass( IpSecurityPolicyCmd )

#--------------------------------------------------------------------------------
# show ip security applied-profile [ vrf VRF_NAME ] [ PROFILE_NAME ]
#--------------------------------------------------------------------------------
@haveValidIpsecLicense
def showIpsecAppliedProfiles( mode, args ):
   ''' function to create list of applied profiles. '''
   profName = args.get( 'PROFILE_NAME' )
   vrfName = args.get( 'VRF', VrfCli.DEFAULT_VRF )

   def fillProfileInfo( profile, profilesModel, profiles, vrfName ):
      profileModel = IpsecModel.IpsecAppliedProfile()
      profileModel.fillIpsecAppliedProfileInfo( ipsecIkeStatusDir,
                                                profiles[ profile ],
                                                vrfName,
                                                ipsecPathStatus,
                                                tunnelIntfStatusDir )
      profilesModel.ipsecAppliedProfiles[ profile ] = profileModel

   profilesModel = IpsecModel.IpsecAppliedProfiles()
   profiles = ipsecIkeStatusDir.appliedProfiles
   if profName is None:
      for profName in profiles:
         fillProfileInfo( profName, profilesModel, profiles, vrfName )
   else:
      if profName in profiles:
         fillProfileInfo( profName, profilesModel, profiles, vrfName )
      else:
         mode.addError( f'IPsec Profile {profName} has not been applied' )

   return profilesModel

class IpSecurityAppliedProfileCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip security applied-profile [ VRF ] [ PROFILE_NAME ]'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'security': matcherSecurity,
      'applied-profile': 'Display Applied Profiles',
      'VRF': vrfExprFactory,
      'PROFILE_NAME': profileNameMatcher,
   }
   handler = showIpsecAppliedProfiles
   cliModel = IpsecModel.IpsecAppliedProfiles

BasicCli.addShowCommandClass( IpSecurityAppliedProfileCmd )

#--------------------------------------------------------------------------------
# show ip security daemon
#--------------------------------------------------------------------------------
@haveValidIpsecLicense
def showIpsecDaemonState( mode, args ):
   ''' function to create list of profiles. '''
   ipsecDaemonStateModel = IpsecModel.IpsecDaemonState()
   ipsecDaemonStateModel.fillIpsecDaemonStateInfo( ipsecDaemonConfigDir,
                                                   ipsecDaemonStatusDir )
   return ipsecDaemonStateModel

class IpSecurityDaemonCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip security daemon [ VRF ]'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'security': matcherSecurity,
      'daemon': matcherDaemon,
      'VRF': vrfExprFactory,
   }
   handler = showIpsecDaemonState
   cliModel = IpsecModel.IpsecDaemonState

BasicCli.addShowCommandClass( IpSecurityDaemonCmd )

#--------------------------------------------------------------------------------
# show ip security daemon log
#--------------------------------------------------------------------------------
@haveValidIpsecLicense
def showIpsecDaemonLog( mode, args ):
   model = IpsecModel.IpsecDaemonLog()

   vrfArg = args.get( 'VRF', VrfCli.DEFAULT_VRF )
   if vrfArg == 'all':
      vrfList = list( ipsecIkeStatusDir.vrfStatus.keys() )
   else:
      vrfList = [ vrfArg ]

   items = ''
   for vrfName in vrfList:
      cmd = f'cat /var/log/agents/charon-VRF-{vrfName}.log'

      output = Tac.run( cmd.split(), stdout=Tac.CAPTURE,
                        stderr=Tac.CAPTURE, ignoreReturnCode=True, asRoot=True )
      if 'Unable to connect: Connection refused' not in output:
         items += output

   model.daemonLog = items.splitlines()
   return model

class IpSecurityDaemonLogCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip security daemon log [ VRF ]'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'security': matcherSecurity,
      'daemon': matcherDaemon,
      'log': 'Show IPsec daemon logs',
      'VRF': vrfExprFactory,
   }
   handler = showIpsecDaemonLog
   cliModel = IpsecModel.IpsecDaemonLog

BasicCli.addShowCommandClass( IpSecurityDaemonLogCmd )

#---------------------------------------------------------------------------------
# show ip security connection [ vrf VRF_NAME ]
#                             [ ( Tunnel | TUNNEL ) |
#                               ( path [ ( name PATH_NAME ) |
#                                        ( peer PEER_ADDR ) ] ) ]
#                             [ detail ]
#--------------------------------------------------------------------------------
@haveValidIpsecLicense
def showIpsecConnectionState( mode, args ):
   ''' function to create list of profiles. '''
   detailed = 'detail' in args
   vrfName = args.get( 'VRF', VrfCli.DEFAULT_VRF )
   tunnelName = args[ 'TUNNEL' ].name if 'TUNNEL' in args else None
   tunnels = 'Tunnel' in args
   pathName = None
   pathPeerIp = None
   paths = False
   pathPeerFqdn = None
   policies = False
   policyConnectionName = None

   if 'path' in args:
      if 'PATH_NAME' in args:
         pathName = args[ 'PATH_NAME' ]
         pathName = pathName[ 0 ].lower() + pathName[ 1 : ]
      elif 'PEER_ADDR' in args:
         pathPeerIp = args[ 'PEER_ADDR' ]
      else:
         paths = True

   if 'policy' in args:
      if 'POLICY_NAME' in args:
         policyConnectionName = args[ 'POLICY_NAME' ]
      else:
         policies = True

   #if tunnelOrPath is not None:
      # following changes are needed when we add pathPeerFqdn
      #if type( tunnelOrPath ) == type( [] ):
      #   for keyValuePair in tunnelOrPath:
      #      key = keyValuePair[ 0 ]
      #      if key == 'pathPeerIp':
      #         pathPeerIp = keyValuePair[ 1 ]
      #      elif key == 'pathPeerFqdn':
      #         pathPeerFqdn = keyValuePair[ 1 ]
      #      break
      #else:

   IpsecConnectionStateModel = IpsecModel.IpsecConnectionState()
   IpsecConnectionStateModel.fillIpsecConnectionStateInfo(
                                       entMan, vrfName,
                                       ipsecIkeStatusDir,
                                       ipsecConnTableDir,
                                       tunnelIntfStatusDir, 
                                       ipsecCounterTable,
                                       ipsecPathStatus,
                                       tunnelName=tunnelName, 
                                       pathName=pathName, 
                                       pathPeerIp=pathPeerIp,
                                       pathPeerFqdn=pathPeerFqdn,
                                       policyConnectionName=policyConnectionName,
                                       tunnels=tunnels,
                                       paths=paths,
                                       policies=policies,
                                       detailed=detailed )
   return IpsecConnectionStateModel

class IpSecurityConnectionCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show ip security connection [ VRF ] '
               '[ ( Tunnel | TUNNEL ) | '
               '  ( path [ ( name PATH_NAME ) | ( peer PEER_ADDR ) ] ) ]'
               '[ detail ]' )
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'security': matcherSecurity,
      'connection': 'Show IPsec Connection information',
      'VRF': vrfExprFactory,
      'Tunnel': 'Tunnel interface',
      'TUNNEL': TunnelIntfCli.TunnelIntf.matcher,
      'path': 'Path information',
      'name': 'Name of the path',
      'PATH_NAME': CliMatcher.PatternMatcher( pattern=r'\S+',
         helpdesc='Name of the path', helpname='WORD' ),
      'peer': 'Peer IP address',
      'PEER_ADDR': IpAddrMatcher.IpAddrMatcher( helpdesc='IP address' ),
      'detail': 'Detailed IKE information',
   }
   handler = showIpsecConnectionState
   cliModel = IpsecModel.IpsecConnectionState

BasicCli.addShowCommandClass( IpSecurityConnectionCmd )

# ---------------------------------------------------------------------------------
# show ip security connection path [ ( name PATH_NAME ) ] events [ EVENT_TYPE ]
# --------------------------------------------------------------------------------
@haveValidIpsecLicense
def showIpsecConnectionEvent( mode, args ):
   acrCommand = AgentCommandRequest.AcrCommand()
   acrCommand.addArgs( args )
   AgentCommandRequest.runSocketCommand(
      entityManager=entMan, dirName=IpsecAgent.name(),
      commandType="IpsecRecordEventAsync",
      command=acrCommand.commandParams(),
      asyncCommand=True, timeout=10,
      outputFormat=mode.session_.outputFormat_ )
   return IpsecModel.IpsecRecordEvents

class IpSecurityConnectionEventCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show ip security connection path '
              '[ ( name PATH_NAME ) ] events [ EVENT_TYPE ] ' )
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'security': matcherSecurity,
      'connection': 'Show IPsec Connection information',
      'path': 'Path information',
      'name': 'Name of the path',
      'PATH_NAME': CliMatcher.PatternMatcher( pattern=r'\S+',
         helpdesc='Name of the path', helpname='WORD' ),
      'events': 'Type of the events',
      'EVENT_TYPE': CliMatcher.EnumMatcher( {
         'sa': 'SA events',
         'path': 'Path events',
         'peer': 'Peer events',
         'local': 'Local events'
      } ),
   }
   handler = showIpsecConnectionEvent
   cliModel = IpsecModel.IpsecRecordEvents

if toggleAutoVpnSnapshotEnabled():
   BasicCli.addShowCommandClass( IpSecurityConnectionEventCmd )

class IpSecurityPolicyConnectionCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show ip security connection [ VRF ]  policy [ POLICY_NAME ] '
               '[ detail ]' )
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'security': matcherSecurity,
      'connection': 'Show IPsec Connection information',
      'VRF': vrfExprFactory,
      'policy': 'Policy based VPN connection information',
      'POLICY_NAME': CliMatcher.PatternMatcher( pattern=r'\S+',
         helpdesc='Policy based VPN connection name', helpname='WORD' ),
      'detail': 'Detailed IKE information',
   }
   handler = showIpsecConnectionState
   cliModel = IpsecModel.IpsecConnectionState

if togglePolicyBasedVpnEnabled():
   BasicCli.addShowCommandClass( IpSecurityPolicyConnectionCmd )

#--------------------------------------------------------------------------------
# show ip security profile <profile>
#--------------------------------------------------------------------------------
@haveValidIpsecLicense
def showIpsecProfiles( mode, args ):
   ''' function to create list of profiles. '''

   pName = args.get( 'PROFILE_NAME' )
   def fillProfileInfo( profName, profilesModel, profiles ):
      profileModel = IpsecModel.IpsecProfile()
      profileModel.fillIpsecProfileInfo( profiles[ profName ], sslStatus )
      profilesModel.ipsecProfiles[profName] = profileModel

   profilesModel = IpsecModel.IpsecProfiles()
   profiles = ipsecIkeConfigDir.ipsecProfile
   if pName is None:
      for pName in profiles:
         fillProfileInfo( pName, profilesModel, profiles )
   else:
      if pName in profiles:
         fillProfileInfo( pName, profilesModel, profiles )
      else:
         mode.addError( f'Profile {pName} has not been configured' )

   return profilesModel

class IpSecurityProfileCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip security profile [ PROFILE_NAME ]'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'security': matcherSecurity,
      'profile': 'Display Profile information',
      'PROFILE_NAME': profileNameMatcher,
   }
   handler = showIpsecProfiles
   cliModel = IpsecModel.IpsecProfiles

BasicCli.addShowCommandClass( IpSecurityProfileCmd )


@haveValidIpsecLicense
def showIpsecKeyControllerPeer( mode, args ):
   ''' function to fill the peer info model '''
   detailed = 'detail' in args
   peerIp = args.get( 'PEER_ADDR' )
   ipsecPeerModel = IpsecModel.IpsecPeerState()
   ipsecPeerModel.fillPeersInfo( ipsecIkeStatusDir, detailed, peerIpStr=peerIp )
   return ipsecPeerModel

matcherKey = CliMatcher.KeywordMatcher( 'key',
      helpdesc='IPsec key information' )
matcherController = CliMatcher.KeywordMatcher( 'controller',
      helpdesc='IPsec key controller information' )

class IpsecKeyControllerPeerCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip security key controller peer [ PEER_ADDR ] [ detail ]'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'security': matcherSecurity,
      'key': matcherKey,
      'controller': matcherController,
      'peer': 'Peer information of the key controller',
      'PEER_ADDR': IpAddrMatcher.IpAddrMatcher( helpdesc='Peer IP address' ),
      'detail': 'Detailed peer information',
   }
   handler = showIpsecKeyControllerPeer
   cliModel = IpsecModel.IpsecPeerState

BasicCli.addShowCommandClass( IpsecKeyControllerPeerCmd )

@haveValidIpsecLicense
def showIpsecKeyControllerLocal( mode, args ):
   ''' function to fill the local key info model'''
   ipsecLocalKeyModel = IpsecModel.IpsecLocalKeyState()
   ipsecLocalKeyModel.fillLocalKeyState( ipsecIkeConfigDir, ipsecIkeStatusDir )
   return ipsecLocalKeyModel

class IpsecKeyControllerLocalCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip security key controller local'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'security': matcherSecurity,
      'key': matcherKey,
      'controller': matcherController,
      'local': 'Local key information of the key controller',
   }
   handler = showIpsecKeyControllerLocal
   cliModel = IpsecModel.IpsecLocalKeyState

BasicCli.addShowCommandClass( IpsecKeyControllerLocalCmd )

TechSupportCli.registerShowTechSupportCmd(
   '2017-03-17 16:14:46',
   cmds=[ 'show ip security applied-profile vrf all',
          'show ip security connection vrf all',
          'show ip security connection vrf all detail',
          'show kernel ipsec vrf all',
          'show kernel ipsec vrf all datapath',
          'show ip security daemon log vrf all',
          'show agent Ipsec logs',
          'show agent Ipsec uptime' ] )

def Plugin( entityManager ):
   global entMan, ipsecPlatformStatus 
   global ipsecIkeConfigDir, ipsecIkeStatusDir, ipsecConnTableDir
   global tunnelIntfStatusDir, ipsecDaemonConfigDir, ipsecDaemonStatusDir
   global ipsecCounterTable, ipsecPathStatus
   global sslStatus

   entMan = entityManager

   ipsecPlatformStatus = LazyMount.mount( entityManager, 
           'ipsec/platform/status', 'Ipsec::Platform::Status', 'r' )
   ipsecDaemonConfigDir = LazyMount.mount( entityManager, 'ipsec/daemonconfig',
                                           'Ipsec::DaemonConfig', 'r' )
   ipsecDaemonStatusDir = LazyMount.mount( entityManager, 'ipsec/daemonstatusdir',
                                           'Ipsec::DaemonStatusDir', 'r' )
   ipsecIkeConfigDir = LazyMount.mount( entityManager, 'ipsec/ike/config',
                                        'Ipsec::Ike::Config', 'r' )
   ipsecIkeStatusDir = LazyMount.mount( entityManager, 'ipsec/ike/status',
                                        'Ipsec::Ike::Status', 'r' )
   ipsecConnTableDir = LazyMount.mount( entityManager,
                                           'ipsec/ike/connectiontable',
                                           'Ipsec::Ike::ConnectionTable', 'r' )
   tunnelIntfStatusDir = LazyMount.mount( entityManager, 
                                           'interface/status/tunnel/intf',
                                           'Interface::TunnelIntfStatusDir', 'r' )
   ipsecCounterTableInfo = SmashLazyMount.mountInfo( 'reader' )
   ipsecCounterTable = SmashLazyMount.mount( entityManager, 'ipsec/countertable',
                                             'IpsecCounters::IpsecCounterTable',
                                             ipsecCounterTableInfo )
   ipsecPathStatus = LazyMount.mount( entityManager, 'ipsec/path/status',
                                      'Ipsec::Path::PathStatusDir', 'r' )
   sslStatus = LazyMount.mount( entityManager, 'mgmt/security/ssl/status',
                                'Mgmt::Security::Ssl::Status', 'r' )

