#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements global DLB configuration.
#-------------------------------------------------------------------------------
'''Configuration commands for global DLB'''

import BasicCli
import CliParser
import CliToken.Ip
import CliToken.Hardware
import CliCommand, ShowCommand, CliMatcher
import LazyMount
import itertools
from CliPlugin import IraCommonCli
from CliPlugin import IraRouteCommon

routingHwDlbStatus = None
routingHardwareStatusCommon = None
routingHardwareStatus = None
aclConfig = None
myEntManager = None

ip = IraRouteCommon.Ip4()
routing = IraRouteCommon.routing( ip )

#-------------------------------------------------------------------------------
# Aliases for some useful token rules.
#-------------------------------------------------------------------------------
configMode = BasicCli.GlobalConfigMode

#-------------------------------------------------------------------------------
# "[no/default] ip hardware fib load-balance distribution [ hash|dynamic]"
#  command
#-------------------------------------------------------------------------------
def dlbEcmpSupportedGuard( mode, token ):
   rhs = routingHardwareStatus
   if rhs is None:
      rhs = mode.sysdbRoot[ 'routing' ][ 'hardware' ][ 'status' ]
   if rhs.dlbEcmpSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def dlbEcmpFlowCliSupportedGuard( mode, token ):
   rhds = routingHwDlbStatus
   if rhds is None:
      rhds = mode.sysdbRoot[ 'routing' ][ 'hardware' ][ 'dlb' ][ 'status' ]
   if rhds.dlbEcmpFlowCliSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def getDlbGroupIdRange( mode, context ):
   # returns the range of dlb group ids that can be provided as input for given 
   # flowset size
   minDlbGroupId = 0
   maxDlbGroupId = routingHwDlbStatus.maxDlbEcmpGroupSize
   #minDlbGroupId 1 for contiguous dlb supported
   if routingHardwareStatus.contiguousDlbIdSupported:
      minDlbGroupId = 1
   if routingHwDlbStatus.flowSetSize == 0:
      #to avoid error when "show ip hardware fib load-balance distribution dynamic
      # group ?" is entered on the terminal while flowSetSize is set to zero
      return ( minDlbGroupId, 127 )
   #max allowable input dlb group ids will reduce by factor of 2 for 
   #each increment in flowSetSize
   return ( minDlbGroupId, 
      maxDlbGroupId // ( 1 << ( routingHwDlbStatus.flowSetSize - 1 ) ) - 1 )

loadBalanceKwMatcher = CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'load-balance',
            helpdesc='Configure load balancing on ECMP routes' ),
         guard=dlbEcmpSupportedGuard )

displayLoadBalanceKwGuard = CliCommand.guardedKeyword( 'load-balance',
      helpdesc='Display load balancing on ECMP routes',
      guard = dlbEcmpSupportedGuard )

flowKwGuard = CliCommand.guardedKeyword( 'flows',
                                    helpdesc='Display active flows',
                                    guard=dlbEcmpFlowCliSupportedGuard )

distributionKwMatcher = CliMatcher.KeywordMatcher(
   'distribution', helpdesc='Configure load balance distribution' )

displayDistributionKwMatcher = CliMatcher.KeywordMatcher(
   'distribution', helpdesc='Display load balance distribution' )

hashKw = CliMatcher.KeywordMatcher( 'hash',
                                    helpdesc='Enable hash based load balancing' )

dynamicKw = CliMatcher.KeywordMatcher( 'dynamic',
                                       helpdesc='Enable dynamic load balancing' )

showDynamicKw = CliMatcher.KeywordMatcher( 'dynamic', 
                                          helpdesc='dynamic load balancing')

groupKw = CliMatcher.KeywordMatcher( 'group',
                                       helpdesc='dynamic load balancing range' )

class FibLoadBalanceDistributionCmd( CliCommand.CliCommandClass ):
   syntax = '''ip hardware fib load-balance distribution ( hash | dynamic )'''
   noOrDefaultSyntax = '''ip hardware fib load-balance distribution ...'''

   data = { "ip": CliToken.Ip.ipMatcherForConfig,
            "hardware": CliToken.Hardware.hardwareForConfigMatcher,
            "fib": IraCommonCli.fibKw,
            "load-balance": loadBalanceKwMatcher,
            "distribution": distributionKwMatcher,
            "hash": hashKw,
            "dynamic": dynamicKw }

   handler = "IraDlbCliHandler.handlerFibLoadBalanceDistributionCmd"
   noOrDefaultHandler = handler

configMode.addCommandClass( FibLoadBalanceDistributionCmd )

#-------------------------------------------------------------------------------
# "show ip hardware fib load-balance distribution" command
#-------------------------------------------------------------------------------
class ShowFibLoadBalanceDistributionCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show ip hardware fib load-balance distribution'''
   data = { 'ip': CliToken.Ip.ipMatcherForShow,
            'hardware': CliToken.Hardware.hardwareForShowMatcher,
            'fib': IraCommonCli.fibKw,
            'load-balance': displayLoadBalanceKwGuard,
            'distribution': displayDistributionKwMatcher
          }

   cliModel = "IraDlbModel.GlobalDlb"
   handler = "IraDlbCliHandler.handlerShowFibLoadBalanceDistributionCmd"

BasicCli.addShowCommandClass( ShowFibLoadBalanceDistributionCmd )

#--------------------------------------------------------------------------------
# "show ip hardware fib load-balance distribution dynamic group flows" command
#-------------------------------------------------------------------------------
class ShowFibLoadBalanceDistributionMacroFlowsCmd(
      ShowCommand.ShowCliCommandClass ):
   syntax = "show ip hardware fib load-balance distribution dynamic group flows"
   data = { 'ip': CliToken.Ip.ipMatcherForShow,
            'hardware': CliToken.Hardware.hardwareForShowMatcher,
            'fib': IraCommonCli.fibKw,
            'load-balance': displayLoadBalanceKwGuard,
            'distribution': displayDistributionKwMatcher,
            'dynamic': showDynamicKw,
            'group': groupKw,
            'flows': flowKwGuard,
          }
   cliModel = "IraDlbModel.MacroFlows"
   handler = "IraDlbCliHandler.handlerShowFibLoadBalanceDistributionMacroFlowsCmd"

BasicCli.addShowCommandClass( ShowFibLoadBalanceDistributionMacroFlowsCmd )

#--------------------------------------------------------------------------------
# "show ip hardware fib load-balance distribution dynamic group RANGE flows"
# command
#-------------------------------------------------------------------------------
class ShowFibLoadBalanceDistributionMacroFlowsDlbIdCmd(
      ShowCommand.ShowCliCommandClass ):
   syntax = "show ip hardware fib load-balance distribution dynamic group RANGE \
flows"
   data = { 'ip': CliToken.Ip.ipMatcherForShow,
            'hardware': CliToken.Hardware.hardwareForShowMatcher,
            'fib': IraCommonCli.fibKw,
            'load-balance': displayLoadBalanceKwGuard,
            'distribution': displayDistributionKwMatcher,
            'dynamic': showDynamicKw,
            'group': groupKw,
            'RANGE': CliMatcher.DynamicIntegerMatcher(
               rangeFn=getDlbGroupIdRange,
               helpname=None,
               helpdesc="Input load-balance id" ),
            'flows': flowKwGuard,
          }
   cliModel = "IraDlbModel.MacroFlows"
   handler = "IraDlbCliHandler.handlerShowFibLoadBalanceDistributionMacroFlowsCmd"

BasicCli.addShowCommandClass( ShowFibLoadBalanceDistributionMacroFlowsDlbIdCmd )



# ---------------------------------------------------------------------------
# " ip hardware fib load-balance distribution dynamic member-selection
#             optimal [ timer | always ] " in exec mode.
#   operational-mode is port_assignment_mode in broadcom specs
# Encoding of DLB operational modes are as follows :
#  0 - Eligibility [ default mode | <cmd> optimal timer ]
#  2 - Packet Spray [ <cmd> optimal always ]
# ---------------------------------------------------------------------------
def constructMemberSelMap( mode ):
   def helpTextVal( mode ):
      memberSelMap = {
         "timer" : "If the inactivity duration has elapsed, pick the "
                         + "optimal member. Else, pick the current assigned member",
         "always" : "Always pick the optimal member, whether or not "
                   + "the inactivity duration has elapsed",
      }
      return memberSelMap
   return helpTextVal( mode )

class FibLoadBalanceDistDynMemberSelCmd( CliCommand.CliCommandClass ):
   syntax = ( 'ip hardware fib load-balance distribution dynamic '
              'member-selection optimal MODE' )
   noOrDefaultSyntax = ( 'ip hardware fib load-balance distribution dynamic '
                         'member-selection ...' )

   data = { "ip": CliToken.Ip.ipMatcherForConfig,
            "hardware": CliToken.Hardware.hardwareForConfigMatcher,
            "fib": IraCommonCli.fibKw,
            "load-balance": loadBalanceKwMatcher,
            "distribution": distributionKwMatcher,
            "dynamic": dynamicKw,
            "member-selection": "Set operational mode",
            "optimal": "Pick optimal port",
            "MODE": CliMatcher.DynamicKeywordMatcher( constructMemberSelMap )
          }
   handler = "IraDlbCliHandler.handlerFibLoadBalanceDistDynMemberSelCmd"
   noOrDefaultHandler = \
      "IraDlbCliHandler.noOrDefaultHandlerFibLoadBalanceDistDynMemberSelCmd"

configMode.addCommandClass( FibLoadBalanceDistDynMemberSelCmd )

#---------------------------------------------------------------------------
# " ip hardware fib load-balance distribution dynamic inactivity-threshold
#                 VALUE mircroseconds" in exec mode.
#   inactivity-threshold is inactivity-duration in broadcom specs
#---------------------------------------------------------------------------

def dlbInactivityDurationRange( mode, context ):
   return ( 16, routingHwDlbStatus.maxInactivityDuration )

class FibLoadBalanceDistDynInactDurCmd( CliCommand.CliCommandClass ):
   syntax = ( 'ip hardware fib load-balance distribution dynamic '
              'inactivity-threshold VALUE microseconds' )
   noOrDefaultSyntax = syntax.replace( 'VALUE microseconds', '...' )

   data = { "ip": CliToken.Ip.ipMatcherForConfig,
            "hardware": CliToken.Hardware.hardwareForConfigMatcher,
            "fib": IraCommonCli.fibKw,
            "load-balance": loadBalanceKwMatcher,
            "distribution": distributionKwMatcher,
            "dynamic": dynamicKw,
            "inactivity-threshold": "Set inactivity threshold",
            "VALUE": CliMatcher.DynamicIntegerMatcher(
                                 rangeFn=dlbInactivityDurationRange,
                                 helpname=None,
                                 helpdesc="Inactivity threshold value "
                                          "in microsecond units" ),
            "microseconds": "Unit in microseconds"
          }

   handler = "IraDlbCliHandler.handlerFibLoadBalanceDistDynInactDurCmd"
   noOrDefaultHandler = \
      "IraDlbCliHandler.noOrDefaultHandlerFibLoadBalanceDistDynInactDurCmd"

configMode.addCommandClass( FibLoadBalanceDistDynInactDurCmd )

#---------------------------------------------------------------------------
# " ip hardware fib load-balance distribution dynamic sampling-period
#                 VALUE microseconds" in exec mode.
#---------------------------------------------------------------------------
class FibLoadBalanceDistDynSamplingPeriodCmd( CliCommand.CliCommandClass ):
   syntax = ( 'ip hardware fib load-balance distribution dynamic '
              'sampling-period VALUE microseconds' )
   noOrDefaultSyntax = syntax.replace( 'VALUE microseconds', '...' )

   data = { "ip": CliToken.Ip.ipMatcherForConfig,
            "hardware": CliToken.Hardware.hardwareForConfigMatcher,
            "fib": IraCommonCli.fibKw,
            "load-balance": loadBalanceKwMatcher,
            "distribution": distributionKwMatcher,
            "dynamic": dynamicKw,
            "sampling-period": "Set sampling period",
            "VALUE": CliMatcher.IntegerMatcher( lbound=1, ubound=255,
                                 helpdesc="Sampling period value "
                                          "in microsecond unit " ),
            "microseconds": "Unit in microseconds"
          }

   handler = "IraDlbCliHandler.handlerFibLoadBalanceDistDynSamplingPeriodCmd"
   noOrDefaultHandler = \
      "IraDlbCliHandler.noOrDefaultHandlerFibLoadBalanceDistDynSamplingPeriodCmd"

configMode.addCommandClass( FibLoadBalanceDistDynSamplingPeriodCmd )

# ---------------------------------------------------------------------------
# " ip hardware fib load-balance distribution dynamic seed VALUE" in exec mode.
# ---------------------------------------------------------------------------
def getDlbEcmpMaxRandomSelectionSeed( mode, context ):
   return ( 0, routingHwDlbStatus.maxRandomSeedSupported )

class FibLoadBalanceDistDynSeedCmd( CliCommand.CliCommandClass ):
   syntax = ( 'ip hardware fib load-balance distribution dynamic '
              'seed VALUE' )
   noOrDefaultSyntax = syntax.replace( 'VALUE', '...' )

   data = { "ip": CliToken.Ip.ipMatcherForConfig,
            "hardware": CliToken.Hardware.hardwareForConfigMatcher,
            "fib": IraCommonCli.fibKw,
            "load-balance": loadBalanceKwMatcher,
            "distribution": distributionKwMatcher,
            "dynamic": dynamicKw,
            "seed": "Set DLB hash seed",
            "VALUE": CliMatcher.DynamicIntegerMatcher(
                                rangeFn=getDlbEcmpMaxRandomSelectionSeed,
                                helpname=None,
                                helpdesc="DLB hash seed" )
           }

   handler = "IraDlbCliHandler.handlerFibLoadBalanceDistDynSeedCmd"
   noOrDefaultHandler = \
      "IraDlbCliHandler.noOrDefaultHandlerFibLoadBalanceDistDynSeedCmd"

configMode.addCommandClass( FibLoadBalanceDistDynSeedCmd )

#---------------------------------------------------------------------------
# " ip hardware fib load-balance distribution dynamic average-traffic-weight
#                 VALUE" in exec mode.
#    average-traffic-weight is port-loading-weight in broadcom term
#---------------------------------------------------------------------------
class FibLoadBalanceDistDynPortLoadingWtCmd( CliCommand.CliCommandClass ):
   syntax = ( 'ip hardware fib load-balance distribution dynamic '
              'average-traffic-weight VALUE' )
   noOrDefaultSyntax = syntax.replace( 'VALUE', '...' )

   data = { "ip": CliToken.Ip.ipMatcherForConfig,
            "hardware": CliToken.Hardware.hardwareForConfigMatcher,
            "fib": IraCommonCli.fibKw,
            "load-balance": loadBalanceKwMatcher,
            "distribution": distributionKwMatcher,
            "dynamic": dynamicKw,
            "average-traffic-weight": "Set Average traffic weight",
            "VALUE": CliMatcher.IntegerMatcher( lbound=1, ubound=15,
                              helpdesc="Average traffic weight value" )
          }

   handler = "IraDlbCliHandler.handlerFibLoadBalanceDistDynPortLoadingWtCmd"
   noOrDefaultHandler = \
      "IraDlbCliHandler.noOrDefaultHandlerFibLoadBalanceDistDynPortLoadingWtCmd"

configMode.addCommandClass( FibLoadBalanceDistDynPortLoadingWtCmd )

# ---------------------------------------------------------------------------
# " ip hardware fib load-balance distribution dynamic flow-set-size VALUE"
#                              in exec mode.
# ---------------------------------------------------------------------------
def constructFlowSetSizeVal( mode ):
   def helpTextVal( mode ):
      rhs = routingHardwareStatus
      if rhs is None:
         rhs = mode.sysdbRoot[ 'routing' ][ 'hardware' ][ 'status' ]
      if rhs.contiguousDlbIdSupported:
         oneEntryUsed = 1
      else:
         oneEntryUsed = 0
      setSizeMap = {
         "1": f"Allow up to {128 - oneEntryUsed} ECMP groups of 256 entries each",
         "2": f"Allow up to {64 - oneEntryUsed} ECMP groups of 512 entries each",
         "3": f"Allow up to {32 - oneEntryUsed} ECMP groups of 1024 entries each",
         "4": f"Allow up to {16 - oneEntryUsed} ECMP groups of 2048 entries each",
         "5": f"Allow up to {8 - oneEntryUsed} ECMP groups of 4096 entries each",
         "6": f"Allow up to {4 - oneEntryUsed} ECMP groups of 8192 entries each",
         "7": "Allow up to 2 ECMP groups of 16384 entries each"
              if oneEntryUsed == 0 else "Allow up to 1 ECMP group of 16384 entries"
      }
      return setSizeMap
   return helpTextVal( mode )

class FibLoadBalanceDistDynFlowSetSizeCmd( CliCommand.CliCommandClass ):
   syntax = ( 'ip hardware fib load-balance distribution dynamic '
              'flow-set-size VALUE' )
   noOrDefaultSyntax = syntax.replace( 'VALUE', '...' )

   data = { "ip": CliToken.Ip.ipMatcherForConfig,
            "hardware": CliToken.Hardware.hardwareForConfigMatcher,
            "fib": IraCommonCli.fibKw,
            "load-balance": loadBalanceKwMatcher,
            "distribution": distributionKwMatcher,
            "dynamic": dynamicKw,
            "flow-set-size": "Set flow set size",
            "VALUE": CliMatcher.DynamicKeywordMatcher( constructFlowSetSizeVal )
          }

   handler = "IraDlbCliHandler.handlerFibLoadBalanceDistDynFlowSetSizeCmd"
   noOrDefaultHandler = \
      "IraDlbCliHandler.noOrDefaultHandlerFibLoadBalanceDistDynFlowSetSizeCmd"

configMode.addCommandClass( FibLoadBalanceDistDynFlowSetSizeCmd )

# ---------------------------------------------------------------------------
# "[no/default] ip hardware fib load-balance distribution dynamic
#     (ipv4|ipv6) access-group <acl>" in exec mode.
# ---------------------------------------------------------------------------

def selectiveDlbSupportedGuard( mode, token ):
   rhs = routingHardwareStatus
   if rhs is None:
      rhs = mode.sysdbRoot[ 'routing' ][ 'hardware' ][ 'status' ]
   if rhs.dlbSelectiveSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

selectiveDlbIpv4Kw = CliCommand.guardedKeyword(
   'ipv4',
   helpdesc='Configuration for IPv4',
   guard=selectiveDlbSupportedGuard )

selectiveDlbIpv6Kw = CliCommand.guardedKeyword(
   'ipv6',
   helpdesc='Configuration for IPv6',
   guard=selectiveDlbSupportedGuard )

# This is already defined in AclCli, but Acl depends on Ira, so
# using it here would create a circular dependency
ipAclNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: itertools.chain( aclConfig.config[ 'ip' ].acl,
                                 aclConfig.config[ 'ipv6' ].acl ),
   'Access list name', priority=CliParser.PRIO_LOW )

class FibLoadBalanceDynamicAclCmd( CliCommand.CliCommandClass ):
   syntax = ( 'ip hardware fib load-balance distribution dynamic '
              '( ipv4 | ipv6 ) access-group ACLNAME' )
   noOrDefaultSyntax = ( 'ip hardware fib load-balance distribution dynamic '
                         '( ipv4 | ipv6 ) ...' )
   data = {
      "ip": CliToken.Ip.ipMatcherForConfig,
      "hardware": CliToken.Hardware.hardwareForConfigMatcher,
      "fib": IraCommonCli.fibKw,
      "load-balance": loadBalanceKwMatcher,
      "distribution": distributionKwMatcher,
      "dynamic": dynamicKw,
      "ipv4": selectiveDlbIpv4Kw,
      "ipv6": selectiveDlbIpv6Kw,
      "access-group": "Configure access control list",
      "ACLNAME": ipAclNameMatcher,
   }

   handler = "IraDlbCliHandler.handlerFibLoadBalanceAclCmd"
   noOrDefaultHandler = "IraDlbCliHandler.noOrDefaultHandlerFibLoadBalanceAclCmd"

configMode.addCommandClass( FibLoadBalanceDynamicAclCmd )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   routing.plugin( entityManager )

   global routingHwDlbStatus
   global routingHardwareStatusCommon
   global routingHardwareStatus
   global aclConfig
   global myEntManager

   myEntManager = entityManager

   routingHwDlbStatus = LazyMount.mount( entityManager,
                                         "routing/hardware/dlb/status",
                                         "Routing::Hardware::DlbStatus", "r" )
   routingHardwareStatusCommon = LazyMount.mount(
      entityManager,
      "routing/hardware/statuscommon",
      "Routing::Hardware::StatusCommon", "r" )
   routingHardwareStatus = LazyMount.mount( entityManager,
                                            "routing/hardware/status",
                                            "Routing::Hardware::Status", "r" )


   aclConfig = LazyMount.mount( entityManager,
                                "acl/config/cli",
                                "Acl::Input::Config", "r" )
