#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
from CliMatcher import KeywordMatcher
import CliParser
import CliToken.Ip
import LazyMount
from Toggles.RoutingLibToggleLib import toggleFlexibleEcmpJ2Enabled

from CliPlugin.IraCommonCli import fibMatcher
from CliPlugin import IraRouteCommon

routingHardwareStatus = None

ip = IraRouteCommon.Ip4()
routing = IraRouteCommon.routing( ip )

def guardEcmpMatcher( mode, token ):
   rhs = routingHardwareStatus
   if rhs is None:
      rhs = mode.sysdbRoot[ 'routing' ][ 'hardware' ][ 'status' ]

   if rhs.flexibleEcmpSupported and toggleFlexibleEcmpJ2Enabled():
      return None
   return CliParser.guardNotThisPlatform

# -------------------------------------------------------------------------------
# Matchers for CLI
# -------------------------------------------------------------------------------
ipForConfigMatcher = CliToken.Ip.ipMatcherForConfig
hardwareMatcher = CliToken.Hardware.hardwareForConfigMatcher
hierarchicalMatcher = KeywordMatcher( 'hierarchical',
                                      helpdesc="Hardware resource configuration" )
nextHopMatcher = KeywordMatcher(
                     'next-hop',
                     helpdesc='Next-hop configuration' )
ecmpMatcher = CliCommand.guardedKeyword(
   'ecmp', helpdesc='ECMP bank level configuration mode', guard=guardEcmpMatcher )

# -------------------------------------------------------------------------------
# ip hardware fib hierarchical next-hop ecmp
# -------------------------------------------------------------------------------
class NexthopEcmpFecModeCmd( CliCommand.CliCommandClass ):
   syntax = 'ip hardware fib hierarchical next-hop ecmp'
   noOrDefaultSyntax = syntax
   data = {
      'ip': ipForConfigMatcher,
      'hardware': hardwareMatcher,
      'fib': fibMatcher,
      'hierarchical': hierarchicalMatcher,
      'next-hop': nextHopMatcher,
      'ecmp': ecmpMatcher
   }

   handler = 'IraEcmpFecCli.gotoNexthopEcmpFecMode'
   noOrDefaultHandler = 'IraEcmpFecCli.HierNhEcmpFecMode.deleteModeConfig'

if toggleFlexibleEcmpJ2Enabled():
   BasicCli.GlobalConfigMode.addCommandClass( NexthopEcmpFecModeCmd )

# -------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
# -------------------------------------------------------------------------------
def Plugin( entityManager ):
   routing.plugin( entityManager )

   global routingHardwareStatus
   routingHardwareStatus = LazyMount.mount( entityManager,
                                            "routing/hardware/status",
                                            "Routing::Hardware::Status", "r" )
