# Copyright (c) 2009-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.


#-------------------------------------------------------------------------------
# This module implements interface-specific IPv6 configuration.
#-------------------------------------------------------------------------------
'''Commands supported in interface config mode'''

import CliParser
from CliPlugin import IntfCli
from CliPlugin import Ip6AddrMatcher
from CliPlugin import IraCommonCli
from CliPlugin import IraUrpfCli
from CliPlugin.IraIp6Model import InterfaceAddressIp6
from CliPlugin.IraIp6Model import Ip6Status
from CliPlugin.IraIp6Model import IpAddress
from CliPlugin.VrfCli import vrfMatcher
import CliCommand
import CliMatcher
import LazyMount
import Tracing
import ConfigMount
import Tac
import os
import CliToken.Ipv6, CliToken.Verify
import CliExtensions
import Arnet
import IpLibTypes
import Toggles.IraToggleLib

ip6StatusDir = None
ip6ConfigDir = None
dynIp6ConfigDir = None
allIntfStatusDir = None
ip4ConfigDir = None
routingHardwareStatus = None
routing6HardwareStatus = None
l3ConfigDir = None
arpCliStatus = None

#Getting additional information from Ipv6RouterAdvt/CliPlugin/Ip6RaCli.py
#Interface name is the parameter that needs to be passed to this procedure
extraIp6NdRaInfoHook = lambda intf, result: None

addrSource = IpLibTypes.addrSource( Tac.Type( 'Arnet::AddressFamily' ).ipv6 )
acceptDad = Tac.Type( 'Ip6::AcceptDad' )

@Tac.memoize
def _mgmtPrefix():
   # MGMT_INTF_PREFIX allows a test to override the prefix that
   # identifies a management interface
   return os.environ.get( "MGMT_INTF_PREFIX", "Management" )

def isManagement( name ):
   return name.startswith( _mgmtPrefix() )

def checkLLAddrAndPrintWarning( mode, addrArgWithMask ):
   if not addrArgWithMask.address.isLinkLocal:
      mode.addWarning( " Address is not a link local address."
                       " Ignoring this command" )
      return False

   if addrArgWithMask.len != 64:
      mode.addWarning( " Address with only /64 mask allowed as link local."
                       " Ignoring this command" )
      return False

   return True

def isLinkLocalAddress( mode, address ):
   try:
      return Arnet.Ip6Addr( address ).isLinkLocal
   except IndexError:
      return False

#-------------------------------------------------------------------------------
# Adds IPv6-specific CLI commands to the "config-if" mode.
#-------------------------------------------------------------------------------

def ipv6SupportedOnIntfConfigMode( intfConfigMode ):
   # Don't configure ipv6 on ineligible interfaces
   return intfConfigMode.intf.routingSupported()

class Ip6IntfConfigModelet( CliParser.Modelet ):
   def __init__( self, mode ):
      CliParser.Modelet.__init__( self, mode )
      self.intf = Ip6Intf( mode.intf, mode.sysdbRoot, createIfMissing=True )

   shouldAddModeletRule = staticmethod( ipv6SupportedOnIntfConfigMode )

   def enableIpv6( self, args ):
      Tracing.trace0( "Enable Ipv6 on", self.intf.name )
      IraCommonCli.printWarningIfNotRoutedPort( self.mode,
                        self.intf.name, configType="IPv6" )
      self.intf.config().enable = True

   def noIpv6Enable( self, args ):
      Tracing.trace0( "Disabling IPv6 on", self.intf.name )
      self.intf.config().enable = False

   def setIp6Addr( self, args ):
      ip6Prefix = args[ 'IP6PREFIX' ]
      Tracing.trace0( "Adding IP6 address", ip6Prefix, "on", self.intf.name )
      IraCommonCli.printWarningIfNotRoutedPort( self.mode,
                                          self.intf.name, configType="IPv6" )
      if self.intf.config().addrSource == addrSource.slaac:
         self.intf.config().addr.clear()
         self.intf.config().addrSource = addrSource.manual
      # deprecated is not set if ns source is configured
      setNsSrc = "ns" in args
      deprecated = not setNsSrc and "source-lp" in args
      self.intf.setAddr( ip6Prefix, deprecated=deprecated )
      if setNsSrc:
         self.intf.setNsSrc( ip6Prefix.address )
      elif self.intf.getNsSrc() == ip6Prefix.address:
         # Only reset if this is the current NS source address
         self.intf.resetNsSrc()

   def setIp6AddrOptions( self, args ):
      # Use the same handler to avoid config churns
      self.setIp6Addr( args )

   def noIp6AddrOptions( self, args ):
      ip6Prefix = args[ 'IP6PREFIX' ]
      # check for 'ns source' and 'source least-preferred'
      if 'ns' in args:
         if self.intf.getNsSrc() == ip6Prefix.address:
            # Only reset if this is the current NS source address
            self.intf.resetNsSrc()
      elif 'source-lp' in args:
         self.intf.resetDeprecate( ip6Prefix )
      else:
         self.mode.addError( 'Unexpected IPv6 address option' )

   def enableSlaac( self, args ):
      Tracing.trace0( "Enabling SLAAC on", self.intf.name )
      IraCommonCli.printWarningIfNotRoutedPort( self.mode,
                                          self.intf.name, configType="IPv6" )
      # pylint: disable-next=superfluous-parens
      if not ( self.intf.config().addrSource == addrSource.slaac ):
         self.intf.config().addr.clear()
         self.intf.config().addrSource = addrSource.slaac

   def disableSlaac( self, args ):
      Tracing.trace0( "Disabling SLAAC on", self.intf.name )
      if self.intf.config().addrSource == addrSource.slaac:
         self.intf.config().addr.clear()
         self.intf.config().addrSource = addrSource.manual

   def noIp6Addr( self, args ):
      ip6Prefix = args[ 'IP6PREFIX' ]
      if self.intf.config().addrSource == addrSource.slaac:
         self.mode.addError( 'Address determined by SLAAC' )
         return
      self.intf.delAddr( ip6Prefix )
      Tracing.trace0( "Removing IPv6 address", ip6Prefix, "on", self.intf.name )

   def noIp6Addrs( self, args ):
      self.intf.config().addrSource = addrSource.manual
      self.intf.delAddrs()
      Tracing.trace0( "Removing Ipv6 addresses on ", self.intf.name )

   def enableAcceptRaDefRtr( self, args ):
      Tracing.trace0( "Accepting default router from RA on", self.intf.name )
      IraCommonCli.printWarningIfNotRoutedPort( self.mode,
                                          self.intf.name, configType="IPv6" )
      self.intf.config().acceptRaDefRtr = True

   def disableAcceptRaDefRtr( self, args ):
      Tracing.trace0( "Not accepting default router from RA on", self.intf.name )
      self.intf.config().acceptRaDefRtr = False

   def enableAcceptRaRtrPref( self, args ):
      Tracing.trace0( "Accepting route preference from RA on", self.intf.name )
      IraCommonCli.printWarningIfNotRoutedPort( self.mode,
                                          self.intf.name, configType="IPv6" )
      self.intf.config().acceptRaRtrPref = True

   def disableAcceptRaRtrPref( self, args ):
      Tracing.trace0( "Not accepting route preference from RA on", self.intf.name )
      self.intf.config().acceptRaRtrPref = False

   def ndNsRetransmitInterval( self, args ):
      ndNsRetransmitIntervalValue = args[ 'INTERVAL' ]
      Tracing.trace0( "Setting ns-interval on ", self.intf.name, " to ",
            ndNsRetransmitIntervalValue )
      IraCommonCli.printWarningIfNotRoutedPort( self.mode,
                                       self.intf.name, configType="IPv6" )
      self.intf.ndNsRetransmitInterval( ndNsRetransmitIntervalValue )

   def noNdNsRetransmitInterval( self, args ):
      Tracing.trace0( "Default ns-interval of 0 on ", self.intf.name )
      self.intf.noNdNsRetransmitInterval()

   def ndDad( self, args ):
      ndDadValue = args.get( 'DAD_VALUE', 'enabled' )
      Tracing.trace0( "Setting accept-dad on ", self.intf.name, "to", ndDadValue )
      IraCommonCli.printWarningIfNotRoutedPort( self.mode,
                                       self.intf.name, configType="IPv6" )
      if ndDadValue == 'disabled':
         self.intf.ndDadDisabled()
      elif ndDadValue == 'strict':
         self.intf.ndDadStrict()
      else:
         self.intf.ndDad()

   def noNdDad( self, args ):
      Tracing.trace0( "No accept-dad on ", self.intf.name )
      self.intf.noNdDad()

   def defaultNdDad( self, args ):
      Tracing.trace0( "Default accept-dad on ", self.intf.name )
      self.intf.defaultNdDad()

   def enableUnsolicitedNaAccept( self, args ):
      Tracing.trace0( "Accepting unsolicited NA on", self.intf.name )
      IraCommonCli.printWarningIfNotRoutedPort( self.mode,
                                          self.intf.name, configType="IPv6" )
      self.intf.config().acceptUnsolicitedNa = True

   def disableUnsolicitedNaAccept( self, args ):
      Tracing.trace0( "Not accepting unsolicited NA on", self.intf.name )
      self.intf.config().acceptUnsolicitedNa = False

   def setUrpf( self, args ):
      urpfMode = ( IraUrpfCli.urpfMode.loose if 'any' in args
                                             else IraUrpfCli.urpfMode.strict )
      allowDefault = 'allow-default' in args or \
                     ( urpfMode == IraUrpfCli.urpfMode.loose and
                       'non-default' not in args )
      IraUrpfCli.doUrpf( self.intf, False, urpfMode,
                         allowDefault, args.get( 'ACL_NAME' ),
                         args.get( 'VRF_NAME' ) )

   def noUrpf( self, args ):
      IraUrpfCli.doUrpf( self.intf, no=True )

   def setIp6LLaddr( self, args ):
      ip6Prefix = args[ 'IP6PREFIX' ]
      if checkLLAddrAndPrintWarning( self.mode, ip6Prefix ):
         self.intf.setLLAddr( ip6Prefix )

   def setIp6LLaddrNoMask( self, args ):
      ip6Addr = args[ 'IP6ADDR' ]
      ip6Prefix = Arnet.Ip6AddrWithMask( ip6Addr, mask=64 )
      if checkLLAddrAndPrintWarning( self.mode, ip6Prefix ):
         self.intf.setLLAddr( ip6Prefix )

   def noIp6LLAddr( self, args ):
      ip6Prefix = args[ 'IP6PREFIX' ]
      if checkLLAddrAndPrintWarning( self.mode, ip6Prefix ):
         self.intf.delLLAddr( ip6Prefix )

   def noIp6LLAddrNoMask( self, args ):
      ip6Addr = args[ 'IP6ADDR' ]
      ip6Prefix = Arnet.Ip6AddrWithMask (ip6Addr, mask=64)
      if checkLLAddrAndPrintWarning( self.mode, ip6Prefix ):
         self.intf.delLLAddr( ip6Prefix )

   def attachedRoutes( self, args ):
      self.intf.attachedRoutes()

   def defaultAttachedRoutes( self, args ):
      self.intf.defaultAttachedRoutes()

   def noAttachedRoutes( self, args ):
      self.intf.noAttachedRoutes()


#-------------------------------------------------------------------------------
# Associate the Ip6IntfConfigModelet with the "config-if" mode.
#-------------------------------------------------------------------------------
IntfCli.IntfConfigMode.addModelet( Ip6IntfConfigModelet )

#-------------------------------------------------------------------------------
# Adds IPv6-specific CLI commands to the "config-if" mode for routed ports.
#-------------------------------------------------------------------------------
class RoutingProtocolIntfConfigModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      # we want to exclude management interfaces and loopback interfaces
      return ( mode.intf.routingSupported() and
               not mode.intf.name.startswith( "Management" ) and
               not mode.intf.name.startswith( "Loopback" ) )

#-------------------------------------------------------------------------------
# Associate the RoutingProtocolIntfConfigModelet with the "config-if" mode.
#-------------------------------------------------------------------------------
IntfCli.IntfConfigMode.addModelet( RoutingProtocolIntfConfigModelet )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 enable
#--------------------------------------------------------------------------------
class Ipv6EnableCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 enable'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'enable': 'Enable IPv6 on an interface',
   }

   handler = Ip6IntfConfigModelet.enableIpv6
   noOrDefaultHandler = Ip6IntfConfigModelet.noIpv6Enable

Ip6IntfConfigModelet.addCommandClass( Ipv6EnableCmd )

matcherAddress = CliMatcher.KeywordMatcher( 'address',
      helpdesc='Set the IPv6 address of an interface' )

matcherNsSource = CliMatcher.KeywordMatcher( 'source',
      helpdesc='Make this IPv6 address the source of all outgoing '
      'ICMPv6 neighbour solicitations' )

matcherSourceLeastPreferred = CliMatcher.KeywordMatcher( 'source',
      helpdesc='Source IPv6 address' )

# --------------------------------------------------------------------------------
# [ no | default ] ipv6 address IP6PREFIX ns source
# [ no | default ] ipv6 address IP6PREFIX ( ns source | source-lp least-preferred )
# --------------------------------------------------------------------------------
class Ipv6AddressIp6PrefixOptionsCmd( CliCommand.CliCommandClass ):
   if Toggles.IraToggleLib.toggleIPv6AddressDeprecationEnabled():
      syntax = ( 'ipv6 address IP6PREFIX'
            '( ( ns source ) | ( source-lp least-preferred ) )' )
   else:
      syntax = 'ipv6 address IP6PREFIX ns source'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'address': matcherAddress,
      'IP6PREFIX': Ip6AddrMatcher.ip6PrefixMatcher,
      'ns': 'ICMPv6 Neighbour Solicitation Options',
      'source': matcherNsSource,
      'source-lp': matcherSourceLeastPreferred,
      'least-preferred': 'Least preferred address'
   }

   handler = Ip6IntfConfigModelet.setIp6AddrOptions
   noOrDefaultHandler = Ip6IntfConfigModelet.noIp6AddrOptions

Ip6IntfConfigModelet.addCommandClass( Ipv6AddressIp6PrefixOptionsCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 address IP6PREFIX
#--------------------------------------------------------------------------------
class Ipv6AddressIp6PrefixCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 address IP6PREFIX'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'address': matcherAddress,
      'IP6PREFIX': Ip6AddrMatcher.ip6PrefixMatcher,
   }

   handler = Ip6IntfConfigModelet.setIp6Addr
   noOrDefaultHandler = Ip6IntfConfigModelet.noIp6Addr

Ip6IntfConfigModelet.addCommandClass( Ipv6AddressIp6PrefixCmd )

#--------------------------------------------------------------------------------
# ( no | default ) ipv6 address
#--------------------------------------------------------------------------------
class Ipv6AddressCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'ipv6 address'
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'address': matcherAddress,
   }

   noOrDefaultHandler = Ip6IntfConfigModelet.noIp6Addrs

Ip6IntfConfigModelet.addCommandClass( Ipv6AddressCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 address auto-config
#--------------------------------------------------------------------------------
class Ipv6AddressAutoConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 address auto-config'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'address': matcherAddress,
      'auto-config': 'Use SLAAC to automatically configure the IPv6 address',
   }

   handler = Ip6IntfConfigModelet.enableSlaac
   noOrDefaultHandler = Ip6IntfConfigModelet.disableSlaac

Ip6IntfConfigModelet.addCommandClass( Ipv6AddressAutoConfigCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 address IP6PREFIX link-local
#--------------------------------------------------------------------------------
class Ipv6AddressIp6PrefixLlCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 address IP6PREFIX link-local'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'address': matcherAddress,
      'IP6PREFIX': Ip6AddrMatcher.ip6PrefixMatcher,
      'link-local': 'Use link-local address',
   }

   handler = Ip6IntfConfigModelet.setIp6LLaddr
   noOrDefaultHandler = Ip6IntfConfigModelet.noIp6LLAddr

Ip6IntfConfigModelet.addCommandClass( Ipv6AddressIp6PrefixLlCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 address IP6ADDR link-local
#--------------------------------------------------------------------------------
class Ipv6AddressIp6AddrLinkLocalCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 address IP6ADDR link-local'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'address': matcherAddress,
      'IP6ADDR': Ip6AddrMatcher.ip6AddrMatcher,
      'link-local': 'Use link-local address',
   }

   handler = Ip6IntfConfigModelet.setIp6LLaddrNoMask
   noOrDefaultHandler = Ip6IntfConfigModelet.noIp6LLAddrNoMask

Ip6IntfConfigModelet.addCommandClass( Ipv6AddressIp6AddrLinkLocalCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 nd ra rx accept default-route
#--------------------------------------------------------------------------------
ndMatcher = CliMatcher.KeywordMatcher( 'nd',
      helpdesc='Neighbor Discovery / Router Advertisement' )
raMatcher = CliMatcher.KeywordMatcher( 'ra',
      helpdesc='Router Advertisement commands' )
naMatcher = CliMatcher.KeywordMatcher( 'na',
      helpdesc='Neighbor Advertisement commands' )
rxMatcher = CliMatcher.KeywordMatcher( 'rx',
      helpdesc='Configure action on receiving RA' )
acceptRaMatcher = CliMatcher.KeywordMatcher( 'accept',
      helpdesc='Accept information on received RA' )
unsolicitedNaMatcher = CliMatcher.KeywordMatcher( 'unsolicited',
      helpdesc='Unsolicited Neighbor Advertisement' )
acceptNaMatcher = CliMatcher.KeywordMatcher( 'accept',
      helpdesc='Accept information on received NA' )

class Ipv6NdRaRxAcceptDefaultRouteCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 nd ra rx accept default-route'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'nd': ndMatcher,
      'ra': raMatcher,
      'rx': rxMatcher,
      'accept': acceptRaMatcher,
      'default-route': 'Accept default router information from RA',
   }

   handler = Ip6IntfConfigModelet.enableAcceptRaDefRtr
   noOrDefaultHandler = Ip6IntfConfigModelet.disableAcceptRaDefRtr

Ip6IntfConfigModelet.addCommandClass( Ipv6NdRaRxAcceptDefaultRouteCmd )

# --------------------------------------------------------------------------------
# [ no | default ] ipv6 nd ra rx accept route-preference
# --------------------------------------------------------------------------------
class Ipv6NdRaRxAcceptRoutePreferenceCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 nd ra rx accept route-preference'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'nd': ndMatcher,
      'ra': raMatcher,
      'rx': rxMatcher,
      'accept': acceptRaMatcher,
      'route-preference': 'Accept router preference information from RA',
   }

   handler = Ip6IntfConfigModelet.enableAcceptRaRtrPref
   noOrDefaultHandler = Ip6IntfConfigModelet.disableAcceptRaRtrPref

Ip6IntfConfigModelet.addCommandClass( Ipv6NdRaRxAcceptRoutePreferenceCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 nd ns-interval INTERVAL
#--------------------------------------------------------------------------------
class NdNsRetransmitIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 nd ns-interval INTERVAL'
   noOrDefaultSyntax = 'ipv6 nd ns-interval ...'
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'nd': ndMatcher,
      'ns-interval': 'Time between neighbor solicitation retransmissions',
      'INTERVAL': CliMatcher.IntegerMatcher( 1000, 0xffffffff,
         helpdesc='Milliseconds' ),
   }

   handler = Ip6IntfConfigModelet.ndNsRetransmitInterval
   noOrDefaultHandler = Ip6IntfConfigModelet.noNdNsRetransmitInterval

Ip6IntfConfigModelet.addCommandClass( NdNsRetransmitIntervalCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 nd dad [ DAD_VALUE ]
#--------------------------------------------------------------------------------
class Ipv6NdDadCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 nd dad [ DAD_VALUE ]'
   noOrDefaultSyntax = 'ipv6 nd dad ...'
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'nd': ndMatcher,
      'dad': 'Configure parameters for duplicate address detection mechanism',
      'DAD_VALUE': CliMatcher.EnumMatcher( {
         'disabled': 'Disable DAD',
         'strict': ( 'Disable the IPv6 interface on detecting a conflict of the '
                      'MAC address based autogenerated link-local address' ),
      } ),
   }

   handler = Ip6IntfConfigModelet.ndDad
   noHandler = Ip6IntfConfigModelet.noNdDad
   defaultHandler = Ip6IntfConfigModelet.defaultNdDad

Ip6IntfConfigModelet.addCommandClass( Ipv6NdDadCmd )

# --------------------------------------------------------------------------------
# [ no | default ] ipv6 nd na unsolicited accept
# --------------------------------------------------------------------------------
class Ipv6NdUnsolicitedNaAcceptCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 nd na unsolicited accept'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'nd': ndMatcher,
      'na': naMatcher,
      'unsolicited': unsolicitedNaMatcher,
      'accept': acceptNaMatcher,
   }

   handler = Ip6IntfConfigModelet.enableUnsolicitedNaAccept
   noOrDefaultHandler = Ip6IntfConfigModelet.disableUnsolicitedNaAccept

Ip6IntfConfigModelet.addCommandClass( Ipv6NdUnsolicitedNaAcceptCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 verify unicast source reachable-via (
#     ( any [ non-default ] [ lookup-vrf VRF_NAME ] [ exception-list ACL_NAME ] ) |
#     ( rx [ allow-default ] [ exception-list ACL_NAME ] ) )
#--------------------------------------------------------------------------------
class UrpfCmd( CliCommand.CliCommandClass ):
   _nonDefaultKwStr = '[ non-default ] ' \
         if Toggles.IraToggleLib.toggleUrpfIgnoreDefaultRouteEnabled() else ''
   _splitVrfUrpfKwStr = '[ lookup-vrf VRF_NAME ] ' \
          if Toggles.IraToggleLib.toggleSplitVrfUrpfResolutionEnabled() else ''
   # pylint: disable-next=consider-using-f-string
   syntax = ( 'ipv6 verify unicast source reachable-via ('
                        '( any %s%s[ exception-list ACL_NAME ] ) |'
                        '( rx [ allow-default ] [ exception-list ACL_NAME ] ) )' %
                        ( _nonDefaultKwStr, _splitVrfUrpfKwStr ) )
   noOrDefaultSyntax = 'ipv6 verify unicast ...'
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'verify': CliToken.Verify.verifyMatcherForConfigIf,
      'unicast': CliCommand.guardedKeyword( 'unicast',
         helpdesc='Enable RPF on unicast traffic',
         guard=IraUrpfCli.urpf6SupportedGuard ),
      'source': IraUrpfCli.urpfSourceMatcher,
      'reachable-via': IraUrpfCli.urpfReachMatcher,
      'any': IraUrpfCli.urpfLooseMatcher,
      'non-default': IraUrpfCli.urpfLooseNonDefaultMatcher,
      'lookup-vrf': IraUrpfCli.urpfLooseLookupVrfMatcher,
      'VRF_NAME': vrfMatcher,
      'rx': IraUrpfCli.urpfStrictMatcher,
      'allow-default': IraUrpfCli.urpfAllowDefaultMatcher,
      'exception-list': CliCommand.guardedKeyword( 'exception-list',
         helpdesc='Exception list',
         guard=IraUrpfCli.urpf6ExceptionSupportedGuard ),
      'ACL_NAME': CliMatcher.DynamicNameMatcher(
         lambda mode: IraUrpfCli.getUserAclNames( [ 'ipv6' ] ),
         helpdesc='Exception Acl' )
   }

   handler = Ip6IntfConfigModelet.setUrpf
   noOrDefaultHandler = Ip6IntfConfigModelet.noUrpf

Ip6IntfConfigModelet.addCommandClass( UrpfCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 attached-routes
#--------------------------------------------------------------------------------
class Ipv6AttachedRoutesCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 attached-routes'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'attached-routes': 'Install attached routes in FIB',
   }

   handler = Ip6IntfConfigModelet.attachedRoutes
   noHandler = Ip6IntfConfigModelet.noAttachedRoutes
   defaultHandler = Ip6IntfConfigModelet.defaultAttachedRoutes

Ip6IntfConfigModelet.addCommandClass( Ipv6AttachedRoutesCmd )

#-------------------------------------------------------------------------------
# Network-layer interface extension for IPv6.
#-------------------------------------------------------------------------------

# canSetIntfIpHook  allows other packages to check if a proposed ip
# address (with mask) is acceptable.  The hook takes an Arnet::Ip6AddrWithMask
# and an interface name as arguments and returns a pair [ b, str ] where b is
# True if the proposed address is acceptable, False otherwise, and str is an
# error or warning string suitable for display in the CLI.
canSetIntfIpHook = CliExtensions.CliHook()
canSetIntfLLAddrHook = CliExtensions.CliHook()
canRemoveIp6AddrsHook = CliExtensions.CliHook()
canDeleteIntfIpHook = CliExtensions.CliHook()

class Ip6Intf( IntfCli.IntfDependentBase ):
   emptyIntf = None
   emptyL3Intf = None
   def __init__( self, intf, sysdbRoot, createIfMissing=False ):
      self.intfStatuses = ip6StatusDir.intf
      self.intfStatus = None
      self.createIfMissing_ = createIfMissing
      self.mode = intf.mode_
      IntfCli.IntfDependentBase.__init__( self, intf, sysdbRoot )

   name = property( lambda self: self.intf_.name )

   #----------------------------------------------------------------------------
   # Returns whether the Ip6::IntfStatus object exists for this interface.
   #----------------------------------------------------------------------------
   def lookup( self ):
      # The Ip6Intf exists if an associated Ip6::IntfStatus object exists. Since
      # Ira is the one responsible for creating the status object, there is a
      # window in which the ip6 intf has been configured in the cli, but
      # 'show interfaces' does not display the ip6 intf status.
      self.intfStatus = self.intfStatuses.get( self.intf_.name )
      return self.intfStatus is not None

   #----------------------------------------------------------------------------
   # Returns the Ip6::IntfConfig object for this interface, creating it if it does
   # not already exist.  Will also create L3Config as side effect
   #----------------------------------------------------------------------------
   def config( self ):
      ( ip6Config, l3Config ) = self.bothConfigs()
      assert ip6Config and l3Config
      return ip6Config

   #----------------------------------------------------------------------------
   # Returns the L3Config object for this interface, creating it if it does
   # not already exist.  Will also create Ip6IntfConfig as a side effect
   #----------------------------------------------------------------------------
   def l3Config( self ):
      ( ip6Config, l3Config ) = self.bothConfigs()
      assert ip6Config and l3Config
      return l3Config

   def bothConfigs( self ):
      l3c = l3ConfigDir.intfConfig.get( self.intf_.name )
      if not l3c:
         if self.createIfMissing_:
            l3c = l3ConfigDir.intfConfig.newMember( self.intf_.name )
         else:
            if not Ip6Intf.emptyL3Intf:
               l3ConfigDirTmp = Tac.newInstance( "L3::Intf::ConfigDir", "" )
               Ip6Intf.emptyL3Intf = l3ConfigDirTmp.intfConfig.newMember( "" )
            l3c = Ip6Intf.emptyL3Intf

      i6ic = ip6ConfigDir.intf.get( self.intf_.name )
      if not i6ic:
         if self.createIfMissing_:
            i6ic = ip6ConfigDir.intf.newMember( self.intf_.name )
            i6ic.l3Config = l3c
         elif self.lookup():
            configSource = self.intfStatus.configSource
            if configSource in dynIp6ConfigDir.entityPtr and \
               configSource in dynIp6ConfigDir.entryState and \
               dynIp6ConfigDir[ configSource ]:
               i6ic = dynIp6ConfigDir[ configSource ].intf.get( self.intf_.name )
         if not i6ic:
            if not Ip6Intf.emptyIntf:
               configDir = Tac.newInstance( "Ip6::Config", "" )
               Ip6Intf.emptyIntf = configDir.intf.newMember( "" )
               Ip6Intf.emptyIntf.l3Config = l3c
            i6ic = Ip6Intf.emptyIntf

      return( i6ic, l3c )

   def setAddr( self, addressWithMask, deprecated=False ):
      a1 = addressWithMask
      addrs = self.config().addr

      ( valid, errorString ) = self.intf_.isValidHostInet6Address( a1 )
      if not valid:
         self.mode.addError( errorString )
         return

      checker = Tac.Value( "Ira::Address6AssignmentChecker" )
      addr6QParams = Tac.newInstance( "Ira::Address6QueryParams", ip6ConfigDir,
                                      self.config().intfId, _mgmtPrefix(),
                                      a1 )
      ans = checker.check( addr6QParams )
      if ans.acceptable:
         if ans.msg:
            self.mode.addWarning( ans.msg )
      else:
         self.mode.addError( ans.msg )
         return

      # Check address against extension hooks, print warning or error message,
      # exit if the address is not acceptable.
      for hook in canSetIntfIpHook.extensions():
         [ accept, message ] = hook( self.name, addressWithMask, mode=self.mode )
         if message:
            if accept:
               self.mode.addWarning( message )
            else:
               self.mode.addError( message )
         if not accept:
            return

      info = Tac.Value( "Ip6::AddressInfo", key=a1 )
      info.deprecated = deprecated
      addrs.addMember( info )

   def setNsSrc( self, addr ):
      Tracing.trace0( "Setting NS source address", addr )
      self.config().nsSourceAddr = addr
      self.config().useNsSourceAddr = True

   def resetNsSrc( self ):
      if self.getUseNsSrc():
         Tracing.trace0( "Re-setting NS source address", self.getNsSrc() )
         self.config().nsSourceAddr = Arnet.Ip6Addr( '::' )
         self.config().useNsSourceAddr = False

   def getNsSrc( self ):
      return self.config().nsSourceAddr

   def getUseNsSrc( self ):
      return self.config().useNsSourceAddr

   def resetDeprecate( self, addressWithMask ):
      addrInfo = self.config().addr.get( addressWithMask )
      if not addrInfo:
         Tracing.trace0( "Can't un-deprecate", addressWithMask, "on",
                         self.name, "address not found in config" )
         return

      addrInfo = Tac.nonConst( addrInfo )
      addrInfo.deprecated = False
      self.config().addr.addMember( addrInfo )
      Tracing.trace0( "Un-Deprecating IP6 address", addressWithMask,
                      "on", self.name )

   def setLLAddr( self, addrArgWithMask ):
      # Check address against extension hooks, print warning or error messa
      # exit if the address is not acceptable.
      for hook in canSetIntfLLAddrHook.extensions():
         [ accept, message ] = hook( self.name, addrArgWithMask )
         if message:
            if accept:
               self.mode.addWarning( message )
            else:
               self.mode.addError( message )
         if not accept:
            return
      self.config().linkLocalAddrWithMask = addrArgWithMask

   def delLLAddr( self, addrArgWithMask ):
      llAddr = self.config().linkLocalAddrWithMask
      if not llAddr == addrArgWithMask:
         self.mode.addWarning( " Address not configured on interface "
                               " Ignoring this command " )
      else:
         nullLLAddr = Arnet.Ip6AddrWithMask( '::/0' )
         self.config().linkLocalAddrWithMask = nullLLAddr

   def delAddr( self, addressWithMask ):
      # Check address against extension hooks, print warning or error message,
      # exit if the address is not acceptable.
      for hook in canDeleteIntfIpHook.extensions():
         [ accept, message ] = hook( self.name, addressWithMask, mode=self.mode )
         if message:
            if accept:
               self.mode.addWarning( message )
            else:
               self.mode.addError( message )
         if not accept:
            return
      if addressWithMask.address.isLinkLocal:
         self.mode.addError( "Configuring link local addresses must use"
                             " 'link-local' keyword " )
         return
      a1 = addressWithMask
      addrs = self.config().addr

      if a1 in addrs:
         if a1.address == self.config().nsSourceAddr:
            # Only reset if this is the current NS source address
            self.resetNsSrc()
         del addrs[ a1 ]
      else:
         self.mode.addError( "Address not configured on interface" )

   def delAddrs( self ):
      # Check address against extension hooks, print warning or error message,
      # exit if the address is not acceptable.
      for hook in canRemoveIp6AddrsHook.extensions():
         [ accept, message ] = hook( self.name, None, mode=self.mode )
         if message:
            if accept:
               self.mode.addWarning( message )
            else:
               self.mode.addError( message )
         if not accept:
            return
      self.config().addr.clear()
      self.resetNsSrc()
      self.delLLAddr( self.config().linkLocalAddrWithMask )

   def ndNsRetransmitInterval( self, ndNsRetransmitIntervalValue ):
      self.config().nsRetransmitInterval = ndNsRetransmitIntervalValue

   def noNdNsRetransmitInterval( self ):
      intfConfig = self.config()
      if not intfConfig:
         return
      intfConfig.nsRetransmitInterval = intfConfig.nsRetransmitIntervalDefault

   def ndDad( self ):
      intfConfig = self.config()
      if not intfConfig:
         return
      intfConfig.acceptDad = acceptDad.dadEnabled

   def ndDadStrict( self ):
      intfConfig = self.config()
      if not intfConfig:
         return
      intfConfig.acceptDad = acceptDad.dadStrict

   def ndDadDisabled( self ):
      intfConfig = self.config()
      if not intfConfig:
         return
      intfConfig.acceptDad = acceptDad.dadDisabled

   def noNdDad( self ):
      intfConfig = self.config()
      if not intfConfig:
         return
      intfConfig.acceptDad = acceptDad.dadGlobal

   def defaultNdDad( self ):
      intfConfig = self.config()
      if not intfConfig:
         return
      intfConfig.acceptDad = acceptDad.dadGlobal

   def attachedRoutes( self ):
      self.config().attachedRoutes = True

   def defaultAttachedRoutes( self ):
      self.config().attachedRoutes = True

   def noAttachedRoutes( self ):
      self.config().attachedRoutes = False

   #----------------------------------------------------------------------------
   # Destroys the IntfConfig object for this interface if it exists.
   #----------------------------------------------------------------------------
   def setDefault( self ):
      intf = self.intf_.name
      del ip6ConfigDir.intf[ intf ]
      if not intf in ip4ConfigDir.ipIntfConfig and intf in l3ConfigDir.intfConfig:
         del l3ConfigDir.intfConfig[ intf ]

   #----------------------------------------------------------------------------
   # Returns the IntfStatus object for this interface.
   #----------------------------------------------------------------------------
   def status( self ):
      return self.intfStatus

   def enabled( self ):
      nullLLAddr = Arnet.Ip6AddrWithMask( '::/0' )
      # pylint: disable-next=superfluous-parens
      llConfigured = not ( self.config().linkLocalAddrWithMask == nullLLAddr )
      return self.config().enable or len( self.config().addr ) > 0 or llConfigured

   #----------------------------------------------------------------------------
   # Called by "show interface" to display addresses on the interface
   #----------------------------------------------------------------------------
   def showNetworkAddr( self ):
      return self.showAddr( lookupIpIntfStatus=True )

   #----------------------------------------------------------------------------
   # Returns an InterfaceAddressIp6 object for this interface.
   #
   # Pass lookupIpIntfStatus=True if you want to return an object only
   # if the interface is not in *enabled* state. Note that aggregate
   # interfaces (eg, port-channel, vlan) may require at least one
   # member before the aggregate can get into the enable state.
   #----------------------------------------------------------------------------
   def showAddr( self, lookupIpIntfStatus=False ):
      if lookupIpIntfStatus and not self.lookup():
         return None

      # Build the InterfaceAddressIp6 object from the interface state, since
      # the operational state may override the configuration state.

      operState = self.show()
      if operState is None:
         return None
      if not ( operState.linkLocal or operState.addresses ):
         return None

      result = InterfaceAddressIp6()
      result.linkLocalIp6 = operState.linkLocal
      result.globalUnicastIp6s = operState.addresses
      result.globalAddressesAreVirtual = operState.globalAddressesAreVirtual
      result.addrSource = operState.addrSource
      if operState.lossOfConnectivityReason:
         result.lossOfConnectivityReason = operState.lossOfConnectivityReason


      return result

   #----------------------------------------------------------------------------
   # Returns an Ip6Status object for this interface.
   #----------------------------------------------------------------------------
   def show( self ):
      self.lookup()
      status = self.status()
      result = Ip6Status()
      result.state = "disabled"
      if self.enabled():
         result.state = "stalled"  # DAD failure on the link local ip
         if self.intf_.name.startswith( "Loopback" ):
            # Loopback interfaces - state is disabled till the 1st global
            # addr is configured on the interface.
            if status and status.addr:
               result.state = "enabled"
               for value in status.addr.values():
                  addr = value.key
                  if addr.address.isLinkLocal:
                     result.linkLocal = IpAddress(
                        address=addr.address.stringValue,
                        subnet=addr.subnet.stringValue,
                        active=True, leastpref=False,
                        dadfailed=False )
                     break
            else:
               result.state = "disabled"
         elif status:
            if not status.l3IntfConnected:
               result.lossOfConnectivityReason = 'cfm'

            for addr, addrInfo in status.addr.items():
               if addr.address.isLinkLocal:
                  # FIXME so far an address being configured and not
                  # showing up in status is a good sign that it is tentative
                  # but I may need something more definite in the future
                  result.state = "enabled"
                  result.linkLocal = IpAddress( address=addr.address.stringValue,
                                                subnet=addr.subnet.stringValue,
                                                active=True,
                                                leastpref=addrInfo.deprecated,
                                                dadfailed=False )
                  break
            # if link-local address is not in the the above collection,
            # look for it in the dadfailed collection
            else:
               for addr in status.dadfailed:
                  if addr.address.isLinkLocal:
                     result.state = "stalled"
                     result.linkLocal = IpAddress( address=addr.address.stringValue,
                                                   subnet=addr.subnet.stringValue,
                                                   active=True, leastpref=False,
                                                   dadfailed=True )
                     break
            if self.config().addrSource == addrSource.slaac:
               intfStatus = allIntfStatusDir.intfStatus[ self.config().intfId ]
               for addr, addrInfo in status.addr.items():
                  if addr.address.isLinkLocal:
                     continue
                  # add the address if it is a SLAAC configured address
                  if not IraCommonCli.isSlaacAddress( addr.address, intfStatus ):
                     continue
                  addr = IpAddress( address = addr.address.stringValue,
                                    subnet=addr.subnet.stringValue,
                                    active=True,
                                    leastpref=addrInfo.deprecated,
                                    dadfailed=False )
                  result.addresses.append( addr )
      config = self.config()
      result.addrSource = config.addrSource
      result.vrf = config.vrf
      result.globalAddressesAreVirtual = bool( config.virtualAddr )
      configAddr = config.virtualAddr if result.globalAddressesAreVirtual \
                   else config.addr
      for addr in configAddr:
         dadfailed = status and addr in status.dadfailed
         addrInfoStatus = status.addr.get( addr ) if status else None
         addr = IpAddress( address=addr.address.stringValue,
                           subnet=addr.subnet.stringValue,
                           active=bool( addrInfoStatus ),
                           leastpref=bool( addrInfoStatus and
                                           addrInfoStatus.deprecated ),
                           dadfailed=bool( dadfailed ) )
         result.addresses.append( addr )
      result.acceptUnsolicitedNa = config.acceptUnsolicitedNa

      # Everything else is status so get out if ipv6 is disabled
      if not self.enabled():
         return result

      if status and status.mcaddr:
         for addr in status.mcaddr:
            result.multicastGroupAddresses.append( addr.stringValue )

      # URPF information
      # We need to display a warning message if V6 URPF is disabled due to
      # conflicts with V4 URPF
      result.urpfV4V6Mismatch = False
      result.urpf = config.urpf.mode
      # For platforms which support explicit loose non-default mode,
      # <mode=loose,allowDefaultRoute=False> => looseNonDefault mode.
      if Toggles.IraToggleLib.toggleUrpfIgnoreDefaultRouteEnabled() and \
         routing6HardwareStatus.urpfLooseNonDefaultSupported and \
         config.urpf.mode == 'loose' and not config.urpf.allowDefaultRoute:
         result.urpf = 'looseNonDefault'
      if not routingHardwareStatus.urpfSupportedOnV4AndV6:
         if self.intf_.name in ip4ConfigDir.ipIntfConfig:
            v4UrpfCfg = ip4ConfigDir.ipIntfConfig[ self.intf_.name ].urpf
            if 'disable' not in [ config.urpf.mode, v4UrpfCfg.mode ] and \
               ( config.urpf.mode != v4UrpfCfg.mode or
                 config.urpf.allowDefaultRoute != v4UrpfCfg.allowDefaultRoute ):
               result.urpf = 'disable'
               result.urpfV4V6Mismatch = True

      result.maxMssIngress = config.maxMssIngress
      result.maxMssEgress = config.maxMssEgress

      arpCliIntfStatus = arpCliStatus.cliIntfStatus.get( self.intf_.name )
      if arpCliIntfStatus is None or not arpCliIntfStatus.timeoutV6:
         result.ndCacheExpiryTime = arpCliStatus.globalNdCacheExpire
      else:
         result.ndCacheExpiryTime = arpCliIntfStatus.timeoutV6

      # Check if Management interface is present in ip6/status to check if multiple
      # SUP have mgmt interface with only one being active at any point. TG series
      # has 2+ mgmt interface with only 1 being active hence kernel intf is missing
      # for the inactive intf.
      extraIp6NdRaInfoHook( self, result )
      return result

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global ip6StatusDir, ip6ConfigDir, dynIp6ConfigDir, l3ConfigDir, ip4ConfigDir
   global routingHardwareStatus, routing6HardwareStatus, allIntfStatusDir
   global arpCliStatus
   ip6StatusDir = LazyMount.mount( entityManager, "ip6/status", "Ip6::Status", "r" )
   ip6ConfigDir = ConfigMount.mount(
      entityManager, "ip6/config", "Ip6::Config", "w" )
   dynIp6ConfigDir = LazyMount.mount( entityManager, "ip6/input/dynIpConfig",
                                      "Tac::Dir", "ri" )
   l3ConfigDir = ConfigMount.mount(
      entityManager, "l3/intf/config", "L3::Intf::ConfigDir", "w" )
   ip4ConfigDir = ConfigMount.mount( entityManager, "ip/config", "Ip::Config", "w" )
   routingHardwareStatus = LazyMount.mount( entityManager, "routing/hardware/status",
                                            "Routing::Hardware::Status", "r" )
   routing6HardwareStatus = LazyMount.mount(
      entityManager, "routing6/hardware/status", "Routing6::Hardware::Status", "r" )
   allIntfStatusDir = LazyMount.mount( entityManager, "interface/status/all",
                                       "Interface::AllIntfStatusDir", "r" )
   arpCliStatus = LazyMount.mount( entityManager, "arp/clistatus",
                                   "Arp::CliStatus", "r" )
   # Cause no interface to destroy us....
   IntfCli.Intf.registerDependentClass( Ip6Intf, priority=10 )
