# Copyright (c) 2005-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

#-------------------------------------------------------------------------------
# This module implements non-interface-specific IP configuration.
#-------------------------------------------------------------------------------
'''Configuration commands supported for IP'''

from collections import defaultdict
import sys

import Arnet
import Arnet.MplsLib
import BasicCli
import Cell
import CliCommand
import CliExtensions
import CliMatcher
import CliParser
import CliToken.Clear
import CliToken.Ip
import CliToken.Icmp
import ConfigMount
import Intf.IntfRange
import CliPlugin.VrfCli as VrfCli # pylint: disable=consider-using-from-import
from CliPlugin import ConfigTagCommon
from CliPlugin import IntfCli
from CliPlugin import Ip6AddrMatcher
from CliPlugin import IpAddrMatcher
from CliPlugin import IpGenAddrMatcher
from CliPlugin import Ip6LlAddr
from CliPlugin import IraCommonCli
from CliPlugin.IraCommonCli import (
   NexthopCliExpression,
   RouteOptionsExprFactory,
   preferenceRangeMatcher,
   tagMatcherForConfig,
   tagNumberMatcher,
)
from CliPlugin import IraIp6Cli
from CliPlugin import IraIpIntfCli
from CliPlugin import IraRouteCommon
from CliPlugin.IraIpIntfCli import zeroIpAddress
from CliPlugin.IraServiceCli import getEffectiveProtocolModel
from CliPlugin.VirtualIntfRule import VirtualIntfMatcher
from CliPlugin.IraIpCliCommon import (
      mergedDict,
      supportedProfileType,
)
from CliPlugin.IraCommonModel import (
      ServiceRoutingProtoModelStatus,
      VrfNotInstalledBfd,
)
from CliPlugin.IraIpModel import (
      InterfaceAddress,
      IpAddress,
      IpRouteSummary,
      IpRouteSummaryForVrf,
      IpModel,
      IpStatus,
      IpStatuses,
      InterfaceAddressBrief,
      IpStatusBrief,
      IpStatusesBrief,
      VrfIpRibRoutes,
      VrfIpRoutes,
      VrfIpRoutesHost,
      bgpTotals,
      isisTotals,
      ospfTotals,
      ospfv3Totals,
      UrpfInterfaces,
      classESubnet,
)
from CliPlugin.VrfCli import (
      ALL_VRF_NAME,
      DEFAULT_VRF,
      VrfExprFactory,
      vrfExists,
)
from CliPlugin.IraIpRouteCliLib import (
      ip,
      noRouteTableForVrfMsg,
      prefixMatcher,
      routeMatcherForConfig,
      routing,
)
from CliPlugin.IraNexthopGroupCli import (
      nexthopGroupSupportedGuard,
      getNexthopGroupNames,
      nexthopGroupNamePattern,
)
from CliPlugin.IpRibLib import ResolutionRibsExprFactory
import LazyMount
import MultiRangeRule
import ShowCommand
import SharkLazyMount
import Tac
import TacSigint
from CliModel import UnknownEntityError
from CliPrint import CliPrint
from IpLibTypes import ProtocolAgentModelType as ProtoAgentModel
from CliToken import IpRibLibCliTokens

import Toggles.IraToggleLib
from Toggles.IraToggleLib import (
   toggleMixedRouteSummaryOutputEnabled,
)
from Toggles.RoutingLibToggleLib import (
   toggleStaticResRibProfilePerViaEnabled,
)
import Toggles.ArnetToggleLib

printer = CliPrint().lib

# pylint: disable-msg=E0602
# Disabling E0602 filewide to prevent getting errors for using log messages

routingHardwareStatusCommon = None
routingHardwareStatus = None
routingHardwareStatus6 = None
routingHardwareRouteStatus = None
ipStatus = None
l3Config = None
allVrfConfig = None
allVrfStatusLocal = None
proxyArpVirtualMacStatus = None
ethIntfStatusDir = None
allVrfRouteSummary = None
arpCliStatus = None

noArpTableForVrfMsg = "ARP table for VRF %s does not exist."
notUnicastPrefix = "The specified prefix %s is not a unicast prefix."
VERIFY_MAX_ROUTES = 100

ipMatcherForConfig = CliToken.Ip.ipMatcherForConfig
ipMatcherForShow = CliToken.Ip.ipMatcherForShow
icmpMatcherForConfig = CliToken.Icmp.icmpMatcherForConfig
rateLimitUnreachableMatcher = CliToken.Icmp.rateLimitUnreachableMatcherForConfig

configTagExpr = ConfigTagCommon.ConfigTagExpr

def vrfRoutingSupportedGuard( mode, token ):
   if routingHardwareStatus.vrfRoutingSupported:
      return None
   return CliParser.guardNotThisPlatform

def routeCacheSupportedGuard( mode, token ):
   if routingHardwareStatus.routeCacheSupported:
      return None
   return CliParser.guardNotThisPlatform

# XXX If the following guard is modified, please update ArosTest plugin
# v4RouteWithV6NhSupported() as well.
def v4RouteWithV6NhSupportedGuard( mode, token ):
   # strictly speaking it's a bit of a hack to base this off the V6
   # ECMP supported capability, but that's really the deciding factor
   # in gated that causes us to not support it on Petra, so...
   if routingHardwareStatus6.ecmpSupported:
      return None
   return CliParser.guardNotThisPlatform

vrfExprFactory = VrfExprFactory(
      helpdesc='Configure routing in a VRF',
      guard=vrfRoutingSupportedGuard )

#-------------------------------------------------------------------------------
# Aliases for some useful token rules.
#-------------------------------------------------------------------------------
configMode = BasicCli.GlobalConfigMode

#-------------------------------------------------------------------------------
# The "ip routing [ipv6 interfaces] [vrf]" and
#     "no|default ip routing [keep-static-routes | delete-static-routes]
#       [addressless] [vrf]"
#     commands.
#-------------------------------------------------------------------------------

class IpRoutingCmd( CliCommand.CliCommandClass ):
   syntax = '''ip routing [ VRF ]'''
   noOrDefaultSyntax = """
      ip ROUTING_UNGUARDED [ keep-static-routes | delete-static-routes ] [ VRF ]"""

   data = { "ip": ipMatcherForConfig,
            "routing": IraIpIntfCli.routingNode,
            "ROUTING_UNGUARDED": IraIpIntfCli.routingMatcher,
            "keep-static-routes": CliCommand.hiddenKeyword( "keep-static-routes" ),
            "delete-static-routes": "Disable IP forwarding, "
                                    "and remove the configured static routes",
            "VRF": vrfExprFactory,
          }

   handler = "IraIpCliHandler.enableIpRouting"
   noOrDefaultHandler = "IraIpCliHandler.disableIpRoutingMaybeClearStaticRoutes"

configMode.addCommandClass( IpRoutingCmd )

class IpRoutingV6IntfCmd( CliCommand.CliCommandClass ):
   syntax = '''ip routing ipv6 interfaces [ VRF ]'''
   noOrDefaultSyntax = syntax.replace( 'routing', 'ROUTING_UNGUARDED' )

   data = { "ip": ipMatcherForConfig,
            "routing": IraIpIntfCli.routingNode,
            "ROUTING_UNGUARDED": IraIpIntfCli.routingMatcher,
            "ipv6": "Global V4 Forwarding for V6 Interfaces",
            "interfaces": "Global V4 Forwarding for V6 Interfaces",
            "VRF": vrfExprFactory,
          }

   handler = "IraIpCliHandler.enableIpRouting"
   noOrDefaultHandler = "IraIpCliHandler.disableIpRouting"

configMode.addCommandClass( IpRoutingV6IntfCmd )

# -------------------------------------------------------------------------------
# The "[ no | defualt ] ip icmp send limit ( ( burst BURSTRATE packets ) |
# ( rate MSGRATE pps ) )" command.
# -------------------------------------------------------------------------------

class IcmpMsgParamsCmd( CliCommand.CliCommandClass ):
   syntax = '''ip icmp send limit
               ( ( burst BURSTRATE packets )
               | ( rate MSGRATE pps ) )'''
   noOrDefaultSyntax = 'ip icmp send limit ( burst | rate ) ... '

   data = { "ip": ipMatcherForConfig,
            "icmp": icmpMatcherForConfig,
            "send": 'Set parameters to control sending ICMP messages',
            "limit": 'Configuration to limit sending ICMP messages',
            "burst": 'Burst size',
            "BURSTRATE": CliMatcher.IntegerMatcher( 0, 2**32 - 1,
                  helpdesc='Set the maximum burst size' ),
            "packets": 'Number of packets',
            "rate": 'Message rate',
            "MSGRATE": CliMatcher.IntegerMatcher( 0, 2**32 - 1,
                  helpdesc='Set the maximum message rate' ),
            "pps": 'Packets per second',
          }

   handler = "IraIpCliHandler.handlerIcmpMsgParamsCmd"
   noOrDefaultHandler = handler

configMode.addCommandClass( IcmpMsgParamsCmd )
#-------------------------------------------------------------------------------
# The "(no|default) ip icmp rate-limit-unreachable 0" command.
#-------------------------------------------------------------------------------
class IcmpUnreachableRateCmd( CliCommand.CliCommandClass ):
   syntax = '''ip icmp rate-limit-unreachable RATE'''

   noOrDefaultSyntax = '''ip icmp rate-limit-unreachable ...'''

   data = { "ip": ipMatcherForConfig,
            "icmp": icmpMatcherForConfig,
            "rate-limit-unreachable": rateLimitUnreachableMatcher,
            "RATE": CliMatcher.IntegerMatcher( 0, 0, helpdesc='Rate' ),
          }

   handler = "IraIpCliHandler.handlerIcmpUnreachableRateCmd"
   noOrDefaultHandler = handler

configMode.addCommandClass( IcmpUnreachableRateCmd )

#-------------------------------------------------------------------------------
# The "[no] ip route" command.
#-------------------------------------------------------------------------------
nexthopAddrMatcher = IpAddrMatcher.IpAddrMatcher(
      'IPv4 address of the nexthop router' )
nexthop6AddrNode = CliCommand.Node(
      Ip6AddrMatcher.Ip6AddrMatcher( 'IPv6 address of the nexthop router' ),
      guard=v4RouteWithV6NhSupportedGuard )
nexthop6LlAddrNode = CliCommand.Node(
      Ip6LlAddr.Ip6LlAddrMatcher( 'IPv6 linklocal address of the nexthop router' ),
      guard=v4RouteWithV6NhSupportedGuard )
intfAddrMatcher = IpAddrMatcher.IpAddrMatcher(
      "Forwarding router's IPv4 address on destination interface" )
intf6AddrNode = CliCommand.Node(
      Ip6AddrMatcher.Ip6AddrMatcher(
         "Forwarding router's IPv6 address on destination interface" ),
      guard=v4RouteWithV6NhSupportedGuard )
nullIntfMatcher = VirtualIntfMatcher( 'Null', 0, 0,
      helpdesc='Interface that drops all traffic' )
nexthopGroupNodeForConfig = CliCommand.Node(
                                  CliMatcher.KeywordMatcher( 'nexthop-group',
                                     helpdesc='Specify nexthop group name'),
                                  guard=nexthopGroupSupportedGuard )

nexthopGroupNameMatcher = CliMatcher.DynamicNameMatcher( getNexthopGroupNames,
                              'Nexthop group name', pattern=nexthopGroupNamePattern )

trackingProtocols = {}
trackingProtocolsHidden = {}

# call addTrackingProto before TrackingProtocolExpr
IraIp6Cli.addTrackingProto( trackingProtocols, trackingProtocolsHidden,
                            'bfd', 'Track this route using BFD' )

vrfExprFactoryForConfig = VrfExprFactory( helpdesc='Configure routes in a VRF' )

def _intfValueFunc( mode, **kwargs ):
   kwargs.setdefault( 'intfnexthop', None )
   return kwargs

def intfAddrRoutesSupportedGuard( mode, token ):
   if ip.intfAddrRoutesSupported:
      return None
   return CliParser.guardNotThisPlatform

# toggle for weight option
weightEnabled = Toggles.IraToggleLib.toggleStaticRouteViaWeightEnabled()

class NexthopAddrWithoutLinkLocalExpr( CliCommand.CliExpression ):
   expression = 'NEXTHOP | NEXTHOP6'
   data = { 'NEXTHOP': nexthopAddrMatcher,
            'NEXTHOP6': nexthop6AddrNode,
           }

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'NEXTHOP_ADDR' ] = {}
      if 'NEXTHOP' in args:
         args[ 'NEXTHOP_ADDR' ][ 'nexthop' ] = args.pop( 'NEXTHOP' )
      elif 'NEXTHOP6' in args:
         args[ 'NEXTHOP_ADDR' ][ 'nexthop6' ] = args.pop( 'NEXTHOP6' )

class NexthopAddrExpr( CliCommand.CliExpression ):
   expression = 'NEXTHOP | NEXTHOP6 | NEXTHOP6LL'
   data = { 'NEXTHOP': nexthopAddrMatcher,
            'NEXTHOP6': nexthop6AddrNode,
            'NEXTHOP6LL': nexthop6LlAddrNode
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'NEXTHOP_INTF' not in args:
         args[ 'NEXTHOP_INTF' ] = {}
      if 'NEXTHOP' in args:
         args[ 'NEXTHOP_INTF' ][ 'nexthop' ] = args.pop( 'NEXTHOP' )
      elif 'NEXTHOP6' in args:
         args[ 'NEXTHOP_INTF' ][ 'nexthop6' ] = args.pop( 'NEXTHOP6' )
      elif 'NEXTHOP6LL' in args:
         args[ 'NEXTHOP_INTF' ][ 'nexthop6ll' ] = args.pop( 'NEXTHOP6LL' )

class NexthopOrIntfNexthopExpr( CliCommand.CliExpression ):
   expression = 'NEXTHOP | NEXTHOP6 | NEXTHOP6LL | NULL0_EXPR | '\
                '( COMMON_INTF INTF_ADDR | INTF6_ADDR )'
   data = { 'NEXTHOP': nexthopAddrMatcher,
            'NEXTHOP6': nexthop6AddrNode,
            'NEXTHOP6LL': nexthop6LlAddrNode,
            'NULL0_EXPR': IraCommonCli.Null0Expr,
            'COMMON_INTF': CliCommand.Node( CliMatcher.WrapperMatcher(
                                               IntfCli.Intf.matcher,
                                               priority=CliParser.PRIO_HIGH ),
                                            guard=intfAddrRoutesSupportedGuard ),
            'INTF_ADDR': intfAddrMatcher,
            'INTF6_ADDR': intf6AddrNode
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'NEXTHOP_INTF' not in args:
         args[ 'NEXTHOP_INTF' ] = {}
      if 'NEXTHOP' in args:
         args[ 'NEXTHOP_INTF' ][ 'nexthop' ] = args.pop( 'NEXTHOP' )
      elif 'NEXTHOP6' in args:
         args[ 'NEXTHOP_INTF' ][ 'nexthop6' ] = args.pop( 'NEXTHOP6' )
      elif 'NEXTHOP6LL' in args:
         args[ 'NEXTHOP_INTF' ][ 'nexthop6ll' ] = args.pop( 'NEXTHOP6LL' )
      elif 'Null0' in args:
         del args[ 'Null0' ]
         args[ 'NEXTHOP_INTF' ][ 'Null0' ] = 'Null0'
      elif 'Null 0' in args:
         args[ 'NEXTHOP_INTF' ][ 'Null 0' ] = args.pop( 'Null 0' )
      elif 'COMMON_INTF' in args:
         args[ 'NEXTHOP_INTF' ][ 'intf' ] = {}
         args[ 'NEXTHOP_INTF' ][ 'intf' ][ 'intf' ] = args.pop( 'COMMON_INTF' )
         if 'INTF_ADDR' in args:
            args[ 'NEXTHOP_INTF' ][ 'intf' ][ 'intfnexthop' ] = \
               args.pop( 'INTF_ADDR' )
         elif 'INTF6_ADDR' in args:
            args[ 'NEXTHOP_INTF' ][ 'intf' ][ 'intfnexthop' ] = \
               args.pop( 'INTF6_ADDR' )

class MplsLabelOrNexthopGroupOrIntfExpr( CliCommand.CliExpression ):
   expression = '( NH label LABEL_VALUE ) | ( nexthop-group NHG_NAME ) | ' \
                'INTF'
   data = { 'NH': IpAddrMatcher.IpAddrMatcher( 'Address of the nexthop router',
                                               priority=CliParser.PRIO_HIGH ),
            'label': IraIp6Cli.labelNode,
            'LABEL_VALUE': Arnet.MplsLib.labelValMatcher,
            'nexthop-group': nexthopGroupNodeForConfig,
            'NHG_NAME': nexthopGroupNameMatcher,
            'INTF': IntfCli.Intf.matcher
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'NEXTHOP_INTF' not in args:
         args[ 'NEXTHOP_INTF' ] = {}
      if 'label' in args:
         args[ 'NEXTHOP_INTF' ][ 'mpls' ] = {}
         args[ 'NEXTHOP_INTF' ][ 'mpls' ][ 'nexthop' ] = args.pop( 'NH' )
         args[ 'NEXTHOP_INTF' ][ 'mpls' ][ 'label' ] = args.pop( 'LABEL_VALUE' )
      elif 'nexthop-group' in args:
         args[ 'NEXTHOP_INTF' ][ 'nexthopGroupName' ] = args.pop( 'NHG_NAME' )
      elif 'INTF' in args:
         args[ 'NEXTHOP_INTF' ][ 'intf' ] = {}
         args[ 'NEXTHOP_INTF' ][ 'intf' ][ 'intf' ] = args.pop( 'INTF' )
         args[ 'NEXTHOP_INTF' ][ 'intf' ][ 'intfnexthop' ] = None

class NhMplsLabelExpr( CliCommand.CliExpression ):
   # this has an adapter, so making it a class
   expression = '( NH label LABEL_VALUE )'
   data = { 'NH': IpAddrMatcher.IpAddrMatcher( 'Address of the nexthop router',
                                               priority=CliParser.PRIO_HIGH ),
            'label': IraIp6Cli.labelNode,
            'LABEL_VALUE': Arnet.MplsLib.labelValMatcher
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'NEXTHOP_INTF' not in args:
         args[ 'NEXTHOP_INTF' ] = {}
      args[ 'NEXTHOP_INTF' ][ 'mpls' ] = {}
      args[ 'NEXTHOP_INTF' ][ 'mpls' ][ 'nexthop' ] = args.pop( 'NH' )
      args[ 'NEXTHOP_INTF' ][ 'mpls' ][ 'label' ] = args.pop( 'LABEL_VALUE' )

class NexthopGroupExpr( CliCommand.CliExpression ):
   # this has an adapter, so making it a class
   expression = '( nexthop-group NHG_NAME )'
   data = { 'nexthop-group': nexthopGroupNodeForConfig,
            'NHG_NAME': nexthopGroupNameMatcher
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'NEXTHOP_INTF' not in args:
         args[ 'NEXTHOP_INTF' ] = {}
      args[ 'NEXTHOP_INTF' ][ 'nexthopGroupName' ] = args.pop( 'NHG_NAME' )

class IpRouteNexthopOrComboCmd( CliCommand.CliCommandClass ):
   '''This class covers nexthop-only routes, combo routes(nexthop+intf)'''
   syntax = '''ip route [ VRF ] PREFIX NEXTHOP_ADDR_OR_INTF [ ROUTE_OPTIONS ]
               [ LEAK_VRF ]'''
   data = { 'ip': ipMatcherForConfig,
            'route': routeMatcherForConfig,
            'VRF': vrfExprFactoryForConfig,
            'PREFIX': prefixMatcher,
            'NEXTHOP_ADDR_OR_INTF': NexthopOrIntfNexthopExpr,
            'ROUTE_OPTIONS': RouteOptionsExprFactory(
                              includePreference=True,
                              includeTag=True,
                              includeCommunity=True,
                              includeNexthopName=True,
                              includeMetric=True,
                              includeWeight=weightEnabled,
                              trackKwargs={
                                 'trackingProtocols': trackingProtocols,
                                 'trackingProtocolsHidden': trackingProtocolsHidden,
                                 'includeIp4': True,
                                 'includeIp6': True,
                                 'ip6Guard': v4RouteWithV6NhSupportedGuard,
                                 'ip6IsHidden': True,
                                 } ),
            'LEAK_VRF': IraIp6Cli.LeakVrfCliExpr
          }

   handler = "IraIpCliHandler.handlerIpRouteNexthopOrComboCmd"

configMode.addCommandClass( IpRouteNexthopOrComboCmd )

class IpRouteNexthopResRibProfileCmd( CliCommand.CliCommandClass ):
   '''This class covers nexthop-only routes with resolution rib profile specified
   for the nexthop'''
   syntax = '''ip route [ VRF ] PREFIX NEXTHOP_ADDR [ ROUTE_OPTIONS ]
               resolution ribs SYS_CON_ONLY_RIBS'''
   data = { 'ip': ipMatcherForConfig,
            'route': routeMatcherForConfig,
            'VRF': vrfExprFactoryForConfig,
            'PREFIX': prefixMatcher,
            'NEXTHOP_ADDR': NexthopAddrWithoutLinkLocalExpr,
            'ROUTE_OPTIONS': RouteOptionsExprFactory(
                              includePreference=True,
                              includeTag=True,
                              includeCommunity=True,
                              includeNexthopName=True,
                              includeMetric=True,
                              includeWeight=weightEnabled,
                              trackKwargs={
                                 'trackingProtocols': trackingProtocols,
                                 'trackingProtocolsHidden': trackingProtocolsHidden,
                                 'includeIp4': True,
                                 'includeIp6': True,
                                 'ip6Guard': v4RouteWithV6NhSupportedGuard,
                                 'ip6IsHidden': True,
                                 } ),
            'resolution': IpRibLibCliTokens.matcherResolution,
            'ribs': IpRibLibCliTokens.matcherRibs,
            'SYS_CON_ONLY_RIBS': ResolutionRibsExprFactory()
          }

   handler = "IraIpCliHandler.handlerIpRouteNexthopResRibProfileCmd"

if toggleStaticResRibProfilePerViaEnabled():
   configMode.addCommandClass( IpRouteNexthopResRibProfileCmd )

class IpRouteMplsLabelCmd( CliCommand.CliCommandClass ):
   '''This class covers mpls route.'''
   syntax = 'ip route [ VRF ] PREFIX NEXTHOP_LABEL [ ROUTE_OPTIONS ] [ LEAK_VRF ]'
   data = { 'ip': ipMatcherForConfig,
            'route': routeMatcherForConfig,
            'VRF': vrfExprFactoryForConfig,
            'PREFIX': prefixMatcher,
            'NEXTHOP_LABEL': NhMplsLabelExpr,
            'ROUTE_OPTIONS': RouteOptionsExprFactory(
                              includePreference=True,
                              includeTag=True,
                              includeCommunity=False,
                              includeNexthopName=True ),
            'LEAK_VRF': IraIp6Cli.LeakVrfCliExpr
          }

   handler = "IraIpCliHandler.handlerIpRouteMplsLabelCmd"

configMode.addCommandClass( IpRouteMplsLabelCmd )

class IpRouteNexthopGroupCmd( CliCommand.CliCommandClass ):
   '''This class covers nexthop-group route'''
   syntax = 'ip route [ VRF ] PREFIX NEXTHOP_GROUP [ ROUTE_OPTIONS ] [ LEAK_VRF ]'
   data = { 'ip': ipMatcherForConfig,
            'route': routeMatcherForConfig,
            'VRF': vrfExprFactoryForConfig,
            'PREFIX': prefixMatcher,
            'NEXTHOP_GROUP': NexthopGroupExpr,
            'ROUTE_OPTIONS': RouteOptionsExprFactory(
                              includePreference=True,
                              includeTag=True,
                              includeCommunity=True,
                              includeNexthopName=True,
                              includeMetric=True,
                              includeWeight=weightEnabled ),
            'LEAK_VRF': IraIp6Cli.LeakVrfCliExpr
          }

   handler = "IraIpCliHandler.handlerIpRouteNexthopGroupCmd"

configMode.addCommandClass( IpRouteNexthopGroupCmd )

class IpRouteInterfaceCmd( CliCommand.CliCommandClass ):
   '''This class  covers interface routes'''
   syntax = 'ip route [ VRF ] PREFIX INTF [ ROUTE_OPTIONS ] [ LEAK_VRF ]'
   data = { 'ip': ipMatcherForConfig,
            'route': routeMatcherForConfig,
            'VRF': vrfExprFactoryForConfig,
            'PREFIX': prefixMatcher,
            'INTF': IntfCli.Intf.matcher,
            'ROUTE_OPTIONS': RouteOptionsExprFactory(
                              includePreference=True,
                              includeTag=True,
                              includeCommunity=True,
                              includeNexthopName=True,
                              includeMetric=True,
                              includeWeight=weightEnabled ),
            'LEAK_VRF': IraIp6Cli.LeakVrfCliExpr
          }

   handler = "IraIpCliHandler.handlerIpRouteInterfaceCmd"

configMode.addCommandClass( IpRouteInterfaceCmd )

class IpRouteEgressVrfCmd( CliCommand.CliCommandClass ):
   syntax = '''ip route [ VRF ] PREFIX EGRESS_VRF NEXTHOP_ADDR [ ROUTE_OPTIONS ]
               [ LEAK_VRF ]'''
   data = { 'ip': ipMatcherForConfig,
            'route': routeMatcherForConfig,
            'VRF': vrfExprFactoryForConfig,
            'PREFIX': prefixMatcher,
            'EGRESS_VRF': IraIp6Cli.EgressVrfCliExpr,
            'NEXTHOP_ADDR': NexthopAddrExpr,
            'ROUTE_OPTIONS': RouteOptionsExprFactory(
                              includePreference=True,
                              includeTag=True,
                              includeNexthopName=True,
                              includeCommunity=False,
                              includeMetric=True,
                              includeWeight=weightEnabled ),
            'LEAK_VRF': IraIp6Cli.LeakVrfCliExpr
          }

   handler = "IraIpCliHandler.handlerIpRouteEgressVrfCmd"

class IpRouteEgressVrfNexthopOrComboCmd( CliCommand.CliCommandClass ):
   syntax = '''ip route [ VRF ] PREFIX EGRESS_VRF NEXTHOP_ADDR_OR_INTF
               [ ROUTE_OPTIONS ] [ LEAK_VRF ]'''
   data = { 'ip': ipMatcherForConfig,
            'route': routeMatcherForConfig,
            'VRF': vrfExprFactoryForConfig,
            'PREFIX': prefixMatcher,
            'EGRESS_VRF': IraIp6Cli.EgressVrfCliExpr,
            'NEXTHOP_ADDR_OR_INTF': NexthopOrIntfNexthopExpr,
            'ROUTE_OPTIONS': RouteOptionsExprFactory(
                              includePreference=True,
                              includeTag=True,
                              includeNexthopName=True,
                              includeCommunity=False,
                              includeMetric=True,
                              includeWeight=weightEnabled ),
            'LEAK_VRF': IraIp6Cli.LeakVrfCliExpr
          }

   handler = "IraIpCliHandler.handlerIpRouteEgressVrfNexthopOrComboCmd"

configMode.addCommandClass( IpRouteEgressVrfNexthopOrComboCmd )

class IpRouteEgressVrfInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'ip route [ VRF ] PREFIX EGRESS_VRF INTF [ ROUTE_OPTIONS ] [ LEAK_VRF ]'
   data = { 'ip': ipMatcherForConfig,
            'route': routeMatcherForConfig,
            'VRF': vrfExprFactoryForConfig,
            'PREFIX': prefixMatcher,
            'EGRESS_VRF': IraIp6Cli.EgressVrfCliExpr,
            'INTF': IntfCli.Intf.matcher,
            'ROUTE_OPTIONS': RouteOptionsExprFactory(
                              includePreference=True,
                              includeTag=True,
                              includeNexthopName=True,
                              includeCommunity=False,
                              includeMetric=True,
                              includeWeight=weightEnabled ),
            'LEAK_VRF': IraIp6Cli.LeakVrfCliExpr
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'NEXTHOP_INTF' not in args:
         args[ 'NEXTHOP_INTF' ] = {}
      args[ 'NEXTHOP_INTF' ][ 'intf' ] = {}
      args[ 'NEXTHOP_INTF' ][ 'intf' ][ 'intf' ] = args.pop( 'INTF' )
      args[ 'NEXTHOP_INTF' ][ 'intf' ][ 'intfnexthop' ] = None

   handler = "IraIpCliHandler.handlerIpRouteEgressVrfInterfaceCmd"

if Toggles.IraToggleLib.toggleIntfEgressVrfRoutesEnabled():
   configMode.addCommandClass( IpRouteEgressVrfInterfaceCmd )

class NoIpRouteCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'ip route [ VRF ] PREFIX ' \
                       '[ ( EGRESS_VRF EGRESS_NEXTHOP ) |' \
                       '  NEXTHOP_ADDR_OR_INTF |' \
                       '  NON_NEXTHOP_ADDR_OR_INTF ] ' \
                       '[ PREFERENCE ] ...'
   data = { 'ip': ipMatcherForConfig,
            'route': routeMatcherForConfig,
            'VRF': vrfExprFactoryForConfig,
            'PREFIX': prefixMatcher,
            'EGRESS_VRF': IraIp6Cli.EgressVrfCliExpr,
            'EGRESS_NEXTHOP': NexthopAddrExpr,
            'NEXTHOP_ADDR_OR_INTF': NexthopOrIntfNexthopExpr,
            'NON_NEXTHOP_ADDR_OR_INTF': MplsLabelOrNexthopGroupOrIntfExpr,
            'PREFERENCE': preferenceRangeMatcher
          }

   noOrDefaultHandler = "IraIpCliHandler.noOrDefaultHandlerNoIpRouteCmd"

configMode.addCommandClass( NoIpRouteCmd )

cachedMatcherForConfig = CliCommand.Node(
      CliMatcher.KeywordMatcher( 'cached', helpdesc='Route cache' ),
      guard=routeCacheSupportedGuard )

class IpRouteCachedCmd( CliCommand.CliCommandClass ):
   syntax = 'ip route [ VRF ] PREFIX cached'
   data = { 'ip': ipMatcherForConfig,
            'route': routeMatcherForConfig,
            'VRF': vrfExprFactoryForConfig,
            'PREFIX': prefixMatcher,
            'cached': cachedMatcherForConfig
            }

   handler = "IraIpCliHandler.handlerIpRouteCachedCmd"

configMode.addCommandClass( IpRouteCachedCmd )

class NoIpRouteCachedCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'ip route [ VRF ] PREFIX cached'
   data = { 'ip': ipMatcherForConfig,
            'route': routeMatcherForConfig,
            'VRF': vrfExprFactoryForConfig,
            'PREFIX': prefixMatcher,
            'cached': cachedMatcherForConfig
            }

   noOrDefaultHandler = "IraIpCliHandler.noOrDefaultHandlerNoIpRouteCachedCmd"

configMode.addCommandClass( NoIpRouteCachedCmd )

#-------------------------------------------------------------------------------
# The "show ip" command.
#-------------------------------------------------------------------------------

# Get the number of configured Vrrp Interfaces
fhrpVrrpIntfsHook = CliExtensions.CliHook()

# Check whether IPv4/IPv6 multicast routing is enabled
mCastRoutingHook = CliExtensions.CliHook()

class ShowIpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show ip'''
   data = { 'ip': ipMatcherForShow }

   cliModel = IpModel
   handler = "IraIpCliHandler.generateIpModel"

BasicCli.addShowCommandClass( ShowIpCmd )

def intfAddressBrief( ipIntfConfig, ipIntfStatus ):
   # We first check ip/status to see the active ip addr
   # If we dont find a valid ip addr in status, we check to see
   # if one has been configured by cli.
   addr = ipIntfStatus.activeAddrWithMask
   if addr == zeroIpAddress:
      addr = ipIntfConfig.addrWithMask
   if addr == zeroIpAddress:
      addr = ipIntfConfig.virtualAddrWithMask

   intfAddrBrief = InterfaceAddressBrief(
      ipAddr=IpAddress( address=addr.address, maskLen=addr.len ) )
   if ipIntfStatus.unnumberedIntfId:
      intfAddrBrief.unnumberedIntf = ipIntfStatus.unnumberedIntfId
   return intfAddrBrief

ipv4Routable240ClassEHook = CliExtensions.CliHook()
ipv4Routable0ClassAHook = CliExtensions.CliHook()

#-------------------------------------------------------------------------------
# The "show ip interface [ intf | ipaddr | vrf ]" command.
#-------------------------------------------------------------------------------

def printIpIntfFull( mode, intfs=None, raiseLookupError=True ):
   results = IpStatuses()
   if intfs is None:
      return results
   for intf in intfs:
      if not IraCommonCli.lookupCommon( intf, useException=raiseLookupError ):
         continue

      config = intf.config() or intf.dynConfig()
      result = IpStatus( description=intf.description(),
                         enabled=config.enabled if config else False,
                         mtu=intf.status().mtu )

      result.name = intf.name
      ( result.lineProtocolStatus, result.interfaceStatus ) = intf.getStatus()

      ipIntf = IraIpIntfCli.IpIntf( intf, mode.sysdbRoot, createIfMissing=False )
      config = ipIntf.config()
      intfStatus = ipIntf.getStatus()
      interfaceAddress = InterfaceAddress()
      if config.addrSource == 'dhcp':
         interfaceAddress.dhcp = True

      if not intfStatus.l3IntfConnected:
         result.lossOfConnectivityReason = 'cfm'

      interfaceAddress.primaryIp = IpAddress(
                                   address=intfStatus.activeAddrWithMask.address,
                                   maskLen=intfStatus.activeAddrWithMask.len )
      # We first look at ip/status, if there's no ip address then we look at
      # ip/config ( written to by the cli ).
      if intfStatus.activeAddrWithMask == zeroIpAddress:
         interfaceAddress.primaryIp = IpAddress(
                                      address=config.addrWithMask.address,
                                      maskLen=config.addrWithMask.len )
      if config.unnumberedIntfId:
         interfaceAddress.unnumberedIntf = config.unnumberedIntfId
      if intfStatus.useVirtualAddr:
         interfaceAddress.primaryIp = IpAddress( address='0.0.0.0', maskLen=0 )

      for sa in config.secondaryWithMask:
         interfaceAddress.secondaryIpsOrderedList.append( IpAddress(
            address=sa.address, maskLen=sa.len ) )

      interfaceAddress.virtualIp = \
          IpAddress( address=config.virtualAddrWithMask.address,
                     maskLen=config.virtualAddrWithMask.len )

      for sa in config.virtualSecondaryWithMask:
         interfaceAddress.virtualSecondaryIpsOrderedList.append( IpAddress(
            address=sa.address, maskLen=sa.len ) )

      ipIntfConfig = ipIntf.config()
      ipIntfStatus = ipIntf.getStatus()
      result.interfaceAddressBrief = intfAddressBrief( ipIntfConfig, ipIntfStatus )
      result.ipv4Routable240 = ipv4Routable240ClassEHook.all()
      result.ipv4Routable0 = ipv4Routable0ClassAHook.all()
      # We don't let user configure broadcast address yet, so it's hard-coded here.
      interfaceAddress.broadcastAddress = "255.255.255.255"

      result.interfaceAddress = interfaceAddress
      result.proxyArp = config.proxyArpEnabled
      result.proxyArpAllowDefault = config.proxyArpAllowDefaultRoute
      result.localProxyArp = config.localProxyArpEnabled
      result.gratuitousArp = config.gratuitousArpAccepted
      arpCliIntfStatus = arpCliStatus.cliIntfStatus.get( intf.name )
      if arpCliIntfStatus is None or not arpCliIntfStatus.timeoutV4:
         result.arpAgingTimeout = arpCliStatus.globalArpTimeout
      else:
         result.arpAgingTimeout = arpCliIntfStatus.timeoutV4
      virtualMac = proxyArpVirtualMacStatus.virtualMac.get( intf.name )
      if virtualMac:
         result.proxyArpVirtualMacAddress = virtualMac.vrMac.arp
      if virtualMac:
         result.vrId = virtualMac.vrId
      if ethIntfStatusDir.intfStatus.get( intf.name ):
         result.routedAddr = ethIntfStatusDir.intfStatus.get( intf.name ).routedAddr
      result.isVrrpBackup = not virtualMac.active if virtualMac else False
      ipIntfStatus = ipStatus.ipIntfStatus.get( intf.name )
      if ipIntfStatus !=  None: # pylint: disable=singleton-comparison
         result.addresslessForwarding = ipIntfStatus.addresslessForwarding
         result.v4IntfForwarding = ipIntfStatus.dpRoutingEnabled
      else:
         result.addresslessForwarding = "isInvalid"
         result.v4IntfForwarding = False
      if config.vrf:
         result.vrf = config.vrf

      # URPF information
      result.urpf = 'disable'
      if intf.name in ip.config.ipIntfConfig:
         # TODO BUG642564: When strictDefault is removed from UrpfMode enum,
         # v4ModeStr should be based on mode as well as allowDefaultRoute.
         v4ModeStr = ip.config.ipIntfConfig[ intf.name ].urpf.mode
         v4AllowDefault = ip.config.ipIntfConfig[ intf.name ].urpf.allowDefaultRoute
         # For platforms which support explicit loose non-default mode,
         # <mode=loose,allowDefaultRoute=False> => looseNonDefault mode.
         if Toggles.IraToggleLib.toggleUrpfIgnoreDefaultRouteEnabled() and \
            routingHardwareStatus.urpfLooseNonDefaultSupported and \
            v4ModeStr == 'loose' and not v4AllowDefault:
            v4ModeStr = 'looseNonDefault'
         result.urpf = v4ModeStr

      result.directedBroadcastEnabled = False
      if intf.name in ip.config.ipIntfConfig:
         enabled = ip.config.ipIntfConfig[ intf.name ].directedBroadcastEnabled
         result.directedBroadcastEnabled = enabled

      result.maxMssIngress = 0
      result.maxMssEgress = 0
      if intf.name in ip.config.ipIntfConfig:
         maxMssIngress = ip.config.ipIntfConfig[ intf.name ].maxMssIngress
         maxMssEgress = ip.config.ipIntfConfig[ intf.name ].maxMssEgress
         result.maxMssIngress = maxMssIngress
         result.maxMssEgress = maxMssEgress

      results.interfaces[ intf.name ] = result

   return results

#-------------------------------------------------------------------------------
# The "show ip interface [ intf | ipaddr | vrf ] brief" command.
#-------------------------------------------------------------------------------

def printIpIntfBrief( mode, intfs=None, raiseLookupError=True ):
   briefModel = IpStatusesBrief()
   if intfs is not None:
      for intf in intfs:
         if not IraCommonCli.lookupCommon( intf, useException=raiseLookupError ):
            continue

         ( lps, intfStatus ) = intf.getStatus()
         briefStatus = IpStatusBrief( name=intf.name,
                                      lineProtocolStatus=lps,
                                      interfaceStatus=intfStatus,
                                      mtu=intf.status().mtu )

         ipIntf = IraIpIntfCli.IpIntf( intf, mode.sysdbRoot, createIfMissing=False )
         ipIntfConfig = ipIntf.config()
         ipIntfStatus = ipIntf.getStatus()
         briefStatus.interfaceAddress = \
               intfAddressBrief( ipIntfConfig, ipIntfStatus )
         briefStatus.ipv4Routable240 = ipv4Routable240ClassEHook.all()
         briefStatus.nonRoutableClassEIntf = ( not briefStatus.ipv4Routable240 and
            classESubnet.contains( briefStatus.interfaceAddress.ipAddr.address ) )
         briefStatus.ipv4Routable0 = ipv4Routable0ClassAHook.all()
         briefModel.interfaces[ intf.name ] = briefStatus

   return briefModel

def isPrimaryIpAddr( ipIntfStatus, intfAddr ):
   return ipIntfStatus.activeAddrWithMask.address == intfAddr

def isSecondIpAddr( ipIntfStatus, intfAddr ):
   for secondAddr in ipIntfStatus.activeSecondaryWithMask:
      if secondAddr.address == intfAddr:
         return True
   return False

def isPrimaryOrSecondIpAddr( ipIntfStatus, intfAddr ):
   return ( isPrimaryIpAddr( ipIntfStatus, intfAddr )
            or isSecondIpAddr( ipIntfStatus, intfAddr ) )

def filterIntfByRouteable( mode, intfs, ifname=None, exceptionOnEmpty=True ):
   filtered = []

   for intf in intfs:
      if intf.routingCurrentlySupported():
         filtered.append( intf )

   if not filtered:
      msg = "No IP capable interfaces"
      if ifname is not None:
         names = list( ifname )
         if len( names ) == 1:
            msg = "%s does not support IP" % names[ 0 ]
         else:
            msg = "No IP capable interfaces in range"

      if exceptionOnEmpty: # pylint: disable=no-else-raise
         raise UnknownEntityError( msg )
      else:
         mode.addError( msg )

   return filtered

def filterIntfByVrf( mode, intfs, vrf ):
   if vrf is None or not vrfExists( vrf ):
      raise UnknownEntityError( noRouteTableForVrfMsg % vrf )

   return [ intf for intf in intfs
      if IraIpIntfCli.IpIntf( intf, mode,
                              createIfMissing=False ).config().vrf == vrf ]

def filterIntfByIntf( mode, ifname ):
   intfs = IntfCli.Intf.getAll( mode, ifname )
   if not intfs:
      raise UnknownEntityError( "Interface does not exist" )

   return intfs

def filterIntfByAddr( mode, intfs, addr ):
   filtered = []
   for ipIntfStatus in ipStatus.ipIntfStatus.values():
      if isPrimaryOrSecondIpAddr( ipIntfStatus, addr ):
         intfType, _ = Intf.IntfRange.intfTypeFromName( ipIntfStatus.intfId )
         intf = intfType.getCliIntf( mode, ipIntfStatus.intfId )
         filtered.append( intf )

   return filtered

def filterIpIntf( mode, dispFcn, intfOrAddr=None, vrfName=None ):
   raiseLookupError = True
   if intfOrAddr and isinstance( intfOrAddr, MultiRangeRule.IntfList ):
      # Filter by interface name.
      intfs = filterIntfByIntf( mode, intfOrAddr )
      intfs = filterIntfByRouteable( mode, intfs, intfOrAddr )
   elif intfOrAddr and not isinstance( intfOrAddr, MultiRangeRule.IntfList ):
      # Filter by ip address.
      intfs = IntfCli.Intf.getAll( mode )
      intfs = filterIntfByRouteable( mode, intfs )
      intfs = filterIntfByAddr( mode, intfs, intfOrAddr )
   else:
      # Get all routeable interfaces.
      intfs = IntfCli.Intf.getAll( mode )
      # exceptionOnEmpty=True is a TARGETED fix for BUG649912, meant specifically to
      # support the case of listing all IP interfaces when there are no routeable
      # interfaces. We do not want to throw an exception so that 'show tech support'
      # succeeds in this circumstance.
      #
      # In an ideal world there would be no exception throwing to signal errors - we
      # should use mode.addError insted. BUG662792 tracks cleaning this up.
      intfs = filterIntfByRouteable( mode, intfs, exceptionOnEmpty=False )
      raiseLookupError = False
   if vrfName:
      # Filter by vrf.
      intfs = filterIntfByVrf( mode, intfs, vrfName )

   return dispFcn( mode, intfs=intfs, raiseLookupError=raiseLookupError )

def showIpIntfFull( mode, args ):
   intfOrAddr = args.get( 'ADDR' ) or args.get( 'INTFS' )
   vrfName = args.get( 'VRF' )
   return filterIpIntf( mode, printIpIntfFull, intfOrAddr, vrfName )

def showIpIntfBrief( mode, args ):
   intfOrAddr = args.get( 'ADDR' ) or args.get( 'INTFS' )
   vrfName = args.get( 'VRF' )
   return filterIpIntf( mode, printIpIntfBrief, intfOrAddr, vrfName )

vrfKwMatcher = CliMatcher.KeywordMatcher( 'vrf', helpdesc='VRF instance' )
vrfNameMatcher = CliMatcher.DynamicNameMatcher( lambda mode: allVrfConfig.vrf,
                                                'VRF name' )

interfaceAfterShowMatcher = CliMatcher.KeywordMatcher(
   'interface',
   helpdesc='Details related to IPv4 interfaces' )

class ShowIpIntfCmd( ShowCommand.ShowCliCommandClass ):
   syntax = """ show ip interface
                [ INTFS | ADDR ]
                [ vrf VRF ]
            """
   data = { 'ip': ipMatcherForShow,
            'interface': interfaceAfterShowMatcher,
            'INTFS': IntfCli.Intf.rangeMatcher,
            'ADDR': IpAddrMatcher.IpAddrMatcher( 'Match this IP Address' ),
            'vrf': vrfKwMatcher,
            'VRF': vrfNameMatcher
          }
   cliModel = IpStatuses

   handler = showIpIntfFull

BasicCli.addShowCommandClass( ShowIpIntfCmd )

briefKwMatcher = CliMatcher.KeywordMatcher( 'brief', helpdesc='Condensed output' )

class ShowIpIntfBriefCmd( ShowCommand.ShowCliCommandClass ):
   syntax = """ show ip interface
                [ INTFS | ADDR ]
                [ vrf VRF ]
                brief
            """
   data = { 'ip': ipMatcherForShow,
            'interface': interfaceAfterShowMatcher,
            'INTFS': IntfCli.Intf.rangeMatcher,
            'ADDR': IpAddrMatcher.IpAddrMatcher( 'Match this IP Address' ),
            'vrf': vrfKwMatcher,
            'VRF': vrfNameMatcher,
            'brief': briefKwMatcher
          }
   cliModel = IpStatusesBrief

   handler = showIpIntfBrief

BasicCli.addShowCommandClass( ShowIpIntfBriefCmd )

#-------------------------------------------------------------------------------
# The "show ip route gateway" command.
#-------------------------------------------------------------------------------
def warnIfRoutingDisabled( mode, vrfName=DEFAULT_VRF ):
   # OSPF (and other protocols in future) allow creating instances even before
   # corresponding VRF is defined. In such cases, vrf/config/<vrfname> may not
   # exist yet.
   if vrfName and not vrfExists( vrfName ):
      mode.addWarning( "IP routing not enabled" )
   elif not routing.config( vrfName ).routing:
      mode.addWarning( "IP routing not enabled" )

#-------------------------------------------------------------------------------
# The "show ip route host" command.
#-------------------------------------------------------------------------------
def showRouteHost( mode, vrfName=None, quantify=False ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   if vrfName and not vrfExists( vrfName ):
      if vrfName != ALL_VRF_NAME:
         mode.addError( "IP routing table %s does not exist." % vrfName )
         return None
   model = IraRouteCommon.showRouteHostCommon( mode, vrfName, quantify,
                        allVrfStatusLocal, routing, allVrfConfig )
   return model

#-------------------------------------------------------------------------------
# The "show ip route fec protocol summary" command.
#-------------------------------------------------------------------------------
showIpFecProtocolSummary = routing.showFecProtocolSummary

#-------------------------------------------------------------------------------
# The "show ip route [vrf <vrfName>|all] ..." command.
#-------------------------------------------------------------------------------
routeMatcherAfterShowIp = CliMatcher.KeywordMatcher( 'route',
      helpdesc='Entries used for routing packets' )

ipPrefixExpr = IpAddrMatcher.ipPrefixExpr(
      'Match this IP address', 'Match this subnet mask',
      'Match this IP prefix', partial=True,
      overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO )

detailKwAfterShowIpRoute = CliCommand.singleKeyword( 'detail',
      helpdesc='All routes' )

sharedCountObj = object()

def _protocolMatcher( protocolName, helpdesc ):
   return CliCommand.Node( CliMatcher.KeywordMatcher(
                              protocolName, helpdesc=helpdesc ),
                           sharedMatchObj=sharedCountObj, maxMatches=1,
                           alias='PROTOCOL' )

class ProtocolCliExpression( CliCommand.CliExpression ):
   expression = "( bgp | connected | kernel | martian | dynamic-policy | " \
                   "ospf | ospfv3 | isis | rip | static | vxlan-control-service | " \
                   "aggregate | cached | gribi | cbf )"

   data = { 'bgp': _protocolMatcher( 'bgp', 'Show only BGP routes' ),
            'connected': _protocolMatcher( 'connected',
                                           'Show only connected routes' ),
            'kernel': _protocolMatcher( 'kernel', 'Show only kernel routes' ),
            'martian': _protocolMatcher( 'martian', 'Show only kernel routes' ),
            'dynamic-policy': _protocolMatcher( 'dynamic-policy',
                                                'Show only Dynamic Policy routes' ),
            'ospf': _protocolMatcher( 'ospf', 'Show only OSPF routes' ),
            'ospfv3': _protocolMatcher( 'ospfv3', 'Show only OSPFv3 routes' ),
            'isis': _protocolMatcher( 'isis', 'Show only IS-IS routes' ),
            'rip': _protocolMatcher( 'rip', 'Show only RIP routes' ),
            'static': _protocolMatcher( 'static', 'Show only static routes' ),
            'vxlan-control-service': _protocolMatcher( 'vxlan-control-service',
                                          'Show only VXLAN Control Service routes' ),
            'aggregate': _protocolMatcher( 'aggregate',
                                           'Show only aggregate routes' ),
            'cached': _protocolMatcher( 'cached', 'Show only Route Cache routes' ),
            'gribi': _protocolMatcher( 'gribi', 'Show only GRIBI routes' ),
          }
   data.update( { 'cbf':
               _protocolMatcher( 'cbf', 'Show only CBF Leaked routes' ) } )

def showRoute( mode, prefix=None, optionsSet=None, vrfName=None,
               routeFilter=None, flattened=False ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   outputFormat = printer.stringToOutputFormat( mode.session_.outputFormat_ )
   fd = sys.stdout.fileno()

   if flattened:
      mode.addWarning( "This command will be deprecated and superseded"
                       " by show ip route" )

   # Return with a warning if a multicast prefix is specified
   if prefix is not None:
      savedPrefix = prefix
      prefix = Arnet.Prefix( prefix ) # coerce to TACC prefix type
      if not prefix.isUnicast:
         mode.addWarning( notUnicastPrefix % savedPrefix )
         return VrfIpRoutes

   if not ( vrfName is None or vrfExists( vrfName ) ):
      if vrfName != ALL_VRF_NAME:
         mode.addError( noRouteTableForVrfMsg % vrfName )
         return None

   p = printer.initPrinter( fd, outputFormat, True )
   printer.startRender( p )
   printer.startDict( p, "vrfs" )
   if vrfName == ALL_VRF_NAME:
      vrfs = sorted( allVrfStatusLocal.vrf )
      vrfs.insert( 0, "default" )
   else:
      vrfs = [ vrfName ]

   for vrf in vrfs:
      # make sure we don't attempt to write if SIGINT happened outside the
      # immediateMode section (since RouteSorter handles those internally)
      #
      # Additionally, "VRF: ..." header is printed inside RouterSorter C++ code
      # for safety when pager exits, see BUG284545
      TacSigint.check()

      printer.startDict( p, vrf )
      keepGoing = showRoutePerVrf( p, mode, prefix, optionsSet,
                                   vrf, routeFilter, flattened )
      printer.endDict( p, vrf )
      if not keepGoing:
         break

   printer.endDict( p, "vrfs" )
   printer.endRender( p )
   printer.deinitPrinter( p )
   return VrfIpRoutes

def showRoutePerVrf( p, mode, prefix=None, optionsSet=None, vrfName="default",
                     routeFilter=None, flattened=False ):
   optionsDict = dict( optionsSet or [] )
   detail = 'detail' in optionsDict
   longerPrefixes = 'longerPrefixes' in optionsDict
   quantify = 'quantify' in optionsDict.values()
   protocol = optionsDict.get( 'protocol' )
   if isinstance( prefix, str ):
      prefix = Arnet.Prefix( prefix )
   # create the outputFormat here, since we will not have access to mode later
   # down the stack, in routing.showRoute
   fmt = mode.session_.outputFormat()
   requestedRevision = mode.session_.requestedModelRevision()
   # In revision 2 of the VrfIpRoutes CAPI model (IraIpModel.py) we moved
   # the Nexthop-Group submodel from the per-route level to the per-via level.
   # If revision 1 is requested, we "degrade" the JSON output by picking
   # the first Nexthop-Group that a route points to and putting it in the per-route
   # level Nexthop-Group submodel.
   nhgModelRevision = 2

   # revision 3 prints the nested tunnel representation for routes resolving over
   # tunnels. If a lower revision is requested, it will flatten the tunnel entries,
   # and print the final nexthop, interface and combined label stack
   flatTunnelRevision = 3

   # revision 4 makes srTePolicy in cbfVias to be optional when RSVP CBF is used
   rsvpCbfViaRevision = 4

   json = not mode.session_.shouldPrint()
   degradeNhgModel = json and requestedRevision < nhgModelRevision
   flattenTunnelOverTunnel = json and requestedRevision < flatTunnelRevision
   degradeCbfViasModel = json and requestedRevision < rsvpCbfViaRevision

   keepGoing = routing.showRoute( prefix, None, protocol, detail, vrfName,
                                  longerPrefixes, quantify=quantify, fmt=fmt,
                                  degradeNhgModel=degradeNhgModel,
                                  flattenTunnelOverTunnel=flattenTunnelOverTunnel,
                                  degradeCbfViasModel=degradeCbfViasModel,
                                  routeFilter=routeFilter, flattened=flattened )
   IraRouteCommon.warnIfRoutingDisabledPrinter( p, mode, vrfName, routing,
                                                allVrfConfig )
   return keepGoing

vrfExprFactoryForShow = VrfExprFactory( helpdesc='Show routes in a VRF',
                                        inclDefaultVrf=True )

def showIpRouteAdapter( mode, args, argsList ):
   ipPrefix = args.pop( 'PREFIX', None )
   if ipPrefix is not None:
      args[ 'PREFIX' ] = ipPrefix
   elif 'ADDR' in args:
      args[ 'PREFIX' ] = args.pop( 'ADDR' )

   args[ 'optionSet' ] = set()
   if 'detail' in args:
      del args[ 'detail' ]
      args[ 'optionSet' ].add( ( 'detail', 'detail' ) )
   if 'PROTOCOL' in args:
      args[ 'optionSet' ].add( ( 'protocol', args.pop( 'PROTOCOL' ) ) )
   if 'quantify' in args:
      del args[ 'quantify' ]
      args[ 'optionSet' ].add( ( 'quantify', 'quantify' ) )

class ShowIpRouteCmd( ShowCommand.ShowCliCommandClass ):
   syntax = """ show ip route
                [ VRF ]
                [ PREFIX | ADDR | flattened ]
                [ { detail | PROTOCOL | quantify } ]
                [ NEXTHOP ]
            """
   data = { 'ip': ipMatcherForShow,
            'route': routeMatcherAfterShowIp,
            'VRF': vrfExprFactoryForShow,
            'PREFIX': ipPrefixExpr,
            'ADDR': IpAddrMatcher.ipAddrMatcher,
            'flattened': 'show flattened adjacencies',
            'detail': detailKwAfterShowIpRoute,
            'PROTOCOL': ProtocolCliExpression,
            'quantify': CliCommand.Node( CliMatcher.KeywordMatcher( 'quantify',
                                         helpdesc='Print the time taken' ),
                                         hidden=True, maxMatches=1 ),
            'NEXTHOP': NexthopCliExpression
         }
   cliModel = VrfIpRoutes

   adapter = showIpRouteAdapter

   @staticmethod
   def handler( mode, args ):
      return showRoute( mode, prefix=args.get( 'PREFIX' ),
               optionsSet=args[ 'optionSet' ], vrfName=args.get( 'VRF' ),
               routeFilter=args.get( 'nexthop-matchers' ),
               flattened=bool( args.get( 'flattened' ) ) )

BasicCli.addShowCommandClass( ShowIpRouteCmd )

#-------------------------------------------------------------------------------
# The "show ip route next-hop nexthop-group" command
#-------------------------------------------------------------------------------

class ShowIpRouteNextHopGroupCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip route [ VRF ] next-hop nexthop-group [ NHG_NAME ]'
   data = { 'ip': ipMatcherForShow,
            'route': routeMatcherAfterShowIp,
            'VRF': vrfExprFactoryForShow,
            'next-hop': 'Filter output by next hop',
            'nexthop-group': 'Show only via next hop group',
            'NHG_NAME': nexthopGroupNameMatcher
            }
   cliModel = VrfIpRoutes
   handler = "IraIpCliHandler.handlerShowIpRouteNextHopGroupCmd"

BasicCli.addShowCommandClass( ShowIpRouteNextHopGroupCmd )

# -------------------------------------------------------------------------------
# The "show ip route [ VRF ] next-hop vxlan [ VTEP_IP ]" command
# -------------------------------------------------------------------------------

class ShowIpRouteNextHopVxlanCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip route [ VRF ] next-hop vxlan [ IP ] [ all-vias ]'
   data = {
      'ip': ipMatcherForShow,
      'route': routeMatcherAfterShowIp,
      'VRF': vrfExprFactoryForShow,
      'next-hop': 'Filter output by next hop',
      'vxlan': 'Show only via next hop VTEP',
      'IP': IpGenAddrMatcher.ipGenAddrMatcher,
      'all-vias': 'Show all the vias of a route if one of those vias matches',
   }
   cliModel = VrfIpRoutes
   handler = "IraIpCliHandler.handlerShowIpRouteNextHopVxlanCmd"

BasicCli.addShowCommandClass( ShowIpRouteNextHopVxlanCmd )

#-------------------------------------------------------------------------------
# The "show ip route static not-installed bfd" command
#-------------------------------------------------------------------------------

class ShowNotInstalledBfdCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip route [ VRF ] static not-installed bfd'
   data = { 'ip': ipMatcherForShow,
            'route': routeMatcherAfterShowIp,
            'VRF': vrfExprFactoryForShow,
            'static': 'Show only static routes',
            'not-installed': 'Show not installed routes',
            'bfd': 'Not installed routes tracked by Bfd'
          }
   cliModel = VrfNotInstalledBfd
   handler = "IraIpCliHandler.handlerShowNotInstalledBfdCmd"

BasicCli.addShowCommandClass( ShowNotInstalledBfdCmd )

def vrfSummaryCommon( mode, ipRouteSummaryModel, vrfs ):
   routeCountByTypeDict = { 'connected':0,
                            'static':0,
                            'staticPersistent':0,
                            'staticNonPersistent':0,
                            'vcs':0,
                            'staticNexthopGroup':0,
                            'rip':0,
                            'internal':0,
                            'attached':0,
                            'dynamicPolicy':0,
                            'gribi': 0,
                            'aggregate': 0,
                            'ospfTotal': 0,
                            'ospfIntraArea': 0,
                            'ospfInterArea': 0,
                            'ospfExternal1': 0,
                            'ospfExternal2': 0,
                            'nssaExternal1': 0,
                            'nssaExternal2': 0,
                            'ospfv3IntraArea': 0,
                            'ospfv3InterArea': 0,
                            'ospfv3External1': 0,
                            'ospfv3External2': 0,
                            'ospfv3NssaExternal1': 0,
                            'ospfv3NssaExternal2': 0,
                            'ospfv3Total': 0,
                            'bgpTotal': 0,
                            'bgpInternal': 0,
                            'bgpExternal': 0,
                            'bgpLocal': 0,
                            'isisTotal': 0,
                            'isisLevel1': 0,
                            'isisLevel2': 0 }
   useAllVrfRouteSummary = (
      getEffectiveProtocolModel( mode ) == ProtoAgentModel.multiAgent and
      allVrfRouteSummary and allVrfRouteSummary.vrfRouteSummary )
   for vrfName in vrfs:
      if useAllVrfRouteSummary:
         vrfRouteSummaryMultiAgent( vrfName, ipRouteSummaryModel,
                                    routeCountByTypeDict )
      else:
         vrfRouteSummarySingleAgent( vrfName, ipRouteSummaryModel,
                                     routeCountByTypeDict )
   if useAllVrfRouteSummary:
      # CountByType totals
      routeCountByTypeDict[ 'ospfTotal' ] += routeCountByTypeDict[ 'ospfIntraArea' ]
      routeCountByTypeDict[ 'ospfTotal' ] += routeCountByTypeDict[ 'ospfInterArea' ]
      routeCountByTypeDict[ 'ospfTotal' ] += routeCountByTypeDict[ 'ospfExternal1' ]
      routeCountByTypeDict[ 'ospfTotal' ] += routeCountByTypeDict[ 'ospfExternal2' ]
      routeCountByTypeDict[ 'ospfTotal' ] += routeCountByTypeDict[ 'nssaExternal1' ]
      routeCountByTypeDict[ 'ospfTotal' ] += routeCountByTypeDict[ 'nssaExternal2' ]
      
      routeCountByTypeDict[ 'ospfv3Total' ] += \
         routeCountByTypeDict[ 'ospfv3InterArea' ]
      routeCountByTypeDict[ 'ospfv3Total' ] += \
         routeCountByTypeDict[ 'ospfv3IntraArea' ]
      routeCountByTypeDict[ 'ospfv3Total' ] += \
         routeCountByTypeDict[ 'ospfv3External1' ]
      routeCountByTypeDict[ 'ospfv3Total' ] += \
         routeCountByTypeDict[ 'ospfv3External2' ]
      routeCountByTypeDict[ 'ospfv3Total' ] += \
         routeCountByTypeDict[ 'ospfv3NssaExternal1' ]
      routeCountByTypeDict[ 'ospfv3Total' ] += \
         routeCountByTypeDict[ 'ospfv3NssaExternal2' ]

      routeCountByTypeDict[ 'bgpTotal' ] += routeCountByTypeDict[ 'bgpInternal' ]
      routeCountByTypeDict[ 'bgpTotal' ] += routeCountByTypeDict[ 'bgpExternal' ]
      routeCountByTypeDict[ 'bgpTotal' ] += routeCountByTypeDict[ 'bgpLocal' ]

      routeCountByTypeDict[ 'isisTotal' ] += routeCountByTypeDict[ 'isisLevel1' ]
      routeCountByTypeDict[ 'isisTotal' ] += routeCountByTypeDict[ 'isisLevel2' ]

   ipRouteSummaryModel.connected = routeCountByTypeDict[ 'connected' ]
   ipRouteSummaryModel.static = routeCountByTypeDict[ 'static' ]
   ipRouteSummaryModel.staticPersistent = routeCountByTypeDict[ 'staticPersistent' ]
   ipRouteSummaryModel.staticNonPersistent = \
       routeCountByTypeDict[ 'staticNonPersistent' ]
   ipRouteSummaryModel.vcs = routeCountByTypeDict[ 'vcs' ]
   ipRouteSummaryModel.staticNexthopGroup = \
       routeCountByTypeDict[ 'staticNexthopGroup' ]
   ipRouteSummaryModel.rip = routeCountByTypeDict[ 'rip' ]
   ipRouteSummaryModel.internal = routeCountByTypeDict[ 'internal' ]
   ipRouteSummaryModel.attached = routeCountByTypeDict[ 'attached' ]
   ipRouteSummaryModel.aggregate = routeCountByTypeDict[ 'aggregate' ]
   ipRouteSummaryModel.dynamicPolicy = routeCountByTypeDict[ 'dynamicPolicy' ]
   ipRouteSummaryModel.gribi = routeCountByTypeDict[ 'gribi' ]

   ospfTotalsModel = ospfTotals()
   ospfTotalsModel.ospfTotal = routeCountByTypeDict[ 'ospfTotal' ]
   ospfTotalsModel.ospfIntraArea = routeCountByTypeDict[ 'ospfIntraArea' ]
   ospfTotalsModel.ospfInterArea = routeCountByTypeDict[ 'ospfInterArea' ]
   ospfTotalsModel.ospfExternal1 = routeCountByTypeDict[ 'ospfExternal1' ]
   ospfTotalsModel.ospfExternal2 = routeCountByTypeDict[ 'ospfExternal2' ]
   ospfTotalsModel.nssaExternal1 = routeCountByTypeDict[ 'nssaExternal1' ]
   ospfTotalsModel.nssaExternal2 = routeCountByTypeDict[ 'nssaExternal2' ]
   ipRouteSummaryModel.ospfCounts = ospfTotalsModel

   ospfv3TotalsModel = ospfv3Totals()
   ospfv3TotalsModel.ospfv3Total = routeCountByTypeDict[ 'ospfv3Total' ]
   ospfv3TotalsModel.ospfv3IntraArea = routeCountByTypeDict[ 'ospfv3IntraArea' ]
   ospfv3TotalsModel.ospfv3InterArea = routeCountByTypeDict[ 'ospfv3InterArea' ]
   ospfv3TotalsModel.ospfv3External1 = routeCountByTypeDict[ 'ospfv3External1' ]
   ospfv3TotalsModel.ospfv3External2 = routeCountByTypeDict[ 'ospfv3External2' ]
   ospfv3TotalsModel.ospfv3NssaExternal1 = \
       routeCountByTypeDict[ 'ospfv3NssaExternal1' ]
   ospfv3TotalsModel.ospfv3NssaExternal2 = \
       routeCountByTypeDict[ 'ospfv3NssaExternal2' ]

   ipRouteSummaryModel.ospfv3Counts = ospfv3TotalsModel

   bgpTotalsModel = bgpTotals()
   bgpTotalsModel.bgpTotal = routeCountByTypeDict[ 'bgpTotal' ]
   bgpTotalsModel.bgpExternal = routeCountByTypeDict[ 'bgpExternal' ]
   bgpTotalsModel.bgpInternal = routeCountByTypeDict[ 'bgpInternal' ]
   bgpTotalsModel.bgpLocal = routeCountByTypeDict[ 'bgpLocal' ]
   ipRouteSummaryModel.bgpCounts = bgpTotalsModel

   isisTotalsModel = isisTotals()
   isisTotalsModel.isisTotal = routeCountByTypeDict[ 'isisTotal' ]
   isisTotalsModel.isisLevel1 = routeCountByTypeDict[ 'isisLevel1' ]
   isisTotalsModel.isisLevel2 = routeCountByTypeDict[ 'isisLevel2' ]
   ipRouteSummaryModel.isisCounts = isisTotalsModel

   return ipRouteSummaryModel

def vrfRouteSummarySingleAgent( vrfName, ipRouteSummaryModel, routeCountByTypeDict ):
   routeConfig = routing.routeConfig( vrfName )
   assert routeConfig
   routeConfigDynamic = routing.routeConfig( vrfName, dynamic=True )
   assert routeConfigDynamic
   routeSummary = Tac.newInstance( "Ira::RouteSummary",
                                   routing.config( vrfName ),
                                   routeConfig,
                                   routeConfigDynamic,
                                   routing.rStatus( vrfName ),
                                   routing.attachRouteStatus( vrfName ),
                                   routing.vrfIdMap(),
                                   routing.ip.ribBypassConfig,
                                   vrfName )
   routeSummary.loadInternalStructures()
   routeSummary.loadAttachedRoutes()
   routeSummary.loadRouteCacheRoutes()

   maskLenCounter = defaultdict( int, ipRouteSummaryModel.maskLen )
   for key, value in routeSummary.routeCountByMask.items():
      maskLenCounter[ key ] += value
   ipRouteSummaryModel.maskLen = dict( maskLenCounter )

   fibSummaryTable = routeSummary.routeCountByType
   for source in routeCountByTypeDict:
      if source in fibSummaryTable:
         routeCountByTypeDict[ source ] += fibSummaryTable[ source ]
   ipRouteSummaryModel.totalRoutes += routeSummary.totalNumRoutes

def vrfRouteSummaryMultiAgent( vrfName, ipRouteSummaryModel, routeCountByTypeDict ):
   # TODO BUG804766:
   # Remove this when routeCache is part of the aggregated allVrfRouteSummary
   routeConfig = routing.routeConfig( vrfName )
   assert routeConfig
   routeConfigDynamic = routing.routeConfig( vrfName, dynamic=True )
   assert routeConfigDynamic
   routeSummary = Tac.newInstance( "Ira::RouteSummary",
                                   routing.config( vrfName ),
                                   routeConfig,
                                   routeConfigDynamic,
                                   routing.rStatus( vrfName ),
                                   routing.attachRouteStatus( vrfName ),
                                   routing.vrfIdMap(),
                                   routing.ip.ribBypassConfig,
                                   vrfName )
   routeSummary.loadRouteCacheRoutes()
   staticNonPersistent = routeSummary.loadStaticNonPersistentCounter()
   routeCountByTypeDict[ 'staticNonPersistent' ] += staticNonPersistent

   maskLenCounter = defaultdict( int, ipRouteSummaryModel.maskLen )
   for key, value in routeSummary.routeCountByMask.items():
      maskLenCounter[ key ] += value

   # routeCache has only static routes
   oldFibSummaryByType = routeSummary.routeCountByType
   totalTypesCount = oldFibSummaryByType.get( 'static', 0 )
   routeCountByTypeDict[ 'static' ] += totalTypesCount
   # routeCache route is always staticPersistent
   routeCountByTypeDict[ 'staticPersistent' ] += totalTypesCount

   # Shark based approach starts from here
   vrfId = routing.vrfId( vrfName )
   assert allVrfRouteSummary
   vrfRouteSummary = allVrfRouteSummary.vrfRouteSummary.get( vrfId )
   assert vrfRouteSummary

   maskLenSummary = vrfRouteSummary.v4MaskLen
   for key, maskLenInfo in maskLenSummary.items():
      maskLenCounter[ key ] += maskLenInfo.count

   fibSummaryTable = vrfRouteSummary.v4RouteCount
   enumTypes = Tac.Type( "Routing::Fib::RouteType" ).attributes
   protocolCodeWordHelper = Tac.Type( 'Ira::ProtocolCodeWordHelper' )
   for enumName in enumTypes:
      fibType = Tac.enumValue( 'Routing::Fib::RouteType', enumName )
      routeType = Tac.newInstance( 'Routing::RouteSummary::V4RouteType', fibType )
      if routeType in fibSummaryTable:
         typeCount = fibSummaryTable[ routeType ].count
         protocolTypeWord = (
            protocolCodeWordHelper.routeTypeToProtocolTypeWord( fibType ) )

         if enumName == 'staticConfig':
            routeCountByTypeDict[ 'staticPersistent' ] += (
               typeCount - staticNonPersistent )

         if protocolTypeWord in routeCountByTypeDict:
            routeCountByTypeDict[ protocolTypeWord ] += typeCount
            totalTypesCount += typeCount

   ipRouteSummaryModel.maskLen = dict( maskLenCounter )
   ipRouteSummaryModel.totalRoutes += totalTypesCount

#-------------------------------------------------------------------------------
# The "show ip route [vrf <vrfName>] summary" command.
#-------------------------------------------------------------------------------
def showRouteSummary( mode, vrfName=None, quantify=False ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   vrfsToShow = []

   if not ( vrfName is None or vrfExists( vrfName ) ):
      if vrfName != ALL_VRF_NAME:
         mode.addError( "Ip routing table %s does not exist." % vrfName )
         return None

   if vrfName == ALL_VRF_NAME:
      vrfsToShow = [ "default" ]
      vrfsToShow.extend( sorted( allVrfStatusLocal.vrf ) )
   else:
      vrfsToShow = [ vrfName ]

   svcRoutingProtoModelStatus = ServiceRoutingProtoModelStatus(
      operatingProtoModel=l3Config.protocolAgentModel,
      configuredProtoModel=l3Config.configuredProtocolAgentModel )
   ipSummary = IpRouteSummary()

   ipSummary.protoModelStatus = svcRoutingProtoModelStatus
   for vrf in vrfsToShow:
      result = showRouteSummaryForVrf( mode, vrfName=vrf,
                                       quantify=quantify )
      if result:
         ipSummary.vrfs[ vrf ] = result

   return ipSummary

def showRouteSummaryForVrf( mode, vrfName=None, quantify=False ):
   beforeSum = Tac.now()
   IpRouteSummaryModel = IpRouteSummaryForVrf()
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   if not ( vrfName is None or vrfExists( vrfName ) ):
      mode.addError( "IP routing table %s does not exist." % vrfName )
      return None

   if not routing.rStatus( vrfName ):
      return None

   vrfs = [vrfName]
   IpRouteSummaryModel.totalRoutes = 0
   IpRouteSummaryModel = vrfSummaryCommon( mode, IpRouteSummaryModel, vrfs )
   # pylint: disable=protected-access
   if quantify:
      afterSum = Tac.now()
      IpRouteSummaryModel._quantify = afterSum-beforeSum
   return IpRouteSummaryModel

summaryAfterShowIpRouteMatcher = CliMatcher.KeywordMatcher( 'summary',
                                     helpdesc='IP routing table summary' )
quantifyKwMatcher = CliCommand.Node( CliMatcher.KeywordMatcher( 'quantify',
                                       helpdesc='Print the time taken' ),
                                     hidden=True )

class ShowIpRouteSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip route [ VRF ] summary [ quantify ]'
   data = { 'ip': ipMatcherForShow,
            'route': routeMatcherAfterShowIp,
            'VRF': vrfExprFactoryForShow,
            'summary': summaryAfterShowIpRouteMatcher,
            'quantify': quantifyKwMatcher
          }
   cliModel = IpRouteSummary

   @staticmethod
   def handler( mode, args ):
      quantify = 'quantify' in args
      return showRouteSummary( mode, vrfName=args.get( 'VRF' ),
                               quantify=quantify )

BasicCli.addShowCommandClass( ShowIpRouteSummaryCmd )

hostAfterShowIpRouteMatcher = CliMatcher.KeywordMatcher(
   'host',
   helpdesc='Host routes' )

fecMatcher = CliMatcher.KeywordMatcher( 'fec', helpdesc="show route adjacencies" )

summaryMatcherAfterShowIp = CliMatcher.KeywordMatcher( 'summary',
                                    helpdesc='show adjacency size and distribution' )

class ShowIpRouteHostCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip route [ VRF ] host [ quantify ]'
   data = { 'ip': ipMatcherForShow,
            'route': routeMatcherAfterShowIp,
            'VRF': vrfExprFactoryForShow,
            'host': hostAfterShowIpRouteMatcher,
            'quantify': quantifyKwMatcher
          }
   cliModel = VrfIpRoutesHost

   @staticmethod
   def handler( mode, args ):
      quantify = 'quantify' in args
      return showRouteHost( mode, args.get( 'VRF' ), quantify=quantify )


BasicCli.addShowCommandClass( ShowIpRouteHostCmd )

class ShowIpRouteFecCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip route [ VRF ] [ PREFIX | ADDR ] fec'
   data = { 'ip': ipMatcherForShow,
            'route': routeMatcherAfterShowIp,
            'VRF': vrfExprFactoryForShow,
            'PREFIX': ipPrefixExpr,
            'ADDR': IpAddrMatcher.ipAddrMatcher,
            'fec': fecMatcher
          }

   cliModel = IraRouteCommon.VrfFecRoutesModel

   @staticmethod
   def adapter( mode, args, argsList ):
      ipPrefix = args.pop( 'PREFIX', None )
      if ipPrefix is not None:
         args[ 'PREFIX' ] = ipPrefix
      elif 'ADDR' in args:
         args[ 'PREFIX' ] = args.pop( 'ADDR' )

   handler = "IraIpCliHandler.handlerShowIpRouteFecCmd"

BasicCli.addShowCommandClass( ShowIpRouteFecCmd )

class ShowIpFecSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip route [ VRF ] fec summary'
   data = { 'ip': ipMatcherForShow,
            'route': routeMatcherAfterShowIp,
            'VRF': vrfExprFactoryForShow,
            'fec': fecMatcher,
            'summary': summaryMatcherAfterShowIp
            }
   cliModel = IraRouteCommon.FecSummaryModel
   handler = "IraIpCliHandler.handlerShowIpFecSummaryCmd"

BasicCli.addShowCommandClass( ShowIpFecSummaryCmd )

class ShowIpFecProtocolSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show ip route fec protocol summary"
   data = { 'ip': ipMatcherForShow,
            'route': routeMatcherAfterShowIp,
            'fec': fecMatcher,
            'protocol': "Group adjacencies by protocol",
            'summary': "Show adjacency distribution by protocol",
            }
   cliModel = IraRouteCommon.FecProtocolSummaryModel

   @staticmethod
   def handler( mode, args ):
      return showIpFecProtocolSummary( mode )

BasicCli.addShowCommandClass( ShowIpFecProtocolSummaryCmd )

#-------------------------------------------------------------------------------
# The "show ip route [vrf vrfName] (prefix) longer-prefixes" command.
#-------------------------------------------------------------------------------

longerPrefixesMatcherAfterShowIpRoute = CliMatcher.KeywordMatcher(
   'longer-prefixes', helpdesc='Match longer prefixes' )

class ShowIpRouteLongerPrefixesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip route [ VRF ] ( PREFIX | ADDR ) ' \
            'longer-prefixes [ { detail | PROTOCOL | quantify } ]'
   data = { 'ip': ipMatcherForShow,
            'route': routeMatcherAfterShowIp,
            'VRF': vrfExprFactoryForShow,
            'PREFIX': ipPrefixExpr,
            'ADDR': IpAddrMatcher.ipAddrMatcher,
            'longer-prefixes': longerPrefixesMatcherAfterShowIpRoute,
            'detail': detailKwAfterShowIpRoute,
            'PROTOCOL': ProtocolCliExpression,
            'quantify': CliCommand.Node( CliMatcher.KeywordMatcher( 'quantify',
                                            helpdesc='Print the time taken' ),
                                         hidden=True, maxMatches=1 )
          }
   cliModel = VrfIpRoutes

   adapter = showIpRouteAdapter
   handler = "IraIpCliHandler.handlerShowIpRouteLongerPrefixesCmd"

BasicCli.addShowCommandClass( ShowIpRouteLongerPrefixesCmd )

#-------------------------------------------------------------------------------
# The "show ip route vrf all summary brief" command.
#-------------------------------------------------------------------------------

allKwMatcher = CliMatcher.KeywordMatcher( ALL_VRF_NAME,
            helpdesc='Show summary of all vrfs' )
vrfForShowMatcher = CliMatcher.KeywordMatcher( 'vrf',
            helpdesc='Specify the vrf' )

class ShowIpRouteSummaryBriefCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip route vrf all summary brief [ quantify ]'
   data = { 'ip': ipMatcherForShow,
            'route': routeMatcherAfterShowIp,
            'vrf': vrfForShowMatcher,
            'all': allKwMatcher,
            'summary': summaryAfterShowIpRouteMatcher,
            'brief': briefKwMatcher,
            'quantify': quantifyKwMatcher
          }
   cliModel = IpRouteSummaryForVrf
   handler = "IraIpCliHandler.handlerShowIpRouteSummaryBriefCmd"

BasicCli.addShowCommandClass( ShowIpRouteSummaryBriefCmd )

#-------------------------------------------------------------------------------
# Determine if assigning the given address/mask to some interface in Ira
# should be allowed

# return a pair of a boolean with a string, where the string is an error or
# warning message
#-------------------------------------------------------------------------------
def canSetIntfIp( intfName, ipAddrWithMask, secondary=False, mode=None ):
   config = routing.routeConfig( vrfName=None, intfName=intfName )
   return IraRouteCommon.canSetIntfIpHelper( config, intfName, ipAddrWithMask,
                                             'ipv4', True )

#-------------------------------------------------------------------------------
# The Resilient Ecmp command
# The syntax of the command is:
#  ip hardware fib ecmp <prefix> capacity <C> redundancy <R>
#  no/default ip hardware fib ecmp <prefix>
#  no/default ip hardware fib ecmp
#-------------------------------------------------------------------------------

def maxResilientEcmpCapacityRangeFn( mode, context ):
   if ( routingHardwareStatus is None or
        routingHardwareStatus.maxLogicalProtocolEcmp == 0 ):
      return( 1, sys.maxsize )
   else :
      return( 2, routingHardwareStatus.maxLogicalProtocolEcmp )

def maxResilientEcmpRedundancyRangeFn( mode, context ):
   if routingHardwareStatus is None or routingHardwareStatus.maxResilientEcmp == 0:
      return( 1, sys.maxsize )
   else:
      return( 1, routingHardwareStatus.maxResilientEcmp // 2 )

def resilientEcmpGuard( mode, token ):
   if routingHardwareStatus.resilientEcmpSupported:
      return None
   return CliParser.guardNotThisPlatform

def orderedRecmpGuard( mode, token ):
   if routingHardwareStatus.orderedEcmpSupported:
      return None
   return CliParser.guardNotThisPlatform

hardwareHelpDesc = "Configure hardware-specific parameters"
fibHelpDesc = "Routing table"
ecmpHelpDesc = "Configure ECMP routes"
resilienceHelpDesc = "Configure Resilience in ECMP routes"
markedRecmpHelpDesc = "Applies to routes marked for ecmp resilience using policy"
orderedRecmpHelpDesc = "Enforce order of next hops in FEC" \
                       " for this resilient configuration"

class ResilientEcmpCmd( CliCommand.CliCommandClass ):
   syntax = """ip hardware fib ecmp resilience
            ( ( PREFIX capacity CAPACITY redundancy REDUNDANCY [ ordered ] )
            | ( marked capacity CAPACITY redundancy REDUNDANCY ) )"""
   noOrDefaultSyntax = "ip hardware fib ecmp resilience [ PREFIX | marked ] ..."

   data = { "ip": ipMatcherForConfig,
            "hardware": hardwareHelpDesc,
            "fib": fibHelpDesc,
            "ecmp": CliCommand.Node( matcher=CliMatcher.KeywordMatcher( "ecmp",
                                                helpdesc=ecmpHelpDesc ),
                                     guard=resilientEcmpGuard ),
            "resilience": resilienceHelpDesc,
            "PREFIX": IpAddrMatcher.ipPrefixExpr(
                         "Network address", "Network mask", "Prefix",
                         overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO,
                         maskKeyword=False ),
            "marked": CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
                                    "marked", helpdesc=markedRecmpHelpDesc ) ),
            "capacity": "specify capacity value",
            "CAPACITY": CliMatcher.DynamicIntegerMatcher(
                                       maxResilientEcmpCapacityRangeFn,
                                       helpdesc="capacity value" ),
            "redundancy": "specify redundancy value",
            "REDUNDANCY": CliMatcher.DynamicIntegerMatcher(
                                       maxResilientEcmpRedundancyRangeFn,
                                       helpdesc="redundancy value" ),
            "ordered": CliCommand.guardedKeyword(
                                    "ordered",
                                    helpdesc=orderedRecmpHelpDesc,
                                    guard=orderedRecmpGuard,
                                    hidden=False )
            }

   handler = "IraIpCliHandler.handlerResilientEcmpCmd"
   noOrDefaultHandler = "IraIpCliHandler.noResilientEcmp"

configMode.addCommandClass( ResilientEcmpCmd )

#-------------------------------------------------------------------------------
# "[no|default] ip hardware fib exact-match prefix-length <length>" command
#  This command is made hidden starting from ga.evans release
#-------------------------------------------------------------------------------

def routesInExactMatchGuard( mode, token ):
   if routingHardwareStatus.prefixLenSupportedInExactMatch:
      return None
   return CliParser.guardNotThisPlatform

def vrfRoutesInExactMatchGuard( mode, token ):
   if routingHardwareStatus.vrfPrefixLensInExactMatchSupported:
      return None
   return CliParser.guardNotThisPlatform

def prefixLengthGuard( mode, token ):
   if int( token ) not in routingHardwareStatus.prefixLenSupportedInExactMatch:
      return CliParser.guardNotThisPlatform
   return None

lenData = { str( i ): 'Prefix length %d' % i for i in range( 0, 33 ) }

def expandLenGuard( mode, token ):
   if int( token ) in routingHardwareStatus.prefixLenNotSupportedInExpansion:
      return CliParser.guardNotThisPlatform
   return None

def compressLenGuard( mode, token ):
   # We do not support prefix compression feature for prefix length part of
   # prefixLenNotSupportedInCompression
   if int( token ) in routingHardwareStatus.prefixLenNotSupportedInCompression:
      return CliParser.guardNotThisPlatform
   return None

class ExactMatchPrefixLenCmd( CliCommand.CliCommandClass ):
   hidden = True
   syntax = "ip hardware fib exact-match prefix-length LENGTH"
   noOrDefaultSyntax = "ip hardware fib exact-match prefix-length ..."

   data = { "ip": ipMatcherForConfig,
            "hardware": hardwareHelpDesc,
            "fib": fibHelpDesc,
            "exact-match": CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher( "exact-match",
                        helpdesc="Configure routes in exact-match table" ),
                  guard=routesInExactMatchGuard ),
            "prefix-length": "Length of the prefix in bits",
            "LENGTH": CliCommand.SetEnumMatcher( lenData, guard=prefixLengthGuard )
          }

   handler = "IraIpCliHandler.handlerExactMatchPrefixLenCmd"
   noOrDefaultHandler = "IraIpCliHandler.noOrDefaultHandlerExactMatchPrefixLenCmd"

configMode.addCommandClass( ExactMatchPrefixLenCmd )

#-------------------------------------------------------------------------------
# "[no|default] ip hardware fib optimize prefix-length <length>
#               [ expand < prefix-lengths > ]
#               [ compress < prefix-lengths > ] [ urpf ]" command
#-------------------------------------------------------------------------------

def urpfExactMatchSupportedGuard( mode, token ):
   if not routingHardwareStatus.urpfExactMatchSupported:
      return CliParser.guardNotThisPlatform
   return None

class OptimizePrefixLenCmd( CliCommand.CliCommandClass ):
   syntax = "ip hardware fib optimize prefix-length LENGTH " \
            "[ { ( expand EXPANDLEN ) | " \
            "( compress COMPRESSLEN ) } ] [ urpf ]"

   noOrDefaultSyntax = "ip hardware fib optimize prefix-length ..." \

   data = { "ip": ipMatcherForConfig,
            "hardware": hardwareHelpDesc,
            "fib": fibHelpDesc,
            "optimize": CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher( "optimize",
                        helpdesc="Optimize IPv4 routes" ),
                  guard=routesInExactMatchGuard ),
            "prefix-length": "Length of the prefix in bits",
            "LENGTH": CliCommand.SetEnumMatcher( lenData, guard=prefixLengthGuard ),
            "expand": CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher( "expand",
                     helpdesc="Expand and optimize routes" ),
                  maxMatches=1 ),
            "EXPANDLEN": CliCommand.SetEnumMatcher( lenData, guard=expandLenGuard ),
            "compress": CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher( "compress",
                     helpdesc="Compress and optimize routes" ),
                  maxMatches=1 ),
            "COMPRESSLEN": CliCommand.SetEnumMatcher( lenData,
                  guard=compressLenGuard ),
            "urpf": CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher( "urpf",
                     helpdesc="Enable RPF lookup" ),
                  guard=urpfExactMatchSupportedGuard )
          }

   handler = "IraIpCliHandler.handlerOptimizePrefixLenCmd"
   noOrDefaultHandler = "IraIpCliHandler.noOrDefaultHandlerOptimizePrefixLenCmd"

configMode.addCommandClass( OptimizePrefixLenCmd )

#-------------------------------------------------------------------------------
# "[no|default] ip hardware fib optimize
#               vrf (<vrf> [ <vrf2> ...])
#               prefix-length <length>
#               [ expand < prefix-lengths > ]
#               [ compress < prefix-lengths > ] [ urpf ]" command
#-------------------------------------------------------------------------------
class VrfOptimizePrefixLenCmd( CliCommand.CliCommandClass ):
   syntax = "ip hardware fib optimize " \
            "vrf { VRFS } " \
            "prefix-length LENGTH " \
            "[ { ( expand EXPANDLEN ) | ( compress COMPRESSLEN ) } ] [ urpf ]"

   noOrDefaultSyntax = "ip hardware fib optimize vrf { VRFS } ..."

   data = mergedDict(
         OptimizePrefixLenCmd.data,
         {
            "vrf": CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher(
                     "vrf", helpdesc="Optimize IPv4 routes for specific VRF(s)" ),
                  guard=vrfRoutesInExactMatchGuard ),
            "VRFS": VrfCli.VrfNameExprFactory( inclDefaultVrf=True ),
         } )

   handler = "IraIpCliHandler.handlerVrfOptimizePrefixLenCmd"
   noOrDefaultHandler = "IraIpCliHandler.noOrDefaultHandlerVrfOptimizePrefixLenCmd"

configMode.addCommandClass( VrfOptimizePrefixLenCmd )

#-------------------------------------------------------------------------------
# "[no|default] ip hardware fib optimize prefixes profile internet|urpf-internet"
# command
#-------------------------------------------------------------------------------
def supportedProfileTypeIpv4( mode ):
   return supportedProfileType( mode, routingHardwareStatus )

class OptimizePrefixesCmd( CliCommand.CliCommandClass ):
   syntax = "ip hardware fib optimize prefixes profile PROFILE"

   noOrDefaultSyntax = "ip hardware fib optimize prefixes ..."

   data = { "ip": ipMatcherForConfig,
            "hardware": hardwareHelpDesc,
            "fib": fibHelpDesc,
            "optimize": CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher( "optimize",
                        helpdesc="Optimize IPv4 routes" ),
                  guard=routesInExactMatchGuard ),
            "prefixes": CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher( "prefixes",
                        helpdesc="Optimize prefixes" ),
                  guard=routesInExactMatchGuard ),
            "profile": CliCommand.Node(
               matcher=CliMatcher.KeywordMatcher( "profile",
                        helpdesc="Optimize based on profiles" ),
               guard=routesInExactMatchGuard ),
            'PROFILE': CliMatcher.DynamicKeywordMatcher( supportedProfileTypeIpv4 )
          }

   handler = "IraIpCliHandler.handlerOptimizePrefixesCmd"
   noOrDefaultHandler = "IraIpCliHandler.noOrDefaultHandlerOptimizePrefixesCmd"

configMode.addCommandClass( OptimizePrefixesCmd )

#-------------------------------------------------------------------------------
# "[no|default] ip hardware fib optimize
#               vrf (<vrf> [ <vrf2> ...])
#               prefixes profile internet|urpf-internet"
# command
#-------------------------------------------------------------------------------
class VrfOptimizePrefixesCmd( CliCommand.CliCommandClass ):
   syntax = "ip hardware fib optimize " \
            "vrf { VRFS } " \
            "prefixes profile PROFILE"

   # The no command for the below commands are the same:
   # - "ip hardware fib optimize vrf { VRFS } prefixes" AND
   # - "ip hardware fib optimize vrf { VRFS } prefix-length"
   #
   # The no command is "no ip hardware fib optimize vrf { VRFS }". The no
   # handler is defined in VrfOptimizePrefixLenCmd class.

   data = mergedDict(
         OptimizePrefixesCmd.data,
         {
            "vrf": CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher(
                     "vrf", helpdesc="Optimize IPv4 routes for specific VRF(s)" ),
                  guard=vrfRoutesInExactMatchGuard ),
            "VRFS": VrfCli.VrfNameExprFactory( inclDefaultVrf=True ),
         } )

   handler = "IraIpCliHandler.handlerVrfOptimizePrefixesCmd"

configMode.addCommandClass( VrfOptimizePrefixesCmd )

#-------------------------------------------------------------------------------
# "[no|default] ip hardware fib optimize [ disable-vrf <vrf-name> ]" command
#-------------------------------------------------------------------------------

class SetDisableVrfCmd( CliCommand.CliCommandClass ):
   syntax = "ip hardware fib optimize [ ( disable-vrf { VRF } ) ]"

   noOrDefaultSyntax = "ip hardware fib optimize [ disable-vrf ] ..."

   data = { "ip": ipMatcherForConfig,
            "hardware": hardwareHelpDesc,
            "fib": fibHelpDesc,
            "optimize": CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher( "optimize",
                        helpdesc="Optimize IPv4 routes" ),
                  guard=routesInExactMatchGuard ),
            "disable-vrf": CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
                                    "disable-vrf",
                                    helpdesc="Disable fib optimizations in Vrf" ),
                                    guard=routesInExactMatchGuard ),
            "VRF": CliMatcher.DynamicNameMatcher(
                           lambda mode: allVrfConfig.vrf, "VRF name" )
          }

   handler = "IraIpCliHandler.handlerSetDisableVrfCmd"
   noOrDefaultHandler = "IraIpCliHandler.noOrDefaultHandlerSetDisableVrfCmd"

configMode.addCommandClass( SetDisableVrfCmd )

#-------------------------------------------------------------------------------
# The "show ip route rpf unicast [ vrf <vrf-name> ] [ prefix ]" command.
#-------------------------------------------------------------------------------

class ShowUrpfCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip route rpf unicast [ VRF ] [ PREFIX | ADDR ]'
   data = { 'ip': ipMatcherForShow,
            'route': routeMatcherAfterShowIp,
            'rpf': 'Reverse Path Forwarding information',
            'unicast': 'Unicast Reverse Path Forwarding information',
            'VRF': VrfExprFactory( helpdesc='Show Unicast Reverse Path '
                                            'Forwarding information in a VRF',
                                   inclDefaultVrf=True ),
            'PREFIX': IpAddrMatcher.ipPrefixExpr(
                         'Destination address',
                         'Destination subnet mask', 'Destination prefix',
                         partial=False,
                         overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO ),
            'ADDR': IpAddrMatcher.IpAddrMatcher( helpdesc='Destination address' )
          }

   cliModel = UrpfInterfaces

   handler = "IraIpCliHandler.handlerShowUrpfCmd"

   @staticmethod
   def adapter( mode, args, argsList ):
      ipPrefix = args.pop( 'PREFIX', None )
      if ipPrefix is not None:
         args[ 'PREFIX' ] = ipPrefix
      elif 'ADDR' in args:
         args[ 'PREFIX' ] = args.pop( 'ADDR' )

BasicCli.addShowCommandClass( ShowUrpfCmd )

#------------------------------------------------------------------------------------
#The "ip icmp redirect" and
#The "no|default ip icmp redirect"
#      commands.
#------------------------------------------------------------------------------------

class IpIcmpRedirectCmd( CliCommand.CliCommandClass ):
   syntax = "ip icmp redirect"

   noSyntax = "ip icmp redirect"

   defaultSyntax = "ip icmp redirect"

   data = { "ip": ipMatcherForConfig,
            "icmp": icmpMatcherForConfig,
            "redirect": "Send ICMP redirects"
          }

   handler = "IraIpCliHandler.handlerIpIcmpRedirectCmd"
   noHandler = "IraIpCliHandler.noHandlerIpIcmpRedirectCmd"
   defaultHandler = "IraIpCliHandler.defaultHandlerIpIcmpRedirectCmd"

configMode.addCommandClass( IpIcmpRedirectCmd )

#-------------------------------------------------------------------------------
# The "ip icmp source-interface" and  "no|default ip icmp source-interface"
# commands.
#-------------------------------------------------------------------------------

sourceInterfaceMatcherForConfig = CliMatcher.KeywordMatcher( 'source-interface',
      helpdesc="Which source interface's address to use" )

class SetIcmpSrcIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'ip icmp source-interface SRC_INTF [ VRF ]'
   noOrDefaultSyntax = 'ip icmp source-interface [ SRC_INTF ] [ VRF ]'
   data = { 'ip': ipMatcherForConfig,
            'icmp': icmpMatcherForConfig,
            'source-interface': sourceInterfaceMatcherForConfig,
            'SRC_INTF': IntfCli.Intf.matcherWithIpSupport,
            'VRF': VrfExprFactory( helpdesc='Configure ICMP source '
                                            'interface in a VRF' ),
          }

   handler = "IraIpCliHandler.handlerSetIcmpSrcIntfCmd"
   noOrDefaultHandler = "IraIpCliHandler.noOrDefaultHandlerSetIcmpSrcIntfCmd"

configMode.addCommandClass( SetIcmpSrcIntfCmd )

myEntManager = None

#-------------------------------------------------------------------------------
# The "ip fib compression redundant-specifics filter" and
# "[no|default] ip fib compression redundant-specifics filter"
# commands.
#-------------------------------------------------------------------------------
def fibSuppressionSupportedGuard( mode, token ):
   rhs = routingHardwareStatus
   if rhs is None:
      rhs = mode.sysdbRoot[ 'routing' ][ 'hardware' ][ 'status' ]
   if rhs.fibSuppressionSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

class FibCompressionCmd( CliCommand.CliCommandClass ):
   syntax = "ip fib compression redundant-specifics filter"

   noOrDefaultSyntax = "ip fib compression redundant-specifics filter"

   data = { "ip": ipMatcherForConfig,
            "fib": fibHelpDesc,
            "compression": CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher( "compression",
                              helpdesc="FIB compression" ),
                  guard=fibSuppressionSupportedGuard ),
            "redundant-specifics": "type of route filter to use",
            "filter": "filter command"
          }

   handler = "IraIpCliHandler.handlerFibCompressionCmd"
   noOrDefaultHandler = "IraIpCliHandler.noOrDefaultHandlerFibCompressionCmd"

configMode.addCommandClass( FibCompressionCmd )

#-------------------------------------------------------------------------------
# The "show ip user route" command
#-------------------------------------------------------------------------------
class ShowIpUserRouteCmd( ShowCommand.ShowCliCommandClass ):
   '''Display routes from the RIB that have the 'dynamic' attribute set to True
      These routes have been programmed via the EOS SDK with the 'persistent' flag
      set to False.'''
   syntax = 'show ip user route [ vrf VRF ] [ tag TAGNUM ] [ CONFIG_TAG_EXPR ]'
   data = {
         'ip': ipMatcherForShow,
         'user': 'Display non-persistent routes',
         'route': 'Static routes',
         'vrf': vrfForShowMatcher,
         'VRF': vrfNameMatcher,
         'tag': tagMatcherForConfig,
         'TAGNUM': tagNumberMatcher,
         'CONFIG_TAG_EXPR': configTagExpr
         }
   cliModel = VrfIpRibRoutes

   handler = "IraIpCliHandler.handlerShowIpUserRouteCmd"

BasicCli.addShowCommandClass( ShowIpUserRouteCmd )

#------------------------------------------------------------------------------
# The "ip address duplicate detection ( logging | disabled )" and
# The "no|default ip address duplicate detection" commands.
#------------------------------------------------------------------------------
class IpDuplicateAddressDetectionCmd( CliCommand.CliCommandClass ):
   '''duplicate address detection'''
   syntax = 'ip address duplicate detection ( logging | disabled )'
   noOrDefaultSyntax = 'ip address duplicate detection ...'
   data = { 'ip': ipMatcherForConfig,
            'address': 'Set IP address of an interface',
            'duplicate': 'Duplicate IP addresses',
            'detection': 'Detection of duplicate IP addresses',
            'logging': 'Enable duplicate address logging',
            'disabled': 'Disable duplicate address detection'
          }

   handler = "IraIpCliHandler.handlerIpDuplicateAddressDetectionCmd"
   noOrDefaultHandler = handler

configMode.addCommandClass( IpDuplicateAddressDetectionCmd )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   routing.plugin( entityManager )

   global routingHardwareStatusCommon
   global routingHardwareStatus
   global routingHardwareStatus6
   global routingHardwareRouteStatus
   global ipStatus
   global allVrfConfig
   global allVrfStatusLocal
   global l3Config
   global myEntManager
   global proxyArpVirtualMacStatus
   global ethIntfStatusDir
   global allVrfRouteSummary
   global arpCliStatus

   myEntManager = entityManager

   allVrfConfig = ConfigMount.mount( entityManager, "ip/vrf/config",
                                   "Ip::AllVrfConfig", "w" )

   routingHardwareStatusCommon = LazyMount.mount(
      entityManager,
      "routing/hardware/statuscommon",
      "Routing::Hardware::StatusCommon", "r" )
   routingHardwareStatus = LazyMount.mount( entityManager,
                                            "routing/hardware/status",
                                            "Routing::Hardware::Status", "r" )
   routingHardwareStatus6 = LazyMount.mount( entityManager,
                                            "routing6/hardware/status",
                                            "Routing6::Hardware::Status", "r" )
   routingHardwareRouteStatus = LazyMount.mount( entityManager,
                                            "routing/hardware/route/status",
                                            "Routing::Hardware::RouteStatus", "r" )

   ConfigMount.mount( entityManager, "routing/vrf/config",
                      "Routing::VrfConfigDir", "wi" )
   ipStatus = LazyMount.mount( entityManager, "ip/status", "Ip::Status", "r" )
   allVrfStatusLocal = LazyMount.mount( entityManager,
                                        Cell.path( "ip/vrf/status/local" ),
                                        "Ip::AllVrfStatusLocal", "r" )
   l3Config = LazyMount.mount( entityManager, "l3/config", "L3::Config", "r" )
   IraIpIntfCli.canSetIntfIpHook.addExtension( canSetIntfIp )
   proxyArpVirtualMacStatus = LazyMount.mount( entityManager,
                                               "routing/proxyArp/virtualMacStatus",
                                               "ProxyArp::VirtualMacStatus",
                                               "r" )

   ethIntfStatusDir = LazyMount.mount( entityManager,
                                       "interface/status/eth/intf",
                                       "Interface::EthIntfStatusDir",
                                       "r" )
   if toggleMixedRouteSummaryOutputEnabled():
      allVrfRouteSummary = SharkLazyMount.mount(
                              entityManager,
                              "routing/routeSummary/aggregated",
                              "Routing::RouteSummary::AllVrfRouteSummary",
                              SharkLazyMount.mountInfo( "shadow" ),
                              autoUnmount=True,
                              unmountTimeout=600 )
   arpCliStatus = LazyMount.mount( entityManager, "arp/clistatus",
                                   "Arp::CliStatus", "r" )
