# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Arnet
from ArnetModel import (
      Ip4Address,
      Ip6Address,
      IpGenericAddress,
      IpGenericPrefix,
)
from ArnetModel import MacAddress
from CliModel import Bool
from CliModel import DeferredModel
from CliModel import Dict
from CliModel import Enum
from CliModel import Float
from CliModel import Int
from CliModel import List
from CliModel import Model
from CliModel import Str
from CliModel import Submodel
from CliPlugin import IntfModel # pylint: disable-msg=F0401
from CliPlugin import IraRouteCommon
from CliPlugin.IraCommonModel import IpRouteSimpleVia, \
      ServiceRoutingProtoModelStatus
from CliPlugin.TunnelModels import (
   TunnelInfo,
   TunnelViaInfo,
   tunnelTypeEnumValues
)
from IntfModels import Interface
from IpLibConsts import DEFAULT_VRF
from TableOutput import createTable, Format
import Tac
import Toggles.RoutingLibToggleLib as RoutingLibToggle

IpUrpfFormatStr = '%-30s %-18s %-20s %-30s'

classESubnet = Tac.Value( "Arnet::IpAddrWithMask", address='240.0.0.0', len=4 )
thisNetworkSubnet = Tac.Value( "Arnet::IpAddrWithMask", address='0.0.0.0', len=8 )

class IpAddress( Model ):
   address = Ip4Address( help="IPv4 address of this interface" )
   maskLen = Int( help="IP address mask length (e.g 24 for a /24)" )

class InterfaceAddress( IntfModel.InterfaceAddressIp4Base ):
   primaryIp = Submodel( valueType=IpAddress, help="Primary IP address" )
   secondaryIps = Dict( valueType=IpAddress, help="Secondary IP addresses" )
   secondaryIpsOrderedList = List( valueType=IpAddress,
                                   help="Secondary IP addresses ordered list", 
                                   optional=True )
   virtualIp = Submodel( valueType=IpAddress, help="Virtual IP address" )
   virtualSecondaryIps = Dict( valueType=IpAddress, 
                               help="Virtual Secondary IP addresses" )
   virtualSecondaryIpsOrderedList = List( valueType=IpAddress,
                                          help=("Virtual Secondary IP addresses "
                                                "ordered list" ) )
   broadcastAddress = Str( help="IP used for broadcast"
                           " (typically '255.255.255.255')",
                           # We don't let user configure broadcast address
                           # yet, so it's hard-coded here:
                           default="255.255.255.255" )
   unnumberedIntf = Interface( help="Donor interface", optional=True )
   dhcp = Bool( help="Whether the address of this interface is determined via DHCP",
                default=False, optional=True )

   def printIntfAddrInfo( self ):
      print( "  Broadcast address is %(broadcastAddress)s" % self )
      if self.dhcp:
         print( "  Address determined by DHCP" )
         
   def render( self ):
      if self.primaryIp is not None and self.primaryIp.address:
         if not self.unnumberedIntf:
            print( "  Internet address is %(address)s/%(maskLen)d" % self.primaryIp )
         else:
            print( "  Internet address is unnumbered. %s/%d borrowed from %s" %
               ( self.primaryIp.address, self.primaryIp.maskLen,
                 self.unnumberedIntf.shortName ) )

         for ip in self.secondaryIpsOrderedList:
            print( "  Secondary address is %(address)s/%(maskLen)d" % ip )

         self.printIntfAddrInfo()

      elif self.virtualIp is not None and self.virtualIp.address:
         print( "  Internet address is virtual %(address)s/%(maskLen)d" %
             self.virtualIp )
         for ip in self.virtualSecondaryIpsOrderedList:
            print( "  Secondary address is virtual %(address)s/%(maskLen)d" % ip )

         self.printIntfAddrInfo()
         
      elif self.dhcp:
         print( "  Address being determined by DHCP" )
      
      else:
         print( "  No Internet protocol address assigned" )

class InterfaceAddressBrief ( IntfModel.InterfaceAddressIp4Base ):
   ipAddr = Submodel( valueType=IpAddress, help="Interface IP address" )
   unnumberedIntf = Interface( help="Donor interface", optional=True )

class IpStatusBase( Model ):
   name = IntfModel.InterfaceStatus.name
   interfaceStatus = IntfModel.InterfaceStatus.interfaceStatus
   lineProtocolStatus = IntfModel.InterfaceStatus.lineProtocolStatus
   mtu = IntfModel.InterfaceStatus.mtu
   interfaceAddressBrief = Submodel( valueType=InterfaceAddressBrief,
                                help="Network layer address for this interface",
                                optional=True )
   ipv4Routable240 = Bool( optional=True,
                           help="Whether or not IPv4 routable 240.0.0.0/4"
                                "is enabled" )
   ipv4Routable0 = Bool( optional=True,
                         help="Whether or not IPv4 routable 0.0.0.0/8"
                              "is enabled" )

   def render( self ):
      print( "%s is %s" % ( self.name.stringValue,
                           IntfModel.formatInterfaceStatus( self ) ) )
      addr = None
      if self.interfaceAddressBrief:
         addr = self.interfaceAddressBrief.ipAddr.address
      disabledRoutePrefix = ''
      if ( addr and classESubnet.contains( addr )
           and not self.ipv4Routable240 ):
         disabledRoutePrefix = '240.0.0.0/4'
      elif ( addr and thisNetworkSubnet.contains( addr )
             and not self.ipv4Routable0 ):
         disabledRoutePrefix = '0.0.0.0/8'

      if disabledRoutePrefix:
         print( "  ! Warning: IPv4 routable {} is disabled".format(
                disabledRoutePrefix ) )

class IpStatus( IpStatusBase ):
   enabled = Bool( help="Whether or not this interface is enabled" )
   description = Str( help="Description of this interface", optional=True )
   interfaceAddress = Submodel( valueType=InterfaceAddress,
                                help="Network Layer Addresses for this interface")
   proxyArp = Bool( help="Whether or not proxy-ARP is enabled" )
   proxyArpAllowDefault = Bool( help="Whether or not proxy-ARP always consider the"
                                     " default route if it is present in the routing"
                                     " table" )
   localProxyArp = Bool( help="Whether or not local proxy-ARP is enabled" )
   gratuitousArp = Bool( help="Whether or not gratuitous ARP is accepted" )
   arpAgingTimeout = Int( help="Configured ARP aging timeout" )
   proxyArpVirtualMacAddress = MacAddress(
      help="The virtual MAC address used in proxy-ARP or local proxy-ARP",
      optional=True )
   routedAddr = MacAddress(
      help="The routed MAC address used in proxy-ARP or local proxy-ARP",
      optional=True )
   isVrrpBackup = Bool( help="Whether if the VRRP instance is in VRRP backup mode",
                        optional=True )
   vrId = Int( help="The VRRP group of the virtual router seleted in proxy ARP",
               optional=True )
   vrf = Str( help="Name of the VRF", optional=True )
   urpf = Enum( values=( "disable", "loose", "strict", "strictDefault",
                         "looseNonDefault" ),
                help="Unicast Reverse Path Forwarding mode on the interface",
                optional=True )
   addresslessForwarding = Enum( values=( "isInvalid", "isFalse", "isTrue" ), 
                              help="V4 Forwarding mode on the V6 interface" )
   v4IntfForwarding = Bool( help="V4 Forwarding is enabled on the interface" )
   directedBroadcastEnabled = Bool(
      help="Whether or not Directed Broadcast is enabled for this intf" )
   maxMssIngress = Int( help="TCP MSS ceiling in bytes enforced on packets"
                             " arriving from the network", optional=True )
   maxMssEgress = Int( help="TCP MSS ceiling in bytes enforced on packets"
                            " forwarded to the network", optional=True )
   lossOfConnectivityReason = Enum( values=( 'cfm', ),
                                    help="Layer 3 interface connectivity status",
                                    optional=True )

   def render( self ):
      super().render()
      if self.description: 
         print( "  Description: %s" % self.description )
      if not self.enabled:
         print( "Internet protocol processing disabled" )
         return
      self.interfaceAddress.render()

      if self.v4IntfForwarding:
         print( "  IPv4 interface forwarding: enabled" )
      else:
         print( "  IPv4 interface forwarding: disabled" )

      if self.addresslessForwarding == "isFalse":
         print( "  IPv6 interface forwarding: disabled" )
      elif self.addresslessForwarding == "isTrue":
         print( "  IPv6 interface forwarding: enabled" )
      else:
         print( "  IPv6 interface forwarding: none" )

      if self.lossOfConnectivityReason:
         print( f"  {self.lossOfConnectivityReason.upper()} "
                "connectivity status is DOWN" )

      if self.urpf != 'disable':
         if self.urpf == 'strict':
            print( "  IP verify unicast source reachable-via RX" )
         elif self.urpf == 'strictDefault':
            print( "  IP verify unicast source reachable-via RX, allow-default" )
         elif self.urpf == 'loose':
            print( "  IP verify unicast source reachable-via ANY" )
         elif self.urpf == 'looseNonDefault':
            print( "  IP verify unicast source reachable-via ANY, non-default" )
      if self.directedBroadcastEnabled:
         print( "  IP Directed Broadcast enabled" )
      self.showIpIntfConfOptions()
      print( "  IP MTU %(mtu)d bytes" % self )

      if self.maxMssIngress and self.maxMssEgress:
         print( "  IPv4 TCP MSS ceiling is %d bytes" % self.maxMssIngress )
      elif self.maxMssIngress:
         print( "  IPv4 TCP MSS ingress ceiling is %d bytes" % self.maxMssIngress )
      elif self.maxMssEgress:
         print( "  IPv4 TCP MSS egress ceiling is %d bytes" % self.maxMssEgress )
      self.showVrfConfOptions()


   def showIpIntfConfOptions( self ):
      if self.interfaceAddress.primaryIp.address or \
         self.interfaceAddress.virtualIp.address:
         if self.proxyArp:
            proxyArpStr = ( "enabled and allows the default route"
                            if self.proxyArpAllowDefault else
                            "enabled" )
            # vrId 256 means VARP
            if self.vrId and self.vrId != 256:
               proxyArpStr += " (VRRP instance %s)" % self.vrId
            if self.isVrrpBackup:
               proxyArpStr = ( "disabled (VRRP instance %s in backup mode)" %
                               self.vrId )
         else:
            proxyArpStr = "disabled"
         if self.localProxyArp:
            localProxyArpStr = ( "enabled" )
            # vrId 256 means VARP
            if self.vrId and self.vrId != 256:
               localProxyArpStr += " (VRRP instance %s)" % self.vrId
            if self.isVrrpBackup:
               localProxyArpStr = ( "disabled (VRRP instance %s in backup mode)" %
                                    self.vrId )
         else:
            localProxyArpStr = "disabled"
         print( "  Proxy-ARP is %s\n"
                "  Local Proxy-ARP is %s"
                % ( proxyArpStr, localProxyArpStr ) )
         if self.proxyArp or self.localProxyArp:
            if mac := self.proxyArpVirtualMacAddress:
               print( "  Proxy-ARP reply MAC address is %s" % mac.stringValue )
            elif self.routedAddr:
               print( "  Proxy-ARP reply MAC address is %s" % self.routedAddr )
         print( "  Gratuitous ARP is %s"
                 % ( "accepted" if self.gratuitousArp else "ignored" ) )
         print( "  ARP aging timeout is %d seconds" % self.arpAgingTimeout )

   def showVrfConfOptions( self ):
      assert self.vrf and self.vrf != ''
      if self.vrf != DEFAULT_VRF:
         print( '  VPN Routing/Forwarding "%(vrf)s"' % self )

class IpStatuses( Model ):
   __revision__ = 2
   interfaces = Dict( keyType=Interface, valueType=IpStatus,
                      help="Maps an interface name to its IP status" )

   def render( self ):
      for key in Arnet.sortIntf( self.interfaces ):
         self.interfaces[ key ].render()

   def degrade( self, dictRepr, revision ):
      # The new revision is introduced because the "injectHosts" bool attribute was
      # removed. So we are putting a default (False) value for this attribute.
      if revision == 1:
         for interface in dictRepr[ 'interfaces' ]:
            dictRepr[ 'interfaces' ][ interface ][ 'injectHosts' ] = False
      return dictRepr

class IpStatusBrief ( IpStatusBase ):
   interfaceAddress = Submodel( valueType=InterfaceAddressBrief,
                                help="Network layer address for this interace" )
   nonRoutableClassEIntf = Bool( optional=True,
                                 help="True if class E address configured and class"
                                      " E address routing is disabled" )

   def addRow( self, tableOutput ):
      addrStr = 'unassigned'
      addr = self.interfaceAddress.ipAddr
      if addr.address or addr.maskLen > 0:
         addrStr = '%s/%d' % ( addr.address, addr.maskLen )
      if addr.address:
         # Check if interface address is reserved or belongs to "this network" and
         # the corresponding routing is enabled ( 'ipv4 routable 240.0.0.0/4' or
         # 'ipv4 routable 0.0.0.0/8' )
         if ( ( classESubnet.contains( addr.address )
                and not self.ipv4Routable240 ) or
              ( thisNetworkSubnet.contains( addr.address ) and not
                self.ipv4Routable0 ) ):
            # Add warning to address string
            addrStr = addrStr + ' !'
      unnumberedIntfStr = ''
      if self.interfaceAddress.unnumberedIntf:
         unnumberedIntfStr = self.interfaceAddress.unnumberedIntf.shortName

      tableOutput.newRow( self.name.stringValue, addrStr,
                          IntfModel.formatShortIntfStatus( self ),
                          self.lineProtocolStatus.lower(),
                          self.mtu, unnumberedIntfStr )

class IpStatusesBrief ( Model ):
   interfaces = Dict( keyType=Interface, valueType=IpStatusBrief,
                      help="Maps an interface name to an abbreviated IP status" )

   def render( self ):
      flh = Format( isHeading=True, justify='left' )
      flh.noPadLeftIs( True )
      fl = Format( justify='left' )
      fl.noPadLeftIs( True )
      fr = Format( justify='right' )

      intfTable = createTable( ( "\nInterface", "\nIP Address",
                                 "\nStatus", "\nProtocol", "\nMTU",
                                 "Address\nOwner" ) )
      intfTable.formatColumns( flh, fl, fl, fl, fr, fl )

      printClassEWarning = False
      printThisNetworkWarning = False
      for key in Arnet.sortIntf( self.interfaces ):
         self.interfaces[ key ].addRow( intfTable )
         # Decide whether we need to print the warning
         if not printClassEWarning and self.interfaces[ key ].nonRoutableClassEIntf:
            printClassEWarning = True
         addr = None
         if self.interfaces[ key ].interfaceAddress:
            addr = self.interfaces[ key ].interfaceAddress.ipAddr.address
         if ( not printThisNetworkWarning and
              not self.interfaces[ key ].ipv4Routable0 and
              addr and thisNetworkSubnet.contains( addr ) ):
            printThisNetworkWarning = True

      print( intfTable.output() )
      # If one interface is a reserved class E address and
      # 'ipv4 routable 240.0.0.0/4' is disabled we print the warning
      if printClassEWarning:
         print( "! Warning: IPv4 routable 240.0.0.0/4 is disabled" )
      # If one interface is a reserved this network address and
      # 'ipv4 routable 0.0.0.0/8' is disabled we print the warning
      if printThisNetworkWarning:
         print( "! Warning: IPv4 routable 0.0.0.0/8 is disabled" )

#-------------------------
# show ip command Models
#------------------------
def enabledDisabled( attr ):
   return 'Enabled' if attr else 'Disabled'

class IpMulticastRouting( Model ):
   ipMulticastEnabled = Bool( help='IPv4 multicast routing is enabled',
                              optional=True )
   ip6MulticastEnabled = Bool( help='IPv6 multicast routing is enabled',
                               optional=True )

   def render( self ):
      if self.ipMulticastEnabled is not None:
         print( "IP Multicast Routing : {}".format(
                enabledDisabled( self.ipMulticastEnabled ) ) )
      if self.ip6MulticastEnabled is not None:
         print( "IPv6 Multicast Routing : {}".format(
                enabledDisabled( self.ip6MulticastEnabled ) ) )

class V6EcmpModel( Model ):
   v6EcmpRouteSupport = Bool( help="IPv6 ECMP route is supported" )
   nextHopIndex = Int( help="IPv6 ECMP route nexthop index", optional=True )
   numPrefixBits = Int( help="IPv6 ECMP route num prefix bits for nexthop index",
                        optional=True )

# This mapping is required to adhere to capi lowerCamelCase enum naming conventions,
# while preserving existing cli show command output.
uRpfStates = {
         'enabled': 'Enabled',
         'disabled': 'Disabled',
         'errorDisabled': 'Error Disabled'
         }

class IpModel( Model ):
   v4RoutingEnabled = Bool( help="IPv4 Routing is enabled" )
   v6RoutingEnabled = Bool( help="IPv6 Routing is enabled" )
   vrrpIntfs = Int( help="Number of interfaces with VRRP configured",
                    optional=True )
   v6IntfForwarding = Bool( help="IPv6 interface forwarding is enabled",
                            optional=True )
   v4uRpfState = Enum( values=uRpfStates,
                       optional=True,
                       help="State of IPv4 uRPF" )
   v6uRpfState = Enum( values=uRpfStates,
                       help="State of IPv6 uRPF",
                       optional=True )
   multicastRouting = Submodel( valueType=IpMulticastRouting,
                                help="IPv4 and IPv6 multicast routing information",
                                optional=True )
   v6EcmpInfo = Submodel( valueType=V6EcmpModel,
                          help="IPv6 ECMP Routing Information",
                          optional=True )

   def render( self ):
      print()
      print( f"IP Routing : {enabledDisabled( self.v4RoutingEnabled )}" )
      if self.multicastRouting:
         self.multicastRouting.render()
      if self.vrrpIntfs:
         if self.vrrpIntfs == 1:
            suffix = ''
         else:
            suffix = 's'
         print( "VRRP: Configured on "
                "{} interface{}".format( self.vrrpIntfs, suffix ) )
      print( f"IPv6 Interfaces Forwarding : {self.v6IntfForwarding}" )
      if self.v4uRpfState:
         print( f"IP uRPF : {uRpfStates[ self.v4uRpfState ]}" )
      print()
      print( "IPv6 Unicast "
             "Routing : {}".format( enabledDisabled( self.v6RoutingEnabled ) ) )
      if self.v6EcmpInfo:
         ecmpInfo = self.v6EcmpInfo
         # If ECMP routing is not supported, print the nexthop information
         # workaround due to EOS not realising IPv6 ECMP is not supported
         # in hardware. - AID1250
         if not ecmpInfo.v6EcmpRouteSupport:
            print( "IPv6 ECMP Route support : False" )
            print( "IPv6 ECMP Route "
                   "nexthop index: {}".format( ecmpInfo.nextHopIndex ) )
            print( "IPv6 ECMP Route num prefix bits for "
                   "nexthop index: {}".format( ecmpInfo.numPrefixBits ) )
      if self.v6uRpfState:
         print( f"IPv6 uRPF : {uRpfStates[ self.v6uRpfState ]}" )

#-------------------------
#show ip route command Models
#----------------------------
class IpRouteNexthopGroup( Model ):
   groupName = Str( help="Nexthop group name" )
   groupId = Int( help="Nexthop group Id" )

class RouteSrTePolicy( Model ):
   endpoint = IpGenericAddress( help="SR-TE policy destination" )
   color = Int( help="SR-TE policy color" )

class CbfColoredTep( Model ):
   endpoint = IpGenericPrefix( help="Tunnel endpoint" )
   color = Int( help="Color assigned to Tunnel" )

class VrfSelectionInfo( Model ):
   policyName = Str( help="VRF Selection Policy Name" )
   vrfName = Str( help="VRF name to which the route is leaked" )

# Class based forwarding via
class CbfVia( Model ):
   # choosing coloredEndpoints a list so that when later requirement comes to have
   # CBF with multiple bgp nexthop with same color then degradation of capi model
   # will not be needed
   coloredEndpoints = List( valueType=CbfColoredTep,
                            help="CBF Endpoint and Color Info", optional=True )
   srTePolicy = Submodel( valueType=RouteSrTePolicy, help="SR-TE policy",
                          optional=True )
   vrfSelectionInfo = Submodel( valueType=VrfSelectionInfo,
                                help="VRF Selection Info",
                                optional=True )
   tunnelSource = Enum( values=tunnelTypeEnumValues, help="Tunnel Type",
                        optional=True )
   dscpValues = List( valueType=int, help="List of DSCP values", optional=True )
   tcValues = List( valueType=int, help="List of traffic class values",
                    optional=True )
   mplsLabel = Int( help="MPLS label (service label); may also be included in "
                         "the labelStack", optional=True )
   labelStack = List( valueType=int, optional=True,
                      help="MPLS label stack (top-of-stack label first)" )

class IpRouteVia( IpRouteSimpleVia ):
   interfaceDescription = Str( help="Interface description", optional=True )
   viaType = Enum( values= ( 'active', 'backup' ),
                   help="Type of Via. PIC has backup Vias", optional=True )
   # A via (optionally) contains a labelStack.  The Fib::Via internal representation
   # has a label in the via and a label stack in the FEC, but together they just
   # define an effective label stack, so that is what is generated here.
   # The mplsLabel may be included in the labelStack as the bottom-most label.
   mplsLabel = Int( help="MPLS label (service label); may also be included in "
                         "the labelStack", optional=True )
   srv6Sid = Ip6Address( help='SRv6 SID (service SID)', optional=True )
   labelStack = List( valueType=int, optional=True,
                      help="MPLS label stack (top-of-stack label first)" )
   viaWeight = Str( help="UCMP load-balance weight", optional=True )
   ucmpGenerationMode = Enum( values=( 'linkBandwidth', ),
                              help="UCMP mode", optional=True )
   vni = Int( help="VXLAN Virtual Network Identifier", optional=True )
   vtepAddr = IpGenericAddress( help="VXLAN remote VTEP IP address", optional=True )
   routerMac = MacAddress( help="VXLAN remote VTEP mac address", optional=True )
   localInterface = Interface( help="Name of the source interface", optional=True )
   # Only present when via is a tunnel.
   vias = List( valueType=TunnelViaInfo, help="List of nested tunnel vias",
                optional=True )
   tunnelDescriptor = Submodel( valueType=TunnelInfo,
                                help="Tunnel information", optional=True )
   nexthopGroup = Submodel( valueType=IpRouteNexthopGroup, help="Nexthop group",
                            optional=True )
   # Only present when via is a tunnel.
   tunViaBackupVias = List( valueType=TunnelViaInfo,
                            help="List of nested tunnel backup vias", optional=True )
   fallbackVrf = Str( help="Fallback VRF", optional=True )
   decapActionType = Enum( values=( 'none', 'ipip' ),
                     help="Decapsulation type", optional=True )

IpRouteVia.__attributes__[ "resolvedVias" ] \
      = List( valueType=IpRouteVia, help="List of resolution vias",
              optional=True )

class IpRouteBase( Model ):
   # these fields are shared by IPv4 and IPv6 routes
   routeAction = Enum( values=( 'forward', 'drop' ), 
                        help="What kind of route it is" )
   hardwareProgrammed = Bool( help="True if this route is programmed in hardware" )
   kernelProgrammed = Bool( help="True if this route is programmed in the kernel" )
   preference = Int( help="Administrative distance; lower values have precedence",
                     optional=True )
   metric = Int( help="Used by routing protocol to determine whether one route "
                  "should be chosen over another.  Lower metrics have precedence",
                  optional=True )
   srTePolicy = List( valueType=RouteSrTePolicy,
         help="SR-TE policy", optional=True )
   directlyConnected = Bool( help="True if this route is directly connected though "
         "an interface, internal, or is a drop route" )
   routeLeaked = Bool( help="True if this route is leaked from different VRF" )
   srcVrf = Str( help="Name of the VRF route is leaked from", optional=True )
   cbfVias = List( valueType=CbfVia, help="List of Class based forwarding vias",
                   optional=True )
   routePriority = Enum( values=( 'low', 'medium', 'high' ),
                         help="Route processing priority",
                         optional=True )
   loopingRoute = Bool( help="True if this route is part of a recursive route"
         " resolution loop", optional=True )

class IpRoute( IpRouteBase ):
   routeType = Enum( values=( 'connected', 'static', 'vcs', 'kernel', 'OSPF',
      'ospfInterArea', 'ospfExternalType1', 'ospfExternalType2',
      'ospfNssaExternalType1', 'ospfNssaExternalType2', 'iBGP', 'eBGP',
      'BGP', 'RIP', 'ISISLevel1', 'ISISLevel2', 'bgpAggregate', 'ospfSummary',
      'nexthopGroupStaticRoute', 'OSPFv3', 'dropRoute', 'martian',
      'dynamicPolicy', 'routeCacheConnected', 'gribi', 'CBFLeaked',
      'dhcp' ), help="The IPv4 route type" )
   # routeSubtype attribute will be used to show ospfv3 subtypes.
   routeSubtype = Enum( values=( 'IntraArea', 'InterArea', 'External1',
      'External2', 'NssaExternal1', 'NssaExternal2' ),
      help='OSPFv3 route type', optional=True )
   vias = List( valueType=IpRouteVia, help="List of vias" )

class IpRoutes( Model ):
   allRoutesProgrammedHardware = Bool( help="True if every route is programmed "
                                       "in hardware" )
   allRoutesProgrammedKernel = Bool( help="True if every route is programmed "
                                     "in kernel" )
   routingDisabled = Bool( help="True if routing is disabled" )
   defaultRouteState = Enum( values=( 'reachable', 'notReachable', 'notSet' ),
                                      help="The state of the default route" )
   routes = Dict( keyType=str, valueType=IpRoute, help="List of routes keyed by "
                  "prefix" )

class VrfIpRoutes ( DeferredModel ):
   '''
   __revision__ 3 prints the nested tunnel representation for routes resolving over
   tunnels. If a lower revision is requested, it will flatten the tunnel entries,
   and print the final nexthop, interface and combined label stack
   '''
   __revision__ = 4
   vrfs = Dict ( keyType=str, valueType = IpRoutes, 
                 help="Route info of each Vrf" )

class VrfLdpIpRoutes( VrfIpRoutes ):
   '''
   Model for "show rib system-tunneling-ldp-rib route ipv4" to print LDP RouteStatus
   Same model as show ip route, but we do not need to support the old revisions here
   '''
   __revision__ = 2

#----------------------------------------------------------
#model for 'show ip|ipv6 user route'
#Based on the IpRoutes model (FIB), but adapted for the RIB
#----------------------------------------------------------
class IpRibRouteVia( IpRouteVia ):
   nexthopName = Str( help='Nexthop name', optional=True )

class IpRibRouteBase( Model ):
   preference = IpRouteBase.preference
   nexthopGroupName = Str( help='Nexthop group name', optional=True )
   nexthopGroupWeight = Int( help='UCMP weight for the nexthop group',
                             optional=True )
   dropNextHopName = Str( help='Nexthop name for drop route', optional=True )
   commandTag = Str( help='Associated command-tag name', optional=True )
   tag = Int( help='Route tag number', optional=True )
   vias = List( valueType=IpRibRouteVia, help="List of vias" )

class IpRibRoute( IpRibRouteBase ):
   routeType = Enum( values=[ 'forward', 'drop', 'nexthopGroup' ],
                     help='RIB route type' )

class IpRibRoutePreference( Model ):
   preference = Dict( keyType=int, valueType=IpRibRoute,
                      help='A mapping from preference to a RIB route' )

class IpRibRoutes( Model ):
   routes = Dict( keyType=str, valueType=IpRibRoutePreference,
                  help='A mapping from prefix to a collection of RIB routes' )

class VrfIpRibRoutes( DeferredModel ):
   vrfs = Dict( keyType=str, valueType=IpRibRoutes,
                help='A mapping from VRF to the RIB routes it contains' )

#---------------------------------------
#model for 'show ip route host'
#---------------------------------------
class IpRouteHostVia( Model ):
   toCpu = Bool( help="True if route goes to cpu" )
   nexthopAddr = IpGenericAddress( help="Nexthop address of the via", optional=True )
   interface = Interface( help="Name of the interface", optional=True )
   vni = Int( help="VXLAN Virtual Network Identifier", optional=True )
   vtepAddr = IpGenericAddress( help="VXLAN remote VTEP IP address", optional=True )
   routerMac = MacAddress( help="VXLAN remote VTEP mac address", optional=True )
    
class IpRouteHost( Model ):
   routeType = Enum( values=( 'receive', 'broadcast', 'directedBcast', 'FIB',
                              'attached' ), help="The ip route host type" )
   hardwareProgrammed = Bool( help="True if this route is "
         "programmed in hardware" )
   vias = List( valueType=IpRouteHostVia, help="List of host vias" )

class IpRoutesHost( DeferredModel):
   routingDisabled = Bool( help="Routing is disabled", optional=True )
   allRoutesProgrammedHardware = Bool( help="Every route is programmed "
                                       "in hardware" )
   routes = Dict( keyType=str, valueType=IpRouteHost, help="IP host routes "
                  "keyed by prefix" )

class VrfIpRoutesHost ( DeferredModel ):
   '''
   __revision__ 2 supports printing all vrfs host routes.
   __revision__ 1 does not support printing all vrfs host routes.
   '''
   __revision__ = 2
   vrfs = Dict( keyType=str, valueType=IpRoutesHost,
                 help="A mapping of VRF to its host routes" )
#-------------------------------------
#model for 'show ip route summary'
#  'show ip route vrf all summary brief'
#-------------------------------------
class ospfTotals( Model ):
   ospfTotal = Int( help="Total number of Ospf routes" ) 
   ospfIntraArea = Int( help="Number of Ospf Intra-area routes" )
   ospfInterArea = Int( help="Number of Ospf Inter-area routes" )
   ospfExternal1 = Int( help="Number of Ospf External-1 routes" )
   ospfExternal2 = Int( help="Number of Ospf External-2 routes" )
   nssaExternal1 = Int( help="Number of Ospf NSSA External-1 routes" )
   nssaExternal2 = Int( help="Number of Ospf NSSA External-2 routes" )

class ospfv3Totals( Model ):
   ospfv3Total = Int( help="Total number of Ospfv3 routes" ) 
   ospfv3IntraArea = Int( help="Number of Ospfv3 Intra-area routes" )
   ospfv3InterArea = Int( help="Number of Ospfv3 Inter-area routes" )
   ospfv3External1 = Int( help="Number of Ospfv3 External-1 routes" )
   ospfv3External2 = Int( help="Number of Ospfv3 External-2 routes" )
   ospfv3NssaExternal1 = Int( help="Number of Ospfv3 NSSA External-1 routes" )
   ospfv3NssaExternal2 = Int( help="Number of Ospfv3 NSSA External-2 routes" )

class bgpTotals( Model ):
   bgpTotal = Int( help="Total number of bgp routes" )
   bgpExternal = Int( help="Number of bgp External routes" )
   bgpInternal = Int( help="Number of bgp Internal routes" )
   bgpLocal = Int( help="Number of bgp Other routes" )

class isisTotals( Model ):
   isisTotal = Int( help="Total number of isis routes" )
   isisLevel1 = Int( help="Number of isis Level-1 routes" )
   isisLevel2 = Int( help="Number of isis Level-2 routes" )

class IpRouteSummaryForVrf( Model ):
   connected = Int( help="Number of connected routes" )
   static = Int( help="Number of static routes" )
   staticPersistent = Int( help="Number of static routes configured to persist" )
   staticNonPersistent = Int( help="Number of static routes configured using the "
                                   "SDK with persistent=False" )
   vrfCount = Int( help="Total number of vrf instances", optional=True )
   vcs = Int( optional=True, help="Number of VXLAN Control Service routes" )
   staticNexthopGroup = Int( help="Number of static nexthop-group routes" )
   dynamicPolicy = Int( help="Number of dynamic policy routes" )
   ospfCounts = Submodel( valueType=ospfTotals, help="OSPF route count breakdown" )
   ospfv3Counts = Submodel( valueType=ospfv3Totals,
                            help="OSPFv3 route count breakdown" )
   bgpCounts = Submodel( valueType=bgpTotals, help="BGP route count breakdown" )
   isisCounts = Submodel( valueType=isisTotals, help="IS-IS route count breakdown" )
   gribi = Int( help="Number of gRIBI routes" )
   rip = Int( help="Number of rip routes" ) 
   internal = Int( help="Number of internal routes" ) 
   attached = Int( help="Number of attached routes" ) 
   aggregate = Int( help="Number of aggregate routes" ) 
   maskLen = Dict( keyType=int, valueType=int, help="Counts for each mask length" )
   totalRoutes = Int( help="Total number of routes" )
   _quantify = Float( optional=True, help="Time (in seconds) taken to populate "
                     "and print the summary" )

   def render( self ):
      beforeRender = Tac.now()
      table = createTable( ( 'Route Source', 'Number Of Routes' ) )
      f1 = Format( justify='left' )
      f2 = Format( justify='right' )
      table.formatColumns( f1, f2 )
      table.newRow( "connected", self.connected )
      table.newRow( "static (persistent)", self.staticPersistent )
      table.newRow( "static (non-persistent)", self.staticNonPersistent )
      table.newRow( "VXLAN Control Service", self.vcs )
      table.newRow( "static nexthop-group", self.staticNexthopGroup )
      table.newRow( "ospf", self.ospfCounts.ospfTotal )
      table.startRow() # format ospf 
      table.newFormattedCell( \
            '  Intra-area: %d Inter-area: %d External-1: %d External-2: %d' \
            % ( self.ospfCounts.ospfIntraArea, \
            self.ospfCounts.ospfInterArea, \
            self.ospfCounts.ospfExternal1, \
            self.ospfCounts.ospfExternal2 ), 2, f1 )
      table.startRow()
      table.newFormattedCell( '  NSSA External-1: %d NSSA External-2: %d' \
            % ( self.ospfCounts.nssaExternal1, \
            self.ospfCounts.nssaExternal2 ), 2, f1 )
      table.newRow( "ospfv3", self.ospfv3Counts.ospfv3Total )
      if RoutingLibToggle.toggleOspf3RouteTypesDisplayEnabled():
         table.startRow() # format ospfv3
         table.newFormattedCell(
               '  Intra-area: %d Inter-area: %d External-1: %d External-2: %d'
               % ( self.ospfv3Counts.ospfv3IntraArea,
               self.ospfv3Counts.ospfv3InterArea,
               self.ospfv3Counts.ospfv3External1,
               self.ospfv3Counts.ospfv3External2 ), 2, f1 )
         table.startRow()
         table.newFormattedCell( '  NSSA External-1: %d NSSA External-2: %d'
            % ( self.ospfv3Counts.ospfv3NssaExternal1,
            self.ospfv3Counts.ospfv3NssaExternal2 ), 2, f1 )
      table.newRow( "bgp", self.bgpCounts.bgpTotal )     
      table.startRow() # format bgp
      table.newFormattedCell( '  External: %d Internal: %d' \
            % ( self.bgpCounts.bgpExternal, \
            self.bgpCounts.bgpInternal ), 2, f1 )

      table.newRow( "isis", self.isisCounts.isisTotal )
      table.startRow() # format isis
      table.newFormattedCell( '  Level-1: %d Level-2: %d' \
            % ( self.isisCounts.isisLevel1, \
            self.isisCounts.isisLevel2 ), 2, f1 )

      table.newRow( "rip", self.rip )
      table.newRow( "internal", self.internal )
      table.newRow( "attached", self.attached )
      table.newRow( "aggregate", self.aggregate )
      table.newRow( "dynamic policy", self.dynamicPolicy )
      table.newRow( "gribi", self.gribi )
      table.newRow( '', '' )
      table.newRow( 'Total Routes', self.totalRoutes )
      print( table.output() )
      IraRouteCommon.showRouteSummaryMaskLengths( sorted( self.maskLen.items() ) ) 
      if self.vrfCount is not None:
         print( "Number of VRFs:", self.vrfCount )
      if self._quantify:
         # we need to add the time spent in render with the time spent in populating
         # the model
         afterRender = Tac.now()
         print( "Time taken for the summary to be printed: %f seconds" % 
               ( ( afterRender-beforeRender)+self._quantify ) )

class IpRouteSummary( Model ):
   __revision__ = 3
   
   protoModelStatus = Submodel( valueType=ServiceRoutingProtoModelStatus,
                                help="Active and configured protocol models" )
   vrfs = Dict( keyType=str, valueType=IpRouteSummaryForVrf,
                help="Route summary of each VRF" )

   def render( self ):
      self.protoModelStatus.render()
      for vrf, vrfSummary in self.vrfs.items():
         print( "\nVRF: %s" % vrf )
         vrfSummary.render()

   def degrade( self, dictRepr, revision ):
      '''
      Revision 2:
      Two new attributes added: 
       'staticNonPersistent'
       'staticPersistent'
      Revision 3:
         instead of returning {rev-2-output }, return
             { "vrfs": { "vrf1": { rev-2-output } } }
         which can now support "show ip route vrf all summary"
      Since the 'all' token was supported only form revision 3, the 
       'show ip route vrf all summary' | json for revisions 1 and 2 will 
       return the dict representation of the default Vrf
      '''
      vrfCount = len( dictRepr[ 'vrfs' ] )
      if vrfCount > 1:
         # The output for 'vrf all', this will have multiple vrfs, the degrade
         # function will return the default vrf as this is always present in 
         # the output dict
         vrfName = DEFAULT_VRF
      else:
         # The 'show ip route vrf <vrfName> summary' command will always return
         # a single vrf 
         vrfName = list( dictRepr[ 'vrfs' ] )[ 0 ]
      if revision == 1:
         del dictRepr[ 'vrfs' ][ vrfName ][ 'staticNonPersistent' ]
         del dictRepr[ 'vrfs' ][ vrfName ][ 'staticPersistent' ]
         dictRepr[ 'vrfs' ][ vrfName ][ 'isis' ] = \
               dictRepr[ 'vrfs' ][ vrfName ][ 'isisCounts' ][ 'isisTotal' ]
         # Leave isisCounts in case somebody has been using it with rev=1
         #del dictRepr[ 'vrfs' ][ vrfName ][ 'isisCounts' ]
      return dictRepr[ 'vrfs' ][ vrfName ]

#-------------------------------------------------------
# show ip route rpf unicast [ VRF ] [ PREFIX | ADDR ]
#-------------------------------------------------------

class UrpfInterface( Model ):
   uRpfMode = Enum( values=( "strict", "loose", "strictDefault", "disable" ),
                    help="URPF mode state" )
   allowDefault = Bool( help=( "Packets are permitted to"
                               " come in on the default interface" ) )
   lookupVrf = Str( help="Name of uRPF lookup VRF", optional=True )

class UrpfInterfaces( Model ):
   interfaces = Dict( keyType=Interface, valueType=UrpfInterface,
                      help=( "Mapping of interface to the additional"
                             " interface state details" ),
                      optional=True )
   prefix = IpGenericPrefix( help="Destination Prefix",
                             optional=True )
   viaDefaultRoute = Bool( help="uRPF is carried out via the default route",
                           optional=True )

   def formatUrpfMode( self, mode ):
      if mode == 'strict':
         return 'Strict'
      elif mode == 'loose':
         return 'Loose'
      elif mode == 'strictDefault':
         return 'Strict'
      else:
         return 'Disable'

   def render( self ):
      if self.interfaces:
         if self.prefix:
            defaultRouteStr = 'via default route' if self.viaDefaultRoute else ""
            print( '\nRPF interfaces for %s %s\n' % ( self.prefix,
                                                      defaultRouteStr ) )
         else:
            print( '\nuRPF configured on interfaces:\n' )

         print( IpUrpfFormatStr % ( 'Interface',
                                    'uRPF mode',
                                    'Allow Default',
                                    'Lookup VRF' ) )
         for intfId in self.interfaces:
            mode = self.interfaces[ intfId ].uRpfMode
            allowDefault = 'Yes' if self.interfaces[ intfId ].allowDefault else "No"
            lookupVrf = self.interfaces[ intfId ].lookupVrf \
               if self.interfaces[ intfId ].lookupVrf else "-"
            print( IpUrpfFormatStr %
                  ( intfId,
                    self.formatUrpfMode( mode ),
                    allowDefault,
                    lookupVrf ) )
