# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliMatcher
import CliParser
import ConfigMount
from CliPlugin import IraIp6IntfCli
from CliPlugin import IraIpIntfCli
from CliPlugin import IraRouteCommon
from CliPlugin.EthIntfCli import isVEos
from CliPlugin.TcpMssCli import CfgTcpMssCeilingCmdBase
import LazyMount
import Tac

ipStatusDir = None
ipConfigDir = None
dynIpConfigDir = None
ip6ConfigDir = None
ip6StatusDir = None
routingHwStatusCommon = None
routingHwStatus = None
routing6HwStatus = None
entityMib = None

ip = IraRouteCommon.Ip4()
routing = IraRouteCommon.routing( ip )
tunModeEnum = Tac.Type( 'Arnet::TunnelIntfMode' )

def getHardwareSupportedModes():
   encapModes = []
   if routingHwStatus and list( routingHwStatus.staticInterfaceTunnelSupport ):
      encapModes = list( routingHwStatus.staticInterfaceTunnelSupport )
   if routing6HwStatus and list( routing6HwStatus.staticInterfaceTunnelSupport ):
      encapModes += list( routing6HwStatus.staticInterfaceTunnelSupport )
   return encapModes

def staticInterfaceTunnelSupported():
   encapModes = getHardwareSupportedModes()
   return encapModes and ( tunModeEnum.tunnelIntfModeGre in encapModes or
         tunModeEnum.tunnelIntfModeIpsec in encapModes )

def tcpMssPlatformGuard( mode, token ):
   if isVEos() or staticInterfaceTunnelSupported():
      return None
   return CliParser.guardNotThisPlatform

#-------------------------------------------------------------------------------
# 'tcp mss ceiling ([ipv4 <64-65515>] [ipv6 <64-65495>]) [ingress | egress ]' .
#-------------------------------------------------------------------------------
class CfgTcpMssCeilingCmd( CfgTcpMssCeilingCmdBase ):
   data = {
      'tcp': CliCommand.Node(
                matcher=CliMatcher.KeywordMatcher( 'tcp', helpdesc='TCP' ),
                guard=tcpMssPlatformGuard ),
   }
   data.update( CfgTcpMssCeilingCmdBase._baseData )

   @staticmethod
   def _getConfig( mode ):
      ipv4Config = ipConfigDir.ipIntfConfig.get( mode.intf.name )
      ipv6Config = ip6ConfigDir.intf.get( mode.intf.name )
      if ipv4Config and ipv6Config:
         return ( ipv4Config.tcpMssConfig, ipv6Config.tcpMssConfig )

      # Create the corresponding ip4 and ip6 config if missing
      if ipv4Config is None:
         IraIpIntfCli.IpIntf( mode.intf, mode.sysdbRoot,
                              createIfMissing=True ).config()
      if ipv6Config is None:
         IraIp6IntfCli.Ip6Intf( mode.intf, mode.sysdbRoot,
                                createIfMissing=True ).config()

      ipv4Config = ipConfigDir.ipIntfConfig.get( mode.intf.name )
      ipv6Config = ip6ConfigDir.intf.get( mode.intf.name )
      return ( ipv4Config.tcpMssConfig, ipv6Config.tcpMssConfig )

   @staticmethod
   def handler( mode, args ):
      ipv4Config, ipv6Config = CfgTcpMssCeilingCmd._getConfig( mode )
      CfgTcpMssCeilingCmdBase._commonHandler( mode, args, ipv4Config, ipv6Config )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ipv4Config, ipv6Config = CfgTcpMssCeilingCmd._getConfig( mode )
      CfgTcpMssCeilingCmdBase._commonNoOrDefaultHandler( mode, ipv4Config,
                                                         ipv6Config )

IraIpIntfCli.RoutingProtocolIntfConfigModelet.addCommandClass( CfgTcpMssCeilingCmd )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   routing.plugin( entityManager )
   global ipStatusDir, ip6StatusDir, ipConfigDir, ip6ConfigDir
   global routingHwStatusCommon
   global routingHwStatus, routing6HwStatus, entityMib
   ipStatusDir = LazyMount.mount( entityManager, "ip/status", "Ip::Status", "r" )
   ip6StatusDir = LazyMount.mount( entityManager, "ip6/status", "Ip6::Status", "r" )
   ipConfigDir = ConfigMount.mount( entityManager, "ip/config", "Ip::Config", "w" )
   ip6ConfigDir = ConfigMount.mount(
                     entityManager, "ip6/config", "Ip6::Config", "w" )
   routingHwStatusCommon = LazyMount.mount( entityManager,
                                            'routing/hardware/statuscommon',
                                            'Routing::Hardware::StatusCommon', 'r' )
   routingHwStatus = LazyMount.mount( entityManager, 'routing/hardware/status',
                                      'Routing::Hardware::Status', 'r' )
   routing6HwStatus = LazyMount.mount( entityManager, 'routing6/hardware/status',
                                        'Routing6::Hardware::Status', 'r' )
   entityMib = LazyMount.mount( entityManager, 'hardware/entmib',
                                'EntityMib::Status', 'r' )
