#!/usr/bin/env python3
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=possibly-unused-variable
# pylint: disable=bad-string-format-type
# pylint: disable=consider-using-f-string
# pylint: disable=import-outside-toplevel

from datetime import datetime
from collections import OrderedDict
from munch import Munch
from operator import itemgetter, attrgetter
from time import strftime, localtime
import socket
import struct
import textwrap

from Toggles import gatedToggleLib, TeToggleLib

from CliModel import Int, Bool, Str, Enum, List, Model, Submodel
from CliModel import Float, GeneratorList, GeneratorDict, Dict

from CliPlugin import IsisCliHelper
from CliPlugin.IsisCliHelper import ( GRACEFUL_RESTART_STATE_MAP,
                                      FLEX_ALGO_METRIC_CAPI_MAP,
                                      SRLG_PROTECTION_MAP,
                                      GR_TS_STATE_MAP)
from CliPlugin import IsisCliModelCommon
from CliPlugin.VrfCli import generateVrfCliModel
from CliPlugin.TunnelModels import ( MplsVia,
                           tunnelTypeEnumValues,
                           TunnelInfo )
from CliPlugin.Srv6Models import Srv6SidFlavorFlags
from CliPlugin.TeCli import ( adminGroupToStr,
                              adminGroupDecimalListToDict,
                              adminGroupDictToDecimalList )
import Tac
from ArnetModel import Ip4Address, Ip6Address, IpGenericAddress, IpGenericPrefix
import Arnet
from DeviceNameLib import intfLongName
from MultiRangeRule import bitMaskToStr, bitMaskCollToStr, strToBitMask
import TableOutput
import PyRibAmiClient
import natsort
from IntfModels import Interface

isTypeHelp = '''Type of IS( Intermediate System ) 
unknown : unknown  IS Type
level1 : IS type is level-1
level2 : IS type is level-2
level1-2 : IS type is level-1-2
   '''
addressFamilyHelp = '''Address Family
none : Address Family disabled
ipv4 : Routes ipv4 only 
ipv6 : Routes ipv6 only
both : Routes both ipv4 and ipv6
   '''
interfaceAddrFamilyHelp = '''Address Family configured on this interface
none : Address Family not configured on interface
ipv4 : IPv4 configured on interface
ipv6 : IPv6 configured on interface
both : IPv4 and IPv6 configured on interface
   '''
neighborAddrFamilyHelp = '''Address Family advertised by neighbor
none : Address Family not advertised by neighbor
ipv4 : IPv4 advertised by neighbor
ipv6 : IPv6 advertised by neighbor
both : IPv4 and IPv6 advertised by neighbor
   '''
authModeHelp = '''
none : No Authentication
md5 : Authentication mode is MD5
text : Authentication mode is TEXT
    '''
authModeHelpL1 = '''Authentication mode Level1''' + authModeHelp

authModeHelpL2 = '''Authentication mode Level2''' + authModeHelp

bfdStateHelp = '''ISIS neighbor BFD state.
  adminDown - Session taken down for administrative purposes 
  init - Waiting for session establishment
  up - Session established successfully
  down - Session is down or has just been created'''

helloPaddingHelp = '''ISIS hello padding behavior
  enabled - Hello packets always padded
  enabledUntilAdj - Hello packets padded until adjacency is up
  disabled - Hello packets never padded'''

ngbHeaderFormatStr = "%-9s %-8s %-16s %-4s %-18s %-17s %-5s %-11s %-20s"
ngbEntryFormatStr = "%-9.9s %-8.8s %-16s %-4s %-18s %-17s %-5s %-11s %-20s"
spfLogHeaderFormatStr = "    %-30s %-10s %-7s %-7s %-17s"
spfLogMtHeaderFormatStr = "    %-30s %-10s %-9s %-9s %-7s %-17s"
spfLogEntryFormatStr = "   %s%-30s %-14.3f %-7d %-7d %-17s"
spfLogMtEntryFormatStr = "   %s%-30s %-14.3f %-9d %-9d %-7d %-17s"
grNgbHeaderFormatStr = "  %-18s %-7s %-18s %-17s %-10s"
isisIntfEntryFormatStr = "%-15s %-15s %-15s %-15s %-15s %-15s"
isisSumAddrEntryFormatStr = "%-28s %6d %18d %s"

ISIS_GR_RCVD_RR = 0x1
ISIS_GR_RCVD_SA = 0x2
ISIS_GR_SENT_RA = 0x4
ISIS_GR_SENT_CSNP = 0x8

MILLISEC_IN_SEC = 1000

LINE_MAX = 78

def structUnpack1B( data ):
   return struct.unpack( '1B', data.encode() )[ 0 ]

class IsisInstanceSummarySpfModel( Model ):
   if gatedToggleLib.toggleIsisSpfMaxIntervalMilliSecEnabled():
      __revision__ = 2
      spfMaxWaitInterval = Int( help='Spf Interval:Max wait(ms)' )
   else:
      __revision__ = 1
      spfMaxWaitInterval = Int( help='Spf Interval:Max wait(s)' )
   spfInitialWaitInterval = Int( help='Spf Interval:Initial wait(ms)' )
   spfHoldInterval = Int( help='Spf Interval:Hold Interval(ms)' )
   spfCurrentHoldIntervalL1 = Int( help='Current SPF hold interval(ms) for Level1', 
                                   optional=True )
   spfCurrentHoldIntervalL2 = Int( help='Current SPF hold interval(ms) for Level2', 
                                   optional=True )
   spfLastRunL1 = Float( help='Last Level 1 SPF ran time', 
                         optional=True )
   spfLastRunL2 = Float( help='Last Level 2 SPF ran time', 
                         optional=True )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         # Convert the spf max interval to seconds
         dictRepr[ 'spfMaxWaitInterval' ] //= MILLISEC_IN_SEC
      return dictRepr
   
class IsisInstanceSummaryLspGenInterval( Model ):
   lspGenMaxWaitInterval = Int( help='Lsp Generation Interval:Max wait(s)' )
   lspGenInitialWaitInterval = Int( help='Lsp Generation Interval:Initial wait(ms)' )
   lspGenHoldInterval = Int( help='Lsp Generation Interval:Hold Interval(s)' )

class IsisInstanceSummaryPreferenceModel( Model ):
   internalV4PreferenceL1 = Int( help='internal preference of level1', 
                               optional=True )
   internalV4PreferenceL2 = Int( help='internal preference of level2', 
                               optional=True )

   internalV6PreferenceL1 = Int( help='internal preference of level1', 
                               optional=True )
   internalV6PreferenceL2 = Int( help='internal preference of level2', 
                               optional=True )

class IsisInstanceSummaryModel( Model ):
   _instanceName = Str( help='Instance Name' )
   _vrf = Str( help='Vrf Name' )
   instanceId = Int( help='Instance ID' )
   hostName = Str( help="Hostname", optional=True )
   systemId = Str( help='System Id' )
   enabled = Bool( default=False, help='Instance Enabled or not' )
   attached = Bool( default=False, help='IPv4 Attached or not' )
   multiTopologyEnabled = Bool( default=False, 
                                help='multiTopology Enabled or not' )
   multiTopologyAttached = Bool( default=False, help='IPv6 Attached or not' )
   isType = Enum( values=( 'unknown', 'level1', 'level2', 'level1-2' ), 
                  default='unknown', help=isTypeHelp ) 
   numActiveInterfaces = Int( help='Number of active interfaces' )
   addressFamily = Enum( values=( 'none', 'ipv4', 'ipv6', 'both' ),  
                         default='none', help=addressFamilyHelp )
   authModeL1 = Enum( values=( 'none', 'md5', 'text', 'sha' ),
                      optional=True, help=authModeHelpL1 )
   authModeKeyIdL1 = Int( help='Authentication mode key id in Level1',
                          optional=True )
   authModeL2 = Enum( values=( 'none', 'md5', 'text', 'sha' ),
                      optional=True, help=authModeHelpL2 )
   authModeKeyIdL2 = Int( help='Authentication mode key id in Level2',
                          optional=True )
   authRxDisabledL1 = Bool( help='Received LSP authentication check is disabled for '
                            'level-1', optional=True )
   authRxDisabledL2 = Bool( help='Received LSP authentication check is disabled for '
                            'level-2', optional=True )
   sharedSecretProfileL1 = Str( help="Shared secret profile for Level-1",
                                optional=True )
   sharedSecretProfileL2 = Str( help="Shared secret profile for Level-2",
                                optional=True )
   preferenceInfo = Submodel( valueType=IsisInstanceSummaryPreferenceModel, 
                              help='Preference Information' )
   spfInfo = Submodel( valueType=IsisInstanceSummarySpfModel, 
                       help='Spf Information' )
   coloredTunnelsInLdpTunnelingSpf = Bool( help='Shortcut SPF enabled for LDP '
                                                'over RSVP-TE tunneling',
                                           optional=True )
   v4ColoredShortcutSpfForIgp = Bool( help='Colored shortcut spf over RSVP-TE '
                                           'tunneling for v4 address enabled',
                                      optional=True )
   v6ColoredShortcutSpfForIgp = Bool( help='Colored shortcut spf over RSVP-TE '
                                           'tunneling for v6 address enabled',
                                      optional=True )
   shortcutSpfForLdpTunneling = Bool( help='Shortcut SPF enabled for LDP tunneling',
                                      optional=True )
   shortcutSpfForIgp = Bool( help='IPv4 Shortcut SPF enabled for IGP',
                             optional=True )
   v6ShortcutSpfForIgp = Bool( help='IPv6 Shortcut SPF enabled for IGP' )
   lspGenInfo = Submodel( valueType=IsisInstanceSummaryLspGenInterval,
                          help="Lsp Gen Information" )
   areaIds = List( valueType=str, help='List of area-ids in this instance' )
   numDisInterfacesL1 = Int( help='Number of DIS interfaces for leve1', 
                             optional=True )
   lsdbSizeL1 = Int( help='LSDB size (number of link state database records)' +  
                     'for level1', optional=True )
   numDisInterfacesL2 = Int( help='Number of DIS interfaces for level2',
                             optional=True )
   lsdbSizeL2 = Int( help='LSDB size (number of link state database records)' +  
                     'for level2', optional=True )
   gracefulRestart = Bool( default=False, help='GR Enabled or not' )
   gracefulRestartHelper = Bool( default=False, help='GR helper Enabled or not' )
   routerIdV4 = Ip4Address( help='ISIS router ID in IP format', optional=True )
   areaLeaderL1 = Str( help='Area leader for Level 1', optional=True )
   areaLeaderL2 = Str( help='Area leader for Level 2', optional=True )
   dynamicFloodingL1 = Bool( default=False,
                             help='Dynamic Flooding is enabled for Level 1' )
   dynamicFloodingL2 = Bool( default=False,
                             help='Dynamic Flooding is enabled for Level 2' )
   lspSizeMaxL1 = Int( help='LSP size maximum for Level 1' )
   lspSizeMaxL2 = Int( help='LSP size maximum for Level 2' )
   csnpGenInterval = Int( help='CSNP generation interval' )
   redistRoutesMaxLimitL1 = \
                  Int( help='Maximum redistributed routes allowed for Level 1' )
   redistRoutesMaxLimitL2 = \
                  Int( help='Maximum redistributed routes allowed for Level 2' )
   redistRoutesCountL1 = Int( help='Redistributed routes count for Level 1' )
   redistRoutesCountL2 = Int( help='Redistributed routes count for Level 2' )
   overloadBitSetL1 = Bool( default=False,
                            help='Overload bit set for Level 1 LSP' )
   overloadBitUpdateTimestampL1 = \
                  Float( help='UTC timestamp when overload bit status for '
                              'Level 1 LSP was updated', 
                         optional=True )
   overloadBitSetL2 = Bool( default=False,
                            help='Overload bit set for Level 2 LSP' )
   overloadBitUpdateTimestampL2 = \
                  Float( help='UTC timestamp when overload bit status for '
                              'Level 2 LSP was updated', 
                         optional=True )
   msdBaseMplsImposition = Int( help='MSD type base MPLS imposition ( '  
                                     'number of MPLS label which can be ' 
                                     'imposed)', optional=True )
   _duplicateSysIdWarning = Str( help='Warning string, addressing the presence'
                                      ' of duplicate nodes in the network' )
   
   def processData( self, data ):
      from CliPlugin.RoutingIsisCli import getHostName
      # importing in function as to avoid circular dependency issue
      self._vrf = data.pop( 'vrf-name' )
      self._instanceName = data.pop( 'instance-name', None )
      self.instanceId = data.pop( 'instance-id' )
      self.systemId = data.pop( 'system-id' )
      self.hostName = getHostName( getRouterId( self.systemId, self.instanceId ),
                                                vrfName=self._vrf )
      if 'router-id-v4' in data:
         self.routerIdV4 = data.pop( 'router-id-v4' )
      else:
         self.routerIdV4 = '0.0.0.0'
      self.attached = ord( data.pop( 'attached' ) ) != 0 
      self.enabled = ord( data.pop( 'enabled' ) ) != 0
      self.multiTopologyEnabled = ord( data.pop( 'mt' ) ) != 0
      self.multiTopologyAttached = ord( data.pop( 'mt-id2-attached' ) ) != 0
      if data[ 'level' ] in IsisCliHelper.LEVEL_MAP:
         self.isType = IsisCliHelper.LEVEL_MAP[ data[ 'level' ] ]
      del data['level']
      if data[ 'addr-family' ] in IsisCliHelper.ADDRESS_FAMILY_MAP:
         self.addressFamily = IsisCliHelper.ADDRESS_FAMILY_MAP[ \
               data[ 'addr-family' ] ]
      del data[ 'addr-family' ]
      self.preferenceInfo = IsisInstanceSummaryPreferenceModel()
      self.numActiveInterfaces = data.pop( 'num-circuits' )
      self.spfInfo = IsisInstanceSummarySpfModel()
      self.spfInfo.spfHoldInterval = data.pop( 'spf-hold-int' )
      convDiv = MILLISEC_IN_SEC
      if gatedToggleLib.toggleIsisSpfMaxIntervalMilliSecEnabled():
         convDiv = 1
      self.spfInfo.spfMaxWaitInterval = data.pop( 'spf-max-int' ) // convDiv
      self.spfInfo.spfInitialWaitInterval = data.pop( 'spf-start-int' )

      self.coloredTunnelsInLdpTunnelingSpf = ord(
                                    data.pop( 'colored-ldp-shortcut-spf' ) ) != 0
      self.v4ColoredShortcutSpfForIgp = ord(
                                    data.pop( 'colored-v4-igp-shortcut-spf' ) ) != 0
      self.v6ColoredShortcutSpfForIgp = ord(
                                    data.pop( 'colored-v6-igp-shortcut-spf' ) ) != 0
      self.shortcutSpfForLdpTunneling = ord( data.pop( 'ldp-shortcut-spf' ) ) != 0
      self.shortcutSpfForIgp = ord( data.pop( 'igp-shortcut-spf' ) ) != 0
      self.v6ShortcutSpfForIgp = ord( data.pop( 'v6-igp-shortcut-spf' ) ) != 0

      self.lspGenInfo = IsisInstanceSummaryLspGenInterval()
      self.lspGenInfo.lspGenMaxWaitInterval = data.pop( 'lsp-gen-max-int' )
      self.lspGenInfo.lspGenInitialWaitInterval = data.pop( 'lsp-gen-start-int' )
      self.lspGenInfo.lspGenHoldInterval = data.pop( 'lsp-gen-hold-int' )
      for i in range( 3 ):
         areaId = data.pop( f'areaId{i + 1}', None )
         if areaId is not None:
            self.areaIds.append( areaId )
            natsort.natsorted( self.areaIds )
      if 'lsdb-size-l1' in data:
         self.lsdbSizeL1 = data.pop( 'lsdb-size-l1' )
         self.numDisInterfacesL1 = data.pop( 'ndis-l1' )
      if 'lsdb-size-l2' in data:
         self.lsdbSizeL2 = data.pop( 'lsdb-size-l2' )
         self.numDisInterfacesL2 = data.pop( 'ndis-l2' )

      from CliPlugin.RoutingIsisCli import getSharedSecretProfileName
      self.sharedSecretProfileL1 = getSharedSecretProfileName(
                                       self._instanceName, 1 )
      self.sharedSecretProfileL2 = getSharedSecretProfileName(
                                       self._instanceName, 2 )
      if 'L1 authentication mode' in data:
         mode = data.pop( 'L1 authentication mode' )
         if mode == 1:
            self.authModeL1 = "md5" 
            if self.sharedSecretProfileL1 and 'L1-authentication-mode-id' in data:
               self.authModeKeyIdL1 = data.pop( 'L1-authentication-mode-id' )
         if mode == 2:
            self.authModeL1 = "text"
         if mode == 3:
            self.authModeL1 = "sha"
            self.authModeKeyIdL1 = data.pop( 'L1-authentication-mode-id' )
      else:
         self.authModeL1 = 'none'
      self.spfInfo.spfCurrentHoldIntervalL1 = \
               data.pop( 'spf-l1-cur-int' )

      # pylint: disable-next=consider-using-in
      if self.isType == 'level1' or self.isType == 'level1-2':
         if 'l1-spf-lastran' in data:
            self.spfInfo.spfLastRunL1 = ( Tac.utcNow() ) - \
            IsisCliHelper.ribdToSecTime( data.pop( 'l1-spf-lastran' ) ) 
      self.preferenceInfo.internalV4PreferenceL1 = \
                  data.pop( 'l1-int-pref-v4' ) 
      self.preferenceInfo.internalV6PreferenceL1 = \
                  data.pop( 'l1-int-pref-v6' ) 

      if 'L2 authentication mode' in data:      
         mode = data.pop( 'L2 authentication mode' )
         if mode == 1:
            self.authModeL2 = "md5" 
            if self.sharedSecretProfileL2 and 'L2-authentication-mode-id' in data:
               self.authModeKeyIdL2 = data.pop( 'L2-authentication-mode-id' )
         if mode == 2:
            self.authModeL2 = "text"
         if mode == 3:
            self.authModeL2 = "sha"
            self.authModeKeyIdL2 = data.pop( 'L2-authentication-mode-id' )
      else:
         self.authModeL2 = 'none'
      if 'auth-rx-disabled-l1' in data:
         self.authRxDisabledL1 = ord( data[ 'auth-rx-disabled-l1' ] ) != 0
      if 'auth-rx-disabled-l2' in data:
         self.authRxDisabledL2 = ord( data[ 'auth-rx-disabled-l2' ] ) != 0

      self.spfInfo.spfCurrentHoldIntervalL2 = \
                              data.pop( 'spf-l2-cur-int' )
      # pylint: disable-next=consider-using-in
      if self.isType == 'level2' or self.isType == 'level1-2':
         if 'l2-spf-lastran' in data:
            self.spfInfo.spfLastRunL2 = ( Tac.utcNow() ) - \
            IsisCliHelper.ribdToSecTime( data.pop( 'l2-spf-lastran' ) )
            
      self.preferenceInfo.internalV4PreferenceL2 = \
                  data.pop( 'l2-int-pref-v4' ) 
      self.preferenceInfo.internalV6PreferenceL2 = \
                  data.pop( 'l2-int-pref-v6' ) 
      self.gracefulRestart = ord( data.pop( 'graceful restart' ) ) != 0
      self.gracefulRestartHelper = ord( data.pop( 'graceful restart helper' ) ) != 0
      self.areaLeaderL1 = data.pop( 'area-leader-l1', "None" )
      self.areaLeaderL2 = data.pop( 'area-leader-l2', "None" )
      self.dynamicFloodingL1 = ord( data.pop( 'dynamic-flooding-l1', '0' ) ) != 0
      self.dynamicFloodingL2 = ord( data.pop( 'dynamic-flooding-l2', '0' ) ) != 0
      self.lspSizeMaxL1 = data.pop( 'lsp-size-max-l1' )
      self.lspSizeMaxL2 = data.pop( 'lsp-size-max-l2' )
      self.csnpGenInterval = data.pop( 'csnp-gen-interval' )
      self.redistRoutesMaxLimitL1 = data.pop( 'max-redist-routes-limit-l1' )
      self.redistRoutesMaxLimitL2 = data.pop( 'max-redist-routes-limit-l2' )
      self.redistRoutesCountL1 = data.pop( 'redist-route-count-l1' )
      self.redistRoutesCountL2 = data.pop( 'redist-route-count-l2' )
      if 'overload-bit-l1' in data:
         self.overloadBitSetL1 = ord( data.pop( 'overload-bit-l1' ) ) != 0
      if 'overload-bit-l2' in data:
         self.overloadBitSetL2 = ord( data.pop( 'overload-bit-l2' ) ) != 0
      if 'overload-bit-update-ts-l1' in data:
         self.overloadBitUpdateTimestampL1 = float( data.pop( 
                                                      'overload-bit-update-ts-l1' ) )
      else:
         self.overloadBitUpdateTimestampL1 = None 
      if 'overload-bit-update-ts-l2' in data:
         self.overloadBitUpdateTimestampL2 = float( data.pop( 
                                                      'overload-bit-update-ts-l2' ) )
      else:
         self.overloadBitUpdateTimestampL2 = None 
      self.msdBaseMplsImposition = data.pop( 'msd-base-mpls-imposition', None )
      if 'dup-sys-id-warning' in data:
         self._duplicateSysIdWarning = data.pop( 'dup-sys-id-warning', None )
         _duplicateSysIdLastChecked = data.pop( 'dup-sys-id-last-checked', None )
         _duplicateSysIdClearWarningTime = data.pop( 
                                                'dup-sys-id-clear-warning-time' )
         lastCheck = datetime.fromtimestamp( 
                                 _duplicateSysIdLastChecked ).strftime(
                                 '%H:%M:%S' )
         clearWarningTime = datetime.fromtimestamp( 
                                 _duplicateSysIdClearWarningTime ).strftime( 
                                 '%H:%M:%S' )
         timeStampMsg = ". Last check at {}. Next check at {}.".format(
                        lastCheck, clearWarningTime )
         self._duplicateSysIdWarning += timeStampMsg

      return data

   def render( self ): # pylint: disable=useless-return
      print( " " )
      print( f"IS-IS Instance: {self._instanceName} VRF: {self._vrf}" )
      print( "  Instance ID: %d" % ( self.instanceId ) )
      print( "  System ID: {}, administratively {}".format( self.systemId,
            "enabled" if self.enabled else "disabled" ) )
      print( "  Router ID: IPv4: %s" % self.routerIdV4 )
      if self.hostName:
         print( "  Hostname: %s" % ( self.hostName ) )
      if self.multiTopologyEnabled:
         mt_str1 = "enabled"
         mt_str2 = "IPv4 " if self.attached else "IPv4 not "
         mt_str3 = ", IPv6 attached" if self.multiTopologyAttached \
                   else ", IPv6 not attached"
      else:
         mt_str1 = "disabled"
         mt_str2 = "" if self.attached else "not "
         mt_str3 = ""
      print( f"  Multi Topology {mt_str1}, {mt_str2}attached{mt_str3}" )
      if self.preferenceInfo:
         prefStr = "  IPv4 Preference:"
         if self.preferenceInfo.internalV4PreferenceL1:
            prefStr += " Level 1: %3d" % \
                       ( self.preferenceInfo.internalV4PreferenceL1 )
            if self.preferenceInfo.internalV4PreferenceL2:
               prefStr += ","
         if self.preferenceInfo.internalV4PreferenceL2:
            prefStr += " Level 2: %3d" % \
                       ( self.preferenceInfo.internalV4PreferenceL2 )
         print( prefStr )
         prefStr = "  IPv6 Preference:"
         if self.preferenceInfo.internalV6PreferenceL1:
            prefStr += " Level 1: %3d" % \
                       ( self.preferenceInfo.internalV6PreferenceL1 )
            if self.preferenceInfo.internalV6PreferenceL2:
               prefStr += ","
         if self.preferenceInfo.internalV6PreferenceL2:
            prefStr += " Level 2: %3d" % \
                       ( self.preferenceInfo.internalV6PreferenceL2 )
         print( prefStr )
      levelStr = IsisCliHelper.IS_TYPE_MAP[ self.isType ]
      print( "  IS-Type: %s, Number active interfaces: %d" % ( levelStr,  
            self.numActiveInterfaces ) )
      routeStr = IsisCliHelper.ROUTE_MAP[ self.addressFamily ]
      print( routeStr )

      lspSizeStr = "  LSP size maximum:"
      lspSizeStr += " Level 1: %4d," % ( self.lspSizeMaxL1 )
      lspSizeStr += " Level 2: %4d" % ( self.lspSizeMaxL2 )
      print( lspSizeStr )
      
      if self.spfInfo:
         if gatedToggleLib.toggleIsisSpfMaxIntervalMilliSecEnabled():
            tableHeadings = [ ( "Interval Type" ), ( "Max Wait" ),
                  ( "Initial Wait" ), ( "Hold Interval" ) ]
            table = TableOutput.createTable( tableHeadings, indent=2 )
            leftColFormat = TableOutput.Format( justify="left" )
            rightColFormat = TableOutput.Format( justify="right" )
            leftColFormat.padLimitIs( True )
            rightColFormat.padLimitIs( True )
            table.formatColumns( leftColFormat, rightColFormat, rightColFormat,
                                 rightColFormat )
            table.newRow( "LSP generation interval",
                          "%d s" % self.lspGenInfo.lspGenMaxWaitInterval,
                          "%d ms" % self.lspGenInfo.lspGenInitialWaitInterval,
                          "%d ms" % self.lspGenInfo.lspGenHoldInterval )
            table.newRow( "SPF interval",
                          "%d ms" % self.spfInfo.spfMaxWaitInterval,
                          "%d ms" % self.spfInfo.spfInitialWaitInterval,
                          "%d ms" % self.spfInfo.spfHoldInterval )
            print( table.output(), end='' )
         else:
            print( " " * 27 + " Max wait(s) Initial wait(ms) Hold interval(ms)" )
            print( "  LSP Generation Interval:%6d %15d %16d" %
            ( self.lspGenInfo.lspGenMaxWaitInterval,
              self.lspGenInfo.lspGenInitialWaitInterval,
              self.lspGenInfo.lspGenHoldInterval ) )

            print( "  SPF Interval:           %6d %15d %16d" %
               ( self.spfInfo.spfMaxWaitInterval,
                 self.spfInfo.spfInitialWaitInterval,
                 self.spfInfo.spfHoldInterval ) )

         currStr = "  Current SPF hold interval(ms):"
         currStr += " Level 1: %d" % ( self.spfInfo.spfCurrentHoldIntervalL1 )
         currStr += ","
         currStr += " Level 2: %d" % ( self.spfInfo.spfCurrentHoldIntervalL2 )
         print( currStr )
         if self.spfInfo.spfLastRunL1 or self.spfInfo.spfLastRunL2:
            spfLastRun = []
            if self.spfInfo.spfLastRunL1:
               spfLastRunL1TimeSec = int ( Tac.utcNow() -
                                           ( self.spfInfo.spfLastRunL1 ) )
               spfLastRunL1Time = IsisCliHelper.secToRibdTime( spfLastRunL1TimeSec )
               spfLastRunL1Time = spfLastRunL1Time if ( spfLastRunL1Time ) else "0"
               spfLastRun.append( ( 1, spfLastRunL1Time ) )

            if self.spfInfo.spfLastRunL2:
               spfLastRunL2TimeSec = int( Tac.utcNow() -
                                          ( self.spfInfo.spfLastRunL2 ) )
               spfLastRunL2Time = IsisCliHelper.secToRibdTime( spfLastRunL2TimeSec )
               spfLastRunL2Time = spfLastRunL2Time if ( spfLastRunL2Time ) else "0"
               spfLastRun.append( ( 2, spfLastRunL2Time ) )
            for ( level, val ) in spfLastRun:
               if len( val ) > 5:
                  suffix = "hours ago"
               elif len( val ) > 2:
                  suffix = "minutes ago"
               else: 
                  suffix = "seconds ago"
               
               print( f"  Last Level {level} SPF run {val} {suffix}" )
      print( "  CSNP generation interval: {} seconds".format(
         self.csnpGenInterval ) )
      if self.dynamicFloodingL1 and self.dynamicFloodingL2:
         currStr = "Level 1 and 2"
      elif self.dynamicFloodingL1:
         currStr = "Level 1"
      elif self.dynamicFloodingL2:
         currStr = "Level 2"
      else:
         currStr = "Disabled"
      print( "  Dynamic Flooding: %s" % currStr )

      if self.coloredTunnelsInLdpTunnelingSpf:
         print( "  Shortcut SPF for LDP tunneling: Enabled(Colored tunnels "
                "included)" )
      elif self.shortcutSpfForLdpTunneling:
         print( "  Shortcut SPF for LDP tunneling: Enabled" )

      if self.v4ColoredShortcutSpfForIgp:
         print( "  IPv4 Shortcut SPF for IGP: Enabled (Colored tunnels included)" )
      elif self.shortcutSpfForIgp:
         print( "  IPv4 Shortcut SPF for IGP: Enabled" )

      if self.v6ColoredShortcutSpfForIgp:
         print( "  IPv6 Shortcut SPF for IGP: Enabled (Colored tunnels included)" )
      elif self.v6ShortcutSpfForIgp:
         print( "  IPv6 Shortcut SPF for IGP: Enabled" )

      authStr = "  Authentication mode:"
      if self.authModeL1:
         if self.sharedSecretProfileL1 or self.sharedSecretProfileL2:
            authStr += "\n   "
         authStr += " Level 1: "
         if self.sharedSecretProfileL1:
            authStr += "Shared-secret profile: %s \
                        \n      Algorithm: " % self.sharedSecretProfileL1 
         authStr += "%s" % ( IsisCliHelper.AUTHENTICATION_MAP[ \
                              self.authModeL1 ] )
         if self.authModeKeyIdL1:
            authStr += " Key ID: %d" % self.authModeKeyIdL1
         if self.authModeL2:
            authStr += ","
      if self.authModeL2:
         if self.sharedSecretProfileL2 or self.sharedSecretProfileL1:
            authStr += "\n   "
         authStr += " Level 2: "
         if self.sharedSecretProfileL2:
            authStr += "Shared-secret profile: %s \
                        \n      Algorithm: " % self.sharedSecretProfileL2
         authStr += "%s" % ( IsisCliHelper.AUTHENTICATION_MAP[ \
                              self.authModeL2 ] )
         if self.authModeKeyIdL2:
            authStr += " Key ID: %d" % self.authModeKeyIdL2
      print( authStr )

      rxDisabledLevels = []
      if self.authRxDisabledL1:
         rxDisabledLevels.append( 'Level 1: disabled' )
      if self.authRxDisabledL2:
         rxDisabledLevels.append( 'Level 2: disabled' )
      if rxDisabledLevels:
         print( "  Received LSP authentication check:",
                ", ".join( rxDisabledLevels ) )
         
      if self.gracefulRestart:
         grStr = "  Graceful Restart: Enabled,"
      else:
         grStr = "  Graceful Restart: Disabled,"
      if self.gracefulRestartHelper:
         grStr += " Graceful Restart Helper: Enabled"
      else:
         grStr += " Graceful Restart Helper: Disabled"
      print( grStr )   

      if len( self.areaIds ) == 1:
         # Render output should show area address like
         # Area addresses: 49.0001
         print( "  Area addresses: %s" % self.areaIds[ 0 ] )
      else:
         # In case of more than one area addresses - each area address will be
         # printed on a separate line, like
         # Area addresses:
         #   49.0001
         #   50.0001.0002
         print( "  Area addresses:" )
         for area in natsort.natsorted( self.areaIds ):
            print( "    %s" % area )
      if self.numDisInterfacesL1 is not None and self.lsdbSizeL1 is not None:
         print( "  level 1: number DIS interfaces: %d, LSDB size: %d" % (
               self.numDisInterfacesL1, self.lsdbSizeL1 ) )
         print( "    Area Leader:", self.areaLeaderL1 )
         if self.overloadBitSetL1:
            print( "    Overload Bit is set.", end=' ' ) 
            setOrClear = "Set on "
         else:
            print( "    Overload Bit is not set.", end=' ' ) 
            setOrClear = "Cleared on "
         if self.overloadBitUpdateTimestampL1 is not None:
            updateTimestamp = datetime.fromtimestamp(
                                 self.overloadBitUpdateTimestampL1 ).strftime(
                                                            '%Y-%m-%d %H:%M:%S')
            print( f"{setOrClear}{updateTimestamp}", end=' ' )
         print( "" )

      if self.numDisInterfacesL2 is not None and self.lsdbSizeL2 is not None:
         print( "  level 2: number DIS interfaces: %d, LSDB size: %d" % ( 
               self.numDisInterfacesL2, self.lsdbSizeL2 ) )
         print( "    Area Leader:", self.areaLeaderL2 )
         if self.overloadBitSetL2:
            print( "    Overload Bit is set.", end=' ' ) 
            setOrClear = "Set on "
         else:
            print( "    Overload Bit is not set.", end=' ' ) 
            setOrClear = "Cleared on "
         if self.overloadBitUpdateTimestampL2 is not None:
            updateTimestamp = datetime.fromtimestamp(
                                 self.overloadBitUpdateTimestampL2 ).strftime(
                                                            '%Y-%m-%d %H:%M:%S')
            print( f"{setOrClear}{updateTimestamp}", end=' ' )
         print( "" )

      if self.redistRoutesMaxLimitL1 == 0:   
         print( "  Redistributed Level 1 routes: %d limit: Not Configured" % (
                    self.redistRoutesCountL1 ) )
      else:
         print( "  Redistributed Level 1 routes: %d limit: %d" % (
                      self.redistRoutesCountL1, self.redistRoutesMaxLimitL1 ) )
         
      if self.redistRoutesMaxLimitL2 == 0:   
         print( "  Redistributed Level 2 routes: %d limit: Not Configured" % (
                    self.redistRoutesCountL2 ) )
      else:
         print( "  Redistributed Level 2 routes: %d limit: %d" % (
                    self.redistRoutesCountL2, self.redistRoutesMaxLimitL2 ) )
      if self.msdBaseMplsImposition is not None: 
         print( "  Maximum SID depth: Enabled" )
         print( "    Base MPLS imposition (MSD type 1): %d" % (
                     self.msdBaseMplsImposition ) )
      return

if gatedToggleLib.toggleIsisSpfMaxIntervalMilliSecEnabled():
   isisSummaryModel = IsisCliModelCommon. \
                   generateIsisDictCliModel( IsisInstanceSummaryModel, revision=2 )
else:
   isisSummaryModel = IsisCliModelCommon. \
                   generateIsisDictCliModel( IsisInstanceSummaryModel, revision=1 )

isisSummaryVRFsModel = generateVrfCliModel( isisSummaryModel,
                                    "ISIS instance information for all VRFs" )

class IsisSumAddrEntryModel( Model ):
   _summaryAddress = Str( help='Summary address' )
   metric = Int( help="Metric value" )
   totalContributors = Int( help="Total number of contributors" )
   advertised = Str( help='Summary address advertised' )
   contributors = List( valueType=IpGenericPrefix, help='List of contributors',
         optional=True )
   _detail = Int( help='Display information in detail' )

   def getKey( self, data ):
      if data.get( 'metric' ) is not None:
         sumAddr = data.get( 'summary-address-v4' ) or data.get(
            'summary-address-v6' )
         return sumAddr + '/' + str( data[ 'mask-length' ] )
      return self._summaryAddress

   def overrideHierarchy( self, data ):
      metric = data.get( 'metric', None )
      if not self._summaryAddress:
         assert metric is not None
         self.metric = data.pop( 'metric', None )
         sumAddr = data.get( 'summary-address-v4' ) or data.get(
            'summary-address-v6' )
         self._summaryAddress = ( sumAddr + '/' + str( data.pop( 'mask-length' ) ) )
         self.totalContributors = data.pop('total-contributors', None )
         self.advertised = "yes" if self.totalContributors else "no"
         self._detail = data.pop( 'detail' )
         if not self._detail:
            self.contributors = None
         return ( data, True )
      else:
         if metric is not None:
            return ( data, False )
         contAddr = data.get( 'contributor-address-v4' ) or data.get(
            'contributor-address-v6' )
         contributorAddress = ( contAddr  + "/" + str( data.pop( 'mask-length' ) ) )
         self.contributors.append( contributorAddress )
         return ( data, True )

   def renderData( self, detail ):
      if not detail:
         print( isisSumAddrEntryFormatStr  % ( self._summaryAddress, self.metric,
            self.totalContributors, self.advertised ) )
      else:
         display = "Advertised" if self.totalContributors else "Not advertised"
         print( "" )
         print( "%s, Metric: %d, Total contributors: %d, %s" % (
            self._summaryAddress, self.metric, self.totalContributors, display ) )
         if self.contributors:
            print( "Contributing prefixes from IS-IS L1 LSP:" )
            # pylint: disable=not-an-iterable
            for contAddr in self.contributors:
               print( "%s" % contAddr )

class IsisLeakedRouteSumAddrInstanceModel( Model ):
   _vrf = Str( help='Vrf Name' )
   _instanceName = Str( help='Instance Name' )
   level1ToLevel2SummaryAddresses = GeneratorDict( keyType=IpGenericPrefix,
         valueType=IsisSumAddrEntryModel,
         help='A dictionary of summary-addresses for L1 to L2 leaked routes' )
   _detailsPresent = Bool( default=False, help='Private attribute to indicate'
         ' if detailed output was requested' )

   def processData( self, data ):
      self._vrf = data.pop( 'vrf-name' )
      self._instanceName = data.pop( 'instance-name', None )

   def render( self ):
      print( "" )
      print( f"IS-IS Instance: {self._instanceName} VRF: {self._vrf}" )
      print( "")
      print( "IS-IS summary-address level-1 to level-2" )
      if not self._detailsPresent:
         print( "")
         print( "Summary address              Metric Total Contributors Advertised" )
         print( "---------------------------- ------ ------------------ ----------" )

      for _, entry in self.level1ToLevel2SummaryAddresses:
         entry.renderData( self._detailsPresent )

isisLeakedRouteSumAddrModel = IsisCliModelCommon. \
                    generateIsisDictCliModel( IsisLeakedRouteSumAddrInstanceModel )
isisLeakedRouteSumAddrVRFsModel = generateVrfCliModel( isisLeakedRouteSumAddrModel,
                    "ISIS  route summary-address for all VRFs" )

LSP_GEN_EVENT_MAP = [
   "backoff started, hold value", 
   "backoff continue, remaining wait time", 
   "backoff restarted, hold value", 
   "out-delay applied, delayed by"
]

class IsisLspLogEntryModel( Model ):
   logTime = Int( help='Log time stamp' )
   logId = Int( help='Log entry identifier' )
   lspLevel = Int( help='Level of the LSP' )
   event = Enum( values=LSP_GEN_EVENT_MAP,
                 help='Event associated with Log' )
   holdInterval = Int( help='Hold interval for the LSP in ms' )
   lspId = Str( help='IS-IS LSPID' )
   lspIdHostname = Str( help='Hostname of the LSP owner', optional=True )
   
   def render( self ):
      helperLspId = self.lspId
      if self.lspIdHostname:
         helperLspId = helperLspId + ' (' + self.lspIdHostname + ')'

      print( ( "%s: %d lsp %s level-%d %s %d msec" ) % (
         strftime( "%Y-%m-%d %H:%M:%S", localtime( self.logTime ) ), 
         self.logId,
         helperLspId, self.lspLevel,
         self.event, self.holdInterval ) )

   def processData( self, data ):
      self.logTime = data.pop( 'time' )
      self.logId = data.pop( 'id' )
      self.lspLevel = ord( data.pop( 'level' ) )
      self.event = LSP_GEN_EVENT_MAP[ ord( data.pop( 'event' ) ) ]
      self.holdInterval = data.pop( 'hold-interval' )
      self.lspId = data.pop( 'lspid' )
      self.lspIdHostname = data.pop( 'lspIdHostname', None )

class IsisLspLogTableModel( Model ):
   _instanceName = Str( help='Instance Name' )
   _vrf = Str( help='Vrf Name' )
   lspLogs = GeneratorList( valueType=IsisLspLogEntryModel,
                              help='List of lsplog Entries' )

   def processData( self, data ):
      self._vrf = data.pop( 'vrf-name' )
      self._instanceName = data.pop( 'instance-name', None )
   
   def render( self ):
      print( "" )
      print( f"IS-IS Instance: {self._instanceName} VRF: {self._vrf}" )
      print( "" )
      for entry in self.lspLogs:
         entry.render()

lspLogTableModel = IsisCliModelCommon. \
                   generateIsisDictCliModel( IsisLspLogTableModel )
lspLogTableVRFsModel = generateVrfCliModel( lspLogTableModel,
                                    "IS-IS instance information for all VRFs" )

class LspFlags( Model ):
   partitionSupport = Bool( default=False,
                            help='Support to Partition Repair' )
   errorAttached = Bool( default=False,
                          help='Attached to other areas by the Error Metric' )
   expenseAttached = Bool( default=False,
                         help='Attached to other areas by the Expense Metric' )
   delayAttached = Bool( default=False,
                          help='Attached to other areas by the Delay Metric' )
   defaultAttached = Bool( default=False,
                            help='Attached to other areas by the Default Metric' )
   dbOverload = Bool( default=False,
                      help='LSP Database Overload' )

def getFlagsString( flags ):
   flagStr = ''
   for flag in IsisCliHelper.ISIS_LSP_FLAGS_MAP:
      if getattr( flags, flag ):
         flagStr += IsisCliHelper.ISIS_LSP_FLAGS_MAP[ flag ] + ' '
   return flagStr.strip()

class LspMtModel( Model ):
   topologyId = Int( help="Topology Id" )
   mtOverloaded = Bool( default=False, help="MultiTopology Overload Flag" )
   mtAttached = Bool( default=False, help="MultiTopology Attached Flag" )

   def processData( self, data ):
      self.topologyId = data.pop( 'topo_id', None ) 
      self.mtOverloaded = ord ( data.pop( 'mt_overload' ) ) != 0
      self.mtAttached = ord ( data.pop( 'mt_attach' ) ) != 0
      return data
   
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_MTID
   
   def render( self ):
      overloadString = ", overloaded" if self.mtOverloaded else ""
      attachString = ", attached" if self.mtAttached else ""
      if self.topologyId == 0:
         print( "      Topology: {} (IPv4){}{}".format( self.topologyId,
                                                   overloadString,
                                                   attachString ) )
      elif self.topologyId == 2:
         print( "      Topology: {} (IPv6){}{}".format( self.topologyId,
                                                   overloadString,
                                                   attachString ) )  
      else:
         print( "      Topology: {} (Unsupported){}{}".format( self.topologyId,
                                                           overloadString,
                                                           attachString ) )
         
class LspAuthModel( Model ):
   authMode = Enum( values=IsisCliHelper.AUTH_MODE_MAP.values(),
                    help="Authentication Mode" )
   authLength = Int( help="Authentication Key Length" )
   authKeyId = Int( help="Authentication Key Id", optional=True )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_AUTH

   def processData( self, data ):
      self.authMode = IsisCliHelper.AUTH_MODE_MAP[ data.pop( 'Authentication mode' \
            , None ) ]
      self.authLength = data.pop( 'Authentication length', None )
      if self.authMode == "SHA":
         self.authKeyId = data.pop( 'Authentication-key-id' )
      return data
   
   def render( self ):
      if self.authMode is None or self.authLength is None :
         return
      else:
         keyIdStr = ""
         if self.authKeyId is not None:
            keyIdStr = " Key id: %d" % self.authKeyId
         print( "      Authentication mode: {}{} Length: {}".format( self.authMode,
                                                                keyIdStr,
                                                                self.authLength ) )
class LspAreaAddress( Model ):
   address = Str( help="Area Address for lsp" )
   
   def processData( self, data ):
      self.address = data.pop( 'area-id' )
      return data
   
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_AREA
   
class LspHostname( Model ):
   name = Str( help="LSP Hostname" )
   
   def processData( self, data ):
      self.name = data.pop( 'hostname' )
      return data
   
   def render( self ):
      print( "      Hostname: %s" % ( self.name ) )
      
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_LSDB_HOSTNAME
   
class InterfaceAddresses( Model ):
   ipv4Address = Ip4Address( help="IPv4 address", optional=True )
   ipv6Address = Ip6Address( help="IPv6 address", optional=True )

   def processData( self, data ):
      if 'ipv4-ifaddr' in data:
         self.ipv4Address = data.pop( 'ipv4-ifaddr' )
      if 'ipv6-ifaddr' in data:
         addr = data.pop( 'ipv6-ifaddr', None )
         if addr and '%' in addr:
            addr = addr[ : addr.index( '%' ) ]
         self.ipv6Address = addr
   def render( self ):
      outputStr = "      Interface address: %s"
      address = self.ipv4Address if self.ipv4Address else self.ipv6Address
      print( outputStr % address )
      
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_ADDRESS

class LspAdjInterfaceAddress( Model ):
   adjInterfaceAddress = Ip4Address( help="Interface Address" )
   
   def processData( self, data ):
      self.adjInterfaceAddress = data.pop( 'intf-addr' )
      
   def render( self ):
      print( "        IPv4 Interface Address: %s" % ( self.adjInterfaceAddress ) )
      
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_ADJ_INTF_ADDR

def setAdjFlagsRenderStr( flagSubmodel, flag ):
   flagSubmodel.adjSet = bool( flag &
         IsisCliHelper.ADJ_SID_SUBTLV_FLAGS[ 'ISIS_SR_ADJ_SET' ] )
   flagSubmodel.adjLocal = bool( flag &
         IsisCliHelper.ADJ_SID_SUBTLV_FLAGS[ 'ISIS_SR_ADJ_LOCAL' ] )
   flagSubmodel.adjValue = bool( flag &
         IsisCliHelper.ADJ_SID_SUBTLV_FLAGS[ 'ISIS_SR_ADJ_VALUE' ] )
   flagSubmodel.adjBackup = bool( flag &
         IsisCliHelper.ADJ_SID_SUBTLV_FLAGS[ 'ISIS_SR_ADJ_BACKUP' ] )
   flagSubmodel.adjAf = bool( flag &
         IsisCliHelper.ADJ_SID_SUBTLV_FLAGS[ 'ISIS_SR_ADJ_AF' ] )

def getAdjFlagsRenderList( flags ):
   adjSet = "S " if ( flags.adjSet )  else ""
   adjLocal = "L " if ( flags.adjLocal ) else ""
   adjValue = "V " if ( flags.adjValue ) else ""
   adjBackup = "B " if ( flags.adjBackup ) else ""
   adjAf = "F " if ( flags.adjAf ) else ""
   return ( adjSet + adjLocal + adjValue + adjBackup +
            adjAf ).strip()

class AdjFlags( Model ):
   adjSet = Bool( help="Sr Adj Set Flag" )
   adjLocal = Bool( help="Sr Adj Local Flag" )
   adjValue = Bool( help="Sr Adj Value Flag" )
   adjBackup = Bool( help="Sr Adj Backup Flag" )
   adjAf = Bool( help="Sr Adj AddressFamily Flag" )
   
class LspAdjLanSid( Model ):
   adjLanSid = Int( help="Adjacency LAN SID" )
   adjLanSystemId = Str( help="Adjacency SystemId" )
   adjLanFlags = Submodel( valueType=AdjFlags, help="Adjacency LAN Flags" )
   adjLanWeight = Int( help="Adjacency LAN Weight" )
   
   def processData( self, data ):
      self.adjLanSid = data.pop( 'adj-lan-sid' )
      self.adjLanSystemId = data.pop( 'adj-lan-system-id' )
      flags = ord( data.pop( 'adj-lan-flags' ) )
      self.adjLanFlags = AdjFlags()
      setAdjFlagsRenderStr( self.adjLanFlags, flags )
      self.adjLanWeight = ord( data.pop( 'adj-lan-weight' ) )
      
   def render( self ):
      adjFlags = getAdjFlagsRenderList( self.adjLanFlags )
      outFmt = "        LAN-Adj-sid: %s flags: [%s] weight: "
      outFmt += "%u system ID: %s"
      print( outFmt % ( self.adjLanSid, adjFlags, self.adjLanWeight,
                       self.adjLanSystemId ) )

   def getMioAttrId( self ):
      return  PyRibAmiClient.MIO_DGET_ISIS_ADJ_LAN_SID

class LspAdjSid( Model ):
   adjSid = Int( help="Adjacency SID" )
   adjFlags = Submodel( valueType=AdjFlags, help="Adjacency Flags" )
   adjWeight = Int( help="Adjacency Weight" )
   
   def processData( self, data ):
      self.adjSid = data.pop( 'adj-sid' )
      flags = ord( data.pop( 'adj-flags' ) )
      self.adjFlags = AdjFlags()
      setAdjFlagsRenderStr( self.adjFlags, flags )
      self.adjWeight = ord( data.pop( 'adj-weight' ) )
      
   def render( self ):
      adjFlags = getAdjFlagsRenderList( self.adjFlags )
      print( "        Adj-sid: {} flags: [{}] weight: 0x{:x}".format(
         self.adjSid, adjFlags, self.adjWeight ) )
      
   def getMioAttrId( self ):
      return  PyRibAmiClient.MIO_DGET_ISIS_ADJ_SID
   
ADJ_MTID_MAP = { 'MTID_IPV6' : 2, 
}

def bw_best_value_units( bw ):
   # bw is in bps, convert it to the most readble units for rendering
   for unit in [ "bps", "Kbps", "Mbps" ]:
      if bw < 1000:
         return bw, unit
      else:
         bw /= 1000
   return bw, "Gbps"

class TeReservablePriorityBandwidth( Model ):
   priority0 = Int( help="Reservable Bandwidth (bps) for priority level 0" )
   priority1 = Int( help="Reservable Bandwidth (bps) for priority level 1" )
   priority2 = Int( help="Reservable Bandwidth (bps) for priority level 2" )
   priority3 = Int( help="Reservable Bandwidth (bps) for priority level 3" )
   priority4 = Int( help="Reservable Bandwidth (bps) for priority level 4" )
   priority5 = Int( help="Reservable Bandwidth (bps) for priority level 5" )
   priority6 = Int( help="Reservable Bandwidth (bps) for priority level 6" )
   priority7 = Int( help="Reservable Bandwidth (bps) for priority level 7" )

   def renderHelper( self, indent=None ):
      indent = indent or " " * 8
      print( "%sUnreserved BW:" % indent )
      NUM_BW_CLASSES = 8
      indent += " " * 4
      PriorityBwHeader1 = "%sTE class %s: %0.2f %s\tTE class %s: %0.2f %s"
      PriorityBwHeader2 = "\tTE class %s: %0.2f %s"
      linesWithThreeClasses = int( NUM_BW_CLASSES / 3 )
      for i in range( linesWithThreeClasses ):
         bw0 = getattr( self, "priority" + str( 3 *i ) )
         bw1 = getattr( self, "priority" + str( 3 *i + 1 ) )
         bw2 = getattr( self, "priority" + str( 3 *i + 2 ) )
         best_value_units_bw0 = bw_best_value_units( bw0 )
         best_value_units_bw1 = bw_best_value_units( bw1 )
         best_value_units_bw2 = bw_best_value_units( bw2 )
         print( ( PriorityBwHeader1 + PriorityBwHeader2 ) % ( indent, 3 * i,
                                    best_value_units_bw0[ 0 ],
                                    best_value_units_bw0[ 1 ],
                                    3*i + 1,
                                    best_value_units_bw1[ 0 ],
                                    best_value_units_bw1[ 1 ],
                                    3 *i + 2,
                                    best_value_units_bw2[ 0 ],
                                    best_value_units_bw2[ 1 ] ) )

      bw0 = getattr( self, "priority" + str( 3 *linesWithThreeClasses ) )
      bw1 = getattr( self, "priority" + str( 3 *linesWithThreeClasses + 1 ) )
      best_value_units_bw0 = bw_best_value_units( bw0 )
      best_value_units_bw1 = bw_best_value_units( bw1 )
      print( PriorityBwHeader1 % ( indent, 3 *linesWithThreeClasses,
                                  best_value_units_bw0[ 0 ],
                                  best_value_units_bw0[ 1 ],
                                  3 *linesWithThreeClasses + 1,
                                  best_value_units_bw1[ 0 ],
                                  best_value_units_bw1[ 1 ] ) )


class LinkTeInfo( Model ):
   administrativeGroup = Int( help="Administrative Group of the Link",
                              optional=True )
   administrativeGroupsExtended = List( valueType=int, optional=True,
         help="Extended Administrative Groups of the Link" )
   metric = Int( help="TE Link cost (if configured on link)", optional=True )
   maxLinkBw = Int( help="Maximum bandwidth (bps) that can be used " \
                      "on Directed Link",
                      optional=True )
   maxReservableBw = Int( help="Maximum bandwidth (bps) that can be reserved "
                            "on Directed Link",
                            optional=True )
   maxReservablePriorityBw = Submodel( valueType=TeReservablePriorityBandwidth,
                                        help="Maximum bandwidth "
                                        "reservable for a priority",
                                        optional=True )
   minDelay = Int( help="Min unidirectional link delay (us)", optional=True )
   maxDelay = Int( help="Max unidirectional link delay (us)", optional=True )
   averageDelay = Int( help="Average unidirectional link delay (us)",
                      optional=True )
   delayVariation = Int( help="Unidirectional link delay variation (us)",
                        optional=True )
   minMaxDelayAnomalous = Bool( help="Anomalous min/max link delay",
                               optional=True )
   localLinkId = Int( help="Local link ID", optional=True )
   remoteLinkId = Int( help="Remote link ID", optional=True )

   def processData( self, data ):
      self.administrativeGroup = data.pop( 'color0', None )
      if self.administrativeGroup == 0:
         # We don't want the rendering code to attempt to display the adminGroup
         # value of 0, so explicitly set to None for this case
         self.administrativeGroup = None
      if TeToggleLib.toggleExtendedAdminGroupEnabled():
         adminGroupDict = { 0: self.administrativeGroup }
         # We support EAGs in the 0-127 range ( 4 indices altogether ), however this
         # is doubled here to attempt to display exactly what was received in the
         # LSP DB
         NUM_EAG_INDICES = 8
         for i in range( 1, NUM_EAG_INDICES ):
            adminGroupDict[ i ] = data.pop( f'color{i}', 0 )
         adminGroupList = adminGroupDictToDecimalList( adminGroupDict )
         self.administrativeGroupsExtended = adminGroupList or None
      else:
         self.administrativeGroupsExtended = None
      self.metric = data.pop( 'metric', None )
      maxLinkBw = data.pop( 'max_link_bw', None )
      self.maxLinkBw = int( maxLinkBw * ( 10 **6 ) ) if maxLinkBw is not None \
                       else None
      maxReservableBw = data.pop( 'max_resv_bw', None )
      self.maxReservableBw = int( maxReservableBw * ( 10 **6 ) ) if maxReservableBw \
                             is not None else None

      maxReservablePriorityBw = TeReservablePriorityBandwidth()
      # pylint: disable=W0612
      priority0 = data.pop( 'max_unresv_bw0', None )
      priority1 = data.pop( 'max_unresv_bw1', None )
      priority2 = data.pop( 'max_unresv_bw2', None )
      priority3 = data.pop( 'max_unresv_bw3', None )
      priority4 = data.pop( 'max_unresv_bw4', None )
      priority5 = data.pop( 'max_unresv_bw5', None )
      priority6 = data.pop( 'max_unresv_bw6', None )
      priority7 = data.pop( 'max_unresv_bw7', None )

      NUM_BW_CLASSES = 8
      for i in range( NUM_BW_CLASSES ):
         val = locals()[ "priority" + str( i ) ]
         if val is not None:
            setattr( maxReservablePriorityBw, "priority" + str( i ),
                     int( val * ( 10 **6 ) ) )
         else:
            setattr( maxReservablePriorityBw, "priority" + str( i ), None )
      if maxReservablePriorityBw.priority0 is not None:
         self.maxReservablePriorityBw = maxReservablePriorityBw

      self.minDelay = data.pop( 'min_delay', None )
      self.maxDelay = data.pop( 'max_delay', None )
      flags = data.pop( 'minmax_delay_flags', None )
      if flags:
         self.minMaxDelayAnomalous = bool(
            ord( flags ) & IsisCliHelper.ISIS_MIN_MAX_DELAY_ABIT )
      self.averageDelay = data.pop( 'average_delay', None )
      self.delayVariation = data.pop( 'delay_variation', None )
      self.localLinkId = data.pop( 'local_linkid', None )
      self.remoteLinkId = data.pop( 'remote_linkid', None )

   def renderHelper( self, indent=None ):
      indent = indent or ""
      maxLinkBwHeader = "%sMaximum link BW: %0.2f %s"
      maxReservableBwHeader = "%sMaximum reservable link BW: %0.2f %s"
      adminColorHeader = "%sAdministrative group (Color): %s"
      adminColorHex = " (0x%x)"
      minMaxDelayHeader = "%sMin/Max unidirectional link delay: %s%d/%d us"
      averageDelayHeader = "%sAverage unidirectional link delay: %d us"
      delayVariationHeader = "%sUnidirectional link delay variation: %d us"
      localLinkHeader = "%sLocal link ID: %d"
      remoteLinkHeader = "%sRemote link ID: %d"
      if ( self.administrativeGroup is not None or
            self.administrativeGroupsExtended is not None ):
         if TeToggleLib.toggleExtendedAdminGroupEnabled():
            adminGroup = adminGroupDecimalListToDict(
                  self.administrativeGroupsExtended )
            print( adminColorHeader % ( indent, adminGroupToStr( adminGroup ) ) )
         else:
            print( ( adminColorHeader + adminColorHex ) %
                  ( indent, adminGroupToStr( self.administrativeGroup ),
                     self.administrativeGroup ) )
      if self.metric is not None:
         print( f"{indent}TE default metric: {self.metric}" )
      if self.maxLinkBw is not None:
         bestValueUnit = bw_best_value_units( self.maxLinkBw )
         print( maxLinkBwHeader % ( indent, bestValueUnit[ 0 ],
                                   bestValueUnit[ 1 ] ) )
      if self.maxReservableBw is not None:
         bestValueUnit = bw_best_value_units( self.maxReservableBw )
         print( maxReservableBwHeader % ( indent, bestValueUnit[ 0 ],
                                         bestValueUnit[ 1 ] ) )
      if self.maxReservablePriorityBw:
         self.maxReservablePriorityBw.renderHelper( indent=indent )

      if self.minDelay:
         print( minMaxDelayHeader % ( indent,
                                     "[A] " if self.minMaxDelayAnomalous else "",
                                     self.minDelay, self.maxDelay ) )
      if self.averageDelay:
         print( averageDelayHeader % ( indent, self.averageDelay ) )
      if self.delayVariation:
         print( delayVariationHeader % ( indent, self.delayVariation ) )

      if self.localLinkId:
         print( localLinkHeader % ( indent, self.localLinkId ) )
      if self.remoteLinkId:
         print( remoteLinkHeader % ( indent, self.remoteLinkId ) )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_LINK_TE

class TeRouterId( Model ):
   v4RouterId = Ip4Address( help="TE IPv4 Router-Id" )
   v6RouterId = Ip6Address( help="TE IPv6 Router-Id", optional=True )

   def render( self ):
      if self.v4RouterId is not None:
         print( "      TE IPv4 router ID: %s" % ( self.v4RouterId ) )
      if self.v6RouterId is not None:
         print( "      TE IPv6 router ID: %s" % ( self.v6RouterId ) )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_TE_RTRID

def getAppsFromBitMask( bitMaskList, standard ):
   # Get application IDs from the given bitMaskList.
   # For example, in bitMaskList = [ 0b1001000 ], bit#0 and bit#3 are set,
   # and hence application IDs [ 0, 3 ] will be returned.
   appIds = []
   for i, bitMask in enumerate( bitMaskList ):
      bitPos = i * 8
      while bitMask > 0:
         if bitMask & 0x80:
            appStr = IsisCliHelper.applicationIdToStr( bitPos, standard )
            appIds.append( appStr )
         bitPos += 1
         bitMask = ( bitMask << 1 ) & 0xff
   return appIds

class ApplicationIdentifiers( Model ):
   standardAppsUseLegacy = Bool( help='Use legacy link attributes', optional=True )
   standardApps = List( valueType=str, help='Standard applications', optional=True )
   userDefinedApps = List( valueType=str, help='User defined applications',
                           optional=True )

   def processData( self, data ):
      self.standardAppsUseLegacy = data.pop( 'sabm-legacy-flag', None ) is not None
      sabm = data.pop( 'sabm', None )
      self.standardApps = getAppsFromBitMask( sabm, True ) if sabm else []
      udabm = data.pop( 'udabm', None )
      self.userDefinedApps = getAppsFromBitMask( udabm, False ) if udabm else []

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_TE_APP

   def renderHelper( self, indent ):
      if self.standardAppsUseLegacy or self.standardApps:
         print( '{}Standard applications: {}{}'.format(
            indent, '[L] ' if self.standardAppsUseLegacy else '',
            ' '.join( self.standardApps ) ) )
      if self.userDefinedApps:
         print( '{}User-defined applications: {}'.format(
            indent, ' '.join( self.userDefinedApps ) ) )

class LspApplicationLinkAttributes( Model ):
   # Application Specific Link Attribute sub-TLV
   applications = Submodel( valueType=ApplicationIdentifiers,
                            help='Application identifiers' )
   linkAttributes = Submodel( valueType=LinkTeInfo, help='TE link attributes',
                              optional=True )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_TE_ASLA

   def render( self ):
      if self.applications:
         indent = ' ' * 8
         print( '%sApplication Specific Link Attributes:' % indent )

         indent += ' ' * 2
         self.applications.renderHelper( indent=indent )

         indent += ' ' * 2
         if self.linkAttributes:
            self.linkAttributes.renderHelper( indent=indent )

class LspNeighbors( Model ):
   systemId = Str( help="IS-IS System Id" )
   metric = Int( help="Link Metric" )
   neighborAddr = Ip4Address( help="Neighbor Interface address" )
   adjTopoId = Int( help="Neighbor topology Id" )
   adjNarrowMetricsEnabled = Bool( help="Neighbor narrow metric" )
   adjInterfaceAddresses = GeneratorList( valueType=LspAdjInterfaceAddress,
                                 help="List of interface addresses" )
   adjSids = GeneratorList( valueType=LspAdjSid, help="List of Adj SID" )
   adjLanSids = GeneratorList( valueType=LspAdjLanSid, help="List of LAN  Adj SID" )
   applicationLinkAttributes = GeneratorList(
      valueType=LspApplicationLinkAttributes,
      help='List of application specific link attributes', optional=True )
   ip6NeighborAddress = Ip6Address( help="Neighbor IPv6 global address",
                                    optional=True )
   ip6GlobalInterfaceAddress = Ip6Address( help="IPv6 global interface address",
                                           optional=True )
   TEInfo = Submodel( valueType=LinkTeInfo,
                              help="TE Information for"
                              "the Link" )
   def processData( self, data ):
      self.systemId = data.pop( 'system-id', None )
      self.metric = data.pop( 'metric', None )
      self.neighborAddr = data.pop( 'ngb-addr', None )
      self.adjTopoId = data.pop( 'adj-topo-id', None )
      self.adjNarrowMetricsEnabled = 'adj-narrow-metrics' in data
      self.ip6NeighborAddress = data.pop( 'ipv6-ngb-addr', None )
      self.ip6GlobalInterfaceAddress = data.pop( 'ipv6-intf-addr', None )
      return data

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_ADJACENCY

   def render( self ):
      if self.adjTopoId:
         if self.adjTopoId == ADJ_MTID_MAP[ 'MTID_IPV6' ]:
            print( "      IS Neighbor (MT-IPv6): %-19s Metric: %s" %
                   ( self.systemId, self.metric ) )
         else:
            print( "      IS Neighbor (MT %s)  : %-19s Metric: %s" %
                   ( self.adjTopoId, self.systemId, self.metric ) )
      elif self.adjNarrowMetricsEnabled:
         fmt = "      IS Neighbor (Narrow metrics, unsupported): %-19s Metric: %s"
         print( fmt % ( self.systemId, self.metric ) )
      else:
         print( "      IS Neighbor          : %-19s Metric: %s" % ( self.systemId,
                                                             self.metric ) )
         if self.neighborAddr:
            print( "        IPv4 Neighbor Address: %s" % ( self.neighborAddr ) )

      for adjInterfaceAddress in self.adjInterfaceAddresses:
         adjInterfaceAddress.render()

      if self.ip6NeighborAddress:
         print( "        IPv6 Neighbor Address: %s" % ( self.ip6NeighborAddress ) )
      if self.ip6GlobalInterfaceAddress:
         print( "        Global IPv6 Interface Address: %s" %
                ( self.ip6GlobalInterfaceAddress ) )

      for adjSid in self.adjSids :
         adjSid.render()
      for adjLanSid in self.adjLanSids :
         adjLanSid.render()
      if self.TEInfo:
         self.TEInfo.renderHelper( indent=' ' * 8 )
      for asla in self.applicationLinkAttributes:
         asla.render()

class SrPrefixOptions( Model ):
   readvert = Bool( help="SR readvertisement Flag" )
   nodeSID = Bool( help="SR Node SID Flag" )
   noPenultimate = Bool( help="SR No Penultimate Hop Flag" )       
   explicitNull = Bool( help="SR Explicit Null Flag" )
   value = Bool( help="Prefix-SID carries a value instead of an index" )
   local = Bool( help="Value/Index carried by the" +
                     " Prefix-SID has local significance" )
   
def setPrefixOptions( subModel, prefixOptions ):
   subModel.readvert = bool( prefixOptions &
         IsisCliHelper.PREFIX_SEGMENT_FLAGS[ 'ISIS_SR_PFX_READVERT' ] )
   subModel.nodeSID = bool( prefixOptions &
         IsisCliHelper.PREFIX_SEGMENT_FLAGS[ 'ISIS_SR_PFX_NODE' ] )
   subModel.noPenultimate = bool( prefixOptions &
         IsisCliHelper.PREFIX_SEGMENT_FLAGS[ 'ISIS_SR_PFX_NOPHP' ] )
   subModel.explicitNull = bool( prefixOptions &
         IsisCliHelper.PREFIX_SEGMENT_FLAGS[ 'ISIS_SR_PFX_EXPLICIT_NULL' ] )
   subModel.value = bool( prefixOptions &
         IsisCliHelper.PREFIX_SEGMENT_FLAGS[ 'ISIS_SR_PFX_VALUE' ] )
   subModel.local = bool( prefixOptions &
         IsisCliHelper.PREFIX_SEGMENT_FLAGS[ 'ISIS_SR_PFX_LOCAL' ] )
   
def getPrefixOptionsFlagsList( prefixOptions ):
   readvert = "R " if ( prefixOptions.readvert ) else ""
   nodeSID = "N " if ( prefixOptions.nodeSID ) else ""
   noPenultimate = "P " if ( prefixOptions.noPenultimate ) else ""
   explicitNull = "E " if ( prefixOptions.explicitNull ) else ""
   value = "V " if ( prefixOptions.value )  else ""
   local = "L " if ( prefixOptions.local ) else "" 
   return ( readvert + nodeSID + noPenultimate + explicitNull +
            value + local ).strip()

class SrPrefixReachability( Model ):
   algo = Str( help="SR Algo Used" )
   algoNum = Int( help="SR Algorithm" )
   sid = Int( help="SR Prefix SID" )
   options = Submodel( valueType=SrPrefixOptions,
                       help="SR Flags for Prefix options" )

   def processData( self, data ):
      from CliPlugin.RoutingIsisCli import getFlexAlgoName
      self.algoNum = ord( data.pop( 'prefix-algo' ) )
      self.algo = getFlexAlgoName( self.algoNum )
      self.sid = data.pop( 'prefix-sid' )
      options = ord( data.pop( 'prefix-option' ) )
      self.options = SrPrefixOptions()
      setPrefixOptions( self.options, options )
      
   def render( self ):
      flagsStr = getPrefixOptionsFlagsList( self.options )
     
      print( "        SR Prefix-SID: %s Flags: [%s] Algorithm: %d" % (
         self.sid, flagsStr, self.algoNum ) )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_PFX_SID

class RouteTag( Model ):
   tag = Int( help="Administrative Tag for IP Prefix" )
   def processData( self, data ):
      self.tag = data.pop( 'route-tag', None )

   def render( self ):
      if self.tag:
         print( "        Route Tag: %d" % self.tag )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_ROUTE_TAG

class Srv6SidStructure( Model ):
   blockLength = Int( help="Block length in bits" )
   nodeLength = Int( help="Node length in bits" )
   functionLength = Int( help="Function length in bits" )
   argumentLength = Int( help="Argument length in bits" )

   def render( self ):
      print( "          SID structure: Block length: %d "\
             "Node length: %d" % ( self.blockLength, self.nodeLength ) )
      print( "                         Function length: %d "\
            "Argument length: %d"  % ( self.functionLength, self.argumentLength ) )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_SRV6_SID_STRUCTURE_TLV

def setSrv6SidFlavors( flavorsModel, flavors ):
   flavorsModel.psp = bool( flavors &
             IsisCliHelper.SRV6SID_FLAVORS_MAP[ 'SRV6_FLAVOR_PSP' ] )
   flavorsModel.usp = bool( flavors &
             IsisCliHelper.SRV6SID_FLAVORS_MAP[ 'SRV6_FLAVOR_USP' ] )
   flavorsModel.usd = bool( flavors &
             IsisCliHelper.SRV6SID_FLAVORS_MAP[ 'SRV6_FLAVOR_USD' ] )
   flavorsModel.nextCsid = bool( flavors &
             IsisCliHelper.SRV6SID_FLAVORS_MAP[ 'SRV6_FLAVOR_NEXT_CSID' ] )
   flavorsModel.nextOnlyCsid = bool( flavors &
             IsisCliHelper.SRV6SID_FLAVORS_MAP[ 'SRV6_FLAVOR_NEXT_ONLY_CSID' ] )

class Srv6EndSid( Model ):
   sidValue = Ip6Address( help="End SID value" )
   flavors = Submodel( valueType=Srv6SidFlavorFlags, help="SID flavors" )
   _behaviorString = Str( help="SID behavior" )
   sidStructure = Submodel( valueType=Srv6SidStructure, help="SID structure",
                            optional=True )

   def processData( self, data ):
      self.sidValue = data.pop( 'sid-value' )
      self._behaviorString = data.pop( 'sid-behavior' )
      flavors = ord( data.pop( 'sid-flavors' ) )
      self.flavors = Srv6SidFlavorFlags()
      setSrv6SidFlavors( self.flavors, flavors )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_SRV6_END_SID_TLV

   def render( self ):
      print( "        SRv6 SID %s" % self._behaviorString )
      print( "          SID : %s" % self.sidValue )
      if self.sidStructure:
         self.sidStructure.render()
  
class Srv6LocatorFlags( Model ):
   down = Bool( help="Up/Down bit" )

   def toStr( self ):
      return "Down: {}".format( "set" if self.down else "unset" )

def setSrv6LocatorFlags( subModel, flags ):
   subModel.down = bool( flags &
             IsisCliHelper.SRV6LOCATOR_FLAGS_MAP[ 'ISIS_SRV6_LOCATOR_DOWN_FLAG' ] )

def getSrv6LocatorFlagsList( subModel ):
   down = "D" if ( subModel.down ) else ""
   return down

class Srv6Locators( Model ):
   locatorTopoId = Int( help="Locator Topology Id" )
   locatorPrefix = IpGenericPrefix( help="Locator prefix" )
   metric = Int( help="Cost of the Locator reachability" )
   algorithm = Int( help="Locator algorithm" )
   flags = Submodel( valueType=Srv6LocatorFlags, help="Locator flags" )
   endSids = GeneratorList( valueType=Srv6EndSid,
                            help="End SID information",
                            optional=True )

   def processData( self, data ):
      self.locatorTopoId = data.pop( 'locator-topo-id' )
      addr = data.pop( 'locator-addr' )
      if addr and '%' in addr:
         addr = addr[ : addr.index( '%' ) ]
      maskLength = data.pop( 'masklen' )
      self.locatorPrefix = Arnet.IpGenPrefix( "%s/%d" % ( addr, maskLength ) )
      self.metric = data.pop( 'metric' )
      self.algorithm = data.pop( 'algo' )
      flags = ord( data.pop( 'flags' ) )
      self.flags = Srv6LocatorFlags()
      setSrv6LocatorFlags( self.flags, flags )
      
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_SRV6_LOCATOR_TLV
   
   def render( self ):
      flags = getSrv6LocatorFlagsList( self.flags )
      print( "      SRv6 Locator: %s Topology: %d" %
            ( self.locatorPrefix, self.locatorTopoId ))
      print( "        Metric: %d Algorithm: %d Flags: [%s]" %
            ( self.metric, self.algorithm, flags ))
      for endSid in self.endSids:
         endSid.render()

class UnsupportedTlv( Model ):
   tlvType = Int( help="TLV type" )
   tlvLength = Int( help="TLV length" )

   def processData( self, data ):
      self.tlvType = data.pop( 'tlv-type', None )
      self.tlvLength = data.pop( 'tlv-length', None )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_UNSUPPORTED_TLV

   def render( self ):
      if self.tlvType is not None and self.tlvLength is not None:
         print( "      Unsupported TLV: Type: {} Length: {}".format( 
                                                               self.tlvType,
                                                               self.tlvLength ) )
   
class LspReachabilities( Model ):
   reachabilityV4Addr = Ip4Address( help="Reachability IPv4 address",
                                    optional=True )
   reachabilityV6Addr = Ip6Address( help="Reachability IPv6 address",
                                    optional=True )
   maskLength = Int( help="Mask Length" )
   metric = Int( help="Cost of the Adjacency" )
   metricType = Enum( values=IsisCliHelper.METRIC_TYPE_MAP.values(),
                      help="Metric Type " )
   reachabilityTopoId = Int( help="Reachable Topology Id" )
   reachabilityNarrowMetrics = Bool( help="Unsupported Reachability Narrow Metrics" )
   reachabilityUpDown = Bool( help="Reachability Status" )
   routeTags = GeneratorList( valueType=RouteTag, help="List of Route Tags",
                            optional=True )
   srPrefixReachabilities = GeneratorList( valueType=SrPrefixReachability,
                                  help="List of SR Prefix Reachability" )
   
   def processData( self, data ):
      self.reachabilityV4Addr = data.pop( 'ipv4-ifaddr', None )
      addr = data.pop( 'ipv6-ifaddr', None )
      if addr and '%' in addr:
         addr = addr[ : addr.index( '%' ) ]
      self.reachabilityV6Addr = addr
      self.maskLength = data.pop( 'masklen', None )
      self.metric = data.pop( 'metric', None )
      self.metricType = IsisCliHelper.METRIC_TYPE_MAP[ int( data.pop( \
            'metric-type' ) ) ]
      if 'reach-topo-id' in data:
         self.reachabilityTopoId = data.pop( 'reach-topo-id' )
      self.reachabilityNarrowMetrics = 'reach-narrow-metrics' in data
      self.reachabilityUpDown =  'reach-updown' in data
      
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_REACHABILITY
   
   def render( self ):
      reachUpDown = "Down" if self.reachabilityUpDown else "Up"
      if self.reachabilityV4Addr is not None:
         if self.reachabilityNarrowMetrics:
            fmt = "      Reachability (Narrow metrics, unsupported): %s/%s Metric:"
            fmt += " %s Type: %s"
            print( fmt % ( self.reachabilityV4Addr, self.maskLength,
                 self.metric, IsisCliHelper.METRIC_TYPE_MAP.reverse()[
                 self.metricType ] ) )
         else:
            fmt = "      Reachability         : %s/%s Metric: %s Type: %s %s"
            print( fmt % ( self.reachabilityV4Addr, self.maskLength,
                          self.metric, IsisCliHelper.METRIC_TYPE_MAP.reverse()[
                          self.metricType ], reachUpDown ) )

      if self.reachabilityV6Addr is not None:
         if self.reachabilityTopoId:
            if self.reachabilityTopoId == ADJ_MTID_MAP[ 'MTID_IPV6' ]:
               print( "      Reachability (MT-IPv6): %s/%s Metric: %s Type: %s %s" \
                      % ( self.reachabilityV6Addr, self.maskLength, self.metric,
                          IsisCliHelper.METRIC_TYPE_MAP.reverse()[ self.metricType ],
                          reachUpDown ) )
            else:
               print( "      Reachability (MT %s)  : %s/%s Metric: %s Type: %s %s" %
                  ( self.reachabilityTopoId, self.reachabilityV6Addr,
                    self.maskLength, self.metric,
                  IsisCliHelper.METRIC_TYPE_MAP.reverse()[ self.metricType ],
                  reachUpDown ) )
         else:
            print( "      Reachability          : %s/%s Metric: %s Type: %s %s" %
               ( self.reachabilityV6Addr, self.maskLength,
                 self.metric, IsisCliHelper.METRIC_TYPE_MAP.reverse()[
                 self.metricType ], reachUpDown ) )

      for routeTag in self.routeTags:
         routeTag.render()

      for srPrefixRechable in self.srPrefixReachabilities:
         srPrefixRechable.render()

class SrAlgo( Model ):
   srAlgo = Str( help="SR Algo Used" )
   algoNum = Int( help="SR Algorithm" )

   def processData( self, data ):
      from CliPlugin.RoutingIsisCli import getFlexAlgoName
      self.algoNum = ord( data.pop( 'sr algo' ) )
      self.srAlgo = getFlexAlgoName( self.algoNum )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_SR_ALGO

class SrCapabilitySrgb( Model ):
   srgbBase = Int( help="Srgb base" )
   srgbRange = Int( help="Srgb range" )

   def processData( self, data ):
      self.srgbBase = data.pop( 'srgb-base', None )
      self.srgbRange = data.pop( 'srgb-range', None )
      
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_SR_CAP_SRGB

   def render( self ):
      out = ""
      if self.srgbBase:
         out = "          SRGB Base: %s" % ( self.srgbBase )
      if self.srgbRange:   
         out += " Range: %s" % ( self.srgbRange )
      if out:
         print( out )
     
class SrCapabilityFlags( Model ):
   mplsV4 = Bool( help="Sr Capability mpls V4 Flag" )
   mplsV6 = Bool( help="Sr capability mpls V6 Flag" )
   srcapV6 = Bool( help="Sr Capability V6 Flag" )

def setSrCapabilityFlags( subModel, flags ):
   subModel.mplsV4 = bool( flags &
         IsisCliHelper.SRCAPABILITY_MAP[ 'ISIS_SR_CAP_MPLS_IPV4' ] )
   subModel.mplsV6 = bool( flags &
         IsisCliHelper.SRCAPABILITY_MAP[ 'ISIS_SR_CAP_MPLS_IPV6' ] )
   subModel.srcapV6 = bool( flags &
         IsisCliHelper.SRCAPABILITY_MAP[ 'ISIS_SR_CAP_IPV6' ] )

def getSrCapabilityFlagsList( subModel ):
   mplsV4 = "I " if ( subModel.mplsV4 ) else ""
   mplsV6 = "V " if ( subModel.mplsV6 ) else ""
   srcapV6 = "H " if ( subModel.srcapV6 ) else ""
   return ( mplsV4 + mplsV6 + srcapV6 ).strip()

class SrCapabilityModel( Model ):
   flags = Submodel( valueType=SrCapabilityFlags, help="SR Capability Flags" )
   srCapabilitySrgb = GeneratorList( valueType=SrCapabilitySrgb,
                            help="Segment Routing Global Block Capability Info" )
   
   def processData( self, data ):
      flags = ord( data.pop( 'flags' ) )
      self.flags = SrCapabilityFlags()
      setSrCapabilityFlags( self.flags, flags )
      
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_SR_CAP

   def render( self ):
      flags = getSrCapabilityFlagsList( self.flags )
      print( "        SR Capability: Flags: [%s]" % ( flags ) )
      for srgbCap in self.srCapabilitySrgb:
         srgbCap.render()

class SrlbRange( Model ):
   srlbBase = Int( help="Minimum label in range" )
   srlbRange = Int( help="Length of label range" )

   def processData( self, data ):
      self.srlbBase = data.pop( "srlb-base", None )
      self.srlbRange = data.pop( "srlb-range", None )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_SR_LB_RANGE

   def render( self ):
      print( ( " " * 10 ) + "SRLB Base: {} Range: {}".format( self.srlbBase,
                                                              self.srlbRange ) )

class SrlbModel( Model ):
   # no flags defined as of draft-ietf-isis-segment-routing-extensions-15
   srlbRanges = List( valueType=SrlbRange,
                      help="Ranges of labels making up SR Local Block" )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_SR_LB

   def render( self ):
      print( "        SR Local Block:" )
      for srlbRange in self.srlbRanges:
         srlbRange.render()

class AreaLeaderModel( Model ):
   prio = Int( help="Area leader priority" )
   algo = Int( help="Area leader algorithm" )

   def processData( self, data ):
      self.prio = ord( data.pop( "prio", chr( 0 ) ) )
      self.algo = ord( data.pop( "algo", chr( 0 ) ) )

   def render( self ):
      print( "        Area leader priority:", self.prio, "algorithm:", self.algo )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_AREA_LEADER

class DynamicFloodingModel( Model ):
   enabled = Bool( default=False, help="Dynamic flooding enabled" )

   def processData( self, data ):
      self.enabled = ord( data.pop( 'dynamic-flooding', '0' ) ) != 0

   def render( self ):
      if self.enabled:
         print( "        Dynamic Flooding: Enabled" )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_DYNAMIC_FLOODING

class NodeMsdModel( Model ):
   baseMplsImposition = Int( help="Base MPLS imposition", optional=True )

   def processData( self, data ):
      self.baseMplsImposition = data.pop( "msd-base-mpls-imposition", None )

   def render( self ):
      if self.baseMplsImposition is not None:
         print( "        Maximum SID depth:" )
         print( "          Base MPLS imposition (MSD type 1): ",
                 self.baseMplsImposition )
   
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_NODE_MSD

class Srv6CapabilityFlags( Model ):
   oamSupport = Bool( help="SRv6 OAM capability flag" )

def setSrv6CapabilityFlags( subModel, flags ):
   subModel.oamSupport = bool( flags &
      IsisCliHelper.SRV6CAPABILITY_FLAGS_MAP[ 'ISIS_SRV6_CAP_OAM_FLAG' ] )

def getSrv6CapabilityFlagsList( subModel ):
   oamSupport = "O" if ( subModel.oamSupport ) else ""
   return oamSupport

class Srv6CapabilityModel( Model ):
   flags = Submodel( valueType=Srv6CapabilityFlags, help="SRv6 Capability flags" )
   
   def processData( self, data ):
      flags = data.pop( 'flags' )
      self.flags = Srv6CapabilityFlags()
      setSrv6CapabilityFlags( self.flags, flags )
      
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_SRV6_CAP

   def render( self ):
      flags = getSrv6CapabilityFlagsList( self.flags )
      print( "        SRv6 Capability: Flags: [%s]" % ( flags ) )

class FlexAlgoExcludeModel( Model ):
   group = Str( help="Administrative Groups" )
   _index = Int( help="Administrative Groups Index" )

   def processData( self, data ):
      val = data.pop( 'group', None )
      self._index = data.pop( 'index', 0 )
      if val:
         self.group = bitMaskToStr( val )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_FLEX_ALGO_EXCLUDE_ADMIN_GROUP

class FlexAlgoIncludeAnyModel( Model ):
   group = Str( help="Administrative Groups" )
   _index = Int( help="Administrative Groups Index" )

   def processData( self, data ):
      val = data.pop( 'group', None )
      self._index = data.pop( 'index', 0 )
      if val:
         self.group = bitMaskToStr( val )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_FLEX_ALGO_INCLUDE_ANY_ADMIN_GROUP

class FlexAlgoIncludeAllModel( Model ):
   group = Str( help="Administrative Groups" )
   _index = Int( help="Administrative Groups Index" )

   def processData( self, data ):
      val = data.pop( 'group', None )
      self._index = data.pop( 'index', 0 )
      if val:
         self.group = bitMaskToStr( val )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_FLEX_ALGO_INCLUDE_ALL_ADMIN_GROUP

class FlexAlgoFlagsModel( Model ):
   flags = Str( help="Flags" )

   def processData( self, data ):
      bytestr = data.pop( "flags" )
      if bytestr:
         # pylint: disable-next=unnecessary-comprehension
         octets = [ octet for octet in bytestr ]
         self.flags = ' '.join( "0x%x" % octet for octet in octets )
      else:
         self.flags = None

   def render( self ):
      if self.flags:
         octets = [ int( octet, 16 ) for octet in self.flags.split() ]
         mFlag = octets[ 0 ] & 0x80
         print( '          Flags: [%s]' % ( "M" if mFlag else "" ), self.flags )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_FLEX_ALGO_FLAGS

class FlexAlgoExcludeSrlgModel( Model ):
   group = Int( help="Shared Risk Link Group" )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_FLEX_ALGO_EXCLUDE_SRLG

class FlexAlgoDefModel( Model ):
   algorithm = Int( help="Algorithm identifier" )
   metricType = Enum( values=( 'IGP Metric', 'Min Unidirectional Delay Metric',
                               'TE Metric', 'Undefined' ), help="Metric type" )
   metricTypeValue = Int( help="Metric type value" )
   calcType = Enum( values=( 'SPF', 'SSPF', 'Undefined' ), help="Calculation type" )
   calcTypeValue = Int( help="Calculation type value" )
   prio = Int( help="Priority" )
   excludeGroups = GeneratorList( valueType=FlexAlgoExcludeModel,
                                  help="Flexible Algorithm Exclude Admin Group" )
   excludeGroupsDecimal = List( valueType=int,
                                help="Flexible Algorithm Exclude Admin Group" )
   includeAllGroups = GeneratorList( valueType=FlexAlgoIncludeAllModel,
                                     help=
                                     "Flexible Algorithm Include All Admin Group" )
   includeAllGroupsDecimal = List( valueType=int,
                                   help="Flexible Algorithm Include All Admin "
                                   "Group" )
   includeAnyGroups = GeneratorList( valueType=FlexAlgoIncludeAnyModel,
                                     help=
                                     "Flexible Algorithm Include Any Admin Group" )
   includeAnyGroupsDecimal = List( valueType=int,
                                   help="Flexible Algorithm Include Any Admin "
                                   "Group" )
   flags = GeneratorList( valueType=FlexAlgoFlagsModel,
                          help="Flexible Algorithm Flags" )
   excludeSrlgGroups = GeneratorList( valueType=FlexAlgoExcludeSrlgModel,
                                      help="Flexible Algorithm Exclude "
                                      "Shared Risk Link Group" )

   def processData( self, data ):
      self.algorithm = ord( data.pop( "algorithm", chr( 0 ) ) )
      self.metricTypeValue = ord( data.pop( "metric-type", chr( 0 ) ) )
      self.calcTypeValue = ord( data.pop( "calc-type", chr( 0 ) ) )
      self.prio = ord( data.pop( "priority", chr( 0 ) ) )
      self.metricType = IsisCliHelper.FLEX_ALGO_METRIC_MAP.get(
         self.metricTypeValue, 'Undefined' )
      self.calcType = IsisCliHelper.FLEX_ALGO_CALC_MAP.get( self.calcTypeValue,
                                                            'Undefined' )
      self.excludeGroupsDecimal = \
            [ int( byte ) for byte in data.pop( 'exclude', [] ) ]
      self.includeAllGroupsDecimal = \
            [ int( byte ) for byte in data.pop( 'include-all', [] ) ]
      self.includeAnyGroupsDecimal = \
            [ int( byte ) for byte in data.pop( 'include-any', [] ) ]

   def render( self ):
      print( "        Flex Algo: Algorithm: %u Metric: %s (%u) Calc: %s (%u) "
              "Prio: %u" %
              ( self.algorithm, self.metricType, self.metricTypeValue, self.calcType,
                self.calcTypeValue, self.prio ) )

      def renderList( title, groups, attr, fmt, delim ):
         if not groups:
            return
         values = sorted( getattr( group, attr ) for group in groups )
         if not values:
            return
         print( '          %s:' % title, delim.join( [ fmt % v
                                                      for v in sorted( values ) ] ) )

      def renderAgList( title, groups, attr ):
         if not groups:
            return
         values = { getattr( group, '_index' ):
                    strToBitMask( getattr( group, 'group' ) ) for group in groups }
         values = bitMaskCollToStr( values )
         if not values:
            return
         print( '          %s:' % title, values )

      renderAgList( 'Exclude admin groups', self.excludeGroups, 'group' )
      renderAgList( 'Include all admin groups', self.includeAllGroups, 'group' )
      renderAgList( 'Include any admin groups', self.includeAnyGroups, 'group' )

      # Flags are a wee bit different, as we try to interpret them.
      for flags in self.flags:
         flags.render()

      renderList( 'Exclude shared risk link groups', self.excludeSrlgGroups, 'group',
                  '%u', ', ' )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_FLEX_ALGO_DEF

class RouterCapabilityFlags( Model ):
   rtrCapS = Bool( help="Router Capability S Flag" )
   rtrCapD = Bool( help="Router Capability D Flag" )
   
def setRouterCapabilityFlags( subModel, flags ):
   subModel.rtrCapS = bool(
        IsisCliHelper.RTR_CAPABILITY_MAP[ 'ISIS_RTR_CAP_FLAG_S' ] & flags )
   subModel.rtrCapD = bool(
         IsisCliHelper.RTR_CAPABILITY_MAP[ 'ISIS_RTR_CAP_FLAG_D' ] & flags )

def getRouterCapabilityFlagsList( subModel ):
   rtrCapS = "S " if ( subModel.rtrCapS ) else ""
   rtrCapD = "D " if ( subModel.rtrCapD ) else ""
   return ( rtrCapS + rtrCapD ).strip()

class RouterCapability( Model ):
   routerId = Str( help="Router Id" )
   flags = Submodel( valueType=RouterCapabilityFlags,
                     help="Router Capability Flags" )
   srlb = Submodel( valueType=SrlbModel, optional=True,
                    help="SR Local Block Information" )
   areaLeader = Submodel( valueType=AreaLeaderModel, optional=True,
                          help="Area Leader" )
   dynamicFlooding = Submodel( valueType=DynamicFloodingModel, optional=True,
                               help="Dynamic Flooding" )
   msd = Submodel( valueType=NodeMsdModel, optional=True,
                   help="Node maximum SID depth" )
   srv6Capabilities = Submodel( valueType=Srv6CapabilityModel, optional=True,
                                help="SRv6 capabilities" )
   # beware, RibCapiLib populates submodels right after populating the
   # parent model, see also isis_dget_lsdb_caps_write
   # Put only Submodels above
   srCapabilities = GeneratorList( valueType=SrCapabilityModel,
                                   help="SR Capabilities" )
   srAlgos = GeneratorList( valueType=SrAlgo, help="Algorithm used" )
   flexAlgoDefs = GeneratorList( valueType=FlexAlgoDefModel,
                                 help="Flexible Algorithm Definitions" )

   def processData( self, data ):
      self.routerId = data.pop( 'router-id' )
      flags = ord( data.pop( 'flags' ) )
      self.flags = RouterCapabilityFlags()
      setRouterCapabilityFlags( self.flags, flags )
      
   def render( self ):
      flags = getRouterCapabilityFlagsList( self.flags )
      print( "      Router Capabilities: Router Id: {} Flags: [{}]".format(
         self.routerId, flags ) )
      for attr in ( 'srlb', 'areaLeader', 'dynamicFlooding',
                    'msd', 'srv6Capabilities' ):
         obj = getattr( self, attr )
         if obj:
            obj.render()

      for cap in self.srCapabilities:
         cap.render()

      values = ', '.join( str( algo.algoNum ) for algo in self.srAlgos )
      if values:
         print( "        Algorithms: ", values )

      for defin in self.flexAlgoDefs:
         defin.render()

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_RTR_CAPABILITY


def setSrBindingFlag( subModel, flag ):
   subModel.attach = bool( flag &
         IsisCliHelper.SR_BINIDING_MAP[ 'ISIS_SR_BINDING_ATTACHED' ] )
   subModel.leak = bool( flag &
         IsisCliHelper.SR_BINIDING_MAP[ 'ISIS_SR_BINDING_LEAK' ] )
   subModel.flood = bool( flag &
         IsisCliHelper.SR_BINIDING_MAP[ 'ISIS_SR_BINDING_FLOOD' ] )
   subModel.mirror = bool( flag &
         IsisCliHelper.SR_BINIDING_MAP[ 'ISIS_SR_BINDING_MIRROR' ] )
   subModel.addrFamily = bool( flag &
         IsisCliHelper.SR_BINIDING_MAP[ 'ISIS_SR_BINDING_AF' ] )

def getSrBindingFlagList( flags ):
   attach = "A " if ( flags.attach ) else ""
   leak = "D " if ( flags.leak ) else ""
   flood = "S " if ( flags.flood ) else ""
   mirror = "M " if ( flags.mirror ) else ""
   addrFamily = "F " if ( flags.addrFamily ) else ""
   return ( attach + leak + flood + mirror + addrFamily ).strip()

class SrBindingFlags( Model ):
   attach = Bool( help="Sr Binding Attach Flags" )
   leak = Bool( help="Sr Binding leak Flag" )
   flood = Bool( help="Sr Binding flood Flag" )
   mirror = Bool( help="Sr Binding Mirror Flag" )
   addrFamily = Bool( help="Sr Binding Address Family Flag" )
   
class SrBindingModel( Model ):
   flags = Submodel( valueType=SrBindingFlags, help="SR Binding Flags" )
   label = Int( help="SR Binding Label" )
   weight = Int( help="SR Binding Weight of the path" )
   prefix = IpGenericAddress( help="SR Binding Ip Prefix" )
   maskLength = Int( help="SR Binding Mask Length" )
   rangeValue = Int( help="SR Binding Range" )
   srBindingTopologyId = Int( help="Sr Binding Topology Id" )
   srPrefixReachabilities = GeneratorList( valueType=SrPrefixReachability,
                                  help="List of SR Reachable Prefixes" )
   
   def processData( self, data ):
      flags = ord( data.pop( 'flag' ) )
      self.flags = SrBindingFlags()
      setSrBindingFlag( self.flags, flags )
      self.weight = ord( data.pop( 'weight' ) ) 
      self.prefix = data.pop( 'prefix' )
      self.maskLength = data.pop( 'mask' )
      self.rangeValue = data.pop( 'range' )
      self.srBindingTopologyId = data.pop( 'mtid', None )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_SR_BINDING
  
   def render( self ):
      mtstr = ""
      if self.srBindingTopologyId is not None:
         if self.srBindingTopologyId == ADJ_MTID_MAP[ 'MTID_IPV6' ]:
            mtstr = " (MT-IPv6)"
         else:
            mtstr = " (MT %s)" % self.srBindingTopologyId
         
      flagsStr = getSrBindingFlagList( self.flags )
      outputFmt = "      Segment Binding%s: Flags: [%s] Weight: %d"
      outputFmt += " Range: %s Pfx %s/%s"
      print( outputFmt % ( mtstr, flagsStr, self.weight, self.rangeValue,
                          self.prefix, self.maskLength ) )

      for srPrefixReachable in self.srPrefixReachabilities:
         srPrefixReachable.render()

class SrlgIds( Model ):
   gid = Int( help="Shared Risk Link Group Identifier" )

   def processData( self, data ):
      self.gid = data.pop( 'srlg-gid' )
      return data

   def render( self ):
      print( "        Group ID: %s" % self.gid )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_TE_SRLG_GID

class TeSrlgBase( Model ):
   ngbSystemId = Str( help="System identifier of the neighbor" )
   srlgIds = GeneratorList( valueType=SrlgIds,
                            help="List of Shared Risk Link Group IDs" )

   def processData( self, data ):
      self.ngbSystemId = data.pop( "system-id" )
      return data

   def render( self ):
      for srlgid in self.srlgIds:
         srlgid.render()

class TeSrlg( TeSrlgBase ):
   flags = Enum( values=( 'Numbered', 'Unnumbered' ), help="SRLG flags" )
   intfAddr = IpGenericAddress( help="Interface address" )
   ngbAddr = IpGenericAddress( help="Neighbor address" )
   localLinkId = Int( help='Local link identifier', optional=True )
   remoteLinkId = Int( help='Remote link identifier', optional=True )

   def processData( self, data ):
      data = TeSrlgBase.processData( self, data )
      self.flags = IsisCliHelper.INTF_SRLG_FLAG_MAP[ ord( data.pop( "flags" ) ) ]
      self.intfAddr = data.pop( "intf-addr", None )
      self.ngbAddr = data.pop( "nbr-addr", None )
      self.localLinkId = data.pop( "local-id", None )
      self.remoteLinkId = data.pop( "remote-id", None )


   def render( self ):
      if self.ngbSystemId:
         srlgHeader = "      Shared Risk Link Group:  Neighbor %s (%s)"
         print( srlgHeader % ( self.ngbSystemId, self.flags ) )
         if self.intfAddr:
            print( "        Interface Address: %s" % self.intfAddr )
         if self.ngbAddr:
            print( "        Neighbor Address: %s" % self.ngbAddr )
         if self.localLinkId:
            print( "        Local link ID: %d" % self.localLinkId )
         if self.remoteLinkId:
            print( "        Remote link ID: %d" % self.remoteLinkId )

         TeSrlgBase.render( self )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_TE_IPV4_SRLG

class v6SrlgFlags( Model ):
   neighborAddressIncluded = Bool( default=False,
                                   help='IPv6 neighbor address is '
                                   'included (NA bit)' )

   def processData( self, data ):
      self.neighborAddressIncluded = ord( data.pop( 'flags' ) ) != 0

class TeIpv6Srlg( TeSrlgBase ):
   flags = Submodel( valueType=v6SrlgFlags, help='IPv6 SRLG flags' )
   v6InterfaceAddress = Ip6Address( help='IPv6 interface address' )
   v6NeighborAddress = Ip6Address( optional=True, help='IPv6 neighbor address' )

   def processData( self, data ):
      data = TeSrlgBase.processData( self, data )
      flags = v6SrlgFlags()
      flags.processData( data )
      self.flags = flags
      self.v6InterfaceAddress = data.pop( 'intf-addr' )
      self.v6NeighborAddress = data.pop( 'nbr-addr', None )

   def render( self ):
      if self.ngbSystemId:
         srlgHeader = "      IPv6 Shared Risk Link Group:  Neighbor %s"
         print( srlgHeader % ( self.ngbSystemId ) )
         print( "        Interface Address: %s" % self.v6InterfaceAddress )
         if self.v6NeighborAddress is not None:
            print( "        Neighbor Address: %s" % self.v6NeighborAddress )
         TeSrlgBase.render( self )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_TE_IPV6_SRLG

class ApplicationSrlg( TeSrlgBase ):
   localLinkId = Int( help='Local link identifier', optional=True )
   remoteLinkId = Int( help='Remote link identifier', optional=True )
   interfaceAddress = IpGenericAddress( help='Interface address', optional=True )
   neighborAddress = IpGenericAddress( help='Neighbor address', optional=True )
   v6InterfaceAddress = Ip6Address( help='IPv6 interface address', optional=True )
   v6NeighborAddress = Ip6Address( help='IPv6 neighbor address', optional=True )
   applications = Submodel( valueType=ApplicationIdentifiers,
                            help='Application identifiers' )

   def processData( self, data ):
      TeSrlgBase.processData( self, data )
      self.localLinkId = data.pop( "local-id", None )
      self.remoteLinkId = data.pop( "remote-id", None )
      self.interfaceAddress = data.pop( "intf-addr", None )
      self.neighborAddress = data.pop( "nbr-addr", None )
      self.v6InterfaceAddress = data.pop( 'intf-v6-addr', None )
      self.v6NeighborAddress = data.pop( 'nbr-v6-addr', None )

   def render( self ):
      indent = ' ' * 6
      if self.ngbSystemId:
         print( '{}Application Specific Shared Risk Link Group:  Neighbor {}'.format(
            indent, self.ngbSystemId ) )
         indent += ' ' * 2
         self.applications.renderHelper( indent=indent )
         if self.localLinkId is not None:
            print( f"{indent}Local link identifier: {self.localLinkId}" )
         if self.remoteLinkId is not None:
            print( f"{indent}Remote link identifier: {self.remoteLinkId}" )
         if self.interfaceAddress:
            print( "{}IPv4 interface address: {}".format( indent,
      self.interfaceAddress ) )
         if self.neighborAddress:
            print( "{}IPv4 neighbor address: {}".format( indent,
      self.neighborAddress ) )
         if self.v6InterfaceAddress is not None:
            print( "{}IPv6 interface address: {}".format( indent,
                                                     self.v6InterfaceAddress ) )
         if self.v6NeighborAddress is not None:
            print( "{}IPv6 neighbor address: {}".format( indent,
      self.v6NeighborAddress ) )
         TeSrlgBase.render( self )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_TE_APP_SRLG

class LspDynFloodNodeId( Model ):
   startIndex = Int( help="Start index of enclosed systems" )
   endIndex = Int( help="End index of enclosed systems" )

   def processData( self, data ):
      self.startIndex = data.pop( 'start-index' )
      self.endIndex = data.pop( 'end-index' )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_DYNFLOOD_NODEID_BRIEF

class LspDynFloodPath( Model ):
   path = Str( help="Path of node indices" )

   def processData( self, data ):
      pathBytes = data.pop( 'path' )
      if isinstance( pathBytes, str ):
         pathBytes = pathBytes.encode()

      pathNode = ( struct.unpack( "!H", pathBytes[ i: i + 2 ] )
                   for i in range( 0, len( pathBytes ), 2 ) )
      self.path = " ".join( "%d" % i for i in pathNode )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_DYNFLOOD_PATH_TLV

   def render( self ):
      if self.path:
         output = "      Dynamic Flooding Path: " + self.path
         while len( output ) > LINE_MAX:
            index = output.rfind( " ", 0, LINE_MAX )
            print( output[ 0 : index ] )
            output = ' ' * 8 + output[ index + 1 : ]
         if len ( output ) > 8:
            print( output )

class SupportedInstTlv( Model ):
   instanceId = Int( help="IS-IS Instance Id" )
   supportedInstances = List( valueType=int, help="List of supported instances" )

   def processData( self, data ):
      instances = data.pop( 'supported_inst' )
      if isinstance( instances, str ):
         instances = instances.encode()
      # Convert every two bytes from the byte string to integers.
      self.supportedInstances = [ struct.unpack( '!H', instances[ i:i + 2 ] )[ 0 ]
                                  for i in range( 0, len( instances ), 2 ) ]
      # First number is the instance ID.
      self.instanceId = self.supportedInstances.pop( 0 )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_INSTANCE_ID_TLV

   def render( self ):
      if self.supportedInstances:
         output = "Instance ID: " + str( self.instanceId ) + \
                  " supported_instance_ids: " + \
                  ' '.join( map( str, self.supportedInstances ) )
         lines = textwrap.wrap( output, width=LINE_MAX, initial_indent=' ' * 6,
                                subsequent_indent=' ' * 8 )
         for line in lines:
            print( line )

class AreaSegmentFlags( Model ):
   areaSegAf = Bool( help="Area segment address family flag" )
   areaSegValue = Bool( help="Area segment value flag" )
   areaSegLocal = Bool( help="Area segment local flag" )

   def set( self, charVal ):
      self.areaSegAf = False
      self.areaSegValue = False
      self.areaSegLocal = False
      if charVal:
         intVal = ord( charVal )
         asf = IsisCliHelper.AREA_SEGMENT_FLAGS
         self.areaSegAf = ( asf[ 'ISIS_SR_AREA_SEGMENT_AF' ] & intVal ) != 0
         self.areaSegValue = ( asf[ 'ISIS_SR_AREA_SEGMENT_VALUE' ] & intVal ) != 0
         self.areaSegLocal = ( asf[ 'ISIS_SR_AREA_SEGMENT_LOCAL' ] & intVal ) != 0

   def render( self ):
      return '{}{}{}'.format(
         'F ' if self.areaSegAf else '',
         'V ' if self.areaSegValue else '',
         'L ' if self.areaSegLocal else '' )

class AreaProxySubTlv( Model ):
   areaProxySystemId = Str( help="Area proxy system ID", optional=True )
   flags = Submodel( valueType=AreaSegmentFlags, help="Area segment flags",
                     optional=True )
   index = Int( help="Area segment index/label", optional=True )
   prefix = IpGenericAddress( help="Area segment IP Prefix", optional=True )
   maskLength = Int( help="Area segment IP Prefix Mask Length", optional=True )

   def processData( self, data ):
      self.areaProxySystemId = data.pop( 'area-proxy-sysid', None )
      if 'area-segment-flags' in data:
         self.flags = AreaSegmentFlags()
         self.flags.set( data.pop( 'area-segment-flags' ) )
      self.index = data.pop( 'area-segment-index', None )
      self.prefix = data.pop( 'area-segment-addr', None )
      if 'area-segment-mask-len' in data:
         self.maskLength = ord( data.pop( 'area-segment-mask-len' ) )

   def render( self ):
      if self.areaProxySystemId:
         print( "        System ID: %s" % self.areaProxySystemId )
      if self.index:
         print( "        Area Segment ID: %d flags: [%s] prefix: %s/%s" % (
            self.index, self.flags.render().strip(),
                  self.prefix, self.maskLength ) )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_AREA_PROXY_SUBTLV

class LspBaseModel( Model ):
   expiryTime = Float( help="UTC time for Future Expiry time of LSP" )
   _lspid = Str( help="LSPID for this LSP" )
   sequence = Int( help="Sequence number for this LSP" )
   checksum = Int( help="Checksum value" )
   length = Int( help="LSP length" )
   intermediateSystemType = Enum( values=( 'none', 'L1', 'L2' ),
                  optional=True, help="IS-type for LSP" )
   flags = Submodel( valueType=LspFlags, optional=True,
                     help="IS-IS Lsp flags" )
   nodeUnreachable = Bool( default=False,
                     help="Flag for LSPs originated by Unreachable/Stale Nodes" )
   hostnameConflict = Bool( default=False,
                            help="LSP originated by node "
                            "with conflicting hostname" )

   def processData( self, data ):
      self.expiryTime = data.pop( 'lifetime' ) + Tac.utcNow()
      self._lspid = data.pop( 'lsid' )
      self.sequence = data.pop( 'sequence' )
      self.checksum = data.pop( 'checksum' )
      self.length = data.pop( 'length' )
      intermediateSystemType = data.pop( 'is-type' )
      self.intermediateSystemType = 'none'
      if intermediateSystemType == 1:
         self.intermediateSystemType = 'L1'
      if intermediateSystemType == 2:
         self.intermediateSystemType = 'L2'
      flags = LspFlags()
      reversedFlagsDict = IsisCliHelper.ISIS_LSP_FLAGS_MAP.reverse()
      if 'flags' in data and data[ 'flags' ]:
         flagList = data.pop( 'flags' ).split()
         for flag in flagList:
            setattr( flags, reversedFlagsDict[ flag ], True )
      self.flags = flags
      self.nodeUnreachable = ( ord( data.pop( 'node-unreachable' ) ) != 0 )
      self.hostnameConflict = ( ord( data.pop( 'hname-conflict' ) ) != 0 )
      return data
   
   def renderData( self, vrf=None, instanceId=0 ):
      def getLspInfoCodes():
         codes = ''
         if self.hostnameConflict:
            codes += 'H'
         if self.nodeUnreachable:
            codes += 'U'
         return codes

      from CliPlugin.RoutingIsisCli import getHostName
      showFormat = " %-2s %-21s %9s %6s %5s %6s %2s  %-21s <%s>"
      hostname = getHostName( getRouterId( self._lspid[ 0:14 ], instanceId ),
                                           vrfName=vrf )
      lspidRender = self._lspid
      if hostname:
         lspidRender = hostname + self._lspid[ -6: ]
      lspRemainingLife = int( self.expiryTime - Tac.utcNow() )
      if lspRemainingLife < 0: # pylint: disable=consider-using-max-builtin
         lspRemainingLife = 0
      print( showFormat % ( getLspInfoCodes(), lspidRender,
                           self.sequence, self.checksum,
                           lspRemainingLife,
                           self.length,
                           self.intermediateSystemType,
                           self._lspid,
                           getFlagsString( self.flags ) ) )
      
   def getKey( self, data ):
      return data[ 'lsid' ]

class LspDetailTlvModel( LspBaseModel ):
   refreshTime = Float( help="Future refresh time of LSP in UTC",
                        optional=True )
   lspGenerationTime = Float( help="Next LSP generation time in UTC",
                              optional=True )
   rcvdLifetime = Int( help="Received remaining lifetime of an LSP ( in seconds )",
                       optional=True )
   modifiedLifetime = Int( help="Modified remaining lifetime of an LSP "
                           "( in seconds )", optional=True )
   
class LspDetailModel( LspBaseModel ):
   addrFamily = Str( help="Address Family", optional=True )
   refreshTime = Float( help="Future refresh time of LSP in UTC", 
                        optional=True )
   lspGenerationTime = Float( help="Next LSP generation time in UTC",
                      optional=True )
   rcvdLifetime = Int( help="Received remaining lifetime of an LSP ( in seconds )",
                        optional=True )
   modifiedLifetime = Int( help="Modified remaining lifetime of an LSP "\
                           "( in seconds )", optional=True )
   hostname = Submodel( valueType=LspHostname, help="IS-IS hostname",
                        optional=True )
   teRouterID = Submodel( valueType=TeRouterId, help="IS-IS TE Router-ID",
                          optional=True )
   auth = Submodel( valueType=LspAuthModel, help="Auth details for the LSP",
                       optional=True )

   areaAddresses = GeneratorList( valueType=LspAreaAddress,
                                  help="List of Area Addresses" )

   multiTopologies = GeneratorList( valueType=LspMtModel,
                           help="Multi Topology details for the LSP",
                           optional=True )

   interfaceAddresses = GeneratorList( valueType=InterfaceAddresses,
                              help="List of Interface Addresses",
                              optional=True )
   neighbors = GeneratorList( valueType=LspNeighbors,
                              help="List of LSP neighbors Details", optional=True )
   reachabilities = GeneratorList( valueType=LspReachabilities ,
                        help="Reachabilities Information of LSP",
                        optional=True )
   srv6Locators = GeneratorList( valueType=Srv6Locators,
                        help="SRv6 Locators information of LSP",
                        optional=True )
   areaProxySubTlvs = GeneratorList( valueType=AreaProxySubTlv,
                                     help="Area proxy information", optional=True )
   routerCapabilities = GeneratorList( valueType=RouterCapability,
                            help="Router Capability Information",
                            optional=True )
   srBindings = GeneratorList( valueType=SrBindingModel,
                         help="Segment Routing Binding Information",
                         optional=True )
   srlgInfos = GeneratorList( valueType=TeSrlg,
                              help="Traffic Engineering Shared Risk "
                              "Link Group information", optional=True )
   ip6SrlgInfos = GeneratorList( valueType=TeIpv6Srlg, optional=True,
                        help="Traffic Engineering IPv6 Shared "
                             "Risk Link Group information" )
   appSrlgInfos = GeneratorList( valueType=ApplicationSrlg, optional=True,
                                 help="Application Specific Shared "
                                 "Risk Link Group information" )
   _generationTimersSet = Bool( default=False ,
                                help="Generation Timers Set" + 
                                "for Self Originated Lsp" )
   dynFloodNodeId = GeneratorList( valueType=LspDynFloodNodeId,
                                   help="Dynamic Flooding Area Node IDs",
                                   optional=True )
   dynFloodPath = GeneratorList( valueType=LspDynFloodPath,
                                 help="Dynamic Flooding Paths",
                                 optional=True )
   purgeOriginator = Str( help="Purge originator", optional=True )
   purgeUpstreamNgb = Str( help="Purge originator upstream neighbor", optional=True )

   supportedInstTlvs = GeneratorList ( valueType=SupportedInstTlv,
                                       help="Supported instance TLV information",
                                       optional=True )

   unsupportedTlvs = GeneratorList( valueType=UnsupportedTlv,
                                    help="Unsupported IS-IS TLV Information",
                                    optional=True )

   def processData( self, data ):
      LspBaseModel.processData( self, data )
      remLspGenerationWaitTime = data.pop( 'rem-wait-time', None )
      if remLspGenerationWaitTime is not None:
         self._generationTimersSet = True
         if remLspGenerationWaitTime > 0:
            self.lspGenerationTime = Tac.utcNow() + float(
               remLspGenerationWaitTime ) / 1000 
      refreshTime = data.pop( 'refresh-time', None )
      if refreshTime is not None:
         self.refreshTime = refreshTime + Tac.utcNow()
      self.rcvdLifetime = data.pop( 'rcvd-lifetime', None )
      self.modifiedLifetime = data.pop( 'modified-lifetime', None )
      addrFamily = data.pop( 'addr-family', None )
      if addrFamily:
         self.addrFamily = ""
         for nlpId in IsisCliHelper.NLPID_MAP:
            if nlpId[ 1 ] & addrFamily:
               self.addrFamily += nlpId[ 0 ]
      self.purgeOriginator = data.pop( 'purge-originator', None )
      self.purgeUpstreamNgb = data.pop( 'purge-upstream-ngb', None )
      return data
   
   def getKey( self, data ):
      return LspBaseModel.getKey( self, data )
      
   def renderData( self, vrf=None, instanceId=0 ):
      LspBaseModel.renderData( self, vrf=vrf, instanceId=instanceId )
      lspGenRender = "      LSP generation remaining wait time: %s ms"
      if self.lspGenerationTime != None: # pylint: disable=singleton-comparison
         generationTime = max( int( ( self.lspGenerationTime - Tac.utcNow() ) \
                                    * 1000 ), 0 )
         print( lspGenRender % ( generationTime ) )
      else:
         if self._generationTimersSet is True:
            print( lspGenRender % ( 0 ) )
      lspRefreshRender = "      Time remaining until refresh: %s s"
      if self.refreshTime is not None:
         lspRemainingRefresh = max( int( self.refreshTime - Tac.utcNow() ), 0 )
         print( lspRefreshRender % lspRemainingRefresh )
      if self.rcvdLifetime is not None and self.modifiedLifetime is not None:
         lifetimeRender = "      Remaining lifetime received: %s s Modified to: %s s"
         print( lifetimeRender % ( self.rcvdLifetime, self.modifiedLifetime ) )
         
      if self.addrFamily:
         print( "      NLPID:%s" % ( self.addrFamily ) )
      if self.hostname:
         self.hostname.render()
      if self.auth:
         self.auth.render()

      if self.purgeOriginator:
         print( "      Purge originator: %s" % ( self.purgeOriginator ) )
      if self.purgeUpstreamNgb:
         print( "      Purge originator upstream neighbor: %s" %
                ( self.purgeUpstreamNgb ) )
      if self.teRouterID:
         self.teRouterID.render()
      addresses = []
      for address in self.areaAddresses:
         addresses.append( address.address )
      if addresses:
         if len( addresses ) == 1:
            print( "      Area addresses: %s" % addresses[ 0 ] )
         else:
            print( "      Area addresses:" )
            for address in natsort.natsorted( addresses ):
               print( "        %s" % address )
      for mtDetail in self.multiTopologies:
         mtDetail.render()
      for interface in self.interfaceAddresses:
         interface.render()
      for neighbor in self.neighbors:
         neighbor.render()
      for reachable in self.reachabilities:
         reachable.render()
      for srv6Locator in self.srv6Locators:
         srv6Locator.render()

      first = True
      for info in self.areaProxySubTlvs:
         if first:
            print( "      Area Proxy:" )
            first = False
         info.render()

      for caps in self.routerCapabilities:
         caps.render()
      for binding in self.srBindings:
         binding.render()
      for srlginfo in self.srlgInfos:
         srlginfo.render()
      for ip6Srlginfo in self.ip6SrlgInfos:
         ip6Srlginfo.render()
      for appSrlgInfo in self.appSrlgInfos:
         appSrlgInfo.render()
      # To save space, format the output of dynamic flooding system IDs here.
      # It shows as
      #       Dynamic Flooding Area Node IDs: n1-n2, n3-n4,
      #         n5-n6, n7-n8, ...
      # One range of indices for each TLV.
      bufPrev = None
      bufPrevLen = 0
      header = ' ' * 6 + 'Dynamic Flooding Area Node IDs:'
      if self.dynFloodNodeId:
         bufPrev = header
         bufPrevLen = len( bufPrev )

      for nodeId in sorted( self.dynFloodNodeId, key=attrgetter( 'startIndex' ) ):
         buf = ' %d-%d,' % ( nodeId.startIndex, nodeId.endIndex )
         bufLen = len( buf )
         if bufLen + bufPrevLen <= LINE_MAX:
            bufPrev += buf
            bufPrevLen += bufLen
         else:
            # If the current node id output does not fit, print out the bufPrev
            # and start a new line
            print( bufPrev )
            bufPrev = ' ' * 8 + buf
            bufPrevLen = len( bufPrev )

      if bufPrev != header:
         # get rid of the "," in the end
         print( bufPrev[ : -1 ] )

      for path in self.dynFloodPath:
         path.render()

      for instance in self.supportedInstTlvs:
         instance.render()

      # unsupported TLVs always rendered at the end.
      for unsupportedTlv in self.unsupportedTlvs:
         unsupportedTlv.render()

class LevelBaseModel( Model ):
   _level = Int( help='Lsdb Level' )
   
   def getKey( self, data ):
      return int( data[ 'level-num' ] )
   
   def processData( self, data ):
      self._level = data.pop( 'level-num' )
      return data
   
   def renderData( self, vrf=None, instanceId=0 ):
      print( "  IS-IS Level %d Link State Database" % ( self._level ) )
      print( " %-2s %-21s %9s %6s %5s %6s %2s  %-21s %5s" %
             ( "", "LSPID", "Seq Num", "Cksum", "Life", "Length", "IS",
                "Received LSPID", "Flags" ) )
      for _,lsp in self.lsps:
         lsp.renderData( vrf=vrf, instanceId=instanceId )
         
class LevelDetailModel( LevelBaseModel ):
   lsps = GeneratorDict( keyType=str, valueType=LspDetailModel,
                help='A mapping of systemId to LSP in LSDB' )
      
   def processData( self, data ):
      LevelBaseModel.processData( self, data )
      
   def renderData( self, vrf=None, instanceId=0 ):
      LevelBaseModel.renderData( self, vrf=vrf, instanceId=instanceId )
   def getKey( self, data ):
      return LevelBaseModel.getKey( self, data )

class LevelSummaryModel( LevelBaseModel ):
   lsps = GeneratorDict( keyType=str, valueType=LspBaseModel,
                help='A mapping of systemId to LSP in LSDB' )
      
   def processData( self, data ):
      LevelBaseModel.processData( self, data )
         
   def renderData( self, vrf=None, instanceId=0 ):
      LevelBaseModel.renderData( self, vrf=vrf, instanceId=instanceId )
         
   def getKey( self, data ):
      return LevelBaseModel.getKey( self, data )
   
class LevelDetailTlvModel( LevelBaseModel ):
   lsps = GeneratorDict( keyType=str, valueType=LspDetailTlvModel,
                         help='A mapping of systemId to LSP in LSDB' )

   def processData( self, data ):
      LevelBaseModel.processData( self, data )

   def renderData( self, vrf=None, instanceId=0 ):
      LevelBaseModel.renderData( self, vrf=vrf, instanceId=instanceId )

   def getKey( self, data ):
      return LevelBaseModel.getKey( self, data )

class IsisLsdbBaseModel( Model ):
   _instanceName = Str( help='ISIS Instance name' )
   _vrf = Str( help='VRF Name' )
   _instanceId = Int( help='IS-IS Instance ID' )
   
   def processData( self, data ):
      self._instanceName = data.pop( 'instance-name', None )
      self._vrf = data.pop( 'vrf-name' )
      self._instanceId = data.pop( 'instance-id' )
      
   def render( self ):
      print( "Legend:" )
      print( "H - hostname conflict" )
      print( "U - node unreachable" )
      print( "" )
      print( f"IS-IS Instance: {self._instanceName} VRF: {self._vrf}" )
      for _,levelEntry in self.level:
         levelEntry.renderData( vrf=self._vrf, instanceId=self._instanceId )
      print( "" )

class IsisLsdbDetailModel( IsisLsdbBaseModel ):
   level = GeneratorDict( keyType=int, valueType=LevelDetailModel,
                                 help="A mapping of isis level to LSDB" )
   def processData( self, data ):
      IsisLsdbBaseModel.processData( self, data )
      
   def render( self ):
      IsisLsdbBaseModel.render( self )
      
class IsisLsdbSummaryModel( IsisLsdbBaseModel ):
   level = GeneratorDict( keyType=int, valueType=LevelSummaryModel,
                                 help="A mapping of isis level to LSDB" )
   def processData( self, data ):
      IsisLsdbBaseModel.processData( self, data )
      
   def render( self ):
      IsisLsdbBaseModel.render( self )

class IsisLsdbDetailTlvModel ( IsisLsdbBaseModel ):
   level = GeneratorDict( keyType=int, valueType=LevelDetailTlvModel,
                          help="A mapping of isis level to LSDB" )

   def processData( self, data ):
      IsisLsdbBaseModel.processData( self, data )

   def render( self ):
      IsisLsdbBaseModel.render( self )

         
lspDbModel = IsisCliModelCommon. \
             generateIsisDictCliModel( IsisLsdbDetailModel )
lspDbVRFModel = generateVrfCliModel( lspDbModel,
                                     "IS-IS instance information for all VRFs" ) 
lspDbSummaryModel = IsisCliModelCommon. \
             generateIsisDictCliModel( IsisLsdbSummaryModel )
lspDbSummaryVRFModel = generateVrfCliModel( lspDbSummaryModel,
                                     "IS-IS instance information for all VRFs" ) 
lspDbDetailTlvModel = IsisCliModelCommon. \
             generateIsisDictCliModel( IsisLsdbDetailTlvModel )
lspDbDetailTlvVRFModel = generateVrfCliModel( lspDbDetailTlvModel,
                                     "IS-IS instance information for all VRFs" )


ISIS_SR_GB_MAX = 65536

ISIS_GLOBAL_HELLO_PADDING_ON = 1
ISIS_GLOBAL_HELLO_PADDING_ALWAYS_OFF = 2
ISIS_HELLO_PADDING_DEFAULT = 2
ISIS_HELLO_PADDING_ON = 1
ISIS_HELLO_PADDING_ALWAYS_OFF = 3

ISIS_INTERFACE_AREA_INDENT = "        "
ISIS_MIO_LEVEL_1 = 1
ISIS_MIO_LEVEL_2 = 2
ISIS_MIO_LEVEL_1_2 = 3
CIRC_FLAG_SENT_RR = 1
CIRC_FLAG_SENT_SA = 2
CIRC_FLAG_RCVD_RA = 4
CIRC_FLAG_RCVD_CSNP = 8

class IsisIntfAreaModel( Model ):
   areaId = Str( help='Area ID' )

   def processData(self, data):
      self.areaId = data.pop('area-id')

class IsisLevelAdjModel( Model ):
   systemId = Str( help='System ID of neighbor' )
   hostname = Str( help='Hostname', optional=True )
   level = Enum( values=IsisCliHelper.INTERFACE_LEVEL_RENDER_MAP,
                 help='Level' )
   state = Enum( values=IsisCliHelper.INTERFACE_ADJ_STATE_MAP,
                 help='Adjacency state' )
   adjType = Enum( values=IsisCliHelper.INTERFACE_ADJ_TYPE_MAP,
                   help='Adjacency type' )
   addrFamily = Enum( values=IsisCliHelper.ADDR_TOPO_RENDER_MAP,
                      help=neighborAddrFamilyHelp )
   adjAddrFamilyMatch = Bool( default=True,
                   help='Address families should match for adjacency formation' )
   holdTime = Int( help='Hold time in seconds' )
   supportedTopologies = Enum( values=IsisCliHelper.ADDR_TOPO_RENDER_MAP,
                               help='Supported topologies',
                               optional=True )
   snpa = Str( help='Subnetwork point of attachment of neighbor',
               optional=True )
   priority = Int( help='Priority', optional=True )
   ipv4IntfAddr = Ip4Address( help='IPv4 interface address', optional=True )
   ipv6IntfAddr = Ip6Address( help='IPv6 interface address', optional=True )
   intfAreas = GeneratorList( valueType=IsisIntfAreaModel, 
                                 help='List of interface areas' )

   def processData( self, data ):
      self.systemId = data.pop('system-id-raw')
      self.hostname = data.pop('system-id')
      if self.systemId == self.hostname:
         self.hostname = None
      levelNum = data.pop('level')
      if levelNum in IsisCliHelper.LEVEL_MAP:
         self.level = IsisCliHelper.LEVEL_MAP[ levelNum ]
      else:
         self.level = "unknown"

      self.state = IsisCliHelper.INTERFACE_ADJ_STATE_MAP.reverse()[ \
            data.pop('state') ]
      self.adjType = IsisCliHelper.INTERFACE_ADJ_TYPE_MAP.reverse()[ \
            data.pop('type') ]

      addrFamilyNum = data.pop('ngb-address-family')
      if addrFamilyNum in IsisCliHelper.ADDRESS_FAMILY_MAP:
         self.addrFamily = IsisCliHelper.ADDRESS_FAMILY_MAP[ addrFamilyNum ]
      else:
         self.addrFamily = "unknown"

      self.adjAddrFamilyMatch = ord( data.pop( 'match-adj-nlpid' ) ) != 0

      self.holdTime = data.pop('hold-time')
      if 'mt' in data and ord(data.pop('mt')) != 0:
         if data[ 'supported-topo' ] in IsisCliHelper.INTERFACE_ADDR_MAP:
            self.supportedTopologies = IsisCliHelper.INTERFACE_ADDR_MAP[ 
                                          data.pop('supported-topo') ]
         else:
            self.supportedTopologies = "none"

      if 'snpa' in data:
         self.snpa = data.pop('snpa')
         self.priority = data.pop('priority')
      if 'ipv4-ifaddr' in data:
         self.ipv4IntfAddr = data.pop('ipv4-ifaddr')
      if 'ipv6-ifaddr' in data:
         addr = data.pop( 'ipv6-ifaddr' )
         if addr and '%' in addr:
            addr = addr[ : addr.index( '%' ) ]
         self.ipv6IntfAddr = addr

      return data

   def render( self ):
      print( "    Adjacency %s:"
             % ( self.hostname if self.hostname else self.systemId ) )
      print( "      State: %s, Level: %s Type: %s"
            % ( IsisCliHelper.INTERFACE_ADJ_STATE_MAP[ self.state ], 
                IsisCliHelper.INTERFACE_LEVEL_RENDER_MAP[ self.level ], 
                IsisCliHelper.INTERFACE_ADJ_TYPE_MAP[ self.adjType ] ) )
      print( "      Advertised hold time: %s" % ( self.holdTime ) )
      print( "      Neighbor supported address families: %s"
             % ( IsisCliHelper.ADDR_TOPO_RENDER_MAP[ self.addrFamily ] ) )
      print( "      Address family match: %s"
             % ( "Enabled" if self.adjAddrFamilyMatch else "Disabled" ) )
      if self.supportedTopologies is not None:
         if self.supportedTopologies != "none":
            print( "      Supported topologies: %s" % (
                   IsisCliHelper.ADDR_TOPO_RENDER_MAP[
                      self.supportedTopologies ] ) )
         else:
            print( "      Supported topologies:" )
      if self.snpa is not None:
         print( "      SNPA: %s, Priority: %d" % ( self.snpa, self.priority ) )
      if self.ipv4IntfAddr is not None:
         print( "      IPv4 interface address: %s" % ( self.ipv4IntfAddr ) )
      if self.ipv6IntfAddr is not None:
         print( "      IPv6 interface address: %s" % ( self.ipv6IntfAddr ) )

      areaIds = []
      for entry in self.intfAreas:
         areaIds.append( entry.areaId )
      if len( areaIds ) == 1:
         print( "      Area addresses: %s" % areaIds[ 0 ] )
      else:
         print( "      Area addresses:" )
         for areaId in natsort.natsorted( areaIds ):
            print( f"{ISIS_INTERFACE_AREA_INDENT}{areaId}" )

class GracefulRestartStatus( Model ):
   rrSent = Bool( help='Restart Request Sent' )
   saSent = Bool( help='Suppress Adjacency Sent' )
   raRcvd = Bool( help='Restart Acknowledgement received' )
   csnpRcvd = Bool( help='CSNP received' )

class BroadcastCircuitInfoModel( Model ):
   lanId = Str( help='LAN ID for broadcast circuit' )
   lanIdWithHostname = Str( help='Lan ID with Hostname for broadcast circuit', 
                              optional=True )
   priority = Int( help='Priority for broadcast circuit' )
   dis = Str( help='System ID of DIS for broadcast circuit' )
   disHostname = Str( help='Hostname of DIS for broadcast circuit', optional=True )
   disPriority = Int( help='DIS Priority for broadcast circuit' )
   
class IsisInterfaceLevelModel( Model ):
   _level = Int(help = 'IS-IS interface level')
   ipv6Metric = Int( help='IPv6 metric', optional=True )
   ipv4Metric = Int( help='IPv4 metric' )
   numAdjacencies = Int( 
                     help='Number of adjacencies (except for loopback interfaces)',
                     optional=True )
   broadcastCircuitInfo = Submodel( valueType=BroadcastCircuitInfoModel,
                  help='Broadcast circuit information', optional=True )
   linkId = Str( help='Link ID for P2P interface', optional=True )
   authenticationMode = Enum( values=IsisCliHelper.AUTH_MODE_MAP.values(),
                              help='Authentication mode', optional=True )
   authenticationModeKeyId = Int( help='Authentication mode key ID',
                                 optional=True )
   sharedSecretProfile = Str( help="Shared secret profile", optional=True )
   _intfName = Str( help="Interface Name", optional=True )
   authRxDisabled = Bool( help='Received hello authentication check is disabled',
                          optional=True )
   grStatus = Submodel( valueType=GracefulRestartStatus,
         help='Status of graceful restart', optional=True )
   isisAdjacencies = GeneratorList( valueType=IsisLevelAdjModel,
         help='List of ISIS adjacency entries' )
   passive = Bool( default=False, help='Passive state' )
   v4Protection = Enum( values=IsisCliHelper.TILFA_PROTECTION_MAP.values(),
                        help='TI-LFA protection mode for IPv4' )
   v6Protection = Enum( values=IsisCliHelper.TILFA_PROTECTION_MAP.values(),
                        help='TI-LFA protection mode for IPv6' )
   v4SrlgProtection = Enum( values=SRLG_PROTECTION_MAP.values(),
                            optional=True,
                            help='TI-LFA SRLG protection mode for IPv4' )
   v6SrlgProtection = Enum( values=SRLG_PROTECTION_MAP.values(),
                            optional=True,
                            help='TI-LFA SRLG protection mode for IPv6' )

   def getKey( self, data ): # pylint: disable=inconsistent-return-statements
      assert data[ 'level-num' ]
      if data[ 'level-num' ] == 1 or data[ 'level-num' ] == 2:
         return str( data[ 'level-num' ] )
      else:
         return

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_LEVEL

   def processData( self, data ):
      self._level = data.pop('level-num')
      metricType = ('mt' in data and ord(data.pop('mt')) != 0)
      self.ipv4Metric = data.pop('metric')
      
      if metricType:
         self.ipv6Metric = data.pop('mt-metric')

      if 'num-adj' in data:
         self.numAdjacencies = int(data.pop('num-adj'))

      if 'lan-id' in data:
         self.broadcastCircuitInfo = BroadcastCircuitInfoModel()
         if data[ 'lan-id-raw' ] == data[ 'lan-id' ]:
            self.broadcastCircuitInfo.lanId = data.pop('lan-id')
            data.pop('lan-id-raw')
         else:
            self.broadcastCircuitInfo.lanId = data.pop('lan-id-raw')
            self.broadcastCircuitInfo.lanIdWithHostname = data.pop('lan-id')
         self.broadcastCircuitInfo.priority = data.pop('priority')
         if data[ 'dis-raw' ] == data[ 'dis' ]:
            self.broadcastCircuitInfo.dis = data.pop('dis')
            data.pop('dis-raw')
         else:
            self.broadcastCircuitInfo.dis = data.pop('dis-raw')
            self.broadcastCircuitInfo.disHostname = data.pop('dis')
         self.broadcastCircuitInfo.disPriority = data.pop('dis-priority')
      elif 'link-id' in data:
         self.linkId = data.pop('link-id')
      
      if 'intf-name' in data:
         self._intfName = data.pop('intf-name')

      if self._intfName is not None:
         from CliPlugin.RoutingIsisCli import getSharedSecretProfileNameIntf
         self.sharedSecretProfile = getSharedSecretProfileNameIntf( self._intfName,
                                                                    self._level )

      if 'Circuit Auth' in data:
         self.authenticationMode = IsisCliHelper.AUTH_MODE_MAP[ 
               int(data.pop('Circuit Auth')) ]
      if ( self.sharedSecretProfile and 'Circuit-Auth-mode-id' in data ) or \
           self.authenticationMode == "SHA":
         self.authenticationModeKeyId = data.pop( 'Circuit-Auth-mode-id' )

      if 'auth-rx-disabled' in data:
         self.authRxDisabled = ord( data[ 'auth-rx-disabled' ] ) != 0

      if 'gr-flags' in data and ord( data[ 'gr-flags' ] ) != 0:
         grFlags = ord( data.pop( 'gr-flags' ) )
         self.grStatus = GracefulRestartStatus()
         self.grStatus.rrSent = bool(grFlags & CIRC_FLAG_SENT_RR)
         self.grStatus.saSent = bool(grFlags & CIRC_FLAG_SENT_SA)
         self.grStatus.raRcvd = bool(grFlags & CIRC_FLAG_RCVD_RA)
         self.grStatus.csnpRcvd = bool(grFlags & CIRC_FLAG_RCVD_CSNP)
      self.passive = ord( data.pop( 'passive' ) ) != 0
      self.v4Protection = IsisCliHelper.TILFA_PROTECTION_MAP[
         ord( data.pop( 'protection-mode-v4' ) ) ]
      self.v6Protection = IsisCliHelper.TILFA_PROTECTION_MAP[
         ord( data.pop( 'protection-mode-v6' ) ) ]
      if 'srlg-protection-v4' in data:
         self.v4SrlgProtection = IsisCliHelper.SRLG_PROTECTION_MAP[
            ord( data.pop( 'srlg-protection-v4' ) ) ]
      if 'srlg-protection-v6' in data:
         self.v6SrlgProtection = IsisCliHelper.SRLG_PROTECTION_MAP[
            ord( data.pop( 'srlg-protection-v6' ) ) ]

   def renderEntryBrief( self, levelName ):
      return self._level

   def renderEntry( self, levelName ):
      print( "    Level %d:" % ( self._level ) )
      passive = ' (passive interface)' if self.passive else ''
      adjStr = str( self.numAdjacencies ) + passive
      if self.numAdjacencies is not None:
         if self.ipv6Metric is not None:
            print(
                  "      IPv4 metric: %d, IPv6 metric: %d, Number of adjacencies: %s"
                  % ( self.ipv4Metric, self.ipv6Metric, adjStr ) 
                   )
         else:
            print( "      Metric: %d, Number of adjacencies: %s"
                   % ( self.ipv4Metric, adjStr ) )
      else:
         if self.ipv6Metric is not None:
            print( "      IPv4 metric: %d, IPv6 metric: %d%s"
                   % ( self.ipv4Metric, self.ipv6Metric, passive ) )
         else:
            print( "      Metric: %d%s" % ( self.ipv4Metric, passive ) )
      
      if self.broadcastCircuitInfo is not None:
         renderLanId = self.broadcastCircuitInfo.lanIdWithHostname \
                           if self.broadcastCircuitInfo.lanIdWithHostname \
                           else self.broadcastCircuitInfo.lanId
         print( "      LAN-ID: %s, Priority: %d" % ( renderLanId,
                                 self.broadcastCircuitInfo.priority ) )
         renderDis = ( self.broadcastCircuitInfo.disHostname 
                        if self.broadcastCircuitInfo.disHostname 
                        else self.broadcastCircuitInfo.dis )
         print( "      DIS: %s, DIS priority: %d" % ( renderDis,
                                       self.broadcastCircuitInfo.disPriority ) )
      elif self.linkId is not None:
         print( "      Link-ID: %s" % ( self.linkId ) )

      modeIdStr = ""
      if self.authenticationModeKeyId:
         modeIdStr = " Key ID: %d" % self.authenticationModeKeyId
      print( "      Authentication mode: ", end="" )
      if self.sharedSecretProfile:
         print( "\n        Shared-secret profile: %s \n        Algorithm: "
                        % self.sharedSecretProfile, end="" ) 
      print( "{}{}".format( self.authenticationMode
                        if self.authenticationMode else 'None', modeIdStr ) )

      if self.authRxDisabled:
         print( "      Received hello authentication check: disabled" )

      if self.grStatus is not None:
         print( "      Graceful restart status: %s%s%s%s"
               % ( "RR sent " if self.grStatus.rrSent else "", 
                  "SA sent " if self.grStatus.saSent else "", 
                  "RA rcvd " if self.grStatus.raRcvd else "", 
                  "CSNP rcvd" if self.grStatus.csnpRcvd else "" ) )

      protectionStr = "      TI-LFA {} protection {}is enabled for the following " \
                      "IPv{} segments: node segments, adjacency segments"
      protectionStr += ", flex-algo node segments"

      disabledStr = "      TI-LFA protection is disabled for IPv{}"

      srlgProtectionFormat = "with SRLG {} protection "
      v4SrlgProtectionStr = ""
      if self.v4SrlgProtection is not None:
         v4SrlgProtectionStr = srlgProtectionFormat.format(
            self.v4SrlgProtection )

      v6SrlgProtectionStr = ""
      if self.v6SrlgProtection is not None:
         v6SrlgProtectionStr = srlgProtectionFormat.format(
            self.v6SrlgProtection )

      if self.v4Protection == 'disabled':
         print( disabledStr.format( 4 ) )
      else:
         print( protectionStr.format( self.v4Protection, v4SrlgProtectionStr,
                                     4 ) )

      if self.v6Protection == 'disabled':
         print( disabledStr.format( 6 ) )
      else:
         print( protectionStr.format( self.v6Protection, v6SrlgProtectionStr,
                                     6 ) )

      for entry in self.isisAdjacencies:
         if self._level is not None:
            entry.render()


class DisabledReason( Model ):
   message = Str( help='Reason for interface being disabled')
   circuitType = Str( help='Configured circuit type on interface' )
   routerCircType = Str( help='Circuit type of router instance' )

class IsisNodeSegmentModel( Model ):
   index = Int( help='Node segment index' )
   ipv6 = Bool( help='Whether node segment index is for IPv6' )
   flexAlgo = Str( help='Flexible algorithm name', optional=True )

   def processData( self, data ):
      from CliPlugin.RoutingIsisCli import getFlexAlgoName
      self.index = data.pop( 'sr-node-sid', None )
      self.ipv6 = bool( data.pop( 'sr-node-sid-v6', None ) )
      algoId = ord( data.pop( 'sr-node-sid-algo', "\0" ) )
      if algoId:
         self.flexAlgo = getFlexAlgoName( algoId )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_NODE_SEGMENT

   def render( self ):
      if self.index:
         flexAlgoStr = ' Algorithm: %s' % self.flexAlgo if self.flexAlgo else ''
         afStr = 'IPv6' if self.ipv6 else 'IPv4'
         print( '    Node segment index %s: %d%s' %
                ( afStr, self.index, flexAlgoStr ) )

class IsisInterface( Model ):
   _interfaceName = Str( help='Interface name' )
   enabled = Bool( help='Interface enabled or not' )
   disabledReason = Submodel( valueType=DisabledReason,
                              help='Reason for interface being disabled', 
                              optional=True )
   index = Int( help='Interface index', optional=True )
   snpa = Str( help='Subnetwork point of attachment', optional=True )
   mtu = Int( help='MTU', optional=True )
   interfaceAddressFamily = Enum( values=( 'none', 'ipv4', 'ipv6', 'both' ),
                                  default='none', help=interfaceAddrFamilyHelp,
                                  optional=True )
   interfaceType = Enum( values=IsisCliHelper.INTERFACE_TYPES_MAP.values(),
                         help="Interface type", optional=True )
   srIndexV4 = Int( help='Node segment index for IPv4', optional=True )
   srIndexV6 = Int( help='Node segment index for IPv6', optional=True )
   srNodeSegments = GeneratorList( valueType=IsisNodeSegmentModel,
                                   help='Node segments for flex-algo',
                                   optional=True )
   topology = Enum( values=IsisCliHelper.INTERFACE_ADDR_MAP.values(),
                    help='Type of topologies enabled',
                    optional=True )
   bfdIpv4Enabled = Bool( help="Whether IPv4 bfd is enabled on the interface", 
                           optional=True )
   bfdIpv6Enabled = Bool( help="Whether IPv6 bfd is enabled on the interface", 
                           optional=True )
   helloPaddingEnabled = Bool( 
                           help="Whether hello padding is enabled on the interface",
                           optional=True )
   helloPadding = Enum( values=( 'enabled', 'disabled', 'enabledUntilAdj' ),
                        help=helloPaddingHelp, optional=True )
   intfLevels = GeneratorDict( valueType=IsisInterfaceLevelModel, 
                               help="""Dictionary of levels of the interface
                               keyed by level name""", 
                               optional=True )
   ipv4MetricProfile = Str( help='IPv4 metric profile', optional=True )
   ipv6MetricProfile = Str( help='IPv6 metric profile', optional=True )
   interfaceSpeed = Int( help='Interface speed in mbps', optional=True )

   ipv4RouteTag = Int( help='IPv4 route tag', optional=True )
   ipv6RouteTag = Int( help='IPv6 route tag', optional=True )
   areaProxyBoundary = Bool( help='Interface enabled for area proxy boundary',
                             optional=True )

   def getKey( self, data ):
      assert data[ 'circ-id' ]
      ifName = intfLongName( data[ 'circ-id' ] )
      return ifName
   
   def fetchExtraDataFromSysDb( self, intfStatus ):
      if intfStatus.metricProfile:
         self.ipv4MetricProfile = intfStatus.metricProfile
      if intfStatus.metricProfileV6:
         self.ipv6MetricProfile = intfStatus.metricProfileV6
      if intfStatus.interfaceSpeed:
         self.interfaceSpeed = intfStatus.interfaceSpeed

   def processData( self, data ): # pylint: disable=inconsistent-return-statements
      if 'circ-id' not in data:
         return

      self._interfaceName = data.pop('circ-id', None)
      self.enabled = True
      self.index = data.pop('circ-index')

      self.interfaceType = IsisCliHelper.INTERFACE_TYPES_MAP[ data.pop('type') ] \
            if data['type'] in IsisCliHelper.INTERFACE_TYPES_MAP else None
      
      if 'sr-index-v4' in data:
         indexV4 = data.pop('sr-index-v4')
         if indexV4 != ISIS_SR_GB_MAX:
            self.srIndexV4 = indexV4
      
      if 'sr-index-v6' in data:
         indexV6 = data.pop('sr-index-v6')
         if indexV6 != ISIS_SR_GB_MAX:
            self.srIndexV6 = indexV6

      intfAF = data.pop( 'intf-address-family' )
      self.interfaceAddressFamily = IsisCliHelper.ADDRESS_FAMILY_MAP[ intfAF ]

      if 'mt' in data and ord(data.pop('mt')) != 0:
         self.topology = IsisCliHelper.INTERFACE_ADDR_MAP[ \
               data.pop('protos-enabled') ]

      bfdConfig = data.pop('bfd-config')
      if bfdConfig == 0:
         bfdGlobal = data.pop('bfd-global')
         if ord(bfdGlobal) == 0 or bfdGlobal == '':
            self.bfdIpv4Enabled = False 
         else:
            self.bfdIpv4Enabled = True
      elif bfdConfig == 1:
         self.bfdIpv4Enabled = True
      else:
         self.bfdIpv4Enabled = False

      bfdConfig6 = data.pop('bfd-config-v6')
      if bfdConfig6 == 0:
         bfdGlobal6 = data.pop('bfd-global-v6')
         if ord(bfdGlobal6) == 0 or bfdGlobal6 == '':
            self.bfdIpv6Enabled = False
         else:
            self.bfdIpv6Enabled = True
      elif bfdConfig6 == 1:
         self.bfdIpv6Enabled = True
      else:
         self.bfdIpv6Enabled = False

      self.ipv4RouteTag = data.pop( 'route-tag-v4', None )
      self.ipv6RouteTag = data.pop( 'route-tag-v6', None )

      helloPaddingIntfConfig = ord( data.pop( 'hello-padding-intf' ) )
      helloPaddingInstConfig = ord( data.pop( 'hello-padding-inst' ) )
      if helloPaddingIntfConfig == ISIS_HELLO_PADDING_DEFAULT \
            and helloPaddingInstConfig == ISIS_GLOBAL_HELLO_PADDING_ON:
         self.helloPaddingEnabled = True
         self.helloPadding = 'enabled'
      elif helloPaddingIntfConfig == ISIS_HELLO_PADDING_DEFAULT \
            and helloPaddingInstConfig == ISIS_GLOBAL_HELLO_PADDING_ALWAYS_OFF:
         self.helloPaddingEnabled = False
         self.helloPadding = 'disabled'
      elif helloPaddingIntfConfig == ISIS_HELLO_PADDING_ON:
         self.helloPaddingEnabled = True
         self.helloPadding = 'enabled'
      elif helloPaddingIntfConfig == ISIS_HELLO_PADDING_ALWAYS_OFF:
         self.helloPaddingEnabled = False
         self.helloPadding = 'disabled'
      else:
         self.helloPaddingEnabled = False
         self.helloPadding = 'enabledUntilAdj'

      areaProxyBoundaryData = ord( data.pop( 'area-proxy-boundary', chr( 0 ) ) )
      self.areaProxyBoundary = ( areaProxyBoundaryData != 0 )

      return data

   def renderEntry( self, intfFullName ):
      if self._interfaceName is None:
         return
      print( "  Interface %s:" % ( intfFullName ) )
      if not self.enabled:
         print( "    %s" % ( self.disabledReason.message ) )
         return
      print( f"    Index: {self.index} SNPA: {self.snpa}" )
      if self.interfaceType is not None:
         print( f"    MTU: {self.mtu} Type: {self.interfaceType}" )
      print( "{}{}".format( "    Supported address families: ",
               IsisCliHelper.ADDR_TOPO_RENDER_MAP[ self.interfaceAddressFamily ] ) )
      if self.areaProxyBoundary:
         print( "    Area proxy boundary is enabled" )
      else:
         print( "    Area proxy boundary is disabled" )
      if self.ipv4RouteTag and self.ipv6RouteTag:
         print( "    Route tag: IPv4: %d, IPv6: %d" % ( self.ipv4RouteTag,
                                                         self.ipv6RouteTag ) )
      elif self.ipv4RouteTag:
         print( "    Route tag: IPv4: %d" % ( self.ipv4RouteTag ) )
      elif self.ipv6RouteTag:
         print( "    Route tag: IPv6: %d" % ( self.ipv6RouteTag ) )

      if self.ipv4MetricProfile:
         print( "    Metric profile for IPv4: '%s'" % self.ipv4MetricProfile )
      if self.ipv6MetricProfile:
         print( "    Metric profile for IPv6: '%s'" % self.ipv6MetricProfile )
      if self.interfaceSpeed:
         print( "    Speed: %d mbps" % self.interfaceSpeed )
      if self.srIndexV4 is not None:
         print( "    Node segment index IPv4: %s" % self.srIndexV4 )
      if self.srIndexV6 is not None:
         print( "    Node segment index IPv6: %s" % self.srIndexV6 )
      if self.srNodeSegments:
         v6NodeSegments = []
         for nodeSeg in self.srNodeSegments:
            if not nodeSeg.ipv6:
               nodeSeg.render()
            else:
               v6NodeSegments.append( nodeSeg )
         # print v6 node segments
         for nodeSeg in v6NodeSegments:
            nodeSeg.render()

      if self.topology is not None:
         print( "    Enabled topologies: %s"
                % ( IsisCliHelper.ADDR_TOPO_RENDER_MAP[ self.topology ] ) )
      
      print( "    %s"
            % ( "BFD IPv4 is enabled" if self.
               bfdIpv4Enabled else "BFD IPv4 is disabled" ) )
      print( "    %s"
            % ( "BFD IPv6 is enabled" if self.
               bfdIpv6Enabled else "BFD IPv6 is disabled" ) )
      if self.helloPadding == 'enabled':
         print( "    Hello padding is enabled" )
      elif self.helloPadding == 'disabled':
         print( "    Hello padding is disabled" )
      else:
         print( "    Hello padding is enabled until adjacency is up" )

      for levelName, level in self.intfLevels:
         level.renderEntry( levelName )

   def renderEntryBrief( self, intfFullName ): 
      if self._interfaceName is None:
         return None
      ipv4Metric = None
      ipv6Metric= None
      numAdjacencies = 0
      level = 0
      levelCount = 0
      intfEntryRow = []
      for levelName, intf in self.intfLevels:
         level = intf.renderEntryBrief( levelName )
         levelCount += 1
         ipv4Metric = intf.ipv4Metric
         ipv6Metric = intf.ipv6Metric if intf.ipv6Metric else ipv4Metric
         if intf.numAdjacencies:
            numAdjacencies += intf.numAdjacencies
         if intf.passive:
            numAdjacencies = "(passive)"
      levelMap = { 1:"L1", 2:"L2" }
      levelTag = "L1L2" if levelCount == 2 else levelMap[ level ]
      intfType = self.interfaceType
      intfEntryRow = [ intfFullName, levelTag, ipv4Metric, ipv6Metric,
                       intfType, numAdjacencies ]
      return intfEntryRow
        
class IsisInterfaceModel( Model ):
   _instanceName = Str( help='Instance Name' )
   _vrf = Str( help='Vrf Name' )
   interfaces = GeneratorDict( keyType=Interface, valueType=IsisInterface, 
                               help="""Dictionary of IS-IS interfaces keyed 
                                       by interface name""" )
   _brief = Bool( default=False, 
                  help='Flag to render brief interface output' )

   def processData( self, data ):
      self._vrf = data.pop('vrf-name')
      self._instanceName = data.pop( 'instance-name', None )

   def renderBrief( self ):
      leftJustify = TableOutput.Format( justify='left' )
      leftJustify.padLimitIs( 1 )
      rightJustify = TableOutput.Format( justify='right' )
      rightJustify.padLimitIs( 1 )

      heads = [ "Interface", "Level", "IPv4 Metric", "IPv6 Metric",
                "Type", "Adjacency" ]

      intfTable = TableOutput.createTable( heads )
      intfTable.formatColumns( leftJustify, leftJustify,
               rightJustify, rightJustify, leftJustify, rightJustify )

      intfCount = 0
      for intfName, intf in self.interfaces:
         if intf.enabled:
            intfEntryRow = intf.renderEntryBrief( intfName )
            if intfEntryRow is not None:
               intfTable.newRow( *intfEntryRow )
               intfCount += 1
      if intfCount:
         print( intfTable.output() )

   def render( self ):
      print( "" )
      print( f"IS-IS Instance: {self._instanceName} VRF: {self._vrf}" )
      print( "" )
      if self._brief:
         self.renderBrief()
      else:
         for intfName, intf in self.interfaces:
            intf.renderEntry( intfName )

interfaceModel = IsisCliModelCommon.generateIsisDictCliModel( IsisInterfaceModel )
interfaceVRFsModel = generateVrfCliModel( interfaceModel,  
                                          "IS-IS interface information for all VRFs"
                                        )

def ngbGrStateString( state ):
   statusString = ''
   rrSa = False
   if state & ISIS_GR_RCVD_SA:
      statusString += "Status: Starting (SA rcvd"
      rrSa = True
   elif state & ISIS_GR_RCVD_RR:
      statusString += "Status: Restarting (RR rcvd"
      rrSa = True

   if rrSa:
      if state & ISIS_GR_SENT_RA:
         statusString += " RA sent"
      if ( state & ISIS_GR_RCVD_RR ) and ( state & ISIS_GR_RCVD_SA ):
         statusString += " RR+SA rcvd"
      if state & ISIS_GR_SENT_CSNP:
         statusString += " CSNP sent"
      statusString += ")"

   return statusString

def getRouterId( systemId, instanceId=0 ):
   RouterId = Tac.Type( "Mpls::RouterId" )
   rId = RouterId()
   rId.stringValue = systemId
   rId.igpInstanceId = instanceId
   return rId


class SrInfoDetail( Model ):
   srLabelV4 = Int( help='Adjacency Label for V4', optional=True )
   srLabelV6 = Int( help='Adjacency Label for V6', optional=True )
   srRouterId = Ip4Address( help='SR Router Id' )
   srGbBase = Int( help='SR Base Value' )
   srGbRange = Int( help='SR Range' )

class IsisNeighborDetails( Model ):
   priority = Int( help='ISIS neighbor priority', optional=True )
   advertisedHoldTime = Int( help='Negotiated hold time \
                           (in seconds) between neighbors' )
   stateChanged = Int( help='UTC Time at which the state changed' )
   interfaceAddressFamily = Enum( values=( 'none', 'ipv4', 'ipv6', 'both' ),
                         default='none', help=interfaceAddrFamilyHelp )
   neighborAddressFamily = Enum( values=( 'none', 'ipv4', 'ipv6', 'both' ),
                         default='none', help=neighborAddrFamilyHelp )
   ip4Address = Ip4Address( help='ISIS Neighbor IPv4 address', optional=True )
   ip6Address = Ip6Address( help='ISIS Neighbor IPv6 address', optional=True )
   ip6GlobalAddress = Ip6Address( help='ISIS Neighbor Global IPv6 address',
                                  optional=True )
   areaIds = List( valueType=str, help='ISIS Neighbor area(s)' )
   bfdIpv4State = Enum( values=( 'up', 'down', 'init', 'adminDown', 'NA' ), 
                        help=bfdStateHelp, optional=True )
   bfdIpv6State = Enum( values=( 'up', 'down', 'init', 'adminDown', 'NA' ), 
                        help=bfdStateHelp, optional=True )
   srEnabled = Bool( help='Shows whether SR enabled on dut' )
   srInfoDetail = Submodel( valueType=SrInfoDetail,
                      help='Information about segment routing', optional=True )
   grSupported = Str( help='Shows whether GR is Supported or Not' )
   grState = Str( help='Shows the state of GR', optional=True )

class IsisNeighborEntry( Model ):
   _instanceName = Str( help='ISIS Instance name' )
   _vrf = Str( help='VRF Name' )
   _instanceId = Int( help='IS-IS Instance ID' )
   hostname = Str( help='Hostname', optional=True )
   circuitId = Str( help='Circuit Identifier' )
   interfaceName = Interface( help='ISIS Interface Name' )
   #No 'failed' state
   state = Enum( values=( 'up', 'down', 'init' ), help='ISIS Adjacency state' )
   lastHelloTime = Int( help='UTC timestamp when the last hello was received' )
   snpa = Str( help='Sub Network Point Of Attachment Address', optional=True )
   level = Enum( values=( 'level-1', 'level-2', 'level-1-2' ),
                     help='Level information of the neighbor' )
   details = Submodel( valueType=IsisNeighborDetails, 
                       help='ISIS neighbor details', optional=True )
   routerIdV4 = Ip4Address( help='ISIS neighbor router ID in IP format',
                            optional=True )
   _matchingLocalAreaAddresses = List( valueType=str,
                           help='Area addresses common between neighbor and router' )

   def processData( self, data ):
      from CliPlugin.RoutingIsisCli import getHostName
      self._instanceName = data.pop( 'instance-name', None )
      self._vrf = data.pop( 'vrf-name' )
      self._instanceId = data.pop( 'instance-id' )
      self.hostname = getHostName( getRouterId( data[ 'system-id' ],
                                                self._instanceId ),
                                   vrfName=self._vrf )
      self.circuitId = data.pop( 'circuit-id' )
      self.level = IsisCliHelper.LEVEL_TYPE_MAP[ data.pop( 'type' ) ]
      self.state = IsisCliHelper.STATE_MAP[ data.pop( 'state' ) ]
      self.interfaceName = intfLongName( data.pop( 'interface-name' ) )
      self.lastHelloTime = data.pop( 'iih-recv-time' )
      self.snpa = data.pop( 'snpa' )
      if 'router-id-v4' in data:
         self.routerIdV4 = data.pop( 'router-id-v4' )
      else:
         self.routerIdV4 = '0.0.0.0'
      details = IsisNeighborDetails()
      setattr( details, 'priority', data.pop( 'priority', None ) )
      intfAF = data.pop( 'intf-address-family', None )
      details.interfaceAddressFamily = \
                 IsisCliHelper.ADDRESS_FAMILY_MAP[ intfAF ]
      ngbAF = data.pop( 'ngb-address-family', None )
      details.neighborAddressFamily = \
                 IsisCliHelper.ADDRESS_FAMILY_MAP[ ngbAF ]
      setattr( details, 'ip4Address', data.pop( 'ipv4-ifaddr', None ) )
      addr = data.pop( 'ipv6-ifaddr', None )
      if addr and '%' in addr:
         addr = addr[ : addr.index( '%' ) ]
      details.ip6Address = addr
      setattr( details, 'ip6GlobalAddress', data.pop( 'ipv6-global-ifaddr', None ) )
      #Convert to list
      if 'area-ids' in data:
         areaAddresses = data.pop( 'area-ids' ).split( "," )
         areaIds = []
         for area in areaAddresses:
            if " (matching local area address)" in area:
               area = area.replace( " (matching local area address)", "" )
               self._matchingLocalAreaAddresses.append( area )
            areaIds.append( area )
         details.areaIds.extend( natsort.natsorted( areaIds ) )
      else:
         details.areaIds = [ '*NONE*' ]
      if data[ 'bfd-neighbor' ] != 4:
         details.bfdIpv4State = IsisCliHelper.BFD_STATE_MAP[ data.pop( \
               'bfd-neighbor' ) ]
      if data[ 'bfd-neighbor-v6' ] != 4:
         details.bfdIpv6State = IsisCliHelper.BFD_STATE_MAP[ data.pop( \
               'bfd-neighbor-v6' ) ]
      details.srEnabled = structUnpack1B( data['sr-enabled'] ) != 0
      srInfoDetail = SrInfoDetail()
      if details.srEnabled:
         srInfoDetail.srGbRange = data.pop( 'srgb-range' )
         srInfoDetail.srGbBase = data.pop( 'srgb-base' )
         if 'sr-label-v4' in data:
            srInfoDetail.srLabelV4 = data.pop( 'sr-label-v4' )
         if 'sr-label-v6' in data:
            srInfoDetail.srLabelV6 = data.pop( 'sr-label-v6' )
         details.srInfoDetail = srInfoDetail
      details.advertisedHoldTime = data.pop( 'hold-time' )
      details.stateChanged = data.pop( 'state-change' )
      grSupported = "Not Supported"
      if structUnpack1B( data.pop( 'gr-capable' ) ):
         grSupported = "Supported"
      details.grSupported = grSupported
      if grSupported == "Supported":
         grState = structUnpack1B( data.pop( 'gr-state' ) )
         details.grState = ngbGrStateString( grState )
      self.details = details

   def renderDetails( self, ifName ):
      details_indent = "  "
      #Accumulating all areas without any separator
      #In future separator can be used
      areaIds = self.details.areaIds if self.details.areaIds else [ "*NONE*" ]
      if len( areaIds ) == 1:
         print( "{}{}: {}".format( details_indent,
               "Area addresses", areaIds[ 0 ] ) )
      else:
         print( "{}{}:".format( details_indent, "Area addresses" ) )
         for area in areaIds:
            if area in self._matchingLocalAreaAddresses:
               area += " (matching local area address)"
            print( f"{details_indent}  {area}" )
      print( "{}{}: {}".format( details_indent, "SNPA", self.snpa ) )
      print( "{}{}: {}".format( details_indent, "Router ID", self.routerIdV4 ) )

      print( "{}{}: {}".format( details_indent, "Advertised Hold Time",
                           self.details.advertisedHoldTime ) )
      print( "{}{}: {}".format( details_indent, "State Changed",
                           IsisCliHelper.elapsedTime( self.details.stateChanged ) +
                           " ago at " + datetime.fromtimestamp(
                           self.details.stateChanged ).
                           strftime( '%Y-%m-%d %H:%M:%S' ) ) )
      #print priority only when LAN neighbor
      if self.details.priority:
         print( "{}{}: {}".format( details_indent, "LAN Priority",
                              self.details.priority ) )
      ip4Address = 'none'
      if self.details.ip4Address:
         ip4Address = self.details.ip4Address
      print( "{}{}: {}".format( details_indent,
            "IPv4 Interface Address", ip4Address ) )
      ip6Address = 'none'
      if self.details.ip6Address:
         ip6Address = self.details.ip6Address
      print( "{}{}: {}".format( details_indent,
            "IPv6 Interface Address", ip6Address ) )
      if self.details.ip6GlobalAddress:
         print( "{}{}: {}".format( details_indent, "Global IPv6 Interface Address",
                              self.details.ip6GlobalAddress ) )
      print( "{}{}: {}".format( details_indent, "Interface name", ifName ) )
      print( "{}{}: {} {}".format( details_indent, "Graceful Restart",
                               self.details.grSupported,
                               self.details.grState if self.details.grState
                               else '' ) )
      #print BFD only when BFD state is UP
      if self.details.bfdIpv4State == 'up':
         print( "{}{}".format( details_indent, "BFD IPv4 state is Up" ) )
      if self.details.bfdIpv6State == 'up':
         print( "{}{}".format( details_indent, "BFD IPv6 state is Up" ) )
      if self.details.srEnabled:
         print( "%sSegment Routing Enabled" % ( details_indent ) )
         print( "{}  SRGB Base: {} Range: {}".format( details_indent,
                                 self.details.srInfoDetail.srGbBase,
                                       self.details.srInfoDetail.srGbRange ) )
         srV4Lbl = self.details.srInfoDetail.srLabelV4
         srV6Lbl = self.details.srInfoDetail.srLabelV6

         if srV4Lbl or srV6Lbl:
            adjacencyLabel = "%s  Adjacency Label" % ( details_indent )
            if srV4Lbl:
               adjacencyLabel += " IPv4: %s" % ( srV4Lbl )
               adjacencyLabel += "," if srV6Lbl else ""
            if srV6Lbl:
               adjacencyLabel += " IPv6: %s" % ( srV6Lbl ) 
            print( adjacencyLabel )

      print( "{}{}{}".format( details_indent, "Supported Address Families: ",
       IsisCliHelper.ADDR_TOPO_RENDER_MAP[
                                self.details.interfaceAddressFamily ] ) )

      print( "{}{}{}".format( details_indent,
            "Neighbor Supported Address Families: ",
       IsisCliHelper.ADDR_TOPO_RENDER_MAP[
                                self.details.neighborAddressFamily ] ) )

   def renderEntry( self, _displayHeader, _detailsPresent, systemId ):
      if _displayHeader:
         print( " " )
         print( ngbHeaderFormatStr % ( "Instance", "VRF", "System Id", "Type", 
                                       "Interface", "SNPA", "State", 
                                       "Hold time", "Circuit Id" ) )
      level = IsisCliHelper.LEVEL_SHORT_MAP[ self.level ]
      ifName = self.interfaceName.stringValue
      state = self.state.upper()
      remHoldTime = self.details.advertisedHoldTime - int( Tac.utcNow() - \
                                                           self.lastHelloTime )
      # BUG243034 - remaining hold time can be negative due to Cli responding 
      # slow in case of high cpu utilization or due to hello packet peeking logic
      # where we keep adjacency up if hello pkts are pending in queue for
      # processing. Display 0 value in those cases as holdTime can't have -ve value 
      remHoldTime = remHoldTime if (remHoldTime > 0) else 0
      if self.hostname:
         print( ngbEntryFormatStr % ( self._instanceName, self._vrf, self.hostname, 
                                  level, ifName, self.snpa, state,
                                  remHoldTime, self.circuitId ) )
      else:
         print( ngbEntryFormatStr % ( self._instanceName, self._vrf, systemId, 
                                     level, ifName, self.snpa, state,
                                     remHoldTime, self.circuitId ) )

      if not  _detailsPresent:
         return
      self.renderDetails( ifName )

class IsisSystemIdNeighbors( Model ):
   adjacencies = List( valueType=IsisNeighborEntry,
                         help="List of neighbors for SystemID" )
   _systemId = Str( help="System Identifier" )

   def getKey( self, data ):
      return data['system-id']

   def overrideHierarchy( self, data ):
      readNext = True
      if not self.adjacencies:
         self._systemId = data[ 'system-id' ]
         neighborData = IsisNeighborEntry()
         neighborData.processData( data )
         self.adjacencies.append( neighborData )
         return ( neighborData, readNext )
      elif self._systemId == self.getKey( data ):
         neighborData = IsisNeighborEntry()
         neighborData.processData( data )
         self.adjacencies.append( neighborData )
         return ( neighborData, readNext )
      else:
         return ( data, False )

   def renderEntry( self, _displayHeader, _detailsPresent, systemId ):
      for neighborData in self.adjacencies :
         neighborData.renderEntry( _displayHeader, _detailsPresent,
                                   systemId )
         _displayHeader = False

class IsisInstanceNeighbors( Model ):
   neighbors = GeneratorDict( valueType=IsisSystemIdNeighbors, keyType=str,
                           help='Dictionary of ISIS neighbors keyed by SystemID' )
   _detailsPresent = Bool( default=False, help='Private attribute to indicate'
                           ' if detailed output was requested' )

   def render( self ):
      #private variable to print header for every instance
      _displayHeader = True
      for systemId, entry in self.neighbors:
         entry.renderEntry( _displayHeader, self._detailsPresent, systemId )
         _displayHeader = False

showNeighborTableModel = IsisCliModelCommon. \
                         generateIsisDictCliModel( IsisInstanceNeighbors )
showNeighborTableVRFsModel = generateVrfCliModel( showNeighborTableModel,
                                    "ISIS instance information for all VRFs" )

class IsisSystemCountersData( Model ):
   authTypeFails = Int( help='Number of authentication type mismatches' )
   authKeyFails = Int( help='Number of authentication key failures' )
   corruptedLsps = Int( help='Number of corrupted in-memory LSPs detected' ) 
   lspErrors = Int( help='Number of received LSPs with errors' )
   databaseOverloads = Int( help='Number of times the database has become'
                            ' overloaded' )
   exceededMaxSeqNums = Int( help='Number of times the system has attempted'
                            ' to exceed the maximum sequence number' )
   ownLspPurges = Int( help='Number of times a zero-aged copy of the system\'s'
                       ' own LSP is received from some other node' )
   ownLspPurgesSent = Int( help='Number of own LSP purges sent' )
   othersLspPurgesSent = Int( help='Number of others\' LSP purges sent' )
   seqNumSkips = Int( help='Number of times a seq number skip has occurred' )
   spfRuns = Int( help='Number of times SPF was run for this level' )
   idLenMismatches = Int( help='Number of times a PDU is received with a different'
                          ' value for ID field length from that of the receiving'
                          ' system' )
   maxAreaAddressMismatches = Int( help='Number of times a PDU is received with a'
                                   ' different value for MaximumAreaAddresses from'
                                   ' that of the receiving system' )
                                                 
class IsisSystemCounters( Model ):
   systemCountersL1 = Submodel( valueType=IsisSystemCountersData,
                                help='System Counters for Level-1' )
   systemCountersL2 = Submodel( valueType=IsisSystemCountersData,
                                help='System Counters for Level-2' )

class IsisInterfaceDropReasonsMalformedPdus( Model ):
   shortHeaders = Int( help='Header length of the IS-IS PDU received is short',
                       optional=True )
   wrongVersion = Int( help='Version of the PDU header is incompatible with the'
                       ' version we support', optional=True )
   idFieldLengthMismatches = Int( help='System Id Length in the PDU is different '
                                  'from the System Id length supported by the sytem'
                                  ' on which the PDU was received. ', optional=True )

   def processData( self, data ):
      self.shortHeaders = data.pop( 'short_hdrs' )
      self.wrongVersion = data.pop( 'wrong_version' )
      self.idFieldLengthMismatches = data.pop( 'id_field_len_mismatch' )

   def getMioAttrId( self ):
      return \
         PyRibAmiClient.MIO_DGET_ISIS_INTERFACE_DROP_COUNTERS_REASONS_MALFORMED_PDUS

class IsisInterfaceDropReasons( Model ):
   badLength = Int( help='Length indicated in the packet header is different from '
                    'the actual length of the packet', optional=True )
   malformedTlv = Int( help='one of the optional TLVs in the PDU could not be'
                       ' validated', optional=True )
   authFailure = Int( help='The PDU could not be authenticated', optional=True )
   iidTlvMismatch = Int( help='Invalid IID-TLV received in the PDU', optional=True)

class IsisInterfaceDropReasonsHellos( IsisInterfaceDropReasons ):
   circuitTypeMismatches = Int( help='The Circuit type indicated in the PDU does not'
                                ' match with that of the interface on which it was '
                                'received For instance, a p2p Hello received on a '
                                'IS-IS Broadcast interface is dropped with this '
                                'reason', optional=True )
   circuitLevelMismatch = Int( help='The circuit level indicated in the PDU does '
                               ' not match with the level on which the IS-IS '
                               'interface receiving this PDU is configured on. ',
                               optional=True )   
   ownSystemIdPdu = Int( help='The System Id on the PDU is similar to that of the '
                         'system on which it is received indicating the system '
                         'received the PDU it has sent (or) some other system is '
                         'using the same system ID. The PDU is dropped.',
                         optional=True )
   unknownCircuitType = Int( help='The circuit type indicated in the PDU is '
                             'unknown. This Happens when a IS-IS Hello has a'
                             ' circuit type other than 1,2 or 3 indicating Level-1'
                             ',Level-2,Level-1-2 respectively', optional=True )
   lengthGreaterThanMTU = Int( help='Length of the PDU is greater than the MTU'
                               ' configured on the interface on which it is '
                               'received.', optional=True )
   badOptionLength = Int( help='Length of one or more of the optional TLVs in the '
                          'PDU is different from that indicated in the Length '
                          'section of that TLV', optional=True )
   noSupportedProtocols = Int( help='Hellos are dropped with this reason. The Hello'
                               ' received does not have NLPID TLV indicating that no'
                               ' protocols are supported by the system sending the'
                               ' Hello', optional=True )
   topoSuppProtocolsMismatch = Int( help='Hellos are dropped with this reaosn. This '
                                    'happens when the MTIDs and NLPIDs in the PDU'
                                    ' mismatch, indicating that there is an '
                                    'inconsistency between the topologies and'
                                    ' protocols supported by the system sending this'
                                    ' Hello.', optional=True )
   protocolAddressesMismatch = Int( help='Hellos are dropped with this reason.'
                                    'The Interface addresses received in the '
                                    'Hello are inconsistent with the NLPIDs received'
                                    ' indicating that there is an inconsistency '
                                    'between the protocols supported and interface '
                                    'addresses configured on the system sending this'
                                    ' Hello', optional=True )
   supportedProtocolsMismatch = Int( help='Hellos are dropped with this reason.'
                                     'The NLPIDs received in the PDU do not match '
                                     'the protocols supported by the system '
                                     'receiving this PDU', optional=True )
   incompatibleAdjAttrs = Int( help='Hellos are dropped with this reason. we drop '
                               'with this reason when there are no usable addresses '
                               'configured on the interface receiving this Hello OR '
                               'the protocols supported by our syatem have changed '
                               'and have become inconsistent with those supported '
                               'by the system sending this Hello.', optional=True )
   badAreaOption = Int( help='Hellos are dropped with this reason. The Area address'
                        ' sub-option in the Areas option is malformed. The areas '
                        ' indicated in the PDU can hence not be used. ',
                        optional=True )
   maxAreaAddressMismatch = Int( help='Hellos are dropped with this reason. The max '
                                 'number of areas indicated in the Hello mismatch '
                                 'with the maximum number of areas supported by the '
                                 'system receiving this hello', optional=True )
   noMatchingAreas = Int( help='Hellos are dropped with this reason. The IS-IS'
                          ' interface on which the Hello was received is in level-1'
                          ' but none of the areas indicated in the Hello match with'
                          ' the areas supported by the system. ', optional=True )
   noRemTimeInRestartTlvWithRA = Int( help='Hellos are dropped with this reason. '
                                      'Remaining time is not indicated in restart '
                                      'TLV with RA.', optional=True )

class IsisInterfaceDropReasonsP2pHellos( IsisInterfaceDropReasonsHellos ):
   noCommonTopology = Int( help='Hellos are dropped with this reason. This happens '
                           'when the system receiving the Hello is in Multi-Topology'
                           ' mode but no MTIDs in the Hello and system are common.',
                           optional=True )
   badAdjStateOption = Int( help='P2P Hellos are dropped with this reason. The '
                            'option length or the Adjacency state indicated in the '
                            'option is incompatible with the state of adjacency in '
                            'which the system receiving this hello currently in. ',
                            optional=True )
   nonMatchingThreeWayHandshake = Int( help='P2P Hellos are dropped with this reason'
                                       '. The neighbour system Id and the Neighbour '
                                       'extended circuit Id indicated in the '
                                       'adjacency state option do not match the '
                                       'system Id and the circuit Id of the system '
                                       'receiving the Hello.', optional=True )

   def processData( self, data ):
      self.badLength = data.pop( 'bad_len' )
      self.malformedTlv = data.pop( 'malformed_tlv' )
      self.authFailure = data.pop( 'auth_failure' )
      self.circuitTypeMismatches = data.pop( 'ckt_type_mismatch' )
      self.circuitLevelMismatch = data.pop( 'ckt_lev_mismatch' )
      self.ownSystemIdPdu = data.pop( 'own_sysid_pdu' )
      self.unknownCircuitType = data.pop( 'unknown_ckt_type' )
      self.lengthGreaterThanMTU = data.pop( 'len_greater_than_mtu' )
      self.badOptionLength = data.pop( 'bad_opt_len' )
      self.noSupportedProtocols = data.pop( 'no_supp_protocols' )
      self.topoSuppProtocolsMismatch = data.pop( 'topo_supp_protocols_mismatch' )
      self.protocolAddressesMismatch = data.pop( 'proto_addr_mismatch' )
      self.supportedProtocolsMismatch = data.pop( 'supp_proto_mismatch' )
      self.incompatibleAdjAttrs = data.pop( 'incomp_adj_attr' )
      self.badAreaOption = data.pop( 'bad_area_opt' )
      self.maxAreaAddressMismatch = data.pop( 'max_area_addr_mismatch' )
      self.noMatchingAreas = data.pop( 'no_matching_areas' )
      self.noRemTimeInRestartTlvWithRA = data.pop( 'no_rem_time_in_rest_tlv_with_'
                                                   'ra' )
      self.noCommonTopology = data.pop( 'no_comm_topology' )
      self.badAdjStateOption = data.pop( 'bad_adj_state_opt' )
      self.nonMatchingThreeWayHandshake = data.pop( 'non_matching_three_way_'
                                                    'handshake' )
      self.iidTlvMismatch = data.pop( 'iid_tlv_error' )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_INTERFACE_DROP_COUNTERS_REASONS_IIH_P2P

class IsisInterfaceDropReasonsLanHellos( IsisInterfaceDropReasonsHellos ):
   noSourceMac = Int( help='LAN Hellos are dropped with this reason. The Hello is '
                      'received without any source MAC address. ', optional=True )
   neighbourSysIdChanged = Int( help='LAN Hellos are Dropped with this reason. '
                                'The system Id of the neighbour sending this Hello'
                                'has changed. We hence drop the Hello and delete the'
                                ' adjacency. ', optional=True )
   dupSysIdDetected = Int( help='We already have an adjacency with the same '
                           'system Id', optional=True )
   missingAreas = Int( help='Hellos are dropped with this reason. The IS-IS '
                       'interface receiving this Hello is on Level-1 but there is '
                       'no area TLV in the Hello. We hence drop the Hello. ',
                       optional=True )
   ngbSysIdMismatchInRestartTLVWithRA = Int( help='Hellos are dropped with this '
                                             'reason. The neighbour system Id '
                                             'indicated in restart TLV with RA '
                                             'does not match our own System Id.',
                                             optional=True )
   def processData( self, data ):
      self.badLength = data.pop( 'bad_len' )
      self.malformedTlv = data.pop( 'malformed_tlv' )
      self.authFailure = data.pop( 'auth_failure' )
      self.circuitTypeMismatches = data.pop( 'ckt_type_mismatch' )
      self.ownSystemIdPdu = data.pop( 'own_sysid_pdu' )
      self.unknownCircuitType = data.pop( 'unknown_ckt_type' )
      self.lengthGreaterThanMTU = data.pop( 'len_greater_than_mtu' )
      self.badOptionLength = data.pop( 'bad_opt_len' )
      self.noSupportedProtocols = data.pop( 'no_supp_protocols' )
      self.topoSuppProtocolsMismatch = data.pop( 'topo_supp_protocols_mismatch' )
      self.protocolAddressesMismatch = data.pop( 'proto_addr_mismatch' )
      self.supportedProtocolsMismatch = data.pop( 'supp_proto_mismatch' )
      self.incompatibleAdjAttrs = data.pop( 'incomp_adj_attr' )
      self.badAreaOption = data.pop( 'bad_area_opt' )
      self.maxAreaAddressMismatch = data.pop( 'max_area_addr_mismatch' )
      self.noMatchingAreas = data.pop( 'no_matching_areas' )
      self.noRemTimeInRestartTlvWithRA = data.pop( 'no_rem_time_in_rest_tlv_'
                                                   'with_ra' )
      self.noSourceMac = data.pop( 'no_src_mac' )
      self.circuitLevelMismatch = data.pop( 'ckt_lev_mismatch' )
      self.neighbourSysIdChanged = data.pop( 'ngb_sysid_changed' )
      self.dupSysIdDetected = data.pop( 'dup_sysid_detected' )
      self.missingAreas = data.pop( 'missing_areas' )
      self.ngbSysIdMismatchInRestartTLVWithRA = data.pop( 'ngb_sysid_mismatch_'
                                                          'in_ra' )
      self.iidTlvMismatch = data.pop( 'iid_tlv_error' )

class IsisInterfaceDropReasonsL1LanHellos( IsisInterfaceDropReasonsLanHellos ):
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_INTERFACE_DROP_COUNTERS_REASONS_IIH_LAN_L1

class IsisInterfaceDropReasonsL2LanHellos( IsisInterfaceDropReasonsLanHellos ):
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_INTERFACE_DROP_COUNTERS_REASONS_IIH_LAN_L2

class IsisInterfaceDropReasonsLsps( IsisInterfaceDropReasons ):
   circuitLevelMismatch = Int( help='The circuit level indicated in the PDU does '
                               ' not match with the level on which the IS-IS '
                               'interface receiving this PDU is configured on. ',
                               optional=True )
   nonAdjSource = Int( help='LSPs, CSNPs or PSNPs are dropped with this reason. '
                       'The system receiving this PDU does not have an adjacency '
                       'with the system sending this PDU. ', optional=True )
   parsingFailure = Int( help='LSPs are dropped with this reasons. The options in'
                         ' the LSP could not be parsed properly, indicating that '
                         'the received LSP is corrupted. ', optional=True )
   invalidPurge = Int( help='LSPs are dropped with this reason. The Purge LSP must '
                       ' contain only the Header (or) the Header and Auth option. If'
                       ' it contains anything more or less, it is dropped. ',
                       optional=True )
   badLspLength = Int( help='Length of the LSP is greater than the length of LSP '
                       'buffer of the receiving system. ', optional=True )
   badChecksum = Int( help='The checksum indicated in the LSP is wrong indicating '
                      'that the Lsp is corrupted. ', optional=True )
   ownLspPurge = Int( help='Self-originated LSP of the system is received with a '
                      'zero-life but the remaining life of the LSP as indicated in '
                      'the LSP database is non-zero. The system drops this LSP and '
                      'regenerates the LSP with a newer sequence number. ',
                      optional=True )
   olderPurgeLsp = Int( help=' Purge LSP with a sequence number older than that in'
                          ' the LSP database of the system is received. The LSP is '
                          ' dropped and the latest LSP is flooded. ' )
   ownExpiredLsp = Int( help='LSP is received with the system Id in the LSPID '
                        'matching the system Id of the system receiving it, but the '
                        'LSP is no longer present in the LSP database of the system.'
                        'This LSP might be from the previous incarnation of the '
                        'system. The LSP is dropped and is force expired. ',
                        optional=True )
   differentChecksumLsp = Int( help='LSP received has samilar sequence number as '
                               'that in the LSP database of the system receiving the'
                               ' LSP, but their checksums are different. The LSP is'
                               ' dropped. If the LSP is self generated then it is '
                               'regenerated with a newer sequence number. If it is'
                               ' not self generated then it is force expired. ',
                               optional=True )
   ownNewerSequenceLsp = Int( help='Self generated LSP with a newer sequence number'
                              ' has been received. It is dropped and a new LSP is '
                              'regenerated and flooded', optional=True )
   olderLsp = Int( help='LSP with older sequence number than that of '
                   'the corresponding LSP in the database of the system is received'
                   '. The LSP is dropped and the latest LSP is flooded into the '
                   'circuit on which the LSP was received ', optional=True )

   def processData( self, data ):
      self.badLength = data.pop( 'bad_len' )
      self.malformedTlv = data.pop( 'malformed_tlv' )
      self.authFailure = data.pop( 'auth_failure' )
      self.circuitLevelMismatch = data.pop( 'ckt_lev_mismatch' )
      self.nonAdjSource = data.pop( 'non_adj_src' )
      self.parsingFailure = data.pop( 'parsing_failure' )
      self.invalidPurge = data.pop( 'invalid_purge' )
      self.badLspLength = data.pop( 'bad_lsp_length' )
      self.badChecksum = data.pop( 'bad_cksum' )
      self.ownLspPurge = data.pop( 'own_lsp_purge' )
      self.olderPurgeLsp = data.pop( 'older_purge_lsp' )
      self.ownExpiredLsp = data.pop( 'own_exp_lsp' )
      self.differentChecksumLsp = data.pop( 'diff_cksum_lsp' )
      self.ownNewerSequenceLsp = data.pop( 'newer_seq_lsp' )
      self.olderLsp = data.pop( 'older_lsp' )
      self.iidTlvMismatch = data.pop( 'iid_tlv_error' )

class IsisInterfaceDropReasonsL1Lsps( IsisInterfaceDropReasonsLsps ):
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_INTERFACE_DROP_COUNTERS_REASONS_LSP_L1

class IsisInterfaceDropReasonsL2Lsps( IsisInterfaceDropReasonsLsps ):
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_INTERFACE_DROP_COUNTERS_REASONS_LSP_L2

class IsisInterfaceDropReasonsSnps( IsisInterfaceDropReasons ):
   circuitLevelMismatch = Int( help='The circuit level indicated in the PDU does '
                               ' not match with the level on which the IS-IS '
                               'interface receiving this PDU is configured on. ',
                               optional=True )
   badOptionLength = Int( help='Length of one or more of the optional TLVs in the '
                          'PDU is different from that indicated in the Length '
                          'section of that TLV', optional=True )
   invalidLspEntryOption = Int( help='CSNPs and PSNPs are dropped with this reason.'
                                ' one or more of the  LSP entry sub-TLVs in the PDU'
                                ' has an invalid length. ', optional=True )
   nonAdjSource = Int( help='LSPs, CSNPs or PSNPs are dropped with this reason. '
                       'The system receiving this PDU does not have an adjacency '
                       'with the system sending this PDU. ', optional=True )

class IsisInterfaceDropReasonsCsnps( IsisInterfaceDropReasonsSnps ):
   def processData( self, data ):
      self.badLength = data.pop( 'bad_len' )
      self.malformedTlv = data.pop( 'malformed_tlv' )
      self.authFailure = data.pop( 'auth_failure' )
      self.circuitLevelMismatch = data.pop( 'ckt_lev_mismatch' )
      self.nonAdjSource = data.pop( 'non_adj_src' )
      self.badOptionLength = data.pop( 'bad_opt_len' )
      self.invalidLspEntryOption = data.pop( 'inv_lsp_entry_opt' )
      self.iidTlvMismatch = data.pop( 'iid_tlv_error' )

class IsisInterfaceDropReasonsL1Csnps( IsisInterfaceDropReasonsCsnps ):
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_INTERFACE_DROP_COUNTERS_REASONS_CSNP_L1

class IsisInterfaceDropReasonsL2Csnps( IsisInterfaceDropReasonsCsnps ):
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_INTERFACE_DROP_COUNTERS_REASONS_CSNP_L2

class IsisInterfaceDropReasonsPsnps( IsisInterfaceDropReasonsSnps ):
   psnpOnNonDisSystem = Int( help='PSNP is received on a circuit of the system of '
                             'type broadcast but system is not a DIS on that '
                             'circuit', optional=True )
   def processData( self, data ):
      self.badLength = data.pop( 'bad_len' )
      self.malformedTlv = data.pop( 'malformed_tlv' )
      self.authFailure = data.pop( 'auth_failure' )
      self.circuitLevelMismatch = data.pop( 'ckt_lev_mismatch' )
      self.badOptionLength = data.pop( 'bad_opt_len' )
      self.invalidLspEntryOption = data.pop( 'inv_lsp_entry_opt' )
      self.psnpOnNonDisSystem = data.pop( 'psnp_on_non_dis_system' )
      self.nonAdjSource = data.pop( 'non_adj_src' )
      self.iidTlvMismatch = data.pop( 'iid_tlv_error' )

class IsisInterfaceDropReasonsL1Psnps( IsisInterfaceDropReasonsPsnps ):
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_INTERFACE_DROP_COUNTERS_REASONS_PSNP_L1

class IsisInterfaceDropReasonsL2Psnps( IsisInterfaceDropReasonsPsnps ):
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_INTERFACE_DROP_COUNTERS_REASONS_PSNP_L2

class IsisInterfaceDropCountersDetails( Model ):
   malformedPdusDropReasons = Submodel( valueType=
                                        IsisInterfaceDropReasonsMalformedPdus,
                                        help='Reasons which led to malformed PDUs'
                                        'getting dropped and their respective '
                                        'counts', optional=True )
   iihP2pDropReasons = Submodel( valueType=IsisInterfaceDropReasonsP2pHellos,
                                 help='Reasons which led IS-IS P2P Hellos'
                                 'getting dropped and their respective '
                                 'counts', optional=True )
   iihLanL1DropReasons = Submodel( valueType=IsisInterfaceDropReasonsL1LanHellos,
                                 help='Reasons which led to Level-1 IS-IS LAN Hellos'
                                 'getting dropped and their respective '
                                   'counts', optional=True )
   iihLanL2DropReasons = Submodel( valueType=IsisInterfaceDropReasonsL2LanHellos,
                                 help='Reasons which led to Level-2 IS-IS LAN Hellos'
                                 'getting dropped and their respective '
                                   'counts', optional=True )
   lspL1DropReasons = Submodel( valueType=IsisInterfaceDropReasonsL1Lsps,
                              help='Reasons which led to IS-IS Level-1 Link State '
                                'PDUs getting dropped and their respective counts',
                                optional=True )
   lspL2DropReasons = Submodel( valueType=IsisInterfaceDropReasonsL2Lsps,
                              help='Reasons which led to IS-IS Level-2 Link State '
                                'PDUs getting dropped and their respective counts',
                                optional=True )
   csnpL1DropReasons = Submodel( valueType=IsisInterfaceDropReasonsL1Csnps,
                              help='Reasons which led to Level-1 IS-IS Complete '
                                 'Sequence number packets getting dropped and their '
                                 'respective counts',
                                 optional=True )
   csnpL2DropReasons = Submodel( valueType=IsisInterfaceDropReasonsL2Csnps,
                              help='Reasons which led to Level-2 IS-IS Complete '
                                 'Sequence number packets getting dropped and their '
                                 'respective counts',
                                 optional=True )
   psnpL1DropReasons = Submodel( valueType=IsisInterfaceDropReasonsL1Psnps,
                              help='Reasons which led to IS-IS Level-1 Partial '
                                 'Sequence number packets getting dropped and their '
                                 'respective counts',
                                 optional=True )
   psnpL2DropReasons = Submodel( valueType=IsisInterfaceDropReasonsL2Psnps,
                              help='Reasons which led to IS-IS Level-2 Partial '
                                 'Sequence number packets getting dropped and their '
                                 'respective counts',
                                 optional=True )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_INTERFACE_DROP_COUNTERS_DETAILS

class IsisInterfaceDropCounters( Model ):
   malformedPdus = Int( help='Number of IS-IS PDUs received malformed. The received '
                        'PDU had either a wrong header length, wrong version, or the'
                        ' System Id length in the PDU is not supported by the system'
                        ' received' )
   iihP2p = Int( help='Number of IS-IS P2P Hello Packets Dropped' )
   iihLanL1 = Int( help='Number of Level-1 IS-IS Lan Hello Packets Dropped' )
   iihLanL2 = Int( help='Number of Level-2 IS-IS Lan Hello Packets Dropped' )
   csnpL1 = Int( help='Number of Level-1 IS-IS Complete Sequence Number Packets '
                 'dropped' )
   csnpL2 = Int( help='Number of Level-2 IS-IS Complete Sequence Number Packets '
                 'dropped' )
   psnpL1 = Int( help='Number of Level-1 IS-IS Partial Sequence Number Packets '
                 'dropped' )
   psnpL2 = Int( help='Number of Level-2 IS-IS Partial Sequence Number Packets '
                 'dropped' )
   lspL1 = Int( help='Number of Level-1 IS-IS Link State PDUs '
                 'dropped' )
   lspL2 = Int( help='Number of Level-2 IS-IS Link State PDUs '
                 'dropped' )
   details = Submodel( valueType=IsisInterfaceDropCountersDetails, help='Reasons of'
                       'drop for each PDU with their respective counts',
                       optional=True )

   def processData( self, data ):
      self.malformedPdus = data.pop( 'malformed_pdus' )
      self.iihP2p = data.pop( 'iih_p2p' )
      self.iihLanL1 = data.pop( 'iih_lan_l1' )
      self.iihLanL2 = data.pop( 'iih_lan_l2' )
      self.csnpL1 = data.pop( 'csn_l1' )
      self.csnpL2 = data.pop( 'csn_l2' )
      self.psnpL1 = data.pop( 'psn_l1' )
      self.psnpL2 = data.pop( 'psn_l2' )
      self.lspL1 = data.pop( 'lsp_l1' )
      self.lspL2 = data.pop( 'lsp_l2' )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_INTERFACE_DROP_COUNTERS

class IsisInterfacePacketCountersRetransmittedData( Model ):
   lspL1 = Int( help='Number of L1 LSPs' )
   lspL2 = Int( help='Number of L2 LSPs' )

class IsisInterfacePacketCountersData( Model ):
   iihL1 = Int( help='Number of L1 LAN IS-IS Hello Packets' )
   iihL2 = Int( help='Nunber of L2 LAN IS-IS Hello Packets' )
   iihP2p = Int( help='Number of p2p Hello Packets' )
   csnpL1 = Int( help='Number of L1 CSN Packets' )
   csnpL2 = Int( help='Number of L2 CSN Packets' )
   psnpL1 = Int( help='Number of L1 PSN Packets' )
   psnpL2 = Int( help='Number of L2 PSN Packets' )
   lspL1 = Int( help='Number of L1 LSPs' )
   lspL2 = Int( help='Number of L2 LSPs' )

class IsisInterfacePacketCounters( Model ):
   transmitted = Submodel( valueType=IsisInterfacePacketCountersData,
                           help='Transmitted packet counters' )
   received = Submodel( valueType=IsisInterfacePacketCountersData,
                        help='Received packet counters' )
   processed = Submodel( valueType=IsisInterfacePacketCountersData,
                         help='Processed packet counters' )
   retransmitted = Submodel( valueType=IsisInterfacePacketCountersRetransmittedData,
                             help='Retransmitted packet counters' )

   def processData( self, data ):
      transmittedData = IsisInterfacePacketCountersData()
      transmittedData.iihL1 = data.pop( 'iih-L1-tx' )
      transmittedData.iihL2 = data.pop( 'iih-L2-tx' )
      transmittedData.iihP2p = data.pop( 'iih-p2p-tx' )
      transmittedData.csnpL1 = data.pop( 'csn-L1-tx' )
      transmittedData.csnpL2 = data.pop( 'csn-L2-tx' )
      transmittedData.psnpL1 = data.pop( 'psn-L1-tx' )
      transmittedData.psnpL2 = data.pop( 'psn-L2-tx' )
      transmittedData.lspL1 = data.pop( 'lsp-L1-tx' )
      transmittedData.lspL2 = data.pop( 'lsp-L2-tx' )
      self.transmitted = transmittedData

      receivedData = IsisInterfacePacketCountersData()
      receivedData.iihL1 = data.pop( 'iih-L1-rx' )
      receivedData.iihL2 = data.pop( 'iih-L2-rx' )
      receivedData.iihP2p = data.pop( 'iih-p2p-rx' )
      receivedData.csnpL1 = data.pop( 'csn-L1-rx' )
      receivedData.csnpL2 = data.pop( 'csn-L2-rx' )
      receivedData.psnpL1 = data.pop( 'psn-L1-rx' )
      receivedData.psnpL2 = data.pop( 'psn-L2-rx' )
      receivedData.lspL1 = data.pop( 'lsp-L1-rx' )
      receivedData.lspL2 = data.pop( 'lsp-L2-rx' )
      self.received = receivedData

      processedData = IsisInterfacePacketCountersData()
      processedData.iihL1 = data.pop( 'iih-L1-processed' )
      processedData.iihL2 = data.pop( 'iih-L2-processed' )
      processedData.iihP2p = data.pop( 'iih-p2p-processed' )
      processedData.csnpL1 = data.pop( 'csn-L1-processed' )
      processedData.csnpL2 = data.pop( 'csn-L2-processed' )
      processedData.psnpL1 = data.pop( 'psn-L1-processed' )
      processedData.psnpL2 = data.pop( 'psn-L2-processed' )
      processedData.lspL1 = data.pop( 'lsp-L1-processed' )
      processedData.lspL2 = data.pop( 'lsp-L2-processed' )
      self.processed = processedData

      retransmittedData = IsisInterfacePacketCountersRetransmittedData()
      retransmittedData.lspL1 = data.pop( 'lsp-L1-retransmit' )
      retransmittedData.lspL2 = data.pop( 'lsp-L2-retransmit' )
      self.retransmitted = retransmittedData


   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_INTERFACE_PKT_COUNTERS
   
class IsisInterfaceCounters( Model ):
   _interfaceName = Interface( help='The IS-IS interface name' )
   packetCounters = Submodel( valueType=IsisInterfacePacketCounters,
                              help='Data pertaining to the packet counters',
                              optional=True )
   dropCounters = Submodel( valueType=IsisInterfaceDropCounters,
                            help='Data pertaining to IS-IS drop counters',
                            optional=True )
   lastResetTime = Int( optional=True, help='UTC timestamp when the counters are' +
                                             ' reset last' )

   def processData( self, data ):
      self._interfaceName = intfLongName( data.pop( 'interface-name' ) )
      if 'last-reset-time' in data:
         self.lastResetTime = data.pop( 'last-reset-time' )

   def getKey( self, data ):
      return intfLongName( data['interface-name'] )


class IsisInstanceCounters( Model ):
   _instanceName = Str( help='Instance Name' )
   _vrf = Str( help='Vrf Name' )
   interfaceCounters = GeneratorDict( valueType=IsisInterfaceCounters,
                                      keyType=Interface,
                                      help='Dictionary of Isis Interface counters'
                                      'keyed by interface name' )
   systemCounters = Submodel( valueType=IsisSystemCounters, 
                              help='Data pertaining to IS-IS System counters',
                              optional=True )
   def processData( self, data ):
      self._instanceName = data.pop( 'instance-name', None )
      self._vrf = data.pop( 'vrf-name' )

   def render( self ):
      def removePadding( table ):            
         table.noPadLeftIs( True )                
         table.padLimitIs( True )                 

      def getReasonsTable( reasonsObj ):
         reasonsDict = vars( reasonsObj )
         reasonsTable = None
         for key, count in reasonsDict.items():
            if not isinstance( getattr( reasonsObj, key ), int ):
               continue
            if count == 0:
               continue
            if not reasonsTable:
               reasonsTable = TableOutput.createTable( dropReasonsHeading )
               f4 = TableOutput.Format( justify='left' )
               f5 = TableOutput.Format( justify='right' )
               removePadding( f4 )      
               removePadding( f5 )
               reasonsTable.formatColumns( f4, f5 )
            reasonsTable.newRow( IsisCliHelper.PKT_DROP_REASON_MAP[ key ], count )
         return reasonsTable

      packetCountersHeadings = ( "Interface", "IIH L1", "IIH L2", "IIH P2P",
                                 "CSN L1", "CSN L2", "PSN L1", "PSN L2", "LSP L1",
                                 "LSP L2", "Last Reset Time" )
      packetCountersRetransmittedHeadings = ( "Interface", "LSP L1", "LSP L2",
                                              "Last Reset Time" )
      dropCountersHeadings = ( "Interface", "Malformed PDUs", "IIH L1", "IIH L2",
                               "IIH P2P", "CSN L1", "CSN L2", "PSN L1", "PSN L2",
                               "LSP L1", "LSP L2", "Last Reset Time" )
      sysCountersHeadings = ( "Events/Errors", "Level 1", "Level 2" )
      dropReasonsHeading = ( "Drop Reason", "Drop Count" )
      transmittedTable = TableOutput.createTable( packetCountersHeadings )
      receivedTable = TableOutput.createTable( packetCountersHeadings )
      processedTable = TableOutput.createTable( packetCountersHeadings )
      retransmittedTable = TableOutput.createTable( 
                              packetCountersRetransmittedHeadings )
      dropCountersTable = TableOutput.createTable( dropCountersHeadings )
      sysCountersTable = TableOutput.createTable( sysCountersHeadings )
      f1 = TableOutput.Format( maxWidth=18, justify='left', wrap=True )
      f2 = TableOutput.Format( maxWidth=10, wrap=True )
      f3 = TableOutput.Format( maxWidth=5, wrap=True )
      f6 = TableOutput.Format( justify='left' )
      f7 = TableOutput.Format( justify='right' )
      f8 = TableOutput.Format( justify='right' )
      removePadding( f1 )
      removePadding( f2 )
      removePadding( f3 )
      removePadding( f6 )
      removePadding( f7 )
      removePadding( f8 )
      transmittedTable.formatColumns( f1, f3, f3, f3, f3, f3, f3, f3, f3, f3 )
      receivedTable.formatColumns( f1, f3, f3, f3, f3, f3, f3, f3, f3, f3 )
      processedTable.formatColumns( f1, f3, f3, f3, f3, f3, f3, f3, f3, f3 )
      retransmittedTable.formatColumns( f1, f3, f3 )         
      dropCountersTable.formatColumns( f1, f2, f3, f3, f3, f3, f3, f3, f3, f3, f3 )
      sysCountersTable.formatColumns( f6, f7, f8 )
      pktCountersPresent = False
      dropCountersPresent = False
      sysCountersPresent = False
      dropReasonsTablePerIntf = OrderedDict()
      interfaceCounters = dict( self.interfaceCounters )
      for intf in sorted( interfaceCounters ):
         counters = interfaceCounters[ intf ]
         packetCounters = counters.packetCounters
         dropCounters = counters.dropCounters
         lastReset = "Never"
         if counters.lastResetTime is not None:
            lastReset = datetime.fromtimestamp( counters.lastResetTime ).strftime(
                                      '%Y-%m-%d %H:%M:%S' )
         if packetCounters != None: # pylint: disable=singleton-comparison
            pktCountersPresent = True
            transmitted = packetCounters.transmitted
            transmittedTable.newRow( intf, transmitted.iihL1, transmitted.iihL2,
                                     transmitted.iihP2p, transmitted.csnpL1,
                                     transmitted.csnpL2, transmitted.psnpL1,
                                     transmitted.psnpL2, transmitted.lspL1, 
                                     transmitted.lspL2, lastReset )
            received = packetCounters.received
            receivedTable.newRow( intf, received.iihL1, received.iihL2,
                                  received.iihP2p, received.csnpL1, received.csnpL2,
                                  received.psnpL1, received.psnpL2, received.lspL1,
                                  received.lspL2, lastReset )
            processed = packetCounters.processed
            processedTable.newRow( intf, processed.iihL1, processed.iihL2,
                                   processed.iihP2p, processed.csnpL1, 
                                   processed.csnpL2, processed.psnpL1, 
                                   processed.psnpL2, processed.lspL1,
                                   processed.lspL2, lastReset )
            retransmitted = packetCounters.retransmitted
            retransmittedTable.newRow( intf, retransmitted.lspL1,
                                       retransmitted.lspL2, lastReset )
         if dropCounters != None: # pylint: disable=singleton-comparison
            dropCountersPresent = True
            dropCountersTable.newRow( intf, dropCounters.malformedPdus,
                                      dropCounters.iihLanL1, dropCounters.iihLanL2,
                                      dropCounters.iihP2p, dropCounters.csnpL1,
                                      dropCounters.csnpL2, dropCounters.psnpL1,
                                      dropCounters.psnpL2, dropCounters.lspL1,
                                      dropCounters.lspL2, lastReset )
            details = dropCounters.details
            if details != None: # pylint: disable=singleton-comparison
               dropReasonTableDict = OrderedDict()
               for dropType in IsisCliHelper.isisDropPduTypes:
                  dropDetailType = IsisCliHelper.PKT_DROP_TYPE_MAP[ dropType ]
                  reasonsTable = getReasonsTable( getattr( details,
                                                           dropDetailType ) )
                  if reasonsTable:
                     dropReasonTableDict[ dropType ] = reasonsTable
               if dropReasonTableDict:
                  dropReasonsTablePerIntf[ intf ] = dropReasonTableDict
      # pylint: disable-next=singleton-comparison
      if self.systemCounters != None:             
         sysCountersPresent = True                
         sysCountersL1 = self.systemCounters.systemCountersL1 
         sysCountersL2 = self.systemCounters.systemCountersL2
         sysCountersTable.newRow( 'Corrupt LSPs detected in IS-IS database',
                                  sysCountersL1.corruptedLsps,
                                  sysCountersL2.corruptedLsps )
         sysCountersTable.newRow( 'IIH PDUs dropped due to maximum area addresses'
                                  ' mismatch',    
                                  sysCountersL1.maxAreaAddressMismatches,
                                  sysCountersL2.maxAreaAddressMismatches )
         sysCountersTable.newRow( 'Number of own LSP purges received',
                                  sysCountersL1.ownLspPurges,
                                  sysCountersL2.ownLspPurges )
         sysCountersTable.newRow( 'Number of own LSP purges sent',
                                  sysCountersL1.ownLspPurgesSent,
                                  sysCountersL2.ownLspPurgesSent )
         sysCountersTable.newRow( 'Number of others\' LSP purges sent',
                                  sysCountersL1.othersLspPurgesSent,
                                  sysCountersL2.othersLspPurgesSent )
         sysCountersTable.newRow( 'Number of sequence number skipped for LSPs',
                                  sysCountersL1.seqNumSkips,
                                  sysCountersL2.seqNumSkips )
         sysCountersTable.newRow( 'Number of SPF runs', sysCountersL1.spfRuns,
                                  sysCountersL2.spfRuns )
         sysCountersTable.newRow( 'Number of times LSP database has overloaded',
                                  sysCountersL1.databaseOverloads, 
                                  sysCountersL2.databaseOverloads )
         sysCountersTable.newRow( 'Number of times system exceeded maximum'
                                  ' sequence number',
                                  sysCountersL1.exceededMaxSeqNums,
                                  sysCountersL2.exceededMaxSeqNums )
         sysCountersTable.newRow( 'PDUs dropped due to authentication key'
                                  ' mismatch', sysCountersL1.authKeyFails,
                                  sysCountersL2.authKeyFails ) 
         sysCountersTable.newRow( 'PDUs dropped due to authentication type'
                                  ' mismatch', sysCountersL1.authTypeFails,
                                  sysCountersL2.authTypeFails )
         sysCountersTable.newRow( 'PDUs dropped due to failure of LSP parsing',
                                  sysCountersL1.lspErrors,
                                  sysCountersL2.lspErrors )
         sysCountersTable.newRow( 'PDUs dropped due to system ID length mismatch',
                                  sysCountersL1.idLenMismatches,
                                  sysCountersL2.idLenMismatches )
      print( f"IS-IS Instance: {self._instanceName} VRF: {self._vrf}" )
      if pktCountersPresent:
         print( "Packet Counters" )
         print( "---------------" )
         print( "Received" )
         print( receivedTable.output() )
         print( "Processed" )
         print( processedTable.output() )
         print( "Transmitted" )
         print( transmittedTable.output() )
         print( "Retransmitted" )
         print( retransmittedTable.output() )
      if dropCountersPresent:
         print( "Drop Counters" )
         print( "-------------" )
         print( dropCountersTable.output() )
      if dropReasonsTablePerIntf:
         print( "Drop Details" )
         print( "------------" )
         for intf, dropDetailDict in dropReasonsTablePerIntf.items():
            print( intf )
            print( "-----------" )
            for key, table in dropDetailDict.items() :
               print( key + ':' )
               print( "----------" )
               print( table.output() )
      if sysCountersPresent:
         print( "System Counters" )
         print( "---------------" )
         print( sysCountersTable.output() )

showCountersInstModel = IsisCliModelCommon. \
                         generateIsisDictCliModel( IsisInstanceCounters )
showCountersVRFsModel = generateVrfCliModel( showCountersInstModel,
                                    "ISIS instance information for all VRFs" )

class IsisSpfLogEntry( Model ):
   startTime = Float( help='SPF start time in UTC' )
   timeTaken = Float( optional=True, help='SPF run duration(ms)' )
   scheduled = Bool( optional=True, help='Scheduled SPF run' ) 
   ipv4Nodes = Int( optional=True,
                    help='Number of IPv4 routers that make up the topology' )
   ipv6Nodes = Int( optional=True,
                    help='Number of IPv6 routers that make up the topology' )
   triggers = Int( help='Number of SPF triggers consumed in this SPF run' )
   reason = Str( help='The first trigger that scheduled the SPF' )
   partial = Bool( optional=True, help='Partial SPF optimization' ) 

   def processData( self, data ):
      if 'duration' in data:
         self.timeTaken = data.pop( 'duration' ) / 1000.0
         self.ipv4Nodes = data.pop( 'nnodes-v4' )
         if 'nnodes-v6' in data:
            self.ipv6Nodes = data.pop( 'nnodes-v6' )
      else:
         self.scheduled = True
      self.startTime = float( data.pop( 'when' ) )
      self.triggers = data.pop( 'count' )
      self.reason = data.pop( 'reason' )
      self.partial = ( ord( data.pop( 'partial', chr( 0 ) ) ) != 0 )

   def renderData( self, _mtEnabled ):
      partialStr = "*" if self.partial else " "
      if self.timeTaken:
         startTimeStamp = datetime.fromtimestamp( self.startTime ).strftime(
                                                            '%Y-%m-%d %H:%M:%S.%f')
         if _mtEnabled:
            print( spfLogMtEntryFormatStr % (
               partialStr, startTimeStamp, self.timeTaken,
               self.ipv4Nodes, self.ipv6Nodes, self.triggers, self.reason ) )
         else:
            print( spfLogEntryFormatStr % (
               partialStr, startTimeStamp, self.timeTaken,
               self.ipv4Nodes, self.triggers, self.reason ) )
      else:
         if _mtEnabled:
            print( "   %s%-65s %-7d %-17s" % (
               partialStr, "(scheduled)", self.triggers, self.reason ) )
         else:
            print( "   %s%-53s %-7d %-17s" % (
               partialStr, "(scheduled)", self.triggers, self.reason ) )


class IsisLevelSpfLogModel( Model ):
   _mtEnabled = Bool( default=False, help='multi-topology enabled or not' )
   spfLogs = GeneratorList( valueType=IsisSpfLogEntry,
             help = 'ISIS SPF log entries' )

   def getKey( self, data ):
      return int( data[ 'level-num' ] )

   def processData( self, data ):
      self._mtEnabled = ord( data.pop( 'mt' ) ) != 0

   def renderData( self, level ):
      print( "  IS-IS SPF Log level-%s" % ( level ) )
      print( "  * - Partial SPF" )
      if self._mtEnabled:
         print( spfLogMtHeaderFormatStr % ( "Start time", "Duration(msec)",
               "Nodes(v4)", "Nodes(v6)", "Count", "Reason" ) )
      else:
         print( spfLogHeaderFormatStr % ( "Start time", "Duration(msec)",
               "Nodes", "Count", "Reason" ) )
      for logEntry in self.spfLogs:
         logEntry.renderData( self._mtEnabled )


class IsisSpfLogTableModel( Model ):
   _instanceName = Str( help='Instance Name' )
   _vrf = Str( help='Vrf Name' )
   levels = GeneratorDict( keyType=int, valueType=IsisLevelSpfLogModel,
           help='Dictionary of SPF logs keyed by level' )

   def processData( self, data ):
      self._vrf = data.pop( 'vrf-name' )
      self._instanceName = data.pop( 'instance-name', None )

   def render( self ):
      print( f"IS-IS Instance: {self._instanceName} VRF: {self._vrf}" )
      for level, entry in self.levels:
         entry.renderData( level )  

spfLogTableModel = IsisCliModelCommon. \
                   generateIsisDictCliModel( IsisSpfLogTableModel )
spfLogTableVRFsModel = generateVrfCliModel( spfLogTableModel,
                       "IS-IS instance information for all VRFs" )

# Models for 'show isis network topology'
class IsisTopologyEntryListModel( Model ):
   metric = Int( help='IS-IS metric value' )
   nexthopSystemId = Str( help='Next Hop' )
   interface = Interface( help='IS-IS interface name', default='' )
   snpa = Str( help='Subnetwork Point of Attachment of Neighbor' )
   nexthopHostname = Str( help='Hostname', optional=True )
   iaMetric = Int( help='IS-IS inter-area metric value' )
   igpShortcutTunnel = Submodel( valueType=TunnelInfo,
                                 help="IGP Shortcut tunnel", optional=True )

   def processData( self, data ):
      self.metric = data.get( 'metric' )
      self.nexthopSystemId = data.pop( 'next-hop' )
      ifName = data.pop( 'interface-name' )
      if ifName != 'N.A.':
         self.interface = intfLongName( ifName )
      self.snpa = data.pop( 'snpa' )
      self.nexthopHostname = data.pop( 'nh-hostname' )
      if self.nexthopHostname == self.nexthopSystemId:
         self.nexthopHostname = None
      self.iaMetric = data.get( 'intra-area-metric' )
      tunnelId = data.get( 'tunnelid' )
      if tunnelId:
         tacTunnelId = Tac.Value( "Tunnel::TunnelTable::TunnelId", int( tunnelId ) )
         tunInfo = TunnelInfo( tunnelIndex=tacTunnelId.tunnelIndex(),
                               tunnelType=tacTunnelId.typeCliStr(),
                               tunnelVias=None )
         self.igpShortcutTunnel = tunInfo

class IsisTopologyNextHopsEntryModel( Model ):
   vias = List( valueType=IsisTopologyEntryListModel,
                            help='Isis nexthops list' )
   destinationHostname = Str( help='Hostname', optional=True )
   _systemId = Str( help='SystemID' )
   _metric = Int( help='Isis Metric Value' )
   _iaMetric = Int( help='Isis Intra-Area Metric Value' )

   def getKey( self, data ):
      return data.get( 'system-id', self._systemId )
   
   def overrideHierarchy( self, data ):
      readNext = True
      if not self.vias:
         self._systemId = data[ 'system-id' ]
         self.destinationHostname = data[ 'sysid-hostname' ]
         if self.destinationHostname == self._systemId:
            self.destinationHostname = None
         self._metric = data[ 'metric' ]
         self._iaMetric = data[ 'intra-area-metric' ]
         nexthopData = IsisTopologyEntryListModel()
         nexthopData.processData( data )
         self.vias.append( nexthopData )
         return (nexthopData, readNext)
      elif self._systemId == self.getKey( data ):
         data[ 'metric' ] = self._metric
         data[ 'intra-area-metric' ] = self._iaMetric
         nexthopData = IsisTopologyEntryListModel()
         nexthopData.processData( data )
         self.vias.append( nexthopData )
         return ( nexthopData, readNext )
      else:
         return ( data, False )

   def renderEntry( self, systemId ):
      for model in self.vias:
         metricValue = " " if systemId == " " else str( model.metric )
         iaMetricValue = " " if systemId == " " else str( model.iaMetric )
         if model.nexthopHostname:
            model.nexthopSystemId = model.nexthopHostname
         if systemId != " " and self.destinationHostname:
            systemId = self.destinationHostname
         if model.igpShortcutTunnel:
            tunInfo = model.igpShortcutTunnel
            ifName = "%s tunnel index %d" % ( tunInfo.tunnelType,
                                              tunInfo.tunnelIndex )
         else:
            ifName = model.interface.stringValue
         if ifName == '':
            ifName = 'N.A.'
         print( "    %-16s %-8s %-9s %-16s %-24s %-17s"
                 % ( systemId, metricValue, iaMetricValue, model.nexthopSystemId,
                     ifName, model.snpa ) )
         systemId = " "


class IsisTopologyNextHopModel( Model ):
   nexthops = GeneratorDict( keyType=str, 
                                valueType=IsisTopologyNextHopsEntryModel,
                                help='Dict of next-hops keyed by system-id' )
   _addrFamily = Int( help='Address Family' )
   _mt = Bool( default=False, help='MultiTopology Enabled or not' )
   _level = Int( help='level entry of ipv4/ipv6' )

   def processData( self, data ):
      self._addrFamily = data[ 'address-family' ]
      self._mt = ord( data[ 'mt' ] ) != 0
      self._level = int( data[ 'level' ] )
   
   def getKey( self, data ):
      # https://tools.ietf.org/html/rfc5120#section-7.5
      # we will return topology-id as key for address-family
      # In case of Non-Mt topology-id returned will be "0"
      # In case of Mt 
      #         if address-family is 'ipv6' return "2"
      #         else if address-family is 'ipv4' return "0:
      if ord(data[ 'mt' ]) != 0 and data[ 'address-family' ] == 2:
         return "2"
      return "0"

   def renderEntry( self ):
      ipstr = ""
      if self._mt:
         ipstr = 'IPv4 ' if self._addrFamily == 1 else 'IPv6 '

      print( "  IS-IS " + ipstr + "paths to level-" +
             str( self._level ) + " routers" )
      print( "    %-16s %-8s %-9s %-16s %-24s %-17s" % ( "System Id", "Metric",
                                                        "IA Metric", "Next-Hop",
                                                        "Interface", "SNPA" ) )
      for systemId, model in self.nexthops:
         model.renderEntry( systemId )

class IsisTopologyEntryModel( Model ):
   topologies = GeneratorDict( keyType=str, valueType=IsisTopologyNextHopModel,
                              help='Dict of address-families keyed by topology-id' )
   
   def getKey( self, data ):
      return int( data[ 'level' ] )

   def renderEntry( self ):
      for _, model in self.topologies:
         model.renderEntry()

class IsisTopologyModel( Model ):
   _vrf = Str( help='Vrf Name' )
   _instanceName = Str( help='Instance Name' )
   levels = GeneratorDict( keyType=int, valueType=IsisTopologyEntryModel,
                          help='A dict of isis Topology keyed by isis level' )
   def processData( self, data ):
      self._vrf = data.pop( 'vrf-name' )
      self._instanceName = data.pop( 'instance-name', None )

   def render( self ):
      print( "" )
      print( f"IS-IS Instance: {self._instanceName} VRF: {self._vrf}" )
      for _, model in self.levels:
         model.renderEntry()

isisTopologyModel = IsisCliModelCommon . \
                    generateIsisDictCliModel( IsisTopologyModel )
isisTopologyVRFsModel = generateVrfCliModel( isisTopologyModel,
                        "Topology information for all VRFs" )

class IsisHostnameEntryModel( Model ):
   hostname = Str( help='Hostname' )
   level = Enum( values=IsisCliHelper.LEVEL_SHORT_MAP.values(),
                 help='IS type' )
   nodeUnreachable = Bool( default=False, help="Node is unreachable" )
   hostnameConflict = Bool( default=False,
                            help="LSP originated by node "
                            "with conflicting hostname" )

   def processData( self, data ):
      self.nodeUnreachable = ( ord( data.pop( 'node-unreachable' ) ) != 0 )
      self.hostnameConflict = ( ord( data.pop( 'hname-conflict' ) ) != 0 )
      return data

   def renderEntry( self, systemId ):
      def getLegendCodes():
         codes = ''
         if self.hostnameConflict:
            codes += 'H'
         if self.nodeUnreachable:
            codes += 'U'
         return codes

      print( " %-2s %-6s %-19s %-s" % ( getLegendCodes(),
                     self.level, systemId, self.hostname ) )
      
   def getKey( self, data ):
      return str( data[ 'system-id' ] )

class IsisHostnameInstanceModel( Model ):
   _vrf = Str( help='Vrf Name' )
   _instanceName = Str( help='Instance Name' )
   systemIds = GeneratorDict( keyType=str, valueType=IsisHostnameEntryModel, 
                             help='Dictionary of IS-IS hostnames keyed by SystemID' )

   def processData( self, data ):
      self._vrf = data.pop( 'vrf-name' )
      self._instanceName = data.pop( 'instance-name', None )

   def render( self ):
      print( "Legend:" )
      print( "H - hostname conflict" )
      print( "U - node unreachable" )
      print( "" )
      print( f"IS-IS Instance: {self._instanceName} VRF: {self._vrf}" )
      print( "    Level  System ID           Hostname" )
      for systemId, model in self.systemIds:
         model.renderEntry( systemId )
      print( "" )


isisHostnameModel = IsisCliModelCommon. \
                     generateIsisDictCliModel( IsisHostnameInstanceModel )
isisHostnameVRFModel = generateVrfCliModel( isisHostnameModel,
                     "Hostname information for all VRFs" )

class IsisMplsLdpSyncInterfaceModel( Model ):
   _instanceName = Str( help='Instance Name' )
   ldpEnabled = Bool( help='LDP Enabled' )
   syncRequired = Bool( help='LDP sync required' )
   syncAchieved = Bool( help='LDP sync achieved', optional=True )
   igpNotifyDelay = Int( help='IGP notify delay in seconds. Immediate if zero',
                         optional=True )
   holddownTime = Int( help='Holddown time in seconds. '
                            'Hold until established if zero',
                       optional=True )

   def toSecondsStr( self, seconds, delay=False ):
      if seconds == 0:
         return "immediate" if delay else "infinite"
      return "%d second%s" % ( seconds, "" if seconds == 1 else "s" )

   def renderEntry( self, instanceName, interface ):
      print( "Interface: " + interface + "; IS-IS instance: " + instanceName )
      ldpEnabled = "No"
      required = "No"
      achieved = "No"
      if self.ldpEnabled:
         ldpEnabled = "Yes"
      if self.syncRequired:
         required = "Yes"
      if self.syncAchieved is None:
         achieved = "Not applicable"
      elif self.syncAchieved:
         achieved = "Yes"

      print( " Ldp Enabled: " + ldpEnabled )
      print( " SYNC information:" )
      print( f"   Required: {required}, Achieved: {achieved}" )
      if self.igpNotifyDelay is not None:
         print( "   IGP Notify Delay: %s" %
                self.toSecondsStr( self.igpNotifyDelay, delay=True ) ) 
      if self.holddownTime is not None:
         print( "   Holddown time: %s" % self.toSecondsStr( self.holddownTime ) )
      print( "" )

class IsisMplsLdpSyncModel( Model ):
   _vrf = Str( 'Vrf Name' )
   _instanceName = Str( 'Instance Name' )
   interfaces = Dict( keyType=Interface, valueType=IsisMplsLdpSyncInterfaceModel,
                      help='MPLS LDP sync information keyed by interface name' )

   def render( self ):
      for interfaceName, interface in self.interfaces.items():
         interface.renderEntry( self._instanceName, interfaceName )

isisMplsLdpSyncModel = IsisCliModelCommon.\
                           generateIsisDictCliModel( IsisMplsLdpSyncModel )
isisMplsLdpSyncVRFsModel = generateVrfCliModel( isisMplsLdpSyncModel,
                                        "MPLS LDP sync information for all vrfs")
# CAPI model for 'show isis lsp purges'

PURGE_GEN_REASON_MAP = {
   "lspLifetimeExpired" : "LSP lifetime expired",
   "receivedOldIncarnation" : "received old incarnation of own LSP",
   "differentChecksum" : "LSP has a different checksum",
   "disResigned" : "resigned as DIS",
   "noContents" : "no contents",
   "userClearedLsp" : "LSP cleared by user",
   "obsoleteProxyLsp" : "received old incarnation of our own Proxy LSP",
   "systemIdChanged": "IS-IS System ID changed"
}

class IsisLspPurgesEntryModel( Model ):
   __revision__ = 2
   logTime = Float( help='Log time stamp' )
   lspId = Str( help='IS-IS LSPID of purge' )
   _vrf = Str( help='VRF Name' )
   _instanceId = Int( help='IS-IS Instance ID' )
   originatorSystemId = Str( help='Purge Originator or receiver SystemID',
                             optional=True )
   originatorHostname = Str( help='Purge Originator or receiver Dynamic Hostname',
                             optional=True )
   upstreamNgbSystemId = Str( help='Neighbour SystemID who send purge',
                              optional=True )
   upstreamNgbHostname = Str( help='Neighbour Hostname who send purge', 
                              optional=True )
   reason = Enum( values=PURGE_GEN_REASON_MAP,
                  help='Reason for purge generation', optional=True )
   lspIdHostname = Str( help='Hostname of the source node of LSP which was purged',
                        optional=True )
   
   def processData( self, data ):
      self.logTime = float( data.pop( 'time' ) )
      self._vrf = data.pop( 'vrf-name' )
      self.lspId = data.pop( 'lspid' )
      self._instanceId = data.pop( 'instance-id' )
      self.lspIdHostname = data.pop( 'lspIdHostname', None )
      if 'originatorSystemId' in data:
         self.originatorSystemId = data.pop( 'originatorSystemId' )
         self.originatorHostname = data.pop( 'originatorHostname', None )
      if 'upstreamNgbSystemId' in data:
         self.upstreamNgbSystemId = data.pop( 'upstreamNgbSystemId' )
         self.upstreamNgbHostname = data.pop( 'upstreamNgbHostname', None )
      if 'reason' in data:
         self.reason = data.pop( 'reason' )
             
   def renderData( self ):
      logTimestampValue = datetime.fromtimestamp( self.logTime ).strftime( 
                                                      '%Y-%m-%d %H:%M:%S.%f' )
      helperOriginator = self.originatorSystemId
      if self.originatorSystemId != 'self' and self.originatorHostname is not None:
         helperOriginator = helperOriginator + ' (' + self.originatorHostname + ')'
      helperUpstreamNgb = self.upstreamNgbSystemId
      if self.upstreamNgbHostname:
         helperUpstreamNgb = helperUpstreamNgb + \
                                    ' (' + self.upstreamNgbHostname + ')'
      helperLspId = self.lspId
      if self.lspIdHostname:
         helperLspId = helperLspId + ' (' + self.lspIdHostname + ')'

      output = "%s Purge of %s received"
      if self.originatorSystemId != 'self' and helperUpstreamNgb is None:
         print( ( output + ", originated by %s" ) % ( logTimestampValue,
                  helperLspId, helperOriginator ) )
      elif self.originatorSystemId != 'self' and \
           helperUpstreamNgb is not None:
         print( ( output + ", flooded by %s towards %s" ) % ( logTimestampValue,
                  helperLspId, helperUpstreamNgb, helperOriginator ) )
      elif self.originatorSystemId == 'self' and helperUpstreamNgb is not None:
         print( ( output + ", flooded by %s towards self" ) % (
                  logTimestampValue, helperLspId, helperUpstreamNgb ) )
      else:
         print( "{} Purge of {} generated by self, {}".format(
            logTimestampValue, helperLspId, 
            PURGE_GEN_REASON_MAP.get( self.reason, "unknown" ) ) )

   def degrade( self, dictRepr, revision ):
      # In old revision '1', 'lspIdHostname' was named as 'lspIdWithHostname'
      if revision == 1:
         if 'lspIdHostname' in dictRepr:
            dictRepr[ 'lspIdWithHostname' ] = dictRepr[ 'lspIdHostname' ]
            del dictRepr[ 'lspIdHostname' ]
      return dictRepr

class IsisLevelLspPurgesModel( Model ):
   purgeLogs = GeneratorList( valueType=IsisLspPurgesEntryModel, 
               help='List of IS-IS purge entries' )
   
   def getKey( self, data ):
      return int( data[ 'level-num' ] )

   def renderData( self, level ):
      print( "IS-IS Level-%s" % ( level ) )
      for purgeEntry in self.purgeLogs:
         purgeEntry.renderData()

class IsisLspPurgesModel( Model ):
   _instanceName = Str( help='IS-IS Instance Name' )
   _vrf = Str( help='VRF Name' )
   levels = GeneratorDict( keyType=int, valueType=IsisLevelLspPurgesModel, 
                           help='Dictionary of purges keyed by level' )
   
   def processData( self, data ):
      self._vrf = data.pop( 'vrf-name' )
      self._instanceName = data.pop( 'instance-name', None )

   def render( self ):
      print( f"IS-IS Instance: {self._instanceName} VRF: {self._vrf}" )
      for level, entry in self.levels:
         entry.renderData( level )
      
isisLspPurgesModel = IsisCliModelCommon.generateIsisDictCliModel(
                  IsisLspPurgesModel, revision=2 )
isisLspPurgesVRFsModel = generateVrfCliModel( isisLspPurgesModel,
                           "IS-IS instance information for all VRFs" )

# CAPI model for 'show isis graceful-restart'
class IsisGrNgbListModel( Model ):
   level = Enum( values=IsisCliHelper.LEVEL_SHORT_MAP.values(),
                 help='IS-IS level' )
   interface = Interface( help='IS-IS interface name' )
   gracefulRestartCapable = Bool( help='Graceful-Restart Capable' )
   status = Enum( values=IsisCliHelper.GRACEFUL_RESTART_STATUS_MAP,
                  help='Graceful-Restart State' )

   def ngbGrState( self, status, gr_capable ):
      if not gr_capable:
         return "na"
      if (status) & ( ISIS_GR_RCVD_SA ):
         return "starting"
      elif (status) & ( ISIS_GR_RCVD_RR ):
         return "reStarting"
      else:
         return "running"

   def getLevel( self, levelType ):
      if levelType == "Level 1 IS":
         return "L1"
      elif levelType == "Level 2 IS":
         return "L2"
      elif levelType == "Level 1 and 2 IS":
         return "L1L2"
      return levelType

   def processData( self, data ):
      self.level = self.getLevel( data.pop( 'type' ) )
      self.interface = intfLongName( data.pop( 'interface-name' ) )
      grC = structUnpack1B( data.pop( 'gr-capable' ) )
      grS = structUnpack1B( data.pop( 'gr-state' ) )

      self.gracefulRestartCapable = bool( grC )
      self.status = self.ngbGrState( grS, grC )

class IsisGracefulRestartNgbEntryModel( Model ):
   hostname = Str( 'Hostname of Neighbor', optional=True )
   adjacencies = List( valueType=IsisGrNgbListModel,
                       help='List of adjacency data for a neighbor' )
   _systemId = Str( 'SystemId' )


   def getKey( self, data ):
      return data[ 'system-id' ]

   def overrideHierarchy( self, data ):
      readNext = True
      if not self.adjacencies:
         self._systemId = data[ 'system-id' ]
         self.hostname = data.get( 'hostname', None )
         entryData = IsisGrNgbListModel()
         entryData.processData( data )
         self.adjacencies.append( entryData )
         return ( entryData, readNext )
      elif self._systemId == self.getKey( data ):
         entryData = IsisGrNgbListModel()
         entryData.processData( data )
         self.adjacencies.append( entryData )
         return ( entryData, readNext )
      else:
         return ( data, False )

   def renderEntry( self, systemId ):
      if self.hostname is not None:
         systemId = self.hostname
      for model in self.adjacencies:
         grC = IsisCliHelper.YES_NO_MAP[ bool( model.gracefulRestartCapable ) ] 
         status = IsisCliHelper.GRACEFUL_RESTART_STATUS_MAP[ model.status ]
         print( grNgbHeaderFormatStr % ( systemId, model.level,
                                        model.interface.stringValue,
                                        grC, status ) )

class IsisGracefulRestartModel( Model ):
   _vrf = Str( help='vrf-name' )
   _instanceName = Str( help='IS-IS instance name' )
   systemId = Str( help='SystemId' )
   gracefulRestart = Enum( values=( IsisCliHelper.ENABLED_DISABLED_MAP ),
                           help='Graceful Restart' )
   gracefulRestartHelper = Enum( values=IsisCliHelper.ENABLED_DISABLED_MAP,
                                 help='Graceful Restart Helper' )
   state = Enum( values=GRACEFUL_RESTART_STATE_MAP.values(),
                 help='Graceful Restart State' )
   t1Timer = Int( help='T1 timer - Time after which an unacknowledged start or ' +
                       'restart attempt will be repeated (in seconds)' )
   t3Timer = Int( help='T3 timer - Time after which the router will declare that ' +
                       'it has failed to achieve database synchronization during ' +
                       'restart (in seconds)',
                       optional=True)
   t2ConfigLevel1 = Int( help='T2 timer - Maximum time to wait for LSP database ' +
                              'synchronization for level-1 (in seconds) ' )
   t2Level1 = Int( help='Remaining T2 for level-1 (in seconds)', optional=True )
   t2ConfigLevel2 = Int( help='T2 timer - Maximum time to wait for LSP database ' +
                              'synchronization for level-2 (in seconds) ' )
   t2Level2 = Int( help='Remaining T2 for level-2 (in seconds)', optional=True )

   grNeighbors = GeneratorDict( keyType=str,
                                 valueType=IsisGracefulRestartNgbEntryModel,
                                 help='A Dict of GracefulRestart Neighbors ' +  
                                      'entries keyed by systemID' )
   grState = Enum( values=GR_TS_STATE_MAP.values(), help='Graceful Restart State',
                   optional=True )
   t2L1Stopped = Float( help='T2 timer - time when Level-1 stopped', optional=True )
   t2L2Stopped =Float( help='T2 timer - time when Level-2 stopped', optional=True )
   t3Stopped = Float( help='T3 timer - time when stopped', optional=True )


   def processData( self, data ):
      self._vrf = data.pop( 'vrf-name' )
      self._instanceName = data.pop( 'instance-name', None )
      self.systemId = data.pop( 'system-id' )
      self.gracefulRestart = ( "enabled"
                               if structUnpack1B( data.pop( "graceful-restart" ) )
                               else "disabled" )
      self.gracefulRestartHelper = ( "enabled"
                                     if structUnpack1B( data.pop( "gr-helper" ) )
                                     else "disabled" )
      self.state = data.pop( 'gr-state' )
      self.t1Timer = 3

      self.t2ConfigLevel1 = data.pop( 'gr-cfg-t2-l1' )
      if 'gr-t2-l1' in data:
         self.t2Level1 = data.pop( 'gr-t2-l1' )

      self.t2ConfigLevel2 = data.pop( 'gr-cfg-t2-l2' )
      if 'gr-t2-l2' in data:
         self.t2Level2 = data.pop( 'gr-t2-l2' )

      if 'gr-t3' in data:
         self.t3Timer = data.pop( 'gr-t3' )

      if 'gr-ts-state' in data:
         self.grState = GR_TS_STATE_MAP[ data.pop( 'gr-ts-state' ) ]
      else:
         self.grState = None
    
      if 'gr-t2-l1-ts' in data:
         self.t2L1Stopped = float( data.pop( 'gr-t2-l1-ts' ) )
      else:
         self.t2L1Stopped = None

      if 'gr-t2-l2-ts' in data:
         self.t2L2Stopped = float( data.pop( 'gr-t2-l2-ts' ) )
      else:
         self.t2L2Stopped = None

      if 'gr-t3-ts' in data:
         self.t3Stopped = float( data.pop( 'gr-t3-ts' ) )
      else:
         self.t3Stopped = None


   def render( self ):
      print( "" )
      print( f"IS-IS Instance: {self._instanceName} VRF: {self._vrf}" )
      print( "  System ID: %s" % self.systemId )
      print( "  Graceful Restart: %s, Graceful Restart Helper: %s" %
               ( IsisCliHelper.ENABLED_DISABLED_MAP[ self.gracefulRestart ], 
                 IsisCliHelper.ENABLED_DISABLED_MAP[ self.gracefulRestartHelper ] ) )
      print("  State: %s" % self.state )
      print("  T1 : 3s" )
      if self.grState is not None:
         tsmsg = ", " + self.grState + " at "
      t2l1 = str( self.t2ConfigLevel1 ) + "s/"
      if self.t2Level1 is not None:
         t2l1 += str( self.t2Level1 ) + "s remaining"
      else:
         t2l1 += "not running"
      if self.t2L1Stopped is not None:
         updateTimeStamp = datetime.fromtimestamp(
                  self.t2L1Stopped ).strftime( '%Y-%m-%d %H:%M:%S' )
         t2l1 += tsmsg
         t2l1 += updateTimeStamp
      t2l2 = str( self.t2ConfigLevel2 ) + "s/"
      if self.t2Level2 is not None:
         t2l2 += str( self.t2Level2 ) + "s remaining"
      else:
         t2l2 += "not running"
      if self.t2L2Stopped is not None:
         updateTimeStamp = datetime.fromtimestamp(
                  self.t2L2Stopped ).strftime( '%Y-%m-%d %H:%M:%S' )
         t2l2 += tsmsg
         t2l2 += updateTimeStamp
      t3 = "not running"
      if self.t3Timer is not None:
         t3 = str( self.t3Timer ) + "s"
      if self.t3Stopped is not None:
         updateTimeStamp = datetime.fromtimestamp(
                  self.t3Stopped ).strftime( '%Y-%m-%d %H:%M:%S' )
         t3 += tsmsg
         t3 += updateTimeStamp
      print( "  T2 (level-1) : %s" % t2l1 )
      print( "  T2 (level-2) : %s" % t2l2 )
      print( "  T3 : %s" % t3 )
      print( "" )
      print( grNgbHeaderFormatStr % ( "System ID", "Type", "Interface",
                                     "Restart Capable", "Status" ) )

      for systemId, model in self.grNeighbors:
         model.renderEntry( systemId )

isisGracefulRestartModel = IsisCliModelCommon. \
                           generateIsisDictCliModel( IsisGracefulRestartModel )
isisGracefulRestartVRFsModel = generateVrfCliModel( isisGracefulRestartModel,
                              "Graceful Restart information for all VRFs" )

class IsisUloopDelayedV4RouteModel( Model ):
   v4DestAddr = Ip4Address( help="IPv4 destination address of delayed route" )
   maskLength = Int( help="Mask Length" )
   algorithm = Str( help="Algorithm identifier", optional=True )

   def processData( self, data ):
      from CliPlugin.RoutingIsisCli import flexAlgoIdToName
      self.v4DestAddr = data.pop( 'ipv4-dest' )
      self.maskLength = data.pop( 'masklen' )
      algo = data.pop( 'algorithm' )
      if algo != '\x00':
         self.algorithm = flexAlgoIdToName( ord( algo ) )
      else:
         self.algorithm = None

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_ULOOP_DELAYED_V4_ROUTES
   
   def render( self ):
      if self.algorithm:
         print( f"        {self.v4DestAddr}/{self.maskLength}, algorithm "
                f"{self.algorithm}" )
      else:
         print( f"        {self.v4DestAddr}/{self.maskLength}" )

class IsisUloopDelayedV6RouteModel( Model ):
   v6DestAddr = Ip6Address( help="IPv6 destination address of delayed route" )
   maskLength = Int( help="Mask Length" )
   algorithm = Str( help="Algorithm identifier", optional=True )

   def processData( self, data ):
      from CliPlugin.RoutingIsisCli import flexAlgoIdToName
      self.v6DestAddr = data.pop( 'ipv6-dest' )
      self.maskLength = data.pop( 'masklen' )
      algo = data.pop( 'algorithm' )
      if algo != '\x00':
         self.algorithm = flexAlgoIdToName( ord( algo ) )
      else:
         self.algorithm = None

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_ULOOP_DELAYED_V6_ROUTES
   
   def render( self ):
      if self.algorithm:
         print( f"        {self.v6DestAddr}/{self.maskLength}, algorithm "
                f"{self.algorithm}" )
      else:
         print( f"        {self.v6DestAddr}/{self.maskLength}" )

class IsisUloopSmModel( Model ):
   level = Int( help='Level' )
   lastAttempt = Bool( default=False, help='Last attempt' )
   state = Enum( values=IsisCliHelper.ULOOP_SM_STATE_RENDER_MAP,
                 help='State of Micro-loop prevention attempt' )
   cause = Enum( values=IsisCliHelper.ULOOP_SM_CAUSE_RENDER_MAP,
                 help='Reason for Micro-loop prevention attempt' )
   result = Enum( values=IsisCliHelper.ULOOP_SM_RESULT_RENDER_MAP,
                  help='Result of Micro-loop prevention attempt' )
   intfName = Interface( help='Protected interface' )
   intfIndex = Int( help='Interface index' )
   protectionV4 = Enum( values=IsisCliHelper.TILFA_PROTECTION_MAP.values(),
                        help='TI-LFA protection mode for IPv4' )
   protectionV6 = Enum( values=IsisCliHelper.TILFA_PROTECTION_MAP.values(),
                        help='TI-LFA protection mode for IPv6' )
   delayV4 = Int( help='IPv4 local convergence delay in msecs' )
   delayV6 = Int( help='IPv6 local convergence delay in msecs' )
   routesDelayedV4 = Int( help='Number of IPv4 routes delayed' )
   routesDelayedV6 = Int( help='Number of IPv6 routes delayed' )
   delayTimerStartedV4 = Int( help='Time delay timer started for IPv4 routes' )
   delayTimerStoppedV4 = Int( help='Time delay timer stopped for IPv4 routes' )
   delayTimeRemainingV4 = Int( help='Delay time remaining for IPv4 routes' )
   delayTimerStartedV6 = Int( help='Time delay timer started for IPv6 routes' )
   delayTimerStoppedV6 = Int( help='Time delay timer stopped for IPv6 routes' )
   delayTimeRemainingV6 = Int( help='Delay time remaining for IPv6 routes' )
   adjacencyDownTimestamp = Int( help='Link or BFD failure timestamp' )

   delayedRoutesV4 = GeneratorList( valueType=IsisUloopDelayedV4RouteModel,
                                    help='List of IPv4 routes delayed',
                                    optional=True )
   delayedRoutesV6 = GeneratorList( valueType=IsisUloopDelayedV6RouteModel,
                                    help='List of IPv6 routes delayed',
                                    optional=True )

   def processData( self, data ):
      self.level = data.pop( 'level' )
      self.lastAttempt = ord( data.pop( 'last-attempt' ) ) != 0
      self.state = data.pop( 'state' )
      if self.state == "delayNotRunning":
         return
      self.cause = data.pop( 'cause' )
      self.result = data.pop( 'result' )
      # uloop does not use the gated interface structure, so
      # can not report the EOS interface name.
      self.intfName = intfLongName( data.pop( 'ifl-name' ) )
      self.intfIndex = data.pop( 'ifl-index' )
      self.protectionV4 = IsisCliHelper.TILFA_PROTECTION_MAP[
         ord( data.pop( 'protection-mode-v4' ) ) ]
      self.protectionV6 = IsisCliHelper.TILFA_PROTECTION_MAP[
         ord( data.pop( 'protection-mode-v6' ) ) ]
      self.delayV4 =  data.pop( 'delay-v4' )
      self.delayV6 =  data.pop( 'delay-v6' )
      self.routesDelayedV4 = data.pop( 'routes-delayed-v4' )
      self.routesDelayedV6 = data.pop( 'routes-delayed-v6' )
      self.delayTimerStartedV4 = data.pop( 'delay-timer-started-v4' )
      self.delayTimerStoppedV4 = data.pop( 'delay-timer-stopped-v4' )
      self.delayTimeRemainingV4 = data.pop( 'delay-time-remaining-v4' )
      self.delayTimerStartedV6 = data.pop( 'delay-timer-started-v6' )
      self.adjacencyDownTimestamp = data.pop( 'adjacency-down-timestamp' )
      self.delayTimerStoppedV6 = data.pop( 'delay-timer-stopped-v6' )
      self.delayTimeRemainingV6 = data.pop( 'delay-time-remaining-v6' )

   def renderEntry( self, detail ):
      if self.state == "delayNotRunning":
         return
      stateStr = IsisCliHelper.ULOOP_SM_STATE_RENDER_MAP.get( self.state, "unknown" )
      print( "" )
      print( "  Level %d" % self.level, end=' ' )
      if self.lastAttempt:
         print( "last attempt due to {} on {},".format(
               IsisCliHelper.ULOOP_SM_CAUSE_RENDER_MAP.get( self.cause, "unknown" ),
               self.intfName.stringValue ), end=' ' )
         if self.result == "delaySucceeded":
            print( IsisCliHelper.ULOOP_SM_RESULT_RENDER_MAP[ self.result ] )
         else:
            # all abort results reported simply as "Failed"
            print( "Failed" )
      else:
         print( "in progress due to {} on {}".format(
               IsisCliHelper.ULOOP_SM_CAUSE_RENDER_MAP.get( self.cause, "unknown" ),
               self.intfName.stringValue ) )

      print( "    Adjacency failure at %s" % datetime.fromtimestamp( 
               self.adjacencyDownTimestamp ).strftime( '%Y-%m-%d %H:%M:%S' ) )

      if self.protectionV4 == "disabled":
         print( "    TI-LFA protection is disabled for IPv4" )
      else:
         print( "    TI-LFA %s protection is enabled for IPv4"  %
                self.protectionV4 )
         if not self.lastAttempt and self.state == 'delaySpfScheduled':
            print( "    %s" % stateStr )
         elif self.state in [ 'delayRoutes', 'delayRoutesNewSpf' ]:
            print( "    IPv4 Routes delayed: %d"  % self.routesDelayedV4 )
         if self.delayTimerStartedV4:
            timeStamp = datetime.fromtimestamp( self.delayTimerStartedV4 ).strftime(
                                                   '%Y-%m-%d %H:%M:%S')
            print( "      Delay timer started at:", timeStamp )
         if self.delayTimeRemainingV4:
            print( "      Delay timer expires in %d secs" %
                   self.delayTimeRemainingV4 )
         if self.delayTimerStoppedV4:
            timeStamp = datetime.fromtimestamp( self.delayTimerStoppedV4 ).strftime(
                                                   '%Y-%m-%d %H:%M:%S')
            print( "      Delay timer stopped at:", timeStamp )
         if ( detail and
              self.state in [ 'delayRoutes', 'delayRoutesNewSpf' ] and 
              self.routesDelayedV4 > 0 ):
            print( "      Delayed routes:" )
            for route in self.delayedRoutesV4:
               route.render()

      if self.protectionV6 == "disabled":
         print( "    TI-LFA protection is disabled for IPv6" )
      else:
         print( "    TI-LFA %s protection is enabled for IPv6"  %
                self.protectionV6 )
         if not self.lastAttempt and self.state == 'delaySpfScheduled':
            print( "    %s" % stateStr )
         elif self.state in [ 'delayRoutes', 'delayRoutesNewSpf' ]:
            print( "    IPv6 Routes delayed: %d"  % self.routesDelayedV6 )
         if self.delayTimerStartedV6:
            timeStamp = datetime.fromtimestamp( self.delayTimerStartedV6 ).strftime(
                                                   '%Y-%m-%d %H:%M:%S')
            print( "      Delay timer started at:", timeStamp )
         if self.delayTimeRemainingV6:
            print( "      Delay timer expires in %d secs" %
                   self.delayTimeRemainingV6 )
         if self.delayTimerStoppedV6:
            timeStamp = datetime.fromtimestamp( self.delayTimerStoppedV6 ).strftime(
                                                   '%Y-%m-%d %H:%M:%S')
            print( "      Delay timer stopped at:", timeStamp )
         if ( detail and
              self.state in [ 'delayRoutes', 'delayRoutesNewSpf' ] and 
              self.routesDelayedV6 > 0 ):
            print( "      Delayed routes:" )
            for route in self.delayedRoutesV6:
               route.render()

class IsisUloopModel( Model ):
   _vrf = Str( help='vrf-name' )
   _instanceName = Str( help='IS-IS instance name' )
   _printDetails = Bool( default=False, help='Display details' )
   systemId = Str( help='System ID' )
   cfgDelayV4 = Int( help='IPv4 local convergence delay' )
   cfgDelayV6 = Int( help='IPv6 local convergence delay' )
   delayAttemptsL1 = Int( help='Number of attempts at level-1' )
   delayAbortsL1 = Int( help='Number of failures at level-1' )
   delayAttemptsL2 = Int( help='Number of attempts at level-2' )
   delayAbortsL2 = Int( help='Number of failures at level-2' )

   uloopSms = GeneratorList( valueType=IsisUloopSmModel,
                             help='List of Micro-loop SMs' )

   def processData( self, data ):
      self._vrf = data.pop( 'vrf-name' )
      self._instanceName = data.pop( 'instance-name', None )
      self.systemId = data.pop( 'system-id' )
      self.cfgDelayV4 = data.pop( 'cfg-delay-v4' )
      self.cfgDelayV6 = data.pop( 'cfg-delay-v6' )
      self.delayAttemptsL1 = data.pop( 'delay-attempts-l1' )
      self.delayAbortsL1 = data.pop( 'delay-aborts-l1' )
      self.delayAttemptsL2 = data.pop( 'delay-attempts-l2' )
      self.delayAbortsL2 = data.pop( 'delay-aborts-l2' )

   def render( self ):
      print( "" )
      print( f"IS-IS Instance: {self._instanceName} VRF: {self._vrf}" )
      print( "  System ID: %s" % self.systemId )
      if self.cfgDelayV4 == 0 and self.cfgDelayV6 == 0:
         print( "  Micro-loop local convergence delay is not configured" )
      else:
         if self.cfgDelayV4:
            print( "  IPv4 local convergence delay configured, %d msecs"
                   % self.cfgDelayV4 )
         if self.cfgDelayV6:
            print( "  IPv6 local convergence delay configured, %d msecs"
                   % self.cfgDelayV6 )
         print( "  Level 1 attempts %d, failures %d" % (
               self.delayAttemptsL1, self.delayAbortsL1 ) )
         print( "  Level 2 attempts %d, failures %d" % (
               self.delayAttemptsL2, self.delayAbortsL2 ) ) 

      for entry in self.uloopSms:
         entry.renderEntry( self._printDetails )

isisUloopModel = IsisCliModelCommon.generateIsisDictCliModel( IsisUloopModel )
isisUloopVRFsModel = generateVrfCliModel( isisUloopModel,
                              "Micro-loop information for all VRFs" )

class TilfaTunnelTableEntry( Model ):
   tunnelType = Enum( values=tunnelTypeEnumValues, help="Tunnel type" )
   tunnelIndex = Int( help="Tunnel index within tunnel type" )
   vias = List( valueType=MplsVia, help="List of tunnel vias" )
   backupVias = List( valueType=MplsVia,
                      help="List of tunnel backup vias" )
   tunnelId = Int( help="Tunnel identifier" )

   def render( self ):
      print( "Tunnel Index %d" % self.tunnelIndex )
      for via in self.vias:
         via.render()
      for via in self.backupVias:
         via.render()

class TilfaTunnelTable( Model ):
   entries = Dict( keyType=int, valueType=TilfaTunnelTableEntry,
                   help="Tunnel table entries keyed by tunnel index" )

   def render( self ):
      for _, tunnelTableTilfaEntry in sorted( self.entries.items() ):
         tunnelTableTilfaEntry.render()

class IsisDynFloodNodeEntryModel( Model ):
   nodeId = Str( 'Node ID' )

   def processData( self, data ):
      self.nodeId = data.pop( 'node' )

   def getKey( self, data ):
      return int( data.get( 'index' ) )

class IsisDynFloodNodeLevelModel( Model ):
   nodes = GeneratorDict( keyType=int,
                          valueType=IsisDynFloodNodeEntryModel,
                          help='A dict of IS-IS nodes keyed by index' )

   def getKey( self, data ):
      return ord( data.get( 'level' ) )

   def render( self ):
      indexNodeFormat = "%-15s %-20s"
      print( " " * 4, indexNodeFormat % ( "Index", "Node ID" ) )
      # print the node in order of the index
      for index, node in sorted( self.nodes, key=itemgetter( 0 ) ):
         print( " " * 4, indexNodeFormat % ( index, node.nodeId ) )

class IsisDynFloodNodeModel( Model ):
   _instanceName = Str( help='Instance Name' )
   _vrf = Str( help='VRF Name' )
   levels = GeneratorDict( keyType=int,
                           valueType=IsisDynFloodNodeLevelModel,
                           help="A dict of IS-IS nodes keyed by level" )

   def processData( self, data ):
      self._instanceName = data.pop( 'instance-name', None )
      self._vrf = data.pop( 'vrf-name' )

   def render( self ):
      print( "" )
      print( f"IS-IS Instance: {self._instanceName} VRF: {self._vrf}" )
      for level, levelModel in self.levels:
         print( " " * 2, "Level %s Nodes:" % level )
         levelModel.render()

isisDynFloodNodeModel = IsisCliModelCommon.generateIsisDictCliModel(
   IsisDynFloodNodeModel )
isisDynFloodNodeVRFsModel = generateVrfCliModel(
   isisDynFloodNodeModel, "Dynamic flooding nodes for all VRFs" )

class IsisDynFloodPathsEntryModel( Model ):
   path = Str( help='Path' )

   def processData( self, data ):
      pathBytes = data.pop( 'path' )
      if isinstance( pathBytes, str ):
         pathBytes = pathBytes.encode()
      pathNode = ( struct.unpack( "!H", pathBytes[ i: i + 2 ] )
                   for i in range( 0, len( pathBytes ), 2 ) )
      self.path = " ".join( "%d" % i for i in pathNode )

   def render( self ):
      if self.path:
         output = "    Path: " + self.path
         while len( output ) > LINE_MAX:
            index = output.rfind( " ", 0, LINE_MAX )
            print( output[ 0 : index ] )
            output = ' ' * 6 + output[ index + 1 : ]
         if len ( output ) > 8:
            print( output )

class IsisDynFloodPathsLevelModel( Model ):
   paths = GeneratorList( valueType=IsisDynFloodPathsEntryModel,
                          help='A list of IS-IS paths' )

   def getKey( self, data ):
      return ord( data.get( 'level' ) )

   def render( self ):
      for path in self.paths:
         path.render()

class IsisDynFloodPathsModel( Model ):
   _instanceName = Str( help='Instance Name' )
   _vrf = Str( help='VRF Name' )
   levels = GeneratorDict( keyType=int,
                           valueType=IsisDynFloodPathsLevelModel,
                           help="A dict of lists of IS-IS paths keyed by level" )

   def processData( self, data ):
      self._instanceName = data.pop( 'instance-name', None )
      self._vrf = data.pop( 'vrf-name' )

   def render( self ):
      print( "" )
      print( f"IS-IS Instance: {self._instanceName} VRF: {self._vrf}" )
      for level, levelModel in self.levels:
         print( "  Level %s:" % level )
         levelModel.render()

isisDynFloodPathsModel = IsisCliModelCommon.generateIsisDictCliModel(
   IsisDynFloodPathsModel )
isisDynFloodPathsVRFsModel = generateVrfCliModel(
   isisDynFloodPathsModel, "Dynamic flooding paths for all VRFs" )

class IsisDynFloodTopologyEntryModel( Model ):
   name = Str( help='A node in a path in the flooding topology' )

   def processData( self, data ):
      self.name = data.pop( 'name', None )

class IsisDynFloodTopologyPathModel( Model ):
   index = Int( help="Path index" )
   path = GeneratorList( valueType=IsisDynFloodTopologyEntryModel,
                         help='A path in the flooding topology' )

   def processData( self, data ):
      self.index = data.pop( 'path', 0 )

   def format( self, string ):
      output = "    Path: " + string
      while len( output ) > LINE_MAX:
         index = output.rfind( " ", 0, LINE_MAX )
         print( output[ 0 : index ] )
         output = ' ' * 4 + output[ index + 1 : ]
      if len ( output ) > 8:
         print( output )

   def render( self ):
      string = ""
      for node in self.path:
         if string:
            string += " "
         string += node.name
      if string:
         self.format( string )

class IsisDynFloodTopologyLevelModel( Model ):
   topology = GeneratorList( valueType=IsisDynFloodTopologyPathModel,
                             help='A list of IS-IS flooding paths' )

   def getKey( self, data ):
      return ord( data[ 'level' ] )

   def render( self ):
      for path in self.topology:
         path.render()

class IsisDynFloodTopologyModel( Model ):
   _instanceName = Str( help='Instance Name' )
   _vrf = Str( help='VRF Name' )
   levels = GeneratorDict( keyType=int,
                           valueType=IsisDynFloodTopologyLevelModel,
                           help="A dict of lists of IS-IS flooding " +
                           "topology keyed by level" )

   def processData( self, data ):
      self._instanceName = data.pop( 'instance-name', None )
      self._vrf = data.pop( 'vrf-name' )

   def render( self ):
      print( "" )
      print( f"IS-IS Instance: {self._instanceName} VRF: {self._vrf}" )
      for level, levelModel in self.levels:
         print( "  Level %s:" % level )
         levelModel.render()

isisDynFloodTopologyModel = IsisCliModelCommon.generateIsisDictCliModel(
   IsisDynFloodTopologyModel )
isisDynFloodTopologyVRFsModel = generateVrfCliModel(
   isisDynFloodTopologyModel, "Dynamic flooding topology for all VRFs" )

class IsisDynFloodInterfacesEntryModel( Model ):
   intf = Interface( help='Interface name' )
   tempIntf = Bool( help='Temporary interface addition to the flooding topology' )

   def processData( self, data ):
      self.intf = intfLongName( data.pop( 'name' ) )
      self.tempIntf = ord( data.pop( 'temp' ) ) != 0

   def render( self ):
      print( '{}{}'.format( self.intf.stringValue, ' (temporary)'
                       if self.tempIntf else '' ) )

class IsisDynFloodInterfacesLevelModel( Model ):
   interfaces = GeneratorList( valueType=IsisDynFloodInterfacesEntryModel,
                          help='A list of IS-IS flooding paths' )

   def getKey( self, data ):
      return ord( data[ 'level' ] )

   def render( self ):
      for item in self.interfaces:
         print( '   ', end=' ' )
         item.render()

class IsisDynFloodInterfacesModel( Model ):
   _instanceName = Str( help='Instance Name' )
   _vrf = Str( help='VRF Name' )
   levels = GeneratorDict( keyType=int,
                           valueType=IsisDynFloodInterfacesLevelModel,
                           help="A dict of lists of IS-IS flooding " +
                           "interfaces keyed by level" )

   def processData( self, data ):
      self._instanceName = data.pop( 'instance-name', None )
      self._vrf = data.pop( 'vrf-name' )

   def render( self ):
      print( "" )
      print( f"IS-IS Instance: {self._instanceName} VRF: {self._vrf}" )
      for level, levelModel in self.levels:
         print( "  Level %s:" % level )
         levelModel.render()

isisDynFloodInterfacesModel = IsisCliModelCommon.generateIsisDictCliModel(
   IsisDynFloodInterfacesModel )
isisDynFloodInterfacesVRFsModel = generateVrfCliModel(
   isisDynFloodInterfacesModel, "Dynamic flooding interfaces for all VRFs" )

#
# Area Proxy Models
#
class IsisShowAreaProxyEntryModel( Model ):
   proxySysId = Str( help='Area proxy system identifier' )
   proxyHostname = Str( help='Area proxy hostname' )
   status = Str( help='Area proxy status' )

   def processData( self, data ):
      self.proxySysId = data.get( 'proxy-sysid', '' )
      self.proxyHostname = data.get( 'proxy-hostname', '' )
      self.status = data.get( 'status', '' )

   def render( self ):
      print( "  System Id: %s" % self.proxySysId )
      print( "  Hostname: %s" % self.proxyHostname )
      print( "  %s" % self.status )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_AREA_PROXY_ENTRY

class IsisShowAreaProxyModel( Model ):
   _instanceName = Str( help='Instance Name' )
   _vrf = Str( help='VRF Name' )
   entry = Submodel( valueType=IsisShowAreaProxyEntryModel,
                     help='Show area proxy information' )

   def processData( self, data ):
      self._instanceName = data.pop( 'instance-name', None )
      self._vrf = data.pop( 'vrf-name' )

   def render( self ):
      print( "" )
      print( f"IS-IS Instance: {self._instanceName} VRF: {self._vrf}" )
      if self.entry:
         self.entry.render()

isisShowAreaProxyModel = IsisCliModelCommon.generateIsisDictCliModel(
   IsisShowAreaProxyModel )
isisShowAreaProxyVRFsModel = generateVrfCliModel(
   isisShowAreaProxyModel, "Area proxy for all VRFs" )

#
# Flex Algo show command models
#
class IsisShowFlexAlgoEntryBase( Model ):
   level = Str( help="IS-IS Level" )
   algorithm = Str( help="Algorithm identifier" )
   advertised = Bool( help="Definition advertised" )
   metric = Enum( values=FLEX_ALGO_METRIC_CAPI_MAP.values(),
                  help="Metric type" )
   system = Str( help="System name" )

   def processData( self, data ):
      def levelName( level ):
         if level == 1:
            return 'L1'
         if level == 2:
            return 'L2'
         return 'L1L2'

      def metricName( metricType ):
         return FLEX_ALGO_METRIC_CAPI_MAP.get(
            metricType, str( metricType ) )

      from CliPlugin.RoutingIsisCli import flexAlgoIdToName
      value = data.get( 'level' )
      self.level = levelName( ord( value ) ) if value else None
      value = data.get( 'algorithm' )
      self.algorithm = flexAlgoIdToName( ord( value ) ) if value else None
      self.system = data.get( 'name' )
      value = data.get( 'metric-type' )
      self.metric = metricName( ord( value ) ) if value else None
      value = data.get( 'advertised' )
      self.advertised = bool( ord( value ) ) if value else False

class IsisShowFlexAlgoSummaryEntryModel( IsisShowFlexAlgoEntryBase ):

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_FLEX_ALGO_ENTRY

class IsisShowFlexAlgoAlternate( Model ):
   name = Str( help="System name" )

   def processData( self, data ):
      self.name = data.get( 'alternate' )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_FLEX_ALGO_ALTERNATE

class IsisShowFlexAlgoDetailEntryModel( IsisShowFlexAlgoEntryBase ):
   priority = Int( help="Priority of the definition" )
   includeAll = Str( help="List of administrative groups a link must include" )
   includeAny = Str( help="List of administrative groups a link may include" )
   exclude = Str( help="List of administrative groups a link must exclude" )
   includeAllDecimal = List( valueType=int,
                             help="List of administrative groups a link must "
                             "include" )
   includeAnyDecimal = List( valueType=int,
                             help="List of administrative groups a link may "
                             "include" )
   excludeDecimal = List( valueType=int,
                          help="List of administrative groups a link must exclude" )
   excludeSrlg = Str( help="List of shared risk link groups that must be excluded" )
   color = Int( help="Color for resulting paths" )
   alternates = GeneratorList( valueType=IsisShowFlexAlgoAlternate,
                               help='Other definition advertisers' )

   def processData( self, data ):
      def bitMaskStrOrNone( key ):
         value = data.get( key )
         if not value:
            return None
         if len( value ) > 1:
            return bitMaskCollToStr( value )
         return bitMaskToStr( value[ 0 ] ) if value[ 0 ] else None

      from CliPlugin.TeCli import getSrlgIdToNameMap
      super().processData( data )
      value = data.get( 'priority' )
      self.priority = ord( value ) if value else None
      self.includeAll = bitMaskStrOrNone( 'include-all' )
      self.includeAllDecimal = adminGroupDictToDecimalList(
            dict( enumerate( data.get( 'include-all', [] ) ) ) )
      self.includeAny = bitMaskStrOrNone( 'include-any' )
      self.includeAnyDecimal = adminGroupDictToDecimalList(
            dict( enumerate( data.get( 'include-any', [] ) ) ) )
      self.exclude = bitMaskStrOrNone( 'exclude' )
      self.excludeDecimal = adminGroupDictToDecimalList(
            dict( enumerate( data.get( 'exclude', [] ) ) ) )
      srlgs = data.get( 'srlg' )
      if srlgs is not None:
         longs = sorted( socket.ntohl( g ) for g in srlgs )
         srlgMap = getSrlgIdToNameMap()
         strs = [ srlgMap.get( g, str( g ) ) for g in longs ]
         srlgs = ' '.join( strs )
      self.excludeSrlg = srlgs
      self.color = data.get( 'color' )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_FLEX_ALGO_DETAIL

   def renderMetric( self, metric ):
      return IsisCliHelper.FLEX_ALGO_METRIC_RENDER_MAP.get( metric,
            metric )

   def renderAdministrativeGroups( self ):
      if not self.includeAny and not self.includeAll and not self.exclude:
         return
      cmd = 'administrative-group'
      if self.includeAll:
         cmd += ' include all ' + self.includeAll
      if self.includeAny:
         cmd += ' include any ' + self.includeAny
      if self.exclude:
         cmd += ' exclude ' + self.exclude
      print( cmd )

   def render( self ):
      heads = [ "Algorithm", "Priority", "Level", "Advertiser" ]
      leftJustify = TableOutput.Format( justify='left' )
      leftJustify.padLimitIs( 1 )
      rightJustify = TableOutput.Format( justify='right' )
      rightJustify.padLimitIs( 1 )

      table = TableOutput.createTable( heads )
      table.formatColumns( leftJustify, rightJustify, leftJustify, leftJustify,
                           rightJustify )
      table.newRow( self.algorithm, self.priority, self.level, self.system )
      print( table.output() )
      print( 'Parameters:' )
      if self.advertised:
         print( 'advertised' )
      print( 'metric', self.renderMetric( self.metric ) )
      self.renderAdministrativeGroups()
      if self.excludeSrlg is not None:
         print( 'srlg exclude', self.excludeSrlg )
      if self.color is not None:
         print( 'color %u' % self.color )
      print( '' )
      names = [ alt.name for alt in self.alternates ]
      if names:
         print( 'Other advertisers:', ', '.join( names ) )

class IsisShowFlexAlgoRouterEntry( Model ):
   router = Str( help='Router name or system id' )
   advertising = Bool( help='Router advertising an algorithm definition '
                       'for this algorithm' )
   priority = Int( help='Priority of the advertised algorithm',
                   optional=True )

   def processData( self, data ):
      self.router = data.pop( 'router' )
      adv = data.get( 'advertised' )
      if adv:
         self.advertising = bool( ord( adv ) )
         priority = data.get( 'priority' )
         if priority is not None:
            self.priority = ord( priority )
      else:
         self.advertising = False

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_FLEX_ALGO_ROUTER_ENTRY

class IsisShowFlexAlgoRouterModel( Model ):
   level = Enum( values=IsisCliHelper.LEVEL_ORD_SHORT_MAP.values(),
                 help='IS-IS Level' )
   algorithm = Str( help="Algorithm identifier" )
   routers = GeneratorList( valueType=IsisShowFlexAlgoRouterEntry,
                            help='Flex-algo algorithm router entry' )

   def processData( self, data ) :
      from CliPlugin.RoutingIsisCli import flexAlgoIdToName
      self.level = IsisCliHelper.LEVEL_ORD_SHORT_MAP[ ord( data.pop( 'level' ) ) ]
      self.algorithm = flexAlgoIdToName( ord( data.pop( 'algorithm' ) ) )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_FLEX_ALGO_ROUTERS

class IsisShowFlexAlgoModel( Model ):
   _instanceName = Str( help='Instance Name' )
   _vrf = Str( help='VRF Name' )
   summaries = GeneratorList( valueType=IsisShowFlexAlgoSummaryEntryModel,
                              help='Flex-algo algorithm summaries' )
   details = GeneratorList( valueType=IsisShowFlexAlgoDetailEntryModel,
                            help='Flex-algo algorithm details' )
   routers = GeneratorList( valueType=IsisShowFlexAlgoRouterModel,
                            help='Flex-algo participating router' )

   def processData( self, data ):
      self._instanceName = data.pop( 'instance-name', None )
      self._vrf = data.pop( 'vrf-name' )

   def render( self ):

      leftJustify = TableOutput.Format( justify='left' )
      leftJustify.padLimitIs( 1 )
      rightJustify = TableOutput.Format( justify='right' )
      rightJustify.padLimitIs( 1 )

      def renderMetric( metric ):
         return IsisCliHelper.FLEX_ALGO_METRIC_RENDER_MAP.get( metric,
                                                               metric )
      def renderAdvertised( advertised ):
         return 'yes' if advertised else 'no'

      def renderHeader():
         print( "" )
         print( f"IS-IS Instance: {self._instanceName} VRF: {self._vrf}" )
         print( "" )

      def renderSummaries():
         sumCount = 0
         summaryHeadings = [ "Algorithm", "Advertised", "Level", "Metric",
                             "Selected" ]
         summaryTable = TableOutput.createTable( summaryHeadings )
         summaryTable.formatColumns( leftJustify, leftJustify, leftJustify,
                                     leftJustify, leftJustify )
         for entry in self.summaries:
            summaryTable.newRow( entry.algorithm,
                                 renderAdvertised( entry.advertised ),
                                 entry.level, renderMetric( entry.metric ),
                                 entry.system )
            sumCount += 1
         if sumCount:
            print( summaryTable.output() )
         return sumCount

      def renderDetails():
         count = 0
         for entry in self.details:
            entry.render()
            count += 1
         return count

      def renderRouters():
         routerHeadings = [ 'Router', 'Level', 'Advertising', 'Priority' ]

         # We can only traverse GeneratorLists once, so collect all data into
         # one structure so that we can transpose it.
         data = {}
         for entry in self.routers:
            level = entry.level
            algorithm = entry.algorithm
            if not data.get( algorithm ):
               data.update( { algorithm: {} } )
            if not data[ algorithm ].get( level ):
               data[ algorithm ].update( { level:{} } )
            for detail in entry.routers:
               cell = { detail.router : Munch ( { 'router' : detail.router,
                                                  'level' : entry.level,
                                                  'advertising' : detail.advertising,
                                                  'priority' : detail.priority } ) }
               data[ algorithm ][ level ].update( cell )

         # Display the data
         for algo in sorted( data.keys() ):
            print( 'Algorithm:', algo )
            print( '' )

            routers = []
            for level in sorted( data[ algo ].keys() ) :
               keys = sorted( data[ algo ][ level ].keys() )
               subset = [ data[ algo ][ level ][ key ] for key in keys ]
               routers += subset

            routerTable = TableOutput.createTable( routerHeadings )
            routerTable.formatColumns( leftJustify, leftJustify, leftJustify,
                                       rightJustify )
            for entry in routers:
               routerTable.newRow( entry.router, entry.level,
                                   'yes' if entry.advertising else 'no',
                                   entry.priority if entry.priority is not None
                                   else '' )
            print( routerTable.output() )

      renderHeader()
      if renderSummaries():
         return
      if renderDetails():
         return
      renderRouters()

isisShowFlexAlgoModel = IsisCliModelCommon.generateIsisDictCliModel(
   IsisShowFlexAlgoModel )
isisShowFlexAlgoVRFsModel = generateVrfCliModel(
   isisShowFlexAlgoModel, "Flex-algo for all VRFs" )
