# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Model, Submodel
from CliModel import ( GeneratorDict,
                       GeneratorList,
                       Int,
                       Str )
from CliPlugin import IsisCliModelCommon
from CliPlugin.VrfCli import generateVrfCliModel
from CliPlugin.IsisCliModels import Srv6LocatorFlags
from ArnetModel import IpGenericPrefix

class IsisSrv6LocatorInstance( Model ):
   systemId = Str( help="System ID" )
   hostName = Str( help="Hostname", optional=True )
   algorithm = Int( help="Locator algorithm" )
   metric = Int( help="Cost of the Locator reachability" )
   flags = Submodel( valueType=Srv6LocatorFlags, help="Locator flags" )

class IsisSrv6Locator( Model ):
   locatorInstances = GeneratorList( valueType=IsisSrv6LocatorInstance,
                            help="SRv6 Locator info advertised by routers" )
   # pylint: disable-msg=arguments-differ
   def render( self, topoId, level, prefix ):
      for locInst in self.locatorInstances:
         if locInst.algorithm == 0:
            algoString = "SPF (0)"
         elif locInst.algorithm == 1:
            algoString = "Strict SPF (1)"
         else:
            algoString = f"{locInst.algorithm}"
         print( f"Locator: {prefix}" )
         if locInst.hostName:
            print( f"System ID: {locInst.hostName}" )
         else:
            print( f"System ID: {locInst.systemId}" )
         print( f"Flags: {locInst.flags.toStr()}" )
         print( f"Level: {level}" )
         # pylint: disable-next=consider-using-f-string
         print( "Algorithm: {}, Topology: {}\n".format( algoString,
                                                                 topoId ) )

class IsisSrv6LevelLocators( Model ):
   locatorPrefixes = GeneratorDict( keyType=IpGenericPrefix,
                            valueType=IsisSrv6Locator,
                            help="SRv6 Locators in the IS-IS level keyed by prefix" )
   # pylint: disable-msg=arguments-differ
   def render( self, topoId, level ):
      for prefix, locator in self.locatorPrefixes:
         locator.render( topoId, level, prefix )

class IsisSrv6TopologyLocators( Model ):
   levels = GeneratorDict( keyType=int, valueType=IsisSrv6LevelLocators,
                         help="SRv6 Locators in the IS-IS topology keyed by level" )
   # pylint: disable-msg=arguments-differ
   def render( self, topoId ):
      for level, levelLocators in self.levels:
         levelLocators.render( topoId, level )

class IsisInstanceSrv6Locators( Model ):
   topologies = GeneratorDict( keyType=int, valueType=IsisSrv6TopologyLocators,
                 help="SRv6 Locators in the IS-IS instance keyed by Topology Id" )
   def render( self ):
      for topoId, topoLocators in self.topologies:
         topoLocators.render( topoId )

IsisSrv6LocatorsModel = IsisCliModelCommon. \
                            generateIsisDictCliModel( IsisInstanceSrv6Locators )
IsisSrv6LocatorsVRFsModel = generateVrfCliModel( IsisSrv6LocatorsModel,
                                 "SRv6 Locators in IS-IS for all VRFs" )
