# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import sys

import BasicCli
import Cell
import CliCommand
from CliDynamicSymbol import LazyCallback
import CliGuards
import CliMatcher
from CliModel import cliPrinted
import CliParser
import CliToken.Ip
import ConfigMount
from CliPlugin import IpAddrMatcher
import CliPlugin.IraIp6Cli as IraIp6Cli # pylint: disable=consider-using-from-import
import CliPlugin.IraIpCli as IraIpCli # pylint: disable=consider-using-from-import
# pylint: disable-next=consider-using-from-import
import CliPlugin.TechSupportCli as TechSupportCli
import CliPlugin.VrfCli as VrfCli # pylint: disable=consider-using-from-import
from CliPlugin import AgentLibShowCommands, ShowTaskSchedulerModel
from CliPlugin.IraKernelIpCli import kernelKwForShow
from CliPlugin.IraRouterKernel import RouterKernelConfigMode
from CliPlugin.IraRouterKernel import routerKernelVrfHook
from CliPlugin.KernelFibCliModels import UnprogrammedRoutesVrfs
from CliPlugin.RcfLibCliModels import ShowAgentRcfModel, ShowAgentRcfDetailModel
from CliPlugin.RouterGeneralCli import (
   matcherSoftware,
   matcherForwarding,
   matcherHardware,
   matcherOffload,
   matcherRoute,
)
from CliPlugin.RcfCliLib import (
   rcfFunctionMatcher,
   rcfFunctionMatcherParensOptional,
   rcfKw,
)
from CliPlugin.IntfCli import Intf
from CliPlugin.EthIntfCli import EthPhyIntf
from CliMode.KernelFib import RouterKernelAfMode
from IpLibConsts import DEFAULT_VRF
from TypeFuture import TacLazyType
import EosCloudInitLib
import LazyMount
import ShowCommand
import SmashLazyMount
import Tracing
import Tac
from Toggles import KernelFibToggleLib, RcfLibToggleLib

traceHandle = Tracing.Handle( 'KernelFibCli' )
t5 = traceHandle.trace5 # Info

kernelFibConstants = TacLazyType( 'KernelFib::KernelFibConstants' )

myEntManager = None
agentCmdConfig = None
routingHardwareStatusCommon = None
kernelfibConfig = None
kernelfibConfig6 = None
vrfNameStatus = None
unprogrammedRoutePrinter = None

matcherKernelfib = CliMatcher.KeywordMatcher( 'kernelFib',
      helpdesc='Show detailed state of KernelFib agent' )
matcherKernel = CliMatcher.KeywordMatcher( 'kernel',
      helpdesc='Show only kernel routes' )
matcherUnprogrammed = CliMatcher.KeywordMatcher( 'unprogrammed',
      helpdesc='Uprogrammed routes information' )
matcherLocal = CliMatcher.KeywordMatcher( 'local',
      helpdesc='Local configuration' )
matcherInterface = CliMatcher.KeywordMatcher( 'interface',
      helpdesc='Specify an interface' )
matcherLookup = CliMatcher.KeywordMatcher( 'lookup',
      helpdesc='Exclude route from SW forwarding table and '
               'Offload route lookup to hardware' )
matcherBgp = CliMatcher.KeywordMatcher( 'bgp',
      helpdesc='Offload route lookup to hardware for Bgp routes' )
matcherSummary = CliMatcher.KeywordMatcher( 'summary',
      helpdesc='Summarized information' )
matcherConfigSanity = CliMatcher.KeywordMatcher( 'config-sanity',
      helpdesc='Show uRPF/ACL config state' )

# CLI guard against platforms that do not fully support txfwd netdevs.
def txFwdUnsupportedGuard( mode, token ):
   return CliGuards.guardNotThisPlatform if EosCloudInitLib.isModeSfa() else None

#-------------------------------------------------------------------------------
# Create per-vrf config ( called when vrf is created/ router-kernel..mode 
# is entered)
#-------------------------------------------------------------------------------
def _routeVrfCreation( vrfName, status ):
   if status:
      kernelfibConfig.vrfConfig.newMember( vrfName )
      kernelfibConfig.vrfConfig[vrfName].standbyRouteCol = ()
      kernelfibConfig6.vrfConfig.newMember( vrfName )
   else:
      del kernelfibConfig.vrfConfig[ vrfName ]
      del kernelfibConfig6.vrfConfig[ vrfName ]

# Register callback handler for creating/deleting vrf config
routerKernelVrfHook.addExtension( _routeVrfCreation )

#-------------------------------------------------------------------------------
# router kernel - address family config mode. 
#-------------------------------------------------------------------------------
class RouterKernelAfConfigMode( RouterKernelAfMode, BasicCli.ConfigModeBase ):
   name = 'Kernel route address family configuration'

   def __init__( self, parent, session, addrFamily ):
      self.vrfName = parent.vrfName
      RouterKernelAfMode.__init__( self, ( self.vrfName, addrFamily ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def clearConfig( self ):
      LazyCallback( "KernelFibCliHandler.setKernelRouteAdminDistance" )(
         self, self.addrFamily, self.vrfName )
      LazyCallback( "KernelFibCliHandler.setSwFwdingHwOffloadLocalIntf" )(
         self, self.addrFamily, self.vrfName )
      LazyCallback( "KernelFibCliHandler.setSwFwdingHwOffloadRcf" )(
         self, self.addrFamily, self.vrfName )
      LazyCallback( "KernelFibCliHandler.setSwFwdingHwOffloadRouteLookup" )(
         self, self.addrFamily, self.vrfName, enable=False )
      LazyCallback( "KernelFibCliHandler.resetKernelFibStandbyRoutes" )(
         self, self.addrFamily, self.vrfName )

#-------------------------------------------------------------------------------
# [ no | default ] address-family ADDR_FAMILY
#-------------------------------------------------------------------------------
class RouterKernelAfConfigModeCmd( CliCommand.CliCommandClass ):
   syntax = 'address-family ADDR_FAMILY'
   noOrDefaultSyntax = syntax
   data = {
      'address-family' : 'Address family configuration options',
      'ADDR_FAMILY' : CliMatcher.EnumMatcher( {
         'ipv4' : 'IPv4 related',
         'ipv6' : 'IPv6 related'
      } ),
   }
   handler = "KernelFibCliHandler.handlerRouterKernelAfConfigModeCmd"
   noOrDefaultHandler =\
      "KernelFibCliHandler.noOrDefaultHandlerRouterKernelAfConfigModeCmd"

RouterKernelConfigMode.addCommandClass( RouterKernelAfConfigModeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] distance DISTANCE
#--------------------------------------------------------------------------------
class DistanceDistanceCmd( CliCommand.CliCommandClass ):
   syntax = 'distance DISTANCE'
   noOrDefaultSyntax = syntax
   data = {
      'distance' : 'Set the admin distance for non EOS routes',
      'DISTANCE' : CliMatcher.IntegerMatcher( 1, 255, helpdesc='Distance value' ),
   }

   # Set the kernel route preference to specified value
   handler = "KernelFibCliHandler.handlerDistanceDistanceCmd"
   # Reset the kernel route preference to default
   noOrDefaultHandler = "KernelFibCliHandler.noOrDefaultHandlerDistanceDistanceCmd"

RouterKernelAfConfigMode.addCommandClass( DistanceDistanceCmd )

#--------------------------------------------------------------------------------
# [ no | default ] software forwarding hardware offload route rcf FUNCTION
#--------------------------------------------------------------------------------
class SwFwdingHwOffloadRouteRcfCmd( CliCommand.CliCommandClass ):
   template_ = 'software forwarding hardware offload route rcf {}'
   syntax = template_.format( 'FUNCTION' )
   noOrDefaultSyntax = template_.format( '...' )
   data = {
      'software' : matcherSoftware,
      'forwarding': matcherForwarding,
      'hardware' : matcherHardware,
      'offload' : matcherOffload,
      'route' : matcherRoute,
      'rcf' : CliCommand.Node( matcher=rcfKw, guard=txFwdUnsupportedGuard ),
      'FUNCTION' : rcfFunctionMatcher,
   }

   # Set the name of the Routing Control Function to the specified value
   handler = "KernelFibCliHandler.handlerSwFwdingHwOffloadRouteRcfCmd"
   # Reset the Routing Control Function name to the default value
   noOrDefaultHandler =\
      "KernelFibCliHandler.noOrDefaultHandlerSwFwdingHwOffloadRouteRcfCmd"

RouterKernelAfConfigMode.addCommandClass( SwFwdingHwOffloadRouteRcfCmd )

#---------------------------------------------------------------------------------
# [ no | default ] software forwarding hardware offload route local interface INTF
#---------------------------------------------------------------------------------
class SwFwdingHwOffloadRouteLocalIntfCmd( CliCommand.CliCommandClass ):
   template_ = 'software forwarding hardware offload route local interface {}'
   syntax = template_.format( 'INTF' )
   noOrDefaultSyntax = template_.format( '...' )
   data = {
      'software' : matcherSoftware,
      'forwarding': matcherForwarding,
      'hardware' : matcherHardware,
      'offload' : matcherOffload,
      'route' : matcherRoute,
      'local' : matcherLocal,
      'interface' : matcherInterface,
      'INTF' : Intf.matcherWithRoutingProtoSupport,
   }

   # Set the local interface name to the specified value
   handler = "KernelFibCliHandler.handlerSwFwdingHwOffloadRouteLocalIntfCmd"
   # Reset the local interface name to the default value
   noOrDefaultHandler =\
      "KernelFibCliHandler.noOrDefaultHandlerSwFwdingHwOffloadRouteLocalIntfCmd"

RouterKernelAfConfigMode.addCommandClass( SwFwdingHwOffloadRouteLocalIntfCmd )

#--------------------------------------------------------------------------------
# [ no | default ] software forwarding hardware offload route lookup bgp
#
# NOTE: only bgp is supported for this release.  We will later add the RCF policy
#       [ no | default ] software forwarding hardware offload route lookup \
#                        ( bgp | rcf FUNCTION )
#--------------------------------------------------------------------------------
class SwFwdingHwOffloadRouteLookupCmd( CliCommand.CliCommandClass ):
   syntax = 'software forwarding hardware offload route lookup bgp'
   noOrDefaultSyntax = syntax
   data = {
         'software' : matcherSoftware,
         'forwarding': matcherForwarding,
         'hardware' : matcherHardware,
         'offload' : matcherOffload,
         'route' : matcherRoute,
         'lookup' : CliCommand.Node( matcher=matcherLookup,
                                     guard=txFwdUnsupportedGuard ),
         'bgp' : matcherBgp
   }

   # Set the offload route option to the specified value
   handler = "KernelFibCliHandler.handlerSwFwdingHwOffloadRouteLookupCmd"
   # Reset the offload route option to the default value
   noOrDefaultHandler =\
      "KernelFibCliHandler.noOrDefaultHandlerSwFwdingHwOffloadRouteLookupCmd"

RouterKernelAfConfigMode.addCommandClass( SwFwdingHwOffloadRouteLookupCmd )

#--------------------------------------------------------------------------------
# [ no | default ] kernel software forwarding ecmp
#--------------------------------------------------------------------------------
class KernelSwFwdingEcmpVrfAllCmd( CliCommand.CliCommandClass ):
   syntax = 'kernel software forwarding ecmp'
   noOrDefaultSyntax = syntax
   data = {
         'kernel' : 'Kernel configuration',
         'software' : matcherSoftware,
         'forwarding': matcherForwarding,
         'ecmp' : 'Program ECMP routes in the kernel',
   }

   # Program all paths from ECMP routes in kernel
   handler = "KernelFibCliHandler.handlerKernelSwFwdingEcmpVrfAllCmd"
   # Program only one path from ECMP routes in kernel
   noOrDefaultHandler =\
      "KernelFibCliHandler.noOrDefaultHandlerKernelSwFwdingEcmpVrfAllCmd"

if KernelFibToggleLib.toggleKernelFibProgramAllEcmpEnabled():
   BasicCli.GlobalConfigMode.addCommandClass( KernelSwFwdingEcmpVrfAllCmd )

#--------------------------------------------------------------------------------
# show ip route [ VRF ] kernel unprogrammed
#--------------------------------------------------------------------------------
class IpRouteKernelUnprogrammedCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip route [ VRF ] kernel unprogrammed'
   data = {
      'ip' : CliToken.Ip.ipMatcherForShow,
      'route' : IraIpCli.routeMatcherAfterShowIp,
      'VRF' : VrfCli.VrfExprFactory( helpdesc='Display VRF state',
                                     inclDefaultVrf=True ),
      'kernel' : matcherKernel,
      'unprogrammed' : matcherUnprogrammed,
   }

   cliModel = UnprogrammedRoutesVrfs

   @staticmethod
   def handler( mode, args ):
      vrfName = args.get( 'VRF', VrfCli.DEFAULT_VRF )
      if vrfNameStatus.vrfIdExists( vrfName ) :
         vrfId = vrfNameStatus.nameToIdMap.vrfNameToId[ vrfName ]
         fd = sys.stdout.fileno()
         fmt = mode.session_.outputFormat()
         unprogrammedRoutePrinter.printRoutes( vrfName, vrfId, False, fd, fmt )
         return cliPrinted( UnprogrammedRoutesVrfs )
      return UnprogrammedRoutesVrfs()

BasicCli.addShowCommandClass( IpRouteKernelUnprogrammedCmd )

#--------------------------------------------------------------------------------
# show ipv6 route [ VRF ] kernel unprogrammed
#--------------------------------------------------------------------------------
def ipV6Vrf( mode, token ):
   if routingHardwareStatusCommon.vrfCapability.\
      ipv6EnabledDefault is False:
      return CliParser.guardNotThisPlatform
   return None

class Ipv6RouteKernelUnprogrammedCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 route [ VRF ] kernel unprogrammed'
   data = {
      'ipv6' : CliToken.Ipv6.ipv6MatcherForShow,
      'route' : IraIp6Cli.routeAfterShowIpv6Matcher,
      'VRF' : VrfCli.VrfExprFactory( helpdesc='Display VRF state',
                                     guard=ipV6Vrf,
                                     inclDefaultVrf=True ),
      'kernel' : matcherKernel,
      'unprogrammed' : matcherUnprogrammed,
   }
   cliModel = UnprogrammedRoutesVrfs

   @staticmethod
   def handler( mode, args ):
      vrfName = args.get( 'VRF', VrfCli.DEFAULT_VRF )
      if vrfNameStatus.vrfIdExists( vrfName ) :
         vrfId = vrfNameStatus.nameToIdMap.vrfNameToId[ vrfName ]
         fd = sys.stdout.fileno()
         fmt = mode.session_.outputFormat()
         unprogrammedRoutePrinter.printRoutes( vrfName, vrfId, True, fd, fmt )
         return cliPrinted( UnprogrammedRoutesVrfs )
      return UnprogrammedRoutesVrfs()

BasicCli.addShowCommandClass( Ipv6RouteKernelUnprogrammedCmd )

# --------------------------------------------------------------------------------
# show kernel software forwarding hardware offload summary [ VRF ]
# --------------------------------------------------------------------------------
class KernelSwFwdingHwOffloadSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show kernel software forwarding hardware offload summary [ VRF ]'
   data = {
      'kernel': kernelKwForShow,
      'software': matcherSoftware,
      'forwarding': matcherForwarding,
      'hardware': matcherHardware,
      'offload': matcherOffload,
      'summary': matcherSummary,
      'VRF': VrfCli.VrfExprFactory( helpdesc='Display VRF state',
                                     inclDefaultVrf=True )
   }

   cliModel = "KernelFibCliDynamicModels.Fwd0EthTypeEntriesVrfs"
   handler = "KernelFibCliHandler.handlerKernelSwFwdingHwOffloadSummaryCmd"

BasicCli.addShowCommandClass( KernelSwFwdingHwOffloadSummaryCmd )

#--------------------------------------------------------------------------------
# show kernel software forwarding hardware offload config-sanity [ VRF ]
#--------------------------------------------------------------------------------
class KernelSwFwdingHwOffloadConfigSanityCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show kernel software forwarding hardware offload config-sanity [ VRF ]'
   data = {
      'kernel' : kernelKwForShow,
      'software' : matcherSoftware,
      'forwarding' : matcherForwarding,
      'hardware' : matcherHardware,
      'offload' : matcherOffload,
      'config-sanity' : matcherConfigSanity,
      'VRF' : VrfCli.VrfExprFactory( helpdesc='Display VRF state',
                                     inclDefaultVrf=True )
   }
   cliModel = "KernelFibCliDynamicModels.Fwd0EthTypeConfigSanityEntriesVrfs"
   handler = "KernelFibCliHandler.handlerKernelSwFwdingHwOffloadConfigSanityCmd"
   
BasicCli.addShowCommandClass( KernelSwFwdingHwOffloadConfigSanityCmd )

#---------------------------------------------------------------------------------
# show tech-support kernelFib [ rcf ] [ detail ]
#--------------------------------------------------------------------------------
class TechSupportKernelfibCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show tech-support kernelFib [ rcf ] [ detail ]'
   data = {
      'tech-support' : TechSupportCli.techSupportKwMatcher,
      'kernelFib' : matcherKernelfib,
      'detail' : 'Show verbose state',
      'rcf' : rcfKw,
   }
   privileged = True
   handler = "KernelFibCliHandler.handlerTechSupportKernelfibCmd"

BasicCli.addShowCommandClass( TechSupportKernelfibCmd )

TechSupportCli.registerShowTechSupportCmd(
   '2014-04-07 04:42:20',
   cmds=[ 'show ip route kernel unprogrammed',
          'show ipv6 route kernel unprogrammed',
          'show kernelfib scheduler' ] )

matcherScheduler = CliCommand.Node(
   matcher=CliMatcher.KeywordMatcher( 'scheduler',
                                      helpdesc='KernelFib scheduler statistics' ) )

#--------------------------------------------------------------------------------
# Deprecate 'show kernelFib scheduler reset'.  It was a
# lazy implementation and doesn't fit the new one.
# --------------------------------------------------------------------------------
class BgpMonitoringInternalSchedulerResetCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show kernelFib scheduler reset'
   data = {
      'kernelFib' : matcherKernelfib,
      'scheduler' : matcherScheduler,
      'reset': CliCommand.singleNode( CliMatcher.KeywordMatcher( 'reset',
         helpdesc='Reset scheduler statistics' ),
         deprecatedByCmd='clear agent kernelFib task scheduler' ),
   }
   privileged = True
   handler = "KernelFibCliHandler.handlerBgpMonitoringInternalSchedulerResetCmd"

BasicCli.addShowCommandClass( BgpMonitoringInternalSchedulerResetCmd )

#--------------------------------------------------------------------------------
# show kernelFib scheduler [ OPTIONS ]
#--------------------------------------------------------------------------------
class KernelfibSchedulerCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show kernelFib scheduler [ OPTIONS ]'
   data = {
      'kernelFib' : matcherKernelfib,
      'scheduler' : matcherScheduler,
      'OPTIONS' : CliCommand.SetEnumMatcher( {
         'verbose' : 'Include more information',
         'internal' : 'Display memory addresses of tasks',
         'history' : 'Display scheduler event history',
      } ),
   }
   privileged = True
   cliModel = ShowTaskSchedulerModel.Overall

   @staticmethod
   def handler( mode, args ):
      # Translate options to those used in AgentCli
      translation = { 'verbose': 'detail',
                      'internal': 'debug',
                      'history': 'history' }
      agentArgs = { 'AGENT': 'KernelFib',
                    # pylint: disable-next=consider-using-dict-items
                    'SCHEDULER_OPTS': [ translation[ i ] for i in translation
                                        if i in args.get( 'OPTIONS', [] ) ],
      }
      return AgentLibShowCommands.showTaskSchedulerHandler( mode, agentArgs )

BasicCli.addShowCommandClass( KernelfibSchedulerCmd )

#--------------------------------------------------------------------------------
# show kernelFib rcf [ <func-name> ]
#--------------------------------------------------------------------------------
class KernelFibRcfCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show kernelFib rcf [ FUNC_NAME ]'
   data = {
      'kernelFib' : matcherKernelfib,
      'rcf' : rcfKw,
      'FUNC_NAME' : rcfFunctionMatcherParensOptional,
   }
   cliModel = ShowAgentRcfModel
   handler = "KernelFibCliHandler.handlerKernelFibRcfCmd"

BasicCli.addShowCommandClass( KernelFibRcfCmd )

#--------------------------------------------------------------------------------
# show kernelFib rcf detail [ <func-name> ]
#--------------------------------------------------------------------------------
class KernelFibRcfDetailCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show kernelFib rcf detail [ FUNC_NAME ]'
   data = {
      'kernelFib' : matcherKernelfib,
      'rcf' : rcfKw,
      'detail' : CliMatcher.KeywordMatcher( "detail", helpdesc="Detailed view" ),
      'FUNC_NAME' : rcfFunctionMatcherParensOptional,
   }
   cliModel = ShowAgentRcfDetailModel
   handler = "KernelFibCliHandler.handlerKernelFibRcfDetailCmd"

if RcfLibToggleLib.toggleRcfShowAgentRcfDetailEnabled():
   BasicCli.addShowCommandClass( KernelFibRcfDetailCmd )

#--------------------------------------------------------------------------------
# [ no | default ] standby route prefix [ interface ] next-hop
#--------------------------------------------------------------------------------
class KernelFibStandbyRouteCmd( CliCommand.CliCommandClass ):
   syntax = 'standby route prefix [ interface ] next-hop'
   noOrDefaultSyntax = 'standby route prefix [interface] [next-hop]' 
   data = {
            'standby': 'Standby supervisor',
            'route': 'Manage static routes on standby supervisor',
            'prefix': IpAddrMatcher.IpPrefixMatcher("Destination IPv4 prefix"),
            'interface': EthPhyIntf.ethOrMgmtMatcher,
            'next-hop': IpAddrMatcher.IpAddrMatcher("IPv4 next hop"),
          }    
   handler = "KernelFibCliHandler.handlerKernelFibStandbyRouteCmd"
   noOrDefaultHandler = ( "KernelFibCliHandler.noOrDefaultHandlerKernelFibStandby"
                        "RouteCmd" )

if KernelFibToggleLib.toggleKernelFibAddStandbyRouteEnabled():
   RouterKernelAfConfigMode.addCommandClass( KernelFibStandbyRouteCmd )

def kernelUnprogrammedRoute( vrfName=DEFAULT_VRF ):
   if vrfName == DEFAULT_VRF:
      return SmashLazyMount.mount( myEntManager,
                              'routing/kernel/unprogrammedRoute',
                              'Smash::KernelUnprogrammedRoute::UnprogrammedRoute',
                              SmashLazyMount.mountInfo( 'reader' ) )
   else:
      return SmashLazyMount.mount( myEntManager,
            'routing/vrf/kernel/unprogrammedRoute',
            'Smash::KernelUnprogrammedRoute::UnprogrammedRoute',
            SmashLazyMount.mountInfo( 'reader' ) ) 

def Plugin( entityManager ):
   global myEntManager
   global routingHardwareStatusCommon
   global kernelfibConfig, kernelfibConfig6
   global vrfNameStatus
   global unprogrammedRoutePrinter

   myEntManager = entityManager
   routingHardwareStatusCommon = LazyMount.mount(
      entityManager, 'routing/hardware/statuscommon',
      'Routing::Hardware::StatusCommon', 'r' )

   kernelfibConfig = ConfigMount.mount( entityManager, 'routing/kernel/config',
                                        'KernelFib::KernelRouteConfigDir', 'w' )
   kernelfibConfig6 = ConfigMount.mount( entityManager, 'routing6/kernel/config',
                                         'KernelFib::KernelRouteConfigDir', 'w' )

   vrfNameStatus = LazyMount.mount( entityManager, 
                                       Cell.path( 'vrf/vrfNameStatus' ), 
                                       'Vrf::VrfIdMap::NameToIdMapWrapper', 'r' )

   unprogrammedRoutePrinter = Tac.newInstance( 
         'KernelFib::UnprogrammedRoutePrinter',
         kernelUnprogrammedRoute( DEFAULT_VRF ),
         kernelUnprogrammedRoute( 'all' ) )


