# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliParser
import LazyMount
from CliPlugin import EbraCliLib
from CliPlugin.L2ProtocolModel import ( L2ProtocolForwardingProfile,
      L2ProtocolForwardingProfiles, L2ProtocolForwardingProfileSummary,
      L2ProtocolForwardingProfileSummaries, L2ProtocolForwardingInfo,
      L2ProtocolForwardingInterface, L2ProtocolForwardingInterfaces )
from CliPlugin.EthIntfCli import EthPhyAutoIntfType
from CliPlugin.SwitchIntfCli import SwitchAutoIntfType
from EbraLib import l2ProtocolFromMacAddress, l2PtActionToCapiAction
from Intf.IntfRange import IntfRangeMatcher
from TypeFuture import TacLazyType
import ShowCommand,  CliMatcher, CliCommand

# Module globals set up by the Plugin function below
l2PtProfileConfig = None
l2PtIntfConfig = None
bridgingHwCapabilities = None
l2ProtocolFwdIntfStatus = None

PortChannelIntfId = TacLazyType( 'Arnet::PortChannelIntfId' )

def l2ProtocolModeGuard( mode, token ):
   if bridgingHwCapabilities.l2ProtocolTransparencySupported:
      return None
   return CliParser.guardNotThisPlatform

matcherL2Protocol = CliMatcher.KeywordMatcher( 'l2-protocol',
                                      helpdesc='L2 Protocol information' )
nodeL2Protocol = CliCommand.Node( matcher=matcherL2Protocol,
                                  guard=l2ProtocolModeGuard )
matcherForwarding = CliMatcher.KeywordMatcher( 'forwarding',
                                      helpdesc='L2 forwarding information' )
matcherProfile = CliMatcher.KeywordMatcher( 'profile',
                                   helpdesc='Forwarding profile information' ) 

def allLagMembersProgrammedInHw( mode, intf, l2ProtocolFwdStatus ):
   if PortChannelIntfId.isPortChannelIntfId( intf ):
      if PortChannelIntfId.subIntf( intf ):
         parentIntf = PortChannelIntfId.parentIntfIdFromSubIntf( intf )
      else:
         parentIntf = intf
      activeMembers = set( l2ProtocolFwdStatus.member )
      configuredMembers = set()
      if EbraCliLib.getLagMembersCallBack:
         configuredMembers = set(
               EbraCliLib.getLagMembersCallBack( # pylint: disable-msg=not-callable
                  mode, parentIntf, useLagConfig=True ) )
      return activeMembers == configuredMembers
   return True

def populateL2ProtocolForwardingProfiles( mode, profileNames, profiles,
                                          summary=None ):
   for profileName in profileNames:
      l2PtProfile = l2PtProfileConfig.profile.get( profileName )
      profile = L2ProtocolForwardingProfileSummary() if summary else \
                L2ProtocolForwardingProfile()
      profiles.profiles[ profileName ] = profile
      # Populate profile info
      if not summary:
         # For IS-IS forward, there are 5 different DMAC sets handled in
         # L2Pt code, which results in 5 redundant entries in this show command.
         # Filter out the redundant entries here.
         processedProtocols = []
         for protocol, action in l2PtProfile.protocolToAction.items():
            info = L2ProtocolForwardingInfo()
            if protocol.addrConfig:
               info.address = protocol.addr
            else:
               info.protocol = l2ProtocolFromMacAddress( protocol.addr )
            if info.configuredUsingAddress() or\
               ( info.protocol, protocol.tagFormat ) not in processedProtocols:
               if not info.configuredUsingAddress():
                  processedProtocols.append( ( info.protocol, protocol.tagFormat ) )
               info.tagFormat = protocol.tagFormat
               info.action = l2PtActionToCapiAction[ action ]
               profile.protocolInfo.append( info )
   if summary:
      # Populate active interfaces
      for intf, intfStatus in l2ProtocolFwdIntfStatus.intfStatus.items():
         profile = profiles.profiles.get( intfStatus.l2PtProfileName )
         if profile:
            if allLagMembersProgrammedInHw( mode, intf, intfStatus ):
               profile.activeIntfs.append( intf )
      # Populate configured interfaces
      for intf, intfProfileName in l2PtIntfConfig.intfToProfile.items():
         profile = profiles.profiles.get( intfProfileName )
         if profile:
            profile.configuredIntfs.append( intf )

def showL2ProtocolForwardingProfile( mode, args ):
   profileName = args.get( 'PROFILE' )
   summary = 'summary' in args
   profiles = L2ProtocolForwardingProfileSummaries() if summary else \
              L2ProtocolForwardingProfiles()
   if profileName and profileName not in l2PtProfileConfig.profile:
      return profiles
   profileNames = [ profileName
                  ] if profileName else l2PtProfileConfig.profile
   populateL2ProtocolForwardingProfiles( mode, profileNames, profiles, summary )
   return profiles

def l2ProtocolForwardingInfos( protocolInfos ):
   l2ProtocolFwdInfos = []
   processedProtocols = []
   for l2ProtocolFwdInfo in protocolInfos.values():
      info = L2ProtocolForwardingInfo()
      l2ProtocolMatch = l2ProtocolFwdInfo.l2ProtocolMatch
      if l2ProtocolMatch.addrConfig:
         continue
      protocol = l2ProtocolFromMacAddress( l2ProtocolMatch.addr )
      if ( protocol, l2ProtocolMatch.tagFormat ) not in processedProtocols:
         processedProtocols.append( ( protocol, l2ProtocolMatch.tagFormat ) )
         info.protocol = protocol
         info.tagFormat = l2ProtocolMatch.tagFormat
         info.action = l2PtActionToCapiAction[ l2ProtocolFwdInfo.action.action ]
         l2ProtocolFwdInfos.append( info )
   return l2ProtocolFwdInfos

def l2ProtocolForwardingInterfaces( mode, intfNames, detail ):
   interfaceInfo = L2ProtocolForwardingInterfaces()
   if detail:
      interfaceInfo.detailIs( True )
   for intf in intfNames:
      # Get the L2ProtocolFwdStatus, if exists
      l2ProtocolFwdStatus = l2ProtocolFwdIntfStatus.intfStatus.get( intf )
      if not l2ProtocolFwdStatus: # pylint: disable=no-else-continue
         continue
      elif not allLagMembersProgrammedInHw( mode, intf, l2ProtocolFwdStatus ):
         continue
      interfaceProfileInfo = L2ProtocolForwardingInterface()
      interfaceInfo.interfaces[ intf ] = interfaceProfileInfo
      interfaceProfileInfo.profileName = l2ProtocolFwdStatus.l2PtProfileName

      if detail:
         # Populate the protocol information
         interfaceProfileInfo.protocolInfo = l2ProtocolForwardingInfos(
            l2ProtocolFwdStatus.seqToProtocolInfo )
         # Populate the PW protocol information
         interfaceProfileInfo.protocolInfo += l2ProtocolForwardingInfos(
            l2ProtocolFwdStatus.pseudoWireProtocolInfo )

   return interfaceInfo

#-------------------------------------------------------------------------------
# "show l2-protocol forwarding interface [intfs] [detail]"
#-------------------------------------------------------------------------------

def showL2ProtocolForwardingInterface( mode, args ):
   intfs = [ intf
             for intf in args.get( 'INTERFACES', l2ProtocolFwdIntfStatus.intfStatus )
             if intf.startswith( ( 'Ethernet', 'Switch', 'Port-Channel' ) ) ]
   detail = 'detail' in args
   return l2ProtocolForwardingInterfaces( mode, intfs, detail )

profileNameRule = CliMatcher.DynamicNameMatcher(
   lambda mode: l2PtProfileConfig.profile,
   helpdesc='L2 protocol forwarding profile name' )

l2ProtocolIntfTypes = [ EthPhyAutoIntfType, SwitchAutoIntfType, ]


class L2ProtocolForwardingInterfacesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show l2-protocol forwarding interfaces [ INTERFACES ] [ detail ]'
   data = {
      'l2-protocol': nodeL2Protocol,
      'forwarding': matcherForwarding,
      'interfaces': 'Interface-specific details',
      'INTERFACES': IntfRangeMatcher( explicitIntfTypes=l2ProtocolIntfTypes ),
      'detail': 'Display information in detail',
   }
   handler = showL2ProtocolForwardingInterface
   cliModel = L2ProtocolForwardingInterfaces

BasicCli.addShowCommandClass( L2ProtocolForwardingInterfacesCmd )

class L2ProtocolForwardingProfileCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show l2-protocol forwarding profile [ PROFILE ]'
   data = {
      'l2-protocol': nodeL2Protocol,
      'forwarding': matcherForwarding,
      'profile': matcherProfile,
      'PROFILE': profileNameRule,
   }
   handler = showL2ProtocolForwardingProfile
   cliModel = L2ProtocolForwardingProfiles

BasicCli.addShowCommandClass( L2ProtocolForwardingProfileCmd )

class L2ProtocolForwardingProfileSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show l2-protocol forwarding profile [ PROFILE ] summary'
   data = {
      'l2-protocol': nodeL2Protocol,
      'forwarding': matcherForwarding,
      'profile': matcherProfile,
      'PROFILE': profileNameRule,
      'summary': 'Summary',
   }
   handler = showL2ProtocolForwardingProfile
   cliModel = L2ProtocolForwardingProfileSummaries

BasicCli.addShowCommandClass( L2ProtocolForwardingProfileSummaryCmd )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global l2PtProfileConfig, l2PtIntfConfig
   global bridgingHwCapabilities
   global l2ProtocolFwdIntfStatus
   l2PtProfileConfig = LazyMount.mount( entityManager,
         "l2protocolforwarding/profileconfig",
         "Ebra::L2Pt::L2PtProfileConfig", "r" )
   l2PtIntfConfig = LazyMount.mount( entityManager,
         "l2protocolforwarding/intfconfig",
         "Ebra::L2Pt::L2PtIntfConfig", "r" )
   bridgingHwCapabilities = LazyMount.mount( entityManager,
         "bridging/hwcapabilities",
         "Bridging::HwCapabilities", "r" )
   l2ProtocolFwdIntfStatus = LazyMount.mount( entityManager,
                              "l2protocolforwarding/intfstatusnew",
                              "Ebra::L2Pt::L2ProtocolFwdIntfStatus", "r" )
