#!/usr/bin/env python3
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#pylint: disable=import-error, ungrouped-imports

import sys
import CliCommand
import CliMatcher
import SmashLazyMount
import Tac, BasicCli, ShowCommand
import Tracing
from collections import defaultdict
from TypeFuture import TacLazyType
from ArnetModel import IpGenericAddr
import CliPlugin.IpGenAddrMatcher as IpGenAddrMatcher
from CliPlugin import TechSupportCli, MacAddr, EthIntfCli, LagIntfCli
from CliPlugin.TunnelCli import (
   getTunnelIdModel,
   getEndpointFromTunnelId,
   getColoredEndpointFromTunnelId )
from CliPlugin.L2RibModel import (
   Summary,
   Dest,
   HostEntry,
   HostEntryAndSource,
   Label,
   LoadBalance,
   HostTable,
   FloodSet,
   FloodSetSummary,
   FloodSetSummaryColl,
   DestsFromHostTable,
   DestSummaryColl,
   DestFloodSetSummary,
   DestFloodSetSummaryColl )
from Ethernet import convertMacAddrToDisplay
from L2RibLib import traverseAndValidateMultiTable

FecIdIntfId = TacLazyType( "Arnet::FecIdIntfId" )
FecIdType = TacLazyType( "Smash::Fib::FecId" )
DropMode = TacLazyType( "Bridging::DropMode" )

t0 = Tracing.trace0
storedEntityManager = None
dynTable = None
bgpTable = None
l2RibSourceHostModel = None
cppPlugin = None
destTypePrettyName = {
   'mpls': 'MPLS',
   'tunnel': 'Tunnel',
   'vxlan': 'VXLAN',
   'cpu': 'CPU',
   'interface': 'Interface'
}

class AliasResolve:
   def __init__( self ):
      self.back = Tac.newInstance( 'L2RibCli::AliasResolver' )
      self.enumDecoder = Tac.Value( 'L2Rib::SourceDecoder' )

   def entryIs( self, commandToken, prettyName, sourceName, tableName, *args ):
      '''set entry. args are additional command aliases'''
      entry = self.back.entryIs( sourceName or "",
                                 commandToken or "",
                                 prettyName or "",
                                 tableName or "" )
      for alias in args:
         entry.commandAlias[ alias ] = True
         self.back.lookup[ alias ] = entry

   def _resolve( self, target, alias=None ):
      '''use key in appropriate list. If no alias provided, return full list.
         If alias provided but not found, return alias'''
      if alias:
         return getattr( self.back.lookup.get( alias.lower(), None ),
                         target, None ) or alias
      return [ getattr( entry, target, None )
               for entry in self.back.repo.values()
               if getattr( entry, target, False ) ]

   def commandToken( self, alias=None ):
      '''returns command token associated with alias,
      or list of all tokens if no alias. Return None if alias not found'''
      return self._resolve( 'commandToken', alias )

   def prettyName( self, alias=None ):
      '''returns display name associated with alias'''
      return self._resolve( 'prettyName',  alias )

   def sourceName( self, alias=None ):
      '''return associated host source name'''
      return self._resolve( 'source', alias )

   def decodedEnum( self, alias=None ):
      '''return associated source name decoded with Tac SourceDecoder '''
      return self.enumDecoder.sourceFromEnum( self._resolve( 'source', alias ) )

   def commandAlias( self, alias=None ):
      '''return associated alias for commands'''
      r = self._resolve( 'commandAlias', alias )
      if isinstance( r, str ):
         return r
      return list( r )

   def tableName( self, alias=None ):
      '''return associated table name'''
      return self._resolve( 'tableName', alias )

# Populate lookup tables
aliasResolve = AliasResolve()
aliasResolve.entryIs( 'local-dynamic', 'Local Dynamic', 'sourceLocalDynamic',
                      'localDynamic', 'dynamic', 'localDynamic' )
aliasResolve.entryIs( 'static', 'Local Static', 'sourceLocalStatic', 'static' )
aliasResolve.entryIs( 'bgp', 'BGP', 'sourceBgp', 'bgp' )
aliasResolve.entryIs( 'mlag', 'MLAG', 'sourceMlag', 'mlag' )
aliasResolve.entryIs( 'router', 'Router MAC', 'sourceRouterMac', 'router' )
aliasResolve.entryIs( 'dot1x', 'Dot1x', 'sourceDot1x', 'dot1x' )
aliasResolve.entryIs( 'vxlan-control-service', 'VXLAN Control Service',
                      'sourceVcs', 'vcs' )
aliasResolve.entryIs( 'vxlan-static', 'VXLAN Static', 'sourceVxlanStatic', 
                      'vxlanStatic' )
aliasResolve.entryIs( 'vxlan-dynamic', 'VXLAN Dynamic', 'sourceVxlanDynamic',
                      'vxlanDynamic' )
aliasResolve.entryIs( None, 'TestProto1', 'sourceTestProto1', None )
aliasResolve.entryIs( None, 'TestProto1', 'sourceTestProto2', None )
aliasResolve.entryIs( None, 'TestProto1', 'sourceTestProto3', None )

# preserve access to deprecated structures for compatibility with existing packages
# use aliasResolve instead of these
class DepDict( dict ):
   def __init__( self, keyfunc, valuefunc ):
      self.valuefunc = valuefunc
      self.keyfunc = keyfunc
   def get( self, key, default=None ):
      return self.valuefunc( key )
   def __getitem__( self, key ):
      return self.valuefunc( key )
   def keys( self ):
      return self.keyfunc()

sourceAliases = DepDict( aliasResolve.commandAlias, aliasResolve.commandToken )
altsByTokan = DepDict( aliasResolve.commandToken, aliasResolve.commandAlias )
tableDisplayNames = DepDict( aliasResolve.tableName, aliasResolve.commandToken )
hostSourceNames = DepDict( aliasResolve.sourceName, aliasResolve.prettyName )

def extractSource( args ):
   if args[ 'SOURCE' ] == 'all':
      return 'all'
   return aliasResolve.commandToken( args[ 'SOURCE' ] )

def l2RibCliHostModel( sourceDirEntry=None ):
   """Return L2RibHostModel model as found from l2RibSourceHostModel dict
   if one exits. Else we construct one from sourceDirEntry."""

   sourceName = sourceDirEntry.source if sourceDirEntry else 'output'
   if not sourceDirEntry:
      multiTableMountInfo = { 'lb' : ( "bridging/l2Rib/lbOutput",
                                       "L2Rib::LoadBalanceOutput" ),
                              'dest' : ( "bridging/l2Rib/destOutput",
                                         "L2Rib::DestOutput" ),
                              'label' : ( "bridging/l2Rib/labelOutput",
                                          "L2Rib::LabelOutput" ) }
   elif sourceDirEntry.multiTable:
      multiTableMountInfo = { 'lb' : ( sourceDirEntry.lbTableSmashPath,
                                       "L2Rib::LoadBalanceTable" ),
                              'dest' : ( sourceDirEntry.destTableSmashPath,
                                         "L2Rib::DestTable" ),
                              'label' : ( sourceDirEntry.labelTableSmashPath,
                                          "L2Rib::LabelTable" ) }
   else:
      multiTableMountInfo = {}

   global l2RibSourceHostModel
   if not l2RibSourceHostModel:
      l2RibSourceHostModel = dict()

   if l2RibSourceHostModel.get( sourceName ):
      return l2RibSourceHostModel[ sourceName ]

   lbInfo = multiTableMountInfo.get( 'lb', None )
   lbTable = SmashLazyMount.mount(
      storedEntityManager, lbInfo[ 0 ], lbInfo[ 1 ],
      SmashLazyMount.mountInfo( 'reader' ) ) \
      if lbInfo and lbInfo[ 0 ] else None
   destInfo = multiTableMountInfo.get( 'dest', None )
   destTable = SmashLazyMount.mount(
      storedEntityManager, destInfo[ 0 ], destInfo[ 1 ],
      SmashLazyMount.mountInfo( 'reader' ) ) \
      if destInfo and destInfo[ 0 ] else None
   labelInfo = multiTableMountInfo.get( 'label', None )
   labelTable = SmashLazyMount.mount(
      storedEntityManager, labelInfo[ 0 ], labelInfo[ 1 ],
      SmashLazyMount.mountInfo( 'reader' ) ) \
      if labelInfo and labelInfo[ 0 ] else None
   l2RibSourceHostModel[ sourceName ] = L2RibHostModel( sourceName, destTable,
                                                        labelTable, lbTable )
   return l2RibSourceHostModel[ sourceName ]

def floodSetTypeStr( h ):
   if h == 'floodSetTypeAll':
      return 'All'
   elif h == 'floodSetTypeAny':
      return 'Any'

   return 'Invalid'

MAX_LEVEL = 5

def hasDest( nextObj, destTable, lbTable, data, level=0, destType='destTypeVxlan' ):
   # If destType is vxlan, data is vtep
   # If destType is intf, data is intfId
   if destType not in [ 'destTypeVxlan', 'destTypeIntf' ]:
      t0( "Unexpected destType", destType )
      return False
   if nextObj.tableType == 'tableTypeInvalid' or level > MAX_LEVEL:
      t0( "Unexpected tableType", nextObj.tableType, " or level", level )
      return False
   if nextObj.tableType == 'tableTypeDest':
      destEntry = destTable.dest.get( nextObj.objId )
      if destEntry and destEntry.destType == destType:
         if destType == 'destTypeVxlan':
            return destEntry.vxlan.vtepAddr == data
         elif destType == 'destTypeIntf':
            return destEntry.intf.intfId == data
   elif nextObj.tableType == 'tableTypeLoadBalance':
      lbEntry = lbTable.lb.get( nextObj.objId )
      if not lbEntry:
         return False
      if isinstance( lbEntry, Tac.Type( "L2Rib::LoadBalanceAndSource" ) ):
         lbEntry = lbEntry.lb
      # if any lb pointer contains DATA, this function returns True
      for lbNext in lbEntry.next.values():
         if hasDest( lbNext, destTable, lbTable, data, level + 1, destType ):
            return True
   return False

def filterOutOfShow( h, mac, vlan, intf, vtep, model ):
   if vlan and h.vlanId != vlan:
      return True
   if mac and convertMacAddrToDisplay( h.macAddr ) != convertMacAddrToDisplay( mac ):
      return True
   if intf:
      if h.nextIsStored:
         return not hasDest( h.next,
                             model.destTable,
                             model.lbTable,
                             intf.name,
                             destType='destTypeIntf'
         )
      if not h.intfIsStored or str( h.intf ) != str( intf ):
         return True
   if vtep:
      return not h.nextIsStored or not hasDest(
         h.next,
         model.destTable,
         model.lbTable,
         vtep
      )
   return None

def allSourceDirEntries():
   allSde = dict()
   for sdeName in cppPlugin.sourceDir.entityPtr:
      sde = cppPlugin.sourceDir.entityPtr[ sdeName ]
      if sde.initialized:
         allSde[ sdeName ] = sde

   for sdeName in cppPlugin.agentSourceDir.source:
      sde = cppPlugin.agentSourceDir.source[ sdeName ]
      if sde.initialized:
         assert sde.name not in allSde
         allSde[ sde.name ] = sde

   return allSde

def updateHostTableModel( hostTable, hostTableModel, cliHostModel,
                          mac=None, vlan=None, intf=None, vtep=None, detail=None,
                          source=None, destModel=None, destType=None ):
   """Walk L2RIB Host Table ( Input or Output ) and add hosts to hostTableModel."""
   # note that these are deliberately not sorted, because sorting them
   # in python becomes very expensive when the tables gets big.  Doing
   # this properly really needs a C++ CLI implementation, like we have
   # for 'show ip route'
   destSet = set()
   for host in hostTable.host.values():
      if filterOutOfShow( host, mac, vlan, intf, vtep, cliHostModel ):
         continue
      ( valid, hostModel ) = cliHostModel.getHostModel( host, detail, source,
                                                        isDestCmd=bool( destModel ) )
      if destModel:
         for dest in hostModel.dests:
            if isinstance( dest, Dest ) and dest not in destSet and \
               ( not destType or dest.destType == destType ):
               destModel.dests.append( dest )
               destSet.add( dest )
         continue
      if valid:
         hostTableModel.hosts.append( hostModel )
      else:
         hostTableModel.invalidHosts.append( hostModel )

def showL2RibInput( mode, source, mac=None, vlan=None, intf=None, vtep=None,
                    detail=None ):
   sourceName = aliasResolve.tableName( source )
   hostTableModel = HostTable( _detail=bool( detail ) )

   allSde = allSourceDirEntries()
   if( sourceName not in allSde and sourceName != 'all' ):
      return hostTableModel

   for sdeName, sde in allSde.items():
      # When hostTableSmashPath is not a valid string, skip the sde to prevent
      # ConfigAgent from crashing.
      if( sourceName in [ 'all', sdeName ] and sde.hostTableSmashPath ):
         cliHostModel = l2RibCliHostModel( sde )
         hostTable = SmashLazyMount.mount( storedEntityManager,
                                           sde.hostTableSmashPath,
                                           'L2Rib::HostInput',
                                           SmashLazyMount.mountInfo( 'reader' ) )
         # When 'all' input tables have to be listed, tag the source
         # name with the host entry to identify the source.
         source = sdeName if sourceName == 'all' else None
         updateHostTableModel( hostTable, hostTableModel, cliHostModel,
                               mac, vlan, intf, vtep, detail, source )
   return hostTableModel

def getInputDestSummaryColl( sourceName, destType=None ):
   allSde = allSourceDirEntries()
   destSummaryColl = DestSummaryColl()

   for sdeName, sde in allSde.items():
      # When hostTableSmashPath is not a valid string, skip the sde to prevent
      # ConfigAgent from crashing.
      if not ( sourceName in [ 'all', sdeName ] and sde.hostTableSmashPath ):
         continue

      cliHostModel = l2RibCliHostModel( sde )
      tableName = f'{aliasResolve.prettyName( sdeName )} Input'
      destModel = DestsFromHostTable( tableName=tableName, destType=destType )
      hostTable = SmashLazyMount.mount( storedEntityManager,
                                          sde.hostTableSmashPath,
                                          'L2Rib::HostInput',
                                          SmashLazyMount.mountInfo( 'reader' ) )
      # When 'all' input tables have to be listed, tag the source
      # name with the host entry to identify the source.
      source = sdeName if sourceName == 'all' else None
      updateHostTableModel( hostTable, None, cliHostModel, source=source,
                              destModel=destModel, destType=destType )
      destSummaryColl.destSummaries.append( destModel )
   return destSummaryColl

def showL2RibInputDest( mode, source, destType=None ):
   sourceName = aliasResolve.tableName( source )

   allSde = allSourceDirEntries()
   if( sourceName not in allSde and sourceName != 'all' ):
      mode.addError( "No input table %s." % source )
      return None
   if destType:
      destType = destTypePrettyName[ destType ]
   return getInputDestSummaryColl( sourceName, destType )   
   
def showL2RibOutput( mode, mac=None, vlan=None, intf=None, vtep=None, detail=None ):
   cliHostModel = l2RibCliHostModel()
   hostTableModel = HostTable( _detail=bool( detail ) )
   updateHostTableModel( cppPlugin.outputTable, hostTableModel, cliHostModel,
                         mac, vlan, intf, vtep, detail, None )
   return hostTableModel

def showL2RibOutputDest( mode, destType=None ):
   destSummaryColl = DestSummaryColl()
   cliHostModel = l2RibCliHostModel()
   if destType:
      destType = destTypePrettyName[ destType ]
   destModel = DestsFromHostTable( tableName='Output', destType=destType )
   updateHostTableModel( cppPlugin.outputTable, None, cliHostModel,
                         destModel=destModel, destType=destType )
   destSummaryColl.destSummaries.append( destModel )
   return destSummaryColl

def getDestModel( dest, level=0, detail=False, populateDomain=False,
                  populateVtepType=False ):
   """Construct Dest L2RibObject using L2Rib::Dest Tac model."""
   destModel = Dest()
   destModel.level = level
   destModel.index = dest.key if detail else None
   def getSrTeDest( intfId ):
      # Given a FecIdIntfId interface, this routine returns a 3-tuple ( tunnelId,
      # endpoint and color ) for the SR-TE policy tunnel table entry.
      if not FecIdIntfId.isFecIdIntfId( intfId ):
         return None, None, None
      fecId = FecIdIntfId.intfIdToFecId( intfId )
      tunnelId = FecIdType.fecIdToTunnelId( fecId )
      endpoint, color = getColoredEndpointFromTunnelId( tunnelId )
      if endpoint:
         return tunnelId, endpoint, color
      return None, None, None
   if dest.destType == 'destTypeMpls':
      destModel.mplsLabel = dest.mpls.label
      # MPLS destination may include a tunnel interface or physical
      # interface. Populate the relevant field in the destination
      # model.
      if dest.mpls.tunnelId:
         destModel.tunnelId = getTunnelIdModel( dest.mpls.tunnelId )
         destModel.tunnelEndPoint = getEndpointFromTunnelId( dest.mpls.tunnelId )
      else:
         # See if the IntfId is a FecId/SR-TE policy tunnel. If so, then the
         # CLI model will contain the SR-TE policy tunnel information instead of
         # the policy FEC interface id.
         tunnelId, endpoint, color = getSrTeDest( dest.mpls.intfId )
         if tunnelId:
            destModel.tunnelId = getTunnelIdModel( tunnelId )
            destModel.tunnelEndPoint = endpoint
            destModel.color = color
         else:
            destModel.interface = dest.mpls.intfId
      if populateDomain:
         # We won't be populating domain for the host table commands
         # i.e 'show l2rib [input <>|output]'
         destModel.domain = 'remote' if dest.mpls.remoteDomain else 'local'
      destModel.destType = 'MPLS'
   elif dest.destType == 'destTypeVxlan':
      destModel.vtepAddr = IpGenericAddr( ip=dest.vxlan.vtepAddr )
      if populateDomain:
         # We won't be populating domain for the host table commands
         # i.e 'show l2rib [input <>|output]'
         destModel.domain = 'remote' if dest.vxlan.vtepType == 'remoteDomainVtep' \
            else 'local'
      if populateVtepType:
         # We populate the vtepType only for the destination commands
         # i.e 'show l2rib [input <>|output] destination [floodset]'
         destModel.vtepType = dest.vxlan.vtepType
      destModel.destType = 'VXLAN'
   elif dest.destType == 'destTypeTunnel':
      # Tunnel destinations can be overloaded to an interface destination.
      destModel.destType = 'Tunnel'
      if dest.tunnel.tunnel:
         destModel.tunnelEndPoint = getEndpointFromTunnelId( dest.tunnel.tunnel )
         destModel.tunnelId = getTunnelIdModel( dest.tunnel.tunnel )
      else:
         # See if the IntfId is a FecId/SR-TE policy tunnel
         tunnelId, endpoint, color = getSrTeDest( dest.tunnel.intfId )
         # CLI model will contain the SR-TE policy tunnel information instead of
         # the policy FEC interface id.
         if tunnelId:
            destModel.tunnelId = getTunnelIdModel( tunnelId )
            destModel.tunnelEndPoint = endpoint
            destModel.color = color
         else:
            destModel.interface = dest.tunnel.intfId
   elif dest.destType == 'destTypeIntf':
      destModel.interface = dest.intf.intfId
      if dest.intf.dropMode == DropMode.dropModeNone:
         destModel.dropMode = 'none'
      elif dest.intf.dropMode == DropMode.dropModeDst:
         destModel.dropMode = 'dst'
      elif dest.intf.dropMode == DropMode.dropModeSrcAndDst:
         destModel.dropMode = 'srcAndDst'
      destModel.destType = 'Interface'
   elif dest.destType == 'destTypeCpu':
      destModel.destType = 'CPU'
   else:
      raise NotImplementedError
   return destModel

def getFloodSetSummaryModel( table, tableName, vlan=None, vtepAddr=None, intfId=None,
                        label=None ):
   assert table
   fsSummary = FloodSetSummary()
   fsSummary.tableName = tableName
   printVlans = list( table.vlanFloodSet )
   if vlan:
      if vlan not in printVlans:
         return fsSummary
      printVlans = [ vlan ]

   for vlanId in sorted( printVlans ):
      vfs = table.vlanFloodSet.get( vlanId )
      if not vfs or vfs.vlanId != vlanId:
         continue

      for macAddr in sorted( vfs.floodSet ):
         fs = vfs.floodSet.get( macAddr )
         if ( not fs or fs.macAddr != macAddr or
              fs.floodSetType == 'floodSetTypeInvalid' ):
            continue

         floodSet = FloodSet()
         floodSet.vlanId = vlanId
         floodSet.macAddr = macAddr
         if fs.floodSetType == 'floodSetTypeAll':
            floodSet.floodType = 'All'
         else:
            floodSet.floodType = 'Any'

         for d in fs.destSet:
            if( vtepAddr and
                ( d.destType != 'destTypeVxlan' or
                  d.vxlan.vtepAddr != vtepAddr ) ):
               continue
            elif( intfId and
                  ( d.destType != 'destTypeIntf' or
                    d.intf.intfId != intfId ) ):
               continue
            elif( label and
                  ( d.destType != 'destTypeMpls' or
                    d.mpls.label != label ) ):
               continue
            dest = getDestModel( d, populateDomain=True )
            floodSet.dests.append( dest )

         if floodSet.dests:
            fsSummary.floodSets.append( floodSet )
   return fsSummary

def getDestFloodSetSummaryModel( table, tableName, destType=None ):
   assert table
   destFsSummary = DestFloodSetSummary()
   destFsSummary.tableName = tableName
   if destType:
      destFsSummary.destType = destTypePrettyName[ destType ]
      destTypes = [ destTypePrettyName[ destType ] ]
   else:
      destTypes = destTypePrettyName.values()
   destDict = defaultdict( list ) # maps the destination to corresponding vlans

   for vlanId in list( table.vlanFloodSet ):
      vfs = table.vlanFloodSet.get( vlanId )
      if not vfs or vfs.vlanId != vlanId:
         continue

      for macAddr in vfs.floodSet:
         fs = vfs.floodSet.get( macAddr )
         if ( not fs or fs.macAddr != macAddr or
              fs.floodSetType == 'floodSetTypeInvalid' ):
            continue
         for d in fs.destSet:
            dest = getDestModel( d, populateDomain=True, populateVtepType=True )
            if dest and dest.destType in destTypes:
               destDict[ dest ].append( vlanId )
   
   for dest, vlanRange in destDict.items():
      dest.vlanRange = vlanRange
      destFsSummary.dests.append( dest )

   return destFsSummary

enumDecoder = Tac.Value( 'L2Rib::SourceDecoder' )

def getInputFloodSetSummaryColl( sourceName, vlan=None, vtep=None ):
   allSde = allSourceDirEntries()
   fsSummaryColl = FloodSetSummaryColl()
   for sdeName in allSde:
      if( sourceName == 'all' or sourceName == sdeName ):
         sde = allSde[ sdeName ]

         tableName = f'{aliasResolve.prettyName( sdeName )} Input'
         fsSummary = getFloodSetSummaryModel( sde, tableName, vlan, vtep,
                                              intfId=None, label=None )
         fsSummaryColl.floodSetSummaries.append( fsSummary )
   return fsSummaryColl

def getInputDestFloodSetSummaryColl( sourceName, destType=None ):
   allSde = allSourceDirEntries()
   destFsSummaryColl = DestFloodSetSummaryColl()
   for sdeName in allSde:
      if( sourceName == 'all' or sourceName == sdeName ):
         sde = allSde[ sdeName ]

         tableName = f'{aliasResolve.prettyName( sdeName )} Input'
         destFsSummary = getDestFloodSetSummaryModel( sde, tableName, destType )
         destFsSummaryColl.destFloodSetSummaries.append( destFsSummary )
   return destFsSummaryColl

def showL2RibInputFloodSet( mode, source, vlan=None, vtep=None ):
   sourceName = aliasResolve.tableName( source )

   allSde = allSourceDirEntries()
   if( sourceName not in allSde and sourceName != 'all' ):
      mode.addError( "No input table %s." % source )
      return None

   return getInputFloodSetSummaryColl( sourceName, vlan, vtep )

def showL2RibInputDestFloodSet( mode, source, destType=None ):
   sourceName = aliasResolve.tableName( source )

   allSde = allSourceDirEntries()
   if( sourceName not in allSde and sourceName != 'all' ):
      mode.addError( "No input table %s." % source )
      return None

   return getInputDestFloodSetSummaryColl( sourceName, destType )

def fsSourceStr( source ):
   prettyNames = list()
   for s in aliasResolve.sourceName():
      if source & aliasResolve.decodedEnum( s ):
         prettyNames.append( aliasResolve.prettyName( s ) )

   if not prettyNames:
      return 'None'

   return ', '.join( prettyNames )

def getOutputFloodSetSummaryColl( vlan=None, vtep=None ):
   tableName = "Output"
   fsSummaryColl = FloodSetSummaryColl()
   fsSummary = getFloodSetSummaryModel( cppPlugin.outputFloodSetTable,
                                        tableName, vlan, vtep,
                                        intfId=None, label=None )
   if cppPlugin.outputFloodSetTable.source:
      fsSummary.source = fsSourceStr(
                         cppPlugin.outputFloodSetTable.source.source )
   fsSummaryColl.floodSetSummaries.append( fsSummary )
   return fsSummaryColl

def showL2RibOutputFloodSet( mode, vlan=None, vtep=None ):
   return getOutputFloodSetSummaryColl( vlan, vtep )

def getOutputDestFloodSetSummaryColl( destType=None ):
   tableName = "Output"
   destFsSummaryColl = DestFloodSetSummaryColl()
   destFsSummary = getDestFloodSetSummaryModel( cppPlugin.outputFloodSetTable,
                                                tableName, destType )
   if cppPlugin.outputFloodSetTable.source:
      destFsSummary.source = fsSourceStr(
                        cppPlugin.outputFloodSetTable.source.source )
   destFsSummaryColl.destFloodSetSummaries.append( destFsSummary )
   return destFsSummaryColl

def showL2RibOutputDestFloodSet( mode, destType=None ):
   return getOutputDestFloodSetSummaryColl( destType )
 
def showL2RibSummary( mode, args ):
   cppPlugin.showL2RibSummary( sys.stdout.fileno(), mode.session_.outputFormat() )
   return Summary

def getLabelModel( label, level, detail ):
   labelModel = Label()
   labelModel.level = level
   labelModel.index = label.key if detail else None
   labelModel.label = label.label
   return labelModel

def getLbModel( lb, level, detail ):
   lbModel = LoadBalance()
   lbModel.num = len( lb.next )
   lbModel.level = level
   lbModel.index = lb.key if detail else None
   return lbModel

class L2RibHostModel:
   def __init__( self, tableName, destTable=None, labelTable=None, lbTable=None ):
      """Provides methods to populate L2RibModel.HostTable."""
      self.tableName = tableName
      self.destTable = destTable
      self.labelTable = labelTable
      self.lbTable = lbTable

   def getHostModel( self, host, detail=True, source=None, isDestCmd=False ):
      """Returns L2RibModel.HostEntry CLI model from host."""
      hostModel = HostEntryAndSource() if self.tableName == 'output' or \
                  source is not None else HostEntry()
      hostModel.macAddress = host.key.macaddr
      hostModel.vlanId = host.key.vlanId
      hostModel.seqNo = host.seqNo
      hostModel.pref = host.preference
      hostModel.entryType = host.entryType
      if self.tableName == 'output':
         hostModel.source = aliasResolve.prettyName( host.source )
      elif source is not None:
         hostModel.source = aliasResolve.prettyName( source )
      retVal = self._updateDestModel( host, hostModel, detail, isDestCmd )
      return ( retVal, hostModel )

   def _updateDestModel( self, host, hostModel, detail, isDestCmd=False ):
      """Populate hostModel.dests for intf/multi-table destination"""
      if host.intfIsStored:
         return self._updateIntfDest( host, hostModel, isDestCmd )
      elif host.nextIsStored:
         return self._updateMtDests( host.next, hostModel, 1, detail, isDestCmd )
      else:
         return False

   def _updateIntfDest( self, host, hostModel, isDestCmd=False ):
      """Construct Dest Model for intf destination objects"""
      dest = Dest()
      # Set level as 0 for 'show l2rib [input <>|output] destination' commands
      dest.level = int( not isDestCmd )
      dest.destType = 'Interface'
      dest.interface = host.intf
      dest.dropMode = 'none'
      hostModel.dests.append( dest )
      return True

   def _updateMtDests( self, nextObj, hostModel, startLevel, detail,
                       isDestCmd=False ):
      def _updateDestAction( objTuple, entry, level ):
         getEntryModel = None
         if objTuple.tableType == 'tableTypeDest':
            getEntryModel = getDestModel
         elif objTuple.tableType == 'tableTypeLabel':
            getEntryModel = getLabelModel
         elif objTuple.tableType == 'tableTypeLoadBalance':
            getEntryModel = getLbModel
         else:
            assert False, "Unexpected table type" % objTuple.tableType
         if isDestCmd:
            # Set level as 0 for 'show l2rib [input <>|output] destination' commands
            level = 0
         if getEntryModel is getDestModel:
            hostModel.dests.append( getEntryModel( entry, level, detail,
                                                   populateDomain=isDestCmd,
                                                   populateVtepType=isDestCmd ) )
         else:
            hostModel.dests.append( getEntryModel( entry, level, detail ) )
      return traverseAndValidateMultiTable( nextObj, self.destTable,
                                            self.labelTable, self.lbTable,
                                            startLevel, _updateDestAction )

# ----------------------------------------------------------------------------------

# sources with helptext
sourceMatchConf = { 'all':'Show all L2 RIB input tables' }
hiddenMatchConf = {}
for tk in aliasResolve.commandToken():
   sourceHelpDesc = 'Show L2 RIB {} input tables'.format(
         aliasResolve.prettyName( tk ) )
   sourceMatchConf[ tk ] = sourceHelpDesc
   aliasList = aliasResolve.commandAlias( tk )
   if isinstance( aliasList, list ):
      hiddenMatchConf.update( { alias:sourceHelpDesc
                                for alias in aliasList } )

class InputMatcher( CliCommand.CliExpression ):
   expression = 'input SOURCE'
   data = { 'input':'Show L2 RIB input tables',
            'SOURCE':CliMatcher.EnumMatcher( sourceMatchConf,
                                             hiddenMatchConf ) }

# ----------------------------------------------------------------------------------

# data for common tokens
l2RibHelpDesc = 'Show L2 RIB information'
outputHelpDesc = 'Show L2 RIB output tables'
destHelpDesc = 'Show L2 RIB destinations'
floodSetHelpDesc = 'Show L2Rib floodset from input source'

# adapters to connect old-parser handlers to new parser
def showL2RibInputAdapter( mode, args ):
   # used 'get' where optional AND default value is NONE. [] otherwise for assert.
   return showL2RibInput( mode=mode, source=extractSource( args ),
                          detail=( 'detail' in args ),
                          vlan=args.get( 'VLAN' ),
                          mac=args.get( 'MAC' ),
                          vtep=args.get( 'VTEP' ),
                          intf=( args.get( 'PHY' ) or args.get( 'VIRT' ) ) )
def showL2RibInputFloodSetAdapter( mode, args ):
   return showL2RibInputFloodSet( mode=mode, source=extractSource( args ),
                                  vlan=args.get( 'VLAN' ),
                                  vtep=args.get( 'VTEP' ) )
def showL2RibInputDestAdapter( mode, args ):
   return showL2RibInputDest( mode=mode, source=extractSource( args ),
                              destType=args.get( 'DEST' ) )
def showL2RibInputDestFloodSetAdapter( mode, args ):
   return showL2RibInputDestFloodSet( mode=mode, source=extractSource( args ),
                                      destType=args.get( 'DEST' ) )
def showL2RibOutputAdapter( mode, args ):
   return showL2RibOutput( mode=mode,
                           detail=( 'detail' in args ),
                           vlan=args.get( 'VLAN' ),
                           mac=args.get( 'MAC' ),
                           vtep=args.get( 'VTEP' ),
                           intf=( args.get( 'PHY' ) or args.get( 'VIRT' ) ) )
def showL2RibOutputFloodSetAdapter( mode, args ):
   return showL2RibOutputFloodSet( mode=mode,
                                   vlan=args.get( 'VLAN' ),
                                   vtep=args.get( 'VTEP' ) )
def showL2RibOutputDestAdapter( mode, args ):
   return showL2RibOutputDest( mode=mode, destType=args.get( 'DEST' ) )
def showL2RibOutputDestFloodSetAdapter( mode, args ):
   return showL2RibOutputDestFloodSet( mode=mode, destType=args.get( 'DEST' ) )

# entry filter tokens, grouped appropriately
class FloodSetFilter( CliCommand.CliExpression ):
   expression = '[ ( vlan VLAN ) ] [ ( vtep VTEP ) ]'
   data = { 'vlan':'Filter by VLAN id',
            'VLAN':CliMatcher.IntegerMatcher( 1, 4094, helpdesc='Choose VLAN ID' ),
            'vtep':'Filter by VTEP',
            'VTEP': IpGenAddrMatcher.IpGenAddrMatcher( 
               helpdesc="IP address of VTEP" ),
           }
class InputFilter( CliCommand.CliExpression ):
   expression = ( '[ FS_FILTER ]' +
                '[ ( mac MAC ) ] [ ( interface ( PHY | VIRT ) ) ] [ detail ]' )
   data = { 'FS_FILTER':FloodSetFilter,
            'mac':'Filter by MAC address',
            'MAC':MacAddr.MacAddrMatcher(),
            'interface':'Filter by destination interface',
            'PHY':EthIntfCli.EthPhyIntf.ethMatcher,
            'VIRT':LagIntfCli.EthLagIntf.matcher,
            'detail':'More comprehensive output' }

class DestFilter( CliCommand.CliExpression ):
   expression = ( '[ DEST ]' )
   data = { 'DEST': CliMatcher.EnumMatcher( {
                     'cpu': 'Show L2 RIB CPU type destinations',
                     'interface': 'Show L2 RIB Interface type destinations',
                     'mpls':'Show L2 RIB MPLS type destinations',
                     'vxlan':'Show L2 RIB VXLAN type destinations',
                     'tunnel': 'Show L2 RIB Tunnel type destinations'  } )
           }
   
# ----------------------------------------------------------------------------------
# add commands
# ----------------------------------------------------------------------------------

# show l2rib summary
class L2RibSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show l2Rib summary'
   data = { 'l2Rib':l2RibHelpDesc,
            'summary':'Show L2 Rib summary of known hosts and sources' }
   handler = showL2RibSummary
   cliModel = Summary
BasicCli.addShowCommandClass( L2RibSummaryCmd )

# input tables
class L2RibInputCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show l2Rib INPUT [ FILTERS ]'
   data = { 'l2Rib':l2RibHelpDesc,
            'INPUT':InputMatcher,
            'FILTERS': InputFilter }
   handler = showL2RibInputAdapter
   cliModel = HostTable
BasicCli.addShowCommandClass( L2RibInputCmd )

# input floodset tables
class L2RibInputFloodSetCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show l2Rib INPUT floodset [ FILTERS ]'
   data = { 'l2Rib':l2RibHelpDesc,
            'INPUT':InputMatcher,
            'floodset':floodSetHelpDesc,
            'FILTERS':FloodSetFilter }
   handler = showL2RibInputFloodSetAdapter
   cliModel = FloodSetSummaryColl
BasicCli.addShowCommandClass( L2RibInputFloodSetCmd )

# input destination tables
class L2RibInputDestCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show l2Rib INPUT destination [ FILTERS ]'
   data = { 'l2Rib':l2RibHelpDesc,
            'INPUT':InputMatcher,
            'destination':destHelpDesc,
            'FILTERS':DestFilter }
   handler = showL2RibInputDestAdapter
   cliModel = DestSummaryColl
BasicCli.addShowCommandClass( L2RibInputDestCmd )

# input destination floodset tables
class L2RibInputDestFloodSetCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show l2Rib INPUT destination floodset [ FILTERS ]'
   data = { 'l2Rib':l2RibHelpDesc,
            'INPUT':InputMatcher,
            'destination':destHelpDesc,
            'floodset':floodSetHelpDesc,
            'FILTERS':DestFilter }
   handler = showL2RibInputDestFloodSetAdapter
   cliModel = DestFloodSetSummaryColl
BasicCli.addShowCommandClass( L2RibInputDestFloodSetCmd )

# output tables
class L2RibOutputCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show l2Rib output [ FILTERS ]'
   data = { 'l2Rib':l2RibHelpDesc,
            'output':outputHelpDesc,
            'FILTERS':InputFilter }
   handler = showL2RibOutputAdapter
   cliModel = HostTable
BasicCli.addShowCommandClass( L2RibOutputCmd )

# output floodset tables
class L2RibOutputFloodSetCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show l2Rib output floodset [ FILTERS ]'
   data = { 'l2Rib':l2RibHelpDesc,
            'output':outputHelpDesc,
            'floodset':floodSetHelpDesc,
            'FILTERS':FloodSetFilter }
   handler = showL2RibOutputFloodSetAdapter
   cliModel = FloodSetSummaryColl
BasicCli.addShowCommandClass( L2RibOutputFloodSetCmd )

# output destination tables
class L2RibOutputDestCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show l2Rib output destination [ FILTERS ]'
   data = { 'l2Rib':l2RibHelpDesc,
            'output':outputHelpDesc,
            'destination':destHelpDesc,
            'FILTERS':DestFilter }
   handler = showL2RibOutputDestAdapter
   cliModel = DestSummaryColl
BasicCli.addShowCommandClass( L2RibOutputDestCmd )

# output destination floodset tables
class L2RibOutputDestFloodSetCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show l2Rib output destination floodset [ FILTERS ]'
   data = { 'l2Rib':l2RibHelpDesc,
            'output':outputHelpDesc,
            'destination':destHelpDesc,
            'floodset':floodSetHelpDesc,
            'FILTERS':DestFilter }
   handler = showL2RibOutputDestFloodSetAdapter
   cliModel = DestFloodSetSummaryColl
BasicCli.addShowCommandClass( L2RibOutputDestFloodSetCmd )

#----------------------------------------------------------------------------------
# register 'show l2rib input all', 'show l2rib input all floodset',
# 'show l2rib output', 'show l2rib output floodset' into 'show tech-support
# extended evpn'.
#-----------------------------------------------------------------------------------
TechSupportCli.registerShowTechSupportCmd(
   '2017-11-03 12:06:06',
   cmds=[ 'show l2rib input all',
          'show l2rib input all floodset',
          'show l2rib output',
          'show l2rib output floodset' ],
   extended='evpn' )

#----------------------------------------------------------------------------------
# register 'show l2rib summary' into 'show tech-support
#-----------------------------------------------------------------------------------
TechSupportCli.registerShowTechSupportCmd(
   '2023-03-13 12:06:06',
   cmds=[ 'show l2rib summary' ] )

def Plugin( entityManager ):
   global storedEntityManager, cppPlugin
   storedEntityManager = entityManager
   cppPlugin = Tac.newInstance( 'L2RibCli::Helper',
                                entityManager.cEntityManager(),
                                aliasResolve.back )
