#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import ArPyUtils
import CliCommand, CliMatcher, ShowCommand
import CommonGuards
import BasicCli
import Tac
from CliPlugin import EventMonCli
from CliPlugin.EthIntfCli import EthPhyAutoIntfType
from CliPlugin.LagIntfCli import LagAutoIntfType
from CliPlugin.LagModel import (
    LacpEventMonReasonDetails,
    LacpEventMonReasons,
)
import sys
import time
from Intf.IntfRange import IntfRangeMatcher

def showLacpTable( mode, args ):
   return showLacpTableVerbose( mode, args )

def showLacpTableVerbose( mode, args ):
   lacpEventMonReason = Tac.Value( "Lacp::LacpEventMonReason" )
   model = LacpEventMonReasons( _detail='verbose' in args )

   with ArPyUtils.FileHandleInterceptor( [ sys.stdout.fileno() ] ) as capturedOut:
      EventMonCli.showTable( mode, args, "lacp" )
      sys.stdout.flush()

   output = capturedOut.contents().split( '\n' )[ : -1 ]
      
   for entry in output:
      originalFields = entry.split( "|" )
      finalFields = {}
      fieldHeaders = [ "time", "intf", "portChannel",  "sm", "state", 
                       "reason", "portEn", "lacpEn", "currEx", "lacpduRx", 
                       "portMoved", "selected", "partnerSync", "count" ]
      if len( originalFields ) != len( fieldHeaders ):
         # Output does not contain proper fields, so it is most likely an
         # error message so we will print the error message and exit
         print( '\n'.join( output ) )
         return model
      for field, header in zip( originalFields, fieldHeaders ):
         if header == "reason":
            key = int( field )
            if finalFields[ "sm" ] == "rx":
               finalFields[ header ] = lacpEventMonReason.rxSmReason[ key ]
            elif finalFields[ "sm" ] == "mux":
               finalFields[ header ] = lacpEventMonReason.muxSmReason[ key ]
         elif header == "portChannel":
            if field == "unknown":
               finalFields[ header ] = ""
            else:
               finalFields[ header ] = field
         elif header == "state":
            if finalFields[ "sm" ] == "rx":
               finalFields[ header ] = field.replace( "rxSm", "" )
            elif finalFields[ "sm" ] == "mux":
               finalFields[ header ] = field.replace( "muxSm", "" )
         elif header == "time":
            # ignore microseconds, if present
            finalFields[ "time" ] = field.split( '.' )[ 0 ]
         else:
            finalFields[ header ] = field

      timestamp = time.mktime( time.strptime( finalFields[ "time" ],
                                              "%Y-%m-%d %H:%M:%S" ) )
      lacpEventModel = LacpEventMonReasonDetails( _rawOutput = entry )
      lacpEventModel.time = timestamp
      lacpEventModel.lag = Tac.Value( 'Arnet::IntfId', finalFields[ "portChannel" ] )
      lacpEventModel.member = Tac.Value( 'Arnet::IntfId', finalFields[ "intf" ] )
      lacpEventModel.state = finalFields[ "state" ]
      lacpEventModel.reason = finalFields[ "reason" ]
      model.eventMonReasons.append( lacpEventModel )

   return model

class EventMonitorLacpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show event-monitor lacp [ verbose ] '
              '[ { ( group-by GROUPBY ) | ( limit LIMIT ) | ' 
              '( match-time TIME ) | '
              '( match-interface ( ETH | LAG ) ) | '
              '( match-sm MATCH_SM ) } ]' )
   data = {
      'event-monitor' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'event-monitor',
            helpdesc='Show event monitor logs' ),
         guard=CommonGuards.standbyGuard ),
      'lacp' : 'Monitor LACP table events',
      'verbose' : 'Print verbose output',
      'group-by' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'group-by',
            helpdesc='Group the results by attribute' ),
         maxMatches=1 ),
      'GROUPBY' : CliCommand.Node(
         matcher=CliMatcher.EnumMatcher( {
            'member': 'Group by member',
            'port-channel': 'Group by port-channel',
            'state-machine': 'Group by state machine',
         } ),
         maxMatches=1 ),
      'limit' : CliCommand.Node( 
         matcher=CliMatcher.KeywordMatcher( 'limit',
            helpdesc='Limit the number of messages' ),
         maxMatches=1 ),
      'LIMIT' : CliCommand.Node(
         matcher=CliMatcher.IntegerMatcher( 1, 15000, helpdesc='Message limit' ),
         maxMatches=1 ),
      'match-time' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'match-time',
            helpdesc='Filter results by time' ),
         maxMatches=1 ),
      'TIME' : CliCommand.Node(
         matcher=CliMatcher.EnumMatcher( {
            'last-minute': 'Match for the last minute',
            'last-hour': 'Match for the last hour',
            'last-day': 'Match for the last day',
            'last-week': 'Match for the last week',
         } ),
         maxMatches=1 ),
      'match-interface': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'match-interface',
            helpdesc='Filter results by intf' ),
         maxMatches=2 ),
      'ETH' : CliCommand.Node(
         matcher=IntfRangeMatcher( explicitIntfTypes={ EthPhyAutoIntfType } ),
         maxMatches=1 ), 
      'LAG' : CliCommand.Node(
         matcher=IntfRangeMatcher( explicitIntfTypes={ LagAutoIntfType } ),
         maxMatches=1 ),
      'match-sm' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'match-sm',
            helpdesc='Filter results by state machine name' ),
         maxMatches=1 ),
      'MATCH_SM' : CliMatcher.EnumMatcher( {
            'mux': 'Mux state machine',
            'rx': 'Receive state machine',
      } ),
   }
   handler = showLacpTable
   cliModel = LacpEventMonReasons
   privileged = True

BasicCli.addShowCommandClass( EventMonitorLacpCmd )
