# Copyright (c) 2008, 2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Tac, Arnet, CliParser, BasicCli, Tracing
from CliPlugin import AclCli
from CliPlugin import EbraCliLib
from CliPlugin import IntfCli
from CliPlugin import IntfRangeCli
from CliPlugin import LagModel
from CliPlugin import WaitForWarmupCli
from CliPlugin import LagCliLib
from CliPlugin import LagCommonCliLib
import CliGlobal
import CliCommand, CliMatcher
import LazyMount, ConfigMount
from LacpConstants import * # pylint: disable=wildcard-import
from LacpLib import * # pylint: disable=wildcard-import
import CliToken.PortChannel, CliToken.LoadBalance, CliToken.Switch
import CliExtensions
from CliMode.LagLoadBalanceMode import LBPoliciesMode
from Intf import IntfRange
from IntfRangePlugin import LagIntf
from Toggles.SwagCoreToggleLib import toggleSwagPhase1Enabled

# pkgdeps: library Lag

traceHandle = Tracing.Handle( 'LagCli' )
t0 = traceHandle.trace0

# Globals written by the Plugin function at the end of this file
lagConfig = None
lagStatus = None
lacpConfig = None
lagGroupDir = None
bridgingConfig = None
bridgingCliConfig = None
ethIntfStatusDir = None
ethPhyIntfConfigSliceDir = None
agentStatus = None
hwConfig = None
hwStatus = None
lagInternalStatus = None
gv = CliGlobal.CliGlobal( hwFcConfig=None, hwFcStatus=None )

lagPrefix = "Port-Channel"
recircPrefix = "Recirc-Channel"
fabricChannelPrefix = "Fabric-Channel"

MinBandwidthTimeoutBase = Tac.Type( "Lag::MinBandwidthTimeoutBase" )
PortChannelNum = LagCliLib.PortChannelNum
PortChannelIntfId = Tac.Type( "Arnet::PortChannelIntfId" )
FabricChannelIntfId = LagCliLib.FabricChannelIntfId
portChannel = PortChannelIntfId.portChannel
portChannelConfigLimit = Tac.Type( "Lag::PortChannelConfigLimit" ).limit
LacpTimeoutMultiplier = Tac.Type( "Lacp::TimeoutMultiplier" )
IntfState = Tac.Type( 'Interface::IntfEnabledState' )

__defaultTraceHandle__ = Tracing.Handle( "EthCli" )

def recircChannelRangeFn( mode, context ):
   return ( PortChannelNum.min, PortChannelNum.max )

def extendedlagIdSupportedGuard( mode, token ):
   if LagCliLib.bridgingHwCapabilities.extendedLagIdSupported:
      return None
   return CliParser.guardNotThisPlatform

def lagHwInterlockSupportedGuard( mode, token ):
   if LagCliLib.bridgingHwCapabilities.lagSupported and \
         LagCliLib.bridgingHwCapabilities.lagHwInterlockSupported:
      return None
   return CliParser.guardNotThisPlatform

def tryWaitForWarmup( mode ):
   if( mode ): # pylint: disable=superfluous-parens
      WaitForWarmupCli.tryWaitForWarmupAfterVlanPortChange( mode )

# explain why we may have disabled bridging, if a Cli command needs an
# explanation. Also return legend for show vlan configured-ports.
def ethPhyIntfInLag( intfName, mode ):
   epilc = LagCliLib.ethPhyIntfLagConfigDir( mode ).get( intfName )
   lag = epilc and epilc.lag
   if lag:
      return( "in " + IntfCli.Intf.getShortname( lag.intfId ),
              "Member of " + lag.intfId,
              "port channel configuration" )
   else:
      return( None, None, None )

def ethPhyIntfInLagRecirculation( intfName, mode ):
   epilc = LagCliLib.ethPhyIntfLagConfigDir( mode ).get( intfName )
   lag = epilc and epilc.lag
   if lag:
      return( "in " + IntfCli.Intf.getShortname( lag.intfId ),
              "Member of " + lag.intfId + " recirc",
              "port channel recirculation configuration" )
   else:
      return( None, None, None )

def hardwareLagConfig( intfId ):
   if FabricChannelIntfId.isFabricChannelIntfId( intfId ):
      return gv.hwFcConfig
   else:
      return hwConfig

def hardwareLagStatus( intfId ):
   if FabricChannelIntfId.isFabricChannelIntfId( intfId ):
      return gv.hwFcStatus
   else:
      return hwStatus

IntfCli.dataLinkExplanationHook.addExtension( ethPhyIntfInLag )
IntfCli.recirculationExplanationHook.addExtension( ethPhyIntfInLagRecirculation )

lagConfigLimitReachedMsg = ( "Port channel config limit %d reached. "
                             "No interfaces were created." % portChannelConfigLimit )

def lagConfigLimitReachedExtension( intfs ):
   # Extension only applies to port channels
   if intfs.type_ is not LagIntf.LagAutoIntfType:
      return None

   # If the number of configs that either exist or are being added
   # is greater than the config limit, return an error message
   total = len( set( intfs ).union( LagCliLib.lagIntfConfigDir.intfConfig ) )
   if total > portChannelConfigLimit:
      return lagConfigLimitReachedMsg

   return None

IntfRangeCli.canCreateIntfsHook.addExtension( lagConfigLimitReachedExtension )

class ChannelGroup:
   def __init__( self, id ): # pylint: disable=redefined-builtin
      self.id_ = id
      self.newborn = 0
      self.lagPrefix_ = lagPrefix

   id = property( lambda self: self.id_ )
   idStr = property( lambda self: f"{self.lagPrefix_}{self.id}" )

   # Create the port-channel interface corresponding to this object.
   # This function is called whenever the config-if channel-group N
   # command is used, or when handling the "interface Port-ChannelN"
   # mode.
   def create( self, mode ):
      lagDir = LagCliLib.lagIntfConfigDir
      if self.idStr not in lagDir.intfConfig:
         if len( LagCliLib.lagIntfConfigDir.intfConfig ) >= portChannelConfigLimit:
            return None
         lagDir.intfConfig.newMember( self.idStr )
         self.newborn = 1
         LagCliLib.updateLacpPortIdRangeForLag( self.idStr )
         if lagDir.intfConfig[ self.idStr ].adminEnabledState() == \
            IntfState.unknownEnabledState:
            lagDir.intfConfig[ self.idStr ].adminEnabled = True
      else:
         self.newborn = 0
      return lagDir.intfConfig[ self.idStr ]

   def get( self, mode ):
      lagDir = LagCliLib.lagIntfConfigDir
      return lagDir.intfConfig.get( self.idStr, None )

class RecircChannelGroup( ChannelGroup ):
   def __init__( self, id ): # pylint: disable=redefined-builtin
      self.id_ = id
      self.newborn = 0
      self.lagPrefix_ = recircPrefix

class FabricChannelGroup( ChannelGroup ):
   def __init__( self, id ): # pylint: disable=redefined-builtin
      self.id_ = id
      self.newborn = 0
      self.lagPrefix_ = fabricChannelPrefix

class EthPhyLagIntfDependent( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      epilc = LagCliLib.ethPhyIntfLagCliConfig( mode=None, create=False,
                                                name=self.intf_.name )
      if epilc:
         del LagCliLib.lacpCliConfig.rateLimitIntfEnable[ epilc.intfId ]
         del LagCliLib.lagConfigCli.phyIntf[ epilc.intfId ]

# Only pass in mode when the mode represents a single EthPhyIntf
def maybeDeleteEthPhyIntfLagConfig( epilc, mode, inputDir=None ):
   if mode and not epilc:
      epilc = LagCliLib.ethPhyIntfLagCliConfig( mode=mode,
                                                create=False, name=mode.intf.name )
   if epilc:
      if inputDir:
         epilcd = inputDir
      else:
         epilcd = LagCliLib.lagConfigCli
      name = epilc.intfId
      if( epilc.lag == None and # pylint: disable=singleton-comparison
          epilc.mode == "lacpModeOff" and
          epilc.priority == PortPriorityDefault and
          epilc.portId == LacpPortIdDefault and
          epilc.timeout == 'lacpLongTimeout' and
          epilc.timeoutMult == LacpTimeoutMultiplier.defaultValue and
          epilc.fwdTxEnabled == True ): # pylint: disable=singleton-comparison
         del epilcd.phyIntf[ name ]

class ChannelGroupModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return intfSupportsLag( mode.intf.name )

#-------------------------------------------------------------------------------
# Rule to match the token class <channelGroup_id>, returning a corresponding
# ChannelGroup object, or <channelGroup_id list>, returning a list of
# corresponding ChannelGroup objects.
#-------------------------------------------------------------------------------
channelGroupMatcher = CliMatcher.DynamicIntegerMatcher( LagIntf.portChannelRangeFn,
                                                        helpdesc='Channel Group ID' )

recircGroupMatcher = CliMatcher.DynamicIntegerMatcher( recircChannelRangeFn,
                                                       helpdesc='Channel Group ID' )

def channelGroups( match ):
   # match is idList
   return list( map( ChannelGroup, match ) )

#-------------------------------------------------------------------------------
# Rule to match the token class <recircChannelGroup_id>, returning a corresponding
# RecircChannelGroup object, or <recirCchannelGroup_id list>, returning a list of
# corresponding RecircChannelGroup objects.
#-------------------------------------------------------------------------------

def recircChannelGroups( match ):
   # match is idList
   return list( map( RecircChannelGroup, match ) )

canSetSystemPriorityHook = CliExtensions.CliHook()

nodeLacp = CliCommand.Node(
   matcher=CliMatcher.KeywordMatcher(
      'lacp', helpdesc='Set Link Aggregation Control Protocol (LACP) parameters' ),
   guard=LagCommonCliLib.lagSupportedGuard )

#--------------------------
# Load-Balance polices mode
#--------------------------

class ConfigLoadBalancePoliciesMode( BasicCli.ConfigModeBase, LBPoliciesMode ):
   name = "LoadBalance Policies"

   def __init__( self, parent, session ):
      LBPoliciesMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#-------------------------------------------------------------------------------
# The "port-channel load-balance [src-mac|dst-mac]" command, in "config" mode.
# Implemented in FocalPoint and Sand.
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# The "[no|default] channel-protocol lacp" command, in "config-if" mode.
# This command is only valid on an aggregable interface, and is a
# no-op; we only support LACP.
#-------------------------------------------------------------------------------

class ChannelProtocolCmd( CliCommand.CliCommandClass ):
   syntax = 'channel-protocol lacp'
   noOrDefaultSyntax = 'channel-protocol ...'
   data = {
      'channel-protocol' : (
         'Set the channel protocol associated with this interface' ),
      'lacp' : nodeLacp
   }
   hidden = True
   handler = lambda mode, args: None
   noOrDefaultHandler = handler

ChannelGroupModelet.addCommandClass( ChannelProtocolCmd )

#-------------------------------------------------------------------------------
# The "[no|default] lacp port-priority N" command, in "config-if" mode.
# This command is only valid on an aggregable interface.
#-------------------------------------------------------------------------------

def setPortPriority( mode, args ):
   portPriority = args.get( 'PRIORITY', PortPriorityDefault )
   epilc = LagCliLib.ethPhyIntfLagCliConfig( mode, create=True )
   epilc.priority = portPriority
   maybeDeleteEthPhyIntfLagConfig( epilc, mode )

class LacpPortPriorityCmd( CliCommand.CliCommandClass ):
   syntax = 'lacp port-priority PRIORITY'
   noOrDefaultSyntax = 'lacp port-priority ...'
   data = {
      'lacp' : nodeLacp,
      'port-priority' : 'Set the port prioriy associated with this interface',
      'PRIORITY' : CliMatcher.IntegerMatcher(
         PortPriorityMin, PortPriorityMax, helpdesc='LACP Port Priority' )
   }
   handler = setPortPriority
   noOrDefaultHandler = setPortPriority

ChannelGroupModelet.addCommandClass( LacpPortPriorityCmd )

#-------------------------------------------------------------------------------
# The "[no|default] lacp port-id N" command, in "config-if" mode.
#-------------------------------------------------------------------------------

def setPortId( mode, args ):
   portId = args.get( 'ID', LacpPortIdDefault )
   epilc = LagCliLib.ethPhyIntfLagCliConfig( mode, create=True )
   epilc.portId = portId
   maybeDeleteEthPhyIntfLagConfig( epilc, mode )

class LacpPortIdCmd( CliCommand.CliCommandClass ):
   syntax = 'lacp port-id ID'
   noOrDefaultSyntax = 'lacp port-id ...'
   data = {
      'lacp' : nodeLacp,
      'port-id' : 'Set the LACP port number associated with this interface ',
      'ID' : CliMatcher.IntegerMatcher(
         LacpPortIdMin, LacpPortIdMax, helpdesc='LACP Port Id' )
   }
   handler = setPortId
   noOrDefaultHandler = setPortId

ChannelGroupModelet.addCommandClass( LacpPortIdCmd )

#----------------------------------------------------------------------------
# The "[no|default] lacp rate-limit" command, in "config-if" mode.
#----------------------------------------------------------------------------

def setRateLimit( mode, args ):
   epilc = LagCliLib.ethPhyIntfLagCliConfig( mode, create=True )
   LagCliLib.lacpCliConfig.rateLimitIntfEnable[ mode.intf.name ] = True
   maybeDeleteEthPhyIntfLagConfig( epilc, mode )

def defaultRateLimit( mode, args ):
   epilc = LagCliLib.ethPhyIntfLagCliConfig( mode, create=True )
   rateLimit = LagCliLib.lacpCliConfig.rateLimitIntfEnable
   del rateLimit[ mode.intf.name ]
   maybeDeleteEthPhyIntfLagConfig( epilc, mode )

def noRateLimit( mode, args ):
   epilc = LagCliLib.ethPhyIntfLagCliConfig( mode, create=True )
   LagCliLib.lacpCliConfig.rateLimitIntfEnable[ mode.intf.name ] = False
   maybeDeleteEthPhyIntfLagConfig( epilc, mode )

class LacpRateLimitCmd( CliCommand.CliCommandClass ):
   syntax = 'lacp rate-limit'
   noOrDefaultSyntax = 'lacp rate-limit'
   data = {
      'lacp' : nodeLacp,
      'rate-limit' : 'Enable LACPDU rate limiting on this port',
   }
   handler = setRateLimit
   defaultHandler = defaultRateLimit
   noHandler = noRateLimit

ChannelGroupModelet.addCommandClass( LacpRateLimitCmd )

#-------------------------------------------------------------------------------
# The "[no|default] lacp timer (normal | fast)" command, in "config-if" mode.
# This command is only valid on an aggregable interface.
#
# legacy:
# "[no|default] lacp rate (normal | fast)""
#-------------------------------------------------------------------------------

def setLacpRate( mode, lacpRate='normal' ):
   epilc = LagCliLib.ethPhyIntfLagCliConfig( mode, create=True )
   if lacpRate == 'fast':
      epilc.timeout = 'lacpShortTimeout'
   else:
      epilc.timeout = 'lacpLongTimeout'
   maybeDeleteEthPhyIntfLagConfig( epilc, mode )

matcherRate = CliMatcher.EnumMatcher( {
   'normal' : '30 seconds while synchronized, 1 second while synchronizing',
   'fast' : '1 second whether synchronized or not',
} )
matcherTimer = CliMatcher.KeywordMatcher(
   'timer', helpdesc='Set the rate at which the peer sends us LACP PDUs' )

class LacpTimerCmd( CliCommand.CliCommandClass ):
   syntax = 'lacp timer RATE'
   noOrDefaultSyntax = 'lacp timer ...'
   data = {
      'lacp' : nodeLacp,
      'timer' : matcherTimer,
      'RATE' : matcherRate
   }
   @staticmethod
   def handler( mode, args ):
      setLacpRate( mode, lacpRate=args[ 'RATE' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setLacpRate( mode )

class LacpRateCmd( CliCommand.CliCommandClass ):
   syntax = 'lacp rate RATE'
   noOrDefaultSyntax = 'lacp rate ...'
   data = {
      'lacp' : nodeLacp,
      'rate' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'rate',
            helpdesc='Set the rate at which the peer sends us LACP PDUs' ),
         deprecatedByCmd='lacp timer' ),
      'RATE' : matcherRate
   }
   @staticmethod
   def handler( mode, args ):
      setLacpRate( mode, lacpRate=args[ 'RATE' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setLacpRate( mode )

ChannelGroupModelet.addCommandClass( LacpRateCmd )
ChannelGroupModelet.addCommandClass( LacpTimerCmd )

#-------------------------------------------------------------------------------
# The "[no|default] lacp timer multiplier <multiplier>" command, in "config-if" mode.
# This command is only valid on an aggregable interface.
#-------------------------------------------------------------------------------

def setLacpTimeoutMultiplier( mode, args ):
   timeoutMultiplier = args.get( 'MULTIPLIER',
                                 LacpTimeoutMultiplier.defaultValue )
   epilc = LagCliLib.ethPhyIntfLagCliConfig( mode, create=True )
   epilc.timeoutMult = timeoutMultiplier
   maybeDeleteEthPhyIntfLagConfig( epilc, mode )

class LacpTimeoutMultiplierCmd( CliCommand.CliCommandClass ):
   syntax = 'lacp timer multiplier MULTIPLIER'
   noOrDefaultSyntax = 'lacp timer multiplier ...'
   data = {
      'lacp' : nodeLacp,
      'timer' : matcherTimer,
      'multiplier' :
         'Set the multiplier rate of missed LACP PDUs time out a member',
      'MULTIPLIER' : CliMatcher.IntegerMatcher(
         LacpTimeoutMultiplier.min, LacpTimeoutMultiplier.max,
         helpdesc=
            'Consecutively missed LACP PDUs to time out a member (default=3)' )
   }
   handler = setLacpTimeoutMultiplier
   noOrDefaultHandler = setLacpTimeoutMultiplier

ChannelGroupModelet.addCommandClass( LacpTimeoutMultiplierCmd )

#-------------------------------------------------------------------------------
# The "[no|default] channel-group N mode {on|active|passive}" command,
# in "config-if" mode. This command is only valid on an aggregable interface.
#-------------------------------------------------------------------------------

nodeChannelGroup = CliCommand.Node(
   matcher=CliMatcher.KeywordMatcher( 'channel-group',
      helpdesc='Set link aggregation group' ),
   guard=LagCommonCliLib.lagSupportedGuard )

def copySwitchportParams( frm, to ):
   '''Copy switchport parameters.  May really belong in Ebra/VlanCli.'''
   to.enabled = frm.enabled
   to.switchportMode = frm.switchportMode
   to.accessVlan = frm.accessVlan
   to.trunkNativeVlan = frm.trunkNativeVlan
   to.trunkAllowedVlans = frm.trunkAllowedVlans
   to.blockUnknownUnicast = frm.blockUnknownUnicast
   to.blockUnknownMulticast = frm.blockUnknownMulticast
   to.trunkGroup.clear()
   for group in frm.trunkGroup:
      to.trunkGroup[ group ] = True

# pylint: disable-next=inconsistent-return-statements
def getChannelModeFromLacpMode( lagMode ):
   if lagMode == 'lacpModeOff':
      return "on"
   elif lagMode == 'lacpModeActive':
      return "active"
   elif lagMode == 'lacpModePassive':
      return "passive"

def reqSetChannelGroupMode( mode, args ):
   channelGroupId = ChannelGroup( args[ 'ID' ] )
   channelGroupMode = args[ 'MODE' ]
   setChannelGroupMode( mode, channelGroupId, channelGroupMode )

def reqSetRecircChannelGroupMode( mode, args ):
   recircChannelGroupId = RecircChannelGroup( args[ 'ID' ] )
   if ( not LagCliLib.recircHwConfig.autoMemberLoopbackConfiguration and
        mode.intf.config() and
        ( mode.intf.config().loopbackMode == 'loopbackNone' ) ):
      warning = 'Interface %s is not configured for system loopback' % \
         mode.intf.name
      mode.addWarning( warning )
   setChannelGroupMode( mode, recircChannelGroupId, 'on' )

def reqSetFabricChannelGroupMode( mode, args ):
   fabricChannelId = FabricChannelGroup( args[ 'FABRIC_CHANNEL_ID' ] )
   # Skip config validity checks for fabric-channels. BUG971926
   setChannelGroupMode( mode, fabricChannelId, 'on', checkValidity=False )

def setChannelGroupMode( mode, channelGroupId, channelGroupMode,
                         checkValidity=True ):
   # Old configuration values in case we have to rolback the current config
   oldMode = None
   def _oldLagState( _mode ):
      _oldLag = None
      _oldLagId = None
      _oldepilc = LagCliLib.ethPhyIntfLagCliConfig( _mode, create=False )
      if _oldepilc:
         _oldLag = _oldepilc.lag
         if _oldLag:
            # There is no validation done for fabric-channel
            # interfaces yet.
            if FabricChannelIntfId.isFabricChannelIntfId( _oldLag.intfId ):
               return ( _oldepilc, None, None )
            _oldLagId = portChannel( _oldLag.intfId )
      return (_oldepilc, _oldLag, _oldLagId)

   oldLag = {}
   with ConfigMount.ConfigMountDisabler( disable=True ):
      oldLag["Sysdb"] = _oldLagState( mode )
   oldLag["ConfigSession"] = _oldLagState( mode )

   def _lagConfigHookArgs( args ):
      args.append( channelGroupId )
      for src in [ 'Sysdb', 'ConfigSession' ]:
         oldLagIds = [ oldLag[ src ][ 2 ] ]
         intfModes = args[0]
         if len( intfModes ) > 1:
            for im in intfModes[1:]:
               if src == 'Sysdb':
                  with ConfigMount.ConfigMountDisabler( disable=True ):
                     _, _, _oldLagId = _oldLagState( im )
               else:
                  _, _, _oldLagId = _oldLagState( im )
            if _oldLagId:
               oldLagIds.append( _oldLagId )
         args.append( oldLagIds )

   for lagConfigCallback in LagCliLib.lagConfigAllowCallbacks:
      if not mode.maybeIntfRangeCliHook(
         lagConfigCallback, lagConfigCallback.__name__, _lagConfigHookArgs ):
         return

   # the config should be written to in session if running in session
   oldepilc, oldLag, oldLagId = oldLag["ConfigSession"]
   # "no channel-group"
   if channelGroupId is None:
      noChannelGroupMode( oldepilc, mode )
      return

   lag = channelGroupId.create( mode )
   if not lag:
      mode.addError( lagConfigLimitReachedMsg )
      return
   # if there are any current members, warn if we're changing the mode
   if channelGroupMode == 'on':
      desiredLagMode = 'lagModeOn'
   else:
      if mode.intf.name in bridgingConfig.switchIntfConfig:
         switchportMode  = bridgingConfig.switchIntfConfig[ mode.intf.name ]. \
             switchportMode
         if switchportMode in [ 'tap', 'tool' ]:
            warningMessage = 'The interface switchport mode of %s is not ' \
                'compatible with LACP protocol, and will not be able to join ' \
                'the LAG.' % switchportMode
            mode.addWarning( warningMessage )
      desiredLagMode = 'lagModeLacp'
   # pylint: disable-next=consider-using-in
   if lag.mode != 'lagModeUnconfigured' and lag.mode != desiredLagMode:
      #XXX This limitation sucks, but is present in the industry standard.
      # Really, we should set all members to desiredLagMode
      # if it is different, and warn about that.
      members = 0
      for intf in LagCliLib.ethPhyIntfLagCliConfigDir( mode ).values():
         if intf.lag == lag:
            members += 1
      if members:
         mode.addError( "Cannot change mode; remove all members and try again." )
         return
   lag.mode = desiredLagMode
   # Create/get LacpIntfConfig for this interface.
   # Note that all interfaces get LacpConfigs, even those that are
   # statically configured, so that the Marker protocol can run.
   epilc = LagCliLib.ethPhyIntfLagCliConfig( mode, create=True )
   oldLag = epilc.lag
   oldMode = epilc.mode

   # BUG2537: switchIntfConfig copying should be done in sysdb plugin
   if channelGroupId.newborn and \
      not FabricChannelIntfId.isFabricChannelIntfId( lag.intfId ):
      switchIntfConfigs = bridgingCliConfig.switchIntfConfig
      lagName = lag.intfId
      intfName = mode.intf.name
      if intfName in switchIntfConfigs:
         if lagName not in switchIntfConfigs:
            switchIntfConfigs.newMember( lagName, 'access' )
         copySwitchportParams( switchIntfConfigs[ intfName ],
                               switchIntfConfigs[ lagName ] )

   # add this interface as a member of lag
   # Set lag attribute first before mode to avoid attrlog race with
   # SysdbReactors for mode attribute
   epilc.lag = lag
   if channelGroupMode == 'active':
      epilc.mode = 'lacpModeActive'
   elif channelGroupMode == 'passive':
      epilc.mode = 'lacpModePassive'
   else:
      epilc.mode = 'lacpModeOff'

   tryWaitForWarmup( mode )
   if checkValidity:
      for lagConfigValidCallback in LagCliLib.lagConfigValidCallbacks:
         ( status, err ) = lagConfigValidCallback(
            mode, epilc.intfId, lag.intfId, oldLag.intfId if oldLag else None )
         if not status:
            if not err:
               err = "Unknown"

            # Revert back the config
            if oldLag:
               oldLagId = portChannel( oldLag.intfId )
               channelGroupId = ChannelGroup( oldLagId )
               channelGroupMode = getChannelModeFromLacpMode( oldMode )
               setChannelGroupMode( mode, channelGroupId, channelGroupMode,
                     checkValidity=False )
            else:
               noChannelGroupMode( epilc, mode )
            mode.addError( 'Cannot add %s to a port channel (%s)' %
                           ( mode.intf.shortname, err ) )
            break
   maybeDeleteEthPhyIntfLagConfig( epilc, mode )

def noChannelGroupMode( epilc, mode ):
   if epilc:
      name = epilc.intfId # pylint: disable=unused-variable
      epilc.lag = None
      epilc.mode = 'lacpModeOff'
   elif mode:
      name = mode.intf.name
   else:
      return
   maybeDeleteEthPhyIntfLagConfig( epilc, mode )
   tryWaitForWarmup( mode )

def reqNoChannelGroupMode( mode, args ):
   setChannelGroupMode( mode, None, None )

class ChannelGroupModeCmd( CliCommand.CliCommandClass ):
   syntax = 'channel-group ID mode MODE'
   noOrDefaultSyntax = 'channel-group ...'
   data = {
      'channel-group' : nodeChannelGroup,
      'ID' : channelGroupMatcher,
      'mode' : 'Select LACP or static link aggregation',
      'MODE' : CliMatcher.EnumMatcher( {
         'on' : 'Enable static link aggregation',
         'active' : 'Enable LACP in active mode',
         'passive' : 'Enable LACP in passive mode',
      } )
   }
   handler = reqSetChannelGroupMode
   noOrDefaultHandler = reqNoChannelGroupMode

class RecircChannelGroupModeCmd( CliCommand.CliCommandClass ):
   syntax = 'channel-group recirculation ID'
   data = {
      'channel-group' : nodeChannelGroup,
      'recirculation' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'recirculation',
            helpdesc='Enable recirculation with static link aggregation' ),
         guard=LagCliLib.recircGuard ),
      'ID' : recircGroupMatcher
   }
   handler = reqSetRecircChannelGroupMode

class FabricChannelGroupModeCmd( CliCommand.CliCommandClass ):
   syntax = 'channel-group fabric FABRIC_CHANNEL_ID'
   data = {
      'channel-group': nodeChannelGroup,
      'fabric': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'fabric',
            helpdesc='Enable fabric channel with static link aggregation' ),
         guard=LagCliLib.fabricChannelGuard ),
      'FABRIC_CHANNEL_ID': LagCliLib.FabricChannelIntfNumberMatcher
   }
   handler = reqSetFabricChannelGroupMode

ChannelGroupModelet.addCommandClass( ChannelGroupModeCmd )
ChannelGroupModelet.addCommandClass( RecircChannelGroupModeCmd )
if toggleSwagPhase1Enabled():
   ChannelGroupModelet.addCommandClass( FabricChannelGroupModeCmd )

#-------------------------------------------------------------------------------
# The "[no|default] forwarding port-channel transmit disabled" command,
# in "config-if" mode.
# This command is only valid on an aggregable interface.
#-------------------------------------------------------------------------------
def setLagPortTransmitDisabled( mode, args ):
   epilc = LagCliLib.ethPhyIntfLagCliConfig( mode, create=True )
   epilc.fwdTxEnabled = False
   maybeDeleteEthPhyIntfLagConfig( epilc, mode )

def setNoLagPortTransmitDisabled( mode, args ):
   epilc = LagCliLib.ethPhyIntfLagCliConfig( mode, create=True )
   epilc.fwdTxEnabled = True
   maybeDeleteEthPhyIntfLagConfig( epilc, mode )

class ForwardingTransmitDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'forwarding port-channel transmit disabled'
   noOrDefaultSyntax = syntax
   data = {
         'forwarding' : CliToken.Switch.matcherForwarding,
         'port-channel' : CliCommand.guardedKeyword(
            'port-channel',
            'Set forwarding when part of a port channel',
            guard=lagHwInterlockSupportedGuard ),
         'transmit' : 'Port transmit forwarding',
         'disabled' : 'Disable port transmit forwarding when part of a port channel'
   }
   handler = setLagPortTransmitDisabled
   noOrDefaultHandler = setNoLagPortTransmitDisabled

ChannelGroupModelet.addCommandClass( ForwardingTransmitDisabledCmd )

# -------------------------------------------------------------------------------
# Associate the ChannelGroupModelet with the "config-if" mode.
# -------------------------------------------------------------------------------
IntfCli.IntfConfigMode.addModelet( ChannelGroupModelet )

# -------------------------------------------------------------------------------
# Register callback to return LAG members
# -------------------------------------------------------------------------------
EbraCliLib.getLagMembersCallBack = LagCliLib.channelPorts

# -------------------------------------------------------------------------------
# The "show lacp" command, in "enable" mode.
#
# The full syntax of this command is:
#
#   show lacp <channelGroupId>
#        [ counters|internal|peer|sys-id|aggregates ]
#        [brief|detailed]
#        [all-ports]
#   legacy:
#    show lacp <channelGroupId> neighbor
#   deprecated by:
#    show lacp <channelGroupId> peer
# So far this is basically a clone of the industry standard interface, but
#  we may want to expose a bit more.
# ------------------------------------------------------------------------------

def cliLacpPorts( mode, channel, channelGroupId, allPorts, activePorts ):
   # Maybe if channelGroupId not = none, then allPorts = True should mean
   # ports in Lag, but not in LACP.  And allPorts == False should mean
   # *all* ports in LACP.
   if allPorts:
      ports = LagCliLib.channelPorts( mode, channel, True )
   else:
      ports = activePorts
   return( ports ) # pylint: disable=superfluous-parens

def lacpPortStatus( port, lacpLagStatus ):
   agg = lacpLagStatus and lacpLagStatus.agg
   if not lacpLagStatus:
      status = 'idle'
   elif not agg:
      status = 'noAgg'
   elif port in agg.aggregate:
      status = 'bundled'
   elif port in agg.standby:
      status = 'standby'
   elif port in lacpLagStatus.otherIntf:
      status = 'mismatchAgg'
   elif port in agg.selected:
      status = 'negotiation'
   else:
      s = LagCliLib.lacpStatus.portStatus.get( port )
      if s and s.selected == "selectedStateUnselected":
         status = 'unselected'
      else:
         status = 'unknown'
   return status

# pylint: disable-next=inconsistent-return-statements
def portCounters( port, channelStatus, lacpCounters, lacpCountersCheckpoint, brief ):
   c = lacpCounters.portCounters.get( port )
   cc = lacpCountersCheckpoint.portCounters.get( port )
   if not c or not cc:
      return

   tempCounter = Tac.newInstance( "Lacp::LacpIntfCounters", "")
   # copy then subtract
   tempCounter.doCopy( c )
   tempCounter.doSubtract( cc )

   details = None
   if not brief:
      details = LagModel.InterfaceLacpCountersDetails(
         lastRxTime = LagModel.toUtc( tempCounter.debugLastRxTime ),
         actorChurnCount = getattr( tempCounter,
                                    "debugActorChurnCount" ),
         actorChangeCount = getattr( tempCounter, "debugActorChangeCount" ),
         actorSyncTransitionCount = getattr( tempCounter,
                                             "debugActorSyncTransitionCount" ),
         partnerChurnCount = getattr( tempCounter,
                                      "debugPartnerChurnCount" ),
         partnerSyncTransitionCount = getattr( tempCounter,
                                               "debugPartnerSyncTransitionCount" ),
         partnerChangeCount = getattr( tempCounter,
                                       "debugPartnerChangeCount" ) )

   return LagModel.InterfaceLacpCounters(
         actorPortStatus = lacpPortStatus( port, channelStatus ),
         lacpdusRxCount = getattr( tempCounter, "statsLACPDUsRx" ),
         lacpdusTxCount = getattr( tempCounter, "statsLACPDUsTx" ),
         markersRxCount = getattr( tempCounter, "statsMarkerPDUsRx" ),
         markersTxCount = getattr( tempCounter, "statsMarkerPDUsTx" ),
         markerResponseRxCount = getattr( tempCounter,
                                          "statsMarkerResponsePDUsRx" ),
         markerResponseTxCount = getattr( tempCounter,
                                          "statsMarkerResponsePDUsTx" ),
         illegalRxCount = tempCounter.statsIllegalRx,
         details = details )

# Hmm... do I show info on channel-groups that have a lacpIntfConfig, but
# have mode lacpModeOff?
# Maybe we should, but only if detailed is true... or maybe
# only if specifically mentioned the port-channel *and* detailed is true.

# pylint: disable-next=inconsistent-return-statements
def portNeighbor( port, lacpStatus, channelStatus, brief ):
   s = lacpStatus.portStatus.get( port )
   if not s:
      return

   # s.lagId.{remote,local} are almost the same as
   # s.{partner,actor}Port, except that the port priority
   # and port number are zero in s.lagId if aggregation
   # is occurring.  (43.3.6.1/f)

   details = None
   if not brief:
      details = LagModel.InterfaceLacpNeighborsDetails(
       aggSelectionState = s.selected,
       partnerChurnState = s.partnerChurnState,
       collectorMaxDelay = s.partnerCollectorMaxDelay )

   r = s.partnerPort
   return LagModel.InterfaceLacpNeighbors(
                  actorPortStatus = lacpPortStatus( port, channelStatus ),
                  partnerSystemId = dot43sysid( r.sysPriority, r.sysId ),
                  partnerPortId = r.portId,
                  partnerPortState = LagCliLib.lacpPortStateModel( s.partnerState ),
                  partnerOperKey = "0x%04x" % r.key,
                  partnerPortPriority = r.portPriority,
                  details = details )

# pylint: disable-next=inconsistent-return-statements
def portInternal( port, lacpStatus, channelStatus, lacpCounters, brief ):
   s = lacpStatus.portStatus.get( port )
   if not s:
      return
   lacpIntfConf = lacpConfig.lacpIntfConfig.get( port )
   if not lacpIntfConf:
      return
   l = s.actorPort
   r = s.partnerPort

   details = None
   if not brief:
      c = lacpCounters.portCounters.get( port )
      details = LagModel.InterfaceLacpInternalDetails(
         aggSelectionState = s.selected,
         actorAdminKey = "0x%04x" % l.key,
         actorChurnState = s.actorChurnState,
         lastRxTime = LagModel.toUtc( c.debugLastRxTime ) if c else 0.0,
         actorRxSmState = s.rxSmState,
         actorMuxSmState = s.muxSmState,
         actorMuxReason = c.debugMuxReason if c else '',
         allowToWarmup = s.allowToWarmup )
      details.actorTimeoutMultiplier = lacpIntfConf.timeoutMult

   return LagModel.InterfaceLacpInternal(
         actorPortStatus = lacpPortStatus( port, channelStatus ),
         partnerSystemId = dot43sysid( r.sysPriority, r.sysId ),
         actorPortId = l.portId,
         actorPortState = LagCliLib.lacpPortStateModel( s.actorState ),
         actorOperKey = "0x%04x" % l.key,
         actorPortPriority = l.portPriority,
         details = details )

def aggNegotiationPorts( agg, lagStatus ): # pylint: disable=redefined-outer-name
   ports = [ i for i in agg.selected
             if not ( i in agg.aggregate or
                      i in agg.standby or
                      ( lagStatus and i in lagStatus.otherIntf ) ) ]
   return Arnet.sortIntf( ports )

def aggOtherPorts( agg ):
   ports = set( agg.standby ).union( agg.selected )
   return Arnet.sortIntf( port for port in ports if port not in agg.aggregate )

# pylint: disable-next=redefined-outer-name
def incompatiblePortsDetail( lacpStatus, lagStatus ):
   aggDict = {}
   incompatPortsDetail = {}
   for portName in lagStatus.otherIntf:
      port = lacpStatus.portStatus.get( portName )
      if not port:
         # Can this happen if the port is in the process of being
         # removed from the lag?
         continue
      aggDict.setdefault( port.lagId, [ ] ).append( portName )
   for aggId, ports in aggDict.items():
      incompatPortsDetail[ dot43lagId( aggId ) ] = \
                         LagModel.IncompatiblePorts( ports=Arnet.sortIntf( ports ) )
   return incompatPortsDetail

# pylint: disable-next=redefined-outer-name
def describeAgg( agg, lagStatus, allPorts, brief ):
   aggregateId = dot43lagId( agg.lagId )
   bundledPorts = Arnet.sortIntf( agg.aggregate )

   ( otherPorts, standbyPorts, negotiationPorts ) = ( [], [], [] )
   if allPorts:
      if not brief:
         otherPorts = aggOtherPorts( agg )
         standbyPorts = Arnet.sortIntf( agg.standby )
      negotiationPorts = aggNegotiationPorts( agg, lagStatus )
   return ( aggregateId, bundledPorts, otherPorts, standbyPorts, negotiationPorts )

AclCli.intfTypes.append( LagIntf.LagAutoIntfType )
intfRangeMatcher = IntfRange.IntfRangeMatcher( explicitIntfTypes=AclCli.intfTypes )
AclCli.ShowAccessListsCmd.data['INTERFACE'] = CliCommand.Node( intfRangeMatcher )
AclCli.ShowIpAccessListsCmd.data['INTERFACE'] = CliCommand.Node( intfRangeMatcher )
AclCli.ShowIp6AccessListsCmd.data['INTERFACE'] = CliCommand.Node( intfRangeMatcher )
AclCli.ShowMacAccessListsCmd.data['INTERFACE'] = CliCommand.Node( intfRangeMatcher )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global lagStatus
   global lacpConfig
   global lagGroupDir
   global bridgingConfig, bridgingCliConfig
   global ethIntfStatusDir, agentStatus
   global ethPhyIntfConfigSliceDir
   global hwConfig
   global hwStatus
   global lagInternalStatus

   lagStatus = LazyMount.mount( entityManager, "lag/status", "Lag::Status", "r" )
   ethPhyIntfConfigSliceDir = ConfigMount.mount(
                         entityManager,
                         "interface/config/eth/phy/slice",
                         "Tac::Dir", "wi" )
   lacpConfig = LazyMount.mount( entityManager, "lag/lacp/config",
                                 "Lacp::Config", "r" )
   lagGroupDir = LazyMount.mount( entityManager, "hardware/lag/internalConfig",
                                  "Hardware::Lag::InternalConfig", "r" )
   bridgingConfig = LazyMount.mount( entityManager, "bridging/config",
                                     "Bridging::Config", "r" )
   bridgingCliConfig = ConfigMount.mount( entityManager,
                                          "bridging/input/config/cli",
                                          "Bridging::Input::CliConfig", "w" )
   ethIntfStatusDir = LazyMount.mount( entityManager, "interface/status/eth/intf",
                                          "Interface::EthIntfStatusDir", "r")
   agentStatus = LazyMount.mount( entityManager, "agent/lag/status",
                                  "Lag::Agent::InitStatus", "r" )
   hwConfig = LazyMount.mount( entityManager, "hardware/lag/config",
                                       "Hardware::Lag::Config", "r" )
   hwStatus = LazyMount.mount( entityManager, "hardware/lag/status",
                               "Hardware::Lag::Status", "r" )
   lagInternalStatus = LazyMount.mount( entityManager,
                                   "lag/internalStatus",
                                   "Lag::InternalStatus", "r" )
   gv.hwFcConfig = LazyMount.mount( entityManager, "hardware/fabricchannel/config",
                                    "Hardware::Lag::Config", "r" )
   gv.hwFcStatus = LazyMount.mount( entityManager, "hardware/fabricchannel/status",
                                    "Hardware::Lag::Status", "r" )

   IntfCli.Intf.registerDependentClass( EthPhyLagIntfDependent )

