#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliMatcher
import CliPlugin.LagCli as LagCli # pylint: disable=consider-using-from-import
import CliPlugin.LagCliLib as LagCliLib # pylint: disable=consider-using-from-import
# pylint: disable-next=consider-using-from-import
import CliPlugin.LagCommonCliLib as LagCommonCliLib
import BasicCliModes
from CliToken.LoadBalance import loadBalanceMatcherForConfig
from CliToken.PortChannel import portChannelMatcherForConfig
import Tac
from TypeFuture import TacLazyType 
from LacpConstants import (
      SystemPriorityDefault, 
      LacpPortIdDefault,
)

TacMinBandwidthTimeoutBase = TacLazyType( "Lag::MinBandwidthTimeoutBase" )

matcherLacp = CliMatcher.KeywordMatcher( 'lacp', 
      helpdesc='Set Link Aggregation Control Protocol (LACP) parameters' )
matcherRateLimit = CliMatcher.KeywordMatcher( 
      'rate-limit', helpdesc='LACPDU Rate Limit Options' )
nodeLacp = CliCommand.Node( 
      matcher=matcherLacp,
      guard=LagCommonCliLib.lagSupportedGuard )
matcherPortIdRange = CliMatcher.IntegerMatcher( 1, 65535, 
      helpdesc='LACP port-ID range' )

def doEnableLacpAudit( mode, args ):
   LagCliLib.lacpCliConfig.enableAudit = True

def noLacpAudit( mode, args ):
   LagCliLib.lacpCliConfig.enableAudit = False

#--------------------------------------------------------------------------------
# [ no | default ] lacp audit
#--------------------------------------------------------------------------------
class LacpAuditCmd( CliCommand.CliCommandClass ):
   syntax = 'lacp audit'
   noOrDefaultSyntax = 'lacp audit ...'
   data = {
      'lacp' : nodeLacp,
      'audit': 'Audit the internal invariants of the LACP state machines',
   }
   handler = doEnableLacpAudit
   noOrDefaultHandler = noLacpAudit

   hidden = True

BasicCliModes.GlobalConfigMode.addCommandClass( LacpAuditCmd )

def setPortIdRange( mode, args ):
   portIdMin = args.get( 'MIN', LacpPortIdDefault )
   portIdMax = args.get( 'MAX', LacpPortIdDefault )
   if portIdMin > portIdMax:
      mode.addError( "Invalid LACP port-id range values" )
      return
   LagCliLib.lacpCliConfig.portIdRange = \
         Tac.Value( 'Lacp::PortIdRange', portIdMin, portIdMax )
   LagCliLib.updateLacpOverridePortIdRange()

#--------------------------------------------------------------------------------
# [ no | default ] lacp port-id range MIN MAX
#--------------------------------------------------------------------------------
class LacpPortIdRangeCmd( CliCommand.CliCommandClass ):
   syntax = 'lacp port-id range MIN MAX'
   noOrDefaultSyntax = 'lacp port-id range ...'
   data = {
      'lacp': nodeLacp,
      'port-id': 'LACP port-id range configuration',
      'range': 'LACP port-id range configuration',
      'MIN' : matcherPortIdRange,
      'MAX' : matcherPortIdRange
   }
   handler = setPortIdRange
   noOrDefaultHandler = setPortIdRange

BasicCliModes.GlobalConfigMode.addCommandClass( LacpPortIdRangeCmd )

def setSystemPriority( mode, args ):
   systemPriority = args.get( 'SYSTEMPRIORITY', SystemPriorityDefault )
   for hook in LagCli.canSetSystemPriorityHook.extensions():
      if not hook( mode, systemPriority ):
         return
   LagCliLib.lacpCliConfig.priority = systemPriority

#--------------------------------------------------------------------------------
# [ no | default ] lacp system-priority SYSTEMPRIORITY
#--------------------------------------------------------------------------------
class LacpSystemPriorityCmd( CliCommand.CliCommandClass ):
   syntax = 'lacp system-priority SYSTEMPRIORITY'
   noOrDefaultSyntax = 'lacp system-priority ...'
   data = {
      'lacp': nodeLacp,
      'system-priority': 'Set local system LACP priority',
      'SYSTEMPRIORITY': CliMatcher.IntegerMatcher( 0, 65535, 
         helpdesc='LACP System Priority' ),
   }
   handler = setSystemPriority
   noOrDefaultHandler = setSystemPriority

BasicCliModes.GlobalConfigMode.addCommandClass( LacpSystemPriorityCmd )

#-----------------------------------------------------------
# "load-balance policies" in global config mode
#-----------------------------------------------------------

def gotoConfigLoadBalancePoliciesMode( mode, args ):
   childMode = mode.childMode( LagCli.ConfigLoadBalancePoliciesMode )
   mode.session_.gotoChildMode( childMode )

def lagLBPoliciesSupportedGuard( mode, token ):
   if LagCliLib.bridgingHwCapabilities.lagLoadBalancePoliciesSupported:
      return None
   return LagCli.CliParser.guardNotThisPlatform

#--------------------------------------------------------------------------------
# load-balance policies
#--------------------------------------------------------------------------------
class LoadBalancePoliciesCmd( CliCommand.CliCommandClass ):
   syntax = 'load-balance policies'
   noOrDefaultSyntax = syntax
   data = {
      'load-balance': loadBalanceMatcherForConfig,
      'policies': CliCommand.guardedKeyword(
            'policies',
            helpdesc='Configure load-balance policies',
            guard=lagLBPoliciesSupportedGuard ),
   }
   handler = gotoConfigLoadBalancePoliciesMode
   noOrDefaultHandler = LagCli.ConfigLoadBalancePoliciesMode.clear

BasicCliModes.GlobalConfigMode.addCommandClass( LoadBalancePoliciesCmd )

#----------------------------------------------------------------------
# The "[no] logging event port-channel member-status global N" command, 
# in "config" mode.
#----------------------------------------------------------------------

def doEnableLagMemberStatusLog( mode, args ):
   LagCliLib.lagConfigCli.memberStatusLogEnabled = True

def noLagMemberStatusLog( mode, args ):
   LagCliLib.lagConfigCli.memberStatusLogEnabled = False

#--------------------------------------------------------------------------------
# [ no | default ] logging event port-channel member-status global ...
#--------------------------------------------------------------------------------
class LoggingEventPortChannelCmd( CliCommand.CliCommandClass ):
   syntax = 'logging event port-channel member-status global ...'
   noOrDefaultSyntax = syntax
   data = {
      'logging': 'Configure logging',
      'event': 'Configure logging events',
      'port-channel': CliCommand.Node( 
         matcher=CliMatcher.KeywordMatcher( 
            'port-channel', helpdesc='Configure port-channel messages' ), 
         guard=LagCommonCliLib.lagSupportedGuard ),
      'member-status': ( 'Enable system logging for interface joining/leaving'
                         ' a port-channel' ),
      'global': 'Configure global port channel member status',
   }
   handler = doEnableLagMemberStatusLog
   noHandler = noLagMemberStatusLog
   defaultHandler = handler

BasicCliModes.GlobalConfigMode.addCommandClass( LoggingEventPortChannelCmd )

#------------------------------------------------------------------------------
# The "[no|default] port-channel min-links review interval <timeout>" command, 
# in "config" mode.
#------------------------------------------------------------------------------

def setMinLinksReviewInterval( mode, args ):
   interval = args.get( 'TIMEOUT', LagCli.MinBandwidthTimeoutBase.timeoutDefault )
   LagCliLib.lagConfigCli.minLinksTimeoutBase = interval

#--------------------------------------------------------------------------------
# [ no | default ] port-channel min-links review interval <timeout>
#--------------------------------------------------------------------------------
class MinLinksReviewIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'port-channel min-links review interval TIMEOUT'
   noOrDefaultSyntax = 'port-channel min-links review interval ...'
   data = {
      'port-channel': portChannelMatcherForConfig,
      'min-links': 'Configure min-links',
      'review': 'Review',
      'interval': 'Port-channel min-links review time in seconds (default = 180)',
      'TIMEOUT': CliMatcher.IntegerMatcher( TacMinBandwidthTimeoutBase.minTimeout,
                                            TacMinBandwidthTimeoutBase.maxTimeout,
                                            helpdesc='Seconds' ),
   }
   handler = setMinLinksReviewInterval
   noOrDefaultHandler = setMinLinksReviewInterval

BasicCliModes.GlobalConfigMode.addCommandClass( MinLinksReviewIntervalCmd )

#------------------------------------------------------------------------------
# The "[no|default] port-channel speed minimum review interval <timeout> seconds" 
# command, in "config" mode.
#------------------------------------------------------------------------------

def setMinSpeedReviewInterval( mode, args ):
   interval = args.get( 'TIMEOUT', LagCli.MinBandwidthTimeoutBase.timeoutDefault )
   LagCliLib.lagConfigCli.minSpeedTimeoutBase = interval

#--------------------------------------------------------------------------------
# [ no | default ] port-channel speed minimum review interval <timeout> seconds
#--------------------------------------------------------------------------------
class MinSpeedReviewIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'port-channel speed minimum review interval TIMEOUT seconds'
   noOrDefaultSyntax = 'port-channel speed minimum review interval ...'
   data = {
      'port-channel': portChannelMatcherForConfig,
      'speed': 'Configure port-channel speed',
      'minimum': 'Minimum',
      'review': 'Review',
      'interval': 'Port-channel minimum speed review time in seconds'
                  ' (default = 180)',
      'TIMEOUT': CliMatcher.IntegerMatcher( TacMinBandwidthTimeoutBase.minTimeout,
                                            TacMinBandwidthTimeoutBase.maxTimeout,
                                            helpdesc='Seconds' ),
      'seconds': 'Seconds'
   }
   handler = setMinSpeedReviewInterval
   noOrDefaultHandler = setMinSpeedReviewInterval

BasicCliModes.GlobalConfigMode.addCommandClass( MinSpeedReviewIntervalCmd )

#-------------------------------------------------------------------------
# The "[no|default] lacp rate-limit default" command, in "config" mode.
#-------------------------------------------------------------------------

def enableGlobalRateLimit( mode, args ):
   LagCliLib.lacpCliConfig.rateLimitEnable = True

def noGlobalRateLimit( mode, args ):
   LagCliLib.lacpCliConfig.rateLimitEnable = False

#--------------------------------------------------------------------------------
# lacp rate-limit default
#--------------------------------------------------------------------------------
class LacpRateLimitCmd( CliCommand.CliCommandClass ):
   syntax = 'lacp rate-limit default'
   noOrDefaultSyntax = syntax
   data = {
      'lacp' : nodeLacp,
      'rate-limit' : 'Set LACPDU Rate Limit Options',
      'default' : 'Enable LACPDU rate limiting by default on all ports',
   }
   handler = enableGlobalRateLimit
   defaultHandler = handler
   noHandler = noGlobalRateLimit

BasicCliModes.GlobalConfigMode.addCommandClass( LacpRateLimitCmd )
