#!/usr/bin/env python3
# Copyright (c) 2009-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from __future__ import absolute_import, division, print_function
import CliCommand
import CliMatcher
import Tac, Tracing
import ConfigMount
import LazyMount
from CliPlugin import IntfCli
from CliPlugin import MlagShowCli
from CliPlugin.LagIntfCli import LagIntfConfigModelet
from CliPlugin.RecircCli import isRecirc
import Toggles.MlagToggleLib

t8 = Tracing.trace8

mlagConfig = None
cliConfig = None
mlagCliSupport = None

PortChannelIntfId = Tac.Type( "Arnet::PortChannelIntfId" )

class MlagIntf( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      t8( "Destroying mlag mapping for", self.intf_.name )
      del mlagConfig.intfConfig[ self.intf_.name ]

#-------------------------------------------------------------------------------
# The "[no|default] mlag" command for config-if-Po* mode
#-------------------------------------------------------------------------------
matcherMlag = CliMatcher.KeywordMatcher( 'mlag', helpdesc="MLAG Membership", )
nodeMlag = CliCommand.Node( matcher=matcherMlag,
      guard=MlagShowCli.mlagSupportedGuard )
MlagNum = Tac.Type( "Mlag::MlagNum" )
matcherInterfaceId = CliMatcher.IntegerMatcher( MlagNum.min, 
                                                MlagNum.max, 
                                                helpdesc='MLAG Interface ID' )

def setMlag( mode, args ):
   mlag = args[ 'MLAG_IDS' ]
   ic = mlagConfig.intfConfig
   name = mode.intf.name

   if isRecirc( name ):
      mode.addError( "Cannot configure MLAG on a Recirc-Channel" )
      return

   # Allow mlag configuration only on port-channels between 1-2000
   channelId = PortChannelIntfId.portChannel( name )
   if channelId > MlagNum.max:
      mode.addError( "Cannot configure MLAG on a Port-Channel "\
                     "with ID higher than %d" % MlagNum.max )
      return
   
   # First look for existing intf with the same mlag #.
   mlagInterface = mlagCliSupport.getMlagInterface( mlagConfig, str(mlag) )

   if mlagInterface and name != mlagInterface:
      mode.addError( "%s is already configured as a member of MLAG %s"
                     % ( mlagInterface, str( mlag ) ) )
      return

   if not Toggles.MlagToggleLib.toggleMlagL2SubinterfacesEnabled():
      sic = cliConfig.switchIntfConfig.get( name )
      if sic and not sic.enabled:
         mode.addError( "Cannot configure MLAG on a routed port" )
         return

   # check if Port-Channel has already been set as "peer-link".
   if mlagConfig.peerLinkIntfId == mode.intf.name:
      mode.addError( "%s already configured as peer-link" % name )
      return

   ic[ name ] = str( mlag )

def noMlag( mode, args ):
   ic = mlagConfig.intfConfig
   name = mode.intf.name
   if name in ic:
      del ic[ name ]

class SetMlagCmd( CliCommand.CliCommandClass ):
   syntax = 'mlag MLAG_IDS'
   noOrDefaultSyntax = 'mlag ...'
   data = {
      'mlag' : nodeMlag,
      'MLAG_IDS' : matcherInterfaceId,
   }
   handler = setMlag
   noOrDefaultHandler = noMlag

LagIntfConfigModelet.addCommandClass( SetMlagCmd )

#-------------------------------------------------------------------------------
# Mounts
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global mlagConfig, cliConfig, mlagCliSupport
   if Toggles.MlagToggleLib.toggleMlagL2SubinterfacesEnabled():
      mlagConfig = ConfigMount.mount( entityManager, 'mlag/input/config/cli',
                                    'Mlag::Config', 'w' )
   else:
      mlagConfig = ConfigMount.mount( entityManager, 'mlag/config',
                                    'Mlag::Config', 'w' )

   cliConfig = LazyMount.mount( entityManager, "bridging/input/config/cli",
                                "Bridging::Input::CliConfig", "r" )

   IntfCli.Intf.registerDependentClass( MlagIntf )
   mlagCliSupport = Tac.newInstance( "Mlag::CliSupport" )
