#!/usr/bin/env python3
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from CliModel import Model, Str, Bool, Int, Enum, Float, List, GeneratorList, Dict
from CliModel import Submodel
import Arnet
from IntfModels import Interface
from CliPlugin import IntfCli
from Intf.IntfRange import intfListToCanonical
import Tac, re
from math import modf
import time
import TableOutput
import LanzWarningsLib

U32_OVERFLOW = 0xFFFFFFFE
U32_MAX_VALUE = 0xFFFFFFFF
U64_MAX_VALUE = 0xFFFFFFFFFFFFFFFF

Microseconds = Tac.Type( "Lanz::Microseconds" )

# Global dict which will be used to cache the canonical ingressPortSet strings
# to be displayed by the `show queue-monitor length` command on Sand systems.
portSetStrepCache = {}

def intfShortToLong( intf ):
   intfStr = intf.replace( 'Et', 'Ethernet' )
   intfStr = intfStr.replace( 'Sw', 'Switch' )
   return intfStr

def getDurationStr( duration, entryType ):
   if duration == -1:
      if entryType == 'E' or entryType == 'GL': # pylint: disable=consider-using-in
         return 'Unknown'
      return 'N/A'
   durationStr = '%u' % duration
   if duration == U32_OVERFLOW:
      durationStr += '+'
   return durationStr

def getSandDurationStr( duration ):
   if duration == -1:
      return 'N/A'
   durationStr = '%u' % duration
   return durationStr

def makoBufferTypeToStr( bufferType, csv=False ):
   # This has to match the enum types in MakoLanz.tac
   # We duplicate the information here to avoid having to use the socket
   # callback to retrieve such information from the platform SM.

   # This is used by statistics record due to its common limited
   # dump method. Lookup is by an integer identifier.
   # This is also used by congestion record to print out full buffer name

   if bufferType in ( 0, 'E' ):
      return 'Egress' if csv else 'egress'
   elif bufferType in ( 1, 'I' ):
      return 'Ingress' if csv else 'ingress'
   elif bufferType in ( 2, 'CM' ):
      return 'CoreMemory' if csv else 'core memory'
   else:
      return None

def eventTypeToStr( eventType ):
   lookup = {
      'S' : 'Start',
      'E' : 'End'
   }
   return lookup[ eventType ]

def getQLenStr( qLen, entryType ):
   if not qLen:
      return 'Unknown'
   elif qLen == -1:
      return 'N/A'
   else:
      qLenStr = '%u' % qLen
      if entryType == 'E' or entryType == 'GL': # pylint: disable=consider-using-in
         qLenStr += '*'
      return qLenStr

def getTimeStr( utcTime, timeOffset=None ):
   if not timeOffset:
      timeStr = time.strftime( "%Y-%m-%d %H:%M:%S", time.gmtime( utcTime ) )
      ( frac, _ ) = modf( utcTime )
      fracString = "%0.5lf" % frac
      timeStr += fracString[ 1: ]
      return timeStr

   timeStr = ''
   timeDelta = timeOffset - utcTime
   seconds = timeDelta % 60
   minutes = ( timeDelta/60 ) % 60
   hours = ( timeDelta/3600 ) % 24
   days =  timeDelta/( 60*60*24 )
   if int( days ) > 0:
      if( days == 1 ): # pylint: disable=superfluous-parens
         dayStr =  "%uday," % days
      else:
         dayStr = "%udays," % days
      timeStr += dayStr
   timeStr +=  "%u:%02u:%02u" % ( hours, minutes, seconds )
   ( frac, _ ) = modf( timeDelta )
   fracString = "%0.5lf" % frac
   timeStr += fracString[ 1: ] + " ago"
   return timeStr

# extract number for ordering from sliceName
# sliceName may not contain a number
def sliceNo( sliceItem ):
   sliceName, _ = sliceItem
   if sliceName.startswith( 'Linecard' ):
      # Extract slice number, which comes after the "Linecard" string prefix
      return int( sliceName[ 8: ] )
   # cater for "FixedSystem" sliceName
   return 0

class CongestionRecordModel( Model ):
   # Not making any attributes common as the help strings
   # differ on different platforms.
   def parseShowQMonLengthRecord( self, entry, cmdType ):
      # Must be overridden
      assert False

   def renderFn( self, cmdType, timeOffset ):
      # Must be overridden
      assert False

class XpCongestionRecord( CongestionRecordModel ):
   entryType = Enum( ('S', 'U', 'E' ),
                     help='Congestion entry type, S-Start, U-Update, E-End' )
   interface = Interface( help='Interface on which the congestion event occurred' )
   trafficClass = Int( help='Traffic-class associated with the congestion' )
   entryTime = Float( help='UTC time in seconds at which the congestion record '
                           'occurred' )
   congestionDuration = Int( help="congestion duration in microseconds" )
   queueLength = Int( help='Queue length in pages of 256 bytes' )
   maxQueueLengthTime = Int( help='Time at which Max Queue length occurred relative '
                             'to congestion start (usecs). '
                             'Only applicable with entries of type E. '
                             'Value of -1 represents an undeterministic value.' )

   def parseShowQMonLengthRecord( self, entry, cmdType ):
      matchRe = ( '(?P<entryType>S|U|E),(?P<timestamp>.*),(?P<intf>.*),'
                  r'(?P<qLen>.*)\*?,(?P<duration>.*),(?P<tc>.*),(?P<timeOfMax>.*)' )
      m = re.match( matchRe, entry )

      self.entryType = m.group( 'entryType' )
      self.interface = re.sub( 'Et', 'Ethernet', m.group( 'intf' ) )
      self.trafficClass = int( m.group( 'tc' ) )

      self.queueLength  = int( m.group( 'qLen' ) )
      self.entryTime = Microseconds.toSeconds( int( m.group( 'timestamp' ) ) )

      self.congestionDuration = int( m.group( 'duration' ) ) 
      if self.congestionDuration == U64_MAX_VALUE:
         self.congestionDuration = -1
      self.maxQueueLengthTime = int( m.group( 'timeOfMax' ) )
      if self.maxQueueLengthTime == U64_MAX_VALUE:
         self.maxQueueLengthTime = -1

   def renderFn( self, cmdType, timeOffset ):
      qLenStr = getQLenStr( self.queueLength, self.entryType )
      durStr = getDurationStr( self.congestionDuration, self.entryType )
      maxQLenTimeStr = getDurationStr( self.maxQueueLengthTime, self.entryType )
      intfTcStr = "{}({})".format(
            IntfCli.Intf.getShortname( self.interface ), self.trafficClass )
      if 'csv' in cmdType:
         formatString = "%s,%s,%s,%s,%s,%s"
         timeStr = getTimeStr( self.entryTime )
         print( formatString % (
                self.entryType, timeStr,
                     intfTcStr, durStr, qLenStr, maxQLenTimeStr ) )
      else:
         formatString = "%-7s%-23s%-31s%-11s%-14s%-11s%-12s"
         timeStr = getTimeStr( self.entryTime, timeOffset )
         absoluteTimeStr = getTimeStr( self.entryTime )
         print( formatString % (
                self.entryType, timeStr, absoluteTimeStr, intfTcStr, durStr,
                qLenStr, maxQLenTimeStr ) )

class SandCongestionRecord( CongestionRecordModel ):
   entryType = Enum( values=( 'S', 'U', 'E', 'P' ),
                  help='Congestion entry type, S-Start, U-Update, E-End, P-Polling' )
   interface = Str( help='Interface on which the congestion event occurred. '
                    'We monitor congestion for the front panel ports as well '
                    'as the cpu port.' )
   trafficClass = Int(
         help="Traffic-class associated with the congestion, if applicable. "
              "Number of traffic classes supported are 8( 0 - 7 )." )
   ingressPortSet = List( valueType=Interface,
                          help="Ingress port-set corresponding to the interface"
                          "with the congestion" )
   queueLength = Int( help='Queue length in bytes' )
   entryTime = Float( help='UTC time in seconds at which the congestion record '
                           'occurred' )
   entryTimeUsecs = Int( help='UTC time in microseconds at which the congestion '
                              'record occurred', optional=True )
   duration = Int( help="congestion duration in seconds" )
   durationUsecs = Int( help="congestion duration in microseconds", optional=True )

   globalProtectionModeEnabled = Bool( help="Global protection mode is enabled",
                                       default=True )

   def parseShowQMonLengthRecord( self, entry, cmdType ):
      matchRe = ( '(?P<entryType>.*),(?P<timestamp>.*),(?P<intf>.*),'
                  r'(?P<qLen>.*)\*?,(?P<duration>.*),(?P<tc>.*),'
                  '(?P<ingressPortSet>.*)' )
      m = re.match( matchRe, entry )

      self.entryType = m.group( 'entryType' )
      self.interface = intfShortToLong( m.group( 'intf' ) )
      if ( self.globalProtectionModeEnabled or
           not self.interface.startswith( "CpuTm" ) ):
         self.trafficClass = int( m.group( 'tc' ) )

      # The `intfListToCanonical` call is heavy. The number of ingressPortSets
      # on a single switch is limited thus we don't need to compute the canonical
      # ingressPortSet string each time we want to display it if it's in the map
      # already.
      intfsList = m.group( 'ingressPortSet' ).split( ':' )
      self.ingressPortSet.extend( intfsList )
      ingressPortSetKey = repr( self.ingressPortSet )
      if ingressPortSetKey not in portSetStrepCache:
         ingressPortSetStr = intfListToCanonical( intfsList )[ 0 ]
         portSetStrepCache[ ingressPortSetKey ] = ingressPortSetStr

      self.entryTimeUsecs = int( m.group( 'timestamp' ) )
      self.entryTime = Microseconds.toSeconds( self.entryTimeUsecs )

      self.queueLength = int( m.group( 'qLen' ) )
      if self.queueLength == U32_MAX_VALUE:
         self.queueLength = -1

      # The Alta platform initially expressed the duration in terms of 
      # seconds and microseconds, but newer versions of EOS now express 
      # the duration value exclusively in terms of microseconds.  The 
      # duration value is thus consistent across platforms.
      self.durationUsecs = int( m.group( 'duration' ) )
      self.duration = int( Microseconds.toSeconds( self.durationUsecs ) )
      if self.durationUsecs == U64_MAX_VALUE:
         self.durationUsecs = -1
         self.duration = -1

   def renderFn( self, cmdType, timeOffset ):
      assert 'csv' in cmdType
      formatString = "%s,%s,%s,%s,%s,%s,%s"
      timeStr = getTimeStr( self.entryTime )
      intfTcStr = "%s" % ( IntfCli.Intf.getShortname( self.interface ) )
      if self.trafficClass is not None:
         intfTcStr += "(%s)" % self.trafficClass
      # This ingressPortSet is supposed to be in the map already as
      # `parseShowQMonLengthRecord` should have been called previously.
      ingressPortSetKey = repr( self.ingressPortSet )
      assert ingressPortSetKey in portSetStrepCache
      ingressPortSetStr = portSetStrepCache[ ingressPortSetKey ]
      if ',' in ingressPortSetStr:
         ingressPortSetStr = '"%s"' % ingressPortSetStr
      print( formatString % (
             self.entryType,
             timeStr, intfTcStr,
             self.queueLength, self.durationUsecs, self.trafficClass,
             ingressPortSetStr ) )

class SandLatencyRecord( CongestionRecordModel ):
   entryTime = Float( help='Time at which the TX latency is recorded.' )
   interface = Interface( help='Destination interface.' )
   trafficClass = Int( help='Traffic class associated with the '
                       'tx delay. Number of traffic classes supported '
                       'are 8( 0 - 7 ).' )
   txLatency = Float( help='Tx-delay information in usecs.' )
   ingressPortSet = List( valueType=Interface,
         help='Ingress port-set corresponding to the interface with the tx-latency' )
   globalProtectionModeEnabled = Bool( help="Global protection mode is enabled",
                                       default=True )

   def parseShowQMonLengthRecord( self, entry, cmdType ):
      matchRe = ( '(?P<timestamp>.*),(?P<intf>.*)\\((?P<tc>\\d+)\\),'
                  '(?P<latency>.*),(?P<portset>.*)' )
      m = re.match( matchRe, entry )

      self.entryTime = Microseconds.toSeconds( int( m.group( 'timestamp' ) ) )
      self.interface = intfShortToLong( m.group( 'intf' ) )
      self.txLatency = float( m.group( 'latency' ) )

      intfsList = m.group( 'portset' ).split( ':' )
      self.ingressPortSet.extend( intfsList )
      ingressPortSetKey = repr( self.ingressPortSet )
      if ingressPortSetKey not in portSetStrepCache:
         ingressPortSetStr = intfListToCanonical( intfsList )[ 0 ]
         portSetStrepCache[ ingressPortSetKey ] = ingressPortSetStr

      if self.globalProtectionModeEnabled:
         self.trafficClass = int( m.group( 'tc' ) )

   def renderFn( self, cmdType, timeOffset ):
      timeStr = getTimeStr( self.entryTime, timeOffset )
      intfShortName = Arnet.IntfId( self.interface ).shortName
      intfTcStr = '%s(%d)' % ( intfShortName, self.trafficClass )
      intfTcStr = intfTcStr.replace( '\'', '' )

      # This ingressPortSet is supposed to be in the map already as
      # `parseShowQMonLengthRecord` should have been called previously.
      ingressPortSetKey = repr( self.ingressPortSet )
      assert ingressPortSetKey in portSetStrepCache
      ingressPortSetStr = portSetStrepCache[ ingressPortSetKey ]
      if ',' in ingressPortSetStr:
         ingressPortSetStr = '%s' % ingressPortSetStr
      print( "%-25s%-15s%-13.3f%-16s" % ( timeStr,
                                          intfTcStr,
                                          self.txLatency,
                                          ingressPortSetStr ) )

class BarefootCongestionRecord( CongestionRecordModel ):
   interface = Interface( help='Interface on which the congestion event occurred' )
   txQueue = Int( help="TX-queue associated with the congestion" )
   queueLength = Int( help='Queue length in segments' )
   entryTime = Float( help='UTC time in seconds at which the congestion record '
                           'occurred' )
   duration = Int( help="Congestion duration in microseconds" )

   def parseShowQMonLengthRecord( self, entry, cmdType ):
      matchRe = ( '(?P<timestamp>.*),(?P<intf>.*),'
                  r'(?P<qLen>.*)\*?,(?P<duration>.*),(?P<txQ>.*)' )
      m = re.match( matchRe, entry )

      self.interface = intfShortToLong( m.group( 'intf' ) )
      self.txQueue = int( m.group( 'txQ' ) )

      self.queueLength = int( m.group( 'qLen' ) )

      timeUsecs = int( m.group( 'timestamp' ) )
      self.entryTime = Microseconds.toSeconds( timeUsecs )

      self.duration = int( m.group( 'duration' ) )

   def renderFn( self, cmdType, timeOffset ):
      assert 'csv' in cmdType
      formatString = "%s,%s,%s,%s,%s"
      timeStr = getTimeStr( self.entryTime )
      intfStr = "%s" % ( IntfCli.Intf.getShortname( self.interface ) )
      print( formatString % ( timeStr, intfStr, self.txQueue, self.duration,
                              self.queueLength ) )

class MakoCongestionRecord( CongestionRecordModel ):
   entryType = Enum( ( 'S', 'E' ),
                     help='Congestion entry type, S-Start, E-End' )
   bufferType = Enum( ( 'I', 'E', 'CM' ),
                      help='Congestion buffer type, I-Ingress, '
                      'E-Egress, CM-Core memory' )
   interface = Interface( help='Interface on which the congestion event occurred' )
   entryTime = Float( help='UTC time in seconds at which the congestion record '
                      'occurred' )
   durationUsecs = Int( help="congestion duration in microseconds" )

   def parseShowQMonLengthRecord( self, entry, cmdType ):
      matchRe = ( '(?P<entryType>S|E),(?P<intf>.*),(?P<bufferType>I|E|CM),'
                  '(?P<timestamp>.*),(?P<durationUsecs>.*)' )
      m = re.match( matchRe, entry )

      self.entryType = m.group( 'entryType' )
      self.interface = intfShortToLong( m.group( 'intf' ) )
      # bufferType here is in string, I/E/CM. CSV output
      # parses them and spells out.
      self.bufferType = m.group( 'bufferType' )

      timeUsecs = int( m.group( 'timestamp' ) )
      self.entryTime = Microseconds.toSeconds( timeUsecs )

      self.durationUsecs = int( m.group( 'durationUsecs' ) )

   def renderFn( self, cmdType, timeOffset ):
      assert 'csv' in cmdType
      formatString = "%s,%s,%s,%s,%s"
      timeStr = getTimeStr( self.entryTime )
      print( formatString % ( eventTypeToStr( self.entryType ),
                              self.interface.stringValue,
                              makoBufferTypeToStr( self.bufferType, True ),
                              timeStr,
                              self.durationUsecs ) )

class BarefootStatisticsRecord( CongestionRecordModel ):
   interface = Interface( help='Interface on which the congestion events occurred' )
   txQueue = Int( help='Traffic class associated with the congestion events' )
   numCongestionEvents = Int( help='Number of times congestion started '
                              'on the interface per TX-Queue' )

   def parseShowQMonLengthRecord( self, entry, cmdType ):
      matchRe = '(?P<intf>.*),(?P<identifier>.*),(?P<stats>.*)'
      m = re.match( matchRe, entry )

      self.interface = intfShortToLong( m.group( 'intf' ) )
      # identifier is a 64 bit field that different platforms can use to
      # uniquely identify a queue for this interface. On Barefoot,
      # we identify the queue by it's ID number
      identifier = int( m.group( 'identifier' ) )
      self.txQueue = identifier
      self.numCongestionEvents = int( m.group( 'stats' ) )

   def renderFn( self, cmdType, timeOffset ):
      intfStr = "%s" % ( IntfCli.Intf.getShortname( self.interface ) )
      print( "%-15s %-15s %-15s" % ( intfStr,
                                     self.txQueue,
                                     self.numCongestionEvents ) )

class MakoStatisticsRecord( CongestionRecordModel ):
   interface = Interface( help='Interface on which the congestion events occurred' )
   bufferType = Int( help='Buffer associated with the congestion event' )
   numCongestionEvents = Int( help='Number of times congestion started '
                              'on the interface per buffer type' )

   def parseShowQMonLengthRecord( self, entry, cmdType ):
      matchRe = '(?P<intf>.*),(?P<bufferType>.*),(?P<stats>.*)'
      m = re.match( matchRe, entry )
      assert m, f"entry is {entry}"

      self.interface = intfShortToLong( m.group( 'intf' ) )
      self.bufferType = int( m.group( 'bufferType' ) )
      self.numCongestionEvents = int( m.group( 'stats' ) )

   def renderFn( self, cmdType, timeOffset ):
      assert False, "rendering of MakoStatisticsRecord is done in table"

class XpStatisticsRecord( CongestionRecordModel ):
   interface = Str( help='Interface on which the congestion '
                          'events occurred.' )
   trafficClass = Int( help='Traffic class associated with the '
                       'congestion events. Number of traffic classes '
                       'supported are 8( 0 - 7 ).' )
   numCongestionEvents = Int( help='Number of times congestion started '
                              'on the interface per traffic-class.' )

   def parseShowQMonLengthRecord( self, entry, cmdType ):
      matchRe = '(?P<intf>.*),(?P<identifier>.*),(?P<stats>.*)'
      m = re.match( matchRe, entry )

      self.interface = m.group( 'intf' )
      # identifier is a 64 bit field that different platforms can use to
      # uniquely identify a queue for this interface.
      identifier = int( m.group( 'identifier' ) )
      self.trafficClass = identifier
      self.numCongestionEvents = int( m.group( 'stats' ) )

   def renderFn( self, cmdType, timeOffset ):
      print( "%-15s %-15s %-15s" % ( self.interface,
                                     self.trafficClass,
                                     self.numCongestionEvents ) )

class StrataCommonRecord( CongestionRecordModel ):
   interface = Interface( help='Interface on which the congestions events occurred' )
   trafficClass = Int( help='Traffic class associated with the congestion '
                       'events. Number of traffic classes supported are '
                       '22 (0-11 for unicast, 12-21 for multicast).' )
   #optional attributes
   cardSlot = Int( help="Card slot associated with the congestion event."
                         "Needed for distinguishing congestion "
                         "records from CPU ports on multi-switch chips. Only "
                         "valid for CPU congestion records.",
                   optional=True)
   chipIndex = Int( help="Chip index associated with the congestion event."
                         "Needed for distinguishing congestion "
                         "records from CPU ports on multi-switch chips. Only "
                         "valid for CPU congestion records.",
                   optional=True)

   def parseShowQMonLengthRecord( self, entry, cmdType ):
      pass

   def renderFn( self, cmdType, timeOffset ):
      pass

   def parseRecord( self, match ):
      if match.group( 'intfType' ) == "Et":
         self.interface = "Ethernet%s" % match.group( 'intfOrCardChip' )
      elif match.group( 'intfType' ) == "Fa":
         self.interface = "Fabric%s" % match.group( 'intfOrCardChip' )
      else:
         self.interface = "Cpu"
         cardChip = match.group( 'intfOrCardChip' ).split( '/' )
         self.cardSlot = int( cardChip[ 0 ] )
         self.chipIndex = int( cardChip[ 1 ] )
      self.trafficClass = int( match.group( 'tc' ) )

   def getInterfaceName( self ):
      intfName = IntfCli.Intf.getShortname( self.interface )
      if intfName == "Cpu":
         intfName += "%u/%u" % ( self.cardSlot, self.chipIndex )
      return intfName

class StrataCongestionRecord( StrataCommonRecord ):
   entryType = Enum( values=( 'S', 'U', 'E' ),
                     help='Congestion entry type, S-Start, U-Update, E-End' )
   entryTime = Float( help='Time at which the congestion entry is generated' )
   congestionDuration = Int( help='Congestion duration associated with the '
                             'congestion event (usecs). '
                             'Only applicable with entries of type E. '
                             'Value -1 represents an undeterministic value.' )
   queueLength = Int( help='Queue length in segments, '
                      'Segment size for S, U, and E type records varies per chip '
                      'model.' )
   maxQueueLengthTime = Int( help='Time at which Max Queue length occurred relative '
                             'to congestion start (usecs). '
                             'Only applicable with entries of type E. '
                             'Value of -1 represents an undeterministic value.' )
   txDrops = Int( help='Number of packet drops seen on the interface due to '
                  'congestion.',
                  default=0 )
   txLatency = Float( help='Tx-delay information per interface (usecs).',
                      default=0.000 )
   fabricPeerIntf = Str( help='Peer interface for the congested fabric '
                               'interface.',
                         optional=True )

   def parseShowQMonLengthRecord( self, entry, cmdType ):
      matchRe = ' (?P<entryType>E|S|U),(?P<timestamp>.*),\
(?P<intfType>Et|Cpu|Fa)(?P<intfOrCardChip>.*)\\((?P<tc>\\d+)\\),(?P<duration>.*),\
(?P<qLen>.*),(?P<timeOfMax>.*)'
      # More info in csv format, so add it
      if 'csv' in cmdType:
         matchRe += ',(?P<latency>.*),(?P<drops>.*)'
      matchRe += ',((?P<peerFabricIntf>Fa.*)|$)'
      m = re.match( matchRe, entry )
      self.parseRecord( m )
      self.entryType = m.group( 'entryType' )
      self.entryTime = Microseconds.toSeconds( int( m.group( 'timestamp' ) ) )
      self.queueLength = int( m.group( 'qLen' ) )
      self.congestionDuration = int( m.group( 'duration' ) )
      if self.congestionDuration == U32_MAX_VALUE:
         self.congestionDuration = -1
      self.maxQueueLengthTime = int( m.group( 'timeOfMax' ) )
      if self.maxQueueLengthTime == U32_MAX_VALUE:
         self.maxQueueLengthTime = -1
      if 'csv' in cmdType:
         self.txLatency = float( m.group( 'latency' ) )
         self.txDrops = int( m.group( 'drops' ) )
      if m.group( 'intfType' ) == "Fa":
         self.fabricPeerIntf = m.group( 'peerFabricIntf' )

   def renderFn( self, cmdType, timeOffset ):
      qLenStr = getQLenStr( self.queueLength, self.entryType )
      durStr = getDurationStr( self.congestionDuration, self.entryType )
      maxQLenTimeStr = getDurationStr( self.maxQueueLengthTime, self.entryType )
      if self.interface:
         intfTcStr = self.getInterfaceName()
         intfTcStr += "(%d)" % ( self.trafficClass )
      else:
         intfTcStr = 'N/A'
      fabricPeerStr = "" if not self.fabricPeerIntf else self.fabricPeerIntf
      if 'csv' in cmdType:
         formatString = " %s,%s,%s,%s,%s,%s,%.3f,%s,%s"
         timeStr = getTimeStr( self.entryTime )
         print( formatString % ( self.entryType, timeStr,
                                 intfTcStr, durStr, qLenStr,
                                 maxQLenTimeStr, self.txLatency,
                                 self.txDrops, fabricPeerStr ) )
      else:
         formatString = " %-4s%-26s%-35s%-15s%-15s%-11s%-15s%-15s"
         timeStr = getTimeStr( self.entryTime, timeOffset )
         absoluteTimeStr = getTimeStr( self.entryTime )
         print( formatString % ( self.entryType,
                                 timeStr, absoluteTimeStr, intfTcStr,
                                 durStr, qLenStr, maxQLenTimeStr,
                                 fabricPeerStr ) )

class StrataDropsRecord( StrataCommonRecord ):
   entryTime = Float( help='UTC time at which congestion drops are seen.' )
   txDrops = Int( help='Number of packet drops seen on the interface due to '
                  'congestion.' )
   def parseShowQMonLengthRecord( self, entry, cmdType ):
      matchRe = '(?P<timestamp>.*),\
(?P<intfType>Et|Cpu|Fa)(?P<intfOrCardChip>.*)\\((?P<tc>\\d+)\\),(?P<drops>.*)'
      m = re.match( matchRe, entry )
      self.parseRecord( m )
      self.entryTime = Microseconds.toSeconds( int( m.group( 'timestamp' ) ) )
      self.txDrops = int( m.group( 'drops' ) )

   def renderFn( self, cmdType, timeOffset ):
      timeStr = getTimeStr( self.entryTime, timeOffset )
      intfTcStr = self.getInterfaceName()
      intfTcStr += "(%d)" % ( self.trafficClass )
      print( "%-40s%-15s%-10u" % ( timeStr,
                                  intfTcStr,
                                  self.txDrops ) )

class StrataLatencyRecord( StrataCommonRecord ):
   entryTime = Float( help='Time at which the tx latency is recorded.' )
   txLatency = Float( help='Tx-delay information in usecs.' ) 

   def parseShowQMonLengthRecord( self, entry, cmdType ):
      matchRe = '(?P<timestamp>.*),\
(?P<intfType>Et|Cpu|Fa)(?P<intfOrCardChip>.*)\\((?P<tc>\\d+)\\),(?P<latency>.*)'
      m = re.match( matchRe, entry )
      self.parseRecord( m )
      self.entryTime = Microseconds.toSeconds( int( m.group( 'timestamp' ) ) )
      self.txLatency = float(  m.group( 'latency' ) )

   def renderFn( self, cmdType, timeOffset ):
      timeStr = getTimeStr( self.entryTime, timeOffset )
      intfTcStr = self.getInterfaceName()
      intfTcStr += "(%d)" % ( self.trafficClass )
      print( "%-30s%-15s%-35.3f" % ( timeStr,
                                    intfTcStr,
                                    self.txLatency ) )

class StrataStatisticsRecord( StrataCommonRecord ):
   numCongestionEvents = Int( help='Number of times congestion started '
                              'on the interface per traffic-class.' )
   def parseShowQMonLengthRecord( self, entry, cmdType ):
      matchRe = '(?P<intfType>Et|Cpu|Fa)(?P<intfOrCardChip>.*),(?P<tc>.*),\
(?P<stats>.*)'
      m = re.match( matchRe, entry )
      self.parseRecord( m )
      self.numCongestionEvents = int( m.group( 'stats' ) )

   def renderFn( self, cmdType, timeOffset ):
      intfName = self.getInterfaceName()
      print( "%-15s %-15s %-15s" % ( intfName,
                                     self.trafficClass,
                                     self.numCongestionEvents ) )

class QueueMonitorLength( Model ):
   lanzEnabled = Bool( help="Queue length monitoring enabled" )
   entryList = GeneratorList( valueType=CongestionRecordModel,
                              help="List of the congestion records" )
   globalHitCount = Int( help='Number of times the global High threshold is hit',
                         default=0 )
   reportTime = Float( help='UTC time at which the congestion report is generated' )
   bytesPerTxmpSegment = Int( help='Number of bytes per txmp segment' )

   #optional attributes
   platformName = Enum( values=( 'Sand', 'Strata', 'Xp', 'Barefoot', 'Mako' ),
                        help="Forwarding plane name",
                        optional=True )
   #private attributes
   _lanzRunning = Bool( help="Lanz agent is running", optional=True )
   _commandType = Str( help="Version of the command requested",
                      default='' ) 

   def printHeader( self ):
      reportTime = time.strftime( "%Y-%m-%d %H:%M:%S",
                                  time.localtime( self.reportTime ) )
      print( '\nReport generated at %s' % reportTime )
      if self.platformName == 'Strata':
         if '-b' not in self._commandType and \
                '-d' not in self._commandType and \
                '-l' not in self._commandType and \
                '-c' not in self._commandType and \
                'csv' not in self._commandType:
            print( "S-Start, U-Update, E-End, TC-Traffic Class" )
            print( "Segment size for S, U and E congestion records is %u bytes" %
                   self.bytesPerTxmpSegment )
            print( "* Max queue length during period of congestion" )
            print( "+ Period of congestion exceeded counter" )
            print( "-------------------------------------------------"
                   "----------------------------------------------"
                   "-----------------------------------" )
            print( "%-8s%-23s%-35s%-15s%-15s%-11s%-15s%-15s" % (
               "Type", "Time", "Absolute", "Interface", "Congestion", "Queue",
               "Time of Max", "Fabric" ) )
            print( "%-31s%-35s%-15s%-15s%-11s%-15s%-15s" % (
               " ", "Time", "(TC)", "duration", "length", "Queue length", "Peer" ) )
            print( "%-81s%-15s%-11srelative to" % ( " ", "(usecs)", "(segments)" ) )
            print( "%-107scongestion" % "" )
            print( "%-107sstart" % "" )
            print( "%-107s(usecs)" % "" )
            print( "-------------------------------------------------"
                   "----------------------------------------------"
                   "-----------------------------------" )
         elif '-d' in self._commandType:
            print( "%-40s%-15sTX Drops" % ( "Time", "Interface(TC)" ) )
            print( "-------------------------------------"
                   "----------------------------" )
         elif '-l' in self._commandType:
            print( "%-30s%-15sTx-Latency (usecs)" % ( "Time", "Interface(TC)" ) )
            print( "-------------------------------------"
                   "----------------------------" )
         elif '-c' in self._commandType:
            print( "%-15s%-15s%-15s" % ( "Interface", "Traffic Class", "Count" ) )
            print( "-------------------------------------"
                   "----------------------------" )
         elif 'csv' in self._commandType:
            print( "Type,Time,Interface,Duration(usecs),Queue-Length,"
                   "Time-Of-Max-Queue(usecs),Latency(usecs),Tx-Drops,Fabric-Peer" )
         else:
            pass
      elif self.platformName == 'Sand':
         if '-l' in self._commandType:
            print( "%-25s%-15s%-13sIngress" % ( "Time", "Intf(TC)", "Tx-Latency" ) )
            print( "%-40s%-13s%-16s" % ( "", "(usecs)", "Port-set" ) )
            print( '-' * 81 )
         elif 'csv' not in self._commandType:
            print( "S-Start, U-Update, E-End, P-Polling, TC-Traffic Class" )
            print( "* Max queue length during period of congestion" )
      elif self.platformName == 'Barefoot':
         if '-c' not in self._commandType:
            print( "Segment size = %u bytes" % self.bytesPerTxmpSegment )

         if 'csv' in self._commandType:
            print( "Time,Interface,TX-Queue,Duration(usecs),Queue-Length(segments)" )
         elif '-c' in self._commandType:
            print( "%-15s%-15s%-15s" % ( "Interface", "TX-Queue", "Count" ) )
            print( "-------------------------------------"
                   "----------------------------" )
      elif self.platformName == 'Mako':
         if 'csv' in self._commandType:
            print( "Type,Interface,BufferType,Time,Duration(usecs)" )
         elif '-c' in self._commandType:
            # statistics record, in table format no header
            pass
         else:
            # Normal output
            print( "Entry type: S-Start, E-End" )
      elif self.platformName == 'Xp':
         if 'csv' not in self._commandType and \
               '-c' not in self._commandType:
            print( "S-Start, U-Update, E-End, TC-Traffic Class" )
            print( "Segment size for congestion records is 256 bytes" )
            print( "* Max queue length during period of congestion" )
            print( "%-7s%-23s%-31s%-11s%-14s%-11sTime of Max" % (
               "Type", "Time", "Absolute", "Intf", "Congestion", "Queue" ) )
            print( "%-30s%-31s%-11s%-14s%-11sQueue length" % (
               " ", "Time", "(TC)", "duration ", "length" ) )
            print( "%-72s%-14s%-11srelative to" % ( " ", "(usecs)", "(segments)" ) )
            print( "%-97scongestion" % "" )
            print( "%-97sstart" % "" )
            print( "%-97s(usecs)" % "" )
            print( "-" * 111 )
         elif '-c' in self._commandType:
            print( "%-15s%-15s%-15s" % ( "Interface", "Traffic Class", "Count" ) )
            print( "-------------------------------------"
                   "----------------------------" )
         else:
            assert 'csv' in self._commandType
            print( "Type,Time,Interface,Duration(usecs),Queue-Length,"
                   "Time-Of-Max-Queue(usecs)" )

   def render( self ):
      if not self._lanzRunning:
         return

      if not self.lanzEnabled:
         print( "queue-monitor length is disabled." )
         return

      self.printHeader()
      # TODO BUG336553
      # SandCongestionRecord does not use the renderFn function, instead, it is
      # defined in-place, in the else clause. The reason is that
      # SandCongestionRecord
      # uses the TableOutput, which needs to be shared between all
      # entries.
      # Sand uses the `renderFn` for the other cases, i.e., 'csv'
      # and the '-l'.
      # Barefoot uses TableOutput for BarefootCongestionRecords
      # Mako uses TableOutput for MakoCongestionRecords
      sandUseTableOutput = self.platformName == 'Sand' and \
            not any( t in self._commandType for t in [ 'csv', '-l' ] )
      barefootUseTableOutput = self.platformName == 'Barefoot' and \
            not any( t in self._commandType for t in [ 'csv', '-c' ] )
      makoUseTableOutput = self.platformName == 'Mako' and \
         'csv' not in self._commandType
      if sandUseTableOutput:
         tableHeadings = ( "Type", "Time", "Absolute\nTime", "Intf(TC)",
                           "Queue\nLength\n(bytes)", "Duration\n\n(usecs)",
                           "Ingress\nPort-set" )
         table = TableOutput.createTable( tableHeadings, tableWidth=184 )
         f1 = TableOutput.Format( justify="left", maxWidth=4 )
         f2 = TableOutput.Format( justify="left", maxWidth=30 )
         f3 = TableOutput.Format( justify="left", maxWidth=30 )
         f4 = TableOutput.Format( justify="left", maxWidth=50 )
         f5 = TableOutput.Format( justify="left", maxWidth=10 )
         f6 = TableOutput.Format( justify="left", maxWidth=10 )
         f7 = TableOutput.Format( justify="left", maxWidth=50, wrap=True )
         f1.noPadLeftIs( True )
         f2.noPadLeftIs( True )
         f3.noPadLeftIs( True )
         f4.noPadLeftIs( True )
         f5.noPadLeftIs( True )
         f6.noPadLeftIs( True )
         f7.noPadLeftIs( True )
         table.formatColumns( f1, f2, f3, f4, f5, f6, f7 )

         for entry in self.entryList:
            timeStr = getTimeStr( entry.entryTime, self.reportTime )
            absoluteTimeStr = getTimeStr( entry.entryTime )
            queueLenStr = getQLenStr( entry.queueLength, entry.entryType )
            durationStr = getSandDurationStr( entry.durationUsecs )
            intfTcStr = "%s" % ( IntfCli.Intf.getShortname( entry.interface ) )
            if entry.trafficClass is not None:
               intfTcStr += "(%s)" % entry.trafficClass
            # This ingressPortSet is supposed to be in the map already as it should
            # have been retrieved through a `SandCongestionRecord`.
            ingressPortSetKey = repr( entry.ingressPortSet )
            assert ingressPortSetKey in portSetStrepCache
            table.newRow( entry.entryType, timeStr, absoluteTimeStr, intfTcStr,
                          queueLenStr, durationStr,
                          portSetStrepCache[ ingressPortSetKey ] )
         print( table.output() )
      elif barefootUseTableOutput:
         tableHeadings = ( "Time", "Absolute\nTime", "Intf(TX-Queue)",
                           "Queue Length\n(segments)", "Duration\n(usecs)" )
         table = TableOutput.createTable( tableHeadings, tableWidth=154 )
         f1 = TableOutput.Format( justify="left", maxWidth=30 )
         f2 = TableOutput.Format( justify="left", maxWidth=30 )
         f3 = TableOutput.Format( justify="left", maxWidth=15 )
         f4 = TableOutput.Format( justify="left", maxWidth=15 )
         f5 = TableOutput.Format( justify="left", maxWidth=16 )
         f1.noPadLeftIs( True )
         f2.noPadLeftIs( True )
         f3.noPadLeftIs( True )
         f4.noPadLeftIs( True )
         f5.noPadLeftIs( True )
         table.formatColumns( f1, f2, f3, f4, f5 )

         for entry in self.entryList:
            timeStr = getTimeStr( entry.entryTime, self.reportTime )
            absoluteTimeStr = getTimeStr( entry.entryTime )
            queueLenStr = "%u" % entry.queueLength
            durationStr = "%u" % entry.duration
            intfTxQStr = "%s" % ( IntfCli.Intf.getShortname( entry.interface ) )
            if entry.txQueue is not None:
               intfTxQStr += "(%u)" % entry.txQueue
            table.newRow( timeStr, absoluteTimeStr, intfTxQStr,
                          queueLenStr, durationStr )
         print( table.output() )
      elif makoUseTableOutput:
         statistics = '-c' in self._commandType

         if statistics:
            tableHeadings = (
               "Interface",
               "Buffer",
               "Congestion Count",
            )
            table = TableOutput.createTable( tableHeadings, tableWidth=90 )
            f1 = TableOutput.Format( justify="left", maxWidth=30 )
            f1.noPadLeftIs( True )
            f1.padLimitIs( True )
            f2 = TableOutput.Format( justify="left", maxWidth=30 )
            f2.noPadLeftIs( True )
            f2.padLimitIs( True )
            f3 = TableOutput.Format( justify="right", maxWidth=30 )
            f3.padLimitIs( True )
            table.formatColumns( f1, f2, f3 )
            for entry in self.entryList:
               intfStr = entry.interface.stringValue
               congestionBufferStr = makoBufferTypeToStr( entry.bufferType )
               numCongestionEvents = entry.numCongestionEvents
               table.newRow(
                  intfStr,
                  congestionBufferStr,
                  numCongestionEvents
               )
         else:
            tableHeadings = (
               "Entry\nType",
               "Interface",
               "Buffer Type",
               "Time",
               "Absolute Time",
               "Duration\n(usecs)"
            )
            table = TableOutput.createTable( tableHeadings, tableWidth=185 )
            f1 = TableOutput.Format( justify="left", maxWidth=5 )
            f1.noPadLeftIs( True )
            f1.padLimitIs( True )
            f2 = TableOutput.Format( justify="left", maxWidth=30, wrap=True )
            f2.noPadLeftIs( True )
            f2.padLimitIs( True )
            f3 = TableOutput.Format( justify="left", maxWidth=20, wrap=True )
            f3.noPadLeftIs( True )
            f3.padLimitIs( True )
            f4 = TableOutput.Format( justify="right", maxWidth=50, wrap=True )
            f4.padLimitIs( True )
            f5 = TableOutput.Format( justify="right", maxWidth=50, wrap=True )
            f5.padLimitIs( True )
            f6 = TableOutput.Format( justify="right", maxWidth=30, wrap=True )
            f6.padLimitIs( True )
            table.formatColumns( f1, f2, f3, f4, f5, f6 )
            for entry in self.entryList:
               entryTypeStr = entry.entryType
               intfStr = entry.interface.stringValue
               congestionBufferStr = makoBufferTypeToStr( entry.bufferType )
               timeStr = getTimeStr( entry.entryTime, self.reportTime )
               absoluteTimeStr = getTimeStr( entry.entryTime )
               durationStr = "%u" % entry.durationUsecs
               table.newRow(
                  entryTypeStr,
                  intfStr,
                  congestionBufferStr,
                  timeStr,
                  absoluteTimeStr,
                  durationStr
               )

         print( table.output() )
      else:
         for entry in self.entryList:
            # pylint: disable-next=unused-variable
            rendered = entry.renderFn( self._commandType, self.reportTime )

class MakoQueueMonitorLengthStatusModel( Model ):
   # Mako LANZ is fairly different from all the other LANZ in terms of
   # queue constructs. Status reporting output would look fairly different.
   lanzEnabled = Bool( help="Queue length monitoring enabled" )
   ingressHighThreshold = Int( help="High threshold on ingress buffer in bytes" )
   ingressLowThreshold = Int( help="Low threshold on ingress buffer in bytes" )
   ingressMemorySize = Int( help="Memory size of ingress buffer in bytes" )

   egressHighThreshold = Int( help="High threshold on egress buffer in bytes" )
   egressLowThreshold = Int( help="Low threshold on egress buffer in bytes" )
   egressMemorySize = Int( help="Memory size of egress buffer in bytes" )

   corememHighThreshold = Int( help="High threshold on core memory buffer in bytes",
                               optional=True )
   corememLowThreshold = Int( help="Low threshold on core memory buffer in bytes",
                              optional=True )
   corememMemorySize = Int( help="Memory size of core memory buffer in bytes",
                            optional=True )

   def render( self ):
      if self.lanzEnabled:
         print( "Buffer monitoring enabled" )
      else:
         print( "Buffer monitoring disabled" )
      print( "All thresholds and sizes in bytes" )

      tableHeadings = (
         "Buffer",
         "High Threshold\n(bytes)",
         "Low Threshold\n(bytes)",
         "Total Size\n(bytes)"
      )
      table = TableOutput.createTable( tableHeadings, tableWidth=80 )
      f1 = TableOutput.Format( justify="left", maxWidth=20 )
      f1.noPadLeftIs( True )
      f1.padLimitIs( True )
      f2 = TableOutput.Format( justify="right", maxWidth=20 )
      f2.padLimitIs( True )
      f3 = TableOutput.Format( justify="right", maxWidth=20 )
      f3.padLimitIs( True )
      f4 = TableOutput.Format( justify="right", maxWidth=20 )
      f4.padLimitIs( True )
      table.formatColumns( f1, f2, f3, f4 )

      table.newRow(
         "ingress",
         self.ingressHighThreshold,
         self.ingressLowThreshold,
         self.ingressMemorySize
      )
      table.newRow(
         "egress",
         self.egressHighThreshold,
         self.egressLowThreshold,
         self.egressMemorySize
      )
      if self.corememMemorySize:
         table.newRow(
            'core memory',
            self.corememHighThreshold,
            self.corememLowThreshold,
            self.corememMemorySize
         )
      print( table.output() )

class QueueMonitorLengthStatus( Model ):
   lanzEnabled = Bool( help="Queue length monitoring enabled" )
   # optional attributes
   platformName = Enum( values=( 'Sand', 'Strata', 'Xp', 'Barefoot', 'Mako' ),
                        help="Forwarding plane name",
                        optional=True )
   makoShowQMonLenStatusModel = Submodel(
      help='Mako queue monitor length status',
      valueType=MakoQueueMonitorLengthStatusModel, optional=True )
   txLatencyEnablingSupported = Bool( help="Latency measurement enabling supported" )
   txLatencyEnabled = Bool( help="Latency measurement enabled" )
   ptpSupported = Bool( help="PTP is supported on current system" )
   # lanzMode and lanzSliceModes are used only on Sand platform
   lanzMode = Enum( values=( 'notifying', 'polling' ),
                    help="Configured queue length monitoring mode",
                    optional=True )
   lanzSliceModes = Dict( keyType=str, valueType=str,
                           help="Mapping of Slice name to Queue length "
                                "monitoring status",
                           optional=True )
   latencyCapableLinecards = Dict( keyType=str, valueType=bool,
                                   help="Linecards capable of measuring ingress"
                                       "latency",
                                   optional=True )
   packetSamplingEnabled = Bool( help="Queue-Monitor Length Mirroring Enabled" )
   mirrorDestIntf = List( valueType=str,
                          help="Mirror destination interfaces" )
   queueLenUnit = Enum( values=( 'segments', 'bytes' ),
                        help="Queue length specified in segments or bytes" )
   maxQueueLength = Int( help="Maximum queue length" )
   bytesPerRxmpSegment = Int( help="Rxmp segment size in bytes" )
   bytesPerTxmpSegment = Int( help="Txmp segment size in bytes" )
   logInterval = Float( help="Syslog interval in seconds" )

   globalBufferSupported = Bool( help="Global buffer monitoring supported" )
   globalBufferEnabled = Bool( help="Global buffer monitoring enabled" )
   globalHighThreshold = Int( help="High thresold on global buffer" )
   globalLowThreshold = Int( help="Low threshold on global buffer" )
   globalLogInterval = Float( help="Syslog interval for global buffer monitoring" )

   numTotalSegments = Int( help="Total buffers in segments" )
   lowThresholdSupported = Bool( help="lower threshold supported" )
   mirroringSupported = Bool( help="Mirroring supported" )
   updateIntervalSupported = Bool( help="Update interval supported" )
   updateIntervalSupportedCards = \
                     Dict( keyType=str, valueType=bool,
                           help="List of cards types that support update interval",
                           optional=True )
   qLenInterval = Int( help="Queue length interval" )

   class InterfaceStatusModel( Model ):
      lanzDisabled = Bool( help="Queue length monitoring disabled status on "
                           "the interface",
                           default=False )
      highThreshold = Int( help="High threshold configured on the interface" )
      lowThreshold = Int( help="Low threshold configured on the interface" )
      mirroringEnabled = Bool( help="Queue-monitor length mirroring enabled\
 on the interface", default=True )

      class InactiveWarningModel( Model ):
         maxQueueLength = Int( help="Maximum VOQ tail drop threshold" )
      inactiveWarning = Submodel( help="Thresholds inactive if greater than the "
                                       "interface's maximum queue length",
                                  valueType=InactiveWarningModel, optional=True )
 
   # interface Dict
   intfStatusAll = Dict( keyType=str,
                         valueType=InterfaceStatusModel, 
                         help="List of Interfaces" )
   def render( self ):
      if self.platformName == 'Mako':
         self.makoShowQMonLenStatusModel.render()
         return

      if not self.lanzEnabled:
         print( "queue-monitor length disabled" )
      else:
         print( "queue-monitor length enabled" )

      if not self.packetSamplingEnabled:
         print( "queue-monitor length packet sampling is disabled" )
      else:
         print( "queue-monitor length packet sampling is enabled" )

      if self.txLatencyEnablingSupported:
         if self.ptpSupported:
            if self.txLatencyEnabled:
               print( "queue-monitor length tx-latency is enabled" )
            else:
               print( "queue-monitor length tx-latency is disabled" )
         else:
            print( "queue-monitor length tx-latency is disabled due to PTP not "
                   "supported" )

      if self.updateIntervalSupported:
         if self.updateIntervalSupportedCards:
            updateSupportedList = ", ".join( self.updateIntervalSupportedCards )
            print( "queue-monitor length update interval supported on: %s" %
                   ( updateSupportedList ) )
         print( "queue-monitor length update interval in micro seconds: %8d " %
                ( self.qLenInterval ) )

      if len ( self.mirrorDestIntf ) > 1:
         print( "Mirror destination interfaces are", end=' ' )
         for key in sorted( self.mirrorDestIntf ):
            print( "%s" % ( key ), end=' ' )
         print( "" )
      elif len ( self.mirrorDestIntf ) == 1:
         print( "Mirror destination interface is %s" % ( self.mirrorDestIntf[ 0 ] ) )

      if self.queueLenUnit == 'segments':
         segments = True
         unit = "segments"
      else:
         segments = False
         unit = "bytes"
      if self.logInterval:
         print( "Syslog interval in seconds       : %5d" % ( self.logInterval ) )
      if self.globalBufferSupported:
         print( "Global Buffer Monitoring" )
         print( "------------------------" )
         if self.globalBufferEnabled:
            print( "Global buffer monitoring is enabled" )
         else:
            print( "Global buffer monitoring is disabled" )
         if segments:
            print( "Segment size in bytes : %5d" % ( self.bytesPerRxmpSegment ) )
      
         print( "Total buffers in segments : %5d" % self.numTotalSegments )
         print( "High threshold : %5d" % ( self.globalHighThreshold ) )
         print( "Low threshold : %5d" % ( self.globalLowThreshold ) )
         if self.globalLogInterval:
            print( "Syslog interval in seconds : %5d" % ( self.globalLogInterval ) )
         print( "\n" )
      print( "Per-Interface Queue Length Monitoring" )
      print( "-------------------------------------" )
      if self.lanzEnabled:
         print( "Queue length monitoring is enabled" )
         # lanzMode and lanzSliceModes are used only on Sand platform
         if self.lanzSliceModes:
            print( "Queue length monitoring mode is %s" % ( self.lanzMode ) )
            print( "Queue length monitoring status is:" )
            for sliceName, sliceStatus in \
                    sorted( self.lanzSliceModes.items(), key=sliceNo ):
               slicesStr = "   %-14s %s" % ( sliceName, sliceStatus )
               if self.txLatencyEnabled and self.latencyCapableLinecards and \
                  sliceName in self.latencyCapableLinecards:
                  slicesStr += ", tx-latency"
               print( slicesStr )
      else:
         print( "Queue length monitoring is disabled" )
         # lanzMode and lanzSliceModes are used only on Sand platform
         if self.lanzSliceModes:
            print( "Queue length monitoring mode is %s" % ( self.lanzMode ) )

      if segments:
         print( "Segment size in bytes : %5d" % ( self.bytesPerTxmpSegment ) )
         
      # pylint: disable-next=bad-string-format-type
      print( "Maximum queue length in %s : %5d" % ( unit, self.maxQueueLength ) )

      # Create a list of max queue lengths that will be printed in the warning
      # summary header.
      inactiveMaxQueueLengths = []
      # pylint: disable-next=unused-variable
      inactiveString = LanzWarningsLib.InactiveThreshold.fullName
      for intf in self.intfStatusAll:
         inactiveWarning = self.intfStatusAll[ intf ].inactiveWarning
         if inactiveWarning:
            maxQueueLength = inactiveWarning.maxQueueLength
            if maxQueueLength not in inactiveMaxQueueLengths:
               inactiveMaxQueueLengths.append( maxQueueLength )

      # Print the warnings
      # pylint: disable-next=use-implicit-booleaness-not-len
      if len( inactiveMaxQueueLengths ):
         print( "Warnings" )
         inactiveMaxQueueLengths.sort()
         inactiveThreshold = LanzWarningsLib.InactiveThreshold()
         for maxQueueLength in inactiveMaxQueueLengths:
            print( "   " + inactiveThreshold.getSummary( maxQueueLength ) )

      print( "Port thresholds in %s:" % ( unit ) )

      perIntfHeader = "%-8s%15s" % ( "Port", "High threshold" )
      
      if self.lowThresholdSupported:
         perIntfHeader += "%15s" % ( "Low threshold" )

      if self.mirroringSupported:
         perIntfHeader += "%20s" % ( "Mirroring Enabled" )

      perIntfHeader += "%15s" % ( "Warnings" )

      print( perIntfHeader )

      for intf in Arnet.sortIntf( self.intfStatusAll ):
         intfStatus = self.intfStatusAll[ intf ]
         if not intfStatus.lanzDisabled:
            intfName = intf
            if not intfName.startswith( "Fa" ):
               intfName = IntfCli.Intf.getShortname( intf )

            perIntfStatus = "%-8s%15d" % ( intfName,
                                           intfStatus.highThreshold )
            if self.lowThresholdSupported:
               perIntfStatus += "%15d" % ( intfStatus.lowThreshold )

            if self.mirroringSupported:
               perIntfStatus += "%20s" % ( intfStatus.mirroringEnabled )

            intfWarnings = []
            inactiveWarning = intfStatus.inactiveWarning
            if inactiveWarning:
               maxQueueLength = inactiveWarning.maxQueueLength
               inactiveThreshold = LanzWarningsLib.InactiveThreshold()
               letterCode = inactiveThreshold.getLetterCode( maxQueueLength )
               intfWarnings.append( letterCode )

            if len( intfWarnings ): # pylint: disable=use-implicit-booleaness-not-len
               intfWarningsString = ", ".join( intfWarnings )
               perIntfStatus += "%15s" % intfWarningsString

            print( perIntfStatus )
         else:
            print( "%-8s%15s" % ( IntfCli.Intf.getShortname( intf ), "disabled" ) )

class ListOfHostnames( Model ):
   hostnames = List( valueType=str,
                     help="List of hostnames" )

class QueueMonitorStreamingClients( Model ):
   __revision__ = 2
   lanzEnabled = Bool( help="Queue length monitoring enabled" )
   streamingEnabled = Bool( help="Queue length streaming enabled" )
   clientConnections = Int(
      help="Total number of clients connected to Lanz Streaming" )
   clientHostnames = Dict( keyType=str, valueType=ListOfHostnames,
                           help="A mapping of client hostname lists to VRF names" )
   _vrf = Str( help="VRF to filter streaming clients by", optional=True )

   def render( self ):
      if not self.lanzEnabled:
         print( "Queue length monitoring is disabled" )
         return

      if not self.streamingEnabled:
         print( "Queue length streaming is disabled" )
         return

      print( "Number of clients connected:", self.clientConnections )
      print( "--------------------------------" )
      for vrfName, hosts in self.clientHostnames.items():
         hostlist = hosts.hostnames
         print( "VRF:", vrfName )
         clients = len( hostlist )
         print( "Client count:", clients )
         if clients < 1:
            print()
            continue
         print( "Client list:" )
         for host in hostlist:
            print( host )
         print()

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         sourceDict = dictRepr.get( 'clientHostnames' )
         destList = []
         for vrfName in sourceDict:
            destList.extend( sourceDict[ vrfName ][ 'hostnames' ] )
         dictRepr[ 'clientHostnames' ] = destList
      return dictRepr
