#!/usr/bin/env python3
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements interface-specific Lanz configuration.
#
#-------------------------------------------------------------------------------
'''Interface configuration commands for Lanz'''
import CliParser
from CliPlugin import IntfCli
from CliPlugin import LanzCli
import ConfigMount
from EthIntfUtil import isEthernetIntf
import LanzLib
import LazyMount
import Tac
import Tracing

__defaultTraceHandle__ = Tracing.Handle( 'LanzCli' )
lanzConfig = None
lanzHwStatus = None

class LanzIntf( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      del lanzConfig.portConfig[ self.intf_.name ]

#-------------------------------------------------------------------------------
# Adds Lanz commands to the "config-if" mode.
#-------------------------------------------------------------------------------

class LanzModelet ( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      # This is a bit hacky, but there's no better way to recognize physical Ethernet
      # interfaces than to look at their name.
      return isEthernetIntf( mode.intf.name ) and not mode.intf.isSubIntf()

modelet = LanzModelet
IntfCli.IntfConfigMode.addModelet( modelet )

# Helper method to check for cpu or fabric in syntax

def isCpuOrFabric( args ):
   if 'cpu' in args:
      return 'cpu'
   elif 'fabric' in args:
      return 'fabric'
   else:
      return None

def disableQMon( mode, intfName ):
   portConfig = Tac.newInstance( 'Lanz::PortConfig', intfName )
   portConfig.highThreshold = lanzHwStatus.maxThreshold
   portConfig.lowThreshold = lanzHwStatus.maxThreshold
   portConfig.disabled = True
   portConfig.mirroringEnabled = lanzHwStatus.mirroringSupported or \
       ( not lanzHwStatus.initialized and not mode.session_.guardsEnabled() )
   lanzConfig.portConfig.addMember( portConfig )

def noQMonitor( mode, cpuOrFabric=None ):
   if cpuOrFabric is not None:
      disableQMon( mode, cpuOrFabric.title() )
   else:
      disableQMon( mode, mode.intf.name )

def defaultQMonitor( mode, cpuOrFabric=None ):
   intfName = cpuOrFabric.title() if cpuOrFabric else mode.intf.name
   del lanzConfig.portConfig[ intfName ]

def qMonitorLengthMirror( mode ):
   intfName = mode.intf.name
   portConfig = Tac.newInstance( 'Lanz::PortConfig', intfName )
   portConfig.mirroringEnabled = True
   if intfName in lanzConfig.portConfig:
      if not lanzConfig.portConfig[ intfName ].disabled and \
         not lanzConfig.portConfig[ intfName ].validThreshold:
         del lanzConfig.portConfig[ intfName ]
      else:
         portConfig.highThreshold = lanzConfig.portConfig[ intfName ].highThreshold
         portConfig.lowThreshold = lanzConfig.portConfig[ intfName ].lowThreshold
         portConfig.validThreshold = lanzConfig.portConfig[ intfName ].validThreshold
         portConfig.disabled = lanzConfig.portConfig[ intfName ].disabled
         lanzConfig.portConfig.addMember( portConfig )

def noQMonitorLengthMirror( mode ):
   intfName = mode.intf.name
   portConfig = Tac.newInstance( 'Lanz::PortConfig', intfName )
   portConfig.mirroringEnabled = False
   # Copy threshold config if it exists
   if intfName in lanzConfig.portConfig:
      portConfig.highThreshold = lanzConfig.portConfig[ intfName ].highThreshold
      portConfig.lowThreshold = lanzConfig.portConfig[ intfName ].lowThreshold
      portConfig.validThreshold = lanzConfig.portConfig[ intfName ].validThreshold
      portConfig.disabled = lanzConfig.portConfig[ intfName ].disabled
   else:
      portConfig.highThreshold = \
            LanzLib.getDefaultHighThreshold( lanzConfig, lanzHwStatus )
      portConfig.lowThreshold = \
            LanzLib.getDefaultLowThreshold( lanzConfig, lanzHwStatus )
      # if the thresholds retrieved above were taken from an uninitialized
      # lanzHwStatus they will be invalid
      portConfig.validThreshold = ( lanzHwStatus.initialized or
                                    lanzConfig.defaultThresholdsConfig.isActive )
      portConfig.disabled = False
   lanzConfig.portConfig.addMember( portConfig )

def setThreshold( mode, highThreshold, lowThreshold, intfName ):
   if not LanzCli.validThresholds( mode, highThreshold, lowThreshold ):
      return

   portConfig = Tac.newInstance( 'Lanz::PortConfig', intfName )
   portConfig.highThreshold = highThreshold
   portConfig.lowThreshold = lowThreshold
   portConfig.validThreshold = True
   portConfig.mirroringEnabled = lanzHwStatus.mirroringSupported or \
       ( not lanzHwStatus.initialized and not mode.session_.guardsEnabled() )
   if intfName in lanzConfig.portConfig:
      mirroringEnabled = lanzConfig.portConfig[ intfName ].mirroringEnabled
      portConfig.mirroringEnabled = mirroringEnabled
   lanzConfig.portConfig.addMember( portConfig )

def configThresholds( mode, highThreshold, lowThreshold, cpuOrFabric=None ):
   if cpuOrFabric is not None:
      setThreshold( mode, highThreshold, lowThreshold, cpuOrFabric.title() )
   else:
      setThreshold( mode, highThreshold, lowThreshold, mode.intf.name )

def setDefaultThresholds( mode, cpuOrFabric=None ):
   if cpuOrFabric is not None:
      intfName = cpuOrFabric.title()
   else:
      intfName = mode.intf.name
   if intfName in lanzConfig.portConfig:
      mirroringEnabled = lanzConfig.portConfig[ intfName ].mirroringEnabled
      # mirroring being disabled is only the default config if mirroring
      # is supported
      if ( lanzHwStatus.mirroringSupported or
           ( not lanzHwStatus.initialized and
             not mode.session_.guardsEnabled() ) ) and not mirroringEnabled:
         portConfig = Tac.newInstance( 'Lanz::PortConfig', intfName )
         if intfName == "Cpu":
            portConfig.highThreshold = lanzHwStatus.defCpuHighThreshold
            portConfig.lowThreshold = lanzHwStatus.defCpuLowThreshold
         elif intfName.startswith( "Fabric" ):
            portConfig.highThreshold = lanzHwStatus.defFabricHighThreshold
            portConfig.lowThreshold = lanzHwStatus.defFabricLowThreshold
         else:
            portConfig.highThreshold = \
                  LanzLib.getDefaultHighThreshold( lanzConfig, lanzHwStatus )
            portConfig.lowThreshold = \
                  LanzLib.getDefaultLowThreshold( lanzConfig, lanzHwStatus )
         portConfig.validThreshold = False
         portConfig.mirroringEnabled = mirroringEnabled
         lanzConfig.portConfig.addMember( portConfig )
      else:
         del lanzConfig.portConfig[ intfName ]

def configThreshold( mode, highThreshold, cpu=None ):
   lowThreshold = highThreshold // 2
   if not cpu:
      setThreshold( mode, highThreshold, lowThreshold, mode.intf.name )
   else:
      setThreshold( mode, highThreshold, lowThreshold, "Cpu" )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------

def Plugin( entityManager ):
   global lanzConfig
   global lanzHwStatus
   lanzConfig = ConfigMount.mount( entityManager, 'lanz/config', 'Lanz::Config',
                                   'w' )
   lanzHwStatus = LazyMount.mount( entityManager, 'lanz/hardware/status',
                                   'Lanz::Hardware::Status', 'r' )
   IntfCli.Intf.registerDependentClass( LanzIntf )
