#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliMatcher
import ShowCommand
import CliPlugin.AclCli as AclCli # pylint: disable=consider-using-from-import
import CliPlugin.LanzCli as LanzCli # pylint: disable=consider-using-from-import

from CliPlugin import VrfCli
from CliPlugin import LanzIntfCli
from CliPlugin.AclCliModel import AllAclList
from CliPlugin import CpuIntfCli
from CliPlugin import EthIntfCli
from CliPlugin import SwitchIntfCli
from CliPlugin.LanzCliModel import ( QueueMonitorLengthStatus,
                                     QueueMonitorStreamingClients,
                                     QueueMonitorLength )
from Intf.IntfRange import IntfRangeMatcher
from CliToken import Lanz

intfTypes = [ EthIntfCli.EthPhyAutoIntfType,
              CpuIntfCli.CpuAutoIntfType, SwitchIntfCli.SwitchAutoIntfType ]
intfRangeMatcher = IntfRangeMatcher( explicitIntfTypes=intfTypes )

#--------------------------------------------------------------------------------
# show queue-monitor length [ ( interface | range ) ]
# [ limit LIMIT ( samples | seconds ) ] [ ( drops | tx-latency ) ]
#--------------------------------------------------------------------------------
#--------------------------------------------------------------------------------
# show queue-monitor length ( cpu | fabric )
# [ limit LIMIT ( samples | seconds ) ] [ drops ]
#--------------------------------------------------------------------------------

class ShowQueueMonitorLengthCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show queue-monitor length \
             [ ( [ ( INTFS ) ] [ limit LIMIT UNIT ] [ ( drops | tx-latency ) ] ) \
             | ( ( cpu | fabric ) [ limit LIMIT UNIT ] [ drops ] ) ]'

   data = {
      'queue-monitor' : Lanz.nodeQueueMonitor,
      'length' : Lanz.matcherQLen,
      'INTFS' : intfRangeMatcher,
      'limit' : Lanz.matcherLimit,
      'LIMIT' : CliMatcher.IntegerMatcher( 1, 1000000, helpdesc='' ),
      'UNIT' : CliMatcher.EnumMatcher( {
                  'samples' : 'Limit to most recent N samples',
                  'seconds' : 'Limit to samples from last N seconds',
               } ),
      'drops' : Lanz.nodeDrops,
      'tx-latency' : Lanz.nodeTxLatency,
      'cpu' : Lanz.nodeCpu,
      'fabric' : Lanz.nodeFabric,
   }

   cliModel = QueueMonitorLength

   @staticmethod
   def handler( mode, args ):
      intfs = LanzIntfCli.isCpuOrFabric( args ) or args.get( 'INTFS' )

      info = {}
      if 'limit' in args:
         info[ 'limit' ] = args[ 'LIMIT' ]
         info[ 'samplesOrSeconds' ] = args[ 'UNIT' ]

      dropsOrLatency = None
      if 'drops' in args:
         dropsOrLatency = 'drops'
      elif 'tx-latency' in args:
         dropsOrLatency = 'tx-latency'

      return LanzCli.showQMonLength( mode, intfs, info, None, dropsOrLatency )

BasicCli.addShowCommandClass( ShowQueueMonitorLengthCmd )

#--------------------------------------------------------------------------------
# show queue-monitor length csv all
#--------------------------------------------------------------------------------
#--------------------------------------------------------------------------------
# show queue-monitor length csv [ ( interface | range ) ]
#--------------------------------------------------------------------------------
#--------------------------------------------------------------------------------
# show queue-monitor length csv ( cpu | fabric )
#--------------------------------------------------------------------------------

class ShowQueueMonitorLengthCsvAllCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show queue-monitor length csv [ ( all |\
                                             [ INTFS ] |\
                                             ( cpu | fabric ) ) ]'
   data = {
      'queue-monitor' : Lanz.nodeQueueMonitor,
      'length' : Lanz.matcherQLen,
      'csv' : Lanz.matcherCsv,
      'all' : Lanz.nodeAll,
      'INTFS' : intfRangeMatcher,
      'cpu' : Lanz.nodeCpu,
      'fabric' : Lanz.nodeFabric,
   }

   cliModel = QueueMonitorLength

   @staticmethod
   def handler( mode, args ):
      if 'all' in args:
         return LanzCli.showQMonLenCSVAll( mode )

      intfs = LanzIntfCli.isCpuOrFabric( args ) or args.get( 'INTFS' )
      return LanzCli.showQMonLength( mode, intfs, None, csv=True )

BasicCli.addShowCommandClass( ShowQueueMonitorLengthCsvAllCmd )

#--------------------------------------------------------------------------------
# show queue-monitor length ( all | global-buffer | statistics )
#--------------------------------------------------------------------------------
class ShowQueueMonitorLengthAllCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show queue-monitor length ( all | global-buffer | statistics )'
   data = {
      'queue-monitor' : Lanz.nodeQueueMonitor,
      'length' : Lanz.matcherQLen,
      'all' : Lanz.nodeAll,
      'global-buffer' : Lanz.nodeGlobalBuffer,
      'statistics' : Lanz.nodeStatistics,
   }

   cliModel = QueueMonitorLength

   @staticmethod
   def handler( mode, args ):
      if 'all' in args:
         return LanzCli.showQMonLenAll( mode )
      elif 'global-buffer' in args:
         return LanzCli.showQMonLenBuffer( mode )
      else:
         return LanzCli.showQMonLength( mode, None, None, None, showStatistics=True )

BasicCli.addShowCommandClass( ShowQueueMonitorLengthAllCmd )

#--------------------------------------------------------------------------------
# show queue-monitor length status
#--------------------------------------------------------------------------------
class QueueMonitorLengthStatusCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show queue-monitor length status'
   data = {
      'queue-monitor' : Lanz.nodeQueueMonitor,
      'length' : Lanz.matcherQLen,
      'status' : 'Display status',
   }
   cliModel = QueueMonitorLengthStatus

   handler = LanzCli.showQMonLenStatus

BasicCli.addShowCommandClass( QueueMonitorLengthStatusCmd )

#--------------------------------------------------------------------------------
# show queue-monitor streaming clients [ VRF ]
#--------------------------------------------------------------------------------
class ShowQueueMonitorStreamingClientsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show queue-monitor streaming clients [ vrf VRF ]'
   data = {
      'queue-monitor' : Lanz.nodeQueueMonitor,
      'streaming' : Lanz.nodeStreaming,
      'clients' : 'Number of connected streaming clients',
      'vrf' : 'Name of VRF to get streaming clients for',
      'VRF' : VrfCli.VrfNameExprFactory()
   }
   cliModel = QueueMonitorStreamingClients

   handler = LanzCli.showQMonStreamingClients

BasicCli.addShowCommandClass( ShowQueueMonitorStreamingClientsCmd )

#--------------------------------------------------------------------------------
# show queue-monitor streaming ( ( ip access-list [ ipAclName ] )\
#                            | ( ipv6 access-list [ ipv6AclName ] ) )'
#--------------------------------------------------------------------------------

class ShowQueueMonitorStreamingIpAccessListCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show queue-monitor streaming ( ( ip access-list [ IP4ACL ] )\
                                          | ( ipv6 access-list [ IP6ACL ] ) )'
   data = {
      'queue-monitor' : Lanz.nodeQueueMonitor,
      'streaming' : Lanz.nodeStreaming,
      'ip' : AclCli.ipKwForShowServiceAcl,
      'ipv6' : AclCli.ipv6KwForShowServiceAcl,
      'access-list' : AclCli.accessListKwMatcherForServiceAcl,
      'IP4ACL' : AclCli.ipAclNameExpression,
      'IP6ACL' : AclCli.ip6AclNameExpression,
   }
   cliModel = AllAclList

   @staticmethod
   def handler( mode, args ):
      aclType = 'ip' if 'ip' in args else 'ipv6'
      name = args[ '<aclNameExpr>' ]
      return LanzCli.showQMonStreamingAcl( mode, name, aclType )

BasicCli.addShowCommandClass( ShowQueueMonitorStreamingIpAccessListCmd )
