# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

#------------------------------------------------------
# This module implements the following configuartion.
#
# daemon DAEMON_NAME
#       exec <name> [<args>]*
#       option <key> value <v>
#       heartbeat <#> [initialization-period <#>]
#       oom score adjustment <#>
#       [no] shutdown
#       (hidden:) command <daemon executable and args>
#-------------------------------------------------------

# Tell package dependency generator that we depend on our SysdbMountProfile
# pkgdeps: rpmwith %{_libdir}/SysdbMountProfiles/ConfigAgent-DaemonCli

import CliPlugin.TechSupportCli
from CliPlugin import AclCli
from CliPlugin import AclCliModel
from CliPlugin import DaemonAgentModel
import AclCliLib
import AgentDirectory
import BasicCli
import BasicCliUtil
import CliCommand
import CliMatcher
import CliToken.Agent
import CliToken.Clear
import ConfigMount
from CliMode.LauncherDaemonMode import LauncherDaemonMode
import CliSession
from LauncherDaemonConstants import DEFAULT_OOM_SCORE_ADJ
import LauncherLib
import LauncherUtil
import LazyMount
from ProcMgrLib import OOM_NONESSENTIAL_PROC
import ShowCommand
import Tac
import Toggles.DaemonStateBrokerToggleLib as DaemonStateBrokerToggle
from TypeFuture import TacLazyType
import Url

import os
import re
import shlex
import time

MAX_COMMAND_ARGV = 256 # Defined in Launcher::AgentConfig::argv
AGENT_TYPE = TacLazyType( 'GenericAgent::AgentTypeEnum' )

configDir = None
statusDir = None
brokerStatusDir = None
agentConfigCliDir = None
aclConfigDir = None
aclStatusDir = None
aclCheckpointDir = None

# on most linux systems, sysconf('SC_CLK_TCK') is 100 ticks/s.
CLK_TCK_HZ = os.sysconf( os.sysconf_names[ 'SC_CLK_TCK' ] )
SYS_UPTIME_FLD = 0
PROC_START_FLD = 21

# Keywords used in commands
daemonKwMatcherForConfig = CliMatcher.KeywordMatcher( 'daemon',
   helpdesc='Configure a new daemon process' )

daemonKwMatcherForShow = CliMatcher.KeywordMatcher( 'daemon',
   helpdesc='Show daemon process information' )

daemonKwMatcherForClear = CliMatcher.KeywordMatcher( 'daemon',
   helpdesc='Clear daemon process information' )

optionTokenMatcher = CliMatcher.KeywordMatcher( 'option',
      helpdesc='Change options configuration for this agent' )

class DaemonConfigModeBase( BasicCli.ConfigModeBase ):
   name = 'Daemon configuration'

   def __init__( self, parent, session, daemonName ):
      self.daemonName = daemonName
      self.genericAgentCfg = configDir.get( daemonName )
      if not self.genericAgentCfg:
         self.genericAgentCfg = configDir.newEntity( 'GenericAgent::Config',
                                                     daemonName )
         self.genericAgentCfg.agentType = AGENT_TYPE.genericAgent
      self.aclConfig = aclConfigDir.get( daemonName )
      if not self.aclConfig:
         self.aclConfig = aclConfigDir.newEntity( 'Acl::ServiceAclTypeVrfMap',
                                                  daemonName )
         aclCheckpointDir.newEntity( 'Acl::CheckpointStatus', daemonName )

      self.launcherCfg = self._getOrCreateLauncherCfg()

   def callModeConstructorsInOrder( self, parent, session, daemonName,
         daemonModeClass ):
      DaemonConfigModeBase.__init__( self, parent, session, daemonName )
      daemonModeClass.__init__( self, daemonName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def _daemonSessionCommitHandler( self ):
      # If daemon is enabled, we need to trigger launcher to react
      # since it's possible that the enabled state has not changed
      # but some of the attributes do.
      #
      # Get the agent config in Sysdb. Note we cannot use self.launcherCfg
      # since it still points to the config session.
      agentCfg = agentConfigCliDir.agent.get( self.daemonName )
      if agentCfg and agentCfg.stable:
         # trigger launcher to re-evaluate.
         agentCfg.stable = False
         agentCfg.stable = True

   def maybeWarnRestartNeeded( self ):
      if self.session.inConfigSession():
         CliSession.registerSessionOnCommitHandler(
               self.session.entityManager,
               "DAEMON:%s" % self.daemonName,
               lambda m, onSessionCommit: self._daemonSessionCommitHandler() )

      elif self.genericAgentCfg.enabled:
         self.addWarning( "New daemon configuration will only take effect after "
                          "restarting the daemon by doing "
                          "'shutdown/no shutdown'." )

   def _enableLauncherCfg( self ):
      if self.launcherCfg.stable:
         # We already have a stable config, so we need to toggle the
         # stable flag to force Launcher to sync the entire config.
         # Note if we are in a config session the toggle won't work,
         # but that's already handled by the on-commit handler.
         self.launcherCfg.stable = False

      # We're enabling - say we are stable so we get this config copied around!
      self.launcherCfg.stable = True

   def shutdownIs( self, shutdown ):
      if not shutdown and not self.launcherCfg.exe:
         self.addError( "No executable specified, cannot start agent." )
         return

      # Verify the executable and warn if none found. This is not fatal since we do
      # support forward configuration, in which case Launcher will do the wait for
      # the executable to materialize and then test if it is an eossdk application
      # that needs the default sysdb mount profile.
      if not shutdown:
         # warn user if the executable is not found
         fexe = self.launcherCfg.exe
         if any( c in set( '`$&;*|()~{}]' ) for c in fexe ):
            self.addError( "%s contains invalid character" % fexe )
            return
         if not fexe.startswith( '/' ):
            self.addError( "%s is not an absolute path" % fexe )
            return
         if fexe.endswith( '/' ):
            self.addError( "%s looks like a directory" % fexe )
            return
         if not LauncherUtil.isExecutable( fexe ):
            if os.path.isfile( fexe ):
               self.addWarning( "%s is not executable: will start it when " 
                                "it becomes executable" % fexe )
            else:
               self.addWarning( "%s not found: will start it when it exists and " 
                                "becomes executable" % fexe )
               fexe = None
         # If this is an EosSdk app and the default/brute-force sysdb profile has to
         # be used, warn about that sub-optimalness (in case the binary is not yet
         # found, those warning will come by Launcher later in its log file).
         if fexe:
            try:
               isEosSdkApp = LauncherUtil.isEosSdkApp( fexe )
            except OSError as e:
               isEosSdkApp = False
               self.addWarning( "Cannot determine app type: %s" % ( e.strerror ) )

            if isEosSdkApp:
               self.addMessage( "This is an EosSdk application" )
               # We remove any extension from the program being run (e.g., '.py') 
               # and substitute underscores for illegal characters."
               fullAgentName =  LauncherUtil.eossdkTitleName( 
                                   LauncherUtil.eossdkAgentNameFromExe ( fexe ), 
                                   self.daemonName )
               self.addMessage( "Full agent name is '%s'" % fullAgentName )
               if "-" in self.daemonName:
                  self.addError( "EosSdk daemon names cannot contain dashes" )
                  return
               if "-" in os.path.basename( fexe ):
                  self.addError( "EosSdk binary names cannot contain dashes" )
                  return
               profileName = "/usr/lib/SysdbMountProfiles/"
               profileName += os.path.basename(fexe)
               profileName64 = "/usr/lib64/SysdbMountProfiles/"
               profileName64 += os.path.basename(fexe)
               if ( os.path.isfile( profileName ) or
                    os.path.isfile( profileName64 ) ):
                  self.addWarning( "Standard profile file '%s' found. The file "
                    "is not necessary. Please consider deleting it. See https:"
                    "//github.com/aristanetworks/EosSdk/wiki/Quickstart%%3A-Hello"
                    "-World#intermezzo-creating-a-mount-profile" % profileName )
               # check if scripts specify an interpreter, else warn
               with open( fexe, 'rb' ) as f:
                  r = f.read( 5 )
               if not r.startswith( ( b'#!', b'\x7fELF' ) ):
                  self.addWarning( "No interpreter (script without #! on line 1" )

      self.genericAgentCfg.enabled = not shutdown

      # This next 'if' statement is needed because Launcher.py's
      # AgentConfigCliReactor only listens when the agent goes stable,
      # and then deletes the listener. So whenever the agent is
      # shutdown, we have to delete the corresponding launcher config.
      if not shutdown:
         self._enableLauncherCfg()

   def agentOptionIs( self, args ):
      option = args[ 'OPTION' ]
      value = args.get( 'VALUE' )
      if value is None:
         del self.genericAgentCfg.option[ option ]
      else:
         self.genericAgentCfg.option[ option ] = value

   def _getOrCreateLauncherCfg( self ):
      """ Returns the Launcher::AgentConfig config for this daemon. If
      none exists, create one."""
      daemonConfig = agentConfigCliDir.agent.get( self.daemonName )
      if daemonConfig:
         return daemonConfig

      daemonConfig = agentConfigCliDir.agent.newMember( self.daemonName )

      daemonConfig.userDaemon = True
      daemonConfig.heartbeatPeriod = 0
      daemonConfig.useEnvvarForSockId = True
      # A daemon's oom_score_adj can be customized via CLI
      if self.daemonName == "TerminAttr":
         daemonConfig.oomScoreAdj = OOM_NONESSENTIAL_PROC
      else:
         daemonConfig.oomScoreAdj = DEFAULT_OOM_SCORE_ADJ


      # Set up the runnability criteria, so the agent only runs when
      # the genericAgentCfg.enabled is True (thus creating this
      # qualpath in the SysdbPlugin state machine).
      daemonConfig.runnability = ( "runnability", )
      daemonConfig.runnability.qualPath = "daemon/agent/runnability/%s" % (
         self.daemonName )

      for redProto in LauncherUtil.allRedProtoSet:
         daemonConfig.criteria[ redProto ] = LauncherUtil.activeSupervisorRoleName

      return daemonConfig

   def doUnsetDaemonExe( self, args ):
      self.maybeWarnRestartNeeded()

      self.launcherCfg.exe = ''
      for argIndex in self.launcherCfg.argv:
         self.launcherCfg.argv[ argIndex ] = ''
      self.genericAgentCfg.defaultEnabled = False

   def doSetDaemonExecutable( self, args ):
      # Note: careful - shlex.split will read from stdin if passed None.
      executable = args[ 'EXECUTABLE' ]
      arguments = args.get( 'ARGV' )
      argv = shlex.split( arguments ) if arguments else []
      self._doSetDaemonExe( executable, argv, defaultEnabled=False )

   def doSetDaemonCommand( self, args ):
      # In the legacy CLI, we set the agent to run after the user
      # enters 'command XXX/XXX --args'. In the commandList string,
      # the first entry is the executable name
      commandList = args[ 'CMD_LINE' ]
      commandList = shlex.split( commandList )
      commandName = commandList[ 0 ]
      argv = commandList[ 1 : ]
      self._doSetDaemonExe( commandName, argv, defaultEnabled=True )

   def _doSetDaemonExe( self, exe, argv, defaultEnabled=False ):
      self.maybeWarnRestartNeeded()

      if len( argv ) > MAX_COMMAND_ARGV:
         self.addError( "Unable to configure an agent with more than %s "
                        "command line arguments." % MAX_COMMAND_ARGV )
         return

      if not LauncherUtil.isExecutable( exe ):
         self.addWarning( "The executable %s does not exist "
                          "(or is not executable)" % exe )

      self.launcherCfg.exe = exe

      for argIndex in self.launcherCfg.argv:
         self.launcherCfg.argv[ argIndex ] = ''
      for i, arg in enumerate( argv ):
         self.launcherCfg.argv[ i ] = arg

      if defaultEnabled:
         # For backwards compatibility, 'command xxx' starts the agent
         # by default.
         self.genericAgentCfg.defaultEnabled = True
         self.shutdownIs( False )
      else:
         self.genericAgentCfg.defaultEnabled = False

   def enableOptionProxyTransport( self, args ):
      self.maybeWarnRestartNeeded()
      self.launcherCfg.userDaemonAccessTokenEnabled = True

   def disableOptionProxyTransport( self, args ):
      self.maybeWarnRestartNeeded()
      self.launcherCfg.userDaemonAccessTokenEnabled = False


class DaemonConfigMode( LauncherDaemonMode, DaemonConfigModeBase ):
   name = 'Daemon configuration'

   def __init__( self, parent, session, daemonName ):
      self.callModeConstructorsInOrder( parent, session, daemonName,
            LauncherDaemonMode )

class HeartbeatConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'heartbeat INTERVAL [ initialization-period PERIOD ]'
   noOrDefaultSyntax = 'heartbeat ...'
   data = {
         'heartbeat': "Manage EOS Agents' execution properties",
         'INTERVAL': CliMatcher.IntegerMatcher( 0, 300,
                    helpdesc='Seconds a daemon may be inactive before termination' ),
         'initialization-period': 'Seconds a daemon may be inactive during startup',
         'PERIOD': CliMatcher.IntegerMatcher( 0, 300,
                                 helpdesc='Initialization period value in seconds' ),
   }

   @staticmethod
   def handler( mode, args ):
      mode.maybeWarnRestartNeeded()
      cfg = mode.launcherCfg
      cfg.heartbeatPeriod = args.get( 'INTERVAL', 0 )
      cfg.startupGracePeriod = args.get( 'PERIOD', cfg.startupGracePeriodDefault )

   noOrDefaultHandler = handler

DaemonConfigMode.addCommandClass( HeartbeatConfigCmd )

daemonOomKwMatcher = CliMatcher.KeywordMatcher( 'oom',
                                    helpdesc='Configure a daemon OOM parameters' )
daemonScoreKwMatcher = CliMatcher.KeywordMatcher( 'score',
                                    helpdesc='Configure a daemon OOM score related '
                                             'parameters' )
daemonScoreAdjKwMatcher = CliMatcher.KeywordMatcher( 'adjustment',
                                    helpdesc='Configure a daemon oom_score_adj '
                                             'parameter. Please see also '
                                             '"show agent oom scores" command' )

class OomScoreAdjConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'oom score adjustment OOM_SCORE_ADJ'
   noOrDefaultSyntax = 'oom score adjustment'
   data = {
            'oom': daemonOomKwMatcher,
            'score': daemonScoreKwMatcher,
            'adjustment': daemonScoreAdjKwMatcher,
            'OOM_SCORE_ADJ': CliMatcher.IntegerMatcher( -1000, 1000,
                                 helpdesc='new oom_score_adj value for the daemon' ),
          }

   @staticmethod
   def handler( mode, args ):
      # We do not set oom_score_adj in /proc/<pid>/oom_score_adj here. To make sure
      # that is changed, one has to do a shut and no shut for the daemon, just like
      # when changing its heartbeat interval
      mode.maybeWarnRestartNeeded()

      oomScoreAdj = args.get( "OOM_SCORE_ADJ" )
      mode.launcherCfg.oomScoreAdj = oomScoreAdj

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.maybeWarnRestartNeeded()

      mode.launcherCfg.oomScoreAdj = DEFAULT_OOM_SCORE_ADJ

DaemonConfigMode.addCommandClass( OomScoreAdjConfigCmd )

def gotoDaemonConfigMode( mode, args ):
   daemonName = args[ 'DAEMON_NAME' ]
   genericAgentCfg = configDir.get( daemonName )
   if( genericAgentCfg and
       genericAgentCfg.agentType != AGENT_TYPE.genericAgent ):
      mode.addError( 'Unable to create daemon \'%s\' because application '
                     '\'%s\' already exists. Please delete the application'
                     % ( daemonName, daemonName ) )
      return
   childMode = mode.childMode( DaemonConfigMode, daemonName=daemonName )
   mode.session_.gotoChildMode( childMode )
    
def doDeleteDaemonConfig( mode, args ):
   daemonName = args[ 'DAEMON_NAME' ]
   genericAgentCfg = configDir.get( daemonName )
   if( genericAgentCfg and
       genericAgentCfg.agentType != AGENT_TYPE.genericAgent ):
      mode.addError( 'Unable to delete daemon \'%s\' because it is an application'
                     % daemonName )
      return
   del agentConfigCliDir.agent[ daemonName ]
   configDir.deleteEntity( daemonName )
   aclConfigDir.deleteEntity( daemonName )
   aclCheckpointDir.deleteEntity( daemonName )

#--------------------------------------------------------------------------------
# [ no | default ] shutdown
#--------------------------------------------------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': 'Disable the daemon',
   }
   handler = lambda mode, args: DaemonConfigMode.shutdownIs( mode, True )
   noHandler = lambda mode, args: DaemonConfigMode.shutdownIs( mode, False )
   defaultHandler = handler

DaemonConfigMode.addCommandClass( ShutdownCmd )

class ServiceAclCmd( CliCommand.CliCommandClass ):
   syntax = '( ip | ipv6 ) access-group NAME [ in ]'
   noOrDefaultSyntax = '( ip | ipv6 ) access-group [ NAME ] [ in ]'
   data = {
         'ip': AclCli.ipMatcherForConfigIf,
         'ipv6': AclCli.ipv6MatcherForConfigIf,
         'access-group': AclCli.accessGroupKwMatcher,
         'NAME': AclCli.standardIpAclNameMatcher,
         'in': AclCli.inKwMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      aclType = 'ip' if 'ip' in args else 'ipv6'
      AclCliLib.setServiceAclTypeVrfMap(
            mode, mode.aclConfig, args[ 'NAME' ], aclType=aclType )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      aclType = 'ip' if 'ip' in args else 'ipv6'
      AclCliLib.noServiceAclTypeVrfMap(
            mode, mode.aclConfig, args.get( 'NAME' ), aclType=aclType )

DaemonConfigMode.addCommandClass( ServiceAclCmd )

#--------------------------------------------------------------------------------
# [ no | default ] option OPTION value VALUE
#--------------------------------------------------------------------------------
KEYMAXLEN = Tac.Type( "GenericAgent::AgentKey" ).maxLength

class AgentOptionCmd( CliCommand.CliCommandClass ):
   syntax = 'option OPTION value VALUE'
   noOrDefaultSyntax = 'option OPTION ...'
   data = {
      'option': optionTokenMatcher,
      'OPTION': CliMatcher.PatternMatcher( pattern='[A-Za-z0-9_-]{1,%s}' % KEYMAXLEN,
         helpdesc='Name of the configuration option', helpname='WORD' ),
      'value': 'Set the value for this option',
      'VALUE': CliMatcher.StringMatcher( helpdesc='Value of the option' ),
   }
   handler = DaemonConfigMode.agentOptionIs
   noOrDefaultHandler = DaemonConfigMode.agentOptionIs

DaemonConfigMode.addCommandClass( AgentOptionCmd )

# --------------------------------------------------------------------------------
# [ no | default ] option proxy transport grpc
# --------------------------------------------------------------------------------
class AgentOptionProxyCmd( CliCommand.CliCommandClass ):
   syntax = 'option proxy transport grpc'
   noOrDefaultSyntax = 'option proxy ...'
   data = {
      'option': optionTokenMatcher,
      'proxy': 'Configure proxy access to the options configured for this agent',
      'transport': 'Configure the transport type',
      'grpc': 'Configure gRPC as the transport type',
   }
   handler = DaemonConfigMode.enableOptionProxyTransport
   noOrDefaultHandler = DaemonConfigMode.disableOptionProxyTransport

if DaemonStateBrokerToggle.toggleOptionProxyCommandEnabled():
   DaemonConfigMode.addCommandClass( AgentOptionProxyCmd )

#--------------------------------------------------------------------------------
# [ no | default ] daemon DAEMON_NAME
#--------------------------------------------------------------------------------
NAMEMAXLEN = Tac.Type( "GenericAgent::AgentName" ).maxLength
daemonNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: list( configDir ),
      'Daemon name',
      pattern='[A-Za-z0-9_-]{1,%s}' % NAMEMAXLEN )

class DaemonCmd( CliCommand.CliCommandClass ):
   syntax = 'daemon DAEMON_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'daemon': daemonKwMatcherForConfig,
      'DAEMON_NAME': daemonNameMatcher
   }
   handler = gotoDaemonConfigMode
   noOrDefaultHandler = doDeleteDaemonConfig

BasicCli.GlobalConfigMode.addCommandClass( DaemonCmd )

#--------------------------------------------------------------------------------
# [ no | default ] command CMD_LINE
#--------------------------------------------------------------------------------
class AgentCommandCmd( CliCommand.CliCommandClass ):
   syntax = 'command CMD_LINE'
   noOrDefaultSyntax = 'command ...'
   data = {
      'command': 'Specify daemon command line',
      'CMD_LINE': CliMatcher.StringMatcher(
         helpdesc='daemon executable and arguments', helpname='commandLine' ),
   }
   hidden = True # Legacy CLI that starts the executable right away
   handler = DaemonConfigMode.doSetDaemonCommand
   noOrDefaultHandler = DaemonConfigMode.doUnsetDaemonExe

DaemonConfigMode.addCommandClass( AgentCommandCmd )

#--------------------------------------------------------------------------------
# [ no | default ] exec EXECUTABLE [ ARGV ]
#--------------------------------------------------------------------------------
# The new CLI which waits for a 'no shutdown' before running the command
class AgentExecCmd( CliCommand.CliCommandClass ):
   syntax = 'exec EXECUTABLE [ ARGV ]'
   noOrDefaultSyntax = 'exec ...'
   data = {
      'exec': 'Specify the executable and arguments to run when enabled',
      'EXECUTABLE': CliMatcher.PatternMatcher( pattern='.+',
         helpdesc='Name or path of the executable', helpname='WORD' ),
      'ARGV': CliMatcher.StringMatcher(
         helpdesc=( 'Startup command line arguments passed to the executable at '
                    'launch' ),
         helpname='ARGV' ),
   }
   handler = DaemonConfigMode.doSetDaemonExecutable
   noOrDefaultHandler = DaemonConfigMode.doUnsetDaemonExe

DaemonConfigMode.addCommandClass( AgentExecCmd )

def _getAgentFullName( cliConfig ):
   # extract binary name
   exe, _ = os.path.splitext( os.path.basename( cliConfig.exe ) )
   return f"{exe}-{cliConfig.name}"

def _getPidFromBash( agentName ):
   # try to get PID from bash, as some agents (such as TerminAttr) do not show up in
   # AgentDirectory, or we want to display the latest PID if the agent isn't running
   # unfortunately, wildcards in Tac.run() throw an error:
   psCmd = [ "ls", "-rt", "/var/log/agents/" ]
   try:
      output = Tac.run( psCmd, stdout=Tac.CAPTURE, stderr=Tac.DISCARD )
      # filter out this agent's logs (note that agent names can contain dashes or
      # be substrings of another agent, and that rotated files are also candidates
      # since during rotation, for a short time, <agent>-<pid> will be missing.
      filterStr = r"^%s-[0-9]+(\.[-0-9_]+(\.gz)*)*$" % agentName
      lines = [ l for l in output.splitlines() if re.match( filterStr, l ) ]
      if not lines:
         return None
      # extract PID
      pidMatch = re.search( r'(?<=%s-)[0-9]+' % agentName, lines[ -1 ] )
      return int( pidMatch.group() ) if pidMatch else None
   except Tac.SystemCommandError:
      pass
   return None

def _getAgentPid( mode, daemonName ):
   sysname = mode.entityManager.sysname()
   # get CLI config
   cliConfig = agentConfigCliDir.agent.get( daemonName )
   if not cliConfig:
      return None
   fullName = _getAgentFullName( cliConfig )
   # AgentDirectory contains agents that show up in /var/run/agents/,
   # such as EosSdk agents.
   agentInfo = AgentDirectory.agent( sysname, fullName )
   # Any executable which links with libeos or python script that imports eossdk
   # is considered an EosSdk app and the logfile name is name <exe>-<daemon name>,
   isEosSdkApp = False
   try:
      isEosSdkApp = LauncherUtil.isEosSdkApp( cliConfig.exe )
   except OSError:
      pass
   agentName = fullName if isEosSdkApp else cliConfig.name
   pid = agentInfo.get( "pid" ) if agentInfo else _getPidFromBash( agentName )
   # Make sure that pid is actually running (in some cases like TerminAttr we have
   # to get the pid from the logfile name), otherwise return None
   if pid:
      try:
         with BasicCliUtil.RootPrivilege( 0 ):
            os.kill( pid, 0 )
      except ProcessLookupError:
         return None
   return pid

def _tickToSec( ticks ):
   # convert a clock tick value (ticks/s) to seconds by dividing by user Hz
   return ticks // CLK_TCK_HZ

def _getProcessTimes( pid ):
   uptime = 0.0
   starttime = 0.0
   def _getField( fd, n ):
      # get nth field from file descriptor fd
      return fd.read().strip().split()[ n ]
   try:
      with open( '/proc/%s/stat' % pid ) as procFd, \
            open( '/proc/uptime' ) as uptimeFd:
         # 22nd field denotes process start time (in clock ticks) since system boot
         relativeStartTime = _tickToSec(
               float( _getField( procFd, PROC_START_FLD ) ) )
         # 1st field denotes system uptime (in sec)
         sysUptime = float( _getField( uptimeFd, SYS_UPTIME_FLD ) )
         starttime = time.time() - sysUptime + relativeStartTime
         uptime = sysUptime - relativeStartTime
   except OSError:
      pass # couldn't open file, or does not exist
   return uptime, starttime

def _buildAgentModel( mode, daemonName ):
   model = DaemonAgentModel.DaemonAgent()
   agent = configDir.get( daemonName )
   if agent is not None:
      model.enabled = agent.enabled
      for key in agent.option:
         model.option[ key ] = agent.option[ key ]

      status = statusDir.get( daemonName )
      if status is not None:
         # This is an SDK agent!
         model.isSdkAgent = True
         model.running = status.running
         for key, value in status.data.items():
            model.data[ key ] = value
      else:
         model.isSdkAgent = False
         # BUG97279 Determine if the process is actually running
         model.running = agent.enabled
      agentCliConfig = agentConfigCliDir.agent.get( daemonName )
      if agentCliConfig and agentCliConfig.userDaemonAccessTokenEnabled:
         # when "option proxy .." is enabled, SDK status data should be ignored
         proxyStatus = brokerStatusDir.get( daemonName )
         if proxyStatus:
            model.data = dict( proxyStatus.data )
         else:
            model.data = {}

      # get PID and other PID-related information:
      model.pid = _getAgentPid( mode, daemonName )
      model.uptime, model.starttime = _getProcessTimes( model.pid )

   return model

def _buildAgentsModel( mode ):
   model = DaemonAgentModel.DaemonAgents()
   for daemonName in configDir:
      model.daemons[ daemonName ] = _buildAgentModel( mode, daemonName )
   return model

#-------------------------------------------------------------------------------
# show daemon [ DAEMON_NAME ]
#-------------------------------------------------------------------------------
def showDaemonAgent( mode, args ):
   daemonName = args.get( 'DAEMON_NAME' )
   model = DaemonAgentModel.DaemonAgents()
   if daemonName is None:
      for daemonName in configDir:
         model.daemons[ daemonName ] = _buildAgentModel( mode, daemonName )
   else:
      if daemonName not in configDir:
         mode.addError( 'Daemon %s has not been configured.' % daemonName )
         return None
      model.daemons[ daemonName ] = _buildAgentModel( mode, daemonName )
   return model

class ShowDaemon( ShowCommand.ShowCliCommandClass ):
   syntax = 'show daemon [ DAEMON_NAME ]'
   data = {
            'daemon': daemonKwMatcherForShow,
            'DAEMON_NAME': daemonNameMatcher,
          }
   cliModel = DaemonAgentModel.DaemonAgents
   privileged = True
   handler = showDaemonAgent

BasicCli.addShowCommandClass( ShowDaemon )

#-------------------------------------------------------------------------------
# show daemon DAEMON_NAME [ip|ipv6] access-list [<acl>] [summary]
#-------------------------------------------------------------------------------
class ShowDaemonAcl( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show daemon DAEMON_NAME'
              '('
              ' ( ip access-list [ <ipAclName> ] ) | '
              ' ( ipv6 access-list [ <ipv6AclName> ] ) |'
              ' ( access-list [ <ipOrIvp6AclName> ] ) '
              ')' )
   data = {
            'daemon': daemonKwMatcherForShow,
            'DAEMON_NAME': daemonNameMatcher,
            'ip': AclCli.ipKwForShowServiceAcl,
            'ipv6': AclCli.ipv6KwForShowServiceAcl,
            'access-list': AclCli.accessListKwMatcherForServiceAcl,
            '<ipAclName>': AclCli.ipAclNameExpression,
            '<ipv6AclName>': AclCli.ip6AclNameExpression,
            '<ipOrIvp6AclName>': AclCli.ipOrIpv6AclNameExpression,
          }
   cliModel = AclCliModel.AllAclList

   @staticmethod
   def handler( mode, args ):
      if 'ip' in args:
         aclType = 'ip'
      elif 'ipv6' in args:
         aclType = 'ipv6'
      else:
         aclType = None

      daemonName = args.get( 'DAEMON_NAME' )
      aclConfig = aclConfigDir.get( daemonName )
      aclStatus = aclStatusDir.get( daemonName )
      aclCheckpoint = aclCheckpointDir.get( daemonName )
      name = args[ '<aclNameExpr>' ]

      if aclConfig and aclStatus and aclCheckpoint:
         return AclCli.showServiceAcl( mode, aclConfig, aclStatus, aclCheckpoint,
                                       aclType, name )
      mode.addError( 'Daemon %s has not been configured.' % daemonName )
      return None

BasicCli.addShowCommandClass( ShowDaemonAcl )

#----------------------------------------------------------------
# "clear daemon DAEMON_NAME counters [ ip | ipv6 ] access-list"
#----------------------------------------------------------------
class ClearDaemonAcl( CliCommand.CliCommandClass ):
   syntax = 'clear daemon DAEMON_NAME counters [ ip | ipv6 ] access-list'
   data = {
            'clear': CliToken.Clear.clearKwNode,
            'daemon': daemonKwMatcherForClear,
            'DAEMON_NAME': daemonNameMatcher,
            'counters': 'counters',
            'ip': AclCli.ipKwForClearServiceAclMatcher,
            'ipv6': AclCli.ipv6KwMatcherForClearServiceAcl,
            'access-list': AclCli.accessListKwMatcherForServiceAcl
          }

   @staticmethod
   def handler( mode, args ):
      if 'ip' in args:
         aclType = 'ip'
      elif 'ipv6' in args:
         aclType = 'ipv6'
      else:
         aclType = None

      daemonName = args.get( 'DAEMON_NAME' )
      aclStatus = aclStatusDir.get( daemonName )
      aclCheckpoint = aclCheckpointDir.get( daemonName )

      if aclStatus and aclCheckpoint:
         AclCli.clearServiceAclCounters( mode, aclStatus, aclCheckpoint, aclType )

BasicCli.EnableMode.addCommandClass( ClearDaemonAcl )

def fsFunc( fs ):
   return fs.scheme in ( 'flash:', 'file:' ) and fs.supportsRead()

class RunEosAppCmd( CliCommand.CliCommandClass ):
   syntax = 'agent run URL [ heartbeat HEARTBEAT ] [ ARGV ]'
   data = {
         'agent': CliToken.Agent.agentKwForConfig,
         'run': 'Run an EOS application in the foreground (interactive)',
         'URL': Url.UrlMatcher( fsFunc, helpdesc='file name of binary to run',
                                allowAllPaths=True ),
         'heartbeat': 'To take down unresponsive agents',
         'HEARTBEAT': CliMatcher.IntegerMatcher( 0, 300,
                    helpdesc='Seconds a daemon may be inactive before termination' ),
         'ARGV' : CliMatcher.StringMatcher( helpname='ARGV',
                    helpdesc='Startup command line arguments passed to the '
                             'executable at launch' ),
   }

   @staticmethod
   def handler( mode, args ):
      argvList = [ 'Cli', str( args[ 'URL' ].pathname ) ]
      argvList.extend( args.get( 'ARGV', '' ).split() )
      import oneTimeLaunch # pylint: disable=import-outside-toplevel
      oneTimeLaunch.run( args.get( 'HEARTBEAT', 0.0 ), argvList )

BasicCli.EnableMode.addCommandClass( RunEosAppCmd )


#--------------------------------------------------------------
# show tech support
#--------------------------------------------------------------
CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2018-01-20 12:00:00',
   cmds=[ 'show daemon' ],
   summaryCmds=[ 'show daemon' ] )

def Plugin( entityManager ):
   global agentConfigCliDir, configDir, statusDir, brokerStatusDir, aclConfigDir
   global aclStatusDir, aclCheckpointDir

   agentConfigCliDir = ConfigMount.mount( entityManager,
                                          LauncherLib.agentConfigCliDirPath,
                                          "Launcher::AgentConfigDir",
                                          "wi" )
   configDir = ConfigMount.mount( entityManager,
                                  'daemon/agent/config',
                                  'Tac::Dir', 'wi' )
   statusDir = LazyMount.mount( entityManager, 'daemon/agent/status',
                                'Tac::Dir', 'ri' )
   # Daemon status set via the DaemonStateBroker proxy agent instead of EOS SDK
   brokerStatusDir = LazyMount.mount( entityManager, 'daemon/broker/status',
                                      'Tac::Dir', 'ri' )
   aclConfigDir = ConfigMount.mount( entityManager, 'daemon/acl/config',
                                     'Tac::Dir', 'wi' )
   aclStatusDir = LazyMount.mount( entityManager,
                                   'daemon/acl/status',
                                   'Tac::Dir', 'ri' )
   aclCheckpointDir = LazyMount.mount( entityManager,
                                       'daemon/acl/checkpoint',
                                       'Tac::Dir', 'wi' )
