# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# -------------------------------------------------------------------------------
# This module implements MPLS configuration.
# -------------------------------------------------------------------------------
'''Configuration commands supported for MPLS'''

from socket import TCP_MD5SIG_MAXKEYLEN, IPPROTO_IP
import CliPlugin.IntfCli as IntfCli
import CliPlugin.MplsCli as MplsCli
import CliPlugin.TechSupportCli
from CliPlugin import AclCli
from CliPlugin import IpAddrMatcher
from CliPlugin.GmpShowTechCli import registerShowTechMulticast, ShowTechMulticastType
import AclCliLib
import AgentDirectory
import BasicCli
import CliCommand
import CliExtensions
import CliMatcher
from CliMode.LdpMode import LdpMode
import CliToken.Clear
import ConfigMount
from IpLibConsts import DEFAULT_VRF
import LazyMount
import ReversibleSecretCli
import Tac
from Toggles import GmpToggleLib, RcfLibToggleLib
from TypeFuture import TacLazyType
import CliParser

# pylint: disable=ungrouped-imports
from CliMode.LdpMode import (
   GrHelperMode,
   GrSpeakerMode,
)

IpGenAddr = TacLazyType( 'Arnet::IpGenAddr' )
LdpIdentifier = TacLazyType( 'Ldp::LdpIdentifier' )
LdpLabelLocalTerminationMode = TacLazyType( 'Ldp::LdpLabelLocalTerminationMode' )
LdpProtoParam = TacLazyType( 'Ldp::LdpProtoParam' )

aclCheckpoint = None
aclConfig = None
aclCpConfig = None
aclStatus = None
mplsHwCapability = None
mplsRoutingConfig = None
routingVrfInfoDir = None

# Support for multiple entityManager, key'd by sysname
ldpEnabledVrfCollMap = {}
ldpConfigCollMap = {}
ldpProtoConfigCollMap = {}
ldpStatusCollMap = {}

def getLdpEnabledVrfColl( entityManager ):
   return ldpEnabledVrfCollMap[ entityManager.sysname() ]

def getLdpConfigColl( entityManager ):
   return ldpConfigCollMap[ entityManager.sysname() ]

def getLdpProtoConfigColl( entityManager ):
   return ldpProtoConfigCollMap[ entityManager.sysname() ]

def getLdpStatusColl( entityManager ):
   return ldpStatusCollMap[ entityManager.sysname() ]

ldpKw = CliMatcher.KeywordMatcher( 'ldp', helpdesc='LDP configuration' )
ldpForShowKw = ldpKw # Legacy use only
ldpForShowTechKw = CliMatcher.KeywordMatcher( 'ldp',
      helpdesc='Show detailed state of LDP agent' )
mldpKw = CliMatcher.KeywordMatcher( 'mldp', helpdesc='MLDP configuration' )
rcfShowTechKw = CliMatcher.KeywordMatcher( 'rcf',
      helpdesc='Show router control-functions related state' )
syncForShowKw = CliMatcher.KeywordMatcher( 'sync',
      helpdesc='IGP sync configuration' )

__ldpConfigPseudowireCleanupHook = CliExtensions.CliHook()

def getLdpConfigPseudowireCleanupHook():
   '''Call this to add extra cleanup routines for when ldp config is cleared.

   It is used to clean up pseudowire config when "no mpls ldp" is run.
   '''

   return __ldpConfigPseudowireCleanupHook

def showLdpConfigWarnings( mode, brief=False ):
   # Assuming default vrf for now
   vrfName = DEFAULT_VRF

   if not AgentDirectory.agentIsRunning( mode.entityManager.sysname(), 'LdpAgent' ):
      mode.addWarning( "Agent 'LdpAgent' is not running" )
   if not AgentDirectory.agentIsRunning( mode.entityManager.sysname(), 'Mpls' ):
      mode.addWarning( "Agent 'Mpls' is not running" )

   if brief:
      # This is for e.g. the general 'show mpls ldp' command
      return

   config = getLdpConfigColl( mode.entityManager ).config.get( vrfName )
   if ( config is None or not config.enabled ):
      mode.addWarning( "LDP is not enabled" )
   else:
      protoConfig = getLdpProtoConfigColl(
                                 mode.entityManager ).protoConfig.get( vrfName )
      if protoConfig is None:
         mode.addWarning( "LDP is operationally down (internal error)" )
      elif protoConfig.runningState != 'ldpRunning':
         mode.addWarning( "LDP is operationally down: %s" %
                           protoConfig.reason )
   if not mplsRoutingConfig.mplsRouting:
      mode.addWarning( "MPLS routing is not enabled" )
   routingInfo = routingVrfInfoDir.get( DEFAULT_VRF )
   if not ( routingInfo and routingInfo.routing ):
      mode.addWarning( "IP routing is not enabled" )

# ------------------------------------------------------------------------------
# mpls ldp configuration mode
# ------------------------------------------------------------------------------
class LdpConfigMode( LdpMode, BasicCli.ConfigModeBase ):
   name = "Mpls Ldp Configuration"

   def __init__( self, parent, session, vrfName='default' ):
      self.vrfName = vrfName
      LdpMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def enableLdpPassword( self, passwd, no ):
      config = getLdpConfigColl(
                  self.entityManager ).config.newMember( self.vrfName )
      if not no:
         if passwd.clearTextSize() > TCP_MD5SIG_MAXKEYLEN:
            self.addError( "Maximum password length exceeded, Max length=80" )
            return
         # Store encoded password
         config.md5Password = passwd
      else:
         config.md5Password = ReversibleSecretCli.getDefaultSecret()

   def authIndexPasswordIs( self, index=None, password=None, enable=True ):
      config = getLdpConfigColl(
                  self.entityManager ).config.newMember( self.vrfName )
      if enable and password:
         assert index
         if password.clearTextSize() > TCP_MD5SIG_MAXKEYLEN:
            self.addError( "Maximum password length exceeded, Max length=80" )
            return
         # Store encoded password
         config.encodedPassword[ index ] = password
      else:
         if index:
            del config.encodedPassword[ index ]
         else:
            config.encodedPassword.clear()

   def authIndexActiveIs( self, index=None, enable=True ):
      config = getLdpConfigColl(
                  self.entityManager ).config.newMember( self.vrfName )
      if enable:
         assert index
         # Set default password
         config.activePasswordIndex = index
      else:
         config.activePasswordIndex = 0

   def neighAuthIndexActiveIs( self, ipAddr=None, index=None, enable=True ):
      config = getLdpConfigColl(
                  self.entityManager ).config.newMember( self.vrfName )
      if enable:
         assert ipAddr
         assert index
         # Set default password
         config.activePasswordIndexForNeighbor[ IpGenAddr( ipAddr ) ] = index
      else:
         if ipAddr:
            del config.activePasswordIndexForNeighbor[ IpGenAddr( ipAddr ) ]
         else:
            config.activePasswordIndexForNeighbor.clear()

   def ldpLabelLocalTermination( self, args ):
      config = getLdpConfigColl(
                  self.entityManager ).config.newMember( self.vrfName )
      if 'explicit-null' in args:
         if not mplsHwCapability.mplsMultiLabelTerminationSupported:
            self.addError( "Multi label termination not supported on this platform" )
            return
         termMode = LdpLabelLocalTerminationMode.explicitNull
      else:
         termMode = LdpLabelLocalTerminationMode.implicitNull
      config.ldpLabelLocalTerminationMode = termMode

   def ldpLinkReadyTimeout( self, timeout ):
      config = getLdpConfigColl(
                  self.entityManager ).config.newMember( self.vrfName )
      if timeout >= 0:
         config.linkReadyTimeout = timeout
      else:
         config.linkReadyTimeout = config.linkReadyTimeoutDefault

   def ldpLinkReadyDelay( self, delay ):
      config = getLdpConfigColl(
                  self.entityManager ).config.newMember( self.vrfName )
      if delay >= 0:
         config.linkReadyDelay = delay
      else:
         config.linkReadyDelay = config.linkReadyDelayDefault

   def pseudowireAgentDel( self ):
      getLdpConfigPseudowireCleanupHook().notifyExtensions( self )

   def shutdownIs( self, no ):
      config = getLdpConfigColl(
                  self.entityManager ).config.newMember( self.vrfName )
      config.enabled = bool( no )
      if config.enabled:
         getLdpEnabledVrfColl( self.entityManager ).enabledVrf[ self.vrfName ] = True
      else:
         del getLdpEnabledVrfColl( self.entityManager ).enabledVrf[ self.vrfName ]

   def entropyLabelIs( self, args ):
      config = getLdpConfigColl(
                  self.entityManager ).config.newMember( self.vrfName )
      enabled = not CliCommand.isNoOrDefaultCmd( args )
      config.entropyLabel = enabled

   def protoParamIs( self, param ):
      config = getLdpConfigColl(
                  self.entityManager ).config.newMember( self.vrfName )
      config.ldpProtoParamSetting = param

   def routerIdIs( self, no, ipAddr, intfId, force ):
      config = getLdpConfigColl(
                  self.entityManager ).config.newMember( self.vrfName )
      if no:
         config.routerId = '0.0.0.0'
         config.routerIdIntfId = ''
      else:
         config.routerId = ipAddr
         config.routerIdIntfId = intfId
      if force:
         config.forceRouterId = Tac.now()

   def transportAddrIntfIdIs( self, no, intfId ):
      config = getLdpConfigColl(
                  self.entityManager ).config.newMember( self.vrfName )
      if no:
         config.ipTransportAddrIntfId = ''
      else:
         config.ipTransportAddrIntfId = intfId

   def targetedIs( self, ipAddr ):
      config = getLdpConfigColl(
                  self.entityManager ).config.newMember( self.vrfName )
      config.staticTarget[ IpGenAddr( ipAddr ) ] = True

   def targetedDel( self, ipAddr ):
      config = getLdpConfigColl(
                  self.entityManager ).config.newMember( self.vrfName )
      del config.staticTarget[ IpGenAddr( ipAddr ) ]

   def targetedDelAll( self ):
      config = getLdpConfigColl(
                  self.entityManager ).config.newMember( self.vrfName )
      config.staticTarget.clear()

   def fecFilterPrefixListNameIs( self, prefixListName ):
      config = getLdpConfigColl(
                  self.entityManager ).config.newMember( self.vrfName )
      config.fecFilterPrefixListName = prefixListName

   def fecFilterPrefixListNameDel( self ):
      config = getLdpConfigColl(
                  self.entityManager ).config.newMember( self.vrfName )
      config.fecFilterPrefixListName = ""

   def helloHoldTimeIs( self, seconds ):
      config = getLdpConfigColl(
                  self.entityManager ).config.newMember( self.vrfName )
      if config is not None:
         config.helloHoldTime = seconds

   def helloIntervalIs( self, seconds ):
      config = getLdpConfigColl(
                  self.entityManager ).config.newMember( self.vrfName )
      if config is not None:
         config.helloInterval = seconds

   def targetedHelloHoldTimeIs( self, seconds ):
      config = getLdpConfigColl(
                  self.entityManager ).config.newMember( self.vrfName )
      if config is not None:
         config.targetedHelloHoldTime = seconds

   def targetedHelloIntervalIs( self, seconds ):
      config = getLdpConfigColl(
                  self.entityManager ).config.newMember( self.vrfName )
      if config is not None:
         config.targetedHelloInterval = seconds

   def aclNameIs( self, aclName ):
      config = getLdpConfigColl( self.entityManager ).config.get( self.vrfName )
      if config is None:
         config = getLdpConfigColl(
                     self.entityManager ).config.newMember( self.vrfName )
      AclCliLib.setServiceAcl( self, 'ldp', IPPROTO_IP,
                               aclConfig, aclCpConfig, aclName, vrfName=self.vrfName,
                               port=[ LdpProtoParam.ldpTcpPort ], tracked=True )

   def aclNameDel( self ):
      AclCliLib.noServiceAcl( self, 'ldp', aclConfig, aclCpConfig, None,
                              vrfName=self.vrfName )

   def ldpIntfsDisabledIs( self, no ):
      config = getLdpConfigColl(
                  self.entityManager ).config.newMember( self.vrfName )
      config.onlyLdpEnabledIntfs = not no

   def gotoLdpGrHelperMode( self ):
      childMode = self.childMode( LdpGrHelperConfigMode )
      childMode.helperIs()
      self.session_.gotoChildMode( childMode )

   def noLdpGrHelperMode( self ):
      childMode = self.childMode( LdpGrHelperConfigMode )
      childMode.noHelper()
      childMode.noRecovery()
      childMode.noNeighborLiveness()

   def gotoLdpGrSpeakerMode( self ):
      childMode = self.childMode( LdpGrSpeakerConfigMode )
      childMode.speakerIs()
      self.session_.gotoChildMode( childMode )

   def noLdpGrSpeakerMode( self ):
      childMode = self.childMode( LdpGrSpeakerConfigMode )
      childMode.noSpeaker()
      childMode.noStateHolding()
      childMode.noReconnect()

   def helloRedundancyIs( self, enabled ):
      config = getLdpConfigColl(
                  self.entityManager ).config.newMember( self.vrfName )
      if config is not None:
         config.helloRedundancy = enabled

   def helloRedundancyTimeoutIs( self, seconds ):
      config = getLdpConfigColl(
                  self.entityManager ).config.newMember( self.vrfName )
      if config is not None:
         if seconds is not None:
            config.helloRedundancyTimeout = seconds
         else:
            config.helloRedundancyTimeout = \
                  LdpProtoParam.defaultHelloRedundancyTimeout

   def allowLuTunnelRedistIs( self, enabled, rcfFn='' ):
      config = getLdpConfigColl(
                  self.entityManager ).config.newMember( self.vrfName )
      if config is not None:
         if enabled:
            # Set rcfForLuTunnelRedist before enable to prevent churn that
            # would occur otherwise when the feature is enabled along with RCF
            # function.
            config.rcfForLuTunnelRedist = rcfFn
            config.allowLuTunnelRedist = True
         else:
            # Clear rcfForLuTunnelRedist after disable to prevent churn that
            # would occur otherwise when the feature is disabled while RCF
            # function is also set.
            config.allowLuTunnelRedist = False
            config.rcfForLuTunnelRedist = ''

   def endOfLibIs( self, enabled ):
      config = getLdpConfigColl(
                  self.entityManager ).config.newMember( self.vrfName )
      if config is not None:
         config.endOfLib = enabled

   # --- TEST CLI ---
   def ignoreHwUnprogrammedIs( self, enabled ):
      config = getLdpConfigColl(
                  self.entityManager ).config.newMember( self.vrfName )
      if config is not None:
         config.ignoreMplsRouteHwUnprogrammed = enabled

# ---------------------------------------------------------------------------------
# clear mpls ldp counters access-list
# ---------------------------------------------------------------------------------
matcherLdp = CliMatcher.KeywordMatcher( 'ldp',
      helpdesc='Clear LDP information' )

class ClearIpAclCounters( CliCommand.CliCommandClass ):
   syntax = 'clear mpls ldp counters access-list'
   data = { 'clear': CliToken.Clear.clearKwNode,
            'mpls': MplsCli.mplsMatcherForClear,
            'ldp': matcherLdp,
            'access-list': AclCli.accessListKwMatcherForServiceAcl,
            'counters': AclCli.countersKwMatcher }

   @staticmethod
   def handler( mode, args ):
      showLdpConfigWarnings( mode )

      aclType = 'ip'
      vrfName = DEFAULT_VRF
      status = getLdpStatusColl( mode.entityManager ).status.get( vrfName )
      if status is None:
         return

      AclCli.clearServiceAclCounters( mode,
                                      aclStatus,
                                      aclCheckpoint,
                                      aclType )

BasicCli.EnableMode.addCommandClass( ClearIpAclCounters )

# --------------------------------------------------------------------------------
# clear mpls ldp neighbor ( * | NEIGHBOR )
# --------------------------------------------------------------------------------
class ClearMplsLdpNeighborCmd( CliCommand.CliCommandClass ):
   syntax = 'clear mpls ldp neighbor ( * | NEIGHBOR )'
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'mpls': MplsCli.mplsMatcherForClear,
      'ldp': matcherLdp,
      'neighbor': 'Neighbor to clear',
      '*': 'Clear all neighbors',
      'NEIGHBOR': IpAddrMatcher.IpAddrMatcher( helpdesc='LDP router ID' ),
   }

   @staticmethod
   def handler( mode, args ):
      showLdpConfigWarnings( mode )
      vrfName = DEFAULT_VRF

      if '*' in args:
         neighborToDelete = LdpIdentifier.allLdpNeighborIdentifier
      else:
         # We only expect lable space to be 0
         neighborToDelete = args[ 'NEIGHBOR' ] + ':0'

      config = getLdpConfigColl( mode.entityManager ).config.get( vrfName )
      config.ldpNeighborClearRequest[ neighborToDelete ] = Tac.now()

BasicCli.EnableMode.addCommandClass( ClearMplsLdpNeighborCmd )

# ------------------------------------------------------------------------------
# mpls ldp graceful-restart role (helper|speaker) config mode
# Graceful-restart is disabled by default.
# Enabling speaker implies support for helper as well. The role configurations
# are not mutually exclusive, as the helper-specific timer configurations are under
# the helper submode.
# ------------------------------------------------------------------------------
class LdpGrHelperConfigMode( GrHelperMode, BasicCli.ConfigModeBase ):
   name = "Mpls Ldp Graceful Restart Helper Configuration"

   def __init__( self, parent, session, vrfName=DEFAULT_VRF ):
      assert vrfName == DEFAULT_VRF
      self.vrfName = vrfName
      GrHelperMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def helperIs( self ):
      config = getLdpConfigColl(
                  self.entityManager ).config.newMember( self.vrfName )
      flag = Tac.nonConst( config.grOperFlag )
      flag.grFlagHelper = True
      config.grOperFlag = flag

   def noHelper( self ):
      config = getLdpConfigColl( self.entityManager ).config.get( self.vrfName )
      if config:
         flag = Tac.nonConst( config.grOperFlag )
         flag.grFlagHelper = False
         config.grOperFlag = flag

   def recoveryIs( self, timeout ):
      config = getLdpConfigColl( self.entityManager ).config.get( self.vrfName )
      if config:
         config.grMaxRecoveryTimeout = timeout

   def noRecovery( self ):
      self.recoveryIs( LdpProtoParam.defaultGrMaxRecoveryTimeout )

   def neighborLivenessIs( self, timeout ):
      config = getLdpConfigColl( self.entityManager ).config.get( self.vrfName )
      if config:
         config.grNeighborLivenessTimeout = timeout

   def noNeighborLiveness( self ):
      self.neighborLivenessIs( LdpProtoParam.defaultGrNeighborLivenessTimeout )

class LdpGrSpeakerConfigMode( GrSpeakerMode, BasicCli.ConfigModeBase ):
   name = "Mpls Ldp Graceful Restart Speaker Configuration"

   def __init__( self, parent, session, vrfName=DEFAULT_VRF ):
      assert vrfName == DEFAULT_VRF
      self.vrfName = vrfName
      GrSpeakerMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def speakerIs( self ):
      config = getLdpConfigColl(
                  self.entityManager ).config.newMember( self.vrfName )
      flag = Tac.nonConst( config.grOperFlag )
      flag.grFlagSpeaker = True
      config.grOperFlag = flag

   def noSpeaker( self ):
      config = getLdpConfigColl( self.entityManager ).config.get( self.vrfName )
      if config:
         flag = Tac.nonConst( config.grOperFlag )
         flag.grFlagSpeaker = False
         config.grOperFlag = flag

   def stateHoldingIs( self, timeout ):
      config = getLdpConfigColl( self.entityManager ).config[ self.vrfName ]
      if config:
         config.grHoldingTimeout = timeout

   def noStateHolding( self ):
      self.stateHoldingIs( LdpProtoParam.defaultGrHoldingTimeout )

   def reconnectIs( self, timeout ):
      config = getLdpConfigColl( self.entityManager ).config[ self.vrfName ]
      if config:
         config.grReconnectTimeout = timeout

   def noReconnect( self ):
      self.reconnectIs( LdpProtoParam.defaultGrReconnectTimeout )

# -----------------------------------------------------------------------------------
class LdpIntf( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      '''Remove interface level ldp igp sync if it was configured'''
      vrfName = DEFAULT_VRF
      ldpConfig = getLdpConfigColl(
                     self.intf_.mode_.entityManager ).config.get( vrfName )
      if ldpConfig:
         del ldpConfig.intfConfigColl[ self.intf_. name ]

# -----------------------------------------------------------------------------------
# Support for show tech-support
# -----------------------------------------------------------------------------------
# Timestamps are made up to maintain historical order within show tech-support
CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2016-02-10 10:13:37',
   cmds=[ 'show mpls ldp detail',
          'show mpls ldp discovery detail',
          'show mpls ldp neighbor detail',
          'show mpls ldp bindings detail',
          'show mpls ldp tunnel',
          'show mpls ldp bindings mldp detail' ],
   cmdsGuard=lambda: mplsHwCapability.mplsSupported,
   summaryCmds=[ 'show mpls ldp neighbor' ],
   summaryCmdsGuard=lambda: mplsHwCapability.mplsSupported )

CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2022-03-02 17:25:46',
   cmds=[ 'show mpls ldp rcf' ],
   cmdsGuard=lambda: mplsHwCapability.mplsSupported )

if RcfLibToggleLib.toggleRcfShowAgentRcfDetailEnabled():
   CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2024-08-15 17:25:46',
   cmds=[ 'show mpls ldp rcf detail' ],
   cmdsGuard=lambda: mplsHwCapability.mplsSupported )

if GmpToggleLib.toggleMulticastShowTechEnabled():
   def mplsSupportedGuard( mode=None, token=None ):
      if mplsHwCapability.mplsSupported:
         return None
      return CliParser.guardNotThisPlatform

   showTechMplsLdpCmds = [ 'show mpls ldp tunnel mldp' ]
   registerShowTechMulticast( showTechMplsLdpCmds,
                              showTechMcastType=ShowTechMulticastType.noVrf,
                              guard=mplsSupportedGuard )
else:
   CliPlugin.TechSupportCli.registerShowTechSupportCmd(
      '2022-03-14 10:00:00',
      cmds=[ 'show mpls ldp tunnel mldp' ],
      cmdsGuard=lambda: mplsHwCapability.mplsSupported )



# -----------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
# -----------------------------------------------------------------------------------
def Plugin( entityManager ):
   global aclCheckpoint
   global aclConfig
   global aclCpConfig
   global aclStatus
   global mplsHwCapability
   global mplsRoutingConfig
   global routingVrfInfoDir

   aclCheckpoint = LazyMount.mount( entityManager, "acl/checkpoint",
                                   "Acl::CheckpointStatus", "w" )
   aclConfig = ConfigMount.mount( entityManager, "acl/config/cli",
                                  "Acl::Input::Config", "w" )
   aclCpConfig = ConfigMount.mount( entityManager, "acl/cpconfig/cli",
                                  "Acl::Input::CpConfig", "w" )
   aclStatus = LazyMount.mount( entityManager, "acl/status/all",
                                "Acl::Status", "r" )
   ldpConfigCollMap[ entityManager.sysname() ] = \
                        ConfigMount.mount( entityManager, "mpls/ldp/ldpConfigColl",
                                           "Ldp::LdpConfigColl", "w" )
   ldpEnabledVrfCollMap[ entityManager.sysname() ] = \
                        ConfigMount.mount( entityManager,
                                        "mpls/ldp/ldpEnabledVrfColl",
                                        "Ldp::LdpEnabledVrfColl", "w" )
   ldpProtoConfigCollMap[ entityManager.sysname() ] = \
                        LazyMount.mount( entityManager,
                                         "mpls/ldp/ldpProtoConfigColl",
                                         "Ldp::LdpProtoConfigColl", "r" )
   ldpStatusCollMap[ entityManager.sysname() ] = \
                        LazyMount.mount( entityManager, "mpls/ldp/ldpStatusColl",
                                         "Ldp::LdpStatusColl", "r" )
   mplsHwCapability = LazyMount.mount( entityManager,
                                       "routing/hardware/mpls/capability",
                                       "Mpls::Hardware::Capability",
                                       "r" )
   mplsRoutingConfig = LazyMount.mount( entityManager, "routing/mpls/config",
                                 "Mpls::Config", "r" )
   routingVrfInfoDir = LazyMount.mount( entityManager,
                                    "routing/vrf/routingInfo/status",
                                    "Tac::Dir", "ri" )
   IntfCli.Intf.registerDependentClass( LdpIntf )
