# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliMatcher
import ShowCommand
import Tac
from TypeFuture import TacLazyType

from CliPlugin.LineSystemConfigCli import portRangeMatcher
from CliPlugin.XcvrShowDomModel import InterfacesTransceiverDom
from CliPlugin.XcvrShowStatusModel import InterfacesTransceiverStatus

# -------------------------------------------------------------------------------
# TAC Types used in commands
# -------------------------------------------------------------------------------
MediaType = TacLazyType( 'Xcvr::MediaType' )

# -------------------------------------------------------------------------------
# Keywords used in the CLI commands
# -------------------------------------------------------------------------------
lineKeyword = CliMatcher.KeywordMatcher( 'line',
                                         helpdesc='Show line system information' )
systemKeyword = CliMatcher.KeywordMatcher( 'system',
                                         helpdesc='Show line system information' )
portKeyword = CliMatcher.KeywordMatcher( 'port',
                                         helpdesc='Number of the port to use' )

# -------------------------------------------------------------------------------
# show line system [PORT X] ...   [ Base class definition ]
# -------------------------------------------------------------------------------
class ShowLineSystemCommand( ShowCommand.ShowCliCommandClass ):
   baseSyntax_ = 'show line system [ PORTRANGE ]'
   baseData_ = {
      'line': lineKeyword,
      'system': systemKeyword,
      'PORTRANGE': portRangeMatcher
   }
   data = {}
   dynamicSyntax_ = True

   @classmethod
   def _generateSyntaxAndData( cls ):
      cls._assert( cls.syntax.startswith( "show line system " ),
                   "must define syntax with 'show line system ...'" )
      for k in cls.baseData_:
         cls._assert( k not in cls.data, f"'{k}' should not be defined in data" )
      cls.data.update( cls.baseData_ )

# -------------------------------------------------------------------------------
# show line system [PORT X] dom [thresholds]
# -------------------------------------------------------------------------------

class ShowLineSystemDomCommand( ShowLineSystemCommand ):
   syntax = 'show line system [ PORTRANGE ] dom [ thresholds ]'
   data = {
      'dom': ( 'Show transceiver Digital Optical Monitoring (DOM) '
               'parameters for the port' ),
      'thresholds': ( 'Show transceiver Digital Optical Monitoring (DOM) '
                      'parameters with thresholds for the interface' )
   }

   handler = "LineSystemShowDomHandler.showLineSystemDom"
   cliModel = InterfacesTransceiverDom

BasicCli.addShowCommandClass( ShowLineSystemDomCommand )

# -------------------------------------------------------------------------------
# show line system [PORT X] idprom
# -------------------------------------------------------------------------------

class ShowLineSystemIdprom( ShowLineSystemCommand ):
   syntax = "show line system [ PORTRANGE ] idprom"
   data = {
      "idprom": "Show line system IDPROM for the port",
   }

   handler = "LineSystemShowIdpromHandler.showLineSystemIdpromHandler"
   cliModel = "LineSystemShowIdpromModel.LineSystemIdpromBase"

BasicCli.addShowCommandClass( ShowLineSystemIdprom )

# -------------------------------------------------------------------------------
# show line system [PORT X] eeprom
# -------------------------------------------------------------------------------

class ShowLineSystemEeprom( ShowLineSystemCommand ):
   syntax = "show line system [ PORTRANGE ] eeprom"
   data = {
      "eeprom": "Show decoded dump of supported EEPROM pages",
   }

   handler = "LineSystemShowEepromHandler.showLineSystemEepromHandler"
   cliModel = "LineSystemShowEepromModel.LineSystemEepromBase"

BasicCli.addShowCommandClass( ShowLineSystemEeprom )

# -------------------------------------------------------------------------------
# show line system [PORT X] status
# -------------------------------------------------------------------------------

class ShowLineSystemStatusCommand( ShowLineSystemCommand ):
   syntax = 'show line system [ PORTRANGE ] status'
   data = {
      'status': ( 'Show transceiver status information' ),
   }

   handler = "LineSystemShowStatusHandler.showLineSystemStatus"
   cliModel = InterfacesTransceiverStatus

BasicCli.addShowCommandClass( ShowLineSystemStatusCommand )
