#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
# pylint: disable-next=consider-using-from-import
import CliPlugin.LldpConfigCli as LldpConfigCli
from CliPlugin.IntfCli import Intf
from CliPlugin.VrfCli import VrfExprFactory
import LldpConstants
from Toggles import LldpToggleLib

matcherLldp = CliMatcher.KeywordMatcher( 'lldp',
                  helpdesc='Set Link Layer Discovery Protocol (LLDP) parameters' )

#------------------------------------------------------------------------------------
# [ no | default ] lldp hold-time SECONDS
#------------------------------------------------------------------------------------
class LldpHoldTimeCmd( CliCommand.CliCommandClass ):
   syntax = 'lldp ( hold-time | holdtime ) SECONDS'
   noOrDefaultSyntax = 'lldp ( hold-time | holdtime ) ...'
   data = {
      'lldp' : matcherLldp,
      'hold-time' : 'Configure the holdtime to be included in transmitted LLDPDUs',
      'holdtime' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'holdtime',
         helpdesc='Configure the holdtime to be included in transmitted LLDPDUs' ),
                                  deprecatedByCmd='lldp hold-time SECONDS' ),
      'SECONDS' : CliMatcher.IntegerMatcher( 3, 65535,
                                            helpdesc='Holdtime (in seconds)' ),
   }

   handler = LldpConfigCli.setLldpHoldtime
   noOrDefaultHandler = LldpConfigCli.noLldpHoldtime

BasicCli.GlobalConfigMode.addCommandClass( LldpHoldTimeCmd )

#------------------------------------------------------------------------------------
# [ no | default ] lldp management-address ( all | [ ipv6 ] INTF | vrf VRF )
#------------------------------------------------------------------------------------
class LldpManagementAddressAllCmd( CliCommand.CliCommandClass ):
   syntax = 'lldp management-address ( all | ( [ ipv6 ] INTF ) | VRF )'
   noOrDefaultSyntax = 'lldp management-address [ ipv6 | VRF ] ...'
   data = {
      'lldp' : matcherLldp,
      'management-address' : ( 'Configure which TLV for the switch management '
                               'addresss are included in transmitted LLDPDUs' ),
      'all' : 'Send all management addresses in LLDPDU',
      'ipv6' : 'IPv6 related',
      'INTF' : Intf.matcherWithIpSupport,
      'VRF' : VrfExprFactory( helpdesc='Configure the source VRF for '
                                       'the management addresses',
                              inclDefaultVrf=True ),
   }

   @staticmethod
   def handler( mode, args ):
      if 'all' in args:
         return LldpConfigCli.setLldpMgmtAddrTxOptionAll( mode )
      if 'ipv6' in args:
         return LldpConfigCli.setLldpIp6MgmtAddrIntfName( mode, args.get( 'INTF' ) )
      if 'INTF' in args:
         return LldpConfigCli.setLldpMgmtAddrTxOptionIntfName( mode,
                                                               args.get( 'INTF' ) )
      if 'VRF' in args:
         return LldpConfigCli.setLldpMgmtAddrVrf( mode, args.get( 'VRF' ) )
      return None

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'ipv6' in args:
         return LldpConfigCli.noLldpIp6MgmtAddrIntfName( mode )
      if 'VRF' in args:
         return LldpConfigCli.noLldpMgmtAddrVrf( mode, args.get( 'VRF' ) )
      return LldpConfigCli.noLldpMgmtAddrTxOption( mode )

BasicCli.GlobalConfigMode.addCommandClass( LldpManagementAddressAllCmd )

#------------------------------------------------------------------------------------
# [ no | default ] lldp run
#------------------------------------------------------------------------------------
class LldpRunCmd( CliCommand.CliCommandClass ):
   syntax = 'lldp run'
   noOrDefaultSyntax = 'lldp run ...'
   data = {
      'lldp' : matcherLldp,
      'run' : 'Enable LLDP',
   }

   @staticmethod
   def handler( mode, args ):
      # Enabled by default.
      LldpConfigCli.enableLldp( mode, not CliCommand.isNoCmd( args ) )

   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( LldpRunCmd )

#------------------------------------------------------------------------------------
# [ no | default ] lldp timer SECONDS
# [ no | default ] lldp (reinit | timer reinitialization) SECONDS
#------------------------------------------------------------------------------------
class LldpTimerCmd( CliCommand.CliCommandClass ):
   syntax = ( 'lldp ( ( reinit RSECONDS ) | '
                    ' ( timer TSECONDS ) ) | '
                    ' ( timer reinitialization RSECONDS )' )
   noOrDefaultSyntax = ( 'lldp ( ( reinit ... ) | '
                               ' ( timer [reinitialization] ... ) )' )
   data = {
      'lldp' : 'Set Link Layer Discovery Protocol (LLDP) parameters',
      'timer' : 'Configure LLDP timers',
      'TSECONDS' : CliMatcher.IntegerMatcher( 1, 32768, helpdesc='Configure the ' +
                                'period at which we transmit LLDPDUs (in seconds)' ),
      'reinitialization' : 'Configure the delay before LLDP re-initialization is ' +
      'attempted',
      'reinit' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'reinit',
                                     helpdesc='Configure the delay before LLDP ' +
                                              're-initialization is attempted' ),
                                  deprecatedByCmd='lldp timer reinitialization' ),
      'RSECONDS' : CliMatcher.IntegerMatcher( 1, 10, helpdesc='Reinitialization ' +
                                                            'delay (in seconds)' ),
   }

   @staticmethod
   def handler( mode, args ):
      if 'reinitialization' in args or 'reinit' in args:
         LldpConfigCli.setLldpReinitDelay( mode, args )
      else:
         LldpConfigCli.setLldpTimer( mode, args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'reinitialization' in args or 'reinit' in args:
         LldpConfigCli.noLldpReinitDelay( mode, args )
      else:
         LldpConfigCli.noLldpTimer( mode, args )

BasicCli.GlobalConfigMode.addCommandClass( LldpTimerCmd )

#------------------------------------------------------------------------------------
# [ no | default ] lldp tlv transmit ( port-description |
#                                      system-capabilities |
#                                      management-address |
#                                      port-vlan |
#                                      link-aggregation-ieee802.1 |
#                                      link-aggregation |
#                                      max-frame-size |
#                                      power-via-mdi |
#                                      med network-policy
#                                      med serial-number )
#------------------------------------------------------------------------------------
tlvKeywords = {
   'port-description' : 'TLV for the port description string',
   'system-capabilities' : 'TLV for advertising system capabilities',
   'management-address' : 'TLV for the switch management address',
   'port-vlan' : 'Port VLAN ID TLV',
   'link-aggregation-ieee802.1' : 'Link Aggregation IEEE802.1 TLV',
   'link-aggregation' : 'Link Aggregation IEEE802.3 TLV',
   'max-frame-size' : 'Maximum Frame Size TLV',
   'power-via-mdi' : 'Power Via MDI TLV (IEEE Std 802.3at)',
   'vlan-name' : 'VLAN Name TLVs (IEEE Std 802.1q)',
}

def getMedTlvKeyword():
   medTlvKeywords = {
      'network-policy' : 'Network Policy TLV (ANSI/TIA-1057)',
   }
   if LldpToggleLib.toggleLldpSerialNumberTlvEnabled():
      medTlvKeywords.update(
         { 'serial-number' : 'Serial Number TLV (ANSI/TIA-1057)' } )

   return medTlvKeywords

class LldpTlvTransmitCmd( CliCommand.CliCommandClass ):
   syntax = 'lldp ( ( tlv transmit ) | tlv-select ) ( TLV | ( med MEDTLV ) )'
   noOrDefaultSyntax = syntax

   data = {
      'lldp' : 'Set Link Layer Discovery Protocol (LLDP) parameters',
      'tlv' : 'Select which TLVs are included in transmitted LLDPDUs',
      'transmit' : 'Select which TLVs are included in transmitted LLDPDUs',
      'tlv-select' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
                  'tlv-select',
                  helpdesc='Select which TLVs are included in transmitted LLDPDUs' ),
                  deprecatedByCmd='lldp tlv transmit' ),
      'TLV' : CliMatcher.EnumMatcher( tlvKeywords ),
      'med' : 'LLDP-MED TLVs (ANSI/TIA-1057)',
      'MEDTLV' : CliMatcher.EnumMatcher( getMedTlvKeyword() )
   }

   @staticmethod
   def handler( mode, args ):
      if 'med' in args:
         LldpConfigCli.enableMedLldpTlv( mode, args.get( 'MEDTLV' ),
                                         enabled=True )
      else:
         LldpConfigCli.enableLldpTlv( mode, args.get( 'TLV' ),
                                      enabled=True )

   @staticmethod
   def noHandler( mode, args ):
      if 'med' in args:
         LldpConfigCli.enableMedLldpTlv( mode, args.get( 'MEDTLV' ),
                                         enabled=False )
      else:
         LldpConfigCli.enableLldpTlv( mode, args.get( 'TLV' ),
                                      enabled=False )

   @staticmethod
   def defaultHandler( mode, args ):
      if args.get( 'TLV' ) in LldpConstants.defaultDisabledTlvs:
         LldpTlvTransmitCmd.noHandler( mode, args )
      else:
         LldpTlvTransmitCmd.handler( mode, args )

BasicCli.GlobalConfigMode.addCommandClass( LldpTlvTransmitCmd )

asciiTlvMatcher = CliMatcher.StringMatcher( pattern=r'.{1,255}',
                                           helpdesc='An alpha-numeric string value' )

tlvWithValueKeywords = {
   'system-name' : 'TLV for the system name string',
   'system-description' : 'TLV for the system description string'
}

# [ no | default ] lldp tlv transmit ( system-name [SYSTEM-NAME] |
#                                      system-description [SYSTEM-DESCRIPTION] )
class LldptlvTransmitWithValueCmd( CliCommand.CliCommandClass ):
   """
   no lldp tlv transmit TLV system-name VALUE
   -> restores default system-name value
   And
   no lldp tlv transmit TLV system-name
   -> disables sending that TLV ( regardless if VALUE was set or not )


   While both
   default lldp tlv transmit TLV system-name
   default lldp tlv transmit TLV system-name VALUE
   -> makes us send default TLV value
   And
   lldp tlv transmit TLV system-name VALUE
   -> make us send VALUE for TLV
   """
   syntax = 'lldp tlv transmit TLV [ VALUE ]'
   noOrDefaultSyntax = syntax

   data = {
      'lldp' : 'Set Link Layer Discovery Protocol (LLDP) parameters',
      'tlv' : 'Select which TLVs are included in transmitted LLDPDUs',
      'transmit' : 'Select which TLVs are included in transmitted LLDPDUs',
      'TLV' : CliMatcher.EnumMatcher( tlvWithValueKeywords ),
      'VALUE' : asciiTlvMatcher
   }

   @staticmethod
   def handler( mode, args ):
      LldpConfigCli.enableLldpTlv( mode, args.get( 'TLV' ),
                                   value=args.get( 'VALUE' ) )

   @staticmethod
   def noHandler( mode, args ):
      value = args.get( 'VALUE' )
      if value:
         # we are removing 'VALUE' -> restore default TLV value
         LldptlvTransmitWithValueCmd.defaultHandler( mode, args )
      else:
         LldpConfigCli.enableLldpTlv( mode, args.get( 'TLV' ), enabled=False )

   @staticmethod
   def defaultHandler( mode, args ):
      LldpConfigCli.enableLldpTlv( mode, args.get( 'TLV' ) )

BasicCli.GlobalConfigMode.addCommandClass( LldptlvTransmitWithValueCmd )

class LldpReceivePacketTaggedDrop( CliCommand.CliCommandClass ):
   syntax = 'lldp receive packet tagged drop'
   noOrDefaultSyntax = syntax

   data = {
      'lldp' : 'Set Link Layer Discovery Protocol (LLDP) parameters',
      'receive' : 'Set receive parameters',
      'packet' : 'Set packet parameters',
      'tagged' : 'Set action for 802.1Q-tagged packets',
      'drop' : 'Drop 802.1Q-tagged packets'
   }

   handler = LldpConfigCli.setLldpReceivePacketTaggedDrop

   noOrDefaultHandler = LldpConfigCli.noLldpReceivePacketTaggedDrop

BasicCli.GlobalConfigMode.addCommandClass( LldpReceivePacketTaggedDrop )
