#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliMatcher
import CliCommand
# pylint: disable-next=consider-using-from-import
import CliPlugin.LldpConfigCli as LldpConfigCli

matcherLldp = CliMatcher.KeywordMatcher( 'lldp',
                     helpdesc='Set Link Layer Discovery Protocol (LLDP) parameters' )
matcherTlv = CliMatcher.KeywordMatcher(
      'tlv', helpdesc='Select which TLVs are included in transmitted LLDPDUs' )
matcherTransmit = CliMatcher.KeywordMatcher(
      'transmit', helpdesc='Select which TLVs are included in transmitted LLDPDUs' )

applicationTypes = {
   'default' : 'default network policy for non specified applications',
   'video-conferencing' : 'streaming-video (6) application type',
   'streaming-video' : 'streaming-video (7) application type',
   'video-signaling' : 'streaming-video (8) application type',
}

#------------------------------------------------------------------------------------
# ( no | default ) lldp med network-policy
#------------------------------------------------------------------------------------
class LldpMedNetworkPolicyCmd( CliCommand.CliCommandClass ):
   syntax = ( 'lldp med network-policy application ( APP | APPID )' +
                                  ' [ dot1q-tag VLANID [cos COS ] ] [ dscp DSCP ]' )

   noOrDefaultSyntax = 'lldp med network-policy [ application ] [ APP | APPID ] ...'
   data = {
      'lldp' : matcherLldp,
      'med' : 'Set Media Endpoint Discovery (LLDP-MED) parameters',
      'network-policy' : 'Set Network Policy to send to Media Enpoint',
      'application' : 'Specify the application type',
      'APP' : CliMatcher.EnumMatcher( applicationTypes ),
      'APPID' : CliMatcher.IntegerMatcher( 6, 255, helpdesc='Specify application ' +
                                                           'by numeric ID' ),
      'dot1q-tag' : 'Set Media Endpoint framing',
      'VLANID' : CliMatcher.IntegerMatcher( 0, 4094,
         helpdesc='Identifier for a Virtual LAN' ),
      'cos' : 'Set CoS of Media Endpoint framing',
      'COS' : CliMatcher.IntegerMatcher( 0, 7, helpdesc='CoS value' ),
      'dscp' : 'Set DSCP of Media Endpoint datagrams',
      'DSCP' : CliMatcher.IntegerMatcher( 0, 63, helpdesc='DSCP value' ),
   }

   handler = LldpConfigCli.configNetworkPolicy

   noOrDefaultHandler = LldpConfigCli.noNetworkPolicy

LldpConfigCli.LldpIntfModelet.addCommandClass( LldpMedNetworkPolicyCmd )

#------------------------------------------------------------------------------------
# [ no | default ] lldp receive
#------------------------------------------------------------------------------------
class LldpReceiveCmd( CliCommand.CliCommandClass ):
   syntax = 'lldp receive'
   noOrDefaultSyntax = 'lldp receive ...'
   data = {
      'lldp' : matcherLldp,
      'receive' : 'Enable LLDP receive on this interface',
   }

   handler = LldpConfigCli.enableLldpReceive
   noOrDefaultHandler = LldpConfigCli.enableLldpReceive

LldpConfigCli.LldpIntfModelet.addCommandClass( LldpReceiveCmd )

#------------------------------------------------------------------------------------
# [ no | default ] lldp transmit
#------------------------------------------------------------------------------------
class LldpTransmitCmd( CliCommand.CliCommandClass ):
   syntax = 'lldp transmit'
   noOrDefaultSyntax = 'lldp transmit ...'
   data = {
      'lldp' : 'Set Link Layer Discovery Protocol (LLDP) parameters',
      'transmit' : 'Enable LLDP transmit on this interface',
   }

   handler = LldpConfigCli.enableLldpTransmit
   noOrDefaultHandler = LldpConfigCli.enableLldpTransmit

LldpConfigCli.LldpIntfModelet.addCommandClass( LldpTransmitCmd )

#------------------------------------------------------------------------------------
# [ no | default ]
# lldp tlv transmit power-via-mdi fallback allocated-power
#      < max-requested-power | XX.XN [ watts ] >
#------------------------------------------------------------------------------------
POWER_RANGE = ( 0.1, 6553.5 )
class LldpTlvTransmitPowerCmd( CliCommand.CliCommandClass ):
   syntax = ( 'lldp tlv transmit power-via-mdi fallback allocated-power '
              '( max-requested-power | ( POWER [watts] ) )' )
   noOrDefaultSyntax = ( 'lldp tlv transmit power-via-mdi fallback allocated-power '
              '[ max-requested-power | ( POWER [watts] ) ]' )
   data = {
      'lldp' : matcherLldp,
      'tlv' : matcherTlv,
      'transmit' : matcherTransmit,
      'power-via-mdi' : 'Power Via MDI TLV (IEEE Std 802.3at)',
      'fallback' : 'Alternative for non powered ports only',
      'allocated-power': 'Granted power by the PSE to the PD',
      'max-requested-power': ( 'Set the allocated-power to the greatest of all '
                               'requested-power on the port' ),
      'POWER': CliMatcher.FloatMatcher( *POWER_RANGE, precisionString='%.0f',
                                        helpdesc="allocated-power value in watts" ),
      'watts': 'The power value is in watts [default]',
   }
   handler = LldpConfigCli.setPoeAllocatedPowerFallback
   noOrDefaultHandler = handler

LldpConfigCli.LldpIntfModelet.addCommandClass( LldpTlvTransmitPowerCmd )

#------------------------------------------------------------------------------------
# [ no | default ]
# lldp tlv transmit ztp vlan <vlan-id>
#------------------------------------------------------------------------------------
class LldpTlvTransmitZtpVlanCmd( CliCommand.CliCommandClass ):
   syntax = ( 'lldp tlv transmit ztp vlan VLANID' )
   noOrDefaultSyntax = syntax
   data = {
         'lldp' : matcherLldp,
         'tlv' : matcherTlv,
         'transmit' : matcherTransmit,
         'ztp' : 'TLV for ZTP boot VLAN',
         'vlan' : 'Configure boot VLAN',
         'VLANID' : CliMatcher.IntegerMatcher( 2, 4094,
                     helpdesc='Identifier for a Virtual LAN' )
   }
   handler = LldpConfigCli.setZtpBootVlan
   noOrDefaultHandler = handler

LldpConfigCli.LldpIntfModelet.addCommandClass( LldpTlvTransmitZtpVlanCmd )
