# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Model, Str, Int, Dict, List, Enum, Float, Bool, Submodel
from IntfModels import Interface

import CliPlugin.IntfCli
import Tac
import TableOutput
import Ark
import IanaAddressFamilyNumbers, LldpConstants
from Arnet import sortIntf
from EnvironmentTypes import PortStatusTacTypes as EnvPwrPortStatus
from LldpTypes import escapedString

prefix = '  '
printPowerFormat = prefix + prefix + '%-28.28s : %s'
btStr = "Lldp::PowerViaMdiInfo::Bt::"
envPwr = EnvPwrPortStatus.namespace + "::"

firstColumnFormat = TableOutput.Format( justify="left" )
firstColumnFormat.noPadLeftIs( True )
defaultColumnFormat = TableOutput.Format( justify="left" )

_tlvDisplayFormat = { 'portDescription' : 'Port Description',
   'systemName' : 'System Name',
   'systemDescription' : 'System Description',
   'systemCapabilities' : 'System Capabilities',
   'managementAddress' : 'Management Address',
   'portVlan' : 'IEEE802.1 Port VLAN ID',
   'portVlanName' : 'IEEE802.1 Port VLAN Name',
   'linkAggregation8021' : 'IEEE802.1 Link Aggregation',
   'linkAggregation8023' : 'IEEE802.3 Link Aggregation',
   'maxFrameSize' : 'IEEE802.3 Maximum Frame Size',
   'powerViaMdi' : 'IEEE802.3 Power Via MDI',
   'medCapabilities' : 'LLDP-MED Capabilities',
   'medNetworkPolicy' : 'LLDP-MED Network Policy',
   'medLocationIdentification' : 'LLDP-MED Location Identification',
   'serialNumber' : 'LLDP-MED Serial Number TLV' }

chassisIdSubtypeMap = {
   'reserved' : 'cidReserved',
   'chassisComponent' : 'cidChassisComponent',
   'interfaceAlias' : 'cidInterfaceAlias',
   'portComponent' : 'cidPortComponent',
   'macAddress' : 'cidMacAddress',
   'networkAddress' : 'cidNetworkAddress',
   'interfaceName' : 'cidInterfaceName',
   'local' : 'cidLocal',
   'unknown' : 'cidUnknown',
}

chassisIdSubtypeStrs = {
   'reserved' : 'Chassis reserved',
   'chassisComponent' : 'Chassis component',
   'interfaceAlias' : 'Interface alias',
   'portComponent' : 'Port component',
   'macAddress' : 'MAC address',
   'networkAddress' : 'Network address',
   'interfaceName' : 'Interface name',
   'local' : 'Locally assigned',
   'unknown' : 'Unknown',
}

chassisIdSubtypeValues = list( chassisIdSubtypeStrs )

portIdSubtypeMap = {
   'reserved' : 'pidReserved',
   'interfaceAlias' : 'pidInterfaceAlias',
   'portComponent' : 'pidPortComponent',
   'macAddress' : 'pidMacAddress',
   'networkAddress' : 'pidNetworkAddress',
   'interfaceName' : 'pidInterfaceName',
   'agentCircuitId' : 'pidAgentCircuitId',
   'local' : 'pidLocal',
   'unknown' : 'pidUnknown',
}

portIdSubtypeStrs = {
   'reserved' : 'Port reserved',
   'interfaceAlias' : 'Interface alias',
   'portComponent' : 'Port component',
   'macAddress' : 'MAC address',
   'networkAddress' : 'Network address',
   'interfaceName' : 'Interface name',
   'agentCircuitId' : 'Agent circuit ID',
   'local' : 'Locally assigned',
   'unknown' : 'Unknown',
}

portIdSubtypeValues = list( portIdSubtypeStrs )

linkAggregationType = {
   'notCapable' : 'Not Capable',
   'capableAndEnabled' : 'Capable, Enabled',
   'capableAndDisabled' : 'Capable, Disabled',
}

linkAggregationTypeValues = list( linkAggregationType )

linkAggrStatusValues = {
   'notCapable' : 0,
   'capableAndDisabled' : 1,
   'capableAndEnabled' : 3,
}

autoNegCapabilityType = {
   'notCapable' : 'Not Supported',
   'capableAndEnabled' : 'Supported, Enabled',
   'capableAndDisabled' : 'Supported, Disabled',
}

autoNegCapabilityTypeValues = list( autoNegCapabilityType )

ifSubtypeStrs = {
   'badType' : 'Unknown',
   'unknown' : 'Unknown',
   'ifIndex' : 'ifIndex',
   'systemPortNumber' : 'System Port Number',
}

mgmtSubtypeUnknown = 'unknown'
managementAddressSubTypeValues = list(
                                 IanaAddressFamilyNumbers.addrFamilyNames.values() )
managementAddressSubTypeValues = [ addrSubtype.lower() for addrSubtype in
                                   managementAddressSubTypeValues ]
managementAddressSubTypeValues += [ mgmtSubtypeUnknown ]

manAddrSubtypes = {
   'ethernet' : 'Ethernet',
   'ipv4' : 'IPv4',
   'ipv6' : 'IPv6',
   'unknown' : 'Unknown',
}

powerPairsType = {
   'signal' : '1',
   'spare'  : '2',
}

powerPairsTypeUnknown = 'unknown'
powerPairsTypeValues = list( powerPairsType ) + [ powerPairsTypeUnknown ]

def powerPortClassToStr( portClass ):
   if portClass == 'pClassPSE':
      s = 'PSE'
   elif portClass == 'pClassPD':
      s = 'PD'
   else:
      s = 'unknown'
   return s

def powerClassToStr( powerClass ):
   # pylint: disable=consider-using-min-builtin
   # This is in-line with Table 79-3b-Power class field
   if powerClass > 5:
      # class 5 & class 6 is mapped to class 4
      powerClass = 5
   if powerClass in [ 1, 2, 3, 4, 5 ]:
      s = 'class %d' % ( powerClass - 1 )
   else:
      s = 'unknown (%d)' % powerClass
   return s.capitalize()

#For IEEE802.3 Power Via MDI
_powerTypeMapping = { 'type1PseDevice' : 'Type 1 PSE',
                      'type1PdDevice' : 'Type 1 PD',
                      'type2PseDevice' : 'Type 2 PSE',
                      'type2PdDevice' : 'Type 2 PD' }

def powerTypeToStr( powerType ):
   return _powerTypeMapping[ powerType ]

_powerSourceMapping = { 'unknown' : 'Unknown',
                        'reserved' : 'Reserved',
                        'pse' : 'Power Sourcing Entity',
                        'pseAndLocal' : 'Power Sourcing Entity and local',
                        'primary' : 'Primary power source',
                        'backup' : 'Backup power source' }

def powerSourceToStr( powerSource ):
   return _powerSourceMapping[ powerSource ]

_powerPriorityMapping = { 'unknown' : 'Unknown',
                          'critical' : 'Critical',
                          'high' : 'High',
                          'low' : 'Low' }

def powerPriorityToStr( powerPriority ):
   return _powerPriorityMapping[ powerPriority ]

_btPowerFieldMapping = {
      None : 'Reserved',
      'twoPairPowering' : "2-pair Powering",
      'fourPairPoweringSingleSignature' : "4-pair Powering Single-signature",
      'fourPairPoweringDualSignature' : "4-pair Powering Dual-signature",
      'poweredSingleSignature' : "Powered Single-signature",
      'twoPairPoweringDualSignature' : "2-pair Powered Dual-signature",
      'powerPairsExtReserved0' : "Reserved",
      'alternativeA' : 'Alternative A',
      'alternativeB' : 'Alternative B',
      'bothAlternatives' : 'Both Alternatives',
      'powerClass1' : 'Class 1',
      'powerClass2' : 'Class 2',
      'powerClass3' : 'Class 3',
      'powerClass4' : 'Class 4',
      'powerClass5' : 'Class 5',
      'singleSignaturePd' : 'Single Signature PD',
      'twoPairOnlyPse' : 'Two Pair Only PSE',
      'powerClassExt1' : 'class 1',
      'powerClassExt2' : 'class 2',
      'powerClassExt3' : 'class 3',
      'powerClassExt4' : 'class 4',
      'powerClassExt5' : 'class 5',
      'powerClassExt6' : 'class 6',
      'powerClassExt7' : 'class 7',
      'powerClassExt8' : 'class 8',
      'dualSignaturePd' : 'Dual Signature Pd',
      'type3Pse' : 'type 3 PSE',
      'type4Pse' : 'type 4 PSE',
      'type3SingleSignaturePd' : 'type 3 single signature PD',
      'type3DualSignaturePd' : 'type 3 dual signature PD',
      'type4SingleSignaturePd' : 'type 4 single signature PD',
      'type4DualSignaturePd' : 'type 4 dual signature PD',
      }

def btPowerFieldToStr( value ):
   return _btPowerFieldMapping[ value ]

_medDeviceTypeMapping = { 'notDefined' : 'Not defined',
                          'endpointClass1' : 'Endpoint Class1',
                          'endpointClass2' : 'Endpoint Class2',
                          'endpointClass3' : 'Endpoint Class3',
                          'networkConnectivity' : 'Network connectivity',
                          'reserved' : 'Reserved' }
def medDeviceTypeToStr( deviceType ):
   return _medDeviceTypeMapping[ deviceType ]

networkPolicyApplicationTypeNames = {
      1 : 'voice',
      2 : 'voice signaling',
      3 : 'guest voice',
      4 : 'guest voice signaling',
      5 : 'softphone voice',
      6 : 'video conferencing',
      7 : 'streaming video',
      8 : 'video signaling',
}

def networkPolicyApplicationTypeToStr( appValue ):
   return networkPolicyApplicationTypeNames.get( appValue, 'unknown' )

class TlvType( Model ):
   tlvType = Enum( values=list( _tlvDisplayFormat ),
      help='portDescription -- Port Description TLV, '
      'systemName -- System Name TLV, '
      'systemDescription -- System Description TLV, '
      'systemCapabilities -- System Capabilities TLV, '
      'managementAddress -- Management Address TLV, '
      'portVlan -- IEEE802.1 Port VLAN ID TLV, '
      'portVlanName -- IEEE802.1 Port VLAN Name TLV, '
      'linkAggregation8021 -- IEEE802.1 Link Aggregation TLV, '
      'linkAggregation8023 -- IEEE802.3 Link Aggregation TLV, '
      'maxFrameSize -- IEEE802.3 Maximum Frame Size TLV, '
      'powerViaMdi -- IEEE802.3 Power Via MDI TLV, '
      'medCapabilities -- LLDP-MED Capabilities TLV, '
      'medNetworkPolicy -- LLDP-MED Network Policy TLV, '
      'medLocationIdentification -- LLDP-MED Location Identification TLV, '
      'serialNumber -- LLDP-MED Serial Number TLV, ' )

class LldpInterface( Model ):
   txEnabled = Bool( help="Indicates whether LLDP frames can be sent from"
         " this interface" )
   rxEnabled = Bool( help="Indicates whether LLDP frames can be received"
         " on this interface" )

class Lldp( Model ):
   messageTxInterval = Int( help="The interval in seconds at which LLDP"
         " frames are transmitted" )
   messageTxHoldTime = Int( help="The holdtime in seconds inserted in the"
         " transmitted LLDP frames" )
   reinitDelay = Int( help="The delay in seconds before LLDP"
         " re-initialization is attempted" )
   managementAddressVrf = Str( help="LLDP Management Address VRF" )
   enabledTlvTypes = List( valueType=TlvType,
         help="Enabled optional TLVs" )
   managementAddressTransmitted = Str( help="Management Address used "
         "in the TLV", optional=True )
   lldpInterfaces = Dict( keyType=Interface, valueType=LldpInterface,
         help="Lldp enabled Interfaces" )

   def render( self ):
      print( 'LLDP transmit interval      : %d seconds' % self.messageTxInterval )
      print( 'LLDP transmit holdtime      : %d seconds' % self.messageTxHoldTime )
      print( 'LLDP reinitialization delay : %d seconds' % self.reinitDelay )
      print( 'LLDP Management Address VRF : %s' % self.managementAddressVrf )

      print()
      print( 'Enabled optional TLVs:' )
      for tlv in self.enabledTlvTypes:
         if tlv.tlvType == 'managementAddress':
            print( '  {} ({})'.format( _tlvDisplayFormat[ tlv.tlvType ],
                   self.managementAddressTransmitted ) )
         else:
            print( '  %s' % _tlvDisplayFormat[ tlv.tlvType ] )
      print()
      table = TableOutput.createTable( [ 'Port', 'Tx Enabled', 'Rx Enabled' ] )
      table.formatColumns( firstColumnFormat,
                           defaultColumnFormat,
                           defaultColumnFormat )

      for intfName in sortIntf( self.lldpInterfaces ):
         shortName = CliPlugin.IntfCli.Intf.getShortname( intfName )
         lldpIntf = self.lldpInterfaces[ intfName ]
         txEnabled = 'Yes' if lldpIntf.txEnabled else 'No'
         rxEnabled = 'Yes' if lldpIntf.rxEnabled else 'No'
         table.newRow( shortName, txEnabled, rxEnabled )
      print( table.output() )

sysCapabilities = [ 'repeater', 'bridge', 'wlanAccessPoint', 'router',
                    'telephone', 'docsisCableDevice', 'stationOnly' ]

def printChassisId( chassisIdType, chassisId ):
   chassisIdTypeOutput = chassisIdSubtypeStrs.get( chassisIdType )
   if chassisIdTypeOutput != 'Unknown':
      value = LldpConstants.chassisIdSubtypeToValue(
            chassisIdSubtypeMap.get( chassisIdType ) )
      chassisIdTypeOutput += ' (%d)' % value
   print( prefix + '- Chassis ID type: %s' % chassisIdTypeOutput )
   print( prefix + '  Chassis ID     : %s' % chassisId )

def printSystemName( sysName ):
   isEmpty = not bool( sysName )
   sysName = '"%s"' % sysName if sysName else '(empty)*'
   print( prefix + '- System Name: %s' % sysName )
   return isEmpty

def printSystemDesc( sysDesc ):
   isEmpty = not bool( sysDesc )
   sysDesc = '"%s"' % sysDesc if sysDesc else '(empty)*'
   print( prefix + '- System Description: %s' % sysDesc )
   return isEmpty

def printSystemCapabilities( sysCapsSupported, sysCapsEnabled ):
   if sysCapsSupported:
      print( prefix + '- System Capabilities : %s' % sysCapsSupported )
   if sysCapsEnabled:
      print( prefix + '  Enabled Capabilities: %s' % sysCapsEnabled )

def printPortId( port ):
   portIdType = portIdSubtypeStrs.get( port.interfaceIdType )
   if portIdType != 'Unknown':
      portValue = LldpConstants.portIdSubtypeToValue(
                        portIdSubtypeMap.get( port.interfaceIdType ) )
      portIdType += '(%d)' % portValue
   print( prefix + '- Port ID type: %s' % portIdType )
   print( prefix + '  Port ID     : %s' % port.interfaceId )

def printPortDesc( description ):
   isEmpty = not bool( description )
   description = '"%s"' % description if description else '(empty)*'
   print( prefix + '- Port Description: %s' % description )
   return isEmpty

def printPortVlanId( vlanId ):
   print( prefix + '- IEEE802.1 Port VLAN ID: %d' % vlanId )

def printLinkAggrInfo( linkAggr, version ):
   """
   - version : One of (802.1, 802.3)
   """

   assert version in ( '802.1', '802.3' )

   if version == '802.1':
      linkAggrStatus = linkAggr.linkAggregation8021Status
      intfId = linkAggr.linkAggregation8021InterfaceId
   else:
      linkAggrStatus = linkAggr.linkAggregation8023Status
      intfId = linkAggr.linkAggregation8023InterfaceId

   linkAggrName = linkAggregationType.get( linkAggrStatus )
   linkAggregationStatusValue = linkAggrStatusValues.get( linkAggrStatus )
   linkAggregationStatus = '%s (0x%02x)' % \
         ( linkAggrName, linkAggregationStatusValue )
   print( prefix + '- IEEE' + version + ' Link Aggregation' )
   print( prefix + '  Link Aggregation Status: %s' % linkAggregationStatus )
   print( prefix + '  Port ID                : %d' % intfId )

def printMaxFrameSize( maxFrameSize ):
   print( prefix + '- IEEE802.3 Maximum Frame Size: %d bytes' % maxFrameSize )

def printPortAndProtocolVlan( neighbor ):
   print( prefix + '- IEEE802.1 Port And Protocol VLAN ID' )
   for vlanId, supported in \
         neighbor.portAndProtocolVlanSupported.items():
      print( prefix + '  VLAN ID  : %d' % vlanId )
      print( prefix + '  Supported: %s' % supported )
      print( prefix + '  Enabled  : %s'
            % neighbor.portAndProtocolVlanEnabled[ vlanId ] )

vlanNameChangedNote = "This list contains VLAN Name TLVs included in the last " + \
               "packet that was sent."
vlanNameIncompleteNote = vlanNameChangedNote[ : -1 ] + \
                         ", but there are more configured on this interface."

def printVlanNames( vlanNames, changed=False, incomplete=False ):
   # FIXME(549814) change print format ( 1 line per TLV) + use 'changed' and
   # 'incomplete' to print "warnings"
   print( prefix + '- IEEE802.1 VLAN Name' )
   if incomplete:
      print( prefix + '  ' + vlanNameIncompleteNote )
   elif changed:
      print( prefix + '  ' + vlanNameChangedNote )
   for vlanId, vlanName in sorted( vlanNames.items() ):
      print( prefix + '  VLAN ID: %d, VLAN Name: "%s"' % ( vlanId, vlanName ) )

def printProtocolIdentityInfo( protocolIdentity ):
   print( prefix + '- IEEE802.1 Protocol Identity' )
   for protocol in protocolIdentity:
      print( prefix + '  %s' % protocol )

def printVidUsageDigest( vidUsageDigest ):
   print( prefix + '- IEEE802.1 VID Usage Digest: 0x%08x' % vidUsageDigest )

def printManagementVid( managementVid ):
   print( prefix + '- IEEE802.1 Management VID: %d' % managementVid )

def printMacPhyConfigStatus( macPhy ):
   autoNegCapabilityName = autoNegCapabilityType.get( \
         macPhy.autoNegCapability )
   print( prefix + '- IEEE802.3 MAC/PHY Configuration/Status' )
   print( prefix + '  Auto-negotiation       : %s' % autoNegCapabilityName )
   for autoNegCap in macPhy.autoNegAdvertisedCapabilities:
      print( prefix + '                           %s' % autoNegCap )
   print( prefix + '  Operational MAU Type   : %s' % macPhy.operMauType )

def printPowerViaMdiInfo( powerInfo ):
   # pylint: disable=protected-access
   def printLine( field, value ):
      print( printPowerFormat % ( field, value ) )

   def btEnumStr( value, intValue, tacType ):
      v = btPowerFieldToStr( value )
      if v == 'Reserved':
         v = 'Reserved (%d)' % intValue
      return v

   def printBtEnum( field, value, intValue, tacType ):
      printLine( field, btEnumStr( value, intValue, tacType ) )

   print( prefix + '- IEEE802.3 Power Via MDI' )
   printLine( "Port Class", powerPortClassToStr( powerInfo.portClass ) )
   printLine( 'PSE MDI Power Support',
              'Supported' if powerInfo.powerOverEthernetSupported else
              'Not Supported' )
   printLine( 'PSE MDI Power State',
              'Enabled' if powerInfo.powerOverEthernetEnabled else 'Disabled' )
   printLine( 'PSE Pairs Control Ability',
              'Controllable' if powerInfo.controlAbility else 'Not controllable' )
   printLine( 'PSE Power Pair', powerInfo.powerPairs.capitalize() )
   if powerInfo._ieeeStandard in [ '802.3bt' ]:
      printLine( 'Power Class', "{}, Ext {}".format( powerClassToStr(
                                                   powerInfo.powerClass ),
                                                btEnumStr( powerInfo.powerClassExt,
                                                           powerInfo._powerClassExt,
                                                           btStr + 'PowerClassExt' )
                                                ) )
   else:
      printLine( 'Power Class', powerClassToStr( powerInfo.powerClass ) )
   if powerInfo._ieeeStandard in [ '802.3at' ]:
      printLine( 'Power Type', powerTypeToStr( powerInfo.powerType ) )
   elif powerInfo._ieeeStandard in [ '802.3bt' ]:
      printLine( 'Power Type', "{}, Ext {}".format( powerTypeToStr(
                                                   powerInfo.powerType ),
                                                btEnumStr( powerInfo.powerTypeExt,
                                                           powerInfo._powerTypeExt,
                                                           envPwr + 'PowerTypeExt' )
                                              ) )

   if powerInfo._ieeeStandard in [ '802.3at', '802.3bt' ]:
      printLine( 'Power Source', powerSourceToStr( powerInfo.powerSource ) )
      printLine( 'Power Priority', powerPriorityToStr( powerInfo.powerPriority ) )
      if powerInfo._ieeeStandard in [ '802.3at' ]:
         printLine( 'Requested Power', '%0.1fW' % powerInfo.pdRequestedPowerValue )
         printLine( 'Allocated Power', '%0.1fW' % powerInfo.pseAllocatedPowerValue )
      else:
         printLine( 'Requested Power',
               '{:0.1f}W, Mode A {:0.1f}W, Mode B {:0.1f}W'.format(
                                       powerInfo.pdRequestedPowerValue,
                                       powerInfo.pdRequestedPowerValueModeA,
                                       powerInfo.pdRequestedPowerValueModeB ) )
         printLine( 'Allocated Power',
               '{:0.1f}W, Mode A {:0.1f}W, Mode B {:0.1f}W'.format(
                                    powerInfo.pseAllocatedPowerValue,
                                    powerInfo.pseAllocatedPowerValueAlternativeA,
                                    powerInfo.pseAllocatedPowerValueAlternativeB ) )

   if powerInfo._ieeeStandard in [ '802.3bt' ]:
      if powerInfo.pd4Pid is not None:
         printLine( 'PD 4PID', powerInfo.pd4Pid )
      if powerInfo._psePoweringStatus is not None:
         printBtEnum( 'PSE Powering Status', powerInfo.psePoweringStatus,
                      powerInfo._psePoweringStatus, envPwr + 'PsePoweringStatus' )
      if powerInfo._pdPoweredStatus is not None:
         printBtEnum( 'PD Powered Status', powerInfo.pdPoweredStatus,
                      powerInfo._pdPoweredStatus, btStr + 'PdPoweredStatus' )
      if powerInfo._psePowerPairsExt is not None:
         printBtEnum( 'Power Pairs Alternative', powerInfo.psePowerPairsExt,
                      powerInfo._psePowerPairsExt, envPwr + 'PsePowerPairsExt' )
      printLine( 'Power Class Signature', 'Mode A {}, Mode B {}'.format(
                 btEnumStr( powerInfo.dualSignaturePowerClassExtModeA,
                            powerInfo._dualSignaturePowerClassExtModeA,
                            envPwr + 'DualSignaturePowerClassExtMode' ),
                 btEnumStr( powerInfo.dualSignaturePowerClassExtModeB,
                            powerInfo._dualSignaturePowerClassExtModeB,
                            envPwr + 'DualSignaturePowerClassExtMode' ) ) )
      if powerInfo.pdLoad is not None:
         printLine( 'PD Load', powerInfo.pdLoad )
      if powerInfo.pseMaximumAvailablePower is not None:
         printLine( 'PSE Maximum Available Power', '%0.1fW' %
                                           powerInfo.pseMaximumAvailablePower )
      if powerInfo.pseAutoclassSupport is not None or \
            powerInfo.autoclassCompleted is not None:
         sup = 'Supported' if powerInfo.pseAutoclassSupport else 'Not supported'
         comp = 'Completed' if powerInfo.autoclassCompleted else 'Not completed'
         printLine( 'PSE Autoclass Measurement', f'{sup}, {comp} ' )
      if powerInfo.autoclassRequest is not None:
         printLine( 'PD Requested Autoclass', powerInfo.autoclassRequest )
      if ( ( powerInfo.powerDownRequest is not None ) or
           ( powerInfo.powerDownTime is not None ) ):
         time = powerInfo.powerDownTime
         powerDownMsg = 'Not Requested' if not powerInfo.powerDownRequest else \
                           'for %d seconds' % time if time else 'Indefinitely'
         printLine( 'PD Power Down Request', powerDownMsg )

def printMedInfo( medInfo ):
   def _printCapabilitiesInfo( capabilities ):
      print( prefix + "- LLDP-MED Capabilities:" )
      print( prefix + "  'Capabilities' capable                 : %s" %
                           capabilities.capabilities )
      print( prefix + "  'Network Policy' capable               : %s" %
                           capabilities.networkPolicy )
      print( prefix + "  'Location Identification' capable      : %s" %
                           capabilities.location )
      print( prefix + "  'Extended Power via MDI - PSE' capable : %s" %
                           capabilities.extendedPse )
      print( prefix + "  'Extended Power via MDI - PD' capable  : %s" %
                           capabilities.extendedPd )
      print( prefix + "  'Inventory' capable                    : %s" %
                           capabilities.inventory )
      print( prefix + "  Device type                            : %s" %
                           medDeviceTypeToStr( capabilities.deviceType ) )

   def _printNetworkPolicies( networkPolicies ):
      print( prefix + '- LLDP-MED Network policies:' )
      for netpol in networkPolicies:
         print( prefix + '  - Application type: %s (%d)' % (
            networkPolicyApplicationTypeToStr( netpol.applicationType ),
            netpol.applicationType ) )
         print( prefix + '    Unknown         : ' + (
            'yes' if netpol.unknownPolicy else 'no' ) )
         print( prefix + '    Tagged          : ' + (
            'yes' if netpol.tagged else 'no' ) )
         print( prefix + '    VLAN ID         : %d' % netpol.vlanId )
         print( prefix + '    CoS             : %d' % netpol.layer2Priority )
         print( prefix + '    DSCP            : %d' % netpol.dscpValue )


   if medInfo.capabilities:
      _printCapabilitiesInfo( medInfo.capabilities )
   if medInfo.networkPolicies:
      _printNetworkPolicies( medInfo.networkPolicies )
   medInfo.printLocation()
   return medInfo.printInventory()

def printZtpBootVlan( vlanId ):
   if vlanId:
      print( prefix + '- ZeroTouch Provisioning Boot VLAN: ' + str( vlanId ) )

class SystemCapabilities( Model ):
   other = Bool( help="Indicates whether the system has any capabilities other than"
         " %s" % ( ",".join( sysCapabilities ) ), optional=True )
   repeater = Bool( help="Indicates whether the system is enabled to be a repeater",
         optional=True )
   bridge = Bool( help="Indicates whether the system is enabled to be a bridge",
         optional=True )
   wlanAccessPoint = Bool( help="Indicates whether the system is enabled to be a"
         " WLAN Access Point", optional=True )
   router = Bool( help="Indicates whether the system is enabled to be a router",
         optional=True )
   telephone = Bool( help="Indicates whether the system is enabled to be a"
         " telephone", optional=True )
   docsisCableDevice = Bool( help="Indicates whether the system is enabled to be a"
         " DOCSIS Cable Device", optional=True )
   stationOnly = Bool( help="Indicates whether the system is enabled to be a"
         " station and nothing else", optional=True )

class UnknownOrgDefTlv( Model ):
   ouiStr = Str( help="Hexdump of Organizationally Unique Identifier of the"
         " information received from the neighbor", optional=True )
   subType = Int( help="Unknown Organizationally-Defined TLV type", optional=True )
   malformedTlvError = Str( help="Error string if the TLV is malformed",
         optional=True )
   tlvLines = List( valueType=str, help="Represents the received Unknown"
         " Organizationally-Defined TLV in either decoded string format or hexdump"
         " depending on whether decoding was successful" )

   def render( self ):
      newPrefix = prefix
      dash = "- "
      if self.malformedTlvError is not None:
         print( prefix + dash + "Malformed TLV:", self.malformedTlvError )
         dash = "  "
      if self.ouiStr:
         # pylint: disable-next=bad-string-format-type
         print( prefix + dash + 'Unknown organizationally-defined TLV'
               ' (OUI %s, subtype %d):' % ( self.ouiStr, self.subType ) )
         newPrefix = prefix + prefix
         dash = "  "
      for line in self.tlvLines:
         print( newPrefix + dash + line )
         dash = "  "

class UnknownTlv( Model ):
   tlvType = Int( help="Unknown TLV type which is not defined in LLDP protocol",
         optional=True )
   tlvLines = List( valueType=str, help="Hexdump of the information about"
         "received Unknown TLV", optional=True )

   def render( self ):
      print( prefix + '- Unknown TLV ( type %d ):' % self.tlvType )
      for line in self.tlvLines :
         print( prefix + '    ' + line )

class ManagementAddr( Model ):
   addressType = Enum( values=managementAddressSubTypeValues,
         help="Type of LLDP management address" )
   address = Str( help="LLDP management address which could either be an IPv4/v6"
         " address or a MAC address, may be used to reach higher layer entities" )
   interfaceNumType = Enum( values=list( ifSubtypeStrs ),
         help="Interface numbering method used by the interfaceNum attribute" )
   interfaceNumTypeValue = Int( help="Interface number subtype value which is"
         " unrecognized", optional=True )
   interfaceNum = Int( help="Interface number associated with the management"
         " address, in the encoding specified by interfaceNumType attribute" )
   oidString = Str( help="OID value used to identify the type of hardware"
         " component or protocol entity associated with the management address" )

   def render( self ):
      # pylint: disable-next=trailing-comma-tuple
      manAddrTypeValue = '%s ' % manAddrSubtypes.get( self.addressType ),
      name = ifSubtypeStrs.get( self.interfaceNumType )
      if self.interfaceNumTypeValue is not None:
         value = self.interfaceNumTypeValue
      else:
         value = LldpConstants.ifSubtypeToValue( self.interfaceNumType )
      ifTypeValue = '%s (%d)' % ( name, value )
      print( prefix +
            '- Management Address Subtype: %s' % manAddrTypeValue )
      print( prefix + '  Management Address        : %s' % self.address )
      print( prefix + '  Interface Number Subtype  : %s'
            % ifTypeValue )
      print( prefix + '  Interface Number          : %d' % self.interfaceNum )
      print( prefix + '  OID String                : %s' % self.oidString )

class LldpDevice( Model ):
   chassisIdType = Enum( values=chassisIdSubtypeValues, help="Type of the chassis" )
   chassisId = Str( help="Chassis identifier string" )
   systemName = Str( help="System name", optional=True )
   systemDescription = Str( help="System description", optional=True )
   systemCapabilities = Submodel( valueType=SystemCapabilities,
         help="Supported system capabilities and information about whether they are"
         " enabled", optional=True )

   def getSystemCapabilities( self ):
      supported = []
      enabled = []
      sysCaps = self.systemCapabilities
      if sysCaps.other is not None:
         supported += [ "Other" ]
         if sysCaps.other is True:
            enabled += [ "Other" ]
      if sysCaps.repeater is not None:
         supported += [ "Repeater" ]
         if sysCaps.repeater is True:
            enabled += [ "Repeater" ]
      if sysCaps.bridge is not None:
         supported += [ "Bridge" ]
         if sysCaps.bridge is True:
            enabled += [ "Bridge" ]
      if sysCaps.wlanAccessPoint is not None:
         supported += [ "WLAN Access Point" ]
         if sysCaps.wlanAccessPoint is True:
            enabled += [ "WLAN Access Point" ]
      if sysCaps.router is not None:
         supported += [ "Router" ]
         if sysCaps.router is True:
            enabled += [ "Router" ]
      if sysCaps.telephone is not None:
         supported += [ "Telephone" ]
         if sysCaps.telephone is True:
            enabled += [ "Telephone" ]
      if sysCaps.docsisCableDevice is not None:
         supported += [ "DOCSIS Cable Device" ]
         if sysCaps.docsisCableDevice is True:
            enabled += [ "DOCSIS Cable Device" ]
      if sysCaps.stationOnly is not None:
         supported += [ "Station Only" ]
         if sysCaps.stationOnly is True:
            enabled += [ "Station Only" ]
      if not supported:
         supported += [ "None" ]
      if not enabled:
         enabled += [ "None" ]
      return ( ", ".join( supported ), ", ".join( enabled ) )

class MedCapabilitiesInfo( Model ):
   capabilities = Bool( help="System supports 'Capabilities' TLV" )
   networkPolicy = Bool( help="System supports 'Network Policy' TLV" )
   location = Bool( help="System supports 'Location Identification' TLV" )
   extendedPse = Bool( help="System supports 'Extended Power via MDI - PSE' TLV" )
   extendedPd = Bool( help="System supports 'Extended Power via MDI - PD' TLV" )
   inventory = Bool( help="System supports 'Inventory' TLV" )
   deviceType = Enum( values=( 'notDefined', 'endpointClass1', 'endpointClass2',
                               'endpointClass3', 'networkConnectivity', 'reserved' ),
                      help="MED device type" )

class MedNetworkPolicyInfo( Model ):
   applicationType = Int( help="Application Type for given policy" )
   unknownPolicy = Bool( help="Endpoint Device wants to explicitly advertise "
                               "that this policy is required by the device, but "
                               "is currently unknown" )
   tagged = Bool( help="The specified application type is using a 'tagged' VLAN" )
   vlanId = Int( help="VLAN identifier for the port as defined in IEEE 802.1Q-2003",
                 optional=True )
   layer2Priority = Int( help="Layer 2 priority to be used for the specified "
                              "application type",
                         optional=True )
   dscpValue = Int( help="DSCP value to be used by Endpoint Device for given "
                         "application to specify priority for IP datagram" )

class SignedFixedPointBase:
   """Two's complement fixed point number
   """
   bitLength = None
   binaryPoint = None

   def __init__( self, fixedPointAsUInt ):
      assert self.bitLength and self.binaryPoint
      assert int( fixedPointAsUInt ) == fixedPointAsUInt
      assert fixedPointAsUInt < ( 1 << self.bitLength )
      self.fixedPointAsUInt = fixedPointAsUInt

   def __float__( self ):
      signed = self.fixedPointAsUInt
      maxInt = 1 << ( self.bitLength - 1 )
      if signed > maxInt:
         complement = 1 << self.bitLength
         signed -= complement
      scale = float( 1 << self.binaryPoint )
      return signed / scale

class Coordinate( SignedFixedPointBase ):
   """Surface latitude and longitude coordinates according to rfc3825
   """
   bitLength = 34
   binaryPoint = 25

class Altitude( SignedFixedPointBase ):
   """According to rfc3825
   """
   bitLength = 30
   binaryPoint = 8

# Geodetic datum code as from rfc6225:
# 0   Reserved
# 1   Vertical datum WGS 84 defined by EPSG CRS Code 4327
# 2   Vertical datum NAD83 defined by EPSG CRS Code 4269
#     with North American Vertical Datum of 1988 ( NAVD88 )
# 3   Vertical datum NAD83 defined by EPSG CRS Code 4269
#     with Mean Lower Low Water ( MLLW ) as associated vertical datum
# 4-7 Unassigned
datumMap = { 0: 'Reserved',
             1: 'WGS 84',
             2: 'NAD83-NAVD88',
             3: 'NAD83-MLLW' }

aTMap = { 1: "meters",
          2: "floors" }

class CoordinateBasedLocationInfo( Model ):
   laRes = Int( help="Latitude resolution as per RFC 3825" )
   latitude = Int( help="Latitude as per RFC 3825: "
                   "fixed point value with 25 bits fractional part" )
   loRes = Int( help="Longitude resolution as per RFC 3825" )
   longitude = Int( help="Longitude as per RFC 3825: "
                    "fixed point value with 25 bits fractional part" )
   aT = Int( help="Altitude type (at) as per RFC 3825" )
   altRes = Int( help="Altitude resolution as per RFC 3825" )
   altitude = Int( help="Altitude as per RFC 3825: "
                   "fixed point value with 8 bits fractional part" )
   datum = Int( help="Datum as per RFC 3825" )

   def populate( self, coordinateBasedData ):
      def copyAttrs( dest, src ):
         for attrName in src.attributes:
            isPrivateAttr = attrName.startswith( '_' )
            if not isPrivateAttr and hasattr( dest, attrName ):
               setattr( dest, attrName, getattr( src, attrName ) )
      copyAttrs( self, coordinateBasedData )

   def render( self ):
      """Based on the examples from rfc3825 and rfc6225

      Coordinate-based LCI:
        Latitude        : +38.89868 degrees (0x04DCC1FC8)  Res: 0.03125 degrees (30)
        Longitude       : -77.03723 degrees (0xF65ECF031)  Res: 0.03125 degrees (30)
        Altitude        : +200.000 meters (0x0F00)  Altitude type: 1
        Datum           : WGS 84 (1)
      """
      fmt = prefix + "  %-13s: %s"
      print( fmt % ( "Coordinate-based LCI", "" ) )
      fmt = prefix + fmt
      self._printCoordinate( fmt, "Latitude", self.latitude, self.laRes )
      self._printCoordinate( fmt, "Longitude", self.longitude, self.loRes )
      self._printAltitude( fmt )
      self._printDatum( fmt )

   def _printCoordinate( self, lineFmt, label, coordinateInt, res ):
      """Print a coordinate value and resolution as specified in rfc3825:
      a 34 bit fixed point value with 9 bits for integer and 25 bits for fraction
      """
      fmt = "%+9.5f degrees (0x%09x)\tRes: %s"
      coordinateFloat = float( Coordinate( coordinateInt ) )
      txt = fmt % ( coordinateFloat, coordinateInt, self._resText( res ) )
      print( lineFmt % ( label, txt ) )

   def _printAltitude( self, lineFmt ):
      """Print the altitude value and resolution as specified in rfc3825:
      30 bit value in 2s-complement fixed-point, 22-bit integer and 8-bit fraction
      """
      fmt = "%+6.2f %s (0x%08x)\tRes: %s bits\tAT: %d"
      altitudeFloat = float( Altitude( self.altitude ) )
      aTText = aTMap.get( self.aT, "UNKNOWN" )
      txt = fmt % ( altitudeFloat, aTText, self.altitude, self.altRes, self.aT )
      print( lineFmt % ( "Altitude", txt ) )

   def _printDatum( self, lineFmt ):
      """Geodetic datum code as from rfc6225"""
      # pylint: disable-next=bad-string-format-type
      txt = "%s (%d)" % ( datumMap.get( self.datum, "UNKNOWN" ), self.datum )
      print( lineFmt % ( "Datum", txt ) )

   def _resText( self, res ):
      """Resolution unsigned int to text
      https://tools.ietf.org/html/rfc3825#appendix-A.1
      """
      if res > Coordinate.bitLength or res < 2:
         # rfc3825: "Values above decimal 34 are undefined and reserved."
         # res < 2: is either invalid or indicates unknown coordinates
         return "UNKNOWN (%d)" % res
      fmt = "%1.5f degrees (%d)"
      return fmt % ( self._resToDegrees( res ), res )

   def _resToDegrees( self, highestOrderValidBits ):
      """Convert the coordinate resolution in number of bits to degrees
      """
      lowestOrderBits = Coordinate.bitLength - highestOrderValidBits
      maxPrecision = 1 << lowestOrderBits
      return float( Coordinate( maxPrecision ) )

class CivicAddressElement( Model ):
   caType = Int( help="Civic Address (CA) Type as per "
                    "draft-ietf-geopriv-dhcp-civil-09" )
   caValue = Str( help="Civic Address (CA) Value as per "
                     "draft-ietf-geopriv-dhcp-civil-09" )

caWhatMapping = {
   0 : 'DHCP server',
   1 : 'network element',
   2 : 'client' }

def caWhatToStr( what ):
   """
   ;;; caWhatToStr( 0 )
   'DHCP server'
   ;;; caWhatToStr( 1 )
   'network element'
   ;;; caWhatToStr( 2 )
   'client'
   ;;; caWhatToStr( 3 )
   'unknown'
   ;;; caWhatToStr( -1 )
   'unknown'

   """
   return caWhatMapping.get( what, 'unknown' )

def _getSortedCA( items ):
   """
   ;;; import collections
   ;;; CA = collections.namedtuple( 'CA', ['caType', 'caValue'] )
   ;;; _getSortedCA( [] )
   []

   ;;; [ ( v.caType, v.caValue ) for v in _getSortedCA( [ 
   ... CA( 5, 1 ), CA( 1, 2 ) ] ) ]
   [(1, 2), (5, 1)]

   ;;; [ ( v.caType, v.caValue ) for v in _getSortedCA( [ 
   ... CA( 0, 1 ), CA( 5, 2 ), CA( 1, 3 ), CA( 0, 4), CA( 100, 5 ), CA( 1, 6 ) ] ) ]
   [(0, 1), (1, 3), (5, 2), (0, 4), (1, 6), (100, 5)]

   ;;; [ ( v.caType, v.caValue ) for v in _getSortedCA( [ 
   ...              CA( 128, 1 ), CA( 56, 2 ), CA( 1, 3 ), CA( 128, 4), CA( 129, 5),
   ...              CA( 1, 6 ) ] ) ]
   [(128, 1), (1, 3), (56, 2), (128, 4), (1, 6), (129, 5)]

   ;;; [ ( v.caType, v.caValue ) for v in _getSortedCA( [ 
   ...                CA( 0, 1 ), CA( 128, 2 ), CA( 2, 3 ), CA( 10, 4 ), CA( 1, 5 ),
   ...                CA( 0, 6 ), CA( 10, 7 ), CA( 2, 8 ), CA( 24, 9 ) ] ) ]
   [(0, 1), (128, 2), (1, 5), (2, 3), (10, 4), (0, 6), (2, 8), (10, 7), (24, 9)]

   """
   def groupByMarker( items, markers=( 0, 128 ) ):
      group = oldIsMarker = 0
      for item in items:
         isMarker = item.caType in markers
         group += isMarker != oldIsMarker
         oldIsMarker = isMarker
         yield group, item
   return [ v for _, v in sorted( groupByMarker( items ) ) ]

class CivicAddress( Model ):
   what = Int( help="Which location the address refers to as per RFC 4776" )
   countryCode = Str( help="Two-letter ISO 3166 country code" )
   civicAddressElements = List( valueType=CivicAddressElement,
                                help="List of Civic Address Elements" )

   def populate( self, civicAddressData ):
      self.what = civicAddressData.what
      self.countryCode = escapedString( civicAddressData.countryCode )
      for caElement in _getSortedCA( list( civicAddressData.caElements.values() ) ):
         elem = CivicAddressElement()
         elem.caType = caElement.caType
         elem.caValue = escapedString( caElement.caValue )
         self.civicAddressElements.append( elem )

   def render( self ):
      def _print( msg, level ):
         print( level * prefix + msg )
      _print( 'Civic Address LCI:', 2 )
      fmt = "%-13s: %s"
      whatTxt = "%s (%d)" % ( caWhatToStr( self.what ), self.what )
      _print( fmt % ( 'What', whatTxt ), 3 )
      _print( fmt % ( 'Country Code', self.countryCode ), 3 )
      if self.civicAddressElements:
         _print( 'CAtype  CAvalue', 4 )
      for elem in self.civicAddressElements:
         _print( '%6d  %s' % ( elem.caType, elem.caValue ), 4 )

class MedLocationIdentificationInfo( Model ):
   coordinateBased = Submodel( valueType=CoordinateBasedLocationInfo,
                               help="Coordinate-based LCI info as per RFC 3825",
                               optional=True )
   civicAddress = Submodel( valueType=CivicAddress,
                            help="Civic Address LCI as per "
                                 "draft-ietf-geopriv-dhcp-civil-09",
                            optional=True )
   ecsElin = Str( help="Emergency Location Identification Number to support "
                       "traditional PSAP-based Emergency Communications Systems",
                  optional=True )

   def render( self ):
      if self.coordinateBased:
         self.coordinateBased.render()
      if self.civicAddress:
         self.civicAddress.render()
      if self.ecsElin:
         print( prefix + prefix + 'ECS ELIN: "%s"' % self.ecsElin )

   def populate( self, locationId ):
      populated = 0
      populated += 1 if self._populateCoordinateBasedLCI(
                                               locationId.coordinateBasedLCI ) else 0

      populated += 1 if self._populateCivicAddressLCI(
                                               locationId.civicAddressLCI ) else 0

      populated += 1 if self._populateEcsElin( locationId.ecsElin ) else 0

      return populated > 0

   def _populateCoordinateBasedLCI( self, coordinateBasedData ):
      if not coordinateBasedData.received:
         return False
      self.coordinateBased = CoordinateBasedLocationInfo()
      self.coordinateBased.populate( coordinateBasedData )
      return True

   def _populateCivicAddressLCI( self, civicAddressData ):
      if not civicAddressData.received:
         return False
      self.civicAddress = CivicAddress()
      self.civicAddress.populate( civicAddressData )
      return True

   def _populateEcsElin( self, ecsElinData ):
      if not ecsElinData.received:
         return False
      self.ecsElin = escapedString( ecsElinData.elin )
      return True

_inventoryTLVs = [ 'hardwareRevisionTlvInfo', 'firmwareRevisionTlvInfo',
                   'softwareRevisionTlvInfo', 'serialNumberTlvInfo',
                   'manufacturerNameTlvInfo', 'modelNameTlvInfo',
                   'assetIDTlvInfo' ]
class LldpMedInfo( Model ):
   capabilities = Submodel( valueType=MedCapabilitiesInfo,
                            help="MED Capabilities", optional=True )
   networkPolicies = List( valueType=MedNetworkPolicyInfo,
                           help="Collection of Network Policies" )
   locationId = Submodel( valueType=MedLocationIdentificationInfo,
                          help="Location Identification information", optional=True )
   hardwareRevisionTlvInfo = Str( help="Hardware Revision Tlv", optional=True )
   firmwareRevisionTlvInfo = Str( help="Firmware Revision Tlv", optional=True )
   softwareRevisionTlvInfo = Str( help="Software Revision Tlv", optional=True )
   serialNumberTlvInfo = Str( help="Serial Number Tlv", optional=True )
   manufacturerNameTlvInfo = Str( help="Manufacturer Name Tlv", optional=True )
   modelNameTlvInfo = Str( help="Model Name Tlv", optional=True )
   assetIDTlvInfo = Str( help="Asset ID Tlv", optional=True )

   def printLocation( self ):
      if not self.locationId:
         return
      print( prefix + '- LLDP-MED Location Identification TLVs:' )
      self.locationId.render()

   def printInventory( self ):
      def _print( field, name ):
         if field is None:
            return False

         value = '"%s"' % field if field else '(empty)*'
         print( prefix + f'- LLDP-MED Inventory {name}: {value}' )
         return value == '(empty)*'
      emptyValuePrinted = False
      emptyValuePrinted |= _print( self.hardwareRevisionTlvInfo,
                                   "Hardware Revision TLV" )
      emptyValuePrinted |= _print( self.firmwareRevisionTlvInfo,
                                   "Firmware Revision TLV" )
      emptyValuePrinted |= _print( self.softwareRevisionTlvInfo,
                                   "Software Revision TLV" )
      emptyValuePrinted |= _print( self.serialNumberTlvInfo,
                                   "Serial Number TLV" )
      emptyValuePrinted |= _print( self.manufacturerNameTlvInfo,
                                   "Manufacturer Name TLV" )
      emptyValuePrinted |= _print( self.modelNameTlvInfo,
                                   "Model Name TLV" )
      emptyValuePrinted |= _print( self.assetIDTlvInfo,
                                   "Asset ID TLV" )
      return emptyValuePrinted

   def populateLocation( self, locationId ):
      assert not self.locationId, "This should not be populated before"
      medLocationIdentificationInfo = MedLocationIdentificationInfo()

      if medLocationIdentificationInfo.populate( locationId ):
         self.locationId = medLocationIdentificationInfo

   def populateInventoryTLVs( self, data ):
      for key, value in data.items():
         if value.isSet:
            setattr( self, key, escapedString( value.value ) )

class LldpInterfaceInfo( Model ):
   interfaceIdType = Enum( values=portIdSubtypeValues,
         help="Type of local/neighbor interface" )
   interfaceId = Str( help="Name of local/neighbor interface"
                           "(may contain escaped quotation marks)" )
   interfaceId_v2 = Str( help="Name of local/neighbor interface" )
   interfaceDescription = Str( help="Description of local/neighbor interface",
         optional=True )
   linkAggregation8021Status = Enum( values=linkAggregationTypeValues,
         help="Link Aggregation 802.1 Status", optional=True )
   linkAggregation8023Status = Enum( values=linkAggregationTypeValues,
         help="Link Aggregation 802.3 Status", optional=True )
   linkAggregation8021InterfaceId = Int(
         help="Link Aggregation 802.1 interface identifier", optional=True )
   linkAggregation8023InterfaceId = Int(
         help="Link Aggregation 802.3 interface identifier", optional=True )
   maxFrameSize = Int( help="Indicates the maximum supported frame size"
         " in octets on the given interface", optional=True )
   portVlanId = Int( help="Port VLAN ID", optional=True )
   vlanNames = Dict( keyType=int, valueType=str,
         help="A mapping between vlan id and vlan name" )

   # PoE TLV
   _powerViaMdiTlvReceived = Bool( help="PowerViaMdiTlv Received" )
   # PoE TLV from IEEE 802.3af
   _ieeeStandard = Enum( values=( '802.3af', '802.3at', '802.3bt' ),
                         help="IEEE Standard version", optional=True )
   portClass = Enum( values=( 'pClassPSE', 'pClassPD', 'unknown' ),
         help="Port class of neighbor interface", optional=True )
   powerOverEthernetSupported = Bool( help="Indicates whether MDI power is"
         " supported on neighbor interface", optional=True )
   powerOverEthernetEnabled = Bool( help="Indicates whether MDI power is"
         " enabled on neighbor interface", optional=True )
   controlAbility = Bool( help="Indicates whether the pair selection"
         " can be controlled on neighbor interface", optional=True )
   powerPairs = Enum( values=powerPairsTypeValues,
         help="PSE power pairs on use on neighbor interface", optional=True )
   powerClass = Int( help="Power classification of neighbor interface",
         optional=True )
   # PoE TLV from IEEE 802.3at
   powerType = Enum( values=( 'type2PseDevice', 'type2PdDevice',
                              'type1PseDevice', 'type1PdDevice' ),
                              help="Power Type - type 1/2 and Power Sourcing Entity"
                                   "/Powered Device",
                              optional=True )
   powerSource = Enum( values=( 'unknown', 'reserved', 'pse', 'pseAndLocal',
                                'primary', 'backup' ),
                       help="Power Source",
                       optional=True )
   powerPriority = Enum( values=( 'unknown', 'critical', 'high', 'low' ),
                         help="Power priority on neighbor interface",
                         optional=True )
   pdRequestedPowerValue = Float( help="Power requested by device in watts",
                                optional=True )
   pseAllocatedPowerValue = Float( help="Power granted to device in watts",
                                 optional=True )

   # PoE TLV from IEEE 802_3bt
   pd4Pid = Bool( help="The PD supports powering of both Modes simultaneously",
                  optional=True )
   pdRequestedPowerValueModeA = Float( help="Maximum input power (in watts) the PD"
                                    " is requesting for the A Mode", optional=True )
   pdRequestedPowerValueModeB = Float( help="Maximum input power (in watts) the PD"
                                    " is requesting for the B Mode", optional=True )
   pseAllocatedPowerValueAlternativeA = Float( help="Maximum output power "
                  "(in watts) the PSE is providing for the A Mode", optional=True )
   pseAllocatedPowerValueAlternativeB = Float( help="Maximum output power "
                  "(in watts) the PSE is providing for the B Mode", optional=True )
   psePoweringStatus = Enum( values=( 'poweringStatusReserved0',
                                      'twoPairPowering',
                                      'fourPairPoweringSingleSignature',
                                      'fourPairPoweringDualSignature' ),
                              help="Existing powering configuration of the PSE",
                              optional=True )
   _psePoweringStatus = Int( help="Real value of psePoweringStatus", optional=True )
   pdPoweredStatus = Enum( values=( 'poweredSingleSignature',
                                    'twoPairPoweringDualSignature',
                                    'fourPairPoweringDualSignature' ),
                           help="Existing powered configuration of the PD",
                           optional=True )
   _pdPoweredStatus = Int( help="Real value of pdPoweredStatus", optional=True )
   psePowerPairsExt = Enum( values=( 'alternativeA',
                                     'alternativeB',
                                     'bothAlternatives' ),
                             help="Alternative used by PSE", optional=True )
   _psePowerPairsExt = Int( help="Real value of psePowerPairsExt", optional=True )
   dualSignaturePowerClassExtModeA = Enum( values=( 'powerClass1',
                                                    'powerClass2',
                                                    'powerClass3',
                                                    'powerClass4',
                                                    'powerClass5',
                                                    'singleSignaturePd',
                                                    'twoPairOnlyPse' ),
                                           help="Power class on Mode A",
                                           optional=True )
   _dualSignaturePowerClassExtModeA = Int( help="Real value of "
                                 "dualSignaturePowerClassExtModeA", optional=True )
   dualSignaturePowerClassExtModeB = Enum( values=( 'powerClass1',
                                                    'powerClass2',
                                                    'powerClass3',
                                                    'powerClass4',
                                                    'powerClass5',
                                                    'singleSignaturePd',
                                                    'twoPairOnlyPse' ),
                                           help="Power class on Mode B",
                                           optional=True )
   _dualSignaturePowerClassExtModeB = Int( help="Real value of "
                                 "dualSignaturePowerClassExtModeB", optional=True )
   powerClassExt = Enum( values=( # 'powerClassExtReserved0',
                                  'powerClassExt1',
                                  'powerClassExt2',
                                  'powerClassExt3',
                                  'powerClassExt4',
                                  'powerClassExt5',
                                  'powerClassExt6',
                                  'powerClassExt7',
                                  'powerClassExt8',
                                  'dualSignaturePd' ),
                         help="Power class ext", optional=True )
   _powerClassExt = Int( help="Real value of powerClassExt", optional=True )
   powerTypeExt = Enum( values=( 'type3Pse',
                                 'type4Pse',
                                 'type3SingleSignaturePd',
                                 'type3DualSignaturePd',
                                 'type4SingleSignaturePd',
                                 'type4DualSignaturePd' ),
                        help="Power Type ext", optional=True )
   _powerTypeExt = Int( help="Real value of powerTypeExt", optional=True )
   pdLoad = Bool( help="PD is dual-signature and power demand on Mode A and Mode B "
                       "are electrically isolated.", optional=True )
   pseMaximumAvailablePower = Float( help="Highest power (in watts) the PSE "
                                               "can grant", optional=True )
   pseAutoclassSupport = Bool( help="PSE supports Autoclass",
                               optional=True )
   autoclassCompleted = Bool( help="PSE has concluded the Autoclass measurement",
                              optional=True )
   autoclassRequest = Bool( help="PD has requested an Autoclass measurement by the "
                                 "PSE", optional=True )
   powerDownRequest = Bool( help="PD has requested a power down", optional=True )
   powerDownTime = Int( help="Time in seconds the PD requests to be unpowered. "
                             "Zero means to remain unpowered indefinitely.",
                        optional=True )

   # LLDP-MED
   medInfo = Submodel( valueType=LldpMedInfo, help="LLDP-MED Info",
                       optional=True )

   unknownOrgDefinedTlvs = List( valueType=UnknownOrgDefTlv,
         help="A list of unknown organizationally specific TLVs" )

   ztpBootVlan = Int( help="ZTP Boot VLAN", optional=True )

   def getOrCreateLldpMedInfo( self ):
      if not self.medInfo:
         self.medInfo = LldpMedInfo()
      return self.medInfo

   def populateMedLocationTLVs( self, locationId ):
      if ( not locationId.coordinateBasedLCI.received and
           not locationId.civicAddressLCI.received and
           not locationId.ecsElin.received ):
         return
      self.getOrCreateLldpMedInfo().populateLocation( locationId )

   def populateMedInventoryTLVs( self, neighbor ):
      data = ( ( k, getattr( neighbor, k ) ) for k in _inventoryTLVs )
      data = { k : v for k, v in data if v }
      if not data:
         return
      self.getOrCreateLldpMedInfo().populateInventoryTLVs( data )

class LocalInterfaceInfo( LldpInterfaceInfo ):
   vlanNamesChanged = Bool( help="List of VLAN Name TLVs might not be up to date" )
   vlanNamesIncomplete = Bool( help="List of VLAN Name TLVs is not complete" )

class LldpLocal( LldpDevice ):
   managementAddresses = List( valueType=ManagementAddr,
         help="A list of management addresses ordered by priority" )
   localInterfaceInfo = Dict( keyType=Interface, valueType=LocalInterfaceInfo,
         help="A mapping between local interface and local Interface information"
         " such as interface id, description, sent TLVs etc" )

   def render( self ):
      print( 'Local System:' )
      printChassisId( self.chassisIdType, self.chassisId )
      printSystemName( self.systemName )
      printSystemDesc( self.systemDescription )
      ( sysCapsSupported, sysCapsEnabled ) = self.getSystemCapabilities()
      printSystemCapabilities( sysCapsSupported, sysCapsEnabled )

      for intfName in sortIntf( self.localInterfaceInfo ):
         print()
         print( 'Interface %s:' % intfName )
         localInterface = self.localInterfaceInfo[ intfName ]
         printPortId( localInterface )
         printPortDesc( localInterface.interfaceDescription )
         for manAddr in self.managementAddresses:
            manAddr.render()
         printPortVlanId( localInterface.portVlanId )
         if localInterface.vlanNames:
            printVlanNames( localInterface.vlanNames,
                            localInterface.vlanNamesChanged,
                            localInterface.vlanNamesIncomplete )
         printLinkAggrInfo( localInterface, version='802.1' )
         printLinkAggrInfo( localInterface, version='802.3' )
         if localInterface._powerViaMdiTlvReceived: #pylint: disable=protected-access
            printPowerViaMdiInfo( localInterface )
         printMaxFrameSize( localInterface.maxFrameSize )
         if localInterface.medInfo:
            printMedInfo( localInterface.medInfo )
         for tlv in localInterface.unknownOrgDefinedTlvs:
            tlv.render()
         printZtpBootVlan( localInterface.ztpBootVlan )

class LldpNeighbor( Model ):
   port = Interface( help="Name of the interface" )
   neighborDevice = Str( help="Name of the neighbor device" )
   neighborPort = Str( help="Name of the neighbor port" )
   ttl = Int( help="The Time To Live value (in seconds) advertised by the"
              " remote system" )

class LldpNeighbors( Model ):
   tablesLastChangeTime = Float( help="The last time tables contained in the"
         " LLDP remote systems MIB are updated", optional=True )
   tablesInserts = Int( help="The number of times the information advertised"
         " by a MSAP has been inserted into tables contained in the LLDP"
         " remote systems MIB" )
   tablesDeletes = Int( help="The number of times the information advertised"
         " by a MSAP has been deleted from tables contained in the LLDP"
         " remote systems MIB" )
   tablesDrops = Int( help="The number of times the information advertised"
         " by a MSAP could not be entered into tables contained in the LLDP"
         " remote systems MIB" )
   tablesAgeOuts = Int( help="The number of times the information advertised"
         " by a MSAP has been deleted from tables contained in the LLDP"
         " remote systems MIB because the information timeliness interval has"
         " expired" )

   lldpNeighbors = List( valueType=LldpNeighbor, help="Lldp Neighbors" )

   def render( self ):
      if self.tablesLastChangeTime is not None:
         lastChangeTimeStr = Ark.timestampToStr(
               self.tablesLastChangeTime - Tac.utcNow() + Tac.now() )
      else:
         lastChangeTimeStr = 'never'

      print( 'Last table change time   : %s' % lastChangeTimeStr )
      print( 'Number of table inserts  : %d' % self.tablesInserts )
      print( 'Number of table deletes  : %d' % self.tablesDeletes )
      print( 'Number of table drops    : %d' % self.tablesDrops )
      print( 'Number of table age-outs : %d' % self.tablesAgeOuts )

      print()
      table = TableOutput.createTable( [ 'Port',
                                         'Neighbor Device ID',
                                         'Neighbor Port ID',
                                         'TTL' ] )
      table.formatColumns( firstColumnFormat,
                           defaultColumnFormat,
                           defaultColumnFormat,
                           defaultColumnFormat )

      for neighbor in self.lldpNeighbors:
         table.newRow( CliPlugin.IntfCli.Intf.getShortname( neighbor.port ),
                       neighbor.neighborDevice,
                       neighbor.neighborPort,
                       neighbor.ttl )
      print( table.output() )

class NeighborInterfaceInfo( LldpInterfaceInfo ):
   portAndProtocolVlanSupported = Dict( keyType=int, valueType=bool,
         help="A mapping between Port And Protocol VLAN ID and whether"
         " it is supported" )
   portAndProtocolVlanEnabled = Dict( keyType=int, valueType=bool,
         help="A mapping between Port And Protocol VLAN ID and whether"
         " it is enabled" )
   protocolIdentityInfo = List( valueType=str,
         help="Protocols enabled on neighbor interface" )
   vidUsageDigest = Int( help="VID usage Digest associated with the neighbor",
         optional=True )
   managementVid = Int( help="Management vlan id", optional=True )
   autoNegCapability = Enum( values=autoNegCapabilityTypeValues,
         help="Auto-negotiation capability of neighbor interface", optional=True )
   autoNegAdvertisedCapabilities = List( valueType=str,
         help="List of auto-negotiation capabilities advertised on neighbor"
         " interface" )
   operMauType = Str( help="Operational Medium Attachment Unit type of the"
         " neighbor interface", optional=True )

   unknownTlvs = List( valueType=UnknownTlv,
         help="List of Unknown Tlvs which are outside the standard TLVs defined"
         " by LLDP protocol" )

class LldpNeighborInfo( LldpDevice ):
   lastContactTime = Float( help="The UTC time at which the neighbor"
         " last communicated with the system" )
   neighborDiscoveryTime = Float( help="The UTC time at which the neighbor was"
         " created" )
   lastChangeTime = Float( help="The UTC time at which the neighbor entry"
         " was last updated" )
   ttl = Int( help="The Time To Live value (in seconds) advertised by the"
         " neighbor" )
   managementAddresses = List( valueType=ManagementAddr,
         help="List of management addresses ordered by priority" )
   neighborInterfaceInfo = Submodel( valueType=NeighborInterfaceInfo,
         help="Information of neighbor interface like interface id, description"
         " along with all the TLVs received on it" )

class LldpNeighborsList( Model ):
   lldpNeighborInfo = List( valueType=LldpNeighborInfo,
         help="List of LLDP neighbors' information" )

class LldpNeighborsDetail( Model ):
   lldpNeighbors = Dict( keyType=Interface, valueType=LldpNeighborsList,
         help="A mapping between local interface and neighbors discovered"
         " on that interface" )

   def printMandatoryTlvs( self, neighbor, ttl ):
      printChassisId( neighbor.chassisIdType, neighbor.chassisId )

      interfaceInfo = neighbor.neighborInterfaceInfo
      printPortId( interfaceInfo )
      print( prefix + '- Time To Live: %d seconds' % ttl )

   def printOptionalTlvs( self, neighbor ):
      emptyValuePrinted = False
      if neighbor.neighborInterfaceInfo.interfaceDescription is not None:
         emptyValuePrinted |= printPortDesc(
                              neighbor.neighborInterfaceInfo.interfaceDescription )
      if neighbor.systemName is not None:
         emptyValuePrinted |= printSystemName( neighbor.systemName )
      if neighbor.systemDescription is not None:
         emptyValuePrinted |= printSystemDesc( neighbor.systemDescription )
      if neighbor.systemCapabilities:
         ( sysCapsSupported, sysCapsEnabled ) = neighbor.getSystemCapabilities()
         printSystemCapabilities( sysCapsSupported, sysCapsEnabled )
      for manAddr in neighbor.managementAddresses:
         manAddr.render()
      return emptyValuePrinted

   def printOrgDefinedTlvs( self, neighbor ):
      emptyValuePrinted = False
      if neighbor.portVlanId is not None:
         printPortVlanId( neighbor.portVlanId )
      if neighbor.portAndProtocolVlanSupported:
         printPortAndProtocolVlan( neighbor )
      if neighbor.vlanNames:
         printVlanNames( neighbor.vlanNames )
      if neighbor.protocolIdentityInfo:
         printProtocolIdentityInfo( neighbor.protocolIdentityInfo )
      if neighbor.vidUsageDigest is not None:
         printVidUsageDigest( neighbor.vidUsageDigest )
      if neighbor.managementVid is not None:
         printManagementVid( neighbor.managementVid )
      if neighbor.linkAggregation8021Status:
         printLinkAggrInfo( neighbor, version='802.1' )
      if neighbor.linkAggregation8023Status:
         printLinkAggrInfo( neighbor, version='802.3' )
      if neighbor.autoNegCapability:
         printMacPhyConfigStatus( neighbor )
      if neighbor._powerViaMdiTlvReceived: #pylint: disable=protected-access
         printPowerViaMdiInfo( neighbor )
      if neighbor.maxFrameSize is not None:
         printMaxFrameSize( neighbor.maxFrameSize )
      printZtpBootVlan( neighbor.ztpBootVlan )
      if neighbor.medInfo:
         emptyValuePrinted |= printMedInfo( neighbor.medInfo )

      return emptyValuePrinted

   def render( self ):
      emptyValuePrinted = False
      for interface in sortIntf( self.lldpNeighbors ):
         neighborDetails = self.lldpNeighbors[ interface ]
         print( 'Interface', interface, 'detected',
               len( neighborDetails.lldpNeighborInfo ), 'LLDP neighbors:' )
         # Print each neighbor info
         for neighborInfo in neighborDetails.lldpNeighborInfo:
            print()
            print( '  Neighbor %s/%s, age %d seconds'
                  % ( neighborInfo.chassisId,
                      neighborInfo.neighborInterfaceInfo.interfaceId,
                      int( round( Tac.utcNow() -
                           neighborInfo.lastContactTime ) ) ) )
            print( '  Discovered {}; Last changed {}'.format(
                  Ark.timestampToStr( neighborInfo.neighborDiscoveryTime -
                        Tac.utcNow() + Tac.now() ),
                  Ark.timestampToStr( neighborInfo.lastChangeTime -
                        Tac.utcNow() + Tac.now() ) ) )

            self.printMandatoryTlvs( neighborInfo, neighborInfo.ttl )
            emptyValuePrinted |= self.printOptionalTlvs( neighborInfo )
            emptyValuePrinted |= self.printOrgDefinedTlvs(
                                       neighborInfo.neighborInterfaceInfo )

            for unknownTlv in neighborInfo.neighborInterfaceInfo.unknownTlvs:
               unknownTlv.render()
            for unknownOrgDefTlv in \
                  neighborInfo.neighborInterfaceInfo.unknownOrgDefinedTlvs:
               unknownOrgDefTlv.render()
         print()
      if emptyValuePrinted:
         print( '* (empty) indicates that the TLV was received but without content' )
         print()

class LldpInterfaceTraffic( Model ):
   txFrames = Int( help="The number of LLDP frames transmitted" )
   txFramesLengthExceeded = Int( help="The number of LLDP frames"
         " transmitted on this port that could not hold all the desired"
         " information" )
   rxFrames = Int( help="The number of LLDP frames received" )
   rxErrors = Int( help="The number of LLDP frames received with one"
         " or more detectable errors" )
   rxDiscards = Int( help="The number of LLDP frames received and"
         " then discarded" )
   tlvsDiscarded = Int( help="The number of LLDP TLVs discarded" )
   tlvsUnknown = Int( help="The number of LLDP TLVs received on this port that"
         " are not recognized" )

class LldpTraffic( Model ):
   interfaces = Dict( valueType=LldpInterfaceTraffic,
         help="Lldp Interface Counters" )

   def render( self ):
      print()
      table = TableOutput.createTable( [ 'Port',
                                         'Tx Frames',
                                         'Tx Length Exceeded' ] )
      table.formatColumns( firstColumnFormat,
                           defaultColumnFormat,
                           defaultColumnFormat )

      for intfName in sortIntf( self.interfaces ):
         intfTraffic = self.interfaces[ intfName ]
         table.newRow( CliPlugin.IntfCli.Intf.getShortname( intfName ),
                       intfTraffic.txFrames,
                       intfTraffic.txFramesLengthExceeded )
      print( table.output() )

      table = TableOutput.createTable( [ 'Port',
                                         'Rx Frames',
                                         'Rx Errors',
                                         'Rx Discard',
                                         'TLVs Discard',
                                         'TLVs Unknown' ] )
      table.formatColumns( firstColumnFormat,
                           defaultColumnFormat,
                           defaultColumnFormat,
                           defaultColumnFormat,
                           defaultColumnFormat,
                           defaultColumnFormat )

      for intfName in sortIntf( self.interfaces ):
         intfTraffic = self.interfaces[ intfName ]
         table.newRow( CliPlugin.IntfCli.Intf.getShortname( intfName ),
                       intfTraffic.rxFrames,
                       intfTraffic.rxErrors,
                       intfTraffic.rxDiscards,
                       intfTraffic.tlvsDiscarded,
                       intfTraffic.tlvsUnknown )
      print( table.output() )

if __name__ == "__main__":
   from doctest import run_docstring_examples
   # pylint: disable-next=consider-using-with
   mySource = open( __file__ ).read().replace( ";;;", ">" * 3 )
   run_docstring_examples( mySource, globals(), name=__file__, verbose=True )
