# Copyright (c) 2006-2011, 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import os
import sys
import subprocess

import Ark
import Arnet
import BasicCli
import BasicCliUtil
import Cell
import CliCommand
import CliMatcher
import CliPlugin.TechSupportCli
import CliPlugin.VrfCli as VrfCli # pylint: disable=consider-using-from-import
from CliPlugin import IntfCli
from CliPlugin.IpGenAddrMatcher import IpGenAddrMatcher
import CliToken.Terminal
import CliToken.Clear
from CliToken.Logging import loggingForConfig
import CliToken.Service
import CliParser
import ConfigMount
import DscpCliLib
import HostnameCli
import IpUtils
import LazyMount
import Logging
import LoggingDefs
import LoggingLib
import ShowCommand
import UtmpDump
import Tac
from TerminalUtil import isConsole, ttyHook
import TimeRangeRule

# pkgdeps: rpm MatchList-lib

#########
#
# This module provides CLI commands related to configuring logging.
# Look for "Implemented" to get the list of currently implemented
# commands.

tacLoggingConfig = None
sysLoggingConfig = None
sysLoggingConfigReq = None
sysLoggingStatus = None
loggingHostType = Tac.Type( "LogMgr::LoggingHost" )
defaultVrf = None
dscpConfig = None
sslConfig = None
matchListConfig = None

def boolToEnabledDisabledString( theBool ):
   if theBool:
      return "enabled"
   else:
      return "disabled"

######### "show logging ..." #########
#
# Implemented
# -----------
#
# show logging
#
# show logging [threshold] <severity-level>
#
# show logging system
#    => print the contents of the system log buffer at /var/log/messages
#
#
# Maybe do later
# --------------
#
# show logging count
#    => print per-message counts, grouped by facility
#
# show logging history
#    => print coarse statistics of logged msgs
#
# show logging summary
#    => print per-slot per-facility counts
#
# show logging xml
#    => print status and messages in XML format

# Helper to show logging configuration.
# The configuration information is not displayed when option to show last N lines is
# given.
def showLoggingConfiguration( mode, sysCfg, sysStatus ):
   print( "Syslog logging:", boolToEnabledDisabledString( sysCfg.loggingOn ) )

   # Buffered logging
   if sysCfg.buffered:
      sevStr = LoggingLib.severityTaccToCli( sysCfg.bufferedSeverity )
      bufferLoggingStr = "level " + sevStr
   else:
      bufferLoggingStr = boolToEnabledDisabledString( sysCfg.buffered )
   print( "Buffer logging: %s" % bufferLoggingStr )

   # Logging to the system console
   consoleLoggingStr = "Console logging: "
   consoleStatus = ''
   if not sysCfg.loggingConsoleEnabled:
      consoleStatus = boolToEnabledDisabledString(
         sysCfg.loggingConsoleEnabled )
   else:
      consoleStatus = "level " + \
          LoggingLib.severityTaccToCli( sysCfg.loggingConsoleSeverity )
   print( consoleLoggingStr + consoleStatus )

   # Logging persistent
   if sysCfg.persistentLog:
      sevStr = LoggingLib.severityTaccToCli( sysStatus.severityThreshold )
      persistentLoggingStr = "level " + sevStr
   else:
      persistentLoggingStr = boolToEnabledDisabledString( sysCfg.persistentLog )
   print( "Persistent logging: %s" % persistentLoggingStr )

   # Logging monitor
   #
   # By default, logging monitor uses the same setting as logging console.
   monitorLoggingStr = "Monitor logging: "
   if sysCfg.loggingMonitorUseConsole:
      monitorStatus = consoleStatus
   else:
      if not sysCfg.loggingMonitorEnabled:
         monitorStatus = boolToEnabledDisabledString(
            sysCfg.loggingMonitorEnabled )
      else:
         monitorStatus = "level " + \
             LoggingLib.severityTaccToCli( sysCfg.loggingMonitorSeverity )
   print( monitorLoggingStr + monitorStatus )

   # Synchronous logging to the console
   synchronousLoggingStr = "Synchronous logging: "
   if not sysCfg.loggingSynchronousEnabled:
      print( synchronousLoggingStr +
             boolToEnabledDisabledString( sysCfg.loggingSynchronousEnabled ) )
   else:
      if sysCfg.loggingSynchronousAll:
         print( synchronousLoggingStr + "all" )
      else:
         print( synchronousLoggingStr + "level " +
                LoggingLib.severityTaccToCli( sysCfg.loggingSynchronousSeverity ) )

   # Trap logging (syslog across the network)
   trapLoggingStr = "Trap logging: "
   # pylint: disable=too-many-nested-blocks
   if not sysCfg.loggingTrapEnabled:
      print( trapLoggingStr +
             boolToEnabledDisabledString( sysCfg.loggingTrapEnabled ) )
   else:
      print( trapLoggingStr + "level " +
             LoggingLib.severityTaccToCli( sysCfg.loggingTrapSeverity ) )

      # Print logging source-address, source-interface, and hosts by vrf starting
      # with default
      vrfList = list( set( sysCfg.srcIpAddress ) |
                      set( sysCfg.srcIp6Address ) |
                      set( sysCfg.srcIntfName ) |
                      set( sysCfg.vrfLoggingHost ) )
      addDefault = False
      if defaultVrf in vrfList:
         addDefault = True
         vrfList.remove( defaultVrf )
      vrfList.sort()
      if addDefault:
         vrfList.insert( 0, defaultVrf )

      statusUnavailableString = ' (unavailable)'
      configErrorString = ' (config error)'
      for vrf in vrfList:
         statusString = ''
         if vrf in sysCfg.srcIpAddress or vrf in sysCfg.srcIp6Address:
            vrfString = ' in VRF %s' % vrf
            intfName, intfString, intf6Name, intf6String = '', '', '', ''
            if vrf in sysCfg.srcIpAddress:
               errorString = ''
               intfName = sysStatus.srcIpIntfName.get( vrf )
               if intfName:
                  intfString = ", assigned to interface %s" % intfName
               else:
                  intfString = ", address unassigned to any interface"
                  errorString = configErrorString
                  if vrf in sysCfg.vrfLoggingHost:
                     statusString = statusUnavailableString
               print( "    Logging source-address '%s'%s%s%s%s" %
                      ( sysCfg.srcIpAddress[ vrf ], statusString,
                        intfString, vrfString, errorString ) )
            if vrf in sysCfg.srcIp6Address:
               errorString = ''
               intf6Name = sysStatus.srcIp6IntfName.get( vrf )
               if intf6Name:
                  intf6String = ", assigned to interface %s" % intf6Name
               else:
                  intf6String = ", address unassigned to any interface"
                  errorString = configErrorString
                  if vrf in sysCfg.vrfLoggingHost:
                     statusString = statusUnavailableString
               print( "    Logging source-address '%s'%s%s%s%s" %
                      ( sysCfg.srcIp6Address[ vrf ], statusString,
                        intf6String, vrfString, errorString ) )
         elif vrf in sysCfg.srcIntfName:
            vrfString, ipString, ip6String = ' in VRF %s' % vrf, '', ''

            ipSrcStatus = sysStatus.srcIntfStatus.get( vrf )
            if ipSrcStatus and ipSrcStatus.activeAddrWithMask.address != '0.0.0.0':
               ipString = ", IP Address %s" % ipSrcStatus.activeAddrWithMask.address

            intf6Status = sysStatus.srcIntf6Status.get( vrf )
            if intf6Status and list( intf6Status.addr ):
               ip6SrcAddr = list( intf6Status.addr )[ 0 ].address.stringValue
               ip6String = ", IPv6 Address %s" % ip6SrcAddr

            errorString = ""
            if not ipString and not ip6String:
               ipString = ", no IP address assigned to interface"
               errorString = configErrorString
               if vrf in sysCfg.vrfLoggingHost:
                  statusString = statusUnavailableString

            print( "    Logging source-interface '%s'%s%s%s%s%s" %
                   ( sysCfg.srcIntfName[ vrf ], statusString,
                     ipString, ip6String, vrfString, errorString ) )

         if vrf in sysCfg.vrfLoggingHost:
            # Print sorted IP Addresses followed by sorted DNS names of hosts
            # to which we are logging
            loggingAddrs = []
            loggingNames = []
            vrfString = ' in VRF %s' % vrf
            loggingHost = sysCfg.vrfLoggingHost[ vrf ].loggingHost
            for lh in loggingHost:
               nameOrIpAddrStr = loggingHost[ lh ].ipAddrOrHostname
               try:
                  Arnet.IpAddress( nameOrIpAddrStr )
                  loggingAddrs.append( nameOrIpAddrStr )
               except ValueError:
                  loggingNames.append( nameOrIpAddrStr )

            loggingAddrs.sort( key=IpUtils.IpAddress )
            for addrStr in loggingAddrs:
               ports = loggingHost[ addrStr ].ports
               protocol = loggingHost[ addrStr ].protocol
               for port in ports:
                  print( "    Logging to '%s' port %d%s via %s" %
                         ( addrStr, port, vrfString, protocol ) )
            for nameStr in sorted( loggingNames ):
               ports = loggingHost[ nameStr ].ports
               protocol = loggingHost[ nameStr ].protocol
               for port in ports:
                  print( "    Logging to '%s' port %d%s via %s" %
                         ( nameStr, port, vrfString, protocol ) )

   seqNoString = "Sequence numbers: "
   print( seqNoString +
          boolToEnabledDisabledString( sysCfg.sequenceNumbersEnabled ) )

   # SysLog Facility
   facility = "Syslog facility: "
   print( facility + LoggingLib.facilityTaccToCli( sysCfg.loggingFacility ) )
   #Hostname logging format
   formatStr =  "Hostname format: "
   if sysCfg.hostnameFormat == 'fqdn':
      print( formatStr + "Fully qualified domain name" )
   elif sysCfg.hostnameFormat == 'ipv4':
      print( formatStr + "ipv4 format" )
   else:
      print( formatStr + "Hostname only" )

   # Relogging interval
   interval = "Repeat logging interval: "
   ri = sysCfg.repeatInterval
   if ri == 0:
      print( interval + "disabled" )
   else:
      print( interval + "%d minutes" % ( sysCfg.repeatInterval ) )

   # Repeat msgs setting
   print( "Repeat messages: %s" %
          ( boolToEnabledDisabledString( sysCfg.repeatMsgs ) ) )

   # Root login msgs setting
   print( "Root login logging: %s" %
          boolToEnabledDisabledString( sysCfg.loggingRootLogin ) )

   # Print included and non-included external configuration files
   includedFiles, nonIncludedFiles = getExternalConfigFiles()
   print( "" )
   print( "External configuration:" )
   print( "    active: %s" % ", ".join( sorted( includedFiles ) ) )
   print( "    inactive: %s" % ", ".join( sorted( nonIncludedFiles ) ) )
   if nonIncludedFiles:
      restartMsg = "! Please restart logging to make all files effective using"
      restartMsg += " \"no logging on/logging on\" or"
      restartMsg += " \"bash sudo systemctl restart rsyslog\"."
      print( "    %s" % restartMsg )

   # Display warning for errors in rsyslog configuration. For some reason
   # rsyslog may print the same error multiple times; just print unique lines
   for err in set( getConfigSyntaxErrors().splitlines() ):
      if "error" in err:
         mode.addWarning( err )

   print( "" )
   print( "Facility                   Severity            Effective Severity" )
   print( "--------------------       -------------       ------------------" )
   for name in sorted( tacLoggingConfig.facilityConfig ):
      level = sysCfg.facilityLogLevel.get( name, sysCfg.facilityLogLevelDefault )
      effectiveSev = Logging.mostSevere( level, sysStatus.severityThreshold )
      print( "%-20s       %-13s       %-13s" %
            ( name.lower(),
              LoggingLib.severityTaccToCli( level ),
              LoggingLib.severityTaccToCli( effectiveSev ) ) )

def getExternalConfigFiles():
   ''' Returns a tuple of lists, ( [active config files], [inactive config files] )
   based on which config files were modified before the rsyslog pid file was
   modified. '''
   activeFiles = []
   inactiveFiles = []
   try:
      syslogMtime = os.path.getmtime( "/var/run/syslogd.pid" )
   except OSError:
      return ( [], [] )

   for confFile in LoggingLib.allExternalConfigFiles():
      confFilePath = os.path.join( LoggingDefs.externalConfigDir, confFile )
      try:
         confFileMTime = os.path.getmtime( confFilePath )
      except OSError:
         continue
      if confFileMTime < syslogMtime:
         activeFiles.append( confFile )
      else:
         inactiveFiles.append( confFile )
   return ( activeFiles, inactiveFiles )

def getConfigSyntaxErrors():
   try:
      if LoggingLib.allExternalConfigFiles():
         Tac.run( [ "rsyslogd", "-N1" ], stderr=Tac.CAPTURE )
   except Tac.SystemCommandError as err:
      return err.output
   else:
      return ""

# pkgdeps: import EosLogUtil (this depends on the FetchLogs utility)
def genLogfileDumpProc( logFilename ):
   # match logfile and logrotated files
   match = os.path.basename( logFilename ) + r'(?=($|\..*\.gz))'
   return subprocess.Popen( [ sys.executable, '/usr/bin/FetchLogs', 'dump', '-n',
                              '-l', os.path.dirname( logFilename ),
                              '-m', match ],
                            stdout=subprocess.PIPE, universal_newlines=True )

loggingForShow = CliMatcher.KeywordMatcher( 'logging',
                                            helpdesc='Details of the system log' )

class ShowLoggingAllCommand( ShowCommand.ShowCliCommandClass ):
   syntax = "show logging all"
   data = {
      'logging' : loggingForShow,
      'all' : 'Show all the lines in the logging buffer'
   }
   @staticmethod
   def handler( mode, args ):
      sysCfg = sysLoggingConfig
      if sysCfg.buffered:
         # The "log buffer" is nothing but the collection of the current
         # and rotated log buffer files with log entries in increasing
         # order of time. To achieve this, we cat the log files starting
         # oldest first (We need to use zcat for all the rotated files
         # since they are compressed)
         p = genLogfileDumpProc( sysCfg.bufferedLogFilename )
         print( p.stdout.read() )

BasicCli.addShowCommandClass( ShowLoggingAllCommand )

@Tac.memoize
def _timeUnitMap():
   d = {}
   for k in ( 'seconds', 'minutes', 'hours', 'days' ):
      d[ k ] = 'Show messages in last <N> ' + k
   return d

class ShowLoggingCommand( ShowCommand.ShowCliCommandClass ):
   syntax = "show logging [ { LASTLINE | " \
            "( ( last LASTTIME UNIT ) | TIMERANGE ) | " \
            "( [ threshold ] SEVERITY ) } ]"

   # make sure 'last' and 'time-range' are exclusive
   __sharedTimeObj__ = object()
   data = {
      'logging' : loggingForShow,
      'LASTLINE' : CliCommand.singleNode( CliMatcher.IntegerMatcher(
         1, 9999,
         helpdesc='Show last number of messages in the logging buffers' ) ),
      'last' : CliCommand.Node( CliMatcher.KeywordMatcher(
         'last', helpdesc='Show messages in last <N> time-units' ),
                                maxMatches=1,
                                sharedMatchObj=__sharedTimeObj__ ),
      'LASTTIME' : CliMatcher.IntegerMatcher(
         1, 9999, helpdesc='Number of time units (sec|min|hr|day)' ),
      'UNIT' : CliMatcher.DynamicKeywordMatcher( lambda mode: _timeUnitMap() ),
      'TIMERANGE' : TimeRangeRule.timeRangeExpression(
         'TIMERANGE', maxMatches=1, sharedMatchObj=__sharedTimeObj__ ),
      'threshold' : CliCommand.singleKeyword(
         'threshold',
         helpdesc='Show only log messages at threshold level or above' ),
      'SEVERITY' : CliCommand.singleNode(
         CliMatcher.DynamicKeywordMatcher(
            lambda mode: dict( LoggingLib.cliSeverities ) ) )
      }

   @staticmethod
   def adapter( mode, args, argsList ):
      # Order of the filters matters. For example, filter by time-range first
      # than take the last 5 lines is differnt from first taking the last 5 lines
      # and then do a time-range filter.
      showLogOptions = []
      for arg, value in argsList:
         if arg in ( 'LASTLINE', 'LASTTIME', 'time-range', 'SEVERITY' ):
            if arg == 'LASTTIME':
               value = ( value, args[ 'UNIT' ][ 0 ] )
            elif arg == 'SEVERITY':
               value = ( 'threshold' in args, value )
            elif arg == 'time-range':
               # note we cannot use TIMERANGE as it's generated by timeRange's
               # adapter so not available in argsList. Instead we use the keyword
               # as order.
               arg = 'TIMERANGE'
               value = args[ 'TIMERANGE' ]
            showLogOptions.append( ( arg, value ) )

      if showLogOptions:
         args[ 'OPTIONS' ] = showLogOptions

   @staticmethod
   def handler( mode, args ):
      sysCfg = sysLoggingConfig
      sysStatus = sysLoggingStatus

      if not sysCfg.buffered:
         return

      # We are not concerned about the errors of  file's existence while
      # printing them to console.
      # The reason is that file might not be present in machine/duts
      # e.g. file is missing currently in etba dut
      # Doing so we choose to ignore all previous errors but we will track
      # all future errors.

      showLogOptions = args.get( 'OPTIONS' )

      # Handle 'show logging'
      if not showLogOptions:
         #if 'show logging' is issued then print the configuration information
         showLoggingConfiguration( mode, sysCfg, sysStatus )
         clearTime = sysLoggingStatus.lastClearLogRequest.timestamp
         if clearTime:
            print( "\nLast clearing of system log messages %s by %s" %
               ( Ark.timestampToStr( clearTime ),
                 sysLoggingStatus.lastClearLogRequest.who ) )

         # the number of lines to print is the size of the logging buffer, if
         # not specified by user
         numLastLines = sysCfg.bufferMessageCount

         print( "\nLog Buffer:" )

         # Need to flush out stdout at this point (i.e. the logging
         # configuration) before we tail the log buffer.
         # This ensures that the log buffer always follows the log configuration
         # output.
         sys.stdout.flush()

         showLogOptions = ( ( 'LASTLINE', numLastLines ), )

      # get all the args for needed filters
      filterArgs = []
      for name, value in showLogOptions:
         if name == 'LASTLINE':
            # Handle 'show logging N'
            cmdArgs = [ "tail", "-n", "%d" % value ]
         elif name == 'LASTTIME':
            # Handle 'show logging last N <sec|min|hour|day>'
            # time-filter -q --seconds=N, for instance.
            cmdArgs = [ "time-filter", "-q", "--" + value[ 1 ] + "=" +
                        str( value[ 0 ] ) ]
         elif name == 'TIMERANGE':
            # Handle 'show logging time-range <begin> <end>'
            # <begin>,<end> is a tuple ((year,month,day),(hour,min,sec))
            beginDate = value[ 0 ][ 0 ]
            beginTime = value[ 0 ][ 1 ]
            endDate = value[ 1 ][ 0 ]
            endTime = value[ 1 ][ 1 ]

            # Convert them to string arguments for time-filter
            # By not converting the time to datetime objects here but in
            # time-filter, we allow human readable inputs
            # for the script as well
            beginRange = TimeRangeRule.datetimeToStr( mode, beginDate, beginTime )
            endRange = TimeRangeRule.datetimeToStr( mode, endDate, endTime )

            if beginRange == "" or endRange == "":
               # nothing to print
               return
            cmdArgs = [ "time-filter", "-q", "-b", beginRange, "-e", endRange ]
         elif name == 'SEVERITY':
            threshold, level = value
            if threshold:
               cmdArgs = [ "sev-filter", "-t", level ]
            else:
               cmdArgs = [ "sev-filter", level ]
         else:
            return

         filterArgs.append( cmdArgs )

      # now run all the filters
      curProc = genLogfileDumpProc( sysCfg.bufferedLogFilename )

      for cmdArgs in filterArgs:
         # Pipe the output of the previous command to the current commmand
         nxtProc = subprocess.Popen( cmdArgs, # pylint: disable=consider-using-with
                                     stdin=curProc.stdout,
                                     stdout=subprocess.PIPE,
                                     universal_newlines=True )
         curProc.stdout.close()
         curProc = nxtProc

      # pipe everything to sed so we always print a final newline
      # pylint: disable-next=consider-using-with
      sedProc = subprocess.Popen( [ "sed", "$a\\" ],
                                  stdin=curProc.stdout, universal_newlines=True )
      curProc.stdout.close()
      sedProc.wait()


BasicCli.addShowCommandClass( ShowLoggingCommand )

def showLoggingFile( mode, filepath, numLastLines=None ):
   """
   Output the ( numLastLines ) lines of filepath.
   """
   # pylint: disable-next=consider-using-with
   catStdout = subprocess.Popen( [ "sudo", "cat", filepath ],
                                 stderr=sys.stderr,
                                 stdout=subprocess.PIPE, universal_newlines=True )
   if numLastLines:
      # pylint: disable-next=consider-using-with
      tailProc = subprocess.Popen( ['tail', "-n", str( numLastLines ) ],
                                   stdin=catStdout.stdout,
                                   stdout=subprocess.PIPE, universal_newlines=True )
      # We use print here because sys.stdout doesn't always
      # redirect correctly to the intended stdout
      print( tailProc.stdout.read() )
   else:
      print( catStdout.stdout.read() )

class ShowLoggingFile( ShowCommand.ShowCliCommandClass ):
   syntax = "show logging system | mce [ LAST ]"
   data = {
      "logging" : loggingForShow,
      "system" : 'Show the contents of the system log buffer',
      "mce" : 'Show the contents of the mcelog buffer',
      "LAST" : CliMatcher.IntegerMatcher(
         1, sys.maxsize,
         helpdesc='Show the last number of lines in the logging buffer' )
      }
   @staticmethod
   def handler( mode, args ):
      numLastLines = args.get( 'LAST' )
      if 'system' in args:
         filename = "/var/log/messages"
      else:
         filename = "/var/log/mcelog"
      showLoggingFile( mode, filename, numLastLines )

BasicCli.addShowCommandClass( ShowLoggingFile )

@BasicCliUtil.EapiIncompatible()
def showLoggingFollow( mode ):
   """Start following /var/log/eos"""

   if not sysLoggingConfig.buffered:
      return
   cmd = [ "LogMonUtil", "-n", str( sysLoggingConfig.bufferMessageCount ),
           sysLoggingConfig.bufferedLogFilename ]
   try:
      Tac.run( cmd,
               stderr=sys.stderr,
               stdout=sys.stdout )
   except Tac.SystemCommandError as e:
      if e.error == -2:
         # Ctrl-C, ignore
         return
      mode.addError( "Could not tail %s (%s)" %
                     ( sysLoggingConfig.bufferedLogFilename, e ) )

class ShowLoggingFollow( ShowCommand.ShowCliCommandClass ):
   syntax = "show logging follow"
   data = {
      "logging" : loggingForShow,
      "follow" : 'Keep following the log buffer as it grows',
      }
   noMore = True

   @staticmethod
   def handler( mode, args ):
      showLoggingFollow( mode )

BasicCli.addShowCommandClass( ShowLoggingFollow )

class ShowLoggingPersistent( ShowCommand.ShowCliCommandClass ):
   syntax = "show logging persistent"
   data = {
      "logging": loggingForShow,
      "persistent" : "Show persistent logging messages stored on flash disk",
      }

   @staticmethod
   def handler( mode, args ):
      cfg = getSysLogConfig()
      showLoggingFile( mode, cfg.persistentLogFilename )

BasicCli.addShowCommandClass( ShowLoggingPersistent )

######### "logging ..." #########
#
# Implemented
# -----------
#
# [no] logging on
#    => enable/disable all logging
#
# logging level <facility> <severity>
#    => set the minimum logging threshold for the given facility
#
# [no] logging buffered [ <len> ] [ <sev> ]
#    => set the size of the logging buffer and/or the minimum logging
#       threshold for messages to be written into the logging buffer.
#
# [no] logging host <ip-addr|hostname> [ protocol <tcp|udp> ]
#    => set an ip address/hostname of a syslog server to which to log
#
# [no] logging trap [severity]
#    => enable/disable logging to syslog servers across the network,
#       and set the minimum threshold for messages to be sent.
#
# [no] logging console [severity]
#    => enable/disable logging to the console,
#       and set the minimum threshold for messages to be logged.
#
# [no] logging monitor [severity]
#    => enable/disable logging to monitoring terminals,
#       and set the minimum threshold for messages to be logged.
#
# [no] logging facility <facility>
#    => set syslog facility to which messages are syslogged across the network
#
# [no] logging source-interface <ifacename>
#    => specify source IP addr of msgs syslogged across the network
#
# [no] logging source-address <ip-addr>
#    => sepcify source IP addr of msgs syslogged across the network
#
# [no] logging synchronous [ level <severity> | all ]
#    => don't intermingle log messages with 'solicited output'
#
# Maybe do later
# --------------
#
# [no] logging buffered xml
#    => log messages in XML format. Wouldn't be hard to do.
#
# [no] logging count
#    => keep per-message counts and last-logged timestamps. Not hard to do.
#
# [no] logging history size <msgcount> <severity>
#    => size of buffer for syslog messages sent to an SNMP server (???)
#
# [no] logging origin-id ...
#    => add the specified identifier to all log messages
#
# [no] logging rate-limit { <number>|all|console } [ except <severity> ]
#    => rate limit log messages
#
# No plans to implement
# ---------------------
#
# [no] logging buffered filtered <severity>
#    => filters implemented in TCL scripts
#
# [no] logging filter <url> ...
#    => specify location of logging filter TCL script
#
# [no] logging linecard
#    => we don't have linecards
#

def getSysLogConfig():
   return sysLoggingConfig

def cmdLoggingLevel( mode, facility, severity ):
   cfg = getSysLogConfig()

   if facility == 'all':
      facilityList = tacLoggingConfig.facilityConfig
   else:
      facilityList = [ facility ]
   for fac in facilityList:
      cfg.facilityLogLevel[ fac ] = severity

def cmdNoLoggingLevel( mode, facility ):
   cfg = getSysLogConfig()
   if facility == 'all':
      cfg.facilityLogLevel.clear()
   else:
      del cfg.facilityLogLevel[ facility ]

def cmdLoggingLevelDefault( mode, severity ):
   cfg = getSysLogConfig()
   cfg.facilityLogLevelDefault = severity

def cmdNoLoggingLevelDefault( mode ):
   cfg = getSysLogConfig()
   cfg.facilityLogLevelDefault = cfg.tacLogLevelDefault

def cmdLoggingBuffered( mode, bufferMsgCount=None, severity=None ):
   cfg = getSysLogConfig()
   cfg.buffered = True

   if bufferMsgCount is not None:
      cfg.bufferMessageCount = bufferMsgCount

   if severity is not None:
      cfg.bufferedSeverity = severity

   # If neither severity nor message count have been
   # specified, we revert to the default
   if severity is None and bufferMsgCount is None:
      cfg.bufferedSeverity = cfg.bufferedSeverityDefault
      cfg.bufferMessageCount = cfg.bufferMessageCountDefault

def cmdNoLoggingBuffered( mode ):
   cfg = getSysLogConfig()
   cfg.buffered = False

   # The industry standard resets the severity and buffer size
   # to their defaults when buffered logging is disabled.
   #
   # NOTE: this means that if someone diddles with the TAC
   #       attributes under the covers, those changes are not
   #       saved by the CLI save code.
   cfg.bufferedSeverity = cfg.bufferedSeverityDefault
   cfg.bufferMessageCount = cfg.bufferMessageCountDefault

def cmdLoggingConsole( mode, severity=None ):
   cfg = getSysLogConfig()
   cfg.loggingConsoleEnabled = True

   if severity is None:
      cfg.loggingConsoleSeverity = cfg.loggingConsoleSeverityDefault
   else:
      cfg.loggingConsoleSeverity = severity

def cmdNoLoggingConsole( mode ):
   cfg = getSysLogConfig()
   cfg.loggingConsoleEnabled = False

   # The industry standard resets the severity and buffer size
   # to their defaults when buffered logging is disabled.
   #
   # NOTE: this means that if someone diddles with the TAC
   #       attributes under the covers, those changes are not
   #       saved by the CLI save code.
   cfg.loggingConsoleSeverity = cfg.loggingConsoleSeverityDefault

def cmdLoggingMonitor( mode, severity=None ):
   cfg = getSysLogConfig()
   cfg.loggingMonitorEnabled = True

   if severity is None:
      cfg.loggingMonitorSeverity = cfg.loggingMonitorSeverityDefault
   else:
      cfg.loggingMonitorSeverity = severity

   cfg.loggingMonitorUseConsole = False

def cmdNoLoggingMonitor( mode ):
   cfg = getSysLogConfig()
   cfg.loggingMonitorEnabled = False

   # The industry standard resets the severity and buffer size
   # to their defaults when buffered logging is disabled.
   #
   # NOTE: this means that if someone diddles with the TAC
   #       attributes under the covers, those changes are not
   #       saved by the CLI save code.
   cfg.loggingMonitorSeverity = cfg.loggingMonitorSeverityDefault
   cfg.loggingMonitorUseConsole = False

def cmdDefaultLoggingMonitor( mode ):
   cfg = getSysLogConfig()
   cfg.loggingMonitorUseConsole = cfg.loggingMonitorUseConsoleDefault
   # Set others to default
   cfg.loggingMonitorEnabled = False
   cfg.loggingMonitorSeverity = cfg.loggingMonitorSeverityDefault

def cmdLoggingTrap( mode, severity=None ):
   cfg = getSysLogConfig()
   cfg.loggingTrapEnabled = True

   if severity is None:
      cfg.loggingTrapSeverity = cfg.loggingTrapSeverityDefault
   else:
      cfg.loggingTrapSeverity = severity

def cmdNoLoggingTrap( mode ):
   cfg = getSysLogConfig()
   cfg.loggingTrapEnabled = False

   # The industry standard resets the severity threshold to its
   # default when 'logging trap' is disabled.
   #
   # NOTE: this means that if someone diddles with the TAC
   #       attributes under the covers, those changes are not
   #       saved by the CLI save code.
   cfg.loggingTrapSeverity = cfg.loggingTrapSeverityDefault

def cmdLoggingSystem( loggingSystem, kwargs ):
   if kwargs:
      config = LoggingLib.logSystemValue( **kwargs )
      loggingSystem[ config ] = True

def cmdNoLoggingSystem( loggingSystem, kwargs ):
   if kwargs:
      config = LoggingLib.logSystemValue( **kwargs )
      del loggingSystem[ config ]
   else:
      loggingSystem.clear()

def cmdLoggingHost( mode, vrfName, ipAddrOrHostname, ports, protocol,
                    sslProfile, overwrite ):
   cfg = getSysLogConfig()

   if not overwrite:
      if not ports:
         mode.addError( "Ports not specified" )
         return

   if not ports:
      ports = [ loggingHostType().portDefault ]

   if protocol is None:
      protocol = loggingHostType().protocolDefault

   # Do a sanity check on the hostname or IP address that the user
   # entered. If it doesn't appear to be legal, print a warning,
   # but don't reject the entry. (The specified hostname may not
   # yet have been configured in DNS, for example.)
   HostnameCli.resolveHostname( mode, ipAddrOrHostname, doWarn=True )

   # Add the host to the list of logging hosts
   if not vrfName:
      vrfName = defaultVrf
   if vrfName not in cfg.vrfLoggingHost:
      vrfHostInfo = cfg.vrfLoggingHost.newMember( vrfName )
   else:
      vrfHostInfo = cfg.vrfLoggingHost[ vrfName ]

   if not overwrite and ipAddrOrHostname in vrfHostInfo.loggingHost:
      ports.extend ( vrfHostInfo.loggingHost[ ipAddrOrHostname ].ports )

   currProto = None
   if ipAddrOrHostname in vrfHostInfo.loggingHost:
      for port in vrfHostInfo.loggingHost[ ipAddrOrHostname ].ports:
         currProto = vrfHostInfo.loggingHost[ ipAddrOrHostname ].protocol
         break
   if not overwrite and currProto is not None and currProto != protocol:
      mode.addWarning( "Protocol %s has already been configured. " % currProto + \
                       "Only one protocol type is allowed on the given host" )
      return

   portsDict = {}
   for port in ports:
      portsDict[ port ] = port

   loggingHost = Tac.Value( "LogMgr::LoggingHost",
                            ipAddrOrHostname=ipAddrOrHostname,
                            protocol=protocol,
                            ports=portsDict,
                            sslProfile=sslProfile )
   vrfHostInfo.loggingHost.addMember( loggingHost )

   updateDscpRules()

def cmdNoLoggingHost( mode, vrfName, ipAddrOrHostname, ports ):
   cfg = getSysLogConfig()
   if not vrfName:
      vrfName = defaultVrf

   # delete the logging host from the list
   if vrfName not in cfg.vrfLoggingHost:
      if vrfName != defaultVrf:
         mode.addWarning( "Host %s not found for logging in VRF %s" %
                          ( ipAddrOrHostname, vrfName ) )
      else:
         mode.addWarning( "Host %s not found for logging" %
                               ipAddrOrHostname )
      return
   # pylint: disable-msg=W0612, R1702
   if ipAddrOrHostname not in cfg.vrfLoggingHost[ vrfName ].loggingHost:
      mode.addWarning( "Host %s not found for logging" % ipAddrOrHostname )
   else:
      if ports == 'all' :
         del cfg.vrfLoggingHost[ vrfName ].loggingHost[ ipAddrOrHostname ]
         if not cfg.vrfLoggingHost[ vrfName ].loggingHost:
            del cfg.vrfLoggingHost[ vrfName ]
      elif ports is None:
         mode.addWarning( "Port is not specified" )
         return
      else:
         # Remove the port specified
         remainingPorts = {}
         for port in cfg.vrfLoggingHost[
               vrfName ].loggingHost[ ipAddrOrHostname ].ports:
            if port not in ports:
               remainingPorts[ port ] = port
         # If there is no port remaining
         if not remainingPorts:
            del cfg.vrfLoggingHost[ vrfName ].loggingHost[ ipAddrOrHostname ]
            if not cfg.vrfLoggingHost[ vrfName ].loggingHost:
               del cfg.vrfLoggingHost[ vrfName ]
         else:
            # Add the remaingPorts back to loggingHost
            loggingHost = Tac.Value( "LogMgr::LoggingHost",
                        ipAddrOrHostname=ipAddrOrHostname,
                        protocol=cfg.vrfLoggingHost[
                           vrfName ].loggingHost[ ipAddrOrHostname ].protocol,
                        ports=remainingPorts )
            cfg.vrfLoggingHost[ vrfName ].loggingHost.addMember( loggingHost )
   updateDscpRules()

def cmdTimestampFormatHighResolution( mode, highResolutionFormat ):
   """ Set the log message timestamp high-resolution format """
   cfg = getSysLogConfig()
   cfg.timestampFormat = LoggingLib.timestampFormatToTacc( highResolutionFormat )
   # Disable the display of timezone and year.
   cfg.timestampYear = False
   cfg.timestampTimezone = False

def cmdFormatSequenceNumber( mode, no=None ):
   cfg = getSysLogConfig()
   cfg.sequenceNumbersEnabled = not bool( no )

def cmdSourceInterface( mode, vrfName, intf ):
   if not vrfName:
      vrfName = defaultVrf
   cfg = getSysLogConfig()
   # cmdSourceInterface is mutually exclusive with cmdSourceAddress on the same VRF
   # BUG795584: Support configuration of sourceAddr and sourceIntf together.
   #
   # Warn the user that source interface configuraiton has overwritten
   # source address configuration in the same VRF
   if vrfName in cfg.srcIpAddress or vrfName in cfg.srcIp6Address:
      overwriteIpWarningStr = "Source interface configuration has overwritten " \
         + "existing source address configuration"
      mode.addWarning( overwriteIpWarningStr + " in VRF %s" % vrfName )
   del cfg.srcIpAddress[ vrfName ]
   del cfg.srcIp6Address[ vrfName ]

   cfg.srcIntfName[ vrfName ] = intf.name
   # Warn the user if a host is not configured for the vrf
   if vrfName not in cfg.vrfLoggingHost:
      noHostWarningStr = "Source interface configuration will be ignored " \
                   + "until a remote Syslog server is configured"
      if vrfName == defaultVrf:
         mode.addWarning( noHostWarningStr )
      else:
         mode.addWarning( noHostWarningStr + " in VRF %s" % vrfName )

def cmdNoSourceInterface( mode, vrfName ):
   if not vrfName:
      vrfName = defaultVrf
   cfg = getSysLogConfig()
   if vrfName in cfg.srcIntfName:
      del cfg.srcIntfName[ vrfName ]

def cmdSourceAddress( mode, vrfName, ipAddr ):
   if not vrfName:
      vrfName = defaultVrf
   cfg = getSysLogConfig()
   # cmdSourceAddress is mutually exclusive with cmdSourceInterface on the same VRF
   # BUG795584: Support configuration of sourceAddr and sourceIntf together.
   #
   # Warn the user that source address configuraiton has overwritten
   # source interface configuration in the same VRF
   if vrfName in cfg.srcIntfName:
      overwriteIntfWarningStr = "Source address configuration has overwritten " \
         + "existing source interface configuration"
      mode.addWarning( overwriteIntfWarningStr + " in VRF %s" % vrfName )
   del cfg.srcIntfName[ vrfName ]

   if ipAddr.af == 'ipv4':
      cfg.srcIpAddress[ vrfName ] = ipAddr.v4Addr
   elif ipAddr.af == 'ipv6':
      cfg.srcIp6Address[ vrfName ] = ipAddr.v6Addr
   else: # Warn the user that the IP address is invalid
      invalidIpWarningStr = "Source address configuration is unsuccessful because " \
         + "the configured IP neither an IPv4 nor IPv6 address"
      mode.addWarning( invalidIpWarningStr )

   # Warn the user if a host is not configured for the vrf
   if vrfName not in cfg.vrfLoggingHost:
      noHostWarningStr = "Source address configuration will be ignored " \
                   + "until a remote Syslog server is configured"
      if vrfName == defaultVrf:
         mode.addWarning( noHostWarningStr )
      else:
         mode.addWarning( noHostWarningStr + " in VRF %s" % vrfName )

def cmdNoSourceAddress( mode, vrfName, ipAddr ):
   if not vrfName:
      vrfName = defaultVrf
   cfg = getSysLogConfig()
   if not ipAddr:
      if vrfName in cfg.srcIpAddress:
         del cfg.srcIpAddress[ vrfName ]
      if vrfName in cfg.srcIp6Address:
         del cfg.srcIp6Address[ vrfName ]
   elif ipAddr.af == 'ipv4' and ipAddr.v4Addr == cfg.srcIpAddress.get( vrfName ):
      del cfg.srcIpAddress[ vrfName ]
   elif ipAddr.af == 'ipv6' and ipAddr.v6Addr == cfg.srcIp6Address.get( vrfName ):
      del cfg.srcIp6Address[ vrfName ]

def cmdLoggingChangeSeverity( mode, message, severity ):
   cfg = getSysLogConfig()
   cfg.msgSeverityLevel[ message ] = severity

def cmdNoLoggingChangeSeverity( mode, message ):
   cfg = getSysLogConfig()
   if message == 'all':
      cfg.msgSeverityLevel.clear()
   else:
      del cfg.msgSeverityLevel[ message ]

levelKwMatcher = CliMatcher.KeywordMatcher(
   'level',
   helpdesc='Configure logging severity' )

trapKwMatcher = CliMatcher.KeywordMatcher(
   'trap',
   helpdesc="Severity of messages sent to the syslog server" )

vrfExprFactory = VrfCli.VrfExprFactory(
   helpdesc='Specify VRF' )

facilityKwMatcher = CliMatcher.KeywordMatcher( 'facility',
                                               helpdesc='Set logging facility' )

def allFacilityNames( mode ):
   names = { n: n + " facility"
             for n in tacLoggingConfig.facilityConfig }
   names[ 'all' ] = 'All facilities'
   return names

def severityValueToTacc( mode, severity ):
   return LoggingLib.severityValueToTacc( severity )

def getSeverity( mode, args ):
   severity = args.get( 'SEVERITY' )
   if severity is not None:
      severity = severityValueToTacc( mode, severity )
   return severity

# Syslog Facility Token
def facilityValueToTacc( mode, facilityNum ):
   return LoggingLib.facilityValueToTacc( facilityNum )

cliSyslogFacilities = [ ( 'kern', 'Kernel messages' ),
                        ( 'user', 'User-level messages' ),
                        ( 'mail', 'Mail system' ),
                        ( 'daemon', 'System daemons' ),
                        ( 'auth', 'Authorization messages' ),
                        ( 'syslog', 'Internal syslogd messages' ),
                        ( 'lpr', 'Line printer subsystem' ),
                        ( 'news', 'Network news subsystem' ),
                        ( 'uucp', 'UUCP subsystem' ),
                        ( 'sys9', 'System use 9' ),
                        ( 'sys10', 'System use 10' ),
                        ( 'sys11', 'System use 11' ),
                        ( 'sys12', 'System use 12' ),
                        ( 'sys13', 'System use 13' ),
                        ( 'sys14', 'System use 14' ),
                        ( 'cron', 'Clock daemon' ),
                        ( 'local0', 'Local use 0' ),
                        ( 'local1', 'Local use 1' ),
                        ( 'local2', 'Local use 2' ),
                        ( 'local3', 'Local use 3' ),
                        ( 'local4', 'Local use 4' ),
                        ( 'local5', 'Local use 5' ),
                        ( 'local6', 'Local use 6' ),
                        ( 'local7', 'Local use 7' ) ]

maxFacilityDescLen = max( len( desc ) for ( n, desc ) in cliSyslogFacilities )

# [no] logging on
class LoggingOn( CliCommand.CliCommandClass ):
   syntax = "logging on"
   noOrDefaultSyntax = syntax
   data = {
      'logging' : loggingForConfig,
      'on' : 'Turn on logging'
      }
   @staticmethod
   def handler( mode, args ):
      cfg = getSysLogConfig()
      cfg.loggingOn = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfg = getSysLogConfig()
      if CliCommand.isDefaultCmd( args ):
         cfg.loggingOn = cfg.loggingOnDefault
      else:
         cfg.loggingOn = False

BasicCli.GlobalConfigMode.addCommandClass( LoggingOn )

class FacilityExpr( CliCommand.CliExpression ):
   expression = "FACILITY_VALUE | FACILITY_NAME"
   data = {
      'FACILITY_VALUE' : CliMatcher.IntegerMatcher(
         0, 23, helpdesc='Syslog facility value' ),
      'FACILITY_NAME' : CliMatcher.DynamicKeywordMatcher(
         # pylint: disable-next=bad-string-format-type
         lambda mode: { fac : '%-*s (facility=%d)' %
                        ( maxFacilityDescLen, desc,
                          LoggingLib.facilityCliToValue( fac ) )
                        for fac, desc in cliSyslogFacilities } )
   }
   @staticmethod
   def adapter( mode, args, argsList ):
      facility = args.pop( 'FACILITY_VALUE', None )
      if facility is None:
         facility = args.pop( 'FACILITY_NAME', None )
         if facility is not None:
            # We can be in an iteration, but all current users
            # only have one occurance (SetRule).
            if isinstance( facility, list ):
               facility = facility[ 0 ]
            facility = LoggingLib.facilityCliToValue( facility )
      else:
         # We can be in an iteration, but all current users
         # only have one occurance (SetRule).
         if isinstance( facility, list ):
            facility = facility[ 0 ]

      if facility is not None:
         args[ 'FACILITY' ] = facilityValueToTacc( mode, facility )

# logging level <facility> <severity>
class LoggingLevel( CliCommand.CliCommandClass ):
   syntax = "logging level ( FACILITY | UNKNOWN_FACILITY ) SEVERITY"
   noOrDefaultSyntax = "logging level ( FACILITY | UNKNOWN_FACILITY )..."
   data = {
      'logging' : loggingForConfig,
      'level' : levelKwMatcher,
      'FACILITY' : CliMatcher.DynamicKeywordMatcher( allFacilityNames ),
      'UNKNOWN_FACILITY': CliCommand.Node(
         matcher=CliMatcher.PatternMatcher( CliParser.namePattern,
            priority=CliParser.PRIO_LOW,
            helpname='facility',
            helpdesc="Unknown Facility" ),
         hidden=True,
      ),
      'SEVERITY' : LoggingLib.SeverityExpression
      }
   @staticmethod
   def handler( mode, args ):
      unknownFacility = args.get( 'UNKNOWN_FACILITY' )
      if unknownFacility:
         mode.addWarning( f'Logging facility {unknownFacility} is not valid' )
         return
      cmdLoggingLevel( mode, args[ 'FACILITY' ], getSeverity( mode, args ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      unknownFacility = args.get( 'UNKNOWN_FACILITY' )
      if unknownFacility:
         mode.addWarning( f'Logging facility {unknownFacility} is not valid' )
         return
      cmdNoLoggingLevel( mode, args[ 'FACILITY' ] )

BasicCli.GlobalConfigMode.addCommandClass( LoggingLevel )

# logging level default <severity>
class LoggingLevelDefault( CliCommand.CliCommandClass ):
   syntax = "logging level default SEVERITY"
   noOrDefaultSyntax = "logging level default..."
   data = {
      'logging': loggingForConfig,
      'level': levelKwMatcher,
      'default': 'Default value for the level',
      'SEVERITY': LoggingLib.SeverityExpression
      }

   @staticmethod
   def handler( mode, args ):
      cmdLoggingLevelDefault( mode, getSeverity( mode, args ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cmdNoLoggingLevelDefault( mode )

BasicCli.GlobalConfigMode.addCommandClass( LoggingLevelDefault )

# [no|default] logging buffered [ <len> ] [ <severity> ]
class LoggingBuffered( CliCommand.CliCommandClass ):
   syntax = "logging buffered [ SIZE ] [ SEVERITY ]"
   noSyntax = "logging buffered ..."
   defaultSyntax = noSyntax
   data = {
      'logging' : loggingForConfig,
      'buffered' : 'Logging buffer configuration',
      'SIZE' : CliMatcher.IntegerMatcher(
         10, 2147483647,
         helpdesc='Logging buffer size (in number of messages)' ),
      'SEVERITY' : LoggingLib.SeverityExpression
      }
   @staticmethod
   def handler( mode, args ):
      cmdLoggingBuffered( mode, args.get( 'SIZE' ), getSeverity( mode, args ) )

   @staticmethod
   def noHandler( mode, args ):
      cmdNoLoggingBuffered( mode )

   @staticmethod
   def defaultHandler( mode, args ):
      cmdLoggingBuffered( mode )

BasicCli.GlobalConfigMode.addCommandClass( LoggingBuffered )

class LoggingTrap( CliCommand.CliCommandClass ):
   syntax = "logging trap [ SEVERITY ]"
   noSyntax = "logging trap ..."
   defaultSyntax = noSyntax
   data = {
      'logging' : loggingForConfig,
      'trap' : trapKwMatcher,
      'SEVERITY' : LoggingLib.SeverityExpression
      }
   @staticmethod
   def handler( mode, args ):
      cmdLoggingTrap( mode, getSeverity( mode, args ) )

   @staticmethod
   def noHandler( mode, args ):
      cmdNoLoggingTrap( mode )

   @staticmethod
   def defaultHandler( mode, args ):
      cmdLoggingTrap( mode )

BasicCli.GlobalConfigMode.addCommandClass( LoggingTrap )

# [no|default] logging trap|buffered system [severity <severity>]
#               [facility <facility>] [tag NAME] [contain STRING]
class LoggingSystem( CliCommand.CliCommandClass ):
   syntax = ( "logging (trap | buffered) system "
              "[ { [ severity SEVERITY ] [ facility FACILITY ] "
              "[ tag TAG ] } ] [ contain CONTAIN ]" )
   noOrDefaultSyntax = syntax
   data = {
      'logging' : loggingForConfig,
      'trap' : trapKwMatcher,
      'buffered': 'Logging buffer configuration',
      'system' : 'Set system log buffer',
      'severity' : CliCommand.singleKeyword( 'severity',
                                             'Specify severity' ),
      'SEVERITY' : LoggingLib.SeverityExpression,
      'facility' : CliCommand.singleKeyword( 'facility',
                                             'Specify severity' ),
      'FACILITY' : FacilityExpr,
      'tag' : CliCommand.singleKeyword( 'tag',
                                        'Specify tag ( program name )' ),
      'TAG' : CliMatcher.PatternMatcher( r'.+', helpname='WORD',
                                         helpdesc='Program name' ),
      'contain' : 'Specify text contained in log message',
      'CONTAIN' : CliMatcher.StringMatcher( helpname='LINE',
                                            helpdesc='Specify text' )

      }

   @staticmethod
   def _setArgs( args ):
      setArgs = {}
      if 'severity' in args:
         setArgs[ 'severity' ] = args[ 'SEVERITY' ]
      if 'facility' in args:
         setArgs[ 'facility' ] = args[ 'FACILITY' ]
      if 'tag' in args:
         setArgs[ 'tag' ] = args[ 'TAG' ][ 0 ]
      if 'contain' in args:
         setArgs[ 'contain' ] = args[ 'CONTAIN' ]
      return setArgs

   @staticmethod
   def _getLoggingSystem( args ):
      cfg = getSysLogConfig()
      if 'trap' in args:
         return cfg.loggingTrapSystem
      return cfg.loggingBufferedSystem

   @staticmethod
   def handler( mode, args ):
      kwargs = LoggingSystem._setArgs( args )
      cmdLoggingSystem( LoggingSystem._getLoggingSystem( args ), kwargs )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      kwargs = LoggingSystem._setArgs( args )
      cmdNoLoggingSystem( LoggingSystem._getLoggingSystem( args ), kwargs )

BasicCli.GlobalConfigMode.addCommandClass( LoggingSystem )


# [no|default] logging message <facility-mnemonic> severity <severity>
class LoggingChangeSeverity( CliCommand.CliCommandClass ):
   syntax = "logging message MESSAGE severity SEVERITY"
   noOrDefaultSyntax = "logging message ( MESSAGE | all ) severity ..."
   data = {
      'logging' : loggingForConfig,
      'message' : 'Configure the given log message',
      'MESSAGE' : CliMatcher.PatternMatcher(
               pattern=r'\S+-\S+',
               helpname='FACILITY-MNEMONIC',
               helpdesc='Log message id' ),
      'severity' : 'Change severity level for the given message',
      'SEVERITY' : LoggingLib.SeverityExpression,
      'all' : 'Clear configurations for all messages'
      }
   @staticmethod
   def handler( mode, args ):
      cmdLoggingChangeSeverity( mode, args[ 'MESSAGE' ],
                                getSeverity( mode, args ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      msg = args.get( 'MESSAGE', 'all' )
      cmdNoLoggingChangeSeverity( mode, msg )

BasicCli.GlobalConfigMode.addCommandClass( LoggingChangeSeverity )


# BUG92: ensure that the address/hostname entered is valid.
# [no] logging [vrf VRF] host HOST [add|remove] [{ ports }]
# [ protocol ( tcp | udp | ( tls ssl-profile PROFILE ) ) ]

class LoggingHost( CliCommand.CliCommandClass ):
   syntax = ( "logging [ VRF ] host HOSTNAME [ add | remove ] [ { PORT } ] "
              "[ protocol ( tcp | udp | ( tls ssl-profile PROFILE ) ) ] " )
   noOrDefaultSyntax = "logging [ VRF ] host HOSTNAME ... "
   data = {
      'logging' : loggingForConfig,
      'VRF' : vrfExprFactory,
      'host' : 'Set syslog server IP address and parameters',
      'HOSTNAME' : HostnameCli.IpAddrOrHostnameMatcher(
         ipv6=True,
         helpname='WORD',
         helpdesc='Hostname or IP address of the syslog server' ),
      'add': 'Configure ports on the given host',
      'remove': 'Remove configured ports from the given host',
      'PORT' : CliMatcher.IntegerMatcher(
         1, 65535, helpdesc='Port of the syslog server (default=514)' ),
      'protocol' : 'Set syslog server transport protocol',
      'tcp' : 'Unprotected TCP',
      'udp' : 'UDP',
      'tls' : 'TCP with TLS',
      'ssl-profile': 'Configure SSL profile to use',
      'PROFILE': CliMatcher.DynamicNameMatcher(
         lambda mode: sslConfig.profileConfig,
         helpname='WORD',
         helpdesc='Profile name',
         priority=CliParser.PRIO_LOW )
   }

   @staticmethod
   def handler( mode, args ):
      protocol = args.get( 'tcp' ) or args.get( 'tls' ) or args.get( 'udp' )
      profileName = args.get( 'PROFILE', '' )
      if 'remove' in args:
         cmdNoLoggingHost( mode, args.get( 'VRF' ),
                           args[ 'HOSTNAME' ], args.get( 'PORT' ) )
      else:
         cmdLoggingHost( mode, args.get( 'VRF' ),
                         args[ 'HOSTNAME' ],
                         args.get( 'PORT' ),
                         protocol,
                         profileName,
                         overwrite='add' not in args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cmdNoLoggingHost( mode, args.get( 'VRF' ),
                        args[ 'HOSTNAME' ], 'all' )

BasicCli.GlobalConfigMode.addCommandClass( LoggingHost )

def updateDscpRules():
   sysCfg = getSysLogConfig()
   dscpValue = sysCfg.dscpValue
   if not dscpValue:
      del dscpConfig.protoConfig[ 'syslog' ]
      return

   protoConfig = dscpConfig.newProtoConfig( 'syslog' )
   ruleColl = protoConfig.rule
   ruleColl.clear()

   for vrf in sysCfg.vrfLoggingHost:
      loggingHost = sysCfg.vrfLoggingHost[ vrf ].loggingHost
      for lh in loggingHost:
         ipOrHostname = loggingHost[ lh ].ipAddrOrHostname
         for port in loggingHost[ lh ].ports:
            proto = loggingHost[ lh ].protocol
            DscpCliLib.addDscpRule( ruleColl, ipOrHostname, port,
                                    False, vrf, proto, dscpValue )
            DscpCliLib.addDscpRule( ruleColl, ipOrHostname, port,
                                    False, vrf, proto, dscpValue, v6=True )



def setDscp( mode, args ):
   sysCfg = getSysLogConfig()
   sysCfg.dscpValue = args[ 'DSCP' ]
   updateDscpRules()

def noDscp( mode, args ):
   sysCfg = getSysLogConfig()
   sysCfg.dscpValue = sysCfg.dscpValueDefault
   updateDscpRules()

# import DscpCliLib
# [no|default] logging qos dscp <dscpValue>
DscpCliLib.addQosDscpCommandClass( BasicCli.GlobalConfigMode, setDscp, noDscp,
                                   loggingForConfig )

# [no|default] logging console [severity]
class LoggingConsole( CliCommand.CliCommandClass ):
   syntax = "logging console [ SEVERITY ]"
   noSyntax = "logging console ..."
   defaultSyntax = noSyntax
   data = {
      "logging" : loggingForConfig,
      "console" : 'Set console logging parameters',
      'SEVERITY' : LoggingLib.SeverityExpression
      }
   @staticmethod
   def handler( mode, args ):
      cmdLoggingConsole( mode, getSeverity( mode, args ) )

   @staticmethod
   def noHandler( mode, args ):
      cmdNoLoggingConsole( mode )

   @staticmethod
   def defaultHandler( mode, args ):
      cmdLoggingConsole( mode )

BasicCli.GlobalConfigMode.addCommandClass( LoggingConsole )

# [no|default] logging monitor [severity]
class LoggingMonitor( CliCommand.CliCommandClass ):
   syntax = "logging monitor [ SEVERITY ]"
   noSyntax = "logging monitor ..."
   defaultSyntax = noSyntax
   data = {
      "logging" : loggingForConfig,
      "monitor" : 'Set terminal monitor parameters',
      'SEVERITY' : LoggingLib.SeverityExpression
      }
   @staticmethod
   def handler( mode, args ):
      cmdLoggingMonitor( mode, getSeverity( mode, args ) )

   @staticmethod
   def noHandler( mode, args ):
      cmdNoLoggingMonitor( mode )

   @staticmethod
   def defaultHandler( mode, args ):
      cmdDefaultLoggingMonitor( mode )

BasicCli.GlobalConfigMode.addCommandClass( LoggingMonitor )

# logging timestamp format traditional [ year | timezone ]
timestampKwMatcher = CliMatcher.KeywordMatcher(
   'timestamp', helpdesc='Set timestamp logging parameters' )
formatKwMatcher = CliMatcher.KeywordMatcher(
   'format', helpdesc='Set logging format parameters' )

class LoggingFormatTimestamp( CliCommand.CliCommandClass ):
   syntax = ( "logging ( format timestamp ) | ( HT HF ) "
              "( traditional [ { year | timezone } ] ) | high-resolution" )
   noOrDefaultSyntax = "logging ( format timestamp ) | ( HT HF ) ..."
   data = {
      'logging' : loggingForConfig,
      'format' : formatKwMatcher,
      'timestamp' : timestampKwMatcher,
      'HT' : CliCommand.Node( timestampKwMatcher,
                              hidden=True ),
      'HF' : CliCommand.Node( formatKwMatcher,
                              hidden=True ),
      'traditional' :
      'Traditional syslog timestamp format as specified in RFC3164',
      'year' : CliCommand.singleKeyword(
         'year',
         helpdesc='Show year in traditional format timestamp' ),
      'timezone' : CliCommand.singleKeyword(
         'timezone',
         helpdesc='Show timezone in traditional format timestamp' ),
      'high-resolution' : 'RFC3339 timestamps'
      }
   @staticmethod
   def handler( mode, args ):
      cfg = getSysLogConfig()
      if 'traditional' in args:
         cfg.timestampFormat = LoggingLib.timestampFormatToTacc( 'traditional' )
         # Enable year/timezone display if enabled.
         cfg.timestampYear = 'year' in args
         cfg.timestampTimezone = 'timezone' in args
      else:
         cfg.timestampFormat = LoggingLib.timestampFormatToTacc(
            'high-resolution' )
         # Disable the display of timezone and year.
         cfg.timestampYear = False
         cfg.timestampTimezone = False

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfg = getSysLogConfig()
      cfg.timestampFormat = cfg.timestampFormatDefault
      # disable the display of timezone and year
      cfg.timestampYear = False
      cfg.timestampTimezone = False

BasicCli.GlobalConfigMode.addCommandClass( LoggingFormatTimestamp )

# [no] logging format rfc5424
class LoggingFormatRFC5424( CliCommand.CliCommandClass ):
   syntax = "logging format rfc5424"
   noOrDefaultSyntax = "logging format rfc5424 ..."
   data = {
      'logging' : loggingForConfig,
      'format' : formatKwMatcher,
      'rfc5424' : 'Forward syslogs in RFC 5424 format'
   }

   @staticmethod
   def handler( mode, args ):
      cfg = getSysLogConfig()
      cfg.syslogFormatRFC5424 = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfg = getSysLogConfig()
      cfg.syslogFormatRFC5424 = cfg.syslogFormatRFC5424Default

BasicCli.GlobalConfigMode.addCommandClass( LoggingFormatRFC5424 )

# [no] logging format hostname { fqdn|ipv4 }
class LoggingFormatHostname( CliCommand.CliCommandClass ):
   syntax = "logging format hostname fqdn | ipv4"
   noOrDefaultSyntax = "logging format hostname ..."
   data = {
      'logging' : loggingForConfig,
      'format' : formatKwMatcher,
      'hostname' : 'Specify hostname logging format',
      'fqdn' : 'Use fully qualified domain name',
      'ipv4' : 'Use IPv4 address'
   }

   @staticmethod
   def handler( mode, args ):
      cfg = getSysLogConfig()
      cfg.hostnameFormat = args.get( 'fqdn' ) or args.get( 'ipv4' ) or 'hostname'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfg = getSysLogConfig()
      cfg.hostnameFormat = cfg.hostnameFormatDefault

BasicCli.GlobalConfigMode.addCommandClass( LoggingFormatHostname )

class LoggingFacility( CliCommand.CliCommandClass ):
   syntax = "logging facility FACILITY"
   noOrDefaultSyntax = "logging facility ..."
   data = {
      'logging' : loggingForConfig,
      'facility' : facilityKwMatcher,
      'FACILITY' : FacilityExpr
   }
   @staticmethod
   def handler( mode, args ):
      cfg = getSysLogConfig()
      cfg.loggingFacility = args[ 'FACILITY' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfg = getSysLogConfig()
      cfg.loggingFacility = cfg.loggingFacilityDefault

BasicCli.GlobalConfigMode.addCommandClass( LoggingFacility )

###########################################################################
# [no|default] logging format sequence-numbers
#
# legacy:
# [no|default] service sequence-numbers
###########################################################################

seqNoKwMatcher = CliMatcher.KeywordMatcher(
   'sequence-numbers',
   helpdesc='Number log messages' )

class LoggingFormatSeq( CliCommand.CliCommandClass ):
   syntax = "logging format sequence-numbers"
   noOrDefaultSyntax = syntax
   data = {
      'logging' : loggingForConfig,
      'format' : formatKwMatcher,
      'sequence-numbers' : seqNoKwMatcher
      }
   @staticmethod
   def handler( mode, args ):
      cmdFormatSequenceNumber( mode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cmdFormatSequenceNumber( mode, no=True )

BasicCli.GlobalConfigMode.addCommandClass( LoggingFormatSeq )

class DeprecatedLoggingFormatSeq( CliCommand.CliCommandClass ):
   syntax = "service sequence-numbers"
   noOrDefaultSyntax = syntax
   data = {
      'service' : CliToken.Service.serviceMatcherForConfig,
      'sequence-numbers' : CliCommand.Node(
         seqNoKwMatcher,
         deprecatedByCmd='logging format sequence-numbers' )
      }
   @staticmethod
   def handler( mode, args ):
      cmdFormatSequenceNumber( mode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cmdFormatSequenceNumber( mode, no=True )

BasicCli.GlobalConfigMode.addCommandClass( DeprecatedLoggingFormatSeq )

# [no] logging source-interface
class LoggingSourceIntf( CliCommand.CliCommandClass ):
   syntax = "logging [ VRF ] source-interface INTF"
   noOrDefaultSyntax = "logging [ VRF ] source-interface ..."
   data = {
      'logging' : loggingForConfig,
      'VRF' : vrfExprFactory,
      'source-interface' :
      'Use IP Address of interface as source IP of log messages',
      'INTF' : IntfCli.Intf.matcherWithIpSupport
      }
   @staticmethod
   def handler( mode, args ):
      cmdSourceInterface( mode, args.get( 'VRF' ), args[ 'INTF' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cmdNoSourceInterface( mode, args.get( 'VRF' ) )

BasicCli.GlobalConfigMode.addCommandClass( LoggingSourceIntf )

# [no] logging source-address
class LoggingSourceAddr( CliCommand.CliCommandClass ):
   syntax = "logging [ VRF ] source-address IP-ADDR"
   noOrDefaultSyntax = syntax
   data = {
      'logging': loggingForConfig,
      'VRF': vrfExprFactory,
      'source-address':
      'Use IP Address of an active interface as source IP of log messages',
      'IP-ADDR': IpGenAddrMatcher( 'IPv4 or IPv6 Address',
                                 helpdesc4='IPv4 address',
                                 helpdesc6='IPv6 address' )
      }

   @staticmethod
   def handler( mode, args ):
      cmdSourceAddress( mode, args.get( 'VRF' ), args[ 'IP-ADDR' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cmdNoSourceAddress( mode, args.get( 'VRF' ), args.get( 'IP-ADDR' ) )

BasicCli.GlobalConfigMode.addCommandClass( LoggingSourceAddr )

# [no] logging relogging-interval <repeatInterval>
class LoggingInterval( CliCommand.CliCommandClass ):
   syntax = "logging relogging-interval INTERVAL"
   noOrDefaultSyntax = "logging relogging-interval ..."
   data = {
      'logging' : loggingForConfig,
      'relogging-interval' :
      'Configure relogging-interval for critical log messages',
      'INTERVAL' :  CliMatcher.IntegerMatcher(
         1, 720,
         helpdesc='Set number of minutes for repeat log interval' )
      }
   @staticmethod
   def handler( mode, args ):
      cfg = getSysLogConfig()
      cfg.repeatInterval = args[ 'INTERVAL' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfg = getSysLogConfig()
      cfg.repeatInterval = 0

BasicCli.GlobalConfigMode.addCommandClass( LoggingInterval )

# [no] logging synchronous [ level <severity> | all ]
class LoggingSynchronous( CliCommand.CliCommandClass ):
   syntax = "logging synchronous [ level SEVERITY | all ]"
   noOrDefaultSyntax = "logging synchronous ..."
   data = {
      'logging' : loggingForConfig,
      'synchronous' : 'Set synchronizing unsolicited with solicited messages',
      'level' : levelKwMatcher,
      'SEVERITY' : LoggingLib.SeverityExpression,
      'all' : 'Log all messages synchronously'
      }
   @staticmethod
   def handler( mode, args ):
      cfg = getSysLogConfig()
      cfg.loggingSynchronousEnabled = True
      if 'all' in args:
         cfg.loggingSynchronousAll = True
      else:
         severity = args.get( 'SEVERITY' )
         if severity is not None:
            severity = severityValueToTacc( mode, severity )
         else:
            severity = cfg.loggingSynchronousSeverityDefault
         cfg.loggingSynchronousSeverity = severity
         cfg.loggingSynchronousAll = False

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfg = getSysLogConfig()
      cfg.loggingSynchronousEnabled = False
      cfg.loggingSynchronousSeverity = cfg.loggingSynchronousSeverityDefault
      cfg.loggingSynchronousAll = cfg.loggingSynchronousAllDefault

BasicCli.GlobalConfigMode.addCommandClass( LoggingSynchronous )

# [no | default ] logging repeat-messages
class LogRepeat( CliCommand.CliCommandClass ):
   syntax = "logging repeat-messages"
   noOrDefaultSyntax = syntax
   data = {
      "logging" : loggingForConfig,
      "repeat-messages" :
      "Repeat messages instead of summarizing number of repeats",
      }
   @staticmethod
   def handler( mode, args ):
      cfg = getSysLogConfig()
      cfg.repeatMsgs = True

   @staticmethod
   def noHandler( mode, args ):
      cfg = getSysLogConfig()
      cfg.repeatMsgs = False

   @staticmethod
   def defaultHandler( mode, args ):
      cfg = getSysLogConfig()
      cfg.repeatMsgs = cfg.repeatMsgsDefault

BasicCli.GlobalConfigMode.addCommandClass( LogRepeat )

class LoggingMatchList( CliCommand.CliCommandClass ):
   syntax = "logging policy match [invert-result] match-list MATCHLIST_NAME discard"
   noOrDefaultSyntax = "logging policy ..."
   data = { "logging": loggingForConfig,
            "policy": "Configure logging policies",
            "match-list": "Configure logging message filtering",
            "MATCHLIST_NAME": CliMatcher.DynamicNameMatcher(
                                 lambda mode: matchListConfig.matchStringList,
                                 'Match list name' ),
            "match": "Configure logging message filtering on matching messages",
            "invert-result": "Invert the match of match-list",
            "discard": "Discard the messages" }

   @staticmethod
   def handler( mode, args ):
      sysLoggingConfig.loggingMatchList = Tac.Value(
         "LogMgr::LoggingMatchListInfo",
         args[ 'MATCHLIST_NAME' ], args.get( 'invert-result' ) == "invert-result" )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      sysLoggingConfig.loggingMatchList = Tac.Value(
         "LogMgr::LoggingMatchListInfo", "", False )

BasicCli.GlobalConfigMode.addCommandClass( LoggingMatchList )

def pidIsRunning( pid ):
   return pid in os.listdir( '/proc' )

######### "terminal monitor" #########
#
# Implemented
# -----------
#
# [no] terminal monitor
#    => enable logging on ssh/telnet sessions
class TerminalMonitor( CliCommand.CliCommandClass ):
   syntax = "terminal monitor"
   noOrDefaultSyntax = syntax
   data = {
      'terminal' : CliToken.Terminal.terminalKwForExec,
      'monitor' : "Display logging output to the current terminal"
      }
   @staticmethod
   def handler( mode, args ):
      # Start the tail process on /var/log/eos-console
      if not isConsole():
         # Start the tail directly
         mode.session_.cliInput.enableLoggingMonitor()
         # Enable log synchronous only for non-console
         # since console already has it enabled.
         mode.session_.cliInput.enableLoggingSync( False )
      else:
         mode.addWarning( "Console always monitors" )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # Stop the tail process on /var/log/eos-console
      if not isConsole():
         # Disable log synchronous
         mode.session_.cliInput.disableLoggingSync()
         # Stop just the tail process itself
         mode.session_.cliInput.disableLoggingMonitor()
      else:
         mode.addWarning( "Console always monitors" )

BasicCli.EnableMode.addCommandClass( TerminalMonitor )

# [no] logging persistent [size]
def cmdLoggingPersistent( mode, logSize=None ):
   cfg = getSysLogConfig()
   cfg.persistentLog = True
   cfg.persistentLogSize = logSize or cfg.persistentLogSizeDefault

def cmdNoLoggingPersistent( mode ):
   cfg = getSysLogConfig()
   cfg.persistentLog = False
   cfg.persistentLogSize = cfg.persistentLogSizeDefault

class LoggingPersistent( CliCommand.CliCommandClass ):
   syntax = "logging persistent [ SIZE ]"
   noOrDefaultSyntax = "logging persistent ..."
   data = {
      'logging': loggingForConfig,
      'persistent' : 'Save logging messages to the flash disk',
      'SIZE' : CliMatcher.IntegerMatcher(
         1024, 2147483647,
         helpdesc='The maximum size (in bytes) of logging file '
                  'stored on flash disk' )
      }
   @staticmethod
   def handler( mode, args ):
      mode.addWarning( "Note: writing system log message on non-volatile "
                       "flash will affect the life expectancy of the flash drive "
                       "due to heavy writing. Please disable persistent logging "
                       "unless needed." )
      cmdLoggingPersistent( mode, args.get( 'SIZE' ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cmdNoLoggingPersistent( mode )

BasicCli.GlobalConfigMode.addCommandClass( LoggingPersistent )

# [no] logging event login root
class LoggingEventRootLogin( IntfCli.LoggingEventGlobalCmd ):
   syntax = "logging event login root"
   noOrDefaultSyntax = syntax
   data = {
      'login' : 'Login activities',
      'root' : 'Root user activities',
      }
   @staticmethod
   def handler( mode, args ):
      sysLoggingConfig.loggingRootLogin = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      sysLoggingConfig.loggingRootLogin = False

BasicCli.GlobalConfigMode.addCommandClass( LoggingEventRootLogin )

######### "clear logging #########
#
# Implemented
# -----------
#
# clear logging
#    => clears system log file and logrotated system log files
#       Example: /var/log/eos & /var/log/eos.*.gz & /mnt/flash/message
#       /var/log/messages is not removed, so logs can be retrieved from there
class ClearLogging( CliCommand.CliCommandClass ):
   syntax = "clear logging"
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'logging' : 'Clear all log files'
      }

   @staticmethod
   def handler( mode, args ):
      info = UtmpDump.getUserInfo()
      who = info[ 'user' ]
      where = info[ 'tty' ]
      timestamp = Tac.now()
      sysLoggingConfigReq.clearLogRequest = \
         Tac.Value( "LogMgr::ClearRequest", timestamp, who, where )

BasicCli.EnableMode.addCommandClass( ClearLogging )

#------------------------------------------------------------------
# Register "show logging" and "show logging threshold errors" into
# "show tech-support".
#------------------------------------------------------------------
# Timestamps are made up to maintain historical order within show tech-support
CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2010-01-01 00:03:30',
   cmds=[ 'show logging' ] )

# Timestamps are made up to maintain historical order within show tech-support
CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2012-03-01 16:49:21',
   cmds=[ 'show logging threshold errors' ],
   summaryCmds=[ 'show logging' ] )

# Capture /var/log/kernel.debug, /var/log/canary.log, /var/log/rbfd.log
# and kernel messages as part of 'show tech'
nLines = 1000
kernelDebugs = '/var/log/kernel.debug'
CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2014-12-09 23:23:23',
   cmds=[ 'bash sudo tail -n %d %s' % ( nLines, kernelDebugs ) ],
   # We may not have /var/log/kernel.debug
   cmdsGuard=lambda: os.path.isfile( kernelDebugs ) )

canaryLog = '/var/log/canary.log'
CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2014-12-09 23:23:23',
   cmds=[ 'bash sudo tail -n %d %s' % ( nLines, canaryLog ) ],
   cmdsGuard=lambda: os.path.isfile( canaryLog ) )

rbfdLog = '/var/log/rbfd.log'
CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2014-12-09 23:23:23',
   cmds=[ 'bash sudo tail -n %d %s' % ( nLines, rbfdLog ) ],
   cmdsGuard=lambda: os.path.isfile( rbfdLog ) )

CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2014-12-09 23:23:23',
   # Show dmesg buffer minus messages at debug level
   cmds=[ 'bash dmesg --time-format iso --time-format raw ' +
          '-x -S | grep -v \':debug :\' | tail -n %d' % nLines ] )

def initLoggingSyncMonitor( session ):
   # terminal monitor is always enabled on console
   if isConsole():
      session.cliInput.enableLoggingSync( True )

def Plugin( entityManager ):
   global tacLoggingConfig, defaultVrf, dscpConfig
   global sysLoggingConfig, sysLoggingConfigReq, sysLoggingStatus
   global sslConfig, matchListConfig

   tacLoggingConfig = LazyMount.mount( entityManager, "logging/config",
                                       "Tac::LogConfig", "r" )
   sysLoggingConfig = ConfigMount.mount( entityManager, "sys/logging/config",
                                         "LogMgr::LogConfig", "w" )
   sysLoggingConfigReq = LazyMount.mount( entityManager, "sys/logging/configReq",
                                         "LogMgr::LogConfigReq", "w" )
   sysLoggingStatus = LazyMount.mount( entityManager,
                                       Cell.path( "sys/logging/status" ),
                                       "LogMgr::LogStatus", "r" )
   defaultVrf = Tac.newInstance( 'L3::VrfName', 'temp-status' ).defaultVrf
   dscpConfig = ConfigMount.mount( entityManager, "mgmt/dscp/config",
                                   "Mgmt::Dscp::Config", "w" )
   sslConfig = LazyMount.mount( entityManager,
                                'mgmt/security/ssl/config',
                                'Mgmt::Security::Ssl::Config', 'r' )
   matchListConfig = LazyMount.mount( entityManager, "matchlist/config/cli",
                                      "MatchList::Config", "r" )

   # We may not know if we are console at this point. Register a callback.
   ttyHook.addExtension( initLoggingSyncMonitor )
