# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliGlobal
import CliMatcher
import CliParser
import CliPlugin.VlanCli as VlanCli # pylint: disable=consider-using-from-import
from CliPlugin import IntfCli, EthIntfCli
import LazyMount
import Tac

gv = CliGlobal.CliGlobal( bridgingHwCapabilities=None )

#-------------------------------------------------------------------------------
# The "[no] bridge mac-address-table aging timeout <age> seconds" command,
# in "config-if" mode.
#
# The full syntax of this command is:
#
#   [no|default] bridge mac-address-table aging timeout <age> seconds
#   [no|default] bridge mac-address-table aging disabled
#-------------------------------------------------------------------------------
def setMacAgingTime( mode, args ):
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   agingTime = args.get( 'AGING_TIME', 0 )
   if args.get( 'disabled', 0 ):
      # Choose an invalid agingTime that will show aging is
      # disabled on this interface
      agingTime = 1 << 30
   sic.agingTime = agingTime

def macAgingSupportedGuard( mode, token ):
   if gv.bridgingHwCapabilities.macIntfAgingSupported:
      return None
   return CliParser.guardNotThisPlatform

bridgeMacMatcher = CliCommand.guardedKeyword( 'mac-address-table',
      helpdesc='Configure MAC Address Table for this interface',
      guard=macAgingSupportedGuard )

class MacAgingCmd( CliCommand.CliCommandClass ):
   syntax = '''bridge mac-address-table aging
               ( disabled | ( timeout AGING_TIME seconds ) )'''
   noOrDefaultSyntax = 'bridge mac-address-table aging ... '
   data = {
      'bridge': 'bridge keyword',
      'mac-address-table': bridgeMacMatcher,
      'aging': 'Aging keyword',
      'timeout': 'Timeout keyword',
      'AGING_TIME': CliMatcher.IntegerMatcher( 10, 1000000,
         helpdesc='Aging time in seconds' ),
      'seconds': 'Seconds Keyword',
      'disabled': 'Disable aging on this interface',
      }

   handler = setMacAgingTime
   noOrDefaultHandler = setMacAgingTime

#---------------------------------------------------------------------------
# Adds bridge CLI commands to the "config-if" mode.
#---------------------------------------------------------------------------
class BridgeModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return isinstance( mode.intf, EthIntfCli.EthIntf )

BridgeModelet.addCommandClass( MacAgingCmd )

IntfCli.IntfConfigMode.addModelet( BridgeModelet )

def Plugin( entityManager ):
   gv.bridgingHwCapabilities = LazyMount.mount( entityManager,
                                     "bridging/hwcapabilities",
                                     "Bridging::HwCapabilities", "r" )
#-------------------------------------------------------------------------------
# The "[no] bridge mac-address-table aging timeout <age> seconds" command,
# in "config-vl" mode.
#
# The full syntax of this command is:
#
#   [no|default] bridge mac-address-table aging timeout <age> seconds
#   [no|default] bridge mac-address-table aging disabled
#-------------------------------------------------------------------------------

def setVlanMacAgingTime( mode, args ):
   agingTime = args.get( 'AGING_TIME', 1 << 30 )
   for vc in mode.bridgingInputConfigs():
      vc.agingTime = agingTime

def noVlanMacAgingTime( mode, args ):
   for vc in mode.bridgingInputConfigs():
      vc.agingTime = 0

def vlanMacAgingSupportedGuard( mode, token ):
   if gv.bridgingHwCapabilities.macVlanAgingSupported:
      return None
   return CliParser.guardNotThisPlatform

bridgeVlanMacMatcher = CliCommand.guardedKeyword( 'mac-address-table',
      helpdesc='Configure MAC Address Table for this VLAN',
      guard=vlanMacAgingSupportedGuard )

class VlanMacAgingCmd( CliCommand.CliCommandClass ):
   syntax = '''bridge mac-address-table aging
               ( disabled | ( timeout AGING_TIME seconds ) )'''
   noOrDefaultSyntax = 'bridge mac-address-table aging ... '
   data = {
      'bridge': 'Configure bridging properties for this VLAN',
      'mac-address-table': bridgeVlanMacMatcher,
      'aging': 'Configure MAC aging for this VLAN',
      'timeout': 'Specify aging timeout',
      'AGING_TIME': CliMatcher.IntegerMatcher( 10, 1000000,
         helpdesc='Aging time in seconds' ),
      'seconds': 'Seconds keyword',
      'disabled': 'Disable aging on this VLAN',
      }

   handler = setVlanMacAgingTime
   noOrDefaultHandler = noVlanMacAgingTime

#---------------------------------------------------------------------------
# Adds Mac Aging CLI command to the "config-vl" mode.
#---------------------------------------------------------------------------

VlanCli.VlanConfigMode.addCommandClass( VlanMacAgingCmd )
