# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Model, Int, Dict
import TableOutput
import operator

class MacLimitPerVlanTable( Model ):
   class LimitedVlanEntry( Model ):
      __revision__ = 2
      limit = Int( help="Maximum number of locally learned MACs" )
      currentAddresses = Int( help="Current number of locally learned MACs" )
      appliedLimit = Int( help="Limit programmed in the hardware" )

      def degrade( self, dictRepr, revision ):
         if revision < 2:
            del dictRepr[ 'appliedLimit' ]
            dictRepr[ 'restricted' ] = \
               ( dictRepr[ 'currentAddresses' ] >= dictRepr[ 'limit' ] )
         return dictRepr

   __revision__ = 2
   limitedVlans = Dict( keyType=int,
                        valueType=LimitedVlanEntry,
                        help="Limited VLANs, keyed by their ID" )

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         totalDynamicAddresses = 0
         for entry in self.limitedVlans.values():
            totalDynamicAddresses += entry.currentAddresses
         dictRepr[ 'totalDynamicAddresses' ] = totalDynamicAddresses
      return dictRepr

   def render( self ):
      columnNames = ( "VLAN", "Limit", "Current Addresses", "Status" )
      table = TableOutput.createTable( columnNames )

      vlanFmt = TableOutput.Format( justify='right', maxWidth=4 )
      limitFmt = TableOutput.Format( justify='right', minWidth=6, maxWidth=6 )
      currentAddressesFmt = TableOutput.Format( justify='right', maxWidth=17 )
      statusFmt = TableOutput.Format( justify='left', minWidth=17, maxWidth=17 )

      formats = [ vlanFmt, limitFmt, currentAddressesFmt, statusFmt ]
      for fmt in formats:
         fmt.padLimitIs( True )

      table.formatColumns( *formats )

      for limitedVlan, entry in sorted( self.limitedVlans.items(),
                                        key=operator.itemgetter( 0 ) ):
         status = ( 'limit reached' if entry.currentAddresses >= entry.limit
                    else 'normal' )
         if entry.appliedLimit != entry.limit:
            status = 'failed to program'
         table.newRow( limitedVlan, entry.limit, entry.currentAddresses, status )

      print( table.output(), end='' )
