# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli, CliPlugin
from CliPlugin import MacsecModel
from CliPlugin.IntfCli import Intf
import SmashLazyMount, Tac
import LazyMount
import CliMatcher
import ShowCommand
from CliToken.Mac import macMatcherForShow

# Globals
_macsecCounters = None
_macsecStatus = None
_macsecMkaStatus = None
_macsecInput = None
_hwStatusSliceDir = None

#----------------------------------------------------------------------------------
# Show tokens
#----------------------------------------------------------------------------------

matcherSecurityForShow = CliMatcher.KeywordMatcher( 'security', 
            helpdesc='MAC security (802.1AE) information' )

matcherDetail = CliMatcher.KeywordMatcher(
          'detail', helpdesc='Display information in detail' )

matcherForProfile = CliMatcher.KeywordMatcher( 'profile',
            helpdesc='MAC security profile name' )

matcherForSource = CliMatcher.KeywordMatcher( 'source',
            helpdesc="Configuration source" )

class MacSecurityInterfaceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mac security interface [ INTFS ] [ detail ]'
   data = {
      'mac': macMatcherForShow,
      'security': matcherSecurityForShow,
      'interface': 'MAC security interface information',
      'INTFS': Intf.rangeMatcher,
      'detail': matcherDetail,
   }

   handler = "MacsecShowCliHandler.showInterfaces"
   cliModel = MacsecModel.MacsecInterfaces

BasicCli.addShowCommandClass( MacSecurityInterfaceCmd )

class MacSecurityMkaCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mac security mka counters [ INTFS ] [ detail ]'
   data = {
      'mac': macMatcherForShow,
      'security': matcherSecurityForShow,
      'mka': 'MAC security key agreement protocol information',
      'counters': 'MAC security counter information',
      'INTFS': Intf.rangeMatcher,
      'detail': matcherDetail,
   }

   handler = "MacsecShowCliHandler.showCounters"
   cliModel = MacsecModel.MacsecMessageCounters

BasicCli.addShowCommandClass( MacSecurityMkaCountersCmd )

class MacSecurityParticipantsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mac security participants [ INTFS ] [ detail ]'
   data = {
      'mac': macMatcherForShow,
      'security': matcherSecurityForShow,
      'participants': 'MAC security participants information',
      'INTFS': Intf.rangeMatcher,
      'detail': matcherDetail,
   }

   handler = "MacsecShowCliHandler.showParticipants"
   cliModel = MacsecModel.MacsecParticipants

BasicCli.addShowCommandClass( MacSecurityParticipantsCmd )

class MacSecurityStatusCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mac security status'
   data = {
      'mac': macMatcherForShow,
      'security': matcherSecurityForShow,
      'status': 'MAC security overall status information',
   }

   handler = "MacsecShowCliHandler.showStatus"
   cliModel = MacsecModel.MacsecStatus

BasicCli.addShowCommandClass( MacSecurityStatusCmd )

#----------------------------------------------------------------------------------
# "show tech-support" commands
#----------------------------------------------------------------------------------
CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2019-01-21 01:41:05',
   cmds=[ 'show mac security status',
          'show mac security mka counters detail',
          'show mac security counters detail', 
          'show mac security interface detail',
          'show mac security participants detail' ],
   cmdsGuard=lambda: _macsecStatus and _macsecStatus.macsecEnabled,
   summaryCmds=[ 'show mac security interface',
                 'show mac security participants',
                 'show mac security mka counters',
                 'show mac security counters' ],
   summaryCmdsGuard=lambda: _macsecStatus and _macsecStatus.macsecEnabled )

CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2022-11-16 03:58:15',
   cmds=[ 'show mac security ptp bypass' ],
   cmdsGuard=lambda: _macsecStatus and _macsecStatus.macsecEnabled )

CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2022-11-23 00:58:58',
   cmds=[ 'show mac security sak' ],
   cmdsGuard=lambda: _macsecStatus and _macsecStatus.macsecEnabled )

def getProfileNames( mode=None ):
   profiles = []
   for src in _macsecInput.values():
      profiles.extend( src.profile )
   return profiles

def getSource( mode ):
   return _macsecInput

class MacSecurityProfileCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mac security profile [ PROFILE_NAME ] [ source SRC_NAME ]'
   data = {
      'mac': macMatcherForShow,
      'security': matcherSecurityForShow,
      'profile': matcherForProfile,
      'PROFILE_NAME' : CliMatcher.DynamicNameMatcher( getProfileNames,
         'Profile name' ),
      'source': matcherForSource,
      'SRC_NAME': CliMatcher.DynamicNameMatcher( getSource, "Source name" ),
   }

   handler = "MacsecShowCliHandler.showMacsecProfile"
   cliModel = MacsecModel.MacsecAllProfiles

BasicCli.addShowCommandClass( MacSecurityProfileCmd )

class MacSecuritySakCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mac security sak [ interface INTFS ]'
   data = {
      'mac': macMatcherForShow,
      'security': matcherSecurityForShow,
      'sak' : 'MAC security sak information',
      'interface' : 'MAC security interface information', 
      'INTFS': Intf.rangeMatcher,
   }
   handler = "MacsecShowCliHandler.showSak"
   cliModel = MacsecModel.MacsecSaks

BasicCli.addShowCommandClass( MacSecuritySakCmd )

class MacSecurityPtpBypassCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mac security ptp bypass'
   data = {
      'mac' : macMatcherForShow,
      'security' : matcherSecurityForShow,
      'ptp' : 'MAC security PTP information',
      'bypass' : 'MAC security bypass information',
   }
   handler = "MacsecShowCliHandler.showPtpBypass"
   cliModel = MacsecModel.MacsecPtpBypass

BasicCli.addShowCommandClass( MacSecurityPtpBypassCmd )

class MacSecurityCapableInterfacesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show interfaces [ INTFS ] mac security capabilities'
   data = {
      'interfaces': 'Details related to interfaces',
      'INTFS': Intf.rangeMatcher,
      'mac': macMatcherForShow,
      'security': matcherSecurityForShow,
      'capabilities': 'MAC security capabilities information',
   }
   handler = "MacsecShowCliHandler.showMacsecCapableInterfaces"
   cliModel = MacsecModel.MacsecCapableInterfaces

BasicCli.addShowCommandClass( MacSecurityCapableInterfacesCmd )

def Plugin( entityManager ):

   global _macsecCounters
   global _macsecStatus
   global _macsecMkaStatus
   global _macsecInput
   global _hwStatusSliceDir

   _macsecCounters = SmashLazyMount.mount( entityManager,
         'macsec/counters', "Macsec::Smash::MessageCounters",
         SmashLazyMount.mountInfo( 'reader' ) )

   _macsecStatus = LazyMount.mount( entityManager,
         'macsec/status', "Macsec::Status", 'r' )

   _macsecMkaStatus = LazyMount.mount( entityManager,
         'macsec/mkaStatus', "Macsec::MkaStatus", 'r' )
  
   _macsecInput = LazyMount.mount( entityManager,
         'macsec/input', "Tac::Dir", 'ri' )

   _hwStatusSliceDir = LazyMount.mount( entityManager,
         'hardware/phy/status/macsec/slice', "Tac::Dir", 'r' )
