#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re, collections
from functools import wraps

import Ark
import Tac
import CliParser
import CliMatcher
import ConfigMount, LazyMount
from TypeFuture import TacLazyType

allProfileConfigDir = None
unitProfileConfigDir = None
maintenanceUnitConfigDir = None
maintenanceUnitStatusDir = None
defaultUnitProfile = None
maintenanceGroupConfigDir = None
maintRunConfigDir = None
onBootDurationMin = Tac.Type(
   "MaintenanceUnitProfile::OnBootDurationRange" ).minOnBootDuration
onBootDurationMax = Tac.Type(
   "MaintenanceUnitProfile::OnBootDurationRange" ).maxOnBootDuration
underMaintenance = Tac.Type( "Maintenance::Unit::State" ).underMaintenance
maintenanceModeEnter = Tac.Type( "Maintenance::Unit::State" ).maintenanceModeEnter
active = Tac.Type( "Maintenance::Unit::State" ).active
underGoingMaintStates = [ maintenanceModeEnter, underMaintenance ]
srcOnBoot = Tac.Type( "Maintenance::Unit::InputSrc" ).srcOnBoot
# States in which we show the on-boot maintenance flag.
onBootMaintShowFlagStates = [ maintenanceModeEnter, underMaintenance ]

showMaintenanceMatcher = CliMatcher.KeywordMatcher( 'maintenance',
                                                    helpdesc='Show Maintenance '
                                                    'information' )
maintenanceKwMatcher = CliMatcher.KeywordMatcher( 'maintenance',
                                          helpdesc='Configure Maintenance '
                                          'mode parameters' )

profileMatcher = CliMatcher.KeywordMatcher( 'profile', helpdesc='Configure profile' )

quiesceMatcher = CliMatcher.KeywordMatcher( 'quiesce',
                                      helpdesc='Execute Maintenance operation' )

defaultMatcher = CliMatcher.KeywordMatcher( 'default',
                                            helpdesc='Default information' )

maintenanceStates = Tac.Type( 'Maintenance::Unit::State' ).attributes

adminMaintenanceStates = [ underMaintenance, active ]

systemBuiltinUnitName = Tac.Type( "Maintenance::Unit::BuiltinUnitName" ).system
linecardBuiltinUnitName = Tac.Type( "Maintenance::Unit::BuiltinUnitName" ).linecard

allEthIntfBuiltinGroupName = \
   Tac.Type( "Group::BuiltinGroupName" ).allEthernetInterface
linecardBuiltinGroupPrefix = Tac.Type( "Group::BuiltinGroupName" ).linecardPrefix
bgpNeighVrfBuiltinGroupPrefix = \
   Tac.Type( "Group::BuiltinGroupName" ).allBgpNeighborVrfPrefix

matcherUnitName = CliMatcher.DynamicNameMatcher(
   lambda mode: maintenanceUnitStatusDir.status,
   pattern='[a-zA-Z0-9_-]+',
   helpname='WORD',
   helpdesc='name of the unit' )

intfGroupType = Tac.Type( "Group::GroupType" ).groupTypeInterface
bgpGroupType = Tac.Type( "Group::GroupType" ).groupTypeBgp
defaultProfile = \
   Tac.Type( 'Maintenance::Profile::DefaultProfile' ).systemDefaultProfileName
intfProfileType = Tac.Type( \
   "Maintenance::Profile::ProfileType" ).profileTypeInterface
bgpProfileType = Tac.Type( "Maintenance::Profile::ProfileType" ).profileTypeBgp
unitProfileType = Tac.Type( "Maintenance::Profile::ProfileType" ).profileTypeUnit
noneProfileType = Tac.Type( "Maintenance::Profile::ProfileType" ).profileTypeNone
dynamicUnitType = Tac.Type( "Maintenance::Unit::UnitType" ).dynamic
builtinUnitType = Tac.Type( "Maintenance::Unit::UnitType" ).builtin
builtinGroupType = Tac.Type( "Group::GroupOrigin" ).builtin
reservedProfileName = Tac.Type( 
   "Maintenance::Profile::DefaultProfile" ).reservedProfileName
nonDynamicUnitTypes = list( set( Tac.Type( 'Maintenance::Unit::UnitType'
                                         ).attributes ) -
                            { Tac.Type( 'Maintenance::Unit::UnitType'
                                           ).dynamic } )

# Any show command which is displaying the maintenance state is expected to
# use this dict to get the display string
maintenanceStateToStr = {
      'init'                    : 'Initial State',
      'active'                  : 'Not Under Maintenance',
      'maintenanceModeEnter'    : 'Entering Maintenance',
      'underMaintenance'        : 'Under Maintenance',
      'failedEnter'             : 'Failed to enter Maintenance',
      'maintenanceModeExit'     : 'Exiting Maintenance',
      'failedExit'              : 'Failed to exit Maintenance'
}

originToStr = {
      'userConfigured'          : 'User Configured',
      'dynamic'                 : 'Dynamic',
      'builtin'                 : 'Built-in'
}

maintEnterStageClass = Tac.Type( "Maintenance::Unit::MaintenanceStageClasses"
                                 ).maintEnterClass
maintExitStageClass = Tac.Type( "Maintenance::Unit::MaintenanceStageClasses"
                                ).maintExitClass
rateMonStage = Tac.Type( "Maintenance::Unit::MaintenanceStageList" ).rateMonStage
bgpStage = Tac.Type( "Maintenance::Unit::MaintenanceStageList" ).bgpStage
mlagStage = Tac.Type( "Maintenance::Unit::MaintenanceStageList" ).mlagStage
linkdownStage = Tac.Type( "Maintenance::Unit::MaintenanceStageList" ).linkDownStage
pimStage = TacLazyType( "Maintenance::Unit::MaintenanceStageList" ).pimStage

eventMgrStageDescription = \
    'Place holder to run scripts configured in event manager'
pimStageDisabledDesc = 'PIM not configured'
pimStageEnabledDesc = 'PIM configured'

StageAndDescription = collections.namedtuple( 'StageAndDescription',
                                              'name description internal' )
stages = []
stages.append( StageAndDescription( 'begin', eventMgrStageDescription, False ) )
stages.append( StageAndDescription( 'before_mlag', eventMgrStageDescription,
                                    False ) )
stages.append( StageAndDescription( mlagStage, 'Mlag Maintenance processing',
                                    True ) )
stages.append( StageAndDescription( 'after_mlag', eventMgrStageDescription, False ) )
stages.append( StageAndDescription( 'before_bgp', eventMgrStageDescription, False ) )
stages.append( StageAndDescription( bgpStage, 'BGP Maintenance processing', 
                                    True ) )
stages.append( StageAndDescription( 'after_bgp', eventMgrStageDescription, False ) )
stages.append( StageAndDescription( 'before_linkdown', eventMgrStageDescription,
                                    False ) )
stages.append( StageAndDescription( linkdownStage, 'Mlag Port Channel processing',
                                    True ) )
stages.append( StageAndDescription( 'after_linkdown', eventMgrStageDescription,
                                    False ) )
stages.append( StageAndDescription( 'before_ratemon', eventMgrStageDescription, 
                                    False ) )
stages.append( StageAndDescription( rateMonStage, 
                                    'Interface Rate Monitoring', True ) )
stages.append( StageAndDescription( 'after_ratemon', eventMgrStageDescription, 
                                    False ) )
stages.append( StageAndDescription( 'end', eventMgrStageDescription, False ) )

def getAdminState( operState ):
   if operState in  underGoingMaintStates:
      return underMaintenance
   return active

def isSubIntf( intfName ):
   return '.' in intfName

def parentIntf( intfName ):
   return intfName.split( '.' )[ 0 ] if isSubIntf( intfName ) else None

def toUtc( timestamp ):
   """ Convert a timestamp got from Tac::now() to UTC """
   utcTime = timestamp + Tac.utcNow() - Tac.now() if timestamp else timestamp
   return utcTime

def utcToStr ( utcTimestamp, relative=True ):
   """ Convert a timestamp, in UTC, to a string """
   return Ark.timestampToStr( utcTimestamp, relative=relative, now=Tac.utcNow() )

def createGroupKey( groupType, groupName ):
   return Tac.Value( "Group::GroupKey", groupType, groupName )

def createProfileKey( profileType, profileName ):
   return Tac.Value( "Maintenance::Profile::ProfileKey", profileType, profileName )

def dynamicName( typeName, componentName, vrfName=None ):
   unitName = "<Dynamic %s>" % typeName # pylint: disable=consider-using-f-string
   unitName += "<" + componentName + ">"
   unitName += "<>" if not vrfName else "<vrf-" + vrfName + ">"
   return unitName

def dynamicUnitName( componentName, vrfName=None ):
   return dynamicName( "Unit", componentName, vrfName=vrfName )

def dynamicGroupName( componentName, vrfName=None ):
   return dynamicName( "Group", componentName, vrfName=vrfName )

def isDynamicUnit( unitConfig ):
   return unitConfig.unitType == dynamicUnitType

DynamicComponentRe = "<Dynamic.+><(.+)><.*>"
VrfRe = "<Dynamic.+><.+><vrf-(.+)>"

def dynamicUnitComponentName( unitConfig ):
   if not isDynamicUnit( unitConfig ):
      return None
   return re.search( DynamicComponentRe, unitConfig.unitName ).group( 1 )

def dynamicUnitVrfName( unitConfig ):
   if not isDynamicUnit( unitConfig ):
      return None
   return re.search( VrfRe, unitConfig.unitName ).group( 1 )

def profileNameMatcher( namesFn ):
   return CliMatcher.DynamicNameMatcher( namesFn,
                                         pattern='[a-zA-Z0-9_-]+',
                                         helpname='WORD',
                                         helpdesc='name of the profile',
                                         priority=CliParser.PRIO_LOW )

def groupNameMatcher( namesFn, pattern='[a-zA-Z0-9_-]+' ):
   return CliMatcher.DynamicNameMatcher( namesFn,
                                         pattern=pattern,
                                         helpname='WORD',
                                         helpdesc='Group name' )


def maintRunningShowCliCheck( func ):
   '''
   Wrap the show maintenance related CLI handlers to check if
   maintenance mode is disabled and display warning indicating 
   the same.
   '''
   @wraps( func )
   def decorated( mode, *args, **kwargs ):
      if 'enabled' not in maintRunConfigDir:
         if kwargs.get( 'ignoreDisplay', False ) is False:
            mode.addWarning( "Maintenance Mode is disabled." )
      return func( mode, *args, **kwargs )
   return decorated

def maintModeAgentIsRunning():
   if 'enabled' in maintRunConfigDir: # pylint: disable=simplifiable-if-statement
      return True
   else:   
      return False   

#
# Profile class: Provides Library to add/remove each configured profile into
# 'all' collection
#

class Profile:
   def addProfile( self, profileKey ):
      allProfileConfigDir.newConfig( profileKey )

   def remProfile( self, profileKey ):
      del allProfileConfigDir.config[ profileKey ]

#
# MaintenanceUnit class: Holds cli state for each configured Maintenance Unit
#
class MaintenanceUnit:
   def __init__( self, unitName ):
      self.name_ = unitName
      maintenanceUnitConfigDir.newConfig( unitName )

   def name( self ):
      return self.name_

   def addGroup( self, groupKey ):
      unitConfig = maintenanceUnitConfigDir.config.get( self.name_ )
      if unitConfig:
         unitConfig.group[ groupKey ] = True

   def remGroup( self, groupKey ):
      unitConfig = maintenanceUnitConfigDir.config.get( self.name_ )
      if unitConfig:
         del unitConfig.group[ groupKey ]

#
# MaintenanceBuiltinUnit class: Holds cli state for each builtin Maintenance Unit
#
class MaintenanceBuiltinUnit:
   def __init__( self, unitName ):
      self.name_ = unitName
      unitConfig = maintenanceUnitConfigDir.newConfig( unitName )
      unitConfig.unitType = builtinUnitType

   def name( self ):
      return self.name_

#
# UnitProfile class: Provides library to add/remove configured unit profile
#
class UnitProfile( Profile ):
   def __init__( self, profileName ):
      self.name_ = profileName
      Profile.__init__( self )

   def addProfile( self, profileName ): # pylint: disable=arguments-renamed
      if profileName not in unitProfileConfigDir.config:
         profileConfig = unitProfileConfigDir.newConfig( profileName )
         Profile.addProfile( self, profileConfig.key )

   def remProfile( self, profileName ): # pylint: disable=arguments-renamed
      profileConfig = unitProfileConfigDir.config.get( profileName )
      if profileConfig:
         Profile.remProfile( self, profileConfig.key )
         del unitProfileConfigDir.config[ profileName ]

   def name( self ):
      return self.name_

#
# MaintenanceGroup class: Holds cli state for each configured maintenance group
#    such as profiles associated with a group
#
class MaintenanceGroup:
   def __init__( self, groupKey ):
      self.groupKey = groupKey

   def addProfile( self, profileKey ):
      maintenanceGroup = \
          maintenanceGroupConfigDir.newMaintenanceGroup( self.groupKey )
      maintenanceGroup.profile[ profileKey ] = True

   def remProfile( self, profileType, profileName ):
      if self.groupKey in maintenanceGroupConfigDir.maintenanceGroup:
         maintenanceGroup = \
             maintenanceGroupConfigDir.maintenanceGroup[ self.groupKey ]
         if profileName is None:
            for profile in maintenanceGroup.profile:
               if profile.type == profileType:
                  profileName = profile.name
         profileKey = Tac.Value( "Maintenance::Profile::ProfileKey", 
                                 type = profileType, name = profileName )
         del maintenanceGroup.profile[ profileKey ]

         if not maintenanceGroup.profile:
            del maintenanceGroupConfigDir.maintenanceGroup[ self.groupKey ]

   def remGroup( self ):
      del maintenanceGroupConfigDir.maintenanceGroup[ self.groupKey ]

def Plugin( entityManager ):
   global allProfileConfigDir, defaultUnitProfile
   global  maintenanceUnitConfigDir, unitProfileConfigDir
   global maintenanceGroupConfigDir, maintenanceUnitStatusDir
   global maintRunConfigDir 
   allProfileConfigDir = ConfigMount.mount( entityManager,
                                            'maintenance/profile/config/all',
                                            'Maintenance::Profile::ConfigDir', 'w' )
   maintenanceUnitConfigDir = ConfigMount.mount( entityManager,
                                                 'maintenance/unit/config',
                                                 'Maintenance::Unit::ConfigDir',
                                                 'w' )
   unitProfileConfigDir = ConfigMount.mount( entityManager,
                                             'maintenance/profile/config/unit',
                                             'MaintenanceUnitProfile::ConfigDir',
                                             'w' )
   defaultUnitProfile = ConfigMount.mount( entityManager,
                                           'maintenance/profile/config/default/unit',
                                           'Maintenance::Profile::DefaultProfile',
                                           'w' )
   maintenanceGroupConfigDir = ConfigMount.mount( 
      entityManager, 'maintenance/group/config',
      'Maintenance::MaintenanceGroup::ConfigDir', 'w' )
   maintenanceUnitStatusDir = LazyMount.mount( entityManager,
                                               'maintenance/unit/status',
                                               'Maintenance::Unit::StatusDir',
                                               'r' )
   maintRunConfigDir = LazyMount.mount(
      entityManager, 'maintenance/runnability/config', 
      'Tac::Dir', 'r' )
