# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import ConfigMount
import LazyMount
from CliMode.Mcs import McsServiceMode
import BasicCli
import CliCommand
from CliPlugin.ControllerCli import (
      CvxConfigMode,
      addNoCvxCallback,
      serviceKwMatcher,
)
import ReversibleSecretCli

mcsCliConfig = None
mcsStatus = None
httpServiceConfig = None
serviceInputConfigDir = None
controllerConfig = None

CVX_SERVICE_NAME = "Mcs"
HTTP_SERVICE_NAME = "mcs"

class McsServiceConfigMode( McsServiceMode, BasicCli.ConfigModeBase ):
   name = "MCS service configuration"

   def __init__( self, parent, session, name=None ):
      McsServiceMode.__init__( self, name )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def enableService( self, args ):
      mcsCliConfig.enabled = True
      httpServiceConfig.service[ HTTP_SERVICE_NAME ].enabled = True
      serviceInputConfigDir.service[ CVX_SERVICE_NAME ].enabled = True
      if not controllerConfig.enabled:
         self.addWarning( "MCS service will not start when CVX is shutdown" )

   def disableService( self, args ):
      mcsCliConfig.enabled = False
      httpServiceConfig.service[ HTTP_SERVICE_NAME ].enabled = False
      serviceInputConfigDir.service[ CVX_SERVICE_NAME ].enabled = False

def _clear( mode ):
   'NoCvx callbacks are called with only `mode`. No choice but to wrap.'
   mode.childMode( McsServiceConfigMode ).clear( mode, {} )

addNoCvxCallback( _clear )

#-----------------------------------------
# (config-cvx)# service mcs
# (config-cvx-mcs)#
#-----------------------------------------
class ServiceMcsCmd( CliCommand.CliCommandClass ):
   syntax = 'service mcs'
   noOrDefaultSyntax = syntax
   data = {
      'service': serviceKwMatcher,
      'mcs': 'Configure Media Control Service',
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( McsServiceConfigMode )
      mode.session_.gotoChildMode( childMode )

   noOrDefaultHandler = McsServiceConfigMode.clear

CvxConfigMode.addCommandClass( ServiceMcsCmd )

#-----------------------------------------
# [ no | default ] shutdown
#-----------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': 'Shutdown Media Control Service',
   }

   handler = McsServiceConfigMode.disableService
   noHandler = McsServiceConfigMode.enableService
   defaultHandler = handler

McsServiceConfigMode.addCommandClass( ShutdownCmd )

#-------------------------------------------------------------------------------
# [no] redis password <secret>
#-------------------------------------------------------------------------------
class RedisPasswordCommand( CliCommand.CliCommandClass ):
   syntax = "redis password SECRET"
   noOrDefaultSyntax = "redis password ..."
   data = {
      "redis": "Configure Redis settings",
      "password": "Configure Redis password",
      "SECRET": ReversibleSecretCli.defaultReversiblePwdCliExpr,
      }

   @staticmethod
   def handler( mode, args ):
      password = args.get( "SECRET" ) or ReversibleSecretCli.getDefaultSecret()
      mcsCliConfig.redisPassword = password

   noOrDefaultHandler = handler

McsServiceConfigMode.addCommandClass( RedisPasswordCommand )

def Plugin( entityManager ):
   global mcsCliConfig
   global mcsStatus
   global httpServiceConfig
   global serviceInputConfigDir
   global controllerConfig

   mcsCliConfig = ConfigMount.mount(
         entityManager, "mcs/config/cli", "Mcs::CliConfig", "w" )
   mcsStatus = LazyMount.mount(
         entityManager, "mcs/status/agent", "Mcs::AgentStatus", "r" )
   httpServiceConfig = ConfigMount.mount(
         entityManager, "mgmt/capi/config", "HttpService::Config", "w" )
   serviceInputConfigDir = ConfigMount.mount(
         entityManager, "controller/service/config",
                              "Controller::ServiceConfigDir", "w" )
   controllerConfig = LazyMount.mount(
         entityManager, "controller/config", "Controllerdb::Config", "r" )
