# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import ConfigMount
import CliMatcher
import LazyMount
from CliMode.McsClient import McsServiceClientMode
from CliMode.McsClient import McsServiceClientSecondaryCvxMode
import CliCommand
from CliPlugin import ControllerClient
from CliPlugin.ControllerClientCvxMgmtConfigMode import HeartbeatIntervalCmd
from CliPlugin.ControllerClientCvxMgmtConfigMode import HeartbeatTimeoutCmd
from CliPlugin.ControllerClientCvxMgmtConfigMode import ServerHostnameCmd
from CliPlugin.ControllerClientCvxMgmtConfigMode import SourceInterfaceCmd
from CliPlugin.ControllerClientCvxMgmtConfigMode import SslProfileNameCmd
from CliPlugin.ControllerClientCvxMgmtConfigMode import VrfNameCmd
from CliPlugin.IpAddrMatcher import IpAddrMatcher
from CliPlugin.ControllerClientCvxMgmtConfigMode import ShutdownCmd as \
      CvxMgmtShutdownCmd
import CliToken.Clear
import Tac

mcsClientConfig = None
mcsClientStatus = None
mcsClientControllerConfig = None
mcsClientControllerStatus = None
serviceConfigDir = None
secondarySysdbConfig = None
secondarySysdbStatus = None

# pylint: disable=protected-access

class McsServiceClientConfigMode( McsServiceClientMode, BasicCli.ConfigModeBase ):
   name = "MCS client configuration"

   def __init__( self, parent, session, name=None ):
      McsServiceClientMode.__init__( self, name )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def enableService( self, args ):
      mcsClientConfig.enabled = True
      serviceConfigDir.service[ 'Mcs' ].enabled = True

   def disableService( self, args ):
      mcsClientConfig.enabled = False
      serviceConfigDir.service[ 'Mcs' ].enabled = False

class McsServiceClientSecondaryCvxConfigMode( McsServiceClientSecondaryCvxMode,
      BasicCli.ConfigModeBase ):
   name = "Secondary Cvx Configuration"

   def __init__( self, parent, session, clusterName=None ):
      McsServiceClientSecondaryCvxMode.__init__( self, clusterName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#-----------------------------------------
# (config)#mcs client
# (config-mcs-client)#
#-----------------------------------------
class McsClientCmd( CliCommand.CliCommandClass ):
   syntax = 'mcs client'
   noOrDefaultSyntax = syntax
   data = {
      'mcs': 'Configure Media Control Service',
      'client': 'Configure Media Control Service Client',
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( McsServiceClientConfigMode )
      mode.session_.gotoChildMode( childMode )

   noOrDefaultHandler = McsServiceClientConfigMode.clear

BasicCli.GlobalConfigMode.addCommandClass( McsClientCmd )

#-----------------------------------------
# [ no | default ] shutdown
#-----------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': 'Shutdown Media Control Service Client',
   }
   handler = McsServiceClientConfigMode.disableService
   noHandler = McsServiceClientConfigMode.enableService
   defaultHandler = handler

McsServiceClientConfigMode.addCommandClass( ShutdownCmd )

#-----------------------------------------
# clear mcs routes all | ( source SOURCE group GROUP )
# ----------------------------------------
class ClearMcsRoutes( CliCommand.CliCommandClass ):
   syntax = 'clear mcs routes all | ( source SOURCE group GROUP )'
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'mcs': 'Media Controller Service',
      'routes': 'Multicast routes',
      'all': 'All multicast routes',
      'source': 'Source address to clear',
      'SOURCE': IpAddrMatcher( 'Source address' ),
      'group': 'Group address to clear',
      'GROUP': IpAddrMatcher( 'Group address' ),
   }

   @staticmethod
   def handler( mode, args ):
      mcsClientConfig.clearConfig = ()
      if 'all' in args:
         mcsClientConfig.clearConfig.all = True
      else:
         mcsClientConfig.clearConfig.all = False
         mcsClientConfig.clearConfig.source = args[ 'SOURCE' ]
         mcsClientConfig.clearConfig.group = args[ 'GROUP' ]
      mcsClientConfig.clearMcsRoutes += 1

BasicCli.EnableMode.addCommandClass( ClearMcsRoutes )

#-----------------------------------------
# (config-mcs-client)# cvx secondary [name]
# (config-mcs-client-cvx)#
#-----------------------------------------
class McsClientSecondaryCvxCmd( CliCommand.CliCommandClass ):
   syntax = 'cvx secondary [ NAME ]'
   noOrDefaultSyntax = 'cvx secondary ...'
   data = {
      'cvx': 'CVX Cluster',
      'secondary': 'Configure Secondary CVX Cluster',
      'NAME': CliMatcher.StringMatcher( helpdesc='Name of cvx cluster' )
   }

   @staticmethod
   def handler( mode, args ):
      clusterName = args.get( 'NAME', ControllerClient.clusterDefaultName )
      mcsClientControllerConfig.clusterName = clusterName
      childMode = mode.childMode( McsServiceClientSecondaryCvxConfigMode,
            clusterName=mcsClientControllerConfig.clusterName )
      mode.session_.gotoChildMode( childMode )

   noOrDefaultHandler = McsServiceClientSecondaryCvxConfigMode.clear

McsServiceClientConfigMode.addCommandClass( McsClientSecondaryCvxCmd )

# Under this (config-mcs-client-cvx)# we will register
# the cvx related cli commands

# Secondary shutdown command
#--------------------------------------------------------------------------------
# [ no | default ] shutdown
#--------------------------------------------------------------------------------
class SecondaryCvxShutdownCmd( CliCommand.CliCommandClass ):
   syntax = CvxMgmtShutdownCmd.syntax
   noOrDefaultSyntax = CvxMgmtShutdownCmd.syntax
   data = CvxMgmtShutdownCmd.data.copy()

   @staticmethod
   def handler( mode, args ):
      mcsClientControllerConfig.enabled = False

   @staticmethod
   def noHandler( mode, args ):
      mcsClientControllerConfig.enabled = True

   defaultHandler = handler

McsServiceClientSecondaryCvxConfigMode.addCommandClass( SecondaryCvxShutdownCmd )

# Secondary heartbeat interval
#--------------------------------------------------------------------------------
# [ no | default ] heartbeat-interval INTERVAL
#--------------------------------------------------------------------------------
class SecondaryCvxHeartbeatIntervalCmd( CliCommand.CliCommandClass ):
   syntax = HeartbeatIntervalCmd.syntax
   noOrDefaultSyntax = HeartbeatIntervalCmd.noOrDefaultSyntax
   data = HeartbeatIntervalCmd.data.copy()

   @staticmethod
   def handler( mode, args ):
      HeartbeatIntervalCmd._setHeartBeatPunchInterval(
            mode, args, mcsClientControllerConfig )

   noOrDefaultHandler = handler

McsServiceClientSecondaryCvxConfigMode.addCommandClass(
      SecondaryCvxHeartbeatIntervalCmd )

# Secondary heartbeat timeout
#--------------------------------------------------------------------------------
# [ no | default ] heartbeat-timeout TIMEOUT
#--------------------------------------------------------------------------------
class SecondaryCvxHeartbeatTimeoutCmd( CliCommand.CliCommandClass ):
   syntax = HeartbeatTimeoutCmd.syntax
   noOrDefaultSyntax = HeartbeatTimeoutCmd.noOrDefaultSyntax
   data = HeartbeatTimeoutCmd.data.copy()

   @staticmethod
   def handler( mode, args ):
      HeartbeatTimeoutCmd._setHeartBeatTimeout(
            mode, args, mcsClientControllerConfig )

   noOrDefaultHandler = handler

McsServiceClientSecondaryCvxConfigMode.addCommandClass(
      SecondaryCvxHeartbeatTimeoutCmd )

# Secondary server host HOSTNAME
#--------------------------------------------------------------------------------
# [ no | default ] server host HOSTNAME [ sysname SYSNAME ]
#--------------------------------------------------------------------------------
class SecondaryServerHostnameCmd( CliCommand.CliCommandClass ):
   syntax = ServerHostnameCmd.syntax
   noOrDefaultSyntax = ServerHostnameCmd.noOrDefaultSyntax
   data = ServerHostnameCmd.data.copy()

   @staticmethod
   def handler( mode, args ):
      ServerHostnameCmd._setServerHostname( mode, args, mcsClientControllerConfig )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ServerHostnameCmd._setNoServerHostname( mode, args, mcsClientControllerConfig )

McsServiceClientSecondaryCvxConfigMode.addCommandClass( SecondaryServerHostnameCmd )

# Secondary source-interface
#--------------------------------------------------------------------------------
# [ no | default ] source-interface INTF ...
#--------------------------------------------------------------------------------
class SecondarySourceInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = SourceInterfaceCmd.syntax
   noOrDefaultSyntax = SourceInterfaceCmd.noOrDefaultSyntax
   data = SourceInterfaceCmd.data.copy()

   @staticmethod
   def handler( mode, args ):
      SourceInterfaceCmd._setSourceIntf( mode, args, mcsClientControllerConfig )

   noOrDefaultHandler = handler

McsServiceClientSecondaryCvxConfigMode.addCommandClass( SecondarySourceInterfaceCmd )

# Secondary ssl profile
#--------------------------------------------------------------------------------
# [ no | default ] ssl profile SSLPROFILENAME
#--------------------------------------------------------------------------------
class SecondarySslProfileCmd( CliCommand.CliCommandClass ):
   syntax = SslProfileNameCmd.syntax
   noOrDefaultSyntax = SslProfileNameCmd.noOrDefaultSyntax
   data = SslProfileNameCmd.data.copy()

   @staticmethod
   def handler( mode, args ):
      SslProfileNameCmd._setSslProfileName( mode, args, mcsClientControllerConfig )

   noOrDefaultHandler = handler

McsServiceClientSecondaryCvxConfigMode.addCommandClass( SecondarySslProfileCmd )

# Secondary vrf
#--------------------------------------------------------------------------------
# [ no | default ] vrf VRF
#--------------------------------------------------------------------------------
class SecondaryVrfNameCmd( CliCommand.CliCommandClass ):
   syntax = VrfNameCmd.syntax
   noOrDefaultSyntax = VrfNameCmd.noOrDefaultSyntax
   data = VrfNameCmd.data.copy()

   @staticmethod
   def handler( mode, args ):
      VrfNameCmd._setVrfName( mode, args, mcsClientControllerConfig )

   noOrDefaultHandler = handler

McsServiceClientSecondaryCvxConfigMode.addCommandClass(
      SecondaryVrfNameCmd )

def Plugin( entityManager ):
   global mcsClientConfig
   global mcsClientStatus
   global mcsClientControllerConfig
   global mcsClientControllerStatus
   global serviceConfigDir
   global secondarySysdbConfig
   global secondarySysdbStatus

   mcsClientConfig = ConfigMount.mount( entityManager,
         "mcs/switch/config", "Mcs::Client::Config", "w" )
   mcsClientStatus = LazyMount.mount( entityManager,
         "mcs/switch/status", "Mcs::Client::Status", "r" )
   mcsClientControllerConfig = ConfigMount.mount( entityManager,
         "mcs/switch/controller/config", "ControllerClient::Config", "w" )
   mcsClientControllerStatus = LazyMount.mount( entityManager,
         "mcs/switch/controller/status", "ControllerClient::Status", "r" )
   serviceConfigDir = ConfigMount.mount( entityManager,
         "mgmt/controller/service/config", "Controller::ServiceConfigDir", "w" )
   secondarySysdbConfig = LazyMount.mount( entityManager,
         "mcs/switch/controller/publish/config",
         "ControllerClient::SysdbConfig", "r" )
   secondarySysdbStatus = LazyMount.mount( entityManager,
         "mcs/switch/controller/publish/status",
         "ControllerClient::SysdbStatus", "r" )
   ControllerClient.enableSecondaryCvxHook(
         mcsClientControllerConfig,
         mcsClientControllerStatus,
         secondarySysdbConfig,
         secondarySysdbStatus )
