# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCliModes
import CliCommand
import CliMatcher
import LazyMount
import ReversibleSecretCli
import Tac
import Toggles.MgmtGnmiClientToggleLib as MgmtGnmiClientToggle
from CliPlugin.ConfigMgmtMode import managementKwMatcher
from CliPlugin.MgmtGnmiClientModeCli import (
   MgmtGnmiClientMode,
   GnmiServerMode,
   GnmiServerGroupMode
)
from HostnameCli import IpAddrOrHostnameMatcher
from LocalUserLib import usernameRe

config = None

# ------------------------------------------------------------
# switch(config)# [no|default] management client api gnmi
# ------------------------------------------------------------
class MgmtGnmiClientModeCmd( CliCommand.CliCommandClass ):
   syntax = 'management client api gnmi'
   noOrDefaultSyntax = syntax
   data = {
      'management': managementKwMatcher,
      'client': 'Configure the switch as a client to external services',
      'api': 'Configure the switch as an API client',
      'gnmi': 'Configure the switch as a gNMI client'
   }
   handler = 'MgmtGnmiClientCliHandler.gotoMgmtGnmiClientMode'
   noOrDefaultHandler = 'MgmtGnmiClientCliHandler.noMgmtGnmiClientMode'

if MgmtGnmiClientToggle.toggleMgmtGnmiClientConfigEnabled():
   BasicCliModes.GlobalConfigMode.addCommandClass( MgmtGnmiClientModeCmd )

# ------------------------------------------------------------
# switch(config-mgmt-client-api-gnmi)# [no|default] server SERVER_NAME
# ------------------------------------------------------------
def getGnmiServerNames( mode=None ):
   return config.gnmiServer.keys()

gnmiServerNameMatcher = CliMatcher.DynamicNameMatcher( getGnmiServerNames,
                                                       'Name of gNMI server' )

class GnmiServerModeCmd( CliCommand.CliCommandClass ):
   syntax = 'server NAME'
   noOrDefaultSyntax = syntax
   data = {
      'server': 'Configure a gNMI server',
      'NAME': gnmiServerNameMatcher
   }
   handler = 'MgmtGnmiClientCliHandler.addGnmiServer'
   noOrDefaultHandler = 'MgmtGnmiClientCliHandler.removeGnmiServer'

MgmtGnmiClientMode.addCommandClass( GnmiServerModeCmd )

# ------------------------------------------------------------
# switch(config-gnmi-server-<SERVER_NAME>)# [no|default] address ADDRESS [port PORT]
# ------------------------------------------------------------
portMatcher = CliMatcher.IntegerMatcher( 1, 65535,
                                         helpdesc='Number of the port to use' )

class GnmiServerAddressCmd( CliCommand.CliCommandClass ):
   syntax = 'address ADDRESS [port PORT]'
   noOrDefaultSyntax = 'address ...'
   data = {
      'address': 'Hostname or IP address of gNMI server',
      'ADDRESS': IpAddrOrHostnameMatcher( helpname='WORD',
                                          helpdesc='Hostname or IP address',
                                          ipv6=True ),
      'port': 'Port of gNMI server (default is 6030)',
      'PORT': portMatcher,
   }
   handler = 'MgmtGnmiClientCliHandler.modifyGnmiServerAddress'
   noOrDefaultHandler = 'MgmtGnmiClientCliHandler.removeGnmiServerAddress'

GnmiServerMode.addCommandClass( GnmiServerAddressCmd )

# ------------------------------------------------------------
# switch(config-gnmi-server-<SERVER_NAME>)# [no|default] username USERNAME password
#                                           PASSWORD
# ------------------------------------------------------------
usernameMatcher = CliMatcher.PatternMatcher( usernameRe, helpname='USERNAME',
                                             helpdesc='Username string' )

class GnmiServerUsernamedPwdCmd( CliCommand.CliCommandClass ):
   syntax = 'username USERNAME password PASSWORD'
   noOrDefaultSyntax = 'username ...'
   data = {
      'username': 'Configure username to use on the gNMI server',
      'USERNAME': usernameMatcher,
      'password': 'Configure password to use on the gNMI server',
      'PASSWORD': ReversibleSecretCli.defaultReversiblePwdCliExpr,
   }
   handler = 'MgmtGnmiClientCliHandler.modifyGnmiServerUsernamePwd'
   noOrDefaultHandler = 'MgmtGnmiClientCliHandler.removeGnmiServerUsernamePwd'

GnmiServerMode.addCommandClass( GnmiServerUsernamedPwdCmd )

# ------------------------------------------------------------
# switch(config-mgmt-client-api-gnmi)# [no|default] group GROUP_NAME
# ------------------------------------------------------------
def getGnmiServerGroupNames( mode=None ):
   return config.gnmiServerGroup.keys()

gnmiServerGroupNameMatcher = CliMatcher.DynamicNameMatcher(
                                getGnmiServerGroupNames, 'Server-group name' )

class GnmiServerGroupModeCmd( CliCommand.CliCommandClass ):
   syntax = 'group GROUPNAME'
   noOrDefaultSyntax = syntax
   data = {
      'group': 'gNMI server-group definitions',
      'GROUPNAME': gnmiServerGroupNameMatcher
   }
   handler = 'MgmtGnmiClientCliHandler.addGnmiServerGroup'
   noOrDefaultHandler = 'MgmtGnmiClientCliHandler.removeGnmiServerGroup'

MgmtGnmiClientMode.addCommandClass( GnmiServerGroupModeCmd )

# ------------------------------------------------------------
# switch(config-gnmi-sg-<GROUP_NAME>)# [no|default] member SERVER_NAME
# ------------------------------------------------------------
class GnmiAddServerToServerGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'member NAME'
   noOrDefaultSyntax = syntax
   data = {
      'member': 'Add a gNMI server to the server-group',
      'NAME': gnmiServerNameMatcher
   }
   handler = 'MgmtGnmiClientCliHandler.addGnmiServerToServerGroup'
   noOrDefaultHandler = 'MgmtGnmiClientCliHandler.removeGnmiServerFromServerGroup'

GnmiServerGroupMode.addCommandClass( GnmiAddServerToServerGroupCmd )

def Plugin( entityManager ):
   global config
   config = LazyMount.mount( entityManager, 'mgmtGnmiClient/config',
                             'MgmtGnmiClient::Config', 'w' )
